/*
  Copyright (C) 1997,1998  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.

   GNU id, written by Arnold Robbins, arnold@audiofax.com,
   with major rewrite by David MacKenzie, djm@gnu.ai.mit.edu,
   is (C) 89, 90, 91, 92, 93, 1994 Free Software Foundation, Inc.
   Parts of id and miscellaneous GNU library functions assembled
   for xispid by Dimitrios P. Bouras <dbouras@hol.gr>
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include <string.h>
#include <varargs.h>
#include <sys/param.h>
#include <sys/stat.h>
#include "common.h"
#include "version.h"

#ifdef SUNOS5x
#include <limits.h>
#endif

struct passwd *getpwuid();
struct group *getgrgid(), *getgrent();
uid_t getuid(), geteuid();
gid_t getgid(), getegid();
void setgrent(), endgrent(), *malloc(), *realloc(), free();
int fflush(), fputs(), _flsbuf(), fprintf(), vfprintf(), printf(), getopt();

#if !defined(NGROUPS_MAX) && defined(NGROUPS)
#define NGROUPS_MAX NGROUPS
#endif

#ifndef SUNOS5x
#ifdef GLIBC2
static const char *private_strerror (errnum)
#else
static char *private_strerror (errnum)
#endif
     int errnum;
{
#ifdef GLIBC2
  extern const char *const sys_errlist[];
#else
  extern char *sys_errlist[];
#endif
  extern int sys_nerr;

  if (errnum > 0 && errnum <= sys_nerr)
    return sys_errlist[errnum];
  return "Unknown system error";
}
#define strerror private_strerror
#endif

void error (va_alist) va_dcl
{
  int status, errnum;
  char *message;
  extern char *program_name;
  va_list args;

  fflush (stdout);
  fprintf (stderr, "%s: ", program_name);

  va_start (args);
  status = va_arg(args, int);
  errnum = va_arg(args, int);
  message = va_arg(args, char*);
  vfprintf (stderr, message, args);
  va_end (args);

  if (errnum)
    fprintf (stderr, ": %s", strerror (errnum));
  putc ('\n', stderr);
  fflush (stderr);
  if (status)
    exit (status);
}

static void *fixup_null_alloc (n)
     size_t n;
{
  void *p;

  p = 0;
  if (n == 0)
    p = malloc ((size_t) 1);
  if (p == 0)
    error (1, 0, "memory exhausted");
  return p;
}

void *xmalloc (n)
     size_t n;
{
  void *p;

  p = malloc (n);
  if (p == 0)
    p = fixup_null_alloc (n);
  return p;
}

void *xrealloc (p, n)
     void *p;
     size_t n;
{
  if (p == 0)
    return xmalloc (n);
  p = realloc (p, n);
  if (p == 0)
    p = fixup_null_alloc (n);
  return p;
}

int getugroups (maxcount, grouplist, username)
     int maxcount;
     int *grouplist;
     char *username;
{
  struct group *grp;
  register char **cp;
  register int count = 0;

  setgrent ();
  while ((grp = getgrent ()) != 0)
    for (cp = grp->gr_mem; *cp; ++cp)
      if (!strcmp (username, *cp))
	{
	  if (maxcount != 0)
	    {
	      if (count >= maxcount)
		{
		  endgrent ();
		  return count;
		}
	      grouplist[count] = grp->gr_gid;
	    }
	  count++;
	}
  endgrent ();
  return count;
}

static void print_user ();
static void print_group ();
static void print_group_list ();
static void print_full_info ();
static void usage ();

/* The name this program was run with. */
char *program_name;

/* If nonzero, output only the group ID(s). -g */
static int just_group = 0;

/* If nonzero, output user/group name instead of ID number. -n */
static int use_name = 0;

/* If nonzero, output real UID/GID instead of default effective UID/GID. -r */
static int use_real = 0;

/* If nonzero, output only the user ID(s). -u */
static int just_user = 0;

/* If nonzero, output only the supplementary groups. -G */
static int just_group_list = 0;

/* The real and effective IDs of the user to print. */
static uid_t ruid, euid;
static gid_t rgid, egid;

/* The number of errors encountered so far. */
static int problems = 0;

/* If non-zero, display usage information and exit.  */
static int show_help;

/* If non-zero, print the version on standard output and exit.  */
static int show_version;

/*
 If non-zero, check the file name argument for the following:
 - name starts with ".xisppipe",
 - mode is 0010600 (i.e. named pipe read-write only by its owner), and
 - owner GID is equal to the caller's GID
*/
static int check_pipe;
static char *pipe_fname;

int main (argc, argv)
     int argc;
     char **argv;
{
  char *p;
  int optc;
  extern int optind;
  extern char *optarg;

  p = RCSid;
  program_name = argv[0];

  while ((optc = getopt (argc, argv, "gnruGhvc:"))
	 != EOF)
    {
      switch (optc)
	{
	case 0:
	  break;
	case 'g':
	  just_group = 1;
	  break;
	case 'n':
	  use_name = 1;
	  break;
	case 'r':
	  use_real = 1;
	  break;
	case 'u':
	  just_user = 1;
	  break;
	case 'G':
	  just_group_list = 1;
	  break;
	case 'v':
	  show_version = 1;
	  break;
	case 'h':
	  usage (0);
	  break;
	case 'c':
	  check_pipe = 1;
	  pipe_fname = optarg;
	  break;
	default:
	  usage (1);
	}
    }

  if (show_version)
    {
      printf ("xispid (X-ISP V.%s%s)\n", Version, PatchLevel);
      exit (0);
    }

  if (show_help)
    usage (0);

  if (check_pipe) {
    struct stat st;
    char *bnp = strstr(pipe_fname, PIPEFNAME);
    int tlen = strlen(PIPEFNAME);

    if (bnp == NULL || strlen(bnp) < tlen) return 1;
    if (strncmp(bnp, PIPEFNAME, tlen)) return 1;
    if (stat(pipe_fname, &st) < 0) return 1;
    if (st.st_mode != 010600 || st.st_gid != getgid()) return 1;
    return 0;
  }

  if (just_user + just_group + just_group_list > 1)
    error (1, 0, "cannot print only user and only group");

  if (just_user + just_group + just_group_list == 0 && (use_real || use_name))
    error (1, 0, "cannot print only names or real IDs in default format");

  if (argc - optind > 1)
    usage (1);

  if (argc - optind == 1)
    {
      struct passwd *pwd = getpwnam (argv[optind]);
      if (pwd == NULL)
	error (1, 0, "%s: No such user", argv[optind]);
      ruid = euid = pwd->pw_uid;
      rgid = egid = pwd->pw_gid;
    }
  else
    {
      euid = geteuid ();
      ruid = getuid ();
      egid = getegid ();
      rgid = getgid ();
    }

  if (just_user)
    print_user (use_real ? ruid : euid);
  else if (just_group)
    print_group (use_real ? rgid : egid);
  else if (just_group_list)
    print_group_list (argv[optind]);
  else
    print_full_info (argv[optind]);
  putchar ('\n');

  return (problems != 0);
}

static void print_user (uid)
     int uid;
{
  struct passwd *pwd = NULL;

  if (use_name)
    {
      pwd = getpwuid (uid);
      if (pwd == NULL)
	problems++;
    }

  if (pwd == NULL)
    printf ("%u", (unsigned) uid);
  else
    printf ("%s", pwd->pw_name);
}

static void print_group (gid)
     int gid;
{
  struct group *grp = NULL;

  if (use_name)
    {
      grp = getgrgid (gid);
      if (grp == NULL)
	problems++;
    }

  if (grp == NULL)
    printf ("%u", (unsigned) gid);
  else
    printf ("%s", grp->gr_name);
}

static void print_group_list (username)
     char *username;
{
  print_group (rgid);
  if (egid != rgid)
    {
      putchar (' ');
      print_group (egid);
    }

  {
    int ngroups;
    gid_t *groups;
    register int i;

    groups = (gid_t *) xmalloc (NGROUPS_MAX * sizeof (gid_t));
    if (username == 0)
      ngroups = getgroups (NGROUPS_MAX, groups);
    else
      ngroups = getugroups (NGROUPS_MAX, groups, username);
    if (ngroups < 0)
      {
	error (0, errno, "cannot get supplemental group list");
	problems++;
	free (groups);
	return;
      }

    for (i = 0; i < ngroups; i++)
      if (groups[i] != rgid && groups[i] != egid)
	{
	  putchar (' ');
	  print_group (groups[i]);
	}
    free (groups);
  }
}

static void print_full_info (username)
     char *username;
{
  struct passwd *pwd;
  struct group *grp;

  printf ("uid=%u", (unsigned) ruid);
  pwd = getpwuid (ruid);
  if (pwd == NULL)
    problems++;
  else
    printf ("(%s)", pwd->pw_name);
  
  printf (" gid=%u", (unsigned) rgid);
  grp = getgrgid (rgid);
  if (grp == NULL)
    problems++;
  else
    printf ("(%s)", grp->gr_name);
  
  if (euid != ruid)
    {
      printf (" euid=%u", (unsigned) euid);
      pwd = getpwuid (euid);
      if (pwd == NULL)
	problems++;
      else
	printf ("(%s)", pwd->pw_name);
    }
  
  if (egid != rgid)
    {
      printf (" egid=%u", (unsigned) egid);
      grp = getgrgid (egid);
      if (grp == NULL)
	problems++;
      else
	printf ("(%s)", grp->gr_name);
    }

  {
    int ngroups;
    gid_t *groups;
    register int i;

    groups = (gid_t *) xmalloc (NGROUPS_MAX * sizeof (gid_t));
    if (username == 0)
      ngroups = getgroups (NGROUPS_MAX, groups);
    else
      ngroups = getugroups (NGROUPS_MAX, groups, username);
    if (ngroups < 0)
      {
	error (0, errno, "cannot get supplemental group list");
	problems++;
	free (groups);
	return;
      }

    if (ngroups > 0)
      fputs (" groups=", stdout);
    for (i = 0; i < ngroups; i++)
      {
	if (i > 0)
	  putchar (',');
	printf ("%u", (unsigned) groups[i]);
	grp = getgrgid (groups[i]);
	if (grp == NULL)
	  problems++;
	else
	  printf ("(%s)", grp->gr_name);
      }
    free (groups);
  }
}

static void usage (status)
     int status;
{
  if (status != 0)
    fprintf (stderr, "Try `%s -h' for more information.\n",
	     program_name);
  else
    {
      printf ("Usage: %s [-gGnruhv] [user name]\n", program_name);
      printf ("   or: %s -c <file name>\n", program_name);
      printf ("\
\n\
  -g,             print only the group ID\n\
  -G,             print only the supplementary groups\n\
  -n,             print a name instead of a number, for -ugG\n\
  -r,             print the real ID instead of effective ID, for -ugG\n\
  -u,             print only the user ID\n\
  -h,             display this help and exit\n\
  -v,             output version information and exit\n\
  -c <file name>  check if the file name starts with %s, it has\n\
                  mode 0010600 and its group is same as caller's group ID\n\
\n\
Without any OPTION, print some useful set of identified information.\n\
", PIPEFNAME);
    }
  exit (status);
}

