/*
Copyright (C) 1996 Csar Crusius

This file is part of the DND Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* TAB SIZE = 8 */

#include <OffiX/DragAndDrop.h>
#include <X11/cursorfont.h>
#include <X11/Xatom.h>
#include <X11/Xmu/WinUtil.h>
#include <stdio.h>
#include <stdlib.h>
#include <values.h>

/* Local variables */
static Display		*dpy;		/* current display		*/
static XButtonEvent	StartEvent;	/* event that started the drag	*/
static int		DragPrecision;	/* minimum dx,dy to start drag	*/
static int		Dragging;	/* Drag state flag		*/
static int		DataOK;		/* Non-zero if data registered	*/
static Atom		DndProtocol;	/* ClientMessage identifier	*/
static Atom		DndSelection;	/* For the data transfers	*/
static Atom		WM_STATE;	/* Needed for icon stuff	*/
static Window		Target;		/* Drop window			*/
static Widget		MainWidget;	/* Main widget of application	*/
static int		DataType;	/* Current drag data type	*/
static XtEventHandler	OtherDrop;	/* Drop on non-reg app widget	*/
static XtEventHandler	RootDrop;	/* How to handle root drops	*/
static XtEventHandler	IconDrop;	/* How to handle icon drops	*/
static int		RootFlag;	/* Non-zero if dropped on root	*/

/*
 * Data for the Dnd cursors
 */
#include "cursor/MASK.xbm"
#include "cursor/file.xbm"
#include "cursor/file_mask.xbm"
#include "cursor/files.xbm"
#include "cursor/files_mask.xbm"
#include "cursor/dir.xbm"
#include "cursor/dir_mask.xbm"
#include "cursor/text.xbm"
#include "cursor/text_mask.xbm"
#include "cursor/grey.xbm"
#include "cursor/link.xbm"
#include "cursor/app.xbm"

typedef struct
{	int	Width,Height;
	char	*ImageData,*MaskData;
	int	HotSpotX,HotSpotY;
	Pixmap	ImagePixmap,MaskPixmap;
	Cursor	CursorID;
} CursorData;

static CursorData DndCursor[DndEND]={
  { 0,0,NULL,NULL,0,0,0 },
  { grey_width,	grey_height,	grey_bits,	MASK_bits	,16,16},
  { file_width, file_height,	file_bits,	file_mask_bits	,16,16},
  { files_width,files_height,	files_bits,	files_mask_bits	,16,16},
  { text_width,text_height,text_bits,text_mask_bits,text_x_hot,text_y_hot },
  { dir_width,	dir_height,     dir_bits,       dir_mask_bits	,16,16},
  { link_width,	link_height,	link_bits,	MASK_bits	,16,16},
  { app_width,  app_height,     app_bits,       MASK_bits	,16,16}
};

/* Local prototypes */
int    DndIsDragging(void);
void   DndStartAction(Widget widget,XtPointer data,XEvent *event,Boolean *p);
void   DndDispatchEvent(Widget widget,XtPointer data,XEvent *event,Boolean* p);
void   DndPropertyHandler(Widget widget,XtPointer data,XEvent *event,Boolean *p);
Widget DndGetMainWidget(Widget widget);

/* DndStartAction
 * 
 * Initializes the drag and drop structures. Binded to ButtonPress event in
 * registered drag widgets.
 * 
 * Last modification
 * 
 * Feb 01 1996 (v0.0) : created
 */
void
DndStartAction(Widget widget,XtPointer data,XEvent *event,Boolean *p)
{
	StartEvent=*((XButtonEvent *)event);
	Dragging=0; DataOK=0;
#ifdef VERBOSE
	fprintf(stderr,"Initializing drag and drop variables...\n");
	fprintf(stderr,"Root 0x%x Window 0x%x Subwindow 0x%x\n",
		StartEvent.root,StartEvent.window,StartEvent.subwindow);
#endif
}

/* DndHandleDragging
 * 
 * Takes care of the drag and drop process. Wait until the pointer had moved
 * a little. Then takes control over the pointer until the buttons are
 * released. After that send a Drag And Drop ClientMessage event. Returns
 * non-zero if a drop did take place.
 * 
 * Last modification
 * 
 * Feb 01 1996 (v0.0) : created
 */
int
DndHandleDragging(Widget widget,XEvent *event)
{
	XEvent	Event;
	Window	root	= RootWindowOfScreen(XtScreenOfObject(widget));
	XtAppContext app= XtWidgetToApplicationContext(widget);
	Window  DispatchWindow;
	
	if(Dragging) return 0;
	if(abs(StartEvent.x_root - event->xmotion.x_root) < DragPrecision && 
	   abs(StartEvent.y_root - event->xmotion.y_root) < DragPrecision)
		return 0;
	
#ifdef VERBOSE
	fprintf(stderr,"Initializing drag...\n");
#endif
	XUngrabPointer(dpy,CurrentTime);
	XGrabPointer(dpy,root,False,
		     ButtonMotionMask|ButtonPressMask|ButtonReleaseMask,
		     GrabModeSync,GrabModeAsync,root,
		     DndCursor[DataType].CursorID,
		     CurrentTime);

	Dragging=1; RootFlag=0;
	while(Dragging)
	{
		XAllowEvents(dpy,SyncPointer,CurrentTime);
		XtAppNextEvent(app,&Event);
		switch(Event.type)
		{
			case ButtonRelease:
				if(Event.xbutton.subwindow) RootFlag=0;
			  	else			    RootFlag=1;
				Dragging=0;
				break;
			default:
				XtDispatchEvent(&Event);
				break;
		 }
	}
	DataOK=0;
	XUngrabPointer(dpy,CurrentTime);
#ifdef VERBOSE
	fprintf(stderr,"Root flag : %d\n",RootFlag);
#endif
	if(!RootFlag)
	{
		Target=XmuClientWindow(dpy,Event.xbutton.subwindow);
#ifdef VERBOSE
		if(Target==Event.xbutton.subwindow)
		  fprintf(stderr,"Dropped on an icon.\n");
#endif
		if(Target==Event.xbutton.subwindow)
			DispatchWindow=Target;
		else	DispatchWindow=PointerWindow;
	}
	else
		Target=DispatchWindow=XtWindow(DndGetMainWidget(widget));
	
     	Event.xclient.type		= ClientMessage;
	Event.xclient.display		= dpy;
	Event.xclient.message_type	= DndProtocol;
	Event.xclient.format		= 32;
	Event.xclient.window		= Target;
	Event.xclient.data.l[0]		= DataType;
	Event.xclient.data.l[1]		= (long)event->xbutton.state;
	Event.xclient.data.l[2]		= (long)widget;
	Event.xclient.data.l[3]		= 0;
	Event.xclient.data.l[4]		= 0;
	
 	XSendEvent(dpy,DispatchWindow,True,NoEventMask,&Event);
#ifdef VERBOSE
	fprintf(stderr,"ClientMessage sent to 0x%x(0x%x).\n",
		DispatchWindow,Target);
#endif
	return 1;
}

/* DndDispatchEvent
 * 
 * This function receives the Dnd ClientMessage event and dispatch it to
 * the correct widget. Binded to the top level widget by DndInitialize.
 * 
 * Last Updated:
 * 
 * Feb 02 1996 (v0.0) : Created
 */
void
DndDispatchEvent(Widget widget,XtPointer data,XEvent *event,Boolean* p)
{
#ifdef VERBOSE
	fprintf(stderr,"Message received! Target 0x%x\n",Target);
#endif
	
	if(event->type!=ClientMessage ||
	   event->xclient.message_type != DndProtocol) return;

	/* So we have a Drop event. Now we must re-dispatch it to
	 * the correct window.
	 */
#ifdef VERBOSE
	fprintf(stderr,"It is a DND protocol message!\n");
#endif
	
	/* Drop in the root window: dispatch to app's rootdrop handler. */
	if(RootFlag)
	{	if(RootDrop!=NULL) RootDrop(widget,data,event,p);
		RootFlag=0; return;
	}
	RootFlag=0;

	/* Drop in the icon or in a widget that is not registered */
	if(!Target)
	{	
		/* Are we iconified ? Take the according action */
		if(DndIsIcon(widget) && IconDrop!=NULL)
			IconDrop(widget,data,event,p);
		if(!DndIsIcon(widget) && OtherDrop!=NULL)
			OtherDrop(widget,data,event,p);
		return;
	}
	
	/* Drop in a registered widget */
	if(XtWindow(widget)==Target)
	{	if(OtherDrop!=NULL) OtherDrop(widget,data,event,p);
		return;
	}
	event->xclient.window = Target;
	XSendEvent(dpy,Target,True,NoEventMask,event);
}

/* DndIsIcon
 * 
 * Return non-zero if the application is iconic (widget=toplevel)
 */
int
DndIsIcon(Widget widget)
{
	Atom JunkAtom;
	int JunkInt;
	unsigned long WinState,JunkLong;
	unsigned char *Property;
		
	XGetWindowProperty(dpy,XtWindow(widget),WM_STATE,
			   0L,2L,False,AnyPropertyType,
			   &JunkAtom,&JunkInt,&WinState,&JunkLong,
			   &Property);
	WinState=(unsigned long)(*((long*)Property));
	return (WinState==3);
}

/* DndInitialize
 * 
 * Must be called anywhere before the top level widget creation and the
 * main loop. Initialize global variables and bind the DndDispatch function
 * to the top level widget. Creates the cursors to be used in drag actions.
 * 
 * Last modification
 * 
 * Feb 01 1996 (v0.0) : created
 */
void
DndInitialize(Widget shell)
{
	int	 screen,i;
	Colormap colormap;
	Window	 root;
	XColor	 Black,White;

	dpy	= XtDisplayOfObject(shell);
	screen	= DefaultScreen(dpy);
	colormap= DefaultColormap(dpy,screen);
	root	= DefaultRootWindow(dpy);
		

	Black.pixel=BlackPixel(dpy,screen);
	White.pixel=WhitePixel(dpy,screen);
	XQueryColor(dpy,colormap,&Black);
	XQueryColor(dpy,colormap,&White);
	
	for(i=1;i!=DndEND;i++)
	{
		DndCursor[i].ImagePixmap=
			XCreateBitmapFromData(dpy,root,
					      DndCursor[i].ImageData,
					      DndCursor[i].Width,
					      DndCursor[i].Height);
		DndCursor[i].MaskPixmap=
			XCreateBitmapFromData(dpy,root,
					      DndCursor[i].MaskData,
					      DndCursor[i].Width,
					      DndCursor[i].Height);
		DndCursor[i].CursorID=
			XCreatePixmapCursor(dpy,DndCursor[i].ImagePixmap,
					    DndCursor[i].MaskPixmap,
					    &Black,&White,
					    DndCursor[i].HotSpotX,
					    DndCursor[i].HotSpotY);
	}
	
	DndCursor[0].CursorID=XCreateFontCursor(dpy,XC_question_arrow);
	
	DndProtocol=XInternAtom(dpy,"DndProtocol",FALSE);
	DndSelection=XInternAtom(dpy,"DndSelection",FALSE);
	WM_STATE=XInternAtom(dpy,"WM_STATE",True);
	Dragging=0;
	DragPrecision=10;
	XtAddEventHandler(shell,NoEventMask,True,DndDispatchEvent,NULL);
	OtherDrop=RootDrop=IconDrop=NULL;
	RootFlag=0;
	MainWidget=shell;
}

/* DndRegisterRootDrop, DndRegisterIconDrop, DndRegisterOtherDrop
 * 
 * Register the application for "unusual" drops.
 */
void DndRegisterRootDrop(XtEventHandler handler)   { RootDrop=handler; }
void DndRegisterIconDrop(XtEventHandler handler)   { IconDrop=handler; }
void DndRegisterOtherDrop(XtEventHandler handler)  { OtherDrop=handler; }

/* DndMultipleShells and DndAddShell (DndGetMainWidget)
 * 
 * An attempt to make DND compatible with multiple-window applications.
 * This should be compatible with the "unmapped" top-level approach
 * for multiple-windows (see Xt manual). To use, call DndMultipleShells
 * just one time, after DndInitialize, and register new shells with
 * DndAddShell. Certainly it's not a definitive solution, but works
 * sometimes.
 */
void DndMultipleShells(void)
{
	MainWidget=0;
}

void DndAddShell(Widget widget)
{
	MainWidget=0;
	XtAddEventHandler(widget,NoEventMask,True,DndDispatchEvent,NULL);
}

Widget DndGetMainWidget(Widget widget)
{
	if(MainWidget) return MainWidget;
#ifdef VERBOSE
	fprintf(stderr,"Multiple shells.\n");
#endif VERBOSE
	
	while(XtParent(widget) && XtIsRealized(XtParent(widget))==True)
		widget=XtParent(widget);
	
	return widget;
}

/* DndUpdateTargetProc
 * 
 * Update the Target global variable, so the main handler can dispatch
 * to the correct one. Based on Enter/Leave events. When the pointer leaves,
 * Target=None. When the pointer enter, then update Target to this widget.
 */
void DndUpdateTargetProc(Widget widget,XtPointer data,XEvent *event,Boolean* p)
{
	if(event->type==EnterNotify)
		Target=XtWindowOfObject(widget);
	else
	  	Target=None;
}

/* DndRegisterDropWidget
 * 
 * Each widget that accepts a drop must register within the protocol. This
 * function binds two functions to the widget:
 * 
 * DndUpdateTargetProc : called by EnterWindow and LeaveWindow events.
 * user handler        : called by the Dnd Event sent by DndDispatchEvent.
 * 
 * Last modification
 * 
 * Feb 01 1996 (v0.0) : created
 */
void
DndRegisterDropWidget(Widget widget,XtEventHandler handler,XtPointer data)
{
	XtAddEventHandler(widget,EnterWindowMask|LeaveWindowMask,
			  False,DndUpdateTargetProc,data);
	XtAddEventHandler(widget,NoEventMask,True,handler,data);
}

/* DndRegisterDragWidget
 * 
 * Each widget that accepts a drag must register within the protocol. This
 * function binds two functions to the widget:
 * 
 * DndStartAction : called by ButtonPress event.
 * user handler   : called by ButtonMotion events.
 * 
 * Last modification
 * 
 * Feb 01 1996 (v0.0) : created
 */
void
DndRegisterDragWidget(Widget widget,XtEventHandler handler,XtPointer data)
{
	XtAddEventHandler(widget,ButtonPressMask,False,DndStartAction,data);
	XtAddEventHandler(widget,ButtonMotionMask,False,handler,data);
}

int DndIsDragging(void) { return Dragging; }

/* DndSetData
 * 
 * Updates the selection data.
 * 
 * Last modification
 * 
 * Feb 05 1996 (v0.0): created
 * Feb 29 1996 (v0.1): long sizes
 */
void DndSetData(int Type,unsigned char *Data,unsigned long Size)
{
	Window root		= DefaultRootWindow(dpy);
	int    AuxSize;

	if(DataOK) return;
	
	/* Set the data type */
	DataType = Type>=DndEND ? 0 : Type ;
		
	/* Set the data */
	AuxSize = Size > MAXINT ? MAXINT : (int)Size ; 
	XChangeProperty(dpy,root,DndSelection,XA_STRING,8,
			PropModeReplace,Data,AuxSize);
	for(Size-=(unsigned long)AuxSize;Size;Size-=(unsigned long)AuxSize)
	{	Data+=AuxSize;
		AuxSize = Size > MAXINT ? MAXINT : (int)Size ;
		XChangeProperty(dpy,root,DndSelection,XA_STRING,8,
				PropModeAppend,Data,AuxSize);
	}
	
	/* Everything is now ok */
	DataOK=1;
}

/* DndGetData
 * 
 * Return a pointer to the current data. Se HOWTO for more details.
 */
void
DndGetData(unsigned char **Data,unsigned long *Size)
{
	Window root		= DefaultRootWindow(dpy);

	Atom	ActualType;
	int	ActualFormat;
	unsigned long RemainingBytes;
      
	XGetWindowProperty(dpy,root,DndSelection,
			   0L,1000000L,
			   FALSE,AnyPropertyType,
			   &ActualType,&ActualFormat,
			   Size,&RemainingBytes,
			   Data);
}	

/* DndDataType DndDragButtons DndSourceWidget
 * 
 * Return information about the Dnd event received. If a non-dnd event is
 * passed, the function DndDataType returns DndNotDnd, and the others
 * return zero.
 * 
 * Last Updated:
 * 
 * Feb 02 1996 (v0.0) : created.
 */
int DndDataType(XEvent *event)
{
	int Type;
	
	if(event->xclient.message_type!=DndProtocol) return DndNotDnd;
	Type=(int)(event->xclient.data.l[0]);
	if(Type>=DndEND) Type=DndUnknown;
	return Type;
}

unsigned int DndDragButtons(XEvent *event)
{
	if(event->xclient.message_type!=DndProtocol) return 0;
	return (unsigned int)(event->xclient.data.l[1]);
}

Widget DndSourceWidget(XEvent *event)
{
	if(event->xclient.message_type!=DndProtocol) return 0;
	return (Widget)(event->xclient.data.l[2]);
}
