/************************************************************************
 * rpncalc.c								*
 *									*
 * A little RPN (Reverse Polish Notation) calculator,                   *
 * rudimentary emulating a HP 28S. 					*
 * 								        *
 * rpncalc is (c) David Frey, 1993, 1994, 1995				*
 *								        * 
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *									* 
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *									* 
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to the Free Software          *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
 *									* 
 ************************************************************************/

/* $Id: rpncalc.c,v 1.2 1996/09/13 20:21:29 david Rel $
 * $Log: rpncalc.c,v $
 * Revision 1.2  1996/09/13 20:21:29  david
 * lclint additions
 *
 * Revision 1.1  1996/07/13 20:53:44  david
 * Added operator completion, renaming due to linting
 *
 * Revision 1.0  1995/12/31 18:14:13  david
 * Initial revision
 * */ 

#include <stdio.h>
#include <stdlib.h> 
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>
#include <signal.h> 

#ifdef HAVE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include "rpncalc.h"
#include "cmds.h"
#include "utils.h"

extern int pushtostack;

char *progname;

struct option const long_options[] =
{
  {"help",    no_argument, 0, 'h'},
  {"help",    no_argument, 0, '?'},
  {"version", no_argument, 0, 'v'},
  {(char *)0, 0,           0, (char)0}
};

double help(void)
{
#define HELP "\
The following operations will be emulated:\n\
  push  pop  pick  swap  over  roll  dup  dupn  drop  dropn  depth\n\
\n\
  chs   +    -     *     /     ^     inv  sqrt  sqr\n\
  sin   cos  tan   asin  acos  atan  atan2\n\
  sinh  cosh tanh  asinh acosh atanh\n\
  ln    log  ld    exp   alog  shl\n\
  j0    j1   jn    y0    y1    yn\n\
  erf   erfc lgamma\n\
  abs   ceil fact  mod  div    gcd   sum  prod\n\
\n\
  hex   dec  oct   and  &      or    |    not   !     xor\n\
  prec\n\
\n\
  The following constants will be recognised: pi, e.\n\
  Delimiters are ',', ';', space, tab and newline.\n"

  printf(HELP);

  return 0.0; /* dummy value */
}

double warranty(void)
{
#define COPYRIGHT "\
This program is free software; you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation; either version 2 of the License, or\n\
(at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with this program; if not, write to the Free Software Foundation,\n\
Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"

  printf(COPYRIGHT);
  return 0.0; /* dummy value */
}

static int cmp(const void *f1, const void *f2)
/* (String-)Compare the two function names, cf. iX 11/95 p 212 */
{
  return strcmp(((const struct cmd *)f1)->fname,
		((const struct cmd *)f2)->fname);
}

/* Evaluate the command line. The work is done in this procedure ! */
static int eval(char line[])
{
  int quit;		/* flags */
  char delimiters[] = " ,;\t\f\n";
  char *cmnd, *rest;		 /* one command line */
  double op1, op2, res;		 /* operands */

  cmnd = strtok(line, delimiters); res = 0; pushtostack=1;
  quit = ((cmnd == NULL) ||
  	  (strncmp(cmnd, "quit", 4) == 0) || (strncmp(cmnd,"q",1) == 0) || 
	  (strncmp(cmnd, "exit", 4) == 0));

  while (!quit)
  {
      /* Is it a number ? It yes, push it onto the stack. */      

      /* First assume it's a double */
      op1 = strtod(cmnd, &rest);
      if (strcmp(rest, "") == 0) 
      {
	(void)push(op1); /* It is a float */
      }
      else
      {
	/* converting number according to the rules of the C programming
	 * language, e.g 0=octal, 0x=hexadecimal
	 */
	op1 = (double)strtol(cmnd, &rest, 0);
	if (strcmp(rest, "") == 0) 
	{
	  (void)push(op1); /* it is an integer */
	}
	else
	{
	  struct cmd dummykey;
	  struct cmd *f; 

	  /* the following bsearch-trick is from iX 11/95 p 212 */
	  dummykey.fname=cmnd;
   	  f=(struct cmd *)bsearch(&dummykey, cmdtab, NCMDTAB,
				   sizeof cmdtab[0], cmp);
	  if (f == NULL)
	  {
	    fprintf(stderr,"%s: unknown command `%s'.\n",progname, cmnd);
	  }
	  else
	  {
	    errno=0;
	    switch (f->argno)
	    {
	      case 0: res=(*f->fnct)(); break;
	      case 1: op1=pop(); res=(*f->fnct)(op1); break;
	      case 2: op2=pop(); op1=pop(); res=(*f->fnct)(op1,op2); break;
	    }
	    if (errno != 0) { perror(NULL); }
	    else
	    {
	      if (f->pushrestostack && pushtostack) (void)push(res);
	    }
	  }
	}
      }
      cmnd = strtok(NULL, delimiters);
      quit = ((cmnd == NULL) ||
  	      (strncmp(cmnd, "quit", 4) == 0) || (strncmp(cmnd,"q",1) == 0) || 
  	      (strncmp(cmnd, "exit", 4) == 0));

  }
    
  /* return quit; */
  return (cmnd != NULL);
}

/* print a short usage statement. Indicate that the argument must be quoted. */
static void usage(void)
{
  fprintf(stderr, "usage: %s [-h][-v] [\"expression\"]\n", progname);
}

#ifdef HAVE_READLINE
/* the initialisation code was taken from:
     fileman.c -- A tiny application which demonstrates how to use the
     GNU Readline library.  This application interactively allows users
     to manipulate files and their modes. 
 */

/* Attempt to complete on the contents of TEXT.  START and END show the
   region of TEXT that contains the word to complete.  We can use the
   entire line in case we want to do some simple parsing.  Return the
   array of matches, or NULL if there aren't any. */

char **rpncalc_completion(char *text, int start, int end)
{
  return completion_matches (text, command_generator);
}

/* Generator function for command completion.  STATE lets us know whether
   to start from scratch; without any state (i.e. STATE == 0), then we
   start at the top of the list. */
char *command_generator (char *text, int state)
{
  static int list_index, len;
  char *name;

  /* If this is a new word to complete, initialize now.  This includes
     saving the length of TEXT for efficiency, and initializing the index
     variable to 0. */
  if (!state) { list_index=0; len=strlen (text); }

  /* Return the next name which partially matches from the command list. */
  while (list_index < NCMDTAB)
  {
    name = cmdtab[list_index].fname; list_index++;

    if (strncmp (name, text, len) == 0) return (dupstr(name));
  }

  /* If no names matched, then return NULL. */
  return ((char *)NULL);
}

/* Tell the GNU Readline library how to complete.  We want to try to complete
   on command names if this is the first word in the line. */
void initialize_readline(int interactive)
{
  /* Allow conditional parsing of the ~/.inputrc file. */
  rl_readline_name = "rpncalc";

  /* Tell the completer that we want a crack first. */
  if (interactive)
  {
    rl_attempted_completion_function = (CPPFunction *)rpncalc_completion;
  }
  else
  {
    rl_bind_key('\t', rl_insert()); 
    /* Suppress key binding of TAB in batch mode */
  }
}

#endif

int main(int argc, char *argv[])
{
  int c, i;
  char *line=NULL;	/* entire line  */
  int quit;			/* should we quit ? */

  progname=strrchr(argv[0],'/');
  if (progname==NULL) progname=argv[0];
  else progname++;

  while ((c = getopt_long(argc, argv, "h?v",
			  long_options, (int *) 0)) != EOF)
  {
    switch (c)
    {
      case 0  : break;
      case 'h':
      case '?': usage(); return 0;
      case 'v': fprintf(stderr,"%s version 1.0\n", progname); return 1;
      default : break;
    }
  }

  fprintf(stderr,"%s version 1.0. Copyright (c) 1993, 1995 David Frey.\n",
	  progname);
  fprintf(stderr,"This is free software with ABSOLUTELY NO WARRANTY.\n");

  signal(SIGFPE,SIG_IGN); /* ignore floating point exceptions. */

  /* Sort the command table, from iX 11/95 p 212 */
  qsort(cmdtab, NCMDTAB, sizeof cmdtab[0], cmp);

  if (argc > optind)
  {
    short int l;

    l=0; for(i=optind;i<argc;i++) l += (strlen(argv[optind])+1);
    line=(char *)xmalloc(l+1);
    strcpy(line, argv[optind]);
    for (i = optind+1; i < argc; i++)
    {
      strcat(line, " "); strcat(line, argv[i]);
    }

    quit=eval(line); (void)showstack();
    free(line); line=NULL;
  }
  else
  {
    int interactive;

    interactive=isatty(0); line=NULL;

#ifdef HAVE_READLINE    
    initialize_readline(interactive);
#endif
    if (interactive)
    {
      fprintf(stderr,"For details, type `warranty'.\n");
      fprintf(stderr,"Type `quit' to quit and `?' to get a summary.\n");

#ifdef HAVE_READLINE    
      using_history();
#endif
    }

    do
    {
      line=getline();
      quit=(line == NULL);
      if (!quit)
      {
	/* Skip white spaces */
	while (*line != '\0' && isspace(*line)) line++;
	if (*line != '\0')
	{
#ifdef HAVE_READLINE          
	  add_history(line);
#endif
	  quit=eval(line);
	  if (!quit) (void)showstack();
	}
      }
      free(line); line=NULL; /* line was malloc'd by readline/getline. */
    }
    while (!quit);
#ifdef HAVE_READLINE 
    remove_history();
#endif    
  }  
  return 0;
}						 /* main */
