#include "wily.h"
#include "data.h"
static char*	badrange ="bad range";

/**********************************************
 * Handle messages to/from remote processes
 **********************************************/

static char*
data_attach(Data*d, int fd, ushort emask)
{
	if(d->fd)
		return "This window already attached";
	else {
		d->fd = fd;
		d->emask = emask;
		return 0;
	}
}

/* Change the d->label to 'label', update d->path.
 * Only called in response to a remote message.
 */
static void
data_changelabel(Data *d,char*label)
{
	pathexpand(label, 0, d->path);
	pathcontract(d->label, d->path);
	tag_setlabel(d->tag, d->label);
}

/* Return newline-separated list of tab-separated (winname,id) tuples
 */
static char *
data_list(void)
{
	static char	*buf=0;
	static int		alloced=0;
	int	size = 0;
	Data	*d;
	char	*ptr;

	/* calculate the size of buffer required */
	size =0;
	for(d=dataroot; d; d=d->next) {
		size += strlen(d->path) + 15;
	}
	
	if(size > alloced) {
		alloced = size;
		buf = srealloc(buf, alloced);
	}
	
	ptr = buf;
	for(d=dataroot; d; d=d->next) {
		sprintf(ptr, "%s\t%d\n", d->path, d->id);
		ptr += strlen(ptr);
	}
	return buf;
}

static int
msg_send(Msg*m, int fd)
{
	static uchar*buf=0;
	static int	alloced=0;
	ulong	size = msg_size(m);

	SETSIZE(buf, alloced, size);
	msg_flatten(m, buf);
	return write(fd,buf,size)!=size;
}

static void
msg_new(Msg*m)
{
	View	*v;
	Data	*d;
	Path	tmp;
	
	pathexpand(m->s, 0, tmp);
	if(!(v = data_find(tmp)))
		v = data_open(tmp, true);
	d = view_data(v);

	/* for now, NO windows created externally are backed up */
	if(!m->flag)
		data_setbackup(d,0);	
	m->w = d->id;
}

/* Process a message which arrived on 'fd'.  Modifies 'm'. */
static void
msg_fill(Msg*m, int fd)
{
	Data	*d=0;
	View *v=0;
	Bool	show;
	char	*s;
	Text	*t;

	/* WMlist or WMnew don't need to be associated with a valid window.
	 */
	switch(m->t) {
	case WMlist:	m->s = data_list();	return;
	case	WMnew:	msg_new(m); return;
	default:	break;
	}

	if((d=data_findid(m->w))) {
		t = d->t;
		v = text_view(t);
	} else  {
		m->s = "No window with this id";
		m->t = WRerror;
		return;
	}

	switch(m->t) {
	case	WMattach:
		if( (s = data_attach(d, fd, m->flag)) ) {
			m->s = s;
			m->t = WRerror;
		}
		break;
	case	WMsetname:	data_changelabel(d, m->s);		break;
	case	WMsettools:	tag_addtool(d->tag, m->s);	break;
	case	WMread:
		if (text_badrange(t,m->r)) {
			m->t = WRerror;
			m->s = badrange;
		} else {
			m->s = text_duputf(t, m->r);
		}
		break;
	case WEreplace:      /* we've already handled this */        break;
	case	WMreplace:
		if (text_badrange(t,m->r)) {
			m->t = WRerror;
			m->s = badrange;
		} else {
			text_replaceutf(t, m->r, m->s);
		}
		break;
	case	WMexec:
	case	WEexec:		run(v, m->s, 0);			break;
	case	WMgoto:
	case	WEgoto:
		show = (m->t == WEgoto) || m->flag;

		if(view_goto(&v, &m->r, m->s)){
			if (show) {
				view_show(v, m->r);
				view_select(v, m->r);
				view_warp(v, m->r);
			}
			m->w = text_data(view_text(v))->id;
		} else {
			m->s = "search failed";
		}
		break;
	default:
		m->t = WRerror;
		m->s = "unknown message type";
	}
}

static Bool
msg_form_and_send(Mtype t,  Id m, Id w, Range	r, ushort	flag, char	*s, int fd)
{
	Msg	msg;

	msg.t = t;
	msg.m = m;
	msg.w = w;
	msg.r = r;
	msg.flag = flag;
	msg.s = s;

	msg_send(&msg, fd);
	return true;
}

/* Process a message which arrived on 'fd' */
void
msg_process(Msg*m, int fd)
{
	Bool	isbounce;

	isbounce = m->t < WEfencepost;
	
	msg_fill(m, fd);

	if(!isbounce){
		if(m->t != WRerror)
			m->t ++;
		(void)msg_send(m,fd);
	} else if (m->t == WRerror) {
		diag(0, "error from bounced event %s", m->s);
		msg_print(m);
	}
}

/*
 * If someone's monitoring this window for 'replace' events,
 * send them one.  If we fail to send the message, or don't need
 * to send the message, return false.
 */
Bool
data_sendreplace(Data *d,Range r, Rstring s)
{
	char	*buf;
	Bool	retval;

	if(! ( d && d->emask & WEreplace) )
		return false;

	buf = salloc(RSLEN(s) * UTFmax);
	buf[texttoutf(buf, s.r0, s.r1)] = '\0';
	retval = msg_form_and_send(WEreplace, 0, d->id, r, 0, buf, d->fd);
	free(buf);
	return retval;
}

/* see data_sendreplace */
Bool
data_sendgoto(Data *d,Range r, char *s)
{
	if(! ( d && d->emask & WEgoto) )
		return false;
	return msg_form_and_send(WEgoto, 0, d->id, r, 0, s, d->fd);
}

/* see data_sendreplace */
Bool
data_sendexec(Data *d,char*cmd, char *arg)
{
	Bool	retval;
	char	*s;

	if(! ( d && d->emask & WEexec) )
		return false;

	if(arg) {
		s = salloc( strlen(cmd) + strlen(arg) + 2);
		sprintf(s, "%s %s", cmd, arg);
	} else {
		s = cmd;
	}
	retval = msg_form_and_send(WEexec, 0, d->id, range(0,0), 0, s, d->fd);
	if(arg)
		free(s);
	return retval;
}

/* Stop any data from sending events to 'fd' */
void
data_fdstop(int fd)
{
	Data	*d;

	for (d=dataroot; d; d = d->next) {
		if(d->fd == fd){
			d->fd = 0;
			d->emask = 0;
		}
	}
}

