#include "wily.h"
#include "key.h"
#include <ctype.h>
#include <sys/wait.h>
#include <signal.h>

static char *	historyfile;
static char *	shell;

static void	ex_parent(ulong, int, ulong, int,  char *, char *, int, View *);
static void	ex_child(int, int, View *, char *, char *, char *);
static void	childfds(int fderr, int fdout, View *v);

/* collect any waiting processes, but don't block */
void
reap(void)
{
	int	stat_loc;

	while ( waitpid(-1, &stat_loc, WNOHANG)> 0 )
		;
}

void
exec_init(void)
{
	historyfile = getenv("HISTORY");
	if(!historyfile)
		historyfile = getenv("history");

	shell =  getenv("SHELL");
	if(!shell)
		shell=DEFAULTSHELL;
}

/* Record 'cmd' in the history file */
static void
history(char *cmd)
{
	FILE *fp;

	if(!historyfile)
		return;

	fp = fopen(historyfile, "a");
	fprintf(fp,"%s\n", cmd);
	fclose(fp);
}

/* Add some stuff to the environment of our child */
static void
childenv(char *label)
{
	Path	buf;

	sprintf(buf, "WILYLABEL=%s", label);
	(void)putenv(strdup(buf));
	sprintf(buf, "w=%s", label);
	(void)putenv(strdup(buf));
}

/* Execute 'cmd', which was selected in 'v'.
 * PRE: the first character of cmd is not 0 or whitespace.
 * Return 0 for success.
 */
int
ex_run(View*v, char *cmd)
{
	Path		dir, label;
	int		pout[2];
	int		perr[2];
	ulong	kout, kerr;
	int		pid;
	View		*vout=0;		/* vout's selection replaced by output */
	View		*vin=0;		/* vin's selection used as input */
	char		op;
	Data		*d;

	assert( *cmd && !isspace(*cmd) );
	
	/* Pipe operations are on the last selected view, or the
	 * body of last_selection if last_selection is a tag.
	 */
	op = cmd[0];
	if (op == '|' || op == '<' || op == '>') {
		View	*vpipe;
		cmd += 1 + strspn(cmd+1, whitespace);	/* elide leading whitespace */
		if (!strlen(cmd))
			return -1;		/* empty command */

		if( ! (vpipe = view_body(last_selection)) )
			return 1;

		if (op == '|' || op == '<')
			vout = vpipe;
		if (op == '|' || op == '>')
			 vin= vpipe;
	}

	/* fderr */
	if (pipe(perr) < 0) {
		diag(0, "pipe"); return 1;
	}
	if (!(kerr = estart(0, perr[0], 0))) {
		diag(0, "estart"); goto cleanerr;
	}

	/* fdout */
	if (vout) {
		if (pipe(pout) < 0) {
			diag(0, "pipe"); goto cleanerr;
		}
		if (!(kout = estart(0, pout[0], 0))) {
			diag(0, "pipe"); goto cleanout;
		}
	} else {
		pout[0] = perr[0];
		pout[1] = perr[1];
		kout = 0;
	}

	d = view_data(v);
	data_context(  d,dir);
	data_path(d, label);

	switch(pid=fork()) {
	case -1:	/* fork failed */
		goto cleanout;
	default:	/* parent */
		close(perr[1]);
		close(pout[1]);
		ex_parent(kerr, perr[0], kout, pout[0], dir, cmd, pid, vout);
		break;
	case 0:	/* child */
		close(perr[0]);
		close(pout[0]);
		ex_child(perr[1], pout[1], vin, dir, cmd, label);
		assert(false);	/* ex_child doesn't return */
	}
	return 0;

cleanout:
	estop(kerr);
	if (kout)
		estop(kout);
	if(vout) {
		close(pout[0]);
		close(pout[1]);
	}
cleanerr:
	close(perr[0]);
	close(perr[1]);

	return 1;
}

static char *
trim_and_add_space(char *cmd)
{
	char	*s, *word;

	word = salloc(strlen(cmd) + 2); /* trailing null, plus we add a space */
	strcpy(word, cmd);

	/* We want the first whitespace-delimited word, then a space */
	if (! (s = strpbrk(word,whitespace) ) ) {
		s = word + strlen(word);
	}
	*s++ = ' ';
	*s = '\0';
	return word;
}

static void
newerrorskey(ulong kerr, int fd, int pid, char *cmd, char *dir, View *v)
{
	Key	*k = key_new(kerr, fd, Kout);
	char	*s;

	k->pid = pid;
	strcpy(k->cmd, cmd);
	if( (s = strpbrk(k->cmd, whitespace)) )
		*s='\0';
	strcpy(k->dir, dir);
	k->first = true;
	k->v = v;
}

/* 	kerr, fderr: event key and fd for fderr
	krepl, fdrepl: event key and fd for stuff to replace selection
	dir: working directory for command
	cmd: the command
	vout:  for inplace replacement
 */
static void
ex_parent(ulong kerr, int fderr, ulong krepl, int fdrepl, 
		char *dir, char *cmd, int pid, View *vout)
{
	char *cmdword;

	cmdword = trim_and_add_space(cmd);
	newerrorskey(kerr, fderr, pid, cmd, dir, 0);
	if(vout) {
		assert(krepl);
		newerrorskey(krepl, fdrepl, pid, cmdword, dir, vout);
	}
	addrunning(cmdword);
	reap();
	free(cmdword);
}

/* Does not return.  Cannot use normal diag stuff: we're a separate
 * process.  'v' is used for fdin, if it is set.
*/
static void
ex_child(int fderr, int fdout, View *v, char *cwd, char *cmd, char *label)
{
	childfds(fderr, fdout, v);
	
	if(setsid()<0)			/* become process group leader */
		perror("setsid");

	if(chdir(cwd)){
		fprintf(stderr, "couldn't chdir(%s)\n", cwd);
		exit(1);
	}

	childenv(label);

	history(cmd);
	execl(shell, shell, "-c", cmd, 0);
	perror(shell);
	exit(1);
}

/* Use replacement 'fderr' and 'fdout'.
 * Get 'fdin' from 'v', or from '/dev/null'
 */
static void
childfds(int fderr, int fdout, View *v)
{
	int	j;
	int	fdin;

	/* redirect fdout and fderr */
	if (dup2(fderr, 2) < 0 || dup2(fdout, 1) < 0) {
		perror("dup2");
		exit(1);
	}

	if (v) {
		fdin = text_fd(view_text(v), view_getsel(v));		/* fd open to read current selection */
		if(fdin<0)
			exit(1);
	} else if ((fdin = open("/dev/null", O_RDONLY, 0)) < 0) {
		perror("open /dev/null");
		fdin = 0;
	}

	if (fdin)
		if (dup2(fdin, 0) < 0) {
			perror("input dup2");
			fdin = 0;
		}
	if (!fdin) {
		/* no need to panic... */
		if (close(0) < 0) {
			/* OK, panic. */
			perror("close fdin");
			exit(1);
		}
	}

	/* Don'v inherit any other open fds */
	for(j=3;j<FOPEN_MAX;j++)
		close(j);
}

/* Kill one of our external processes.  If 's' is set, use it,
 * otherwise just kill the first process we find.
 */
void
ex_kill(View *v, char *s)
{
	Key	*k = 0;
	const char *sep = " \t\n";

	if (s) {
		for (s = strtok(s, sep); s; s = strtok(NULL, sep)) {
			if ((k = key_findcmd(s))) {
				/* only output keys have a prog attached */
				assert(k->t == Kout); 
				if(kill(- k->pid, SIGKILL))
					diag(0, "kill %s", s);
			}
		}
	} else {
		while ((k = key_nextkey(k)))
			diag(0, "Kill %s", k->cmd);
	}
}

