/** 
 *
 * $Id: Frame.c,v 1.20 1996/04/30 01:36:25 miers Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/
 
static char rcsid[] = "$Id: Frame.c,v 1.20 1996/04/30 01:36:25 miers Exp $";

#include <LTconfig.h>
#include <Xm/XmP.h>
#include <Xm/DebugUtil.h>
#include <Xm/BaseClassP.h>
#include <Xm/FrameP.h>
#include <Xm/TransltnsP.h>
#include <stdio.h>

#define _XmMax(a,b) ((a) > (b) ? (a) : (b))
#define _XmMin(a,b) ((a) < (b) ? (a) : (b))

/*
 * Forward Declarations
 */
/* core */
static void class_initialize(void);
static void class_part_initialize(WidgetClass widget_class);
static void initialize(Widget request,
		       Widget new,
		       ArgList args,
		       Cardinal *num_args);
static void destroy(Widget w);
static void realize(Widget w,
		    XtValueMask *value_mask,
		    XSetWindowAttributes *attributes);
static void resize(Widget w);
static Boolean set_values(Widget current,
			  Widget request,
			  Widget new,
			  ArgList args,
			  Cardinal *num_args);
static void expose(Widget w, XEvent *event, Region region);
static XtGeometryResult query_geometry(Widget w,
				       XtWidgetGeometry *request,
				       XtWidgetGeometry *reply);
/* composite */
static XtGeometryResult geometry_manager(Widget w,
					 XtWidgetGeometry *request,
					 XtWidgetGeometry *reply);
static void change_managed(Widget w);
/* constraint */
static void constraint_initialize(Widget request,
				  Widget new,
				  ArgList args,
				  Cardinal *num_args);
static Boolean constraint_set_values(Widget current, 
				     Widget request,
				     Widget new,
				     ArgList args,
				     Cardinal *num_args);
/* helper */
static void _XmFrameLayout(Widget w, Boolean ParentResize,
			   Widget child, Boolean TestMode,
			   XtWidgetGeometry *childgeom);

/* actions */
static void Enter(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void GetFocus(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void Arm(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void Activate(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void ManagerGadgetDrag(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void ManagerParentActivate(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void ManagerParentCancel(Widget w, XEvent *event, String *params, Cardinal *num_params);

char _XmFrame_defaultTranslations[] = 
   "<EnterWindow>:        Enter()\n\
    <FocusIn>:            FocusIn()\n\
    <Btn1Down>:           Arm()\n\
    <Btn1Up>:             Activate()\n\
    <Btn2Down>:           ManagerGadgetDrag()\n\
    <Key>osfActivate:     ManagerParentActivate()\n\
    <Key>osfCancel:       ManagerParentCancel()\n\
    ~s ~m ~a <Key>Return: ManagerParentActivate()";

static XtActionsRec actions[] = {
    {"Enter",			Enter},
    {"FocusIn",			GetFocus},
    {"Arm",			Arm},
    {"Activate",		Activate},
    {"ManagerGadgetDrag",	ManagerGadgetDrag},
    {"ManagerParentActivate",	ManagerParentActivate},
    {"ManagerParentCancel",	ManagerParentCancel}
};

/*
 * resources for the frame class
 */
#define Offset(field) XtOffsetOf(XmFrameRec, frame.field)
static XtResource resources[] = {
    {
	XmNmarginWidth, XmCMarginWidth, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_width),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNmarginHeight, XmCMarginHeight, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_height),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNshadowType, XmCShadowType, XmRShadowType,
	sizeof(unsigned char), Offset(shadow_type),
	XmRImmediate, (XtPointer)((unsigned char)XmUNSPECIFIED)
	/* this was XmRCallProc, (XtPointer)_XmFrameShadowTypeDefault */
    },
    /* manager override */
    {
	XmNshadowThickness, XmCShadowThickness, XmRHorizontalDimension,
	sizeof(Dimension), XtOffsetOf(XmFrameRec, manager.shadow_thickness),
	XmRImmediate, (XtPointer)XmINVALID_DIMENSION
	/* this was XmRCallProc, (XtPointer)_XmFrameShadowThicknessDefault */
    },
};

static XmSyntheticResource syn_resources[] = {
    {
	XmNmarginWidth,
	sizeof(Dimension), Offset(margin_width),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginHeight,
	sizeof(Dimension), Offset(margin_height),
	_XmFromVerticalPixels, _XmToVerticalPixels
    }
};

/*
 * Constraint Resources for frame's children
 */
#undef Offset
#define Offset(field) XtOffsetOf(XmFrameConstraintRec, frame.field)

static XtResource frameConstraintResources[] = {
    {
	XmNchildType, XmCChildType, XmRChildType,
	sizeof(unsigned char), Offset(child_type),
	XtRImmediate, (XtPointer)XmFRAME_WORKAREA_CHILD
    },
    {
	XmNchildHorizontalAlignment, XmCChildHorizontalAlignment, XmRChildHorizontalAlignment,
	sizeof(unsigned char), Offset(child_h_alignment),
	XtRImmediate, (XtPointer)XmALIGNMENT_BEGINNING
    },
    {
	XmNchildVerticalAlignment, XmCChildVerticalAlignment, XmRChildVerticalAlignment,
	sizeof(unsigned char), Offset(child_v_alignment),
	XtRImmediate, (XtPointer)XmALIGNMENT_CENTER
    },
    {
	XmNchildHorizontalSpacing, XmCChildHorizontalSpacing, XmRHorizontalDimension,
	sizeof(Dimension), Offset(child_h_spacing),
	XmRImmediate, (XtPointer)XmINVALID_DIMENSION
	/* this was XmRCallProc, (XtPointer)_XmFrameHorizSpaceDefault */
    }
};

static XmSyntheticResource constraint_syn_resources[] = {
    {
	XmNchildHorizontalSpacing,
	sizeof(Dimension), Offset(child_h_spacing),
	_XmFromVerticalPixels, _XmToVerticalPixels
    }
};

static XmBaseClassExtRec _XmFrameCoreClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,                             
    /* version                   */ XmBaseClassExtVersion,
    /* size                      */ sizeof(XmBaseClassExtRec),
    /* initialize_prehook        */ NULL, /* FIXME */
    /* set_values_prehook        */ NULL, /* FIXME */
    /* initialize_posthook       */ NULL, /* FIXME */
    /* set_values_posthook       */ NULL, /* FIXME */
    /* secondary_object_class    */ NULL, /* FIXME */
    /* secondary_object_create   */ NULL, /* FIXME */
    /* get_secondary_resources   */ NULL, /* FIXME */
    /* fast_subclass             */ { 0 }, /* FIXME */
    /* get_values_prehook        */ NULL, /* FIXME */
    /* get_values_posthook       */ NULL, /* FIXME */
    /* class_part_init_prehook   */ NULL, /* FIXME */
    /* class_part_init_posthook  */ NULL, /* FIXME */
    /* ext_resources             */ NULL, /* FIXME */
    /* compiled_ext_resources    */ NULL, /* FIXME */
    /* num_ext_resources         */ 0, /* FIXME */
    /* use_sub_resources         */ FALSE, /* FIXME */
    /* widget_navigable          */ NULL, /* FIXME */
    /* focus_change              */ NULL, /* FIXME */
    /* wrapper_data              */ NULL
};

#undef Offset
static CompositeClassExtensionRec frameCompositeExt = {
    /* next_extension */  NULL,
    /* record_type    */  NULLQUARK,
    /* version        */  XtCompositeExtensionVersion,
    /* record_size    */  sizeof(CompositeClassExtensionRec),
    /* accepts_objects */ True,
#if XtSpecificationRelease >= 6
    /* allows_change_managed_set */ True
#endif
};

static XmManagerClassExtRec _XmFrameMClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,
    /* version                   */ XmManagerClassExtVersion,
    /* record_size               */ sizeof(XmManagerClassExtRec),
    /* traversal_children        */ NULL /* FIXME */
};

XmFrameClassRec xmFrameClassRec = {
    /* Core class part */
    {
	/* superclass            */ (WidgetClass) &xmManagerClassRec,
        /* class_name            */ "XmFrame",
	/* widget_size           */ sizeof(XmFrameRec),
	/* class_initialize      */ class_initialize,
	/* class_part_initialize */ class_part_initialize,
	/* class_inited          */ FALSE,
	/* initialize            */ initialize,
	/* initialize_hook       */ NULL,
	/* realize               */ realize,
	/* actions               */ actions,
	/* num_actions           */ XtNumber(actions),
	/* resources             */ resources,
	/* num_resources         */ XtNumber(resources),
	/* xrm_class             */ NULLQUARK,
	/* compress_motion       */ TRUE,
	/* compress_exposure     */ XtExposeCompressMultiple,
	/* compress_enterleave   */ TRUE,
	/* visible_interest      */ FALSE,
	/* destroy               */ destroy,
	/* resize                */ resize,
	/* expose                */ expose,
	/* set_values            */ set_values,
	/* set_values_hook       */ NULL,
	/* set_values_almost     */ XtInheritSetValuesAlmost,
	/* get_values_hook       */ NULL,
	/* accept_focus          */ NULL,
	/* version               */ XtVersion,
	/* callback offsets      */ NULL,
	/* tm_table              */ _XmFrame_defaultTranslations,
	/* query_geometry        */ query_geometry,
	/* display_accelerator   */ XtInheritDisplayAccelerator,
	/* extension             */ (XtPointer)&_XmFrameCoreClassExtRec
    },
    /* Composite class part */
    {
	/* geometry manager */ geometry_manager, 
        /* change_managed   */ change_managed, 
        /* insert_child     */ XtInheritInsertChild,
        /* delete_child     */ XtInheritDeleteChild,
        /* extension        */ (XtPointer)&frameCompositeExt,
    },
    /* Constraint class part */
    {
	/* subresources      */ frameConstraintResources,
        /* subresource_count */ XtNumber(frameConstraintResources), 
        /* constraint_size   */ sizeof(XmFrameConstraintRec),
        /* initialize        */ constraint_initialize,
        /* destroy           */ NULL,
        /* set_values        */ constraint_set_values,
        /* extension         */ NULL,  
    },
    /* XmManager class part */
    {
	/* translations                 */ XmInheritTranslations,
        /* syn_resources                */ syn_resources,
        /* num_syn_resources            */ XtNumber(syn_resources),
        /* syn_constraint_resources     */ constraint_syn_resources,
        /* num_syn_constraint_resources */ XtNumber(constraint_syn_resources),
        /* parent_process               */ XmInheritParentProcess,
	/* extension                    */ (XtPointer)&_XmFrameMClassExtRec
    },
    /* XmFrame Area part */
    {
	/* extension */ NULL,
    },
};

WidgetClass xmFrameWidgetClass = (WidgetClass)&xmFrameClassRec;

/* core methods */

static void 
class_initialize()
{
    _XmFrameCoreClassExtRec.record_type = XmQmotif;
}

static void
class_part_initialize(WidgetClass widget_class)
{
    CompositeClassExtension ext, *extptr;
    XmFrameWidgetClass fclass = (XmFrameWidgetClass)widget_class;

    extptr = (CompositeClassExtension*)_XmGetClassExtensionPtr((XmGenericClassExt*)&(fclass->composite_class.extension),
							       NULLQUARK);

    if (extptr == NULL || *extptr == NULL)
    {
	ext = (CompositeClassExtension) XtNew(CompositeClassExtensionRec);
	if (ext != NULL)
	{
	    ext->next_extension = fclass->composite_class.extension;
	    ext->record_type = NULLQUARK;
	    ext->version = XtCompositeExtensionVersion;
	    ext->record_size = sizeof(CompositeClassExtensionRec);
	    ext->accepts_objects = True;
#if XtSpecificationRelease >= 6
	    ext->allows_change_managed_set = True;
#endif
	    fclass->composite_class.extension = (XtPointer) ext;
	}
    }
    _XmFastSubclassInit(widget_class, XmFRAME_BIT);
}

static void
initialize(Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    Frame_TitleArea(new) = NULL;
    Frame_WorkArea(new) = NULL;

    if (Frame_ShadowType(new) == (unsigned char)XmUNSPECIFIED)
    {
	if (XtIsShell(XtParent(new)))
	    Frame_ShadowType(new) = XmSHADOW_ETCHED_OUT;
	else
	    Frame_ShadowType(new) = XmSHADOW_IN;
    }

    if (MGR_ShadowThickness(new) == XmINVALID_DIMENSION)
	MGR_ShadowThickness(new) = 2;

    _XmFrameLayout(new, True, NULL, False, NULL);
}

static void
destroy(Widget w)
{
}

static Boolean
set_values(Widget current,
	   Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    Boolean redisplay = False;

    if (Frame_MarginWidth(new) != Frame_MarginWidth(current) ||
	Frame_MarginHeight(new) != Frame_MarginHeight(current) ||
	Frame_ShadowType(new) != Frame_ShadowType(current))
	redisplay = True;

    if (redisplay)
	_XmFrameLayout(new, True, NULL, False, NULL);

    return redisplay;
}

static void
realize(Widget w, 
	XtValueMask *value_mask, 
	XSetWindowAttributes *attributes)
{
#define superclass (&xmManagerClassRec)
    (*superclass->core_class.realize)(w, value_mask, attributes);
#undef superclass

    _XmFrameLayout(w, True, NULL, False, NULL);

    XdbDebug(__FILE__, w, "Frame Realize\n");
}

static void
resize(Widget w)
{
    XdbDebug(__FILE__, w, "Frame resize (%d %d)\n", XtWidth(w), XtHeight(w));

    _XmFrameLayout(w, False, NULL, False, NULL);
}

static void
expose(Widget w,
       XEvent *event,
       Region region)
{
    Position topline_y = 0;

    if (Frame_TitleArea(w))
    {
	switch (FrameC_ChildVAlignment(Frame_TitleArea(w)))
	{
	case XmALIGNMENT_CENTER:
	    topline_y = XtHeight(Frame_TitleArea(w)) / 2;
	    break;
	case XmALIGNMENT_BASELINE_TOP:
	    /* change this to work with XmWidgetBaseline */
	    topline_y = XtHeight(w);
	    break;
	case XmALIGNMENT_WIDGET_TOP:
	    topline_y = XtHeight(Frame_TitleArea(w));
	    break;
	case XmALIGNMENT_WIDGET_BOTTOM:
	    topline_y = 0;
	    break;
	}
    }

    _XmDrawShadows(XtDisplay(w),
		   XtWindow(w),
		   MGR_TopShadowGC(w),
		   MGR_BottomShadowGC(w),
		   0, 
		   topline_y, 
		   XtWidth(w),
		   XtHeight(w) - topline_y,
		   MGR_ShadowThickness(w),
		   Frame_ShadowType(w));

    /* redisplay our gadget children -- if any */
    _XmRedisplayGadgets(w, event, region);
}

static XtGeometryResult
query_geometry(Widget w,
	       XtWidgetGeometry *request,
	       XtWidgetGeometry *reply)
{
    XtWidgetGeometry want;

    _XmFrameLayout(w, True, w, True, &want);

    *reply = want;
    reply->width = want.width;
    reply->height = want.height;
    reply->request_mode = CWWidth|CWHeight;

    if ((request->request_mode & (CWWidth|CWHeight)) == (CWWidth|CWHeight) &&
	reply->width == request->width && reply->height == request->height)
	return XtGeometryYes;
    else if (((request->request_mode & CWWidth) && (request->width == XtWidth(w))) 
             && ((request->request_mode & CWHeight) && (request->height == XtHeight(w))))
	return XtGeometryNo;
    else
	return XtGeometryAlmost;
}

static XtGeometryResult
geometry_manager(Widget w,
		 XtWidgetGeometry *request,
		 XtWidgetGeometry *reply)
{
    XtWidgetGeometry want;

    if (FrameC_ChildType(w) != XmFRAME_WORKAREA_CHILD &&
	FrameC_ChildType(w) != XmFRAME_TITLE_CHILD)
	return XtGeometryNo;

    want = *request;

    /* FIXME -- there is an implicit assumption that the child has set
     * CWWidth and CWHeight in the request_mode */
    _XmFrameLayout(XtParent(w), True, w, True, &want);

    if (request->request_mode & (CWX|CWY))
	return XtGeometryNo;

    *reply = want;
    if ((request->request_mode & (CWWidth|CWHeight)) == (CWWidth|CWHeight) &&
	reply->width == request->width && reply->height == request->height)
	return XtGeometryYes;
    else if (request->width == XtWidth(w) && request->height == XtHeight(w))
	return XtGeometryNo;
    else
	return XtGeometryAlmost;
}

static void
change_managed(Widget w)
{
    int i;
    Widget child;

    Frame_TitleArea(w) = NULL;
    Frame_WorkArea(w) = NULL;

    for (i=0; i < ((XmFrameWidget)w)->composite.num_children; i++)
    {
	child = ((XmFrameWidget)w)->composite.children[i];

	if (!XtIsManaged(child))
	    continue;
	else if (FrameC_ChildType(child) != XmFRAME_GENERIC_CHILD)
	{
	    if (FrameC_ChildType(child) == XmFRAME_TITLE_CHILD)
	    {
		if (!Frame_TitleArea(w))
		    Frame_TitleArea(w) = child;
	    }
	    else if (FrameC_ChildType(child) == XmFRAME_WORKAREA_CHILD)
	    {
		if (!Frame_WorkArea(w))
		    Frame_WorkArea(w) = child;
	    }
	    else {
		XdbDebug(__FILE__, w, "XmFrame illegal child type resource\n");
	    }
	}
    }
 
    _XmFrameLayout(w, True, NULL, False, NULL);

    _XmNavigChangeManaged(w);
}

/* Constraint methods */

static void 
constraint_initialize(Widget request, 
		      Widget new, 
		      ArgList args, 
		      Cardinal *num_args)
{
    if (FrameC_ChildHSpacing(new) == XmINVALID_DIMENSION)
	FrameC_ChildHSpacing(new) = Frame_MarginWidth(XtParent(new));
}

static Boolean
constraint_set_values(Widget current, 
		      Widget request, 
		      Widget new, 
		      ArgList args, 
		      Cardinal *num_args)
{
    Boolean redisplay = False;

    if (FrameC_ChildType(current) != FrameC_ChildType(new) ||
	FrameC_ChildHAlignment(current) != FrameC_ChildHAlignment(new) ||
	FrameC_ChildHSpacing(current) != FrameC_ChildHSpacing(new) ||
	FrameC_ChildVAlignment(current) != FrameC_ChildVAlignment(new))
	redisplay = True;

    if (redisplay)
	_XmFrameLayout(XtParent(new), True, NULL, False, NULL);

    return redisplay;
}

static void
_XmFrameLayout(Widget w, Boolean ParentResize,
	       Widget child, Boolean TestMode, XtWidgetGeometry *childgeom)
{
    int title_x = 0, title_y = 0;
    int workarea_x = 0, workarea_y = 0;
    Dimension curw, curh;
    XtWidgetGeometry geo;
    XtGeometryResult result;

    XdbDebug(__FILE__, w, "_XmFrameLayout %s %s\n", ParentResize ? "(ParentResize)" : "",
		TestMode ? "(TestMode)" : "");

    curw = curh = 0;

    if (Frame_TitleArea(w))
    {
	if (TestMode && child == Frame_TitleArea(w)) {
	    curh += childgeom->height;
	    curw = childgeom->width;
	}
	else {
	    /* FIXME -- when all children answer right, the curh/curw
	     * assignment should be changed to use the geo results */
	    XtQueryGeometry(Frame_TitleArea(w), NULL, &geo);
	    curh += XtHeight(Frame_TitleArea(w));
	    curw = XtWidth(Frame_TitleArea(w));
	}
    }

    if (Frame_WorkArea(w))
    {
	if (TestMode && child == Frame_WorkArea(w)) {
	    curh += childgeom->height;
	    curw = childgeom->width;
	}
	else {
	    /* FIXME -- when all children answer right, the curh/curw
	     * assignment should be changed to use the geo results */
	    XtQueryGeometry(Frame_WorkArea(w), NULL, &geo);
	    curh += XtHeight(Frame_WorkArea(w));
	    curw = _XmMax(curw, XtWidth(Frame_WorkArea(w)));
	}
    }

    curh += 2 * Frame_MarginHeight(w) + 2 * MGR_ShadowThickness(w);
    curw += 2 * Frame_MarginWidth(w) + 2 * MGR_ShadowThickness(w);

    /* this condition is met in the query_geometry call */
    if (TestMode && w == child) {
	childgeom->width = curw;
	childgeom->height = curh;
	return;
    }

    /* FIXME -- this needs to be beefed up to properly set child values
     * when XtGeometryYes */
    if (TestMode) {
	geo.width = curw;
	geo.height = curh;
	geo.request_mode = XtCWQueryOnly|CWWidth|CWHeight;
	result = XtMakeGeometryRequest(w, &geo, &geo);

	switch (result) {
	case XtGeometryNo:
	    childgeom->width = XtWidth(child);
	    childgeom->height = XtHeight(child);
	    /* go back, no layout change */
	    return;

	case XtGeometryYes:
	    /* implicit assumption:  since we don't override the values in
	     * childgeom, Xt should take care of the layout changes for us.
	     * So we continue with the layout method, attempting to resize
	     * ourselves, and setting the fields appropriately */
	    break;

	case XtGeometryAlmost:
	    /* we need to compute the layout based on the size our parent said
	     * we could attain. */
	    if (Frame_TitleArea(w) && child == Frame_TitleArea(w))
	    {
		childgeom->width =
			 _XmMin(XtWidth(Frame_TitleArea(w)),
				curw - 2 * MGR_ShadowThickness(w)
				- 2 * Frame_MarginWidth(w)
				- 2 * FrameC_ChildHSpacing(Frame_TitleArea(w)));
	    }
	    if (Frame_WorkArea(w) && child == Frame_WorkArea(w))
	    {
		childgeom->width = curw - 2 * MGR_ShadowThickness(w)
				       - 2 * Frame_MarginWidth(w);
		childgeom->height = curh - XtY(Frame_WorkArea(w))
				        - Frame_MarginHeight(w)
				        - MGR_ShadowThickness(w);
	    }
	    return;

	/* shouldn't happen */
	case XtGeometryDone:
	default:
	    _XmWarning(w, "Parent returned unexpected geometry result.");
	    return;
	}
    }

    if (ParentResize) {
	do {
	    result = XtMakeResizeRequest(w, curw, curh, &curw, &curh);
	} while (result == XtGeometryAlmost);

	if (result == XtGeometryYes) {
	    XtWidth(w) = curw;
	    XtHeight(w) = curh;
	}
	if (result == XtGeometryNo) {
	    curw = XtWidth(w);
	    curh = XtHeight(w);
	}
	XdbDebug(__FILE__, NULL, "_XmFrameLayout %s (%d %d) => %s\n",
		 XtName(w), curw, curh, XdbGeometryResult2String(result));
    }
    else {
	curw = XtWidth(w);
	curh = XtHeight(w);
    }

    workarea_y = MGR_ShadowThickness(w);

    if (Frame_TitleArea(w))
    {
	switch (FrameC_ChildVAlignment(Frame_TitleArea(w)))
	{
	case XmALIGNMENT_CENTER:
	    title_y = 0;
	    workarea_y = XtHeight(Frame_TitleArea(w)) + Frame_MarginHeight(w);
	    break;
	case XmALIGNMENT_BASELINE_BOTTOM:
	    /* fill this out */
	    break;
	case XmALIGNMENT_BASELINE_TOP:
	    title_y = 0;
	    workarea_y = MGR_ShadowThickness(w)
			 + Frame_MarginHeight(w);
	    break;
	case XmALIGNMENT_WIDGET_TOP:
	    title_y = 0;
	    workarea_y = XtHeight(Frame_TitleArea(w)) + MGR_ShadowThickness(w)
			 + Frame_MarginHeight(w);
	    break;
	case XmALIGNMENT_WIDGET_BOTTOM:
	    title_y = MGR_ShadowThickness(w) + Frame_MarginHeight(w);
	    workarea_y = MGR_ShadowThickness(w) + Frame_MarginHeight(w);
	    break;
	}

	switch (FrameC_ChildHAlignment(Frame_TitleArea(w)))
	{
	case XmALIGNMENT_BEGINNING:
	    title_x = 20;
	    break;
	case XmALIGNMENT_END:
	    title_x = curw - XtWidth(Frame_TitleArea(w)) - 20;
	    break;
	case XmALIGNMENT_CENTER:
	    title_x = (int)((float)(curw/2)
			  - (float)(XtWidth(Frame_TitleArea(w))/2));
	    break;
	}

	if (TestMode && Frame_TitleArea(w) &&
	    child == Frame_TitleArea(w)) {
	    XtX(Frame_TitleArea(w)) = title_x;
	    XtY(Frame_TitleArea(w)) = title_y;
	    XtWidth(Frame_TitleArea(w)) =
		 _XmMin(XtWidth(Frame_TitleArea(w)),
			curw - 2 * MGR_ShadowThickness(w)
			- 2 * Frame_MarginWidth(w)
			- 2 * FrameC_ChildHSpacing(Frame_TitleArea(w)));
	}
	else
	    _XmConfigureObject(Frame_TitleArea(w),
			       title_x, title_y,
			       _XmMin(XtWidth(Frame_TitleArea(w)),
					curw - 2 * MGR_ShadowThickness(w)
					- 2 * Frame_MarginWidth(w)
					- 2 * FrameC_ChildHSpacing(Frame_TitleArea(w))),
			       XtHeight(Frame_TitleArea(w)),
			       XtBorderWidth(Frame_TitleArea(w)));
    }
	
    if (Frame_WorkArea(w))
    {
	workarea_x = MGR_ShadowThickness(w) + Frame_MarginWidth(w);

	if (TestMode && Frame_WorkArea(w) && child == Frame_WorkArea(w)) {
	    XtX(Frame_WorkArea(w)) = workarea_x;
	    XtY(Frame_WorkArea(w)) = workarea_y;
	    XtWidth(Frame_WorkArea(w)) = curw - 2 * MGR_ShadowThickness(w)
					       - 2 * Frame_MarginWidth(w);
	    XtHeight(Frame_WorkArea(w)) = curh - XtY(Frame_WorkArea(w))
					        - Frame_MarginHeight(w)
					        - MGR_ShadowThickness(w);
	}
	else
	    _XmConfigureObject(Frame_WorkArea(w),
			       workarea_x, workarea_y,
			       curw - 2 * MGR_ShadowThickness(w)
				       - 2 * Frame_MarginWidth(w),
			       curh - XtY(Frame_WorkArea(w))
				        - Frame_MarginHeight(w)
				        - MGR_ShadowThickness(w),
			       XtBorderWidth(Frame_WorkArea(w)));
    }
}

static void
Enter(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
}

static void
GetFocus(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
}

static void
Arm(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
}

static void
Activate(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
}

static void
ManagerGadgetDrag(Widget w, XEvent *event,
		  String *params, Cardinal *num_params)
{
}

static void
ManagerParentActivate(Widget w, XEvent *event,
		      String *params, Cardinal *num_params)
{
}

static void
ManagerParentCancel(Widget w, XEvent *event,
		    String *params, Cardinal *num_params)
{
}

Widget 
XmCreateFrame(Widget parent,
	      char *name,
	      Arg *arglist,
	      Cardinal argCount)
{
    return XtCreateManagedWidget(name,
		                 xmFrameWidgetClass,
				 parent,
				 arglist,
				 argCount);
}
