/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** storeRef.h
*/

# ifndef STOREREF_H
# define STOREREF_H

/*
** note: forwardTypes defines sRef
*/

/*
** kinds of storage references 
*/

typedef enum {
  SR_NOTHING,
  SR_INTERNAL,
  SR_SPECSTATE,
  SR_SYSTEM,
} speckind;
    
typedef enum { 
  SK_PARAM, 
  SK_ARRAYFETCH, 
  SK_FIELD,
  SK_PTR, 
  SK_ADR, 
  SK_CONST,
  SK_CVAR, 
  SK_UNCONSTRAINED,
  SK_OBJECT, 
  SK_CONJ, 
  SK_EXTERNAL,
  SK_DERIVED,
  SK_NEW, 
  SK_TYPE, 
  SK_RESULT,
  SK_SPECIAL,
  SK_UNKNOWN 
} skind;

typedef struct _cref
{
  int    lexlevel;
  usymId index;
} *cref;

typedef struct _ainfo
{
  /*@exposed@*/ /*@notnull@*/ sRef arr;
  bool indknown;
  int  ind;
} *ainfo;

typedef struct _finfo
{
  /*@exposed@*/ /*@notnull@*/  sRef rec;
  /*@observer@*/ cstring field;
} *finfo ;

typedef struct _cjinfo
{
  /*@exposed@*/ /*@notnull@*/ sRef a;
  /*@exposed@*/ /*@notnull@*/ sRef b;
} *cjinfo ;

typedef union _sinfo
{
  /*@only@*/ cref     cvar;
             int      paramno;
  /*@only@*/ ainfo    arrayfetch;
  /*@only@*/ finfo    field;
             ctype    object;
  /*@observer@*/ cstring fname; /* unconstrained, new */
  /*@exposed@*/ /*@notnull@*/ sRef     ref;
  /*@only@*/ cjinfo   conj;
             speckind   spec;
} *sinfo;
  
typedef /*@null@*/ struct _alinfo
{
  /*@only@*/ fileloc  loc;
  /*@observer@*/ sRef ref;
  /*@observer@*/ uentry ue;
} *alinfo;

struct _sRef
{
  /* does it need to be inside parens (macros) */
  bool safe      BOOLBITS; 

  /* has it been modified */
  bool modified  BOOLBITS;

  skind kind;
  ctype type;

  sstate defstate;
  nstate nullstate;

  alkind aliaskind;
  alkind oaliaskind;
  
  exkind expkind;     /* exposed, observer, normal */
  exkind oexpkind;
  
  /* where it becomes observer/exposed */
  /*@null@*/ /*@only@*/ alinfo expinfo;  

  /* where it changed alias kind */
  /*@null@*/ /*@only@*/ alinfo aliasinfo;

  /* where it is defined/allocated */
  /*@null@*/ /*@only@*/ alinfo definfo;  

  /* where it changes null state */
  /*@null@*/ /*@only@*/ alinfo nullinfo;  

  /*@only@*/ /*@relnull@*/ sinfo info;
  
  /* stores fields for structs, elements for arrays, derefs for pointers */
  /*@only@*/ sRefSet deriv; 
} ;

extern bool sRef_perhapsNull (sRef s);
extern bool sRef_possiblyNull (sRef s);
extern bool sRef_definitelyNull (sRef s);

extern void sRef_setNullUnknown (sRef s, fileloc loc);
extern void sRef_setNotNull (sRef s, fileloc loc);
extern void sRef_setNullState (sRef s, nstate n, fileloc loc);
extern void sRef_setNullStateInnerComplete (sRef s, nstate n, fileloc loc);
extern void sRef_setPosNull (sRef s, fileloc loc);
extern void sRef_setDefNull (sRef s, fileloc loc);

extern /*@truenull@*/ bool sRef_isInvalid (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isValid (sRef s) /*@*/ ;

/*@constant null sRef sRef_undefined; @*/
# define sRef_undefined    ((sRef) NULL)
# define sRef_isInvalid(s) ((s) == NULL)
# define sRef_isValid(s)   ((s) != NULL)

extern bool sRef_isRecursiveField (sRef s) /*@*/ ;
extern void sRef_copyRealDerivedComplete (sRef s1, sRef s2) /*@modifies s1@*/ ;
extern nstate sRef_getNullState (/*@sef@*/ sRef s) /*@*/ ;
extern bool sRef_isNotNull (sRef s) /*@*/ ;

# define sRef_getNullState(s)    (sRef_isValid(s) ? (s)->nullstate : NS_UNKNOWN)
extern bool sRef_isDefinitelyNull (sRef s) /*@*/ ;
						      
extern /*@falsenull@*/ bool sRef_isLocalVar (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isNSLocalVar (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isRealLocalVar (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isLocalParamVar (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isKnown (/*@sef@*/ sRef s) /*@*/ ;

extern bool sRef_hasLastReference (sRef s) /*@*/ ;
# define sRef_hasLastReference(s) (sRef_hasAliasInfoRef (s))

# define sRef_isKnown(s)          (sRef_isValid(s) && (s)->kind != SK_UNKNOWN)

extern bool sRef_isMeaningful (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isMeaningful(s)     (sRef_isValid(s) && sRef_isKnown(s) \
				   && (s)->kind != SK_NEW && (s)->kind != SK_TYPE)
extern bool sRef_isNew (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isNew(s)            (sRef_isValid(s) && (s)->kind == SK_NEW)

extern bool sRef_isType (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isType(s)           (sRef_isValid(s) && (s)->kind == SK_TYPE)

extern bool sRef_isSafe (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isSafe(s)           (sRef_isValid(s) && (s)->safe)

extern bool sRef_isUnsafe (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isUnsafe(s)         (sRef_isValid(s) && !(s)->safe)

extern void sRef_clearAliasKind (/*@sef@*/ sRef s) /*@modifies s@*/ ;
# define sRef_clearAliasKind(s)   (sRef_isValid(s) ? (s)->aliaskind = AK_UNKNOWN : AK_ERROR)

extern bool sRef_stateKnown (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_stateKnown(s)       (sRef_isValid(s) && (s)->defstate != SS_UNKNOWN)

extern alkind sRef_getAliasKind (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_getAliasKind(s)     (sRef_isValid(s) ? (s)->aliaskind : AK_ERROR)

extern alkind sRef_getOrigAliasKind (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_getOrigAliasKind(s) (sRef_isValid(s) ? (s)->oaliaskind : AK_ERROR)

extern /*@falsenull@*/ bool sRef_isConj (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isConj(s)           (sRef_isValid(s) && (s)->kind == SK_CONJ)

extern /*@exposed@*/ sRef sRef_buildArrow (sRef s, /*@dependent@*/ cstring f);
extern /*@exposed@*/ sRef sRef_makeArrow (sRef s, /*@dependent@*/ cstring f);

extern bool sRef_isAllocIndexRef (sRef s) /*@*/ ;
extern void sRef_setAliasKind (sRef s, alkind ak, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setPdefined (sRef s, fileloc loc) /*@modifies s@*/ ;

extern /*@unused@*/ bool sRef_hasDerived (sRef s) /*@*/ ;
extern void sRef_clearDerived (sRef s);
extern void sRef_clearDerivedComplete (sRef s);
extern /*@exposed@*/ sRef sRef_getBaseSafe (sRef s);

extern /*@observer@*/ sRefSet sRef_derivedFields (/*@dependent@*/ sRef rec) /*@*/ ;
extern bool sRef_sameName (sRef s1, sRef s2) /*@*/ ;
extern bool sRef_isDirectParam (sRef s) /*@*/ ;
extern /*@exposed@*/ sRef sRef_makeAnyArrayFetch (/*@exposed@*/ sRef arr);
extern bool sRef_isUnknownArrayFetch (sRef arr) /*@*/ ;

extern void sRef_setPartialDefinedComplete (sRef s, fileloc loc);
extern bool sRef_isMacroParamRef (sRef s) /*@*/ ;

extern void sRef_destroyMod (void) /*@modifies internalState@*/ ;

extern bool sRef_deepPred (bool (predf) (sRef), sRef s);

extern bool sRef_aliasCompleteSimplePred (bool (predf) (sRef), sRef s);

extern void sRef_clearExKindComplete (sRef s, fileloc loc);

extern /*@falsenull@*/ bool sRef_isKindSpecial (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isKindSpecial(s) (sRef_isValid (s) && (s)->kind == SK_SPECIAL)

extern /*@observer@*/ cstring sRef_nullMessage (sRef s) /*@*/ ;

extern bool sRef_isSystemState (sRef s) /*@*/ ;
extern bool sRef_isInternalState (sRef s) /*@*/ ;
extern bool sRef_isResult (sRef s) /*@*/ ;
extern bool sRef_isSpecInternalState (sRef s) /*@*/ ;
extern bool sRef_isSpecState (sRef s) /*@*/ ;
extern bool sRef_isNothing (sRef s) /*@*/ ;

extern bool sRef_isGlobal (sRef s) /*@*/ ;
extern bool sRef_isReference (sRef s) /*@*/ ;

extern ctype sRef_deriveType (sRef s, uentryList cl) /*@*/ ;
extern ctype sRef_getType (sRef s) /*@*/ ;

extern /*@falsenull@*/ bool sRef_isAddress (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isArrayFetch (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isConst (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isCvar (sRef s) /*@*/ ; 
extern /*@falsenull@*/ bool sRef_isField (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isParam (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isPointer (sRef s) /*@*/ ;

extern void sRef_setType (sRef s, ctype ct);
extern void sRef_mergeNullState (sRef s, nstate n);
extern void sRef_setLastReference (sRef s, sRef ref, fileloc loc);
extern bool sRef_canModify (sRef s, sRefSet sl) /*@modifies s@*/ ;
extern bool sRef_canModifyVal (sRef s, sRefSet sl) /*@modifies s@*/ ;
extern bool sRef_isIReference (sRef s) /*@*/ ;
extern bool sRef_isIndexKnown (sRef arr) /*@*/ ;
extern bool sRef_isModified (sRef s) /*@*/ ;

extern bool sRef_isExternallyVisible (sRef s) /*@*/ ;
extern bool sRef_realSame (sRef s1, sRef s2) /*@*/ ;
extern bool sRef_same (sRef s1, sRef s2) /*@*/ ;
extern bool sRef_similar (sRef s1, sRef s2) /*@*/ ;
extern /*@observer@*/ cstring sRef_getField (sRef rec) /*@*/ ;
extern /*@only@*/ cstring sRef_unparse (sRef s) /*@*/ ;
extern /*@observer@*/ cstring sRef_stateVerb (sRef s) /*@*/ ;
extern /*@observer@*/ cstring sRef_stateAltVerb (sRef s) /*@*/ ;
extern /*@only@*/ cstring sRef_unparseOpt (sRef s) /*@*/ ;
extern /*@only@*/ cstring sRef_unparseDebug (sRef s) /*@*/ ;
extern void sRef_killComplete (sRef s, fileloc loc) /*@modifies s@*/ ;
extern int sRef_getIndex (sRef arr) /*@*/ ;
extern /*@dependent@*/ sRef sRef_fixOuterRef (/*@returned@*/ sRef s);
extern void sRef_setDefinedComplete (sRef s, fileloc loc);
extern void sRef_setDefinedNCComplete (sRef s, fileloc loc);
extern int sRef_getParam (sRef s) /*@*/ ;
extern int sRef_lexLevel (sRef s) /*@*/ ;
extern void sRef_setOrigAliasKind (sRef s, alkind kind);
extern /*@exposed@*/ sRef 
  sRef_fixBase (/*@returned@*/ sRef s, /*@returned@*/ sRef base)
  /*@modifies s, base@*/ ;

extern void sRef_showNotReallyDefined (sRef s) /*@modifies stderr@*/ ;

extern void sRef_enterFunctionScope (void);
extern void sRef_setGlobalScope (void);
extern void sRef_exitFunctionScope (void);
extern void sRef_clearGlobalScopeSafe (void);
extern void sRef_setGlobalScopeSafe (void);

extern /*@notnull@*/ /*@exposed@*/ sRef 
  sRef_buildArrayFetch (/*@exposed@*/ sRef arr);
extern /*@notnull@*/ /*@exposed@*/ sRef sRef_buildArrayFetchKnown (/*@exposed@*/ sRef arr, int i);
extern /*@exposed@*/ sRef
  sRef_buildField (sRef rec, /*@dependent@*/ cstring f) 
  /*@modifies rec@*/ ;
extern /*@exposed@*/ sRef sRef_buildPointer (/*@exposed@*/ sRef s) 
  /*@modifies s@*/ ;

extern /*@exposed@*/ sRef sRef_makeAddress (/*@exposed@*/ sRef s);
extern /*@notnull@*/ /*@dependent@*/ sRef sRef_makeUnconstrained (/*@exposed@*/ cstring) /*@*/ ;

extern /*@falsenull@*/ bool sRef_isUnconstrained (sRef s) /*@*/ ;

extern /*@observer@*/ cstring sRef_unconstrainedName (sRef s) /*@*/ ;

extern /*@notnull@*/ /*@exposed@*/ sRef sRef_makeArrayFetch (sRef arr) /*@*/ ;
extern /*@notnull@*/ /*@exposed@*/ sRef
  sRef_makeArrayFetchKnown (sRef arr, int i);
extern /*@notnull@*/ /*@dependent@*/ sRef
  sRef_makeConj (/*@exposed@*/ /*@returned@*/ sRef a, /*@exposed@*/ sRef b);
extern /*@notnull@*/ /*@dependent@*/ sRef
  sRef_makeCvar (int level, usymId index, ctype ct);
extern /*@notnull@*/ /*@dependent@*/ sRef
  sRef_makeConst (ctype ct);
extern /*@exposed@*/ sRef
  sRef_makeField (sRef rec, /*@dependent@*/ cstring f);
extern /*@notnull@*/ /*@dependent@*/ sRef 
  sRef_makeGlobal (usymId uid, ctype ct);
extern /*@exposed@*/ sRef
  sRef_makeNCField (sRef rec, /*@dependent@*/ cstring f) /*@*/ ;
extern void sRef_maybeKill (sRef s, fileloc loc) /*@modifies s@*/ ;
extern /*@unused@*/ /*@notnull@*/ /*@dependent@*/ sRef 
  sRef_makeObject (ctype c) /*@*/ ;
extern /*@notnull@*/ /*@dependent@*/ sRef sRef_makeType (ctype c) /*@*/ ;
extern /*@notnull@*/ /*@dependent@*/ sRef sRef_makeParam (int l, ctype ct) /*@*/ ;
extern /*@exposed@*/ sRef sRef_makePointer (sRef s) /*@modifies s@*/ ;
extern void sRef_makeSafe (sRef s) /*@modifies s@*/ ;
extern void sRef_makeUnsafe (sRef s) /*@modifies s@*/ ;
extern sRef sRef_makeUnknown (void) /*@*/ ;

extern sRef sRef_makeNothing (void) /*@*/ ;
extern sRef sRef_makeInternalState (void) /*@*/ ;
extern sRef sRef_makeSpecState (void) /*@*/ ;
extern sRef sRef_makeSystemState (void) /*@*/ ;

extern /*@notnull@*/ sRef sRef_makeResult (void) /*@*/ ;
extern /*@exposed@*/ sRef 
  sRef_fixResultType (/*@returned@*/ sRef s, ctype typ, uentry ue)
  /*@modifies s@*/;

extern void sRef_setParamNo (sRef s, int l) /*@modifies s;@*/;

extern /*@notnull@*/ sRef 
  sRef_makeNew (ctype ct, sRef sr, /*@exposed@*/ cstring name);

extern usymId sRef_getScopeIndex (sRef s) /*@*/ ;
extern /*@exposed@*/ uentry sRef_getBaseUentry (sRef s);

extern /*@exposed@*/ sRef 
  sRef_fixBaseParam (/*@returned@*/ sRef s, exprNodeList args)
  /*@modifies s@*/ ;

extern bool sRef_isUnionField (sRef s);
extern void sRef_setModified (sRef s);

extern void sRef_resetState (sRef s);
extern void sRef_resetStateComplete (sRef s);

extern void sRef_storeState (sRef s);
extern /*@exposed@*/ sRef sRef_getBase (sRef t) /*@*/ ;
extern /*@exposed@*/ sRef sRef_getRootBase (sRef s) /*@*/ ;

extern /*@observer@*/ uentry sRef_getUentry (sRef s);

extern cstring sRef_dump (sRef s) /*@*/ ;
extern cstring sRef_dumpGlobal (sRef s) /*@*/ ;
extern /*@exposed@*/ sRef sRef_undump (char **c) /*@modifies *c@*/ ;
extern /*@exposed@*/ sRef sRef_undumpGlobal (char **c) /*@modifies *c@*/ ;

extern /*@only@*/ sRef sRef_saveCopy (sRef s);
extern /*@dependent@*/ sRef sRef_copy (sRef s);

extern cstring sRef_unparseState (sRef s) /*@*/ ;
extern ynm sRef_isWriteable (sRef s) /*@*/ ;
extern ynm sRef_isReadable (sRef s) /*@*/ ;
extern bool sRef_isStrictReadable (sRef s) /*@*/ ;
extern bool sRef_hasNoStorage (sRef s) /*@*/ ;
extern void sRef_showExpInfo (sRef s) /*@modifies stderr*/ ;
extern void sRef_setDefined (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setUndefined (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setOnly (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setDependent (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setOwned (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setKept (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setKeptComplete (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setFresh (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setShared (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_showAliasInfo (sRef s) /*@modifies stderr@*/ ;
extern void sRef_showNullInfo (sRef s) /*@modifies stderr@*/ ;
extern void sRef_showStateInfo (sRef s) /*@modifies stderr@*/ ;
extern void sRef_setStateFromType (sRef s, ctype ct) /*@modifies s@*/ ;
extern void sRef_kill (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setAllocated (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setAllocatedShallowComplete (sRef s, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setAllocatedComplete (sRef s, fileloc loc) /*@modifies s@*/ ;
extern /*@only@*/ cstring sRef_unparseKindNamePlain (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isRealGlobal(sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isFileStatic (sRef s) /*@*/ ;
extern int sRef_getScope (sRef s) /*@*/ ;
extern /*@observer@*/ cstring sRef_getScopeName (sRef s) /*@*/ ;

/* must be real function (passed as function param) */
extern /*@falsenull@*/ bool sRef_isDead (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isDeadStorage (sRef s) /*@*/ ;
extern bool sRef_isStateLive (sRef s) /*@*/ ;
extern /*@falsenull@*/ bool sRef_isPossiblyDead (sRef s) /*@*/ ;
extern /*@truenull@*/ bool sRef_isStateUndefined (sRef s) /*@*/ ;
extern bool sRef_isUnuseable (sRef s) /*@*/ ;

extern /*@exposed@*/ sRef sRef_constructDeref (sRef t) 
   /*@modifies t@*/ ;

extern /*@exposed@*/ sRef sRef_constructDeadDeref (sRef t) 
   /*@modifies t@*/ ;

extern bool sRef_isJustAllocated (sRef s) /*@*/ ;

extern /*@falsenull@*/ bool sRef_isAllocated (sRef s) /*@*/ ;

extern bool sRef_isUndefGlob (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isUndefGlob(s) \
    ((sRef_isValid(s)) \
     && ((s)->defstate == SS_UNDEFGLOB || (s)->defstate == SS_UNDEFKILLED))

extern bool sRef_isKilledGlob (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isKilledGlob(s) \
    ((sRef_isValid(s)) \
     && ((s)->defstate == SS_KILLED || (s)->defstate == SS_UNDEFKILLED))

extern bool sRef_isRelDef (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isRelDef(s) \
  ((sRef_isValid(s)) && ((s)->defstate == SS_RELDEF))

extern bool sRef_isPartial (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isPartial(s) \
  ((sRef_isValid(s)) && ((s)->defstate == SS_PARTIAL))

extern bool sRef_isStateSpecial (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isStateSpecial(s) \
  ((sRef_isValid(s)) && ((s)->defstate == SS_SPECIAL))

extern bool sRef_isStateDefined (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isStateDefined(s) \
  ((sRef_isValid(s)) && (((s)->defstate == SS_DEFINED) \
			 || (s)->defstate == SS_RELDEF))

extern bool sRef_isAnyDefined (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isAnyDefined(s)   ((sRef_isValid(s)) && \
				 (((s)->defstate == SS_DEFINED) \
				  || ((s)->defstate == SS_RELDEF) \
				  || ((s)->defstate == SS_PARTIAL)))

extern /*@falsenull@*/ bool sRef_isPdefined (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isPdefined(s) \
  ((sRef_isValid(s)) && ((s)->defstate == SS_PDEFINED))

extern bool sRef_isReallyDefined (sRef s) /*@*/ ;

extern bool sRef_isStateUnknown (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isStateUnknown(s) \
  ((sRef_isValid(s)) && ((s)->defstate == SS_UNKNOWN))

extern /*@falsenull@*/ bool sRef_isRefCounted (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isRefCounted(s) \
  ((sRef_isValid(s)) && ((s)->aliaskind == AK_REFCOUNTED))

extern /*@falsenull@*/ bool sRef_isNewRef (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isNewRef(s) \
  ((sRef_isValid(s)) && ((s)->aliaskind == AK_NEWREF))

extern /*@falsenull@*/ bool sRef_isKillRef (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isKillRef(s) \
  ((sRef_isValid(s)) && ((s)->aliaskind == AK_KILLREF))

extern bool sRef_isOnly (sRef s) /*@*/ ;
extern bool sRef_isDependent (sRef s) /*@*/ ;
extern bool sRef_isOwned (/*@sef@*/ sRef s) /*@*/ ;
extern bool sRef_isKeep (/*@sef@*/ sRef s) /*@*/ ;

extern bool sRef_isKept (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isKept(s) \
  ((sRef_isValid(s)) && ((s)->aliaskind == AK_KEPT))

extern /*@unused@*/ bool sRef_isTemp (sRef s) /*@*/ ;

extern bool sRef_isStack (sRef s) /*@*/ ;
extern bool sRef_isLocalState (sRef s) /*@*/ ;
extern bool sRef_isUnique (sRef s) /*@*/ ;
extern bool sRef_isShared (sRef s) /*@*/ ;
extern bool sRef_isExposed (sRef s) /*@*/ ;
extern bool sRef_isObserver (sRef s) /*@*/ ;
extern bool sRef_isFresh (sRef s) /*@*/ ;

extern bool sRef_isRefsField (/*@sef@*/ sRef s) /*@*/ ;
# define sRef_isRefsField(s) \
  ((sRef_isValid(s)) && ((s)->aliaskind == AK_REFS))

extern void sRef_protectDerivs (void) /*@modifies internalState@*/ ;
extern void sRef_clearProtectDerivs (void) /*@modifies internalState@*/ ;

extern exkind sRef_getExKind (sRef s) /*@*/ ;
extern exkind sRef_getOrigExKind (sRef s) /*@*/ ;
extern void sRef_setExKind (sRef s, exkind ex, fileloc loc) /*@modifies s@*/ ;
extern void sRef_setExposed (sRef s, fileloc loc) /*@modifies s@*/;

extern bool sRef_isAnyParam (sRef s) /*@*/ ;
extern /*@observer@*/ sRef sRef_getAliasInfoRef (/*@exposed@*/ sRef s) /*@*/ ;
extern bool sRef_hasAliasInfoRef (sRef s) /*@*/ ;

extern /*@exposed@*/ sRef sRef_constructPointer (sRef t) /*@modifies t*/ ;
extern bool sRef_isAliasCheckedGlobal (sRef s) /*@*/ ;
extern bool sRef_includedBy (sRef small, sRef big) /*@*/ ;
extern /*@dependent@*/ /*@exposed@*/ sRef sRef_makeExternal (/*@exposed@*/ sRef s) /*@*/ ;
extern bool sRef_similarRelaxed (sRef s1, sRef s2) /*@*/ ;
extern /*@only@*/ cstring sRef_unparseKindName (sRef s) /*@*/ ;
extern void sRef_copyState (sRef s1, sRef s2) /*@modifies s1@*/ ;
extern /*@unused@*/ bool sRef_isObject (sRef s) /*@*/ ;
extern bool sRef_isNotUndefined (sRef s) /*@*/ ;
extern bool sRef_isExternal (sRef s) /*@*/ ;
extern cstring sRef_unparseDeep (sRef s) /*@*/ ;
extern cstring sRef_unparseFull (sRef s) /*@*/ ;
extern /*@observer@*/ cstring sRef_unparseScope (sRef s) /*@*/ ;
extern void sRef_mergeState (sRef res, sRef other, clause cl, fileloc loc)
   /*@modifies res, other@*/ ;
extern void sRef_mergeOptState (sRef res, sRef other, clause cl, fileloc loc)
   /*@modifies res, other@*/ ;
extern void sRef_mergeStateQuiet (sRef res, sRef other)
   /*@modifies res@*/ ;
extern void sRef_mergeStateQuietReverse (sRef res, sRef other)
   /*@modifies res@*/ ;
extern void sRef_setStateFromUentry (sRef s, uentry ue)
   /*@modifies s@*/ ;
extern bool sRef_isStackAllocated (sRef s) /*@*/ ;
extern bool sRef_modInFunction (void) /*@*/ ;
extern void sRef_clearAliasState (sRef s, fileloc loc)
   /*@modifies s@*/ ;
extern void sRef_setPartial (sRef s, fileloc loc)
   /*@modifies s@*/ ;
extern void sRef_setDerivNullState (sRef set, sRef guide, nstate ns)
   /*@modifies set@*/ ;

extern void sRef_clearGlobalScope (void) /*@modifies internalState@*/ ;

extern sRef sRef_makeDerived (/*@exposed@*/ sRef t);

extern sstate sRef_getDefState (sRef s) /*@*/ ;
extern void sRef_setDefState (sRef s, sstate defstate, fileloc loc);
extern void sRef_showRefLost (sRef s);
extern void sRef_showRefKilled (sRef s);
extern /*@exposed@*/ sRef sRef_updateSref (sRef s);

extern void sRef_aliasCheckPred (bool (predf) (sRef, exprNode, sRef, exprNode),
				 /*@null@*/ bool (checkAliases) (sRef),
				 sRef s, exprNode e, exprNode err);
extern bool sRef_aliasCheckSimplePred (sRefTest predf, sRef s);

extern void sRef_showStateInconsistent (sRef s);

extern void sRef_setDependentComplete (sRef s, fileloc loc);

extern void sRef_setAliasKindComplete (sRef s, alkind kind, fileloc loc);

extern bool sRef_isThroughArrayFetch (sRef s) /*@*/ ;

extern /*@exposed@*/ /*@notnull@*/ sRef sRef_getConjA (sRef s) /*@*/ ; 
extern /*@exposed@*/ /*@notnull@*/ sRef sRef_getConjB (sRef s) /*@*/ ;  

extern /*@only@*/ cstring sRef_unparsePreOpt (sRef s) /*@*/ ;

extern void sRef_free (/*@only@*/ sRef s);

extern void sRef_setObserver (sRef s, fileloc loc) /*@modifies s@*/ ;

# else
# error "Multiple include"
# endif













