
/*
 * DREADERD/SUBS.C
 *
 *	Misc subroutines.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype char *parseword(char **pptr, char *toks);
Prototype void SetCurrentGroup(Connection *conn, const char *group);
Prototype int shash(const char *s);
Prototype void SanitizeString(char *name);
Prototype int ValidGroupName(const char *name);
Prototype int SetTimeRestrict(TimeRestrict *tr, const char *yymmdd, const char *hhmmss, const char *gmt);

char *
parseword(char **pptr, char *toks)
{
    char *base = *pptr;
    int i;

    for (i = 0; base[i] && strchr(toks, base[i]) == NULL; ++i)
	;
    if (base[i]) {
	int j;

	base[i] = 0;
	for (j = i + 1; base[j] && strchr(toks, base[j]) != NULL; ++j)
	    ;
	*pptr = base + j;
    } else {
	*pptr = base + i;
	if (i == 0)
	    base = NULL;
    }
    return(base);
}

void
SetCurrentGroup(Connection *conn, const char *group)
{
    zfreeStr(&conn->co_MemPool, &conn->co_GroupName);
    conn->co_GroupName = zallocStr(&conn->co_MemPool, group);
}

int
shash(const char *s)
{
    int hv = 0xAFC32344;

    while (*s) {
	hv = (hv << 5) ^ *s ^ (hv >> 23);
	++s;
    }
    return(hv ^ (hv >> 16));
}

void
SanitizeString(char *name)
{
    for (; *name; ++name) {
	if (*name >= 'a' && *name <= 'z')
	    continue;
	if (*name >= 'A' && *name <= 'Z')
	    continue;
	if (*name >= '0' && *name <= '9')
	    continue;
	if (*name == '\t')
	    *name = ' ';
	if (*name == '@' ||
	    *name == '-' ||
	    *name == '+' ||
	    *name == '_' ||
	    *name == '.' ||
	    *name == ',' ||
	    *name == ' '
	) {
	    continue;
	}
	*name = '?';
    }
}

int 
ValidGroupName(const char *name)
{
    int r = 0;

    if (name[0] == 0 || name[0] == '.') {
	r = -1;
    } else {
	for (; *name; ++name) {
	    if (*name >= 'a' && *name <= 'z')
		continue;
	    if (*name >= 'A' && *name <= 'Z')
		continue;
	    if (*name >= '0' && *name <= '9')
		continue;
	    if (*name == '+')
		continue;
	    if (*name == '-')
		continue;
	    if (*name == '.') {
		if (name[-1] == '.' || name[1] == '.' || name[1] == 0)
		    r = -1;
	    }
	}
    }
    return(r);
}

int 
SetTimeRestrict(TimeRestrict *tr, const char *yymmdd, const char *hhmmss, const char *gmt)
{
    time_t t = time(NULL);
    struct tm tmv = { 0 };
    struct tm *tp = localtime(&t);

    if (yymmdd == NULL || hhmmss == NULL)
	return(-1);

    if (sscanf(yymmdd, "%2d%2d%2d", &tmv.tm_year,&tmv.tm_mon,&tmv.tm_mday) != 3)
	return(-1);
    if (sscanf(hhmmss, "%2d%2d%2d", &tmv.tm_hour,&tmv.tm_min,&tmv.tm_sec) != 3)
	return(-1);
    if (--tmv.tm_mon < 0 || tmv.tm_mday <= 0)
	return(-1);

    /*
     * Handle year rollover, aka '00 - 99' == -99.
     */
    tmv.tm_year += (tp->tm_year / 100) * 100;

    if (tp->tm_year - tmv.tm_year < -50)
	tmv.tm_year -= 100;
    else if (tp->tm_year - tmv.tm_year > 50)
	tmv.tm_year += 100;

    tmv.tm_isdst = -1;

    t = mktime(&tmv);

    /*
     * don't use localtime, use gmt.  Figure out the GMT offset.
     */

    if (gmt && strcmp(gmt, "GMT") == 0) {
        struct tm lt = *localtime(&t);
        struct tm gt = *gmtime(&t);
        int dt =
            (lt.tm_sec + lt.tm_min * 60 + lt.tm_hour * (60 * 60)) -
            (gt.tm_sec + gt.tm_min * 60 + gt.tm_hour * (60 * 60));
        if (lt.tm_mday > gt.tm_mday)
            dt += 24 * 60 * 60;
        if (lt.tm_mday < gt.tm_mday)
            dt -= 24 * 60 * 60;
        t += dt;
    }
    tr->tr_Time = t;
    return(0);
}

