/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		TCPListenerSocket.cpp

	Contains:	implements TCPListenerSocket class
					
	$Log: TCPListenerSocket.cpp,v $
	Revision 1.2  1999/02/19 23:13:21  ds
	Created
	
	
*/

#ifndef __MW_
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <unistd.h>
#include <sys/errno.h>
#endif

#include "TCPListenerSocket.h"
#include "Task.h"

QTSS_ErrorCode TCPListenerSocket::Listen(UInt32 queueLength)
{
	if (fSocket == kInvalidSocket)
		return QTSS_FileNotOpen;
		
	int err = ::listen(fSocket, queueLength);
	if (err != 0)
		return (QTSS_ErrorCode)OSThread::GetErrno();
	return QTSS_NoErr;
}

QTSS_ErrorCode TCPListenerSocket::Initialize(UInt32 addr, UInt16 port)
{
	QTSS_ErrorCode err = this->TCPSocket::Open();
	if (err != 0)
		return err;
	
	this->ReuseAddr();
	err = this->Bind(addr, port);
	if (err == 0)
		err = this->Listen(kListenQueueLength);
	return err;
}

void TCPListenerSocket::ProcessEvent(Task::EventFlags /*theEvent*/)
{
	//we are executing on the same thread as every other goddam
	//socket, so whatever you do here has to be fast.
	
	struct sockaddr_in addr;
	int size = sizeof(addr);
	Task* theTask = NULL;
	TCPSocket* theSocket = NULL;
	
	//fSocket data member of TCPSocket.
	while (true)
	{
		int osSocket = accept(fSocket, (struct sockaddr*)&addr, &size);
		if (osSocket == -1)
		{
			//take a look at what this error is.
			int acceptError = OSThread::GetErrno();
			
			//If it's EWOULDBLOCK, there's nothing on the listen queue right now,
			//so modwatch and return
			if (acceptError == EWOULDBLOCK)
				break;
			//if this error gets returned, we're out of file desciptors, so let's
			//idle the listener until the condition clears up.
			else if (acceptError == EMFILE)
			{
				fOutOfDescriptors = true;
				//this->SetIdleTimer(kTimeBetweenAcceptsInMsec);
				return;//don't modwatch, because we're out of file descriptors anyway.
			}
			else
			{
#if DEBUG
				printf("TCP Listener socket got %d err from accept\n",acceptError);
#endif
				continue;
			}
		}
		if (!fListening)
{
#if !__MacOSX__
			removeevent(osSocket);
#endif
			//this should be a disconnect. do an ioctl call?
			close(osSocket);
}
		else if ((theTask = this->GetSessionTask(&theSocket)) == NULL)
{
#if !__MacOSX__
			removeevent(osSocket);
#endif
			//this should be a disconnect. do an ioctl call?
			close(osSocket);
}
		else
		{	
			Assert(osSocket != kInvalidSocket);
			
			//set options on the socket
			//we are a server, always disable nagle algorithm
			int one = 1;
			int err = ::setsockopt(osSocket, IPPROTO_TCP, TCP_NODELAY, (char*)&one, sizeof(int));
			AssertV(err == 0, OSThread::GetErrno());

			err = ::setsockopt(osSocket, SOL_SOCKET, SO_KEEPALIVE, (char*)&one, sizeof(int));
			AssertV(err == 0, OSThread::GetErrno());

			//setup the socket. When there is data on the socket,
			//theTask will get an S_DATA event
			theSocket->Set(osSocket, &addr, this, theTask);
		}
	}
	
	fOutOfDescriptors = false;
	//after every accept we should modwatch to make sure we will continue
	//to get listen events
	this->Modwatch();
}
SInt64 TCPListenerSocket::Run()
{
	//This function will get called when we have run out of file descriptors.
	//All we need to do is check the listen queue to see if the situation has
	//cleared up.
	(void)this->GetEvents();
	this->ProcessEvent(Task::kReadEvent);
	return 0;
}
