/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFrdTAhead (FILE *fp)

Purpose:
  Get file format information from a text audio file

Description:
  This routine reads the header for a text audio file.  The header information
  is used to set the file data format information in the audio file pointer
  structure.

  Text audio files have been used to store audio information (specifically
  noise samples) on CDROM's.  The audio data itself is stored as text.  The
  text specifies integer values, with one value per line.  Non-sequential
  access to the data involves rewinding the file for each such access.

  Text audio header:
   Offset Length Type    Contents
      0     4    char   File identifier ("%//")
      -     -    char   Sampling frequency ("%sampling rate: <value> KHz")
      -     -    char   Audio data
  For text audio files, the audio data is the character representation of
  integer values, with one value per line.

Parameters:
  <-  AFILE *AFrdTAhead
      Audio file pointer for the audio file
   -> FILE *fp
      File pointer for the file

Author / revision:
  P. Kabal  Copyright (C) 1998
  $Revision: 1.19 $  $Date: 1998/06/19 19:14:03 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFrdTAhead.c 1.19 1998/06/19 libtsp-v3r0 $";

#include <string.h>

#include <libtsp.h>
#include <libtsp/AFdataio.h>
#include <libtsp/AFheader.h>
#include <libtsp/Xstdio.h>
#include <libtsp/AFpar.h>
#include <libtsp/AFmsg.h>

#define FM_TXAUD	"%//"
#define MAX_NHEAD	100


AFILE *
AFrdTAhead (fp)

     FILE *fp;

{
  AFILE *AFp;
  char *line;
  double FkHz;
  int nc, c, n, ErrCode;
  char Info[AF_MAXINFO];
  struct AF_info Hinfo;

  AFp = NULL;
  ErrCode = 0;

/* Check the file magic */
  line = AFgetLine (fp, &ErrCode);
  if (line == NULL) {
    UTwarn ("AFrdTAhead - %s", AFM_UEoF);
    return NULL;
  }
  if (strcmp (line, FM_TXAUD) != 0) {
    UTwarn ("AFrdTAhead - %s", AFM_TA_BadId);
    return NULL;
  }

/* Check the header */
  FkHz = 0.0;
  Hinfo.Info = Info;
  Hinfo.N = 0;
  for (n = 0; n < MAX_NHEAD; ++n) {

    /* Peek ahead to see if the header has ended */
    c = fgetc (fp);
    if (c == EOF)
      break;
    ungetc (c, fp);
    if (c != '%')
      break;

    /* Read a line */
    line = AFgetLine (fp, &ErrCode);
    if (ErrCode)
      return NULL;

    /* Save the line in the AFsp information structure */
    nc = strlen (line);
    if (nc < AF_MAXINFO - Hinfo.N + 1) {
      if (Hinfo.N > 0)
	Hinfo.Info[Hinfo.N] = '\n';	/* Replace null with line-feed */
      strcpy (&Hinfo.Info[Hinfo.N + 1], &line[1]);
      Hinfo.N += nc;
    }

    /* Check for a sampling frequency record */
    if (strncmp (line, "%%sampling rate:", sizeof "%%sampling rate:"))
      sscanf (line, "%%sampling rate:%lg KHz", &FkHz); /* "KHz" is ignored */
  }

/* Error checks */
  if (n >= MAX_NHEAD) {
    UTwarn ("AFrdTAhead - %s", AFM_TA_LongHead);
    return NULL;
  }
  if (FkHz == 0.0) {
    UTwarn ("AFrdTAhead - %s", AFM_TA_NoNSamp);
    return NULL;
  }

/* Set the parameters for file access */
  AFp = AFsetRead (fp, FT_TXAUD, FD_TEXT, DS_NATIVE, 1000. * FkHz, 1.0, 1L,
		   AF_LDATA_UNDEF, AF_NSAMP_UNDEF, &Hinfo, AF_NOFIX);

  return AFp;
}
