/*

  File: "nucleic.c"

  Author: Marc Feeley (feeley@iro.umontreal.ca)

  Last modified: June 6, 1997

  This program is a modified version of the program described in the paper:

    M. Feeley, M. Turcotte, G. Lapalme, "Using Multilisp for Solving
    Constraint Satisfaction Problems: an Application to Nucleic Acid 3D
    Structure Determination" published in the journal "Lisp and Symbolic
    Computation".

  The differences between this program and the original are described in
  the paper:

    "???" published in the "Journal of Functional Programming".

*/

/*
  Define the symbol "lazy" if you want lazy computation.
*/

#define not_lazy

#include <stdio.h>
#include <math.h>

/*---------------------------------------------------------------------------*/

/* MATH UTILITIES */

typedef double FLOAT; /* define precision of floating point numbers */

#ifndef M_PI
#define M_PI   3.14159265358979323846
#endif
#ifndef M_PI_2
#define M_PI_2 1.57079632679489661923
#endif

struct struct_pt
{
  FLOAT x, y, z;
#ifdef lazy
  int relative; /* true if point remains to be converted to absolute coord. */
#endif
};

typedef struct struct_pt *pt;

#define make_pt(a,b,c,p){(p)->x=(a);(p)->y=(b);(p)->z=(c);}

struct struct_tfo
{
  FLOAT m[4][3];
};

typedef struct struct_tfo *tfo;

#define make_tfo(a,b,c,d,e,f,g,h,i,x,y,z,t)\
{\
(t)->m[0][0]=(a);(t)->m[0][1]=(b);(t)->m[0][2]=(c);\
(t)->m[1][0]=(d);(t)->m[1][1]=(e);(t)->m[1][2]=(f);\
(t)->m[2][0]=(g);(t)->m[2][1]=(h);(t)->m[2][2]=(i);\
(t)->m[3][0]=(x);(t)->m[3][1]=(y);(t)->m[3][2]=(z);\
}

struct struct_A
{
  struct struct_pt n6, n7, n9, c8, h2, h61, h62, h8;
};

struct struct_C
{
  struct struct_pt n4, o2, h41, h42, h5, h6;
};

struct struct_G
{
  struct struct_pt n2, n7, n9, c8, o6, h1, h21, h22, h8;
};

struct struct_U
{
  struct struct_pt o2, o4, h3, h5, h6;
};

struct struct_nuc
{
  char type; /* 'A', 'C', 'G', or 'U' */
  struct struct_tfo
    dgf_base_tfo, p_o3_275_tfo, p_o3_180_tfo, p_o3_60_tfo;
  struct struct_pt
    p, o1p, o2p, o5_, c5_, h5_, h5__, c4_, h4_, o4_, c1_, h1_,
    c2_, h2__, o2_, h2_, c3_, h3_, o3_, n1, n3, c2, c4, c5, c6;
  union
  {
    struct struct_A A;
    struct struct_C C;
    struct struct_G G;
    struct struct_U U;
  } _;
};

typedef struct struct_nuc *nuc;

/* NUCLEIC ACID CONFORMATIONS DATA BASE */
struct struct_nuc rA01, rA02, rA03, rA04, rA05, rA06, rA07, rA08, rA09, rA10;
struct struct_nuc rC01, rC02, rC03, rC04, rC05, rC06, rC07, rC08, rC09, rC10;
struct struct_nuc rG01, rG02, rG03, rG04, rG05, rG06, rG07, rG08, rG09, rG10;
struct struct_nuc rU01, rU02, rU03, rU04, rU05, rU06, rU07, rU08, rU09, rU10;
struct struct_nuc *rAs[11], rA;
struct struct_nuc *rCs[11], rC;
struct struct_nuc *rGs[11], rG, rG_;
struct struct_nuc *rUs[11], rU, rU_;

void init_A()
{
  rA.type = 'A';
  make_tfo(  -0.0018,  -0.8207,   0.5714,
              0.2679,  -0.5509,  -0.7904,
              0.9634,   0.1517,   0.2209,
              0.0073,   8.4030,   0.6232, &rA.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rA.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rA.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rA.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rA.p         );
  make_pt(    1.6980,   7.6960,  -3.5570, &rA.o1p       );
  make_pt(    3.2260,   9.5010,  -4.4020, &rA.o2p       );
  make_pt(    4.1590,   7.6040,  -3.0340, &rA.o5_       );
  make_pt(    5.4550,   8.2120,  -2.8810, &rA.c5_       );
  make_pt(    5.4546,   8.8508,  -1.9978, &rA.h5_       );
  make_pt(    5.7588,   8.6625,  -3.8259, &rA.h5__      );
  make_pt(    6.4970,   7.1480,  -2.5980, &rA.c4_       );
  make_pt(    7.4896,   7.5919,  -2.5214, &rA.h4_       );
  make_pt(    6.1630,   6.4860,  -1.3440, &rA.o4_       );
  make_pt(    6.5400,   5.1200,  -1.4190, &rA.c1_       );
  make_pt(    7.2763,   4.9681,  -0.6297, &rA.h1_       );
  make_pt(    7.1940,   4.8830,  -2.7770, &rA.c2_       );
  make_pt(    6.8667,   3.9183,  -3.1647, &rA.h2__      );
  make_pt(    8.5860,   5.0910,  -2.6140, &rA.o2_       );
  make_pt(    8.9510,   4.7626,  -1.7890, &rA.h2_       );
  make_pt(    6.5720,   6.0040,  -3.6090, &rA.c3_       );
  make_pt(    5.5636,   5.7066,  -3.8966, &rA.h3_       );
  make_pt(    7.3801,   6.3562,  -4.7350, &rA.o3_       );
  make_pt(    4.7150,   0.4910,  -0.1360, &rA.n1        );
  make_pt(    6.3490,   2.1730,  -0.6020, &rA.n3        );
  make_pt(    5.9530,   0.9650,  -0.2670, &rA.c2        );
  make_pt(    5.2900,   2.9790,  -0.8260, &rA.c4        );
  make_pt(    3.9720,   2.6390,  -0.7330, &rA.c5        );
  make_pt(    3.6770,   1.3160,  -0.3660, &rA.c6        );
  make_pt(    2.4280,   0.8450,  -0.2360, &rA._.A.n6    );
  make_pt(    3.1660,   3.7290,  -1.0360, &rA._.A.n7    );
  make_pt(    5.3170,   4.2990,  -1.1930, &rA._.A.n9    );
  make_pt(    4.0100,   4.6780,  -1.2990, &rA._.A.c8    );
  make_pt(    6.6890,   0.1903,  -0.0518, &rA._.A.h2    );
  make_pt(    1.6470,   1.4460,  -0.4040, &rA._.A.h61   );
  make_pt(    2.2780,  -0.1080,  -0.0280, &rA._.A.h62   );
  make_pt(    3.4421,   5.5744,  -1.5482, &rA._.A.h8    );

  rA01.type = 'A';
  make_tfo(  -0.0043,  -0.8175,   0.5759,
              0.2617,  -0.5567,  -0.7884,
              0.9651,   0.1473,   0.2164,
              0.0359,   8.3929,   0.5532, &rA01.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rA01.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rA01.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rA01.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rA01.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rA01.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rA01.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rA01.o5_     );
  make_pt(    5.4352,   8.2183,  -2.7757, &rA01.c5_     );
  make_pt(    5.3830,   8.7883,  -1.8481, &rA01.h5_     );
  make_pt(    5.7729,   8.7436,  -3.6691, &rA01.h5__    );
  make_pt(    6.4830,   7.1518,  -2.5252, &rA01.c4_     );
  make_pt(    7.4749,   7.5972,  -2.4482, &rA01.h4_     );
  make_pt(    6.1626,   6.4620,  -1.2827, &rA01.o4_     );
  make_pt(    6.5431,   5.0992,  -1.3905, &rA01.c1_     );
  make_pt(    7.2871,   4.9328,  -0.6114, &rA01.h1_     );
  make_pt(    7.1852,   4.8935,  -2.7592, &rA01.c2_     );
  make_pt(    6.8573,   3.9363,  -3.1645, &rA01.h2__    );
  make_pt(    8.5780,   5.1025,  -2.6046, &rA01.o2_     );
  make_pt(    8.9516,   4.7577,  -1.7902, &rA01.h2_     );
  make_pt(    6.5522,   6.0300,  -3.5612, &rA01.c3_     );
  make_pt(    5.5420,   5.7356,  -3.8459, &rA01.h3_     );
  make_pt(    7.3487,   6.4089,  -4.6867, &rA01.o3_     );
  make_pt(    4.7442,   0.4514,  -0.1390, &rA01.n1      );
  make_pt(    6.3687,   2.1459,  -0.5926, &rA01.n3      );
  make_pt(    5.9795,   0.9335,  -0.2657, &rA01.c2      );
  make_pt(    5.3052,   2.9471,  -0.8125, &rA01.c4      );
  make_pt(    3.9891,   2.5987,  -0.7230, &rA01.c5      );
  make_pt(    3.7016,   1.2717,  -0.3647, &rA01.c6      );
  make_pt(    2.4553,   0.7925,  -0.2390, &rA01._.A.n6  );
  make_pt(    3.1770,   3.6859,  -1.0198, &rA01._.A.n7  );
  make_pt(    5.3247,   4.2695,  -1.1710, &rA01._.A.n9  );
  make_pt(    4.0156,   4.6415,  -1.2759, &rA01._.A.c8  );
  make_pt(    6.7198,   0.1618,  -0.0547, &rA01._.A.h2  );
  make_pt(    1.6709,   1.3900,  -0.4039, &rA01._.A.h61 );
  make_pt(    2.3107,  -0.1627,  -0.0373, &rA01._.A.h62 );
  make_pt(    3.4426,   5.5361,  -1.5199, &rA01._.A.h8  );

  rA02.type = 'A';
  make_tfo(   0.5566,   0.0449,   0.8296,
              0.5125,   0.7673,  -0.3854,
             -0.6538,   0.6397,   0.4041,
             -9.1161,  -3.7679,  -2.9968, &rA02.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rA02.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rA02.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rA02.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rA02.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rA02.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rA02.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rA02.o5_     );
  make_pt(    4.5778,   6.6594,  -4.0364, &rA02.c5_     );
  make_pt(    4.9220,   7.1963,  -4.9204, &rA02.h5_     );
  make_pt(    3.7996,   5.9091,  -4.1764, &rA02.h5__    );
  make_pt(    5.7873,   5.8869,  -3.5482, &rA02.c4_     );
  make_pt(    6.0405,   5.0875,  -4.2446, &rA02.h4_     );
  make_pt(    6.9135,   6.8036,  -3.4310, &rA02.o4_     );
  make_pt(    7.7293,   6.4084,  -2.3392, &rA02.c1_     );
  make_pt(    8.7078,   6.1815,  -2.7624, &rA02.h1_     );
  make_pt(    7.1305,   5.1418,  -1.7347, &rA02.c2_     );
  make_pt(    7.2040,   5.1982,  -0.6486, &rA02.h2__    );
  make_pt(    7.7417,   4.0392,  -2.3813, &rA02.o2_     );
  make_pt(    8.6785,   4.1443,  -2.5630, &rA02.h2_     );
  make_pt(    5.6666,   5.2728,  -2.1536, &rA02.c3_     );
  make_pt(    5.1747,   5.9805,  -1.4863, &rA02.h3_     );
  make_pt(    4.9997,   4.0086,  -2.1973, &rA02.o3_     );
  make_pt(   10.3245,   8.5459,   1.5467, &rA02.n1      );
  make_pt(    9.8051,   6.9432,  -0.1497, &rA02.n3      );
  make_pt(   10.5175,   7.4328,   0.8408, &rA02.c2      );
  make_pt(    8.7523,   7.7422,  -0.4228, &rA02.c4      );
  make_pt(    8.4257,   8.9060,   0.2099, &rA02.c5      );
  make_pt(    9.2665,   9.3242,   1.2540, &rA02.c6      );
  make_pt(    9.0664,  10.4462,   1.9610, &rA02._.A.n6  );
  make_pt(    7.2750,   9.4537,  -0.3428, &rA02._.A.n7  );
  make_pt(    7.7962,   7.5519,  -1.3859, &rA02._.A.n9  );
  make_pt(    6.9479,   8.6157,  -1.2771, &rA02._.A.c8  );
  make_pt(   11.4063,   6.9047,   1.1859, &rA02._.A.h2  );
  make_pt(    8.2845,  11.0341,   1.7552, &rA02._.A.h61 );
  make_pt(    9.6584,  10.6647,   2.7198, &rA02._.A.h62 );
  make_pt(    6.0430,   8.9853,  -1.7594, &rA02._.A.h8  );

  rA03.type = 'A';
  make_tfo(  -0.5021,   0.0731,   0.8617,
             -0.8112,   0.3054,  -0.4986,
             -0.2996,  -0.9494,  -0.0940,
              6.4273,  -5.1944,  -3.7807, &rA03.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rA03.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rA03.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rA03.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rA03.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rA03.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rA03.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rA03.o5_     );
  make_pt(    4.1214,   6.7116,  -1.9049, &rA03.c5_     );
  make_pt(    3.3465,   5.9610,  -2.0607, &rA03.h5_     );
  make_pt(    4.0789,   7.2928,  -0.9837, &rA03.h5__    );
  make_pt(    5.4170,   5.9293,  -1.8186, &rA03.c4_     );
  make_pt(    5.4506,   5.3400,  -0.9023, &rA03.h4_     );
  make_pt(    5.5067,   5.0417,  -2.9703, &rA03.o4_     );
  make_pt(    6.8650,   4.9152,  -3.3612, &rA03.c1_     );
  make_pt(    7.1090,   3.8577,  -3.2603, &rA03.h1_     );
  make_pt(    7.7152,   5.7282,  -2.3894, &rA03.c2_     );
  make_pt(    8.5029,   6.2356,  -2.9463, &rA03.h2__    );
  make_pt(    8.1036,   4.8568,  -1.3419, &rA03.o2_     );
  make_pt(    8.3270,   3.9651,  -1.6184, &rA03.h2_     );
  make_pt(    6.7003,   6.7565,  -1.8911, &rA03.c3_     );
  make_pt(    6.5898,   7.5329,  -2.6482, &rA03.h3_     );
  make_pt(    7.0505,   7.2878,  -0.6105, &rA03.o3_     );
  make_pt(    9.6740,   4.7656,  -7.6614, &rA03.n1      );
  make_pt(    9.0739,   4.3013,  -5.3941, &rA03.n3      );
  make_pt(    9.8416,   4.2192,  -6.4581, &rA03.c2      );
  make_pt(    7.9885,   5.0632,  -5.6446, &rA03.c4      );
  make_pt(    7.6822,   5.6856,  -6.8194, &rA03.c5      );
  make_pt(    8.5831,   5.5215,  -7.8840, &rA03.c6      );
  make_pt(    8.4084,   6.0747,  -9.0933, &rA03._.A.n6  );
  make_pt(    6.4857,   6.3816,  -6.7035, &rA03._.A.n7  );
  make_pt(    6.9740,   5.3703,  -4.7760, &rA03._.A.n9  );
  make_pt(    6.1133,   6.1613,  -5.4808, &rA03._.A.c8  );
  make_pt(   10.7627,   3.6375,  -6.4220, &rA03._.A.h2  );
  make_pt(    7.6031,   6.6390,  -9.2733, &rA03._.A.h61 );
  make_pt(    9.1004,   5.9708,  -9.7893, &rA03._.A.h62 );
  make_pt(    5.1705,   6.6830,  -5.3167, &rA03._.A.h8  );

  rA04.type = 'A';
  make_tfo(  -0.5426,  -0.8175,   0.1929,
              0.8304,  -0.5567,  -0.0237,
              0.1267,   0.1473,   0.9809,
             -0.5075,   8.3929,   0.2229, &rA04.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rA04.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rA04.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rA04.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rA04.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rA04.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rA04.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rA04.o5_     );
  make_pt(    5.4352,   8.2183,  -2.7757, &rA04.c5_     );
  make_pt(    5.3830,   8.7883,  -1.8481, &rA04.h5_     );
  make_pt(    5.7729,   8.7436,  -3.6691, &rA04.h5__    );
  make_pt(    6.4830,   7.1518,  -2.5252, &rA04.c4_     );
  make_pt(    7.4749,   7.5972,  -2.4482, &rA04.h4_     );
  make_pt(    6.1626,   6.4620,  -1.2827, &rA04.o4_     );
  make_pt(    6.5431,   5.0992,  -1.3905, &rA04.c1_     );
  make_pt(    7.2871,   4.9328,  -0.6114, &rA04.h1_     );
  make_pt(    7.1852,   4.8935,  -2.7592, &rA04.c2_     );
  make_pt(    6.8573,   3.9363,  -3.1645, &rA04.h2__    );
  make_pt(    8.5780,   5.1025,  -2.6046, &rA04.o2_     );
  make_pt(    8.9516,   4.7577,  -1.7902, &rA04.h2_     );
  make_pt(    6.5522,   6.0300,  -3.5612, &rA04.c3_     );
  make_pt(    5.5420,   5.7356,  -3.8459, &rA04.h3_     );
  make_pt(    7.3487,   6.4089,  -4.6867, &rA04.o3_     );
  make_pt(    3.6343,   2.6680,   2.0783, &rA04.n1      );
  make_pt(    5.4505,   3.9805,   1.2446, &rA04.n3      );
  make_pt(    4.7540,   3.3816,   2.1851, &rA04.c2      );
  make_pt(    4.8805,   3.7951,   0.0354, &rA04.c4      );
  make_pt(    3.7416,   3.0925,  -0.2305, &rA04.c5      );
  make_pt(    3.0873,   2.4980,   0.8606, &rA04.c6      );
  make_pt(    1.9600,   1.7805,   0.7462, &rA04._.A.n6  );
  make_pt(    3.4605,   3.1184,  -1.5906, &rA04._.A.n7  );
  make_pt(    5.3247,   4.2695,  -1.1710, &rA04._.A.n9  );
  make_pt(    4.4244,   3.8244,  -2.0953, &rA04._.A.c8  );
  make_pt(    5.0814,   3.4352,   3.2234, &rA04._.A.h2  );
  make_pt(    1.5423,   1.6454,  -0.1520, &rA04._.A.h61 );
  make_pt(    1.5716,   1.3398,   1.5392, &rA04._.A.h62 );
  make_pt(    4.2675,   3.8876,  -3.1721, &rA04._.A.h8  );

  rA05.type = 'A';
  make_tfo(  -0.5891,   0.0449,   0.8068,
              0.5375,   0.7673,   0.3498,
             -0.6034,   0.6397,  -0.4762,
             -0.3019,  -3.7679,  -9.5913, &rA05.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rA05.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rA05.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rA05.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rA05.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rA05.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rA05.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rA05.o5_     );
  make_pt(    4.5778,   6.6594,  -4.0364, &rA05.c5_     );
  make_pt(    4.9220,   7.1963,  -4.9204, &rA05.h5_     );
  make_pt(    3.7996,   5.9091,  -4.1764, &rA05.h5__    );
  make_pt(    5.7873,   5.8869,  -3.5482, &rA05.c4_     );
  make_pt(    6.0405,   5.0875,  -4.2446, &rA05.h4_     );
  make_pt(    6.9135,   6.8036,  -3.4310, &rA05.o4_     );
  make_pt(    7.7293,   6.4084,  -2.3392, &rA05.c1_     );
  make_pt(    8.7078,   6.1815,  -2.7624, &rA05.h1_     );
  make_pt(    7.1305,   5.1418,  -1.7347, &rA05.c2_     );
  make_pt(    7.2040,   5.1982,  -0.6486, &rA05.h2__    );
  make_pt(    7.7417,   4.0392,  -2.3813, &rA05.o2_     );
  make_pt(    8.6785,   4.1443,  -2.5630, &rA05.h2_     );
  make_pt(    5.6666,   5.2728,  -2.1536, &rA05.c3_     );
  make_pt(    5.1747,   5.9805,  -1.4863, &rA05.h3_     );
  make_pt(    4.9997,   4.0086,  -2.1973, &rA05.o3_     );
  make_pt(   10.2594,  10.6774,  -1.0056, &rA05.n1      );
  make_pt(    9.7528,   8.7080,  -2.2631, &rA05.n3      );
  make_pt(   10.4471,   9.7876,  -1.9791, &rA05.c2      );
  make_pt(    8.7271,   8.5575,  -1.3991, &rA05.c4      );
  make_pt(    8.4100,   9.3803,  -0.3580, &rA05.c5      );
  make_pt(    9.2294,  10.5030,  -0.1574, &rA05.c6      );
  make_pt(    9.0349,  11.3951,   0.8250, &rA05._.A.n6  );
  make_pt(    7.2891,   8.9068,   0.3121, &rA05._.A.n7  );
  make_pt(    7.7962,   7.5519,  -1.3859, &rA05._.A.n9  );
  make_pt(    6.9702,   7.8292,  -0.3353, &rA05._.A.c8  );
  make_pt(   11.3132,  10.0537,  -2.5851, &rA05._.A.h2  );
  make_pt(    8.2741,  11.2784,   1.4629, &rA05._.A.h61 );
  make_pt(    9.6733,  12.1368,   0.9529, &rA05._.A.h62 );
  make_pt(    6.0888,   7.3990,   0.1403, &rA05._.A.h8  );

  rA06.type = 'A';
  make_tfo(  -0.9815,   0.0731,  -0.1772,
              0.1912,   0.3054,  -0.9328,
             -0.0141,  -0.9494,  -0.3137,
              5.7506,  -5.1944,   4.7470, &rA06.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rA06.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rA06.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rA06.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rA06.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rA06.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rA06.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rA06.o5_     );
  make_pt(    4.1214,   6.7116,  -1.9049, &rA06.c5_     );
  make_pt(    3.3465,   5.9610,  -2.0607, &rA06.h5_     );
  make_pt(    4.0789,   7.2928,  -0.9837, &rA06.h5__    );
  make_pt(    5.4170,   5.9293,  -1.8186, &rA06.c4_     );
  make_pt(    5.4506,   5.3400,  -0.9023, &rA06.h4_     );
  make_pt(    5.5067,   5.0417,  -2.9703, &rA06.o4_     );
  make_pt(    6.8650,   4.9152,  -3.3612, &rA06.c1_     );
  make_pt(    7.1090,   3.8577,  -3.2603, &rA06.h1_     );
  make_pt(    7.7152,   5.7282,  -2.3894, &rA06.c2_     );
  make_pt(    8.5029,   6.2356,  -2.9463, &rA06.h2__    );
  make_pt(    8.1036,   4.8568,  -1.3419, &rA06.o2_     );
  make_pt(    8.3270,   3.9651,  -1.6184, &rA06.h2_     );
  make_pt(    6.7003,   6.7565,  -1.8911, &rA06.c3_     );
  make_pt(    6.5898,   7.5329,  -2.6482, &rA06.h3_     );
  make_pt(    7.0505,   7.2878,  -0.6105, &rA06.o3_     );
  make_pt(    6.6624,   3.5061,  -8.2986, &rA06.n1      );
  make_pt(    6.5810,   3.2570,  -5.9221, &rA06.n3      );
  make_pt(    6.5151,   2.8263,  -7.1625, &rA06.c2      );
  make_pt(    6.8364,   4.5817,  -5.8882, &rA06.c4      );
  make_pt(    7.0116,   5.4064,  -6.9609, &rA06.c5      );
  make_pt(    6.9173,   4.8260,  -8.2361, &rA06.c6      );
  make_pt(    7.0668,   5.5163,  -9.3763, &rA06._.A.n6  );
  make_pt(    7.2573,   6.7070,  -6.5394, &rA06._.A.n7  );
  make_pt(    6.9740,   5.3703,  -4.7760, &rA06._.A.n9  );
  make_pt(    7.2238,   6.6275,  -5.2453, &rA06._.A.c8  );
  make_pt(    6.3146,   1.7741,  -7.3641, &rA06._.A.h2  );
  make_pt(    7.2568,   6.4972,  -9.3456, &rA06._.A.h61 );
  make_pt(    7.0437,   5.0478, -10.2446, &rA06._.A.h62 );
  make_pt(    7.4108,   7.6227,  -4.8418, &rA06._.A.h8  );

  rA07.type = 'A';
  make_tfo(   0.2379,   0.1310,  -0.9624,
             -0.5876,  -0.7696,  -0.2499,
             -0.7734,   0.6249,  -0.1061,
             30.9870, -26.9344,  42.6416, &rA07.dgf_base_tfo );
  make_tfo(   0.7529,   0.1548,   0.6397,
              0.2952,  -0.9481,  -0.1180,
              0.5882,   0.2777,  -0.7595,
            -58.8919, -11.3095,   6.0866, &rA07.p_o3_275_tfo );
  make_tfo(  -0.0239,   0.9667,  -0.2546,
              0.9731,  -0.0359,  -0.2275,
             -0.2290,  -0.2532,  -0.9399,
              3.5401, -29.7913,  52.2796, &rA07.p_o3_180_tfo );
  make_tfo(  -0.8912,  -0.4531,   0.0242,
             -0.1183,   0.1805,  -0.9764,
              0.4380,  -0.8730,  -0.2145,
             19.9023,  54.8054,  15.2799, &rA07.p_o3_60_tfo  );
  make_pt(   41.8210,   8.3880,  43.5890, &rA07.p       );
  make_pt(   42.5400,   8.0450,  44.8330, &rA07.o1p     );
  make_pt(   42.2470,   9.6920,  42.9910, &rA07.o2p     );
  make_pt(   40.2550,   8.2030,  43.7340, &rA07.o5_     );
  make_pt(   39.3505,   8.4697,  42.6565, &rA07.c5_     );
  make_pt(   39.1377,   7.5433,  42.1230, &rA07.h5_     );
  make_pt(   39.7203,   9.3119,  42.0717, &rA07.h5__    );
  make_pt(   38.0405,   8.9195,  43.2869, &rA07.c4_     );
  make_pt(   37.3687,   9.3036,  42.5193, &rA07.h4_     );
  make_pt(   37.4319,   7.8146,  43.9387, &rA07.o4_     );
  make_pt(   37.1959,   8.1354,  45.3237, &rA07.c1_     );
  make_pt(   36.1788,   8.5202,  45.3970, &rA07.h1_     );
  make_pt(   38.1721,   9.2328,  45.6504, &rA07.c2_     );
  make_pt(   39.1555,   8.7939,  45.8188, &rA07.h2__    );
  make_pt(   37.7862,  10.0617,  46.7013, &rA07.o2_     );
  make_pt(   37.3087,   9.6229,  47.4092, &rA07.h2_     );
  make_pt(   38.1844,  10.0268,  44.3367, &rA07.c3_     );
  make_pt(   39.1578,  10.5054,  44.2289, &rA07.h3_     );
  make_pt(   37.0547,  10.9127,  44.3441, &rA07.o3_     );
  make_pt(   34.8811,   4.2072,  47.5784, &rA07.n1      );
  make_pt(   35.1084,   6.1336,  46.1818, &rA07.n3      );
  make_pt(   34.4108,   5.1360,  46.7207, &rA07.c2      );
  make_pt(   36.3908,   6.1224,  46.6053, &rA07.c4      );
  make_pt(   36.9819,   5.2334,  47.4697, &rA07.c5      );
  make_pt(   36.1786,   4.1985,  48.0035, &rA07.c6      );
  make_pt(   36.6103,   3.2749,  48.8452, &rA07._.A.n6  );
  make_pt(   38.3236,   5.5522,  47.6595, &rA07._.A.n7  );
  make_pt(   37.3887,   7.0024,  46.2437, &rA07._.A.n9  );
  make_pt(   38.5055,   6.6096,  46.9057, &rA07._.A.c8  );
  make_pt(   33.3553,   5.0152,  46.4771, &rA07._.A.h2  );
  make_pt(   37.5730,   3.2804,  49.1507, &rA07._.A.h61 );
  make_pt(   35.9775,   2.5638,  49.1828, &rA07._.A.h62 );
  make_pt(   39.5461,   6.9184,  47.0041, &rA07._.A.h8  );

  rA08.type = 'A';
  make_tfo(   0.1084,  -0.0895,  -0.9901,
              0.9789,  -0.1638,   0.1220,
             -0.1731,  -0.9824,   0.0698,
             -2.9039,  47.2655,  33.0094, &rA08.dgf_base_tfo );
  make_tfo(   0.7529,   0.1548,   0.6397,
              0.2952,  -0.9481,  -0.1180,
              0.5882,   0.2777,  -0.7595,
            -58.8919, -11.3095,   6.0866, &rA08.p_o3_275_tfo );
  make_tfo(  -0.0239,   0.9667,  -0.2546,
              0.9731,  -0.0359,  -0.2275,
             -0.2290,  -0.2532,  -0.9399,
              3.5401, -29.7913,  52.2796, &rA08.p_o3_180_tfo );
  make_tfo(  -0.8912,  -0.4531,   0.0242,
             -0.1183,   0.1805,  -0.9764,
              0.4380,  -0.8730,  -0.2145,
             19.9023,  54.8054,  15.2799, &rA08.p_o3_60_tfo  );
  make_pt(   41.8210,   8.3880,  43.5890, &rA08.p       );
  make_pt(   42.5400,   8.0450,  44.8330, &rA08.o1p     );
  make_pt(   42.2470,   9.6920,  42.9910, &rA08.o2p     );
  make_pt(   40.2550,   8.2030,  43.7340, &rA08.o5_     );
  make_pt(   39.4850,   8.9301,  44.6977, &rA08.c5_     );
  make_pt(   39.0638,   9.8199,  44.2296, &rA08.h5_     );
  make_pt(   40.0757,   9.0713,  45.6029, &rA08.h5__    );
  make_pt(   38.3102,   8.0414,  45.0789, &rA08.c4_     );
  make_pt(   37.7842,   8.4637,  45.9351, &rA08.h4_     );
  make_pt(   37.4200,   7.9453,  43.9769, &rA08.o4_     );
  make_pt(   37.2249,   6.5609,  43.6273, &rA08.c1_     );
  make_pt(   36.3360,   6.2168,  44.1561, &rA08.h1_     );
  make_pt(   38.4347,   5.8414,  44.1590, &rA08.c2_     );
  make_pt(   39.2688,   5.9974,  43.4749, &rA08.h2__    );
  make_pt(   38.2344,   4.4907,  44.4348, &rA08.o2_     );
  make_pt(   37.6374,   4.0386,  43.8341, &rA08.h2_     );
  make_pt(   38.6926,   6.6079,  45.4637, &rA08.c3_     );
  make_pt(   39.7585,   6.5640,  45.6877, &rA08.h3_     );
  make_pt(   37.8238,   6.0705,  46.4723, &rA08.o3_     );
  make_pt(   33.9162,   6.2598,  39.7758, &rA08.n1      );
  make_pt(   34.6709,   6.5759,  42.0215, &rA08.n3      );
  make_pt(   33.7257,   6.5186,  41.0858, &rA08.c2      );
  make_pt(   35.8935,   6.3324,  41.5018, &rA08.c4      );
  make_pt(   36.2105,   6.0601,  40.1932, &rA08.c5      );
  make_pt(   35.1538,   6.0151,  39.2537, &rA08.c6      );
  make_pt(   35.3088,   5.7642,  37.9649, &rA08._.A.n6  );
  make_pt(   37.5818,   5.8677,  40.0507, &rA08._.A.n7  );
  make_pt(   37.0932,   6.3197,  42.1810, &rA08._.A.n9  );
  make_pt(   38.0509,   6.0354,  41.2635, &rA08._.A.c8  );
  make_pt(   32.6830,   6.6898,  41.3532, &rA08._.A.h2  );
  make_pt(   36.2305,   5.5855,  37.5925, &rA08._.A.h61 );
  make_pt(   34.5056,   5.7512,  37.3528, &rA08._.A.h62 );
  make_pt(   39.1318,   5.8993,  41.2285, &rA08._.A.h8  );

  rA09.type = 'A';
  make_tfo(   0.8467,   0.4166,  -0.3311,
             -0.3962,   0.9089,   0.1303,
              0.3552,   0.0209,   0.9346,
            -42.7319, -26.6223, -29.8163, &rA09.dgf_base_tfo );
  make_tfo(   0.7529,   0.1548,   0.6397,
              0.2952,  -0.9481,  -0.1180,
              0.5882,   0.2777,  -0.7595,
            -58.8919, -11.3095,   6.0866, &rA09.p_o3_275_tfo );
  make_tfo(  -0.0239,   0.9667,  -0.2546,
              0.9731,  -0.0359,  -0.2275,
             -0.2290,  -0.2532,  -0.9399,
              3.5401, -29.7913,  52.2796, &rA09.p_o3_180_tfo );
  make_tfo(  -0.8912,  -0.4531,   0.0242,
             -0.1183,   0.1805,  -0.9764,
              0.4380,  -0.8730,  -0.2145,
             19.9023,  54.8054,  15.2799, &rA09.p_o3_60_tfo  );
  make_pt(   41.8210,   8.3880,  43.5890, &rA09.p       );
  make_pt(   42.5400,   8.0450,  44.8330, &rA09.o1p     );
  make_pt(   42.2470,   9.6920,  42.9910, &rA09.o2p     );
  make_pt(   40.2550,   8.2030,  43.7340, &rA09.o5_     );
  make_pt(   39.3505,   8.4697,  42.6565, &rA09.c5_     );
  make_pt(   39.1377,   7.5433,  42.1230, &rA09.h5_     );
  make_pt(   39.7203,   9.3119,  42.0717, &rA09.h5__    );
  make_pt(   38.0405,   8.9195,  43.2869, &rA09.c4_     );
  make_pt(   37.6479,   8.1347,  43.9335, &rA09.h4_     );
  make_pt(   38.2691,  10.0933,  44.0524, &rA09.o4_     );
  make_pt(   37.3999,  11.1488,  43.5973, &rA09.c1_     );
  make_pt(   36.5061,  11.1221,  44.2206, &rA09.h1_     );
  make_pt(   37.0364,  10.7838,  42.1836, &rA09.c2_     );
  make_pt(   37.8636,  11.0489,  41.5252, &rA09.h2__    );
  make_pt(   35.8275,  11.3133,  41.7379, &rA09.o2_     );
  make_pt(   35.6214,  12.1896,  42.0714, &rA09.h2_     );
  make_pt(   36.9316,   9.2556,  42.2837, &rA09.c3_     );
  make_pt(   37.1778,   8.8260,  41.3127, &rA09.h3_     );
  make_pt(   35.6285,   8.9334,  42.7926, &rA09.o3_     );
  make_pt(   38.1482,  15.2833,  46.4641, &rA09.n1      );
  make_pt(   37.3641,  13.0968,  45.9007, &rA09.n3      );
  make_pt(   37.5032,  14.1288,  46.7300, &rA09.c2      );
  make_pt(   37.9570,  13.3377,  44.7113, &rA09.c4      );
  make_pt(   38.6397,  14.4660,  44.3267, &rA09.c5      );
  make_pt(   38.7473,  15.5229,  45.2609, &rA09.c6      );
  make_pt(   39.3720,  16.6649,  45.0297, &rA09._.A.n6  );
  make_pt(   39.1079,  14.3351,  43.0223, &rA09._.A.n7  );
  make_pt(   38.0132,  12.4868,  43.6280, &rA09._.A.n9  );
  make_pt(   38.7058,  13.1402,  42.6620, &rA09._.A.c8  );
  make_pt(   37.0731,  14.0857,  47.7306, &rA09._.A.h2  );
  make_pt(   39.8113,  16.8281,  44.1350, &rA09._.A.h61 );
  make_pt(   39.4100,  17.3741,  45.7478, &rA09._.A.h62 );
  make_pt(   39.0412,  12.9660,  41.6397, &rA09._.A.h8  );

  rA10.type = 'A';
  make_tfo(   0.7063,   0.6317,  -0.3196,
             -0.0403,  -0.4149,  -0.9090,
             -0.7068,   0.6549,  -0.2676,
              6.4402, -52.1496,  30.8246, &rA10.dgf_base_tfo );
  make_tfo(   0.7529,   0.1548,   0.6397,
              0.2952,  -0.9481,  -0.1180,
              0.5882,   0.2777,  -0.7595,
            -58.8919, -11.3095,   6.0866, &rA10.p_o3_275_tfo );
  make_tfo(  -0.0239,   0.9667,  -0.2546,
              0.9731,  -0.0359,  -0.2275,
             -0.2290,  -0.2532,  -0.9399,
              3.5401, -29.7913,  52.2796, &rA10.p_o3_180_tfo );
  make_tfo(  -0.8912,  -0.4531,   0.0242,
             -0.1183,   0.1805,  -0.9764,
              0.4380,  -0.8730,  -0.2145,
             19.9023,  54.8054,  15.2799, &rA10.p_o3_60_tfo  );
  make_pt(   41.8210,   8.3880,  43.5890, &rA10.p       );
  make_pt(   42.5400,   8.0450,  44.8330, &rA10.o1p     );
  make_pt(   42.2470,   9.6920,  42.9910, &rA10.o2p     );
  make_pt(   40.2550,   8.2030,  43.7340, &rA10.o5_     );
  make_pt(   39.4850,   8.9301,  44.6977, &rA10.c5_     );
  make_pt(   39.0638,   9.8199,  44.2296, &rA10.h5_     );
  make_pt(   40.0757,   9.0713,  45.6029, &rA10.h5__    );
  make_pt(   38.3102,   8.0414,  45.0789, &rA10.c4_     );
  make_pt(   37.7099,   7.8166,  44.1973, &rA10.h4_     );
  make_pt(   38.8012,   6.8321,  45.6380, &rA10.o4_     );
  make_pt(   38.2431,   6.6413,  46.9529, &rA10.c1_     );
  make_pt(   37.3505,   6.0262,  46.8385, &rA10.h1_     );
  make_pt(   37.8484,   8.0156,  47.4214, &rA10.c2_     );
  make_pt(   38.7381,   8.5406,  47.7690, &rA10.h2__    );
  make_pt(   36.8286,   8.0368,  48.3701, &rA10.o2_     );
  make_pt(   36.8392,   7.3063,  48.9929, &rA10.h2_     );
  make_pt(   37.3576,   8.6512,  46.1132, &rA10.c3_     );
  make_pt(   37.5207,   9.7275,  46.1671, &rA10.h3_     );
  make_pt(   35.9985,   8.2392,  45.9032, &rA10.o3_     );
  make_pt(   39.9117,   2.2278,  48.8527, &rA10.n1      );
  make_pt(   38.6207,   3.6941,  47.4757, &rA10.n3      );
  make_pt(   38.9872,   2.4888,  47.9057, &rA10.c2      );
  make_pt(   39.2961,   4.6720,  48.1174, &rA10.c4      );
  make_pt(   40.2546,   4.5307,  49.0912, &rA10.c5      );
  make_pt(   40.5932,   3.2189,  49.4985, &rA10.c6      );
  make_pt(   41.4938,   2.9317,  50.4229, &rA10._.A.n6  );
  make_pt(   40.7195,   5.7755,  49.5060, &rA10._.A.n7  );
  make_pt(   39.1730,   6.0305,  47.9170, &rA10._.A.n9  );
  make_pt(   40.0413,   6.6250,  48.7728, &rA10._.A.c8  );
  make_pt(   38.5257,   1.5960,  47.4838, &rA10._.A.h2  );
  make_pt(   41.9907,   3.6753,  50.8921, &rA10._.A.h61 );
  make_pt(   41.6848,   1.9687,  50.6599, &rA10._.A.h62 );
  make_pt(   40.3571,   7.6321,  49.0452, &rA10._.A.h8  );

  rAs[0] = &rA10;
  rAs[1] = &rA09;
  rAs[2] = &rA08;
  rAs[3] = &rA07;
  rAs[4] = &rA06;
  rAs[5] = &rA05;
  rAs[6] = &rA04;
  rAs[7] = &rA03;
  rAs[8] = &rA02;
  rAs[9] = &rA01;
  rAs[10]= 0;
}

void init_C()
{
  rC.type = 'C';
  make_tfo(  -0.0359,  -0.8071,   0.5894,
             -0.2669,   0.5761,   0.7726,
             -0.9631,  -0.1296,  -0.2361,
              0.1584,   8.3434,   0.5434, &rC.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rC.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rC.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rC.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rC.p         );
  make_pt(    1.4950,  -7.6230,   3.4770, &rC.o1p       );
  make_pt(    2.9490,  -9.4640,   4.3740, &rC.o2p       );
  make_pt(    3.9730,  -7.5950,   3.0340, &rC.o5_       );
  make_pt(    5.2430,  -8.2420,   2.8260, &rC.c5_       );
  make_pt(    5.1974,  -8.8497,   1.9223, &rC.h5_       );
  make_pt(    5.5548,  -8.7348,   3.7469, &rC.h5__      );
  make_pt(    6.3140,  -7.2060,   2.5510, &rC.c4_       );
  make_pt(    7.2954,  -7.6762,   2.4898, &rC.h4_       );
  make_pt(    6.0140,  -6.5420,   1.2890, &rC.o4_       );
  make_pt(    6.4190,  -5.1840,   1.3620, &rC.c1_       );
  make_pt(    7.1608,  -5.0495,   0.5747, &rC.h1_       );
  make_pt(    7.0760,  -4.9560,   2.7270, &rC.c2_       );
  make_pt(    6.7770,  -3.9803,   3.1099, &rC.h2__      );
  make_pt(    8.4500,  -5.1930,   2.5810, &rC.o2_       );
  make_pt(    8.8309,  -4.8755,   1.7590, &rC.h2_       );
  make_pt(    6.4060,  -6.0590,   3.5580, &rC.c3_       );
  make_pt(    5.4021,  -5.7313,   3.8281, &rC.h3_       );
  make_pt(    7.1570,  -6.4240,   4.7070, &rC.o3_       );
  make_pt(    5.2170,  -4.3260,   1.1690, &rC.n1        );
  make_pt(    4.2960,  -2.2560,   0.6290, &rC.n3        );
  make_pt(    5.4330,  -3.0200,   0.7990, &rC.c2        );
  make_pt(    2.9930,  -2.6780,   0.7940, &rC.c4        );
  make_pt(    2.8670,  -4.0630,   1.1830, &rC.c5        );
  make_pt(    3.9570,  -4.8300,   1.3550, &rC.c6        );
  make_pt(    2.0187,  -1.8047,   0.5874, &rC._.C.n4    );
  make_pt(    6.5470,  -2.5560,   0.6290, &rC._.C.o2    );
  make_pt(    1.0684,  -2.1236,   0.7109, &rC._.C.h41   );
  make_pt(    2.2344,  -0.8560,   0.3162, &rC._.C.h42   );
  make_pt(    1.8797,  -4.4972,   1.3404, &rC._.C.h5    );
  make_pt(    3.8479,  -5.8742,   1.6480, &rC._.C.h6    );

  rC01.type = 'C';
  make_tfo(  -0.0137,  -0.8012,   0.5983,
             -0.2523,   0.5817,   0.7733,
             -0.9675,  -0.1404,  -0.2101,
              0.2031,   8.3874,   0.4228, &rC01.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rC01.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rC01.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rC01.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rC01.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rC01.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rC01.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rC01.o5_     );
  make_pt(    5.2416,  -8.2422,   2.8181, &rC01.c5_     );
  make_pt(    5.2050,  -8.8128,   1.8901, &rC01.h5_     );
  make_pt(    5.5368,  -8.7738,   3.7227, &rC01.h5__    );
  make_pt(    6.3232,  -7.2037,   2.6002, &rC01.c4_     );
  make_pt(    7.3048,  -7.6757,   2.5577, &rC01.h4_     );
  make_pt(    6.0635,  -6.5092,   1.3456, &rC01.o4_     );
  make_pt(    6.4697,  -5.1547,   1.4629, &rC01.c1_     );
  make_pt(    7.2354,  -5.0043,   0.7018, &rC01.h1_     );
  make_pt(    7.0856,  -4.9610,   2.8521, &rC01.c2_     );
  make_pt(    6.7777,  -3.9935,   3.2487, &rC01.h2__    );
  make_pt(    8.4627,  -5.1992,   2.7423, &rC01.o2_     );
  make_pt(    8.8693,  -4.8638,   1.9399, &rC01.h2_     );
  make_pt(    6.3877,  -6.0809,   3.6362, &rC01.c3_     );
  make_pt(    5.3770,  -5.7562,   3.8834, &rC01.h3_     );
  make_pt(    7.1024,  -6.4754,   4.7985, &rC01.o3_     );
  make_pt(    5.2764,  -4.2883,   1.2538, &rC01.n1      );
  make_pt(    4.3777,  -2.2062,   0.7229, &rC01.n3      );
  make_pt(    5.5069,  -2.9779,   0.9088, &rC01.c2      );
  make_pt(    3.0693,  -2.6246,   0.8500, &rC01.c4      );
  make_pt(    2.9279,  -4.0146,   1.2149, &rC01.c5      );
  make_pt(    4.0101,  -4.7892,   1.4017, &rC01.c6      );
  make_pt(    2.1040,  -1.7437,   0.6331, &rC01._.C.n4  );
  make_pt(    6.6267,  -2.5166,   0.7728, &rC01._.C.o2  );
  make_pt(    1.1496,  -2.0600,   0.7287, &rC01._.C.h41 );
  make_pt(    2.3303,  -0.7921,   0.3815, &rC01._.C.h42 );
  make_pt(    1.9353,  -4.4465,   1.3419, &rC01._.C.h5  );
  make_pt(    3.8895,  -5.8371,   1.6762, &rC01._.C.h6  );

  rC02.type = 'C';
  make_tfo(   0.5141,   0.0246,   0.8574,
             -0.5547,  -0.7529,   0.3542,
              0.6542,  -0.6577,  -0.3734,
             -9.1111,  -3.4598,  -3.2939, &rC02.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rC02.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rC02.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rC02.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rC02.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rC02.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rC02.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rC02.o5_     );
  make_pt(    4.3825,  -6.6585,   4.0489, &rC02.c5_     );
  make_pt(    4.6841,  -7.2019,   4.9443, &rC02.h5_     );
  make_pt(    3.6189,  -5.8889,   4.1625, &rC02.h5__    );
  make_pt(    5.6255,  -5.9175,   3.5998, &rC02.c4_     );
  make_pt(    5.8732,  -5.1228,   4.3034, &rC02.h4_     );
  make_pt(    6.7337,  -6.8605,   3.5222, &rC02.o4_     );
  make_pt(    7.5932,  -6.4923,   2.4548, &rC02.c1_     );
  make_pt(    8.5661,  -6.2983,   2.9064, &rC02.h1_     );
  make_pt(    7.0527,  -5.2012,   1.8322, &rC02.c2_     );
  make_pt(    7.1627,  -5.2525,   0.7490, &rC02.h2__    );
  make_pt(    7.6666,  -4.1249,   2.4880, &rC02.o2_     );
  make_pt(    8.5944,  -4.2543,   2.6981, &rC02.h2_     );
  make_pt(    5.5661,  -5.3029,   2.2009, &rC02.c3_     );
  make_pt(    5.0841,  -6.0018,   1.5172, &rC02.h3_     );
  make_pt(    4.9062,  -4.0452,   2.2042, &rC02.o3_     );
  make_pt(    7.6298,  -7.6136,   1.4752, &rC02.n1      );
  make_pt(    8.6945,  -8.7046,  -0.2857, &rC02.n3      );
  make_pt(    8.6943,  -7.6514,   0.6066, &rC02.c2      );
  make_pt(    7.7426,  -9.6987,  -0.3801, &rC02.c4      );
  make_pt(    6.6642,  -9.5742,   0.5722, &rC02.c5      );
  make_pt(    6.6391,  -8.5592,   1.4526, &rC02.c6      );
  make_pt(    7.9033, -10.6371,  -1.3010, &rC02._.C.n4  );
  make_pt(    9.5840,  -6.8186,   0.6136, &rC02._.C.o2  );
  make_pt(    7.2009, -11.3604,  -1.3619, &rC02._.C.h41 );
  make_pt(    8.7058, -10.6168,  -1.9140, &rC02._.C.h42 );
  make_pt(    5.8585, -10.3083,   0.5822, &rC02._.C.h5  );
  make_pt(    5.8197,  -8.4773,   2.1667, &rC02._.C.h6  );

  rC03.type = 'C';
  make_tfo(  -0.4993,   0.0476,   0.8651,
              0.8078,  -0.3353,   0.4847,
              0.3132,   0.9409,   0.1290,
              6.2989,  -5.2303,  -3.8577, &rC03.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rC03.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rC03.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rC03.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rC03.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rC03.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rC03.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rC03.o5_     );
  make_pt(    3.9938,  -6.7042,   1.9023, &rC03.c5_     );
  make_pt(    3.2332,  -5.9343,   2.0319, &rC03.h5_     );
  make_pt(    3.9666,  -7.2863,   0.9812, &rC03.h5__    );
  make_pt(    5.3098,  -5.9546,   1.8564, &rC03.c4_     );
  make_pt(    5.3863,  -5.3702,   0.9395, &rC03.h4_     );
  make_pt(    5.3851,  -5.0642,   3.0076, &rC03.o4_     );
  make_pt(    6.7315,  -4.9724,   3.4462, &rC03.c1_     );
  make_pt(    7.0033,  -3.9202,   3.3619, &rC03.h1_     );
  make_pt(    7.5997,  -5.8018,   2.4948, &rC03.c2_     );
  make_pt(    8.3627,  -6.3254,   3.0707, &rC03.h2__    );
  make_pt(    8.0410,  -4.9501,   1.4724, &rC03.o2_     );
  make_pt(    8.2781,  -4.0644,   1.7570, &rC03.h2_     );
  make_pt(    6.5701,  -6.8129,   1.9714, &rC03.c3_     );
  make_pt(    6.4186,  -7.5809,   2.7299, &rC03.h3_     );
  make_pt(    6.9357,  -7.3841,   0.7235, &rC03.o3_     );
  make_pt(    6.8024,  -5.4718,   4.8475, &rC03.n1      );
  make_pt(    7.9218,  -5.5700,   6.8877, &rC03.n3      );
  make_pt(    7.8908,  -5.0886,   5.5944, &rC03.c2      );
  make_pt(    6.9789,  -6.3827,   7.4823, &rC03.c4      );
  make_pt(    5.8742,  -6.7319,   6.6202, &rC03.c5      );
  make_pt(    5.8182,  -6.2769,   5.3570, &rC03.c6      );
  make_pt(    7.1702,  -6.7511,   8.7402, &rC03._.C.n4  );
  make_pt(    8.7747,  -4.3728,   5.1568, &rC03._.C.o2  );
  make_pt(    6.4741,  -7.3461,   9.1662, &rC03._.C.h41 );
  make_pt(    7.9889,  -6.4396,   9.2429, &rC03._.C.h42 );
  make_pt(    5.0736,  -7.3713,   6.9922, &rC03._.C.h5  );
  make_pt(    4.9784,  -6.5473,   4.7170, &rC03._.C.h6  );

  rC04.type = 'C';
  make_tfo(  -0.5669,  -0.8012,   0.1918,
             -0.8129,   0.5817,   0.0273,
             -0.1334,  -0.1404,  -0.9811,
             -0.3279,   8.3874,   0.3355, &rC04.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rC04.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rC04.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rC04.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rC04.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rC04.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rC04.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rC04.o5_     );
  make_pt(    5.2416,  -8.2422,   2.8181, &rC04.c5_     );
  make_pt(    5.2050,  -8.8128,   1.8901, &rC04.h5_     );
  make_pt(    5.5368,  -8.7738,   3.7227, &rC04.h5__    );
  make_pt(    6.3232,  -7.2037,   2.6002, &rC04.c4_     );
  make_pt(    7.3048,  -7.6757,   2.5577, &rC04.h4_     );
  make_pt(    6.0635,  -6.5092,   1.3456, &rC04.o4_     );
  make_pt(    6.4697,  -5.1547,   1.4629, &rC04.c1_     );
  make_pt(    7.2354,  -5.0043,   0.7018, &rC04.h1_     );
  make_pt(    7.0856,  -4.9610,   2.8521, &rC04.c2_     );
  make_pt(    6.7777,  -3.9935,   3.2487, &rC04.h2__    );
  make_pt(    8.4627,  -5.1992,   2.7423, &rC04.o2_     );
  make_pt(    8.8693,  -4.8638,   1.9399, &rC04.h2_     );
  make_pt(    6.3877,  -6.0809,   3.6362, &rC04.c3_     );
  make_pt(    5.3770,  -5.7562,   3.8834, &rC04.h3_     );
  make_pt(    7.1024,  -6.4754,   4.7985, &rC04.o3_     );
  make_pt(    5.2764,  -4.2883,   1.2538, &rC04.n1      );
  make_pt(    3.8961,  -3.0896,  -0.1893, &rC04.n3      );
  make_pt(    5.0095,  -3.8907,  -0.0346, &rC04.c2      );
  make_pt(    3.0480,  -2.6632,   0.8116, &rC04.c4      );
  make_pt(    3.4093,  -3.1310,   2.1292, &rC04.c5      );
  make_pt(    4.4878,  -3.9124,   2.3088, &rC04.c6      );
  make_pt(    2.0216,  -1.8941,   0.4804, &rC04._.C.n4  );
  make_pt(    5.7005,  -4.2164,  -0.9842, &rC04._.C.o2  );
  make_pt(    1.4067,  -1.5873,   1.2205, &rC04._.C.h41 );
  make_pt(    1.8721,  -1.6319,  -0.4835, &rC04._.C.h42 );
  make_pt(    2.8048,  -2.8507,   2.9918, &rC04._.C.h5  );
  make_pt(    4.7491,  -4.2593,   3.3085, &rC04._.C.h6  );

  rC05.type = 'C';
  make_tfo(  -0.6298,   0.0246,   0.7763,
             -0.5226,  -0.7529,  -0.4001,
              0.5746,  -0.6577,   0.4870,
             -0.0208,  -3.4598,  -9.6882, &rC05.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rC05.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rC05.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rC05.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rC05.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rC05.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rC05.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rC05.o5_     );
  make_pt(    4.3825,  -6.6585,   4.0489, &rC05.c5_     );
  make_pt(    4.6841,  -7.2019,   4.9443, &rC05.h5_     );
  make_pt(    3.6189,  -5.8889,   4.1625, &rC05.h5__    );
  make_pt(    5.6255,  -5.9175,   3.5998, &rC05.c4_     );
  make_pt(    5.8732,  -5.1228,   4.3034, &rC05.h4_     );
  make_pt(    6.7337,  -6.8605,   3.5222, &rC05.o4_     );
  make_pt(    7.5932,  -6.4923,   2.4548, &rC05.c1_     );
  make_pt(    8.5661,  -6.2983,   2.9064, &rC05.h1_     );
  make_pt(    7.0527,  -5.2012,   1.8322, &rC05.c2_     );
  make_pt(    7.1627,  -5.2525,   0.7490, &rC05.h2__    );
  make_pt(    7.6666,  -4.1249,   2.4880, &rC05.o2_     );
  make_pt(    8.5944,  -4.2543,   2.6981, &rC05.h2_     );
  make_pt(    5.5661,  -5.3029,   2.2009, &rC05.c3_     );
  make_pt(    5.0841,  -6.0018,   1.5172, &rC05.h3_     );
  make_pt(    4.9062,  -4.0452,   2.2042, &rC05.o3_     );
  make_pt(    7.6298,  -7.6136,   1.4752, &rC05.n1      );
  make_pt(    8.5977,  -9.5977,   0.7329, &rC05.n3      );
  make_pt(    8.5951,  -8.5745,   1.6594, &rC05.c2      );
  make_pt(    7.7372,  -9.7371,  -0.3364, &rC05.c4      );
  make_pt(    6.7596,  -8.6801,  -0.4476, &rC05.c5      );
  make_pt(    6.7338,  -7.6721,   0.4408, &rC05.c6      );
  make_pt(    7.8849, -10.7881,  -1.1289, &rC05._.C.n4  );
  make_pt(    9.3993,  -8.5377,   2.5743, &rC05._.C.o2  );
  make_pt(    7.2499, -10.8809,  -1.9088, &rC05._.C.h41 );
  make_pt(    8.6122, -11.4649,  -0.9468, &rC05._.C.h42 );
  make_pt(    6.0317,  -8.6941,  -1.2588, &rC05._.C.h5  );
  make_pt(    5.9901,  -6.8809,   0.3459, &rC05._.C.h6  );

  rC06.type = 'C';
  make_tfo(  -0.9837,   0.0476,  -0.1733,
             -0.1792,  -0.3353,   0.9249,
             -0.0141,   0.9409,   0.3384,
              5.7793,  -5.2303,   4.5997, &rC06.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rC06.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rC06.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rC06.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rC06.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rC06.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rC06.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rC06.o5_     );
  make_pt(    3.9938,  -6.7042,   1.9023, &rC06.c5_     );
  make_pt(    3.2332,  -5.9343,   2.0319, &rC06.h5_     );
  make_pt(    3.9666,  -7.2863,   0.9812, &rC06.h5__    );
  make_pt(    5.3098,  -5.9546,   1.8564, &rC06.c4_     );
  make_pt(    5.3863,  -5.3702,   0.9395, &rC06.h4_     );
  make_pt(    5.3851,  -5.0642,   3.0076, &rC06.o4_     );
  make_pt(    6.7315,  -4.9724,   3.4462, &rC06.c1_     );
  make_pt(    7.0033,  -3.9202,   3.3619, &rC06.h1_     );
  make_pt(    7.5997,  -5.8018,   2.4948, &rC06.c2_     );
  make_pt(    8.3627,  -6.3254,   3.0707, &rC06.h2__    );
  make_pt(    8.0410,  -4.9501,   1.4724, &rC06.o2_     );
  make_pt(    8.2781,  -4.0644,   1.7570, &rC06.h2_     );
  make_pt(    6.5701,  -6.8129,   1.9714, &rC06.c3_     );
  make_pt(    6.4186,  -7.5809,   2.7299, &rC06.h3_     );
  make_pt(    6.9357,  -7.3841,   0.7235, &rC06.o3_     );
  make_pt(    6.8024,  -5.4718,   4.8475, &rC06.n1      );
  make_pt(    6.6920,  -5.0495,   7.1354, &rC06.n3      );
  make_pt(    6.6201,  -4.5500,   5.8506, &rC06.c2      );
  make_pt(    6.9254,  -6.3614,   7.4926, &rC06.c4      );
  make_pt(    7.1046,  -7.2543,   6.3718, &rC06.c5      );
  make_pt(    7.0391,  -6.7951,   5.1106, &rC06.c6      );
  make_pt(    6.9614,  -6.6648,   8.7815, &rC06._.C.n4  );
  make_pt(    6.4083,  -3.3696,   5.6340, &rC06._.C.o2  );
  make_pt(    7.1329,  -7.6280,   9.0324, &rC06._.C.h41 );
  make_pt(    6.8204,  -5.9469,   9.4777, &rC06._.C.h42 );
  make_pt(    7.2954,  -8.3135,   6.5440, &rC06._.C.h5  );
  make_pt(    7.1753,  -7.4798,   4.2735, &rC06._.C.h6  );

  rC07.type = 'C';
  make_tfo(   0.0033,   0.2720,  -0.9623,
              0.3013,  -0.9179,  -0.2584,
             -0.9535,  -0.2891,  -0.0850,
             43.0403,  13.7233,  34.5710, &rC07.dgf_base_tfo );
  make_tfo(   0.9187,   0.2887,   0.2694,
              0.0302,  -0.7316,   0.6811,
              0.3938,  -0.6176,  -0.6808,
            -48.4330,  26.3254,  13.6383, &rC07.p_o3_275_tfo );
  make_tfo(  -0.1504,   0.7744,  -0.6145,
              0.7581,   0.4893,   0.4311,
              0.6345,  -0.4010,  -0.6607,
            -31.9784, -13.4285,  44.9650, &rC07.p_o3_180_tfo );
  make_tfo(  -0.6236,  -0.7810,  -0.0337,
             -0.6890,   0.5694,  -0.4484,
              0.3694,  -0.2564,  -0.8932,
             12.1105,  30.8774,  46.0946, &rC07.p_o3_60_tfo  );
  make_pt(   33.3400,  11.0980,  46.1750, &rC07.p       );
  make_pt(   34.5130,  10.2320,  46.4660, &rC07.o1p     );
  make_pt(   33.4130,  12.3960,  46.9340, &rC07.o2p     );
  make_pt(   31.9810,  10.3390,  46.4820, &rC07.o5_     );
  make_pt(   30.8152,  11.1619,  46.2003, &rC07.c5_     );
  make_pt(   30.4519,  10.9454,  45.1957, &rC07.h5_     );
  make_pt(   31.0379,  12.2016,  46.4400, &rC07.h5__    );
  make_pt(   29.7081,  10.7448,  47.1428, &rC07.c4_     );
  make_pt(   28.8710,  11.4416,  47.0982, &rC07.h4_     );
  make_pt(   29.2550,   9.4394,  46.8162, &rC07.o4_     );
  make_pt(   29.3907,   8.5625,  47.9460, &rC07.c1_     );
  make_pt(   28.4416,   8.5669,  48.4819, &rC07.h1_     );
  make_pt(   30.4468,   9.2031,  48.7952, &rC07.c2_     );
  make_pt(   31.4222,   8.9651,  48.3709, &rC07.h2__    );
  make_pt(   30.3701,   8.9157,  50.1624, &rC07.o2_     );
  make_pt(   30.0652,   8.0304,  50.3740, &rC07.h2_     );
  make_pt(   30.1622,  10.6879,  48.6120, &rC07.c3_     );
  make_pt(   31.0952,  11.2399,  48.7254, &rC07.h3_     );
  make_pt(   29.1076,  11.1535,  49.4702, &rC07.o3_     );
  make_pt(   29.7883,   7.2209,  47.5235, &rC07.n1      );
  make_pt(   29.1825,   5.0438,  46.8275, &rC07.n3      );
  make_pt(   28.8008,   6.2912,  47.2263, &rC07.c2      );
  make_pt(   30.4888,   4.6890,  46.7186, &rC07.c4      );
  make_pt(   31.5034,   5.6405,  47.0249, &rC07.c5      );
  make_pt(   31.1091,   6.8691,  47.4156, &rC07.c6      );
  make_pt(   30.8109,   3.4584,  46.3336, &rC07._.C.n4  );
  make_pt(   27.6171,   6.5989,  47.3189, &rC07._.C.o2  );
  make_pt(   31.7923,   3.2301,  46.2638, &rC07._.C.h41 );
  make_pt(   30.0880,   2.7857,  46.1215, &rC07._.C.h42 );
  make_pt(   32.5542,   5.3634,  46.9395, &rC07._.C.h5  );
  make_pt(   31.8523,   7.6279,  47.6603, &rC07._.C.h6  );

  rC08.type = 'C';
  make_tfo(   0.0797,  -0.6026,  -0.7941,
              0.7939,   0.5201,  -0.3150,
              0.6028,  -0.6054,   0.5198,
            -36.8341,  41.5293,   1.6628, &rC08.dgf_base_tfo );
  make_tfo(   0.9187,   0.2887,   0.2694,
              0.0302,  -0.7316,   0.6811,
              0.3938,  -0.6176,  -0.6808,
            -48.4330,  26.3254,  13.6383, &rC08.p_o3_275_tfo );
  make_tfo(  -0.1504,   0.7744,  -0.6145,
              0.7581,   0.4893,   0.4311,
              0.6345,  -0.4010,  -0.6607,
            -31.9784, -13.4285,  44.9650, &rC08.p_o3_180_tfo );
  make_tfo(  -0.6236,  -0.7810,  -0.0337,
             -0.6890,   0.5694,  -0.4484,
              0.3694,  -0.2564,  -0.8932,
             12.1105,  30.8774,  46.0946, &rC08.p_o3_60_tfo  );
  make_pt(   33.3400,  11.0980,  46.1750, &rC08.p       );
  make_pt(   34.5130,  10.2320,  46.4660, &rC08.o1p     );
  make_pt(   33.4130,  12.3960,  46.9340, &rC08.o2p     );
  make_pt(   31.9810,  10.3390,  46.4820, &rC08.o5_     );
  make_pt(   31.8779,   9.9369,  47.8760, &rC08.c5_     );
  make_pt(   31.3239,  10.6931,  48.4322, &rC08.h5_     );
  make_pt(   32.8647,   9.6624,  48.2489, &rC08.h5__    );
  make_pt(   31.0429,   8.6773,  47.9401, &rC08.c4_     );
  make_pt(   31.0779,   8.2331,  48.9349, &rC08.h4_     );
  make_pt(   29.6956,   8.9669,  47.5983, &rC08.o4_     );
  make_pt(   29.2784,   8.1700,  46.4782, &rC08.c1_     );
  make_pt(   28.8006,   7.2731,  46.8722, &rC08.h1_     );
  make_pt(   30.5544,   7.7940,  45.7875, &rC08.c2_     );
  make_pt(   30.8837,   8.6410,  45.1856, &rC08.h2__    );
  make_pt(   30.5100,   6.6007,  45.0582, &rC08.o2_     );
  make_pt(   29.6694,   6.4168,  44.6326, &rC08.h2_     );
  make_pt(   31.5146,   7.5954,  46.9527, &rC08.c3_     );
  make_pt(   32.5255,   7.8261,  46.6166, &rC08.h3_     );
  make_pt(   31.3876,   6.2951,  47.5516, &rC08.o3_     );
  make_pt(   28.3976,   8.9302,  45.5933, &rC08.n1      );
  make_pt(   26.2155,   9.6135,  44.9910, &rC08.n3      );
  make_pt(   27.0281,   8.8961,  45.8192, &rC08.c2      );
  make_pt(   26.7044,  10.3489,  43.9595, &rC08.c4      );
  make_pt(   28.1088,  10.3837,  43.7247, &rC08.c5      );
  make_pt(   28.8978,   9.6708,  44.5535, &rC08.c6      );
  make_pt(   25.8715,  11.0249,  43.1749, &rC08._.C.n4  );
  make_pt(   26.5733,   8.2371,  46.7484, &rC08._.C.o2  );
  make_pt(   26.2707,  11.5609,  42.4177, &rC08._.C.h41 );
  make_pt(   24.8760,  10.9939,  43.3427, &rC08._.C.h42 );
  make_pt(   28.5089,  10.9722,  42.8990, &rC08._.C.h5  );
  make_pt(   29.9782,   9.6687,  44.4097, &rC08._.C.h6  );

  rC09.type = 'C';
  make_tfo(   0.8727,   0.4760,  -0.1091,
             -0.4188,   0.6148,  -0.6682,
             -0.2510,   0.6289,   0.7359,
             -8.1687, -52.0761, -25.0726, &rC09.dgf_base_tfo );
  make_tfo(   0.9187,   0.2887,   0.2694,
              0.0302,  -0.7316,   0.6811,
              0.3938,  -0.6176,  -0.6808,
            -48.4330,  26.3254,  13.6383, &rC09.p_o3_275_tfo );
  make_tfo(  -0.1504,   0.7744,  -0.6145,
              0.7581,   0.4893,   0.4311,
              0.6345,  -0.4010,  -0.6607,
            -31.9784, -13.4285,  44.9650, &rC09.p_o3_180_tfo );
  make_tfo(  -0.6236,  -0.7810,  -0.0337,
             -0.6890,   0.5694,  -0.4484,
              0.3694,  -0.2564,  -0.8932,
             12.1105,  30.8774,  46.0946, &rC09.p_o3_60_tfo  );
  make_pt(   33.3400,  11.0980,  46.1750, &rC09.p       );
  make_pt(   34.5130,  10.2320,  46.4660, &rC09.o1p     );
  make_pt(   33.4130,  12.3960,  46.9340, &rC09.o2p     );
  make_pt(   31.9810,  10.3390,  46.4820, &rC09.o5_     );
  make_pt(   30.8152,  11.1619,  46.2003, &rC09.c5_     );
  make_pt(   30.4519,  10.9454,  45.1957, &rC09.h5_     );
  make_pt(   31.0379,  12.2016,  46.4400, &rC09.h5__    );
  make_pt(   29.7081,  10.7448,  47.1428, &rC09.c4_     );
  make_pt(   29.4506,   9.6945,  47.0059, &rC09.h4_     );
  make_pt(   30.1045,  10.9634,  48.4885, &rC09.o4_     );
  make_pt(   29.1794,  11.8418,  49.1490, &rC09.c1_     );
  make_pt(   28.4388,  11.2210,  49.6533, &rC09.h1_     );
  make_pt(   28.5211,  12.6008,  48.0367, &rC09.c2_     );
  make_pt(   29.1947,  13.3949,  47.7147, &rC09.h2__    );
  make_pt(   27.2316,  13.0683,  48.3134, &rC09.o2_     );
  make_pt(   27.0851,  13.3391,  49.2227, &rC09.h2_     );
  make_pt(   28.4131,  11.5507,  46.9391, &rC09.c3_     );
  make_pt(   28.4451,  12.0512,  45.9713, &rC09.h3_     );
  make_pt(   27.2707,  10.6955,  47.1097, &rC09.o3_     );
  make_pt(   29.8751,  12.7405,  50.0682, &rC09.n1      );
  make_pt(   30.7172,  13.1841,  52.2328, &rC09.n3      );
  make_pt(   30.0617,  12.3404,  51.3847, &rC09.c2      );
  make_pt(   31.1834,  14.3941,  51.8297, &rC09.c4      );
  make_pt(   30.9913,  14.8074,  50.4803, &rC09.c5      );
  make_pt(   30.3434,  13.9610,  49.6548, &rC09.c6      );
  make_pt(   31.8090,  15.1847,  52.6957, &rC09._.C.n4  );
  make_pt(   29.6470,  11.2494,  51.7616, &rC09._.C.o2  );
  make_pt(   32.1422,  16.0774,  52.3606, &rC09._.C.h41 );
  make_pt(   31.9392,  14.8893,  53.6527, &rC09._.C.h42 );
  make_pt(   31.3632,  15.7771,  50.1491, &rC09._.C.h5  );
  make_pt(   30.1742,  14.2374,  48.6141, &rC09._.C.h6  );

  rC10.type = 'C';
  make_tfo(   0.1549,   0.8710,  -0.4663,
              0.6768,  -0.4374,  -0.5921,
             -0.7197,  -0.2239,  -0.6572,
             25.2447, -14.1920,  50.3201, &rC10.dgf_base_tfo );
  make_tfo(   0.9187,   0.2887,   0.2694,
              0.0302,  -0.7316,   0.6811,
              0.3938,  -0.6176,  -0.6808,
            -48.4330,  26.3254,  13.6383, &rC10.p_o3_275_tfo );
  make_tfo(  -0.1504,   0.7744,  -0.6145,
              0.7581,   0.4893,   0.4311,
              0.6345,  -0.4010,  -0.6607,
            -31.9784, -13.4285,  44.9650, &rC10.p_o3_180_tfo );
  make_tfo(  -0.6236,  -0.7810,  -0.0337,
             -0.6890,   0.5694,  -0.4484,
              0.3694,  -0.2564,  -0.8932,
             12.1105,  30.8774,  46.0946, &rC10.p_o3_60_tfo  );
  make_pt(   33.3400,  11.0980,  46.1750, &rC10.p       );
  make_pt(   34.5130,  10.2320,  46.4660, &rC10.o1p     );
  make_pt(   33.4130,  12.3960,  46.9340, &rC10.o2p     );
  make_pt(   31.9810,  10.3390,  46.4820, &rC10.o5_     );
  make_pt(   31.8779,   9.9369,  47.8760, &rC10.c5_     );
  make_pt(   31.3239,  10.6931,  48.4322, &rC10.h5_     );
  make_pt(   32.8647,   9.6624,  48.2489, &rC10.h5__    );
  make_pt(   31.0429,   8.6773,  47.9401, &rC10.c4_     );
  make_pt(   30.0440,   8.8473,  47.5383, &rC10.h4_     );
  make_pt(   31.6749,   7.6351,  47.2119, &rC10.o4_     );
  make_pt(   31.9159,   6.5022,  48.0616, &rC10.c1_     );
  make_pt(   31.0691,   5.8243,  47.9544, &rC10.h1_     );
  make_pt(   31.9300,   7.0685,  49.4493, &rC10.c2_     );
  make_pt(   32.9024,   7.5288,  49.6245, &rC10.h2__    );
  make_pt(   31.5672,   6.1750,  50.4632, &rC10.o2_     );
  make_pt(   31.8416,   5.2663,  50.3200, &rC10.h2_     );
  make_pt(   30.8618,   8.1514,  49.3749, &rC10.c3_     );
  make_pt(   31.1122,   8.9396,  50.0850, &rC10.h3_     );
  make_pt(   29.5351,   7.6245,  49.5409, &rC10.o3_     );
  make_pt(   33.1890,   5.8629,  47.7343, &rC10.n1      );
  make_pt(   34.4004,   4.2636,  46.4828, &rC10.n3      );
  make_pt(   33.2062,   4.8497,  46.7851, &rC10.c2      );
  make_pt(   35.5600,   4.6374,  47.0822, &rC10.c4      );
  make_pt(   35.5444,   5.6751,  48.0577, &rC10.c5      );
  make_pt(   34.3565,   6.2450,  48.3432, &rC10.c6      );
  make_pt(   36.6977,   4.0305,  46.7598, &rC10._.C.n4  );
  make_pt(   32.1661,   4.5034,  46.2348, &rC10._.C.o2  );
  make_pt(   37.5405,   4.3347,  47.2259, &rC10._.C.h41 );
  make_pt(   36.7033,   3.2923,  46.0706, &rC10._.C.h42 );
  make_pt(   36.4713,   5.9811,  48.5428, &rC10._.C.h5  );
  make_pt(   34.2986,   7.0426,  49.0839, &rC10._.C.h6  );

  rCs[0] = &rC10;
  rCs[1] = &rC09;
  rCs[2] = &rC08;
  rCs[3] = &rC07;
  rCs[4] = &rC06;
  rCs[5] = &rC05;
  rCs[6] = &rC04;
  rCs[7] = &rC03;
  rCs[8] = &rC02;
  rCs[9] = &rC01;
  rCs[10]= 0;
}

void init_G()
{
  rG.type = 'G';
  make_tfo(  -0.0018,  -0.8207,   0.5714,
              0.2679,  -0.5509,  -0.7904,
              0.9634,   0.1517,   0.2209,
              0.0073,   8.4030,   0.6232, &rG.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rG.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rG.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rG.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rG.p         );
  make_pt(    1.6980,   7.6960,  -3.5570, &rG.o1p       );
  make_pt(    3.2260,   9.5010,  -4.4020, &rG.o2p       );
  make_pt(    4.1590,   7.6040,  -3.0340, &rG.o5_       );
  make_pt(    5.4550,   8.2120,  -2.8810, &rG.c5_       );
  make_pt(    5.4546,   8.8508,  -1.9978, &rG.h5_       );
  make_pt(    5.7588,   8.6625,  -3.8259, &rG.h5__      );
  make_pt(    6.4970,   7.1480,  -2.5980, &rG.c4_       );
  make_pt(    7.4896,   7.5919,  -2.5214, &rG.h4_       );
  make_pt(    6.1630,   6.4860,  -1.3440, &rG.o4_       );
  make_pt(    6.5400,   5.1200,  -1.4190, &rG.c1_       );
  make_pt(    7.2763,   4.9681,  -0.6297, &rG.h1_       );
  make_pt(    7.1940,   4.8830,  -2.7770, &rG.c2_       );
  make_pt(    6.8667,   3.9183,  -3.1647, &rG.h2__      );
  make_pt(    8.5860,   5.0910,  -2.6140, &rG.o2_       );
  make_pt(    8.9510,   4.7626,  -1.7890, &rG.h2_       );
  make_pt(    6.5720,   6.0040,  -3.6090, &rG.c3_       );
  make_pt(    5.5636,   5.7066,  -3.8966, &rG.h3_       );
  make_pt(    7.3801,   6.3562,  -4.7350, &rG.o3_       );
  make_pt(    4.7150,   0.4910,  -0.1360, &rG.n1        );
  make_pt(    6.3490,   2.1730,  -0.6020, &rG.n3        );
  make_pt(    5.9530,   0.9650,  -0.2670, &rG.c2        );
  make_pt(    5.2900,   2.9790,  -0.8260, &rG.c4        );
  make_pt(    3.9720,   2.6390,  -0.7330, &rG.c5        );
  make_pt(    3.6770,   1.3160,  -0.3660, &rG.c6        );
  make_pt(    6.8426,   0.0056,  -0.0019, &rG._.G.n2    );
  make_pt(    3.1660,   3.7290,  -1.0360, &rG._.G.n7    );
  make_pt(    5.3170,   4.2990,  -1.1930, &rG._.G.n9    );
  make_pt(    4.0100,   4.6780,  -1.2990, &rG._.G.c8    );
  make_pt(    2.4280,   0.8450,  -0.2360, &rG._.G.o6    );
  make_pt(    4.6151,  -0.4677,   0.1305, &rG._.G.h1    );
  make_pt(    6.6463,  -0.9463,   0.2729, &rG._.G.h21   );
  make_pt(    7.8170,   0.2642,  -0.0640, &rG._.G.h22   );
  make_pt(    3.4421,   5.5744,  -1.5482, &rG._.G.h8    );

  rG01.type = 'G';
  make_tfo(  -0.0043,  -0.8175,   0.5759,
              0.2617,  -0.5567,  -0.7884,
              0.9651,   0.1473,   0.2164,
              0.0359,   8.3929,   0.5532, &rG01.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rG01.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rG01.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rG01.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rG01.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rG01.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rG01.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rG01.o5_     );
  make_pt(    5.4352,   8.2183,  -2.7757, &rG01.c5_     );
  make_pt(    5.3830,   8.7883,  -1.8481, &rG01.h5_     );
  make_pt(    5.7729,   8.7436,  -3.6691, &rG01.h5__    );
  make_pt(    6.4830,   7.1518,  -2.5252, &rG01.c4_     );
  make_pt(    7.4749,   7.5972,  -2.4482, &rG01.h4_     );
  make_pt(    6.1626,   6.4620,  -1.2827, &rG01.o4_     );
  make_pt(    6.5431,   5.0992,  -1.3905, &rG01.c1_     );
  make_pt(    7.2871,   4.9328,  -0.6114, &rG01.h1_     );
  make_pt(    7.1852,   4.8935,  -2.7592, &rG01.c2_     );
  make_pt(    6.8573,   3.9363,  -3.1645, &rG01.h2__    );
  make_pt(    8.5780,   5.1025,  -2.6046, &rG01.o2_     );
  make_pt(    8.9516,   4.7577,  -1.7902, &rG01.h2_     );
  make_pt(    6.5522,   6.0300,  -3.5612, &rG01.c3_     );
  make_pt(    5.5420,   5.7356,  -3.8459, &rG01.h3_     );
  make_pt(    7.3487,   6.4089,  -4.6867, &rG01.o3_     );
  make_pt(    4.7442,   0.4514,  -0.1390, &rG01.n1      );
  make_pt(    6.3687,   2.1459,  -0.5926, &rG01.n3      );
  make_pt(    5.9795,   0.9335,  -0.2657, &rG01.c2      );
  make_pt(    5.3052,   2.9471,  -0.8125, &rG01.c4      );
  make_pt(    3.9891,   2.5987,  -0.7230, &rG01.c5      );
  make_pt(    3.7016,   1.2717,  -0.3647, &rG01.c6      );
  make_pt(    6.8745,  -0.0224,  -0.0058, &rG01._.G.n2  );
  make_pt(    3.1770,   3.6859,  -1.0198, &rG01._.G.n7  );
  make_pt(    5.3247,   4.2695,  -1.1710, &rG01._.G.n9  );
  make_pt(    4.0156,   4.6415,  -1.2759, &rG01._.G.c8  );
  make_pt(    2.4553,   0.7925,  -0.2390, &rG01._.G.o6  );
  make_pt(    4.6497,  -0.5095,   0.1212, &rG01._.G.h1  );
  make_pt(    6.6836,  -0.9771,   0.2627, &rG01._.G.h21 );
  make_pt(    7.8474,   0.2424,  -0.0653, &rG01._.G.h22 );
  make_pt(    3.4426,   5.5361,  -1.5199, &rG01._.G.h8  );

  rG02.type = 'G';
  make_tfo(   0.5566,   0.0449,   0.8296,
              0.5125,   0.7673,  -0.3854,
             -0.6538,   0.6397,   0.4041,
             -9.1161,  -3.7679,  -2.9968, &rG02.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rG02.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rG02.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rG02.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rG02.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rG02.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rG02.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rG02.o5_     );
  make_pt(    4.5778,   6.6594,  -4.0364, &rG02.c5_     );
  make_pt(    4.9220,   7.1963,  -4.9204, &rG02.h5_     );
  make_pt(    3.7996,   5.9091,  -4.1764, &rG02.h5__    );
  make_pt(    5.7873,   5.8869,  -3.5482, &rG02.c4_     );
  make_pt(    6.0405,   5.0875,  -4.2446, &rG02.h4_     );
  make_pt(    6.9135,   6.8036,  -3.4310, &rG02.o4_     );
  make_pt(    7.7293,   6.4084,  -2.3392, &rG02.c1_     );
  make_pt(    8.7078,   6.1815,  -2.7624, &rG02.h1_     );
  make_pt(    7.1305,   5.1418,  -1.7347, &rG02.c2_     );
  make_pt(    7.2040,   5.1982,  -0.6486, &rG02.h2__    );
  make_pt(    7.7417,   4.0392,  -2.3813, &rG02.o2_     );
  make_pt(    8.6785,   4.1443,  -2.5630, &rG02.h2_     );
  make_pt(    5.6666,   5.2728,  -2.1536, &rG02.c3_     );
  make_pt(    5.1747,   5.9805,  -1.4863, &rG02.h3_     );
  make_pt(    4.9997,   4.0086,  -2.1973, &rG02.o3_     );
  make_pt(   10.3245,   8.5459,   1.5467, &rG02.n1      );
  make_pt(    9.8051,   6.9432,  -0.1497, &rG02.n3      );
  make_pt(   10.5175,   7.4328,   0.8408, &rG02.c2      );
  make_pt(    8.7523,   7.7422,  -0.4228, &rG02.c4      );
  make_pt(    8.4257,   8.9060,   0.2099, &rG02.c5      );
  make_pt(    9.2665,   9.3242,   1.2540, &rG02.c6      );
  make_pt(   11.6077,   6.7966,   1.2752, &rG02._.G.n2  );
  make_pt(    7.2750,   9.4537,  -0.3428, &rG02._.G.n7  );
  make_pt(    7.7962,   7.5519,  -1.3859, &rG02._.G.n9  );
  make_pt(    6.9479,   8.6157,  -1.2771, &rG02._.G.c8  );
  make_pt(    9.0664,  10.4462,   1.9610, &rG02._.G.o6  );
  make_pt(   10.9838,   8.7524,   2.2697, &rG02._.G.h1  );
  make_pt(   12.2274,   7.0896,   2.0170, &rG02._.G.h21 );
  make_pt(   11.8502,   5.9398,   0.7984, &rG02._.G.h22 );
  make_pt(    6.0430,   8.9853,  -1.7594, &rG02._.G.h8  );

  rG03.type = 'G';
  make_tfo(  -0.5021,   0.0731,   0.8617,
             -0.8112,   0.3054,  -0.4986,
             -0.2996,  -0.9494,  -0.0940,
              6.4273,  -5.1944,  -3.7807, &rG03.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rG03.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rG03.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rG03.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rG03.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rG03.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rG03.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rG03.o5_     );
  make_pt(    4.1214,   6.7116,  -1.9049, &rG03.c5_     );
  make_pt(    3.3465,   5.9610,  -2.0607, &rG03.h5_     );
  make_pt(    4.0789,   7.2928,  -0.9837, &rG03.h5__    );
  make_pt(    5.4170,   5.9293,  -1.8186, &rG03.c4_     );
  make_pt(    5.4506,   5.3400,  -0.9023, &rG03.h4_     );
  make_pt(    5.5067,   5.0417,  -2.9703, &rG03.o4_     );
  make_pt(    6.8650,   4.9152,  -3.3612, &rG03.c1_     );
  make_pt(    7.1090,   3.8577,  -3.2603, &rG03.h1_     );
  make_pt(    7.7152,   5.7282,  -2.3894, &rG03.c2_     );
  make_pt(    8.5029,   6.2356,  -2.9463, &rG03.h2__    );
  make_pt(    8.1036,   4.8568,  -1.3419, &rG03.o2_     );
  make_pt(    8.3270,   3.9651,  -1.6184, &rG03.h2_     );
  make_pt(    6.7003,   6.7565,  -1.8911, &rG03.c3_     );
  make_pt(    6.5898,   7.5329,  -2.6482, &rG03.h3_     );
  make_pt(    7.0505,   7.2878,  -0.6105, &rG03.o3_     );
  make_pt(    9.6740,   4.7656,  -7.6614, &rG03.n1      );
  make_pt(    9.0739,   4.3013,  -5.3941, &rG03.n3      );
  make_pt(    9.8416,   4.2192,  -6.4581, &rG03.c2      );
  make_pt(    7.9885,   5.0632,  -5.6446, &rG03.c4      );
  make_pt(    7.6822,   5.6856,  -6.8194, &rG03.c5      );
  make_pt(    8.5831,   5.5215,  -7.8840, &rG03.c6      );
  make_pt(   10.9733,   3.5117,  -6.4286, &rG03._.G.n2  );
  make_pt(    6.4857,   6.3816,  -6.7035, &rG03._.G.n7  );
  make_pt(    6.9740,   5.3703,  -4.7760, &rG03._.G.n9  );
  make_pt(    6.1133,   6.1613,  -5.4808, &rG03._.G.c8  );
  make_pt(    8.4084,   6.0747,  -9.0933, &rG03._.G.o6  );
  make_pt(   10.3759,   4.5855,  -8.3504, &rG03._.G.h1  );
  make_pt(   11.6254,   3.3761,  -7.1879, &rG03._.G.h21 );
  make_pt(   11.1917,   3.0460,  -5.5593, &rG03._.G.h22 );
  make_pt(    5.1705,   6.6830,  -5.3167, &rG03._.G.h8  );

  rG04.type = 'G';
  make_tfo(  -0.5426,  -0.8175,   0.1929,
              0.8304,  -0.5567,  -0.0237,
              0.1267,   0.1473,   0.9809,
             -0.5075,   8.3929,   0.2229, &rG04.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rG04.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rG04.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rG04.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rG04.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rG04.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rG04.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rG04.o5_     );
  make_pt(    5.4352,   8.2183,  -2.7757, &rG04.c5_     );
  make_pt(    5.3830,   8.7883,  -1.8481, &rG04.h5_     );
  make_pt(    5.7729,   8.7436,  -3.6691, &rG04.h5__    );
  make_pt(    6.4830,   7.1518,  -2.5252, &rG04.c4_     );
  make_pt(    7.4749,   7.5972,  -2.4482, &rG04.h4_     );
  make_pt(    6.1626,   6.4620,  -1.2827, &rG04.o4_     );
  make_pt(    6.5431,   5.0992,  -1.3905, &rG04.c1_     );
  make_pt(    7.2871,   4.9328,  -0.6114, &rG04.h1_     );
  make_pt(    7.1852,   4.8935,  -2.7592, &rG04.c2_     );
  make_pt(    6.8573,   3.9363,  -3.1645, &rG04.h2__    );
  make_pt(    8.5780,   5.1025,  -2.6046, &rG04.o2_     );
  make_pt(    8.9516,   4.7577,  -1.7902, &rG04.h2_     );
  make_pt(    6.5522,   6.0300,  -3.5612, &rG04.c3_     );
  make_pt(    5.5420,   5.7356,  -3.8459, &rG04.h3_     );
  make_pt(    7.3487,   6.4089,  -4.6867, &rG04.o3_     );
  make_pt(    3.6343,   2.6680,   2.0783, &rG04.n1      );
  make_pt(    5.4505,   3.9805,   1.2446, &rG04.n3      );
  make_pt(    4.7540,   3.3816,   2.1851, &rG04.c2      );
  make_pt(    4.8805,   3.7951,   0.0354, &rG04.c4      );
  make_pt(    3.7416,   3.0925,  -0.2305, &rG04.c5      );
  make_pt(    3.0873,   2.4980,   0.8606, &rG04.c6      );
  make_pt(    5.1433,   3.4373,   3.4609, &rG04._.G.n2  );
  make_pt(    3.4605,   3.1184,  -1.5906, &rG04._.G.n7  );
  make_pt(    5.3247,   4.2695,  -1.1710, &rG04._.G.n9  );
  make_pt(    4.4244,   3.8244,  -2.0953, &rG04._.G.c8  );
  make_pt(    1.9600,   1.7805,   0.7462, &rG04._.G.o6  );
  make_pt(    3.2489,   2.2879,   2.9191, &rG04._.G.h1  );
  make_pt(    4.6785,   3.0243,   4.2568, &rG04._.G.h21 );
  make_pt(    5.9823,   3.9654,   3.6539, &rG04._.G.h22 );
  make_pt(    4.2675,   3.8876,  -3.1721, &rG04._.G.h8  );

  rG05.type = 'G';
  make_tfo(  -0.5891,   0.0449,   0.8068,
              0.5375,   0.7673,   0.3498,
             -0.6034,   0.6397,  -0.4762,
             -0.3019,  -3.7679,  -9.5913, &rG05.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rG05.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rG05.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rG05.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rG05.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rG05.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rG05.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rG05.o5_     );
  make_pt(    4.5778,   6.6594,  -4.0364, &rG05.c5_     );
  make_pt(    4.9220,   7.1963,  -4.9204, &rG05.h5_     );
  make_pt(    3.7996,   5.9091,  -4.1764, &rG05.h5__    );
  make_pt(    5.7873,   5.8869,  -3.5482, &rG05.c4_     );
  make_pt(    6.0405,   5.0875,  -4.2446, &rG05.h4_     );
  make_pt(    6.9135,   6.8036,  -3.4310, &rG05.o4_     );
  make_pt(    7.7293,   6.4084,  -2.3392, &rG05.c1_     );
  make_pt(    8.7078,   6.1815,  -2.7624, &rG05.h1_     );
  make_pt(    7.1305,   5.1418,  -1.7347, &rG05.c2_     );
  make_pt(    7.2040,   5.1982,  -0.6486, &rG05.h2__    );
  make_pt(    7.7417,   4.0392,  -2.3813, &rG05.o2_     );
  make_pt(    8.6785,   4.1443,  -2.5630, &rG05.h2_     );
  make_pt(    5.6666,   5.2728,  -2.1536, &rG05.c3_     );
  make_pt(    5.1747,   5.9805,  -1.4863, &rG05.h3_     );
  make_pt(    4.9997,   4.0086,  -2.1973, &rG05.o3_     );
  make_pt(   10.2594,  10.6774,  -1.0056, &rG05.n1      );
  make_pt(    9.7528,   8.7080,  -2.2631, &rG05.n3      );
  make_pt(   10.4471,   9.7876,  -1.9791, &rG05.c2      );
  make_pt(    8.7271,   8.5575,  -1.3991, &rG05.c4      );
  make_pt(    8.4100,   9.3803,  -0.3580, &rG05.c5      );
  make_pt(    9.2294,  10.5030,  -0.1574, &rG05.c6      );
  make_pt(   11.5110,  10.1256,  -2.7114, &rG05._.G.n2  );
  make_pt(    7.2891,   8.9068,   0.3121, &rG05._.G.n7  );
  make_pt(    7.7962,   7.5519,  -1.3859, &rG05._.G.n9  );
  make_pt(    6.9702,   7.8292,  -0.3353, &rG05._.G.c8  );
  make_pt(    9.0349,  11.3951,   0.8250, &rG05._.G.o6  );
  make_pt(   10.9013,  11.4422,  -0.9512, &rG05._.G.h1  );
  make_pt(   12.1031,  10.9341,  -2.5861, &rG05._.G.h21 );
  make_pt(   11.7369,   9.5180,  -3.4859, &rG05._.G.h22 );
  make_pt(    6.0888,   7.3990,   0.1403, &rG05._.G.h8  );

  rG06.type = 'G';
  make_tfo(  -0.9815,   0.0731,  -0.1772,
              0.1912,   0.3054,  -0.9328,
             -0.0141,  -0.9494,  -0.3137,
              5.7506,  -5.1944,   4.7470, &rG06.dgf_base_tfo );
  make_tfo(  -0.8143,  -0.5091,  -0.2788,
             -0.0433,  -0.4257,   0.9038,
             -0.5788,   0.7480,   0.3246,
              1.5227,   6.9114,  -7.0765, &rG06.p_o3_275_tfo );
  make_tfo(   0.3822,  -0.7477,   0.5430,
              0.4552,   0.6637,   0.5935,
             -0.8042,   0.0203,   0.5941,
             -6.9472,  -4.1186,  -5.9108, &rG06.p_o3_180_tfo );
  make_tfo(   0.5640,   0.8007,  -0.2022,
             -0.8247,   0.5587,  -0.0878,
              0.0426,   0.2162,   0.9754,
              6.2694,  -7.0540,   3.3316, &rG06.p_o3_60_tfo  );
  make_pt(    2.8930,   8.5380,  -3.3280, &rG06.p       );
  make_pt(    1.6980,   7.6960,  -3.5570, &rG06.o1p     );
  make_pt(    3.2260,   9.5010,  -4.4020, &rG06.o2p     );
  make_pt(    4.1590,   7.6040,  -3.0340, &rG06.o5_     );
  make_pt(    4.1214,   6.7116,  -1.9049, &rG06.c5_     );
  make_pt(    3.3465,   5.9610,  -2.0607, &rG06.h5_     );
  make_pt(    4.0789,   7.2928,  -0.9837, &rG06.h5__    );
  make_pt(    5.4170,   5.9293,  -1.8186, &rG06.c4_     );
  make_pt(    5.4506,   5.3400,  -0.9023, &rG06.h4_     );
  make_pt(    5.5067,   5.0417,  -2.9703, &rG06.o4_     );
  make_pt(    6.8650,   4.9152,  -3.3612, &rG06.c1_     );
  make_pt(    7.1090,   3.8577,  -3.2603, &rG06.h1_     );
  make_pt(    7.7152,   5.7282,  -2.3894, &rG06.c2_     );
  make_pt(    8.5029,   6.2356,  -2.9463, &rG06.h2__    );
  make_pt(    8.1036,   4.8568,  -1.3419, &rG06.o2_     );
  make_pt(    8.3270,   3.9651,  -1.6184, &rG06.h2_     );
  make_pt(    6.7003,   6.7565,  -1.8911, &rG06.c3_     );
  make_pt(    6.5898,   7.5329,  -2.6482, &rG06.h3_     );
  make_pt(    7.0505,   7.2878,  -0.6105, &rG06.o3_     );
  make_pt(    6.6624,   3.5061,  -8.2986, &rG06.n1      );
  make_pt(    6.5810,   3.2570,  -5.9221, &rG06.n3      );
  make_pt(    6.5151,   2.8263,  -7.1625, &rG06.c2      );
  make_pt(    6.8364,   4.5817,  -5.8882, &rG06.c4      );
  make_pt(    7.0116,   5.4064,  -6.9609, &rG06.c5      );
  make_pt(    6.9173,   4.8260,  -8.2361, &rG06.c6      );
  make_pt(    6.2717,   1.5402,  -7.4250, &rG06._.G.n2  );
  make_pt(    7.2573,   6.7070,  -6.5394, &rG06._.G.n7  );
  make_pt(    6.9740,   5.3703,  -4.7760, &rG06._.G.n9  );
  make_pt(    7.2238,   6.6275,  -5.2453, &rG06._.G.c8  );
  make_pt(    7.0668,   5.5163,  -9.3763, &rG06._.G.o6  );
  make_pt(    6.5754,   2.9964,  -9.1545, &rG06._.G.h1  );
  make_pt(    6.1908,   1.1105,  -8.3354, &rG06._.G.h21 );
  make_pt(    6.1346,   0.9352,  -6.6280, &rG06._.G.h22 );
  make_pt(    7.4108,   7.6227,  -4.8418, &rG06._.G.h8  );

  rG07.type = 'G';
  make_tfo(   0.0894,  -0.6059,   0.7905,
             -0.6810,   0.5420,   0.4924,
             -0.7268,  -0.5824,  -0.3642,
             34.1424,  45.9610, -11.8600, &rG07.dgf_base_tfo );
  make_tfo(  -0.8644,  -0.4956,  -0.0851,
             -0.0427,   0.2409,  -0.9696,
              0.5010,  -0.8345,  -0.2294,
              4.0167,  54.5377,  12.4779, &rG07.p_o3_275_tfo );
  make_tfo(   0.3706,  -0.6167,   0.6945,
             -0.2867,  -0.7872,  -0.5460,
              0.8834,   0.0032,  -0.4686,
            -52.9020,  18.6313,  -0.6709, &rG07.p_o3_180_tfo );
  make_tfo(   0.4155,   0.9025,  -0.1137,
              0.9040,  -0.4236,  -0.0582,
             -0.1007,  -0.0786,  -0.9918,
             -7.6624, -25.2080,  49.5181, &rG07.p_o3_60_tfo  );
  make_pt(   31.3810,   0.1400,  47.5810, &rG07.p       );
  make_pt(   29.9860,   0.6630,  47.6290, &rG07.o1p     );
  make_pt(   31.7210,  -0.6460,  48.8090, &rG07.o2p     );
  make_pt(   32.4940,   1.2540,  47.2740, &rG07.o5_     );
  make_pt(   33.8709,   0.7918,  47.2113, &rG07.c5_     );
  make_pt(   34.1386,   0.5870,  46.1747, &rG07.h5_     );
  make_pt(   34.0186,  -0.0095,  47.9353, &rG07.h5__    );
  make_pt(   34.7297,   1.9687,  47.6685, &rG07.c4_     );
  make_pt(   35.7723,   1.6845,  47.8113, &rG07.h4_     );
  make_pt(   34.6455,   2.9768,  46.6660, &rG07.o4_     );
  make_pt(   34.1690,   4.1829,  47.2627, &rG07.c1_     );
  make_pt(   35.0437,   4.7633,  47.5560, &rG07.h1_     );
  make_pt(   33.4145,   3.7532,  48.4954, &rG07.c2_     );
  make_pt(   32.4340,   3.3797,  48.2001, &rG07.h2__    );
  make_pt(   33.3209,   4.6953,  49.5217, &rG07.o2_     );
  make_pt(   33.2374,   5.6059,  49.2295, &rG07.h2_     );
  make_pt(   34.2724,   2.5970,  48.9773, &rG07.c3_     );
  make_pt(   33.6373,   1.8935,  49.5157, &rG07.h3_     );
  make_pt(   35.3453,   3.1884,  49.7285, &rG07.o3_     );
  make_pt(   34.0511,   7.8930,  43.7791, &rG07.n1      );
  make_pt(   34.9937,   6.3369,  45.3199, &rG07.n3      );
  make_pt(   35.0882,   7.3126,  44.4200, &rG07.c2      );
  make_pt(   33.7190,   5.9650,  45.5374, &rG07.c4      );
  make_pt(   32.5845,   6.4770,  44.9458, &rG07.c5      );
  make_pt(   32.7430,   7.5179,  43.9914, &rG07.c6      );
  make_pt(   36.3030,   7.7827,  44.1036, &rG07._.G.n2  );
  make_pt(   31.4499,   5.8335,  45.4368, &rG07._.G.n7  );
  make_pt(   33.2760,   4.9817,  46.4043, &rG07._.G.n9  );
  make_pt(   31.9235,   4.9639,  46.2934, &rG07._.G.c8  );
  make_pt(   31.8602,   8.1000,  43.3695, &rG07._.G.o6  );
  make_pt(   34.2623,   8.6223,  43.1283, &rG07._.G.h1  );
  make_pt(   36.5188,   8.5081,  43.4347, &rG07._.G.h21 );
  make_pt(   37.0888,   7.3524,  44.5699, &rG07._.G.h22 );
  make_pt(   31.0815,   4.4201,  46.7218, &rG07._.G.h8  );

  rG08.type = 'G';
  make_tfo(   0.2224,   0.6335,   0.7411,
             -0.3644,  -0.6510,   0.6659,
              0.9043,  -0.4181,   0.0861,
            -47.6824,  -0.5823, -31.7554, &rG08.dgf_base_tfo );
  make_tfo(  -0.8644,  -0.4956,  -0.0851,
             -0.0427,   0.2409,  -0.9696,
              0.5010,  -0.8345,  -0.2294,
              4.0167,  54.5377,  12.4779, &rG08.p_o3_275_tfo );
  make_tfo(   0.3706,  -0.6167,   0.6945,
             -0.2867,  -0.7872,  -0.5460,
              0.8834,   0.0032,  -0.4686,
            -52.9020,  18.6313,  -0.6709, &rG08.p_o3_180_tfo );
  make_tfo(   0.4155,   0.9025,  -0.1137,
              0.9040,  -0.4236,  -0.0582,
             -0.1007,  -0.0786,  -0.9918,
             -7.6624, -25.2080,  49.5181, &rG08.p_o3_60_tfo  );
  make_pt(   31.3810,   0.1400,  47.5810, &rG08.p       );
  make_pt(   29.9860,   0.6630,  47.6290, &rG08.o1p     );
  make_pt(   31.7210,  -0.6460,  48.8090, &rG08.o2p     );
  make_pt(   32.4940,   1.2540,  47.2740, &rG08.o5_     );
  make_pt(   32.5924,   2.3488,  48.2255, &rG08.c5_     );
  make_pt(   33.3674,   2.1246,  48.9584, &rG08.h5_     );
  make_pt(   31.5994,   2.5917,  48.6037, &rG08.h5__    );
  make_pt(   33.0722,   3.5577,  47.4258, &rG08.c4_     );
  make_pt(   33.0310,   4.4778,  48.0089, &rG08.h4_     );
  make_pt(   34.4173,   3.3055,  47.0316, &rG08.o4_     );
  make_pt(   34.5056,   3.3910,  45.6094, &rG08.c1_     );
  make_pt(   34.7881,   4.4152,  45.3663, &rG08.h1_     );
  make_pt(   33.1122,   3.1198,  45.1010, &rG08.c2_     );
  make_pt(   32.9230,   2.0469,  45.1369, &rG08.h2__    );
  make_pt(   32.7946,   3.6590,  43.8529, &rG08.o2_     );
  make_pt(   33.5170,   3.6707,  43.2207, &rG08.h2_     );
  make_pt(   32.2730,   3.8173,  46.1566, &rG08.c3_     );
  make_pt(   31.3094,   3.3123,  46.2244, &rG08.h3_     );
  make_pt(   32.2391,   5.2039,  45.7807, &rG08.o3_     );
  make_pt(   39.3337,   2.7157,  44.1441, &rG08.n1      );
  make_pt(   37.4430,   3.8242,  45.0824, &rG08.n3      );
  make_pt(   38.7276,   3.7646,  44.7403, &rG08.c2      );
  make_pt(   36.7791,   2.6963,  44.7704, &rG08.c4      );
  make_pt(   37.2860,   1.5653,  44.1678, &rG08.c5      );
  make_pt(   38.6647,   1.5552,  43.8235, &rG08.c6      );
  make_pt(   39.5123,   4.8216,  44.9936, &rG08._.G.n2  );
  make_pt(   36.2829,   0.6110,  44.0078, &rG08._.G.n7  );
  make_pt(   35.4394,   2.4314,  44.9931, &rG08._.G.n9  );
  make_pt(   35.2180,   1.1815,  44.5128, &rG08._.G.c8  );
  make_pt(   39.2907,   0.6514,  43.2796, &rG08._.G.o6  );
  make_pt(   40.3076,   2.8048,  43.9352, &rG08._.G.h1  );
  make_pt(   40.4994,   4.9066,  44.7977, &rG08._.G.h21 );
  make_pt(   39.0738,   5.6108,  45.4464, &rG08._.G.h22 );
  make_pt(   34.3856,   0.4842,  44.4185, &rG08._.G.h8  );

  rG09.type = 'G';
  make_tfo(  -0.9699,  -0.1688,  -0.1753,
             -0.1050,  -0.3598,   0.9271,
             -0.2196,   0.9176,   0.3312,
             45.6217, -38.9484, -12.3208, &rG09.dgf_base_tfo );
  make_tfo(  -0.8644,  -0.4956,  -0.0851,
             -0.0427,   0.2409,  -0.9696,
              0.5010,  -0.8345,  -0.2294,
              4.0167,  54.5377,  12.4779, &rG09.p_o3_275_tfo );
  make_tfo(   0.3706,  -0.6167,   0.6945,
             -0.2867,  -0.7872,  -0.5460,
              0.8834,   0.0032,  -0.4686,
            -52.9020,  18.6313,  -0.6709, &rG09.p_o3_180_tfo );
  make_tfo(   0.4155,   0.9025,  -0.1137,
              0.9040,  -0.4236,  -0.0582,
             -0.1007,  -0.0786,  -0.9918,
             -7.6624, -25.2080,  49.5181, &rG09.p_o3_60_tfo  );
  make_pt(   31.3810,   0.1400,  47.5810, &rG09.p       );
  make_pt(   29.9860,   0.6630,  47.6290, &rG09.o1p     );
  make_pt(   31.7210,  -0.6460,  48.8090, &rG09.o2p     );
  make_pt(   32.4940,   1.2540,  47.2740, &rG09.o5_     );
  make_pt(   33.8709,   0.7918,  47.2113, &rG09.c5_     );
  make_pt(   34.1386,   0.5870,  46.1747, &rG09.h5_     );
  make_pt(   34.0186,  -0.0095,  47.9353, &rG09.h5__    );
  make_pt(   34.7297,   1.9687,  47.6685, &rG09.c4_     );
  make_pt(   34.5880,   2.8482,  47.0404, &rG09.h4_     );
  make_pt(   34.3575,   2.2770,  49.0081, &rG09.o4_     );
  make_pt(   35.5157,   2.1993,  49.8389, &rG09.c1_     );
  make_pt(   35.9424,   3.2010,  49.8893, &rG09.h1_     );
  make_pt(   36.4701,   1.2820,  49.1169, &rG09.c2_     );
  make_pt(   36.1545,   0.2498,  49.2683, &rG09.h2__    );
  make_pt(   37.8262,   1.4547,  49.4008, &rG09.o2_     );
  make_pt(   38.0227,   1.6945,  50.3094, &rG09.h2_     );
  make_pt(   36.2242,   1.6797,  47.6725, &rG09.c3_     );
  make_pt(   36.4297,   0.8197,  47.0351, &rG09.h3_     );
  make_pt(   37.0289,   2.8480,  47.4426, &rG09.o3_     );
  make_pt(   34.3005,   3.5042,  54.6070, &rG09.n1      );
  make_pt(   34.7693,   3.7936,  52.2874, &rG09.n3      );
  make_pt(   34.4484,   4.2541,  53.4939, &rG09.c2      );
  make_pt(   34.9354,   2.4584,  52.2785, &rG09.c4      );
  make_pt(   34.8092,   1.5915,  53.3422, &rG09.c5      );
  make_pt(   34.4646,   2.1367,  54.6085, &rG09.c6      );
  make_pt(   34.2514,   5.5708,  53.6503, &rG09._.G.n2  );
  make_pt(   35.0641,   0.2835,  52.9337, &rG09._.G.n7  );
  make_pt(   35.2669,   1.6690,  51.1915, &rG09._.G.n9  );
  make_pt(   35.3288,   0.3954,  51.6563, &rG09._.G.c8  );
  make_pt(   34.3151,   1.5317,  55.6650, &rG09._.G.o6  );
  make_pt(   34.0623,   3.9797,  55.4539, &rG09._.G.h1  );
  make_pt(   33.9950,   6.0502,  54.5016, &rG09._.G.h21 );
  make_pt(   34.3512,   6.1432,  52.8242, &rG09._.G.h22 );
  make_pt(   35.5414,  -0.6006,  51.2679, &rG09._.G.h8  );

  rG10.type = 'G';
  make_tfo(  -0.0980,  -0.9723,   0.2122,
             -0.9731,   0.1383,   0.1841,
             -0.2083,  -0.1885,  -0.9597,
             17.8469,  38.8265,  37.0475, &rG10.dgf_base_tfo );
  make_tfo(  -0.8644,  -0.4956,  -0.0851,
             -0.0427,   0.2409,  -0.9696,
              0.5010,  -0.8345,  -0.2294,
              4.0167,  54.5377,  12.4779, &rG10.p_o3_275_tfo );
  make_tfo(   0.3706,  -0.6167,   0.6945,
             -0.2867,  -0.7872,  -0.5460,
              0.8834,   0.0032,  -0.4686,
            -52.9020,  18.6313,  -0.6709, &rG10.p_o3_180_tfo );
  make_tfo(   0.4155,   0.9025,  -0.1137,
              0.9040,  -0.4236,  -0.0582,
             -0.1007,  -0.0786,  -0.9918,
             -7.6624, -25.2080,  49.5181, &rG10.p_o3_60_tfo  );
  make_pt(   31.3810,   0.1400,  47.5810, &rG10.p       );
  make_pt(   29.9860,   0.6630,  47.6290, &rG10.o1p     );
  make_pt(   31.7210,  -0.6460,  48.8090, &rG10.o2p     );
  make_pt(   32.4940,   1.2540,  47.2740, &rG10.o5_     );
  make_pt(   32.5924,   2.3488,  48.2255, &rG10.c5_     );
  make_pt(   33.3674,   2.1246,  48.9584, &rG10.h5_     );
  make_pt(   31.5994,   2.5917,  48.6037, &rG10.h5__    );
  make_pt(   33.0722,   3.5577,  47.4258, &rG10.c4_     );
  make_pt(   34.0333,   3.3761,  46.9447, &rG10.h4_     );
  make_pt(   32.0890,   3.8338,  46.4332, &rG10.o4_     );
  make_pt(   31.6377,   5.1787,  46.5914, &rG10.c1_     );
  make_pt(   32.2499,   5.8016,  45.9392, &rG10.h1_     );
  make_pt(   31.9167,   5.5319,  48.0305, &rG10.c2_     );
  make_pt(   31.1507,   5.0820,  48.6621, &rG10.h2__    );
  make_pt(   32.0865,   6.8890,  48.3114, &rG10.o2_     );
  make_pt(   31.5363,   7.4819,  47.7942, &rG10.h2_     );
  make_pt(   33.2398,   4.8224,  48.2563, &rG10.c3_     );
  make_pt(   33.3166,   4.5570,  49.3108, &rG10.h3_     );
  make_pt(   34.2528,   5.7056,  47.7476, &rG10.o3_     );
  make_pt(   28.2782,   6.3049,  42.9364, &rG10.n1      );
  make_pt(   30.4001,   5.8547,  43.9258, &rG10.n3      );
  make_pt(   29.6195,   6.1568,  42.8913, &rG10.c2      );
  make_pt(   29.7005,   5.7006,  45.0649, &rG10.c4      );
  make_pt(   28.3383,   5.8221,  45.2343, &rG10.c5      );
  make_pt(   27.5519,   6.1461,  44.0958, &rG10.c6      );
  make_pt(   30.1838,   6.3385,  41.6890, &rG10._.G.n2  );
  make_pt(   27.9936,   5.5926,  46.5651, &rG10._.G.n7  );
  make_pt(   30.2046,   5.3825,  46.3136, &rG10._.G.n9  );
  make_pt(   29.1371,   5.3398,  47.1506, &rG10._.G.c8  );
  make_pt(   26.3361,   6.3024,  44.0495, &rG10._.G.o6  );
  make_pt(   27.8122,   6.5394,  42.0833, &rG10._.G.h1  );
  make_pt(   29.7125,   6.5595,  40.8235, &rG10._.G.h21 );
  make_pt(   31.1859,   6.2231,  41.6389, &rG10._.G.h22 );
  make_pt(   28.9406,   5.1504,  48.2059, &rG10._.G.h8  );

  rGs[0] = &rG10;
  rGs[1] = &rG09;
  rGs[2] = &rG08;
  rGs[3] = &rG07;
  rGs[4] = &rG06;
  rGs[5] = &rG05;
  rGs[6] = &rG04;
  rGs[7] = &rG03;
  rGs[8] = &rG02;
  rGs[9] = &rG01;
  rGs[10]= 0;
}

void init_U()
{
  rU.type = 'U';
  make_tfo(  -0.0359,  -0.8071,   0.5894,
             -0.2669,   0.5761,   0.7726,
             -0.9631,  -0.1296,  -0.2361,
              0.1584,   8.3434,   0.5434, &rU.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rU.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rU.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rU.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rU.p         );
  make_pt(    1.4950,  -7.6230,   3.4770, &rU.o1p       );
  make_pt(    2.9490,  -9.4640,   4.3740, &rU.o2p       );
  make_pt(    3.9730,  -7.5950,   3.0340, &rU.o5_       );
  make_pt(    5.2430,  -8.2420,   2.8260, &rU.c5_       );
  make_pt(    5.1974,  -8.8497,   1.9223, &rU.h5_       );
  make_pt(    5.5548,  -8.7348,   3.7469, &rU.h5__      );
  make_pt(    6.3140,  -7.2060,   2.5510, &rU.c4_       );
  make_pt(    7.2954,  -7.6762,   2.4898, &rU.h4_       );
  make_pt(    6.0140,  -6.5420,   1.2890, &rU.o4_       );
  make_pt(    6.4190,  -5.1840,   1.3620, &rU.c1_       );
  make_pt(    7.1608,  -5.0495,   0.5747, &rU.h1_       );
  make_pt(    7.0760,  -4.9560,   2.7270, &rU.c2_       );
  make_pt(    6.7770,  -3.9803,   3.1099, &rU.h2__      );
  make_pt(    8.4500,  -5.1930,   2.5810, &rU.o2_       );
  make_pt(    8.8309,  -4.8755,   1.7590, &rU.h2_       );
  make_pt(    6.4060,  -6.0590,   3.5580, &rU.c3_       );
  make_pt(    5.4021,  -5.7313,   3.8281, &rU.h3_       );
  make_pt(    7.1570,  -6.4240,   4.7070, &rU.o3_       );
  make_pt(    5.2170,  -4.3260,   1.1690, &rU.n1        );
  make_pt(    4.2960,  -2.2560,   0.6290, &rU.n3        );
  make_pt(    5.4330,  -3.0200,   0.7990, &rU.c2        );
  make_pt(    2.9930,  -2.6780,   0.7940, &rU.c4        );
  make_pt(    2.8670,  -4.0630,   1.1830, &rU.c5        );
  make_pt(    3.9570,  -4.8300,   1.3550, &rU.c6        );
  make_pt(    6.5470,  -2.5560,   0.6290, &rU._.U.o2    );
  make_pt(    2.0540,  -1.9000,   0.6130, &rU._.U.o4    );
  make_pt(    4.4300,  -1.3020,   0.3600, &rU._.U.h3    );
  make_pt(    1.9590,  -4.4570,   1.3250, &rU._.U.h5    );
  make_pt(    3.8460,  -5.7860,   1.6240, &rU._.U.h6    );

  rU01.type = 'U';
  make_tfo(  -0.0137,  -0.8012,   0.5983,
             -0.2523,   0.5817,   0.7733,
             -0.9675,  -0.1404,  -0.2101,
              0.2031,   8.3874,   0.4228, &rU01.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rU01.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rU01.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rU01.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rU01.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rU01.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rU01.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rU01.o5_     );
  make_pt(    5.2416,  -8.2422,   2.8181, &rU01.c5_     );
  make_pt(    5.2050,  -8.8128,   1.8901, &rU01.h5_     );
  make_pt(    5.5368,  -8.7738,   3.7227, &rU01.h5__    );
  make_pt(    6.3232,  -7.2037,   2.6002, &rU01.c4_     );
  make_pt(    7.3048,  -7.6757,   2.5577, &rU01.h4_     );
  make_pt(    6.0635,  -6.5092,   1.3456, &rU01.o4_     );
  make_pt(    6.4697,  -5.1547,   1.4629, &rU01.c1_     );
  make_pt(    7.2354,  -5.0043,   0.7018, &rU01.h1_     );
  make_pt(    7.0856,  -4.9610,   2.8521, &rU01.c2_     );
  make_pt(    6.7777,  -3.9935,   3.2487, &rU01.h2__    );
  make_pt(    8.4627,  -5.1992,   2.7423, &rU01.o2_     );
  make_pt(    8.8693,  -4.8638,   1.9399, &rU01.h2_     );
  make_pt(    6.3877,  -6.0809,   3.6362, &rU01.c3_     );
  make_pt(    5.3770,  -5.7562,   3.8834, &rU01.h3_     );
  make_pt(    7.1024,  -6.4754,   4.7985, &rU01.o3_     );
  make_pt(    5.2764,  -4.2883,   1.2538, &rU01.n1      );
  make_pt(    4.3777,  -2.2062,   0.7229, &rU01.n3      );
  make_pt(    5.5069,  -2.9779,   0.9088, &rU01.c2      );
  make_pt(    3.0693,  -2.6246,   0.8500, &rU01.c4      );
  make_pt(    2.9279,  -4.0146,   1.2149, &rU01.c5      );
  make_pt(    4.0101,  -4.7892,   1.4017, &rU01.c6      );
  make_pt(    6.6267,  -2.5166,   0.7728, &rU01._.U.o2  );
  make_pt(    2.1383,  -1.8396,   0.6581, &rU01._.U.o4  );
  make_pt(    4.5223,  -1.2489,   0.4716, &rU01._.U.h3  );
  make_pt(    2.0151,  -4.4065,   1.3290, &rU01._.U.h5  );
  make_pt(    3.8886,  -5.7486,   1.6535, &rU01._.U.h6  );

  rU02.type = 'U';
  make_tfo(   0.5141,   0.0246,   0.8574,
             -0.5547,  -0.7529,   0.3542,
              0.6542,  -0.6577,  -0.3734,
             -9.1111,  -3.4598,  -3.2939, &rU02.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rU02.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rU02.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rU02.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rU02.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rU02.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rU02.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rU02.o5_     );
  make_pt(    4.3825,  -6.6585,   4.0489, &rU02.c5_     );
  make_pt(    4.6841,  -7.2019,   4.9443, &rU02.h5_     );
  make_pt(    3.6189,  -5.8889,   4.1625, &rU02.h5__    );
  make_pt(    5.6255,  -5.9175,   3.5998, &rU02.c4_     );
  make_pt(    5.8732,  -5.1228,   4.3034, &rU02.h4_     );
  make_pt(    6.7337,  -6.8605,   3.5222, &rU02.o4_     );
  make_pt(    7.5932,  -6.4923,   2.4548, &rU02.c1_     );
  make_pt(    8.5661,  -6.2983,   2.9064, &rU02.h1_     );
  make_pt(    7.0527,  -5.2012,   1.8322, &rU02.c2_     );
  make_pt(    7.1627,  -5.2525,   0.7490, &rU02.h2__    );
  make_pt(    7.6666,  -4.1249,   2.4880, &rU02.o2_     );
  make_pt(    8.5944,  -4.2543,   2.6981, &rU02.h2_     );
  make_pt(    5.5661,  -5.3029,   2.2009, &rU02.c3_     );
  make_pt(    5.0841,  -6.0018,   1.5172, &rU02.h3_     );
  make_pt(    4.9062,  -4.0452,   2.2042, &rU02.o3_     );
  make_pt(    7.6298,  -7.6136,   1.4752, &rU02.n1      );
  make_pt(    8.6945,  -8.7046,  -0.2857, &rU02.n3      );
  make_pt(    8.6943,  -7.6514,   0.6066, &rU02.c2      );
  make_pt(    7.7426,  -9.6987,  -0.3801, &rU02.c4      );
  make_pt(    6.6642,  -9.5742,   0.5722, &rU02.c5      );
  make_pt(    6.6391,  -8.5592,   1.4526, &rU02.c6      );
  make_pt(    9.5840,  -6.8186,   0.6136, &rU02._.U.o2  );
  make_pt(    7.8505, -10.5925,  -1.2223, &rU02._.U.o4  );
  make_pt(    9.4601,  -8.7514,  -0.9277, &rU02._.U.h3  );
  make_pt(    5.9281, -10.2509,   0.5782, &rU02._.U.h5  );
  make_pt(    5.8831,  -8.4931,   2.1028, &rU02._.U.h6  );

  rU03.type = 'U';
  make_tfo(  -0.4993,   0.0476,   0.8651,
              0.8078,  -0.3353,   0.4847,
              0.3132,   0.9409,   0.1290,
              6.2989,  -5.2303,  -3.8577, &rU03.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rU03.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rU03.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rU03.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rU03.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rU03.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rU03.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rU03.o5_     );
  make_pt(    3.9938,  -6.7042,   1.9023, &rU03.c5_     );
  make_pt(    3.2332,  -5.9343,   2.0319, &rU03.h5_     );
  make_pt(    3.9666,  -7.2863,   0.9812, &rU03.h5__    );
  make_pt(    5.3098,  -5.9546,   1.8564, &rU03.c4_     );
  make_pt(    5.3863,  -5.3702,   0.9395, &rU03.h4_     );
  make_pt(    5.3851,  -5.0642,   3.0076, &rU03.o4_     );
  make_pt(    6.7315,  -4.9724,   3.4462, &rU03.c1_     );
  make_pt(    7.0033,  -3.9202,   3.3619, &rU03.h1_     );
  make_pt(    7.5997,  -5.8018,   2.4948, &rU03.c2_     );
  make_pt(    8.3627,  -6.3254,   3.0707, &rU03.h2__    );
  make_pt(    8.0410,  -4.9501,   1.4724, &rU03.o2_     );
  make_pt(    8.2781,  -4.0644,   1.7570, &rU03.h2_     );
  make_pt(    6.5701,  -6.8129,   1.9714, &rU03.c3_     );
  make_pt(    6.4186,  -7.5809,   2.7299, &rU03.h3_     );
  make_pt(    6.9357,  -7.3841,   0.7235, &rU03.o3_     );
  make_pt(    6.8024,  -5.4718,   4.8475, &rU03.n1      );
  make_pt(    7.9218,  -5.5700,   6.8877, &rU03.n3      );
  make_pt(    7.8908,  -5.0886,   5.5944, &rU03.c2      );
  make_pt(    6.9789,  -6.3827,   7.4823, &rU03.c4      );
  make_pt(    5.8742,  -6.7319,   6.6202, &rU03.c5      );
  make_pt(    5.8182,  -6.2769,   5.3570, &rU03.c6      );
  make_pt(    8.7747,  -4.3728,   5.1568, &rU03._.U.o2  );
  make_pt(    7.1154,  -6.7509,   8.6509, &rU03._.U.o4  );
  make_pt(    8.7055,  -5.3037,   7.4491, &rU03._.U.h3  );
  make_pt(    5.1416,  -7.3178,   6.9665, &rU03._.U.h5  );
  make_pt(    5.0441,  -6.5310,   4.7784, &rU03._.U.h6  );

  rU04.type = 'U';
  make_tfo(  -0.5669,  -0.8012,   0.1918,
             -0.8129,   0.5817,   0.0273,
             -0.1334,  -0.1404,  -0.9811,
             -0.3279,   8.3874,   0.3355, &rU04.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rU04.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rU04.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rU04.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rU04.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rU04.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rU04.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rU04.o5_     );
  make_pt(    5.2416,  -8.2422,   2.8181, &rU04.c5_     );
  make_pt(    5.2050,  -8.8128,   1.8901, &rU04.h5_     );
  make_pt(    5.5368,  -8.7738,   3.7227, &rU04.h5__    );
  make_pt(    6.3232,  -7.2037,   2.6002, &rU04.c4_     );
  make_pt(    7.3048,  -7.6757,   2.5577, &rU04.h4_     );
  make_pt(    6.0635,  -6.5092,   1.3456, &rU04.o4_     );
  make_pt(    6.4697,  -5.1547,   1.4629, &rU04.c1_     );
  make_pt(    7.2354,  -5.0043,   0.7018, &rU04.h1_     );
  make_pt(    7.0856,  -4.9610,   2.8521, &rU04.c2_     );
  make_pt(    6.7777,  -3.9935,   3.2487, &rU04.h2__    );
  make_pt(    8.4627,  -5.1992,   2.7423, &rU04.o2_     );
  make_pt(    8.8693,  -4.8638,   1.9399, &rU04.h2_     );
  make_pt(    6.3877,  -6.0809,   3.6362, &rU04.c3_     );
  make_pt(    5.3770,  -5.7562,   3.8834, &rU04.h3_     );
  make_pt(    7.1024,  -6.4754,   4.7985, &rU04.o3_     );
  make_pt(    5.2764,  -4.2883,   1.2538, &rU04.n1      );
  make_pt(    3.8961,  -3.0896,  -0.1893, &rU04.n3      );
  make_pt(    5.0095,  -3.8907,  -0.0346, &rU04.c2      );
  make_pt(    3.0480,  -2.6632,   0.8116, &rU04.c4      );
  make_pt(    3.4093,  -3.1310,   2.1292, &rU04.c5      );
  make_pt(    4.4878,  -3.9124,   2.3088, &rU04.c6      );
  make_pt(    5.7005,  -4.2164,  -0.9842, &rU04._.U.o2  );
  make_pt(    2.0800,  -1.9458,   0.5503, &rU04._.U.o4  );
  make_pt(    3.6834,  -2.7882,  -1.1190, &rU04._.U.h3  );
  make_pt(    2.8508,  -2.8721,   2.9172, &rU04._.U.h5  );
  make_pt(    4.7188,  -4.2247,   3.2295, &rU04._.U.h6  );

  rU05.type = 'U';
  make_tfo(  -0.6298,   0.0246,   0.7763,
             -0.5226,  -0.7529,  -0.4001,
              0.5746,  -0.6577,   0.4870,
             -0.0208,  -3.4598,  -9.6882, &rU05.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rU05.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rU05.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rU05.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rU05.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rU05.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rU05.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rU05.o5_     );
  make_pt(    4.3825,  -6.6585,   4.0489, &rU05.c5_     );
  make_pt(    4.6841,  -7.2019,   4.9443, &rU05.h5_     );
  make_pt(    3.6189,  -5.8889,   4.1625, &rU05.h5__    );
  make_pt(    5.6255,  -5.9175,   3.5998, &rU05.c4_     );
  make_pt(    5.8732,  -5.1228,   4.3034, &rU05.h4_     );
  make_pt(    6.7337,  -6.8605,   3.5222, &rU05.o4_     );
  make_pt(    7.5932,  -6.4923,   2.4548, &rU05.c1_     );
  make_pt(    8.5661,  -6.2983,   2.9064, &rU05.h1_     );
  make_pt(    7.0527,  -5.2012,   1.8322, &rU05.c2_     );
  make_pt(    7.1627,  -5.2525,   0.7490, &rU05.h2__    );
  make_pt(    7.6666,  -4.1249,   2.4880, &rU05.o2_     );
  make_pt(    8.5944,  -4.2543,   2.6981, &rU05.h2_     );
  make_pt(    5.5661,  -5.3029,   2.2009, &rU05.c3_     );
  make_pt(    5.0841,  -6.0018,   1.5172, &rU05.h3_     );
  make_pt(    4.9062,  -4.0452,   2.2042, &rU05.o3_     );
  make_pt(    7.6298,  -7.6136,   1.4752, &rU05.n1      );
  make_pt(    8.5977,  -9.5977,   0.7329, &rU05.n3      );
  make_pt(    8.5951,  -8.5745,   1.6594, &rU05.c2      );
  make_pt(    7.7372,  -9.7371,  -0.3364, &rU05.c4      );
  make_pt(    6.7596,  -8.6801,  -0.4476, &rU05.c5      );
  make_pt(    6.7338,  -7.6721,   0.4408, &rU05.c6      );
  make_pt(    9.3993,  -8.5377,   2.5743, &rU05._.U.o2  );
  make_pt(    7.8374, -10.6990,  -1.1008, &rU05._.U.o4  );
  make_pt(    9.2924, -10.3081,   0.8477, &rU05._.U.h3  );
  make_pt(    6.0932,  -8.6982,  -1.1929, &rU05._.U.h5  );
  make_pt(    6.0481,  -6.9515,   0.3446, &rU05._.U.h6  );

  rU06.type = 'U';
  make_tfo(  -0.9837,   0.0476,  -0.1733,
             -0.1792,  -0.3353,   0.9249,
             -0.0141,   0.9409,   0.3384,
              5.7793,  -5.2303,   4.5997, &rU06.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rU06.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rU06.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rU06.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rU06.p       );
  make_pt(    1.4950,  -7.6230,   3.4770, &rU06.o1p     );
  make_pt(    2.9490,  -9.4640,   4.3740, &rU06.o2p     );
  make_pt(    3.9730,  -7.5950,   3.0340, &rU06.o5_     );
  make_pt(    3.9938,  -6.7042,   1.9023, &rU06.c5_     );
  make_pt(    3.2332,  -5.9343,   2.0319, &rU06.h5_     );
  make_pt(    3.9666,  -7.2863,   0.9812, &rU06.h5__    );
  make_pt(    5.3098,  -5.9546,   1.8564, &rU06.c4_     );
  make_pt(    5.3863,  -5.3702,   0.9395, &rU06.h4_     );
  make_pt(    5.3851,  -5.0642,   3.0076, &rU06.o4_     );
  make_pt(    6.7315,  -4.9724,   3.4462, &rU06.c1_     );
  make_pt(    7.0033,  -3.9202,   3.3619, &rU06.h1_     );
  make_pt(    7.5997,  -5.8018,   2.4948, &rU06.c2_     );
  make_pt(    8.3627,  -6.3254,   3.0707, &rU06.h2__    );
  make_pt(    8.0410,  -4.9501,   1.4724, &rU06.o2_     );
  make_pt(    8.2781,  -4.0644,   1.7570, &rU06.h2_     );
  make_pt(    6.5701,  -6.8129,   1.9714, &rU06.c3_     );
  make_pt(    6.4186,  -7.5809,   2.7299, &rU06.h3_     );
  make_pt(    6.9357,  -7.3841,   0.7235, &rU06.o3_     );
  make_pt(    6.8024,  -5.4718,   4.8475, &rU06.n1      );
  make_pt(    6.6920,  -5.0495,   7.1354, &rU06.n3      );
  make_pt(    6.6201,  -4.5500,   5.8506, &rU06.c2      );
  make_pt(    6.9254,  -6.3614,   7.4926, &rU06.c4      );
  make_pt(    7.1046,  -7.2543,   6.3718, &rU06.c5      );
  make_pt(    7.0391,  -6.7951,   5.1106, &rU06.c6      );
  make_pt(    6.4083,  -3.3696,   5.6340, &rU06._.U.o2  );
  make_pt(    6.9679,  -6.6901,   8.6800, &rU06._.U.o4  );
  make_pt(    6.5626,  -4.3957,   7.8812, &rU06._.U.h3  );
  make_pt(    7.2781,  -8.2254,   6.5350, &rU06._.U.h5  );
  make_pt(    7.1657,  -7.4312,   4.3503, &rU06._.U.h6  );

  rU07.type = 'U';
  make_tfo(  -0.9434,   0.3172,   0.0971,
              0.2294,   0.4125,   0.8816,
              0.2396,   0.8539,  -0.4619,
              8.3625, -52.7147,   1.3745, &rU07.dgf_base_tfo );
  make_tfo(   0.2765,  -0.1121,  -0.9545,
             -0.8297,   0.4733,  -0.2959,
              0.4850,   0.8737,   0.0379,
            -14.7774, -45.2464,  21.9088, &rU07.p_o3_275_tfo );
  make_tfo(   0.1063,  -0.6334,  -0.7665,
             -0.5932,  -0.6591,   0.4624,
             -0.7980,   0.4055,  -0.4458,
             43.7634,   4.3296,  28.4890, &rU07.p_o3_180_tfo );
  make_tfo(   0.7136,  -0.5032,  -0.4873,
              0.6803,   0.3317,   0.6536,
             -0.1673,  -0.7979,   0.5791,
            -17.1858,  41.4390, -27.0751, &rU07.p_o3_60_tfo  );
  make_pt(   21.3880,  15.0780,  45.5770, &rU07.p       );
  make_pt(   21.9980,  14.5500,  46.8210, &rU07.o1p     );
  make_pt(   21.1450,  14.0270,  44.5420, &rU07.o2p     );
  make_pt(   22.1250,  16.3600,  44.9460, &rU07.o5_     );
  make_pt(   21.5037,  16.8594,  43.7323, &rU07.c5_     );
  make_pt(   20.8147,  17.6663,  43.9823, &rU07.h5_     );
  make_pt(   21.1086,  16.0230,  43.1557, &rU07.h5__    );
  make_pt(   22.5654,  17.4874,  42.8616, &rU07.c4_     );
  make_pt(   22.1584,  17.7243,  41.8785, &rU07.h4_     );
  make_pt(   23.0557,  18.6826,  43.4751, &rU07.o4_     );
  make_pt(   24.4788,  18.6151,  43.6455, &rU07.c1_     );
  make_pt(   24.9355,  19.0840,  42.7739, &rU07.h1_     );
  make_pt(   24.7958,  17.1427,  43.6474, &rU07.c2_     );
  make_pt(   24.5652,  16.7400,  44.6336, &rU07.h2__    );
  make_pt(   26.1041,  16.8773,  43.2455, &rU07.o2_     );
  make_pt(   26.7516,  17.5328,  43.5149, &rU07.h2_     );
  make_pt(   23.8109,  16.5979,  42.6377, &rU07.c3_     );
  make_pt(   23.5756,  15.5686,  42.9084, &rU07.h3_     );
  make_pt(   24.2890,  16.7447,  41.2729, &rU07.o3_     );
  make_pt(   24.9420,  19.2174,  44.8923, &rU07.n1      );
  make_pt(   25.2655,  20.5636,  44.8883, &rU07.n3      );
  make_pt(   25.1663,  21.2219,  43.8561, &rU07.c2      );
  make_pt(   25.6911,  21.1219,  46.0494, &rU07.c4      );
  make_pt(   25.8051,  20.4068,  47.2048, &rU07.c5      );
  make_pt(   26.2093,  20.9962,  48.2534, &rU07.c6      );
  make_pt(   25.4692,  19.0221,  47.2053, &rU07._.U.o2  );
  make_pt(   25.0502,  18.4827,  46.0370, &rU07._.U.o4  );
  make_pt(   25.9599,  22.1772,  46.0966, &rU07._.U.h3  );
  make_pt(   25.5545,  18.4409,  48.1234, &rU07._.U.h5  );
  make_pt(   24.7854,  17.4265,  45.9883, &rU07._.U.h6  );

  rU08.type = 'U';
  make_tfo(  -0.0080,  -0.7928,   0.6094,
             -0.7512,   0.4071,   0.5197,
             -0.6601,  -0.4536,  -0.5988,
             44.1482,  30.7036,   2.1088, &rU08.dgf_base_tfo );
  make_tfo(   0.2765,  -0.1121,  -0.9545,
             -0.8297,   0.4733,  -0.2959,
              0.4850,   0.8737,   0.0379,
            -14.7774, -45.2464,  21.9088, &rU08.p_o3_275_tfo );
  make_tfo(   0.1063,  -0.6334,  -0.7665,
             -0.5932,  -0.6591,   0.4624,
             -0.7980,   0.4055,  -0.4458,
             43.7634,   4.3296,  28.4890, &rU08.p_o3_180_tfo );
  make_tfo(   0.7136,  -0.5032,  -0.4873,
              0.6803,   0.3317,   0.6536,
             -0.1673,  -0.7979,   0.5791,
            -17.1858,  41.4390, -27.0751, &rU08.p_o3_60_tfo  );
  make_pt(   21.3880,  15.0780,  45.5770, &rU08.p       );
  make_pt(   21.9980,  14.5500,  46.8210, &rU08.o1p     );
  make_pt(   21.1450,  14.0270,  44.5420, &rU08.o2p     );
  make_pt(   22.1250,  16.3600,  44.9460, &rU08.o5_     );
  make_pt(   23.5096,  16.1227,  44.5783, &rU08.c5_     );
  make_pt(   23.5649,  15.8588,  43.5222, &rU08.h5_     );
  make_pt(   23.9621,  15.4341,  45.2919, &rU08.h5__    );
  make_pt(   24.2805,  17.4138,  44.7151, &rU08.c4_     );
  make_pt(   25.3492,  17.2309,  44.6030, &rU08.h4_     );
  make_pt(   23.8497,  18.3471,  43.7208, &rU08.o4_     );
  make_pt(   23.4090,  19.5681,  44.3321, &rU08.c1_     );
  make_pt(   24.2595,  20.2496,  44.3524, &rU08.h1_     );
  make_pt(   23.0418,  19.1813,  45.7407, &rU08.c2_     );
  make_pt(   22.0532,  18.7224,  45.7273, &rU08.h2__    );
  make_pt(   23.1307,  20.2521,  46.6291, &rU08.o2_     );
  make_pt(   22.8888,  21.1051,  46.2611, &rU08.h2_     );
  make_pt(   24.0799,  18.1326,  46.0700, &rU08.c3_     );
  make_pt(   23.6490,  17.4370,  46.7900, &rU08.h3_     );
  make_pt(   25.3329,  18.7227,  46.5109, &rU08.o3_     );
  make_pt(   22.2515,  20.1624,  43.6698, &rU08.n1      );
  make_pt(   22.4760,  21.0609,  42.6406, &rU08.n3      );
  make_pt(   23.6229,  21.3462,  42.3061, &rU08.c2      );
  make_pt(   21.3986,  21.6081,  42.0236, &rU08.c4      );
  make_pt(   20.1189,  21.3012,  42.3804, &rU08.c5      );
  make_pt(   19.1599,  21.8516,  41.7578, &rU08.c6      );
  make_pt(   19.8919,  20.3745,  43.4387, &rU08._.U.o2  );
  make_pt(   20.9790,  19.8423,  44.0440, &rU08._.U.o4  );
  make_pt(   21.5235,  22.3222,  41.2097, &rU08._.U.h3  );
  make_pt(   18.8732,  20.1200,  43.7312, &rU08._.U.h5  );
  make_pt(   20.8545,  19.1313,  44.8608, &rU08._.U.h6  );

  rU09.type = 'U';
  make_tfo(  -0.0317,   0.1374,   0.9900,
             -0.3422,  -0.9321,   0.1184,
              0.9391,  -0.3351,   0.0765,
            -32.1929,  25.8198, -28.5088, &rU09.dgf_base_tfo );
  make_tfo(   0.2765,  -0.1121,  -0.9545,
             -0.8297,   0.4733,  -0.2959,
              0.4850,   0.8737,   0.0379,
            -14.7774, -45.2464,  21.9088, &rU09.p_o3_275_tfo );
  make_tfo(   0.1063,  -0.6334,  -0.7665,
             -0.5932,  -0.6591,   0.4624,
             -0.7980,   0.4055,  -0.4458,
             43.7634,   4.3296,  28.4890, &rU09.p_o3_180_tfo );
  make_tfo(   0.7136,  -0.5032,  -0.4873,
              0.6803,   0.3317,   0.6536,
             -0.1673,  -0.7979,   0.5791,
            -17.1858,  41.4390, -27.0751, &rU09.p_o3_60_tfo  );
  make_pt(   21.3880,  15.0780,  45.5770, &rU09.p       );
  make_pt(   21.9980,  14.5500,  46.8210, &rU09.o1p     );
  make_pt(   21.1450,  14.0270,  44.5420, &rU09.o2p     );
  make_pt(   22.1250,  16.3600,  44.9460, &rU09.o5_     );
  make_pt(   21.5037,  16.8594,  43.7323, &rU09.c5_     );
  make_pt(   20.8147,  17.6663,  43.9823, &rU09.h5_     );
  make_pt(   21.1086,  16.0230,  43.1557, &rU09.h5__    );
  make_pt(   22.5654,  17.4874,  42.8616, &rU09.c4_     );
  make_pt(   23.0565,  18.3036,  43.3915, &rU09.h4_     );
  make_pt(   23.5375,  16.5054,  42.4925, &rU09.o4_     );
  make_pt(   23.6574,  16.4257,  41.0649, &rU09.c1_     );
  make_pt(   24.4701,  17.0882,  40.7671, &rU09.h1_     );
  make_pt(   22.3525,  16.9643,  40.5396, &rU09.c2_     );
  make_pt(   21.5993,  16.1799,  40.6133, &rU09.h2__    );
  make_pt(   22.4693,  17.4849,  39.2515, &rU09.o2_     );
  make_pt(   23.0899,  17.0235,  38.6827, &rU09.h2_     );
  make_pt(   22.0341,  18.0633,  41.5279, &rU09.c3_     );
  make_pt(   20.9509,  18.1709,  41.5846, &rU09.h3_     );
  make_pt(   22.7249,  19.3020,  41.2100, &rU09.o3_     );
  make_pt(   23.8580,  15.0648,  40.5757, &rU09.n1      );
  make_pt(   25.1556,  14.5982,  40.4523, &rU09.n3      );
  make_pt(   26.1047,  15.3210,  40.7448, &rU09.c2      );
  make_pt(   25.3391,  13.3315,  40.0020, &rU09.c4      );
  make_pt(   24.2974,  12.5148,  39.6749, &rU09.c5      );
  make_pt(   24.5450,  11.3410,  39.2610, &rU09.c6      );
  make_pt(   22.9633,  12.9979,  39.8053, &rU09._.U.o2  );
  make_pt(   22.8009,  14.2648,  40.2524, &rU09._.U.o4  );
  make_pt(   26.3414,  12.9194,  39.8855, &rU09._.U.h3  );
  make_pt(   22.1227,  12.3533,  39.5486, &rU09._.U.h5  );
  make_pt(   21.7989,  14.6788,  40.3650, &rU09._.U.h6  );

  rU10.type = 'U';
  make_tfo(  -0.9674,   0.1021,  -0.2318,
             -0.2514,  -0.2766,   0.9275,
              0.0306,   0.9555,   0.2933,
             27.8571, -42.1305, -24.4563, &rU10.dgf_base_tfo );
  make_tfo(   0.2765,  -0.1121,  -0.9545,
             -0.8297,   0.4733,  -0.2959,
              0.4850,   0.8737,   0.0379,
            -14.7774, -45.2464,  21.9088, &rU10.p_o3_275_tfo );
  make_tfo(   0.1063,  -0.6334,  -0.7665,
             -0.5932,  -0.6591,   0.4624,
             -0.7980,   0.4055,  -0.4458,
             43.7634,   4.3296,  28.4890, &rU10.p_o3_180_tfo );
  make_tfo(   0.7136,  -0.5032,  -0.4873,
              0.6803,   0.3317,   0.6536,
             -0.1673,  -0.7979,   0.5791,
            -17.1858,  41.4390, -27.0751, &rU10.p_o3_60_tfo  );
  make_pt(   21.3880,  15.0780,  45.5770, &rU10.p       );
  make_pt(   21.9980,  14.5500,  46.8210, &rU10.o1p     );
  make_pt(   21.1450,  14.0270,  44.5420, &rU10.o2p     );
  make_pt(   22.1250,  16.3600,  44.9460, &rU10.o5_     );
  make_pt(   23.5096,  16.1227,  44.5783, &rU10.c5_     );
  make_pt(   23.5649,  15.8588,  43.5222, &rU10.h5_     );
  make_pt(   23.9621,  15.4341,  45.2919, &rU10.h5__    );
  make_pt(   24.2805,  17.4138,  44.7151, &rU10.c4_     );
  make_pt(   23.8509,  18.1819,  44.0720, &rU10.h4_     );
  make_pt(   24.2506,  17.8583,  46.0741, &rU10.o4_     );
  make_pt(   25.5830,  18.0320,  46.5775, &rU10.c1_     );
  make_pt(   25.8569,  19.0761,  46.4256, &rU10.h1_     );
  make_pt(   26.4410,  17.1555,  45.7033, &rU10.c2_     );
  make_pt(   26.3459,  16.1253,  46.0462, &rU10.h2__    );
  make_pt(   27.7649,  17.5888,  45.6478, &rU10.o2_     );
  make_pt(   28.1004,  17.9719,  46.4616, &rU10.h2_     );
  make_pt(   25.7796,  17.2997,  44.3513, &rU10.c3_     );
  make_pt(   25.9478,  16.3824,  43.7871, &rU10.h3_     );
  make_pt(   26.2154,  18.4984,  43.6541, &rU10.o3_     );
  make_pt(   25.7321,  17.6281,  47.9726, &rU10.n1      );
  make_pt(   25.5136,  18.5779,  48.9560, &rU10.n3      );
  make_pt(   25.2079,  19.7276,  48.6503, &rU10.c2      );
  make_pt(   25.6482,  18.1987,  50.2518, &rU10.c4      );
  make_pt(   25.9847,  16.9266,  50.6092, &rU10.c5      );
  make_pt(   26.0918,  16.6439,  51.8416, &rU10.c6      );
  make_pt(   26.2067,  15.9515,  49.5943, &rU10._.U.o2  );
  make_pt(   26.0713,  16.3497,  48.3080, &rU10._.U.o4  );
  make_pt(   25.4890,  18.9105,  51.0618, &rU10._.U.h3  );
  make_pt(   26.4742,  14.9310,  49.8682, &rU10._.U.h5  );
  make_pt(   26.2346,  15.6394,  47.4975, &rU10._.U.h6  );

  rUs[0] = &rU10;
  rUs[1] = &rU09;
  rUs[2] = &rU08;
  rUs[3] = &rU07;
  rUs[4] = &rU06;
  rUs[5] = &rU05;
  rUs[6] = &rU04;
  rUs[7] = &rU03;
  rUs[8] = &rU02;
  rUs[9] = &rU01;
  rUs[10]= 0;
}

/*---------------------------------------------------------------------------*/

/* POINTS */

void pt_sub( p1, p2, p3 )
pt p1, p2, p3;
{
  p3->x = p1->x - p2->x;
  p3->y = p1->y - p2->y;
  p3->z = p1->z - p2->z;
}

FLOAT pt_dist( p1, p2 )
pt p1, p2;
{
  FLOAT dx = p1->x - p2->x, dy = p1->y - p2->y, dz = p1->z - p2->z;
  return sqrt( dx*dx + dy*dy + dz*dz );
}

FLOAT pt_phi( p )
pt p;
{
  FLOAT x = p->x, y = p->y, z = p->z;
  FLOAT b = atan2( x, z );
  return atan2( cos(b)*z + sin(b)*x, y );
}

FLOAT pt_theta( p )
pt p;
{
  return atan2( p->x, p->z );
}

/*---------------------------------------------------------------------------*/

/* COORDINATE TRANSFORMATIONS */

struct struct_tfo tfo_id =
{
  1.0, 0.0, 0.0,
  0.0, 1.0, 0.0,
  0.0, 0.0, 1.0,
  0.0, 0.0, 0.0
};

void tfo_apply( t, p1, p2 )
tfo t;
pt p1, p2;
{
  FLOAT x = p1->x, y = p1->y, z = p1->z;
  p2->x = x*t->m[0][0] + y*t->m[1][0] + z*t->m[2][0] + t->m[3][0];
  p2->y = x*t->m[0][1] + y*t->m[1][1] + z*t->m[2][1] + t->m[3][1];
  p2->z = x*t->m[0][2] + y*t->m[1][2] + z*t->m[2][2] + t->m[3][2];
}

void tfo_combine( a, b, t )
tfo a, b, t;
{
  FLOAT x, y, z;

  x = a->m[0][0];  y = a->m[0][1];  z = a->m[0][2];
  t->m[0][0] = x*b->m[0][0] + y*b->m[1][0] + z*b->m[2][0];
  t->m[0][1] = x*b->m[0][1] + y*b->m[1][1] + z*b->m[2][1];
  t->m[0][2] = x*b->m[0][2] + y*b->m[1][2] + z*b->m[2][2];

  x = a->m[1][0];  y = a->m[1][1];  z = a->m[1][2];
  t->m[1][0] = x*b->m[0][0] + y*b->m[1][0] + z*b->m[2][0];
  t->m[1][1] = x*b->m[0][1] + y*b->m[1][1] + z*b->m[2][1];
  t->m[1][2] = x*b->m[0][2] + y*b->m[1][2] + z*b->m[2][2];

  x = a->m[2][0];  y = a->m[2][1];  z = a->m[2][2];
  t->m[2][0] = x*b->m[0][0] + y*b->m[1][0] + z*b->m[2][0];
  t->m[2][1] = x*b->m[0][1] + y*b->m[1][1] + z*b->m[2][1];
  t->m[2][2] = x*b->m[0][2] + y*b->m[1][2] + z*b->m[2][2];

  x = a->m[3][0];  y = a->m[3][1];  z = a->m[3][2];
  t->m[3][0] = x*b->m[0][0] + y*b->m[1][0] + z*b->m[2][0] + b->m[3][0];
  t->m[3][1] = x*b->m[0][1] + y*b->m[1][1] + z*b->m[2][1] + b->m[3][1];
  t->m[3][2] = x*b->m[0][2] + y*b->m[1][2] + z*b->m[2][2] + b->m[3][2];
}

void tfo_inv_ortho( t1, t2 )
tfo t1, t2;
{
  FLOAT tx = t1->m[3][0], ty = t1->m[3][1], tz = t1->m[3][2];
  t2->m[0][0] = t1->m[0][0];
  t2->m[0][1] = t1->m[1][0];
  t2->m[0][2] = t1->m[2][0];
  t2->m[1][0] = t1->m[0][1];
  t2->m[1][1] = t1->m[1][1];
  t2->m[1][2] = t1->m[2][1];
  t2->m[2][0] = t1->m[0][2];
  t2->m[2][1] = t1->m[1][2];
  t2->m[2][2] = t1->m[2][2];
  t2->m[3][0] = -(tx*t1->m[0][0] + ty*t1->m[0][1] + tz*t1->m[0][2]);
  t2->m[3][1] = -(tx*t1->m[1][0] + ty*t1->m[1][1] + tz*t1->m[1][2]);
  t2->m[3][2] = -(tx*t1->m[2][0] + ty*t1->m[2][1] + tz*t1->m[2][2]);
}

void tfo_align( p1, p2, p3, t )
pt p1, p2, p3;
tfo t;
{
  FLOAT x1 = p1->x, y1 = p1->y, z1 = p1->z;
  FLOAT x3 = p3->x, y3 = p3->y, z3 = p3->z;
  FLOAT x31 = x3-x1, y31 = y3-y1, z31 = z3-z1;
  struct struct_pt rotpy;
  pt_sub( p2, p1, &rotpy );
  {
    FLOAT phi = pt_phi( &rotpy ), theta = pt_theta( &rotpy );
    FLOAT sinp = sin(phi), sint = sin(theta);
    FLOAT cosp = cos(phi), cost = cos(theta);
    FLOAT sinpsint = sinp*sint;
    FLOAT sinpcost = sinp*cost;
    FLOAT cospsint = cosp*sint;
    FLOAT cospcost = cosp*cost;
    struct struct_pt rotpz;
    make_pt( cost*x31 - sint*z31,
             sinpsint*x31 + cosp*y31 + sinpcost*z31,
             cospsint*x31 - sinp*y31 + cospcost*z31,
             &rotpz );
    {
      FLOAT rho = pt_theta( &rotpz );
      FLOAT cosr = cos( rho ), sinr = sin( rho );
      FLOAT x = -x1*cost + z1*sint;
      FLOAT y = -x1*sinpsint - y1*cosp - z1*sinpcost;
      FLOAT z = -x1*cospsint + y1*sinp - z1*cospcost;
      t->m[0][0] = cost*cosr - cospsint*sinr;
      t->m[0][1] = sinpsint;
      t->m[0][2] = cost*sinr + cospsint*cosr;
      t->m[1][0] = sinp*sinr;
      t->m[1][1] = cosp;
      t->m[1][2] = -sinp*cosr;
      t->m[2][0] = -sint*cosr - cospcost*sinr;
      t->m[2][1] = sinpcost;
      t->m[2][2] = -sint*sinr + cospcost*cosr;
      t->m[3][0] = x*cosr - z*sinr;
      t->m[3][1] = y;
      t->m[3][2] = x*sinr + z*cosr;
    }
  }
}

/*---------------------------------------------------------------------------*/

void init_nucleotides()
{
  init_A();
  init_C();
  init_G();
  init_U();

  rG_.type = 'G';
  make_tfo(  -0.2067,  -0.0264,   0.9780,
              0.9770,  -0.0586,   0.2049,
              0.0519,   0.9979,   0.0379,
              1.0331, -46.8078, -36.4742, &rG_.dgf_base_tfo );
  make_tfo(  -0.8644,  -0.4956,  -0.0851,
             -0.0427,   0.2409,  -0.9696,
              0.5010,  -0.8345,  -0.2294,
              4.0167,  54.5377,  12.4779, &rG_.p_o3_275_tfo );
  make_tfo(   0.3706,  -0.6167,   0.6945,
             -0.2867,  -0.7872,  -0.5460,
              0.8834,   0.0032,  -0.4686,
            -52.9020,  18.6313,  -0.6709, &rG_.p_o3_180_tfo );
  make_tfo(   0.4155,   0.9025,  -0.1137,
              0.9040,  -0.4236,  -0.0582,
             -0.1007,  -0.0786,  -0.9918,
             -7.6624, -25.2080,  49.5181, &rG_.p_o3_60_tfo  );
  make_pt(   31.3810,   0.1400,  47.5810, &rG_.p        );
  make_pt(   29.9860,   0.6630,  47.6290, &rG_.o1p      );
  make_pt(   31.7210,  -0.6460,  48.8090, &rG_.o2p      );
  make_pt(   32.4940,   1.2540,  47.2740, &rG_.o5_      );
  make_pt(   32.1610,   2.2370,  46.2560, &rG_.c5_      );
  make_pt(   31.2986,   2.8190,  46.5812, &rG_.h5_      );
  make_pt(   32.0980,   1.7468,  45.2845, &rG_.h5__     );
  make_pt(   33.3476,   3.1959,  46.1947, &rG_.c4_      );
  make_pt(   33.2668,   3.8958,  45.3630, &rG_.h4_      );
  make_pt(   33.3799,   3.9183,  47.4216, &rG_.o4_      );
  make_pt(   34.6515,   3.7222,  48.0398, &rG_.c1_      );
  make_pt(   35.2947,   4.5412,  47.7180, &rG_.h1_      );
  make_pt(   35.1756,   2.4228,  47.4827, &rG_.c2_      );
  make_pt(   34.6778,   1.5937,  47.9856, &rG_.h2__     );
  make_pt(   36.5631,   2.2672,  47.4798, &rG_.o2_      );
  make_pt(   37.0163,   2.6579,  48.2305, &rG_.h2_      );
  make_pt(   34.6953,   2.5043,  46.0448, &rG_.c3_      );
  make_pt(   34.5444,   1.4917,  45.6706, &rG_.h3_      );
  make_pt(   35.6679,   3.3009,  45.3487, &rG_.o3_      );
  make_pt(   37.4804,   4.0914,  52.2559, &rG_.n1       );
  make_pt(   36.9670,   4.1312,  49.9281, &rG_.n3       );
  make_pt(   37.8045,   4.2519,  50.9550, &rG_.c2       );
  make_pt(   35.7171,   3.8264,  50.3222, &rG_.c4       );
  make_pt(   35.2668,   3.6420,  51.6115, &rG_.c5       );
  make_pt(   36.2037,   3.7829,  52.6706, &rG_.c6       );
  make_pt(   39.0869,   4.5552,  50.7092, &rG_._.G.n2   );
  make_pt(   33.9075,   3.3338,  51.6102, &rG_._.G.n7   );
  make_pt(   34.6126,   3.6358,  49.5108, &rG_._.G.n9   );
  make_pt(   33.5805,   3.3442,  50.3425, &rG_._.G.c8   );
  make_pt(   35.9958,   3.6512,  53.8724, &rG_._.G.o6   );
  make_pt(   38.2106,   4.2053,  52.9295, &rG_._.G.h1   );
  make_pt(   39.8218,   4.6863,  51.3896, &rG_._.G.h21  );
  make_pt(   39.3420,   4.6857,  49.7407, &rG_._.G.h22  );
  make_pt(   32.5194,   3.1070,  50.2664, &rG_._.G.h8   );

  rU_.type = 'U';
  make_tfo(  -0.0109,   0.5907,   0.8068,
              0.2217,  -0.7853,   0.5780,
              0.9751,   0.1852,  -0.1224,
             -1.4225, -11.0956,  -2.5217, &rU_.dgf_base_tfo );
  make_tfo(  -0.8313,  -0.4738,  -0.2906,
              0.0649,   0.4366,  -0.8973,
              0.5521,  -0.7648,  -0.3322,
              1.6833,   6.8060,  -7.0011, &rU_.p_o3_275_tfo );
  make_tfo(   0.3445,  -0.7630,   0.5470,
             -0.4628,  -0.6450,  -0.6082,
              0.8168,  -0.0436,  -0.5753,
             -6.8179,  -3.9778,  -5.9887, &rU_.p_o3_180_tfo );
  make_tfo(   0.5855,   0.7931,  -0.1682,
              0.8103,  -0.5790,   0.0906,
             -0.0255,  -0.1894,  -0.9816,
              6.1203,  -7.1051,   3.1984, &rU_.p_o3_60_tfo  );
  make_pt(    2.6760,  -8.4960,   3.2880, &rU_.p        );
  make_pt(    1.4950,  -7.6230,   3.4770, &rU_.o1p      );
  make_pt(    2.9490,  -9.4640,   4.3740, &rU_.o2p      );
  make_pt(    3.9730,  -7.5950,   3.0340, &rU_.o5_      );
  make_pt(    5.2430,  -8.2420,   2.8260, &rU_.c5_      );
  make_pt(    5.1974,  -8.8497,   1.9223, &rU_.h5_      );
  make_pt(    5.5548,  -8.7348,   3.7469, &rU_.h5__     );
  make_pt(    6.3140,  -7.2060,   2.5510, &rU_.c4_      );
  make_pt(    5.8744,  -6.2116,   2.4731, &rU_.h4_      );
  make_pt(    7.2798,  -7.2260,   3.6420, &rU_.o4_      );
  make_pt(    8.5733,  -6.9410,   3.1329, &rU_.c1_      );
  make_pt(    8.9047,  -6.0374,   3.6446, &rU_.h1_      );
  make_pt(    8.4429,  -6.6596,   1.6327, &rU_.c2_      );
  make_pt(    9.2880,  -7.1071,   1.1096, &rU_.h2__     );
  make_pt(    8.2502,  -5.2799,   1.4754, &rU_.o2_      );
  make_pt(    8.7676,  -4.7284,   2.0667, &rU_.h2_      );
  make_pt(    7.1642,  -7.4416,   1.3021, &rU_.c3_      );
  make_pt(    7.4125,  -8.5002,   1.2260, &rU_.h3_      );
  make_pt(    6.5160,  -6.9772,   0.1267, &rU_.o3_      );
  make_pt(    9.4531,  -8.1107,   3.4087, &rU_.n1       );
  make_pt(   11.5931,  -9.0015,   3.6357, &rU_.n3       );
  make_pt(   10.8101,  -7.8950,   3.3748, &rU_.c2       );
  make_pt(   11.1439, -10.2744,   3.9206, &rU_.c4       );
  make_pt(    9.7056, -10.4026,   3.9332, &rU_.c5       );
  make_pt(    8.9192,  -9.3419,   3.6833, &rU_.c6       );
  make_pt(   11.3013,  -6.8063,   3.1326, &rU_._.U.o2   );
  make_pt(   11.9431, -11.1876,   4.1375, &rU_._.U.o4   );
  make_pt(   12.5840,  -8.8673,   3.6158, &rU_._.U.h3   );
  make_pt(    9.2891, -11.2898,   4.1313, &rU_._.U.h5   );
  make_pt(    7.9263,  -9.4537,   3.6977, &rU_._.U.h6   );
}

/*---------------------------------------------------------------------------*/

/* PARTIAL INSTANTIATIONS */

struct struct_var
{
  int i;
  tfo t;
  nuc n;
  int on_stack;
};

typedef struct struct_var *var;

typedef struct struct_var *var_list;

struct struct_sol_list
{
  var_list sol;
  struct struct_sol_list *next;
};

typedef struct struct_sol_list *sol_list;

struct struct_var partial_inst[200];
int partial_inst_length;

#ifdef lazy

#define absolute_pos(var,pt1,pt2)\
{ if ((pt1)->relative) { tfo_apply((var)->t,pt1,pt1); (pt1)->relative = 0; }\
  *(pt2) = *(pt1);\
}

void make_relative_nuc( t, n1, n2 )
tfo t;
nuc n1, n2;
{
  *n2 = *n1;
  n2->p.relative = 1;
  n2->o1p.relative = 1;
  n2->o2p.relative = 1;
  n2->o5_.relative = 1;
  n2->c5_.relative = 1;
  n2->h5_.relative = 1;
  n2->h5__.relative = 1;
  n2->c4_.relative = 1;
  n2->h4_.relative = 1;
  n2->o4_.relative = 1;
  n2->c1_.relative = 1;
  n2->h1_.relative = 1;
  n2->c2_.relative = 1;
  n2->h2__.relative = 1;
  n2->o2_.relative = 1;
  n2->h2_.relative = 1;
  n2->c3_.relative = 1;
  n2->h3_.relative = 1;
  n2->o3_.relative = 1;
  n2->n1.relative = 1;
  n2->n3.relative = 1;
  n2->c2.relative = 1;
  n2->c4.relative = 1;
  n2->c5.relative = 1;
  n2->c6.relative = 1;

  switch (n2->type)
  {
    case 'A':
      n2->_.A.n6.relative = 1;
      n2->_.A.n7.relative = 1;
      n2->_.A.n9.relative = 1;
      n2->_.A.c8.relative = 1;
      n2->_.A.h2.relative = 1;
      n2->_.A.h61.relative = 1;
      n2->_.A.h62.relative = 1;
      n2->_.A.h8.relative = 1;
      break;

    case 'C':
      n2->_.C.n4.relative = 1;
      n2->_.C.o2.relative = 1;
      n2->_.C.h41.relative = 1;
      n2->_.C.h42.relative = 1;
      n2->_.C.h5.relative = 1;
      n2->_.C.h6.relative = 1;
      break;

    case 'G':
      n2->_.G.n2.relative = 1;
      n2->_.G.n7.relative = 1;
      n2->_.G.n9.relative = 1;
      n2->_.G.c8.relative = 1;
      n2->_.G.o6.relative = 1;
      n2->_.G.h1.relative = 1;
      n2->_.G.h21.relative = 1;
      n2->_.G.h22.relative = 1;
      n2->_.G.h8.relative = 1;
      break;

    case 'U':
      n2->_.U.o2.relative = 1;
      n2->_.U.o4.relative = 1;
      n2->_.U.h3.relative = 1;
      n2->_.U.h5.relative = 1;
      n2->_.U.h6.relative = 1;
      break;
  }
}  

#else

#define absolute_pos(var,pt1,pt2)tfo_apply((var)->t,pt1,pt2)

#endif

#define atom_pos(atom,var,pos)\
{ pt x = &(var)->n->atom; absolute_pos(var,x,pos); }

var get_var( i )
int i;
{
  var v = &partial_inst[partial_inst_length-1];
  while (v->i != i) v--;
  return v;
}

/*---------------------------------------------------------------------------*/

/* SEARCH */

void (*seq)();
int (*constraint)();
sol_list solutions, last_solution;

void search( problem_domains, problem_constraint )
void (*problem_domains)();
int (*problem_constraint)();
{
  seq = problem_domains;
  constraint = problem_constraint;
  solutions = NULL;
  last_solution = NULL;
  partial_inst_length = 0;
  (*seq)();
}

void try( i, t, n )
int i;
tfo t;
nuc n;
{
  var v = &partial_inst[partial_inst_length];
#ifdef lazy
  struct struct_nuc rel_n;
  make_relative_nuc( t, n, &rel_n );
  v->n = &rel_n;
#else
  v->n = n;
#endif
  v->i = i;
  v->t = t;
  v->on_stack = 1;
  if ((*constraint)( v ))
  {
    partial_inst_length++;
    (*seq)();
    partial_inst_length--;
  }
}

void found_solution()
{
  int i;
  sol_list sols;
  var_list sol;
  sol = (var_list)malloc( (partial_inst_length+1)*sizeof(struct struct_var) );
  if (sol==NULL) { printf( "memory overflow\n" ); exit(1); }
  for (i=0; i<partial_inst_length; i++)
  {
    if (partial_inst[i].on_stack)
    {
      tfo t;
#ifdef lazy
      nuc n;
#endif
      t = (tfo)malloc( sizeof(struct struct_tfo) );
      if (t==NULL) { printf( "memory overflow\n" ); exit(1); }
      *t = *partial_inst[i].t;
      partial_inst[i].t = t;
#ifdef lazy
      n = (nuc)malloc( sizeof(struct struct_nuc) );
      if (n==NULL) { printf( "memory overflow\n" ); exit(1); }
      *n = *partial_inst[i].n;
      partial_inst[i].n = n;
#endif
      partial_inst[i].on_stack = 0;
    }
    sol[i].i = partial_inst[i].i;
    sol[i].t = partial_inst[i].t;
    sol[i].n = partial_inst[i].n;
  }
  sol[i].i = -1;
  sols = (sol_list)malloc( sizeof(struct struct_sol_list) );
  if (sols==NULL) { printf( "memory overflow\n" ); exit(1); }
  sols->sol = sol;
  sols->next = NULL;
  if (last_solution == NULL)
    solutions = sols;
  else
    last_solution->next = sols;
  last_solution = sols;
}

/*---------------------------------------------------------------------------*/

/* DOMAINS */

void dgf_base( t1, ref, n, t2 )
tfo t1;
var ref;
nuc n;
tfo t2;
{
  nuc ref_n = ref->n;
  struct struct_pt p1, p2, p3;
  struct struct_tfo m, align;
  switch (ref_n->type)
  {
    case 'A':
      atom_pos( c1_, ref, &p1 );
      atom_pos( _.A.n9, ref, &p2 );
      atom_pos( c4, ref, &p3 );
      break;
    case 'C':
      atom_pos( c1_, ref, &p1 );
      atom_pos( n1, ref, &p2 );
      atom_pos( c2, ref, &p3 );
      break;
    case 'G':
      atom_pos( c1_, ref, &p1 );
      atom_pos( _.G.n9, ref, &p2 );
      atom_pos( c4, ref, &p3 );
      break;
    case 'U':
      atom_pos( c1_, ref, &p1 );
      atom_pos( n1, ref, &p2 );
      atom_pos( c2, ref, &p3 );
      break;
  }
  tfo_align( &p1, &p2, &p3, &m );
  tfo_inv_ortho( &m, &align );
  tfo_combine( t1, &align, &m );
  tfo_combine( &n->dgf_base_tfo, &m, t2 );
}

void reference( n, i )
nuc n;
int i;
{
  try( i, &tfo_id, n );
}

struct struct_tfo wc_tfo =
{
  -1.0000,  0.0028, -0.0019,
   0.0028,  0.3468, -0.9379,
  -0.0019, -0.9379, -0.3468,
  -0.0080,  6.0730,  8.7208
};

void wc( n, i, j )
nuc n;
int i, j;
{
  struct struct_tfo t;
  dgf_base( &wc_tfo, get_var(j), n, &t );
  try( i, &t, n );
}

struct struct_tfo wc_dumas_tfo =
{
  -0.9737, -0.1834,  0.1352,
  -0.1779,  0.2417, -0.9539,
   0.1422, -0.9529, -0.2679,
   0.4837,  6.2649,  8.0285
 };

void wc_dumas( n, i, j )
nuc n;
int i, j;
{
  struct struct_tfo t;
  dgf_base( &wc_dumas_tfo, get_var(j), n, &t );
  try( i, &t, n );
}

struct struct_tfo helix5_tfo =
{
   0.9886, -0.0961,  0.1156,
   0.1424,  0.8452, -0.5152,
  -0.0482,  0.5258,  0.8492,
  -3.8737,  0.5480,  3.8024
};

void helix5( n, i, j )
nuc n;
int i, j;
{
  struct struct_tfo t;
  dgf_base( &helix5_tfo, get_var(j), n, &t );
  try( i, &t, n );
}

struct struct_tfo helix3_tfo =
{
   0.9886,  0.1424, -0.0482,
  -0.0961,  0.8452,  0.5258,
   0.1156, -0.5152,  0.8492,
   3.4426,  2.0474, -3.7042
 };

void helix3( n, i, j )
nuc n;
int i, j;
{
  struct struct_tfo t;
  dgf_base( &helix3_tfo, get_var(j), n, &t );
  try( i, &t, n );
}

struct struct_tfo g37_a38_tfo =
{
   0.9991,  0.0164, -0.0387,
  -0.0375,  0.7616, -0.6470,
   0.0189,  0.6478,  0.7615,
  -3.3018,  0.9975,  2.5585
};
  
void G37_A38( n, i, j )
nuc n;
int i, j;
{
  struct struct_tfo t;
  dgf_base( &g37_a38_tfo, get_var(j), n, &t );
  try( i, &t, n );
}

void stacked5( n, i, j )
nuc n;
int i, j;
{
  G37_A38( n, i, j );
  helix5( n, i, j );
}

struct struct_tfo a38_g37_tfo =
{
   0.9991, -0.0375,  0.0189,
   0.0164,  0.7616,  0.6478,
  -0.0387, -0.6470,  0.7615,
   3.3819,  0.7718, -2.5321
};

void A38_G37( n, i, j )
nuc n;
int i, j;
{
  struct struct_tfo t;
  dgf_base( &a38_g37_tfo, get_var(j), n, &t );
  try( i, &t, n );
}

void stacked3( n, i, j )
nuc n;
int i, j;
{
  A38_G37( n, i, j );
  helix3( n, i, j );
}

void P_O3( n, i, j )
nuc *n;
int i, j;
{
  var ref = get_var(j);
  struct struct_pt p1, p2, p3;
  struct struct_tfo m, align;

  atom_pos( o3_, ref, &p1 );
  atom_pos( c3_, ref, &p2 );
  atom_pos( c4_, ref, &p3 );
  tfo_align( &p1, &p2, &p3, &m );
  tfo_inv_ortho( &m, &align );

  while (*n!=NULL)
  {
    tfo_combine( &(*n)->p_o3_60_tfo, &align, &m );
    try( i, &m, *n );
    tfo_combine( &(*n)->p_o3_180_tfo, &align, &m );
    try( i, &m, *n );
    tfo_combine( &(*n)->p_o3_275_tfo, &align, &m );
    try( i, &m, *n );
    n++;
  }
}

/*---------------------------------------------------------------------------*/

/* PROBLEM STATEMENT */

/* Anticodon problem */

void anticodon_domains()
{
  switch (partial_inst_length)
  {
    case  0: reference( &rC, 27     ); break;
    case  1: helix5(    &rC, 28, 27 ); break;
    case  2: helix5(    &rA, 29, 28 ); break;
    case  3: helix5(    &rG, 30, 29 ); break;
    case  4: helix5(    &rA, 31, 30 ); break;
    case  5: wc(        &rU, 39, 31 ); break;
    case  6: helix5(    &rC, 40, 39 ); break;
    case  7: helix5(    &rU, 41, 40 ); break;
    case  8: helix5(    &rG, 42, 41 ); break;
    case  9: helix5(    &rG, 43, 42 ); break;
    case 10: stacked3(  &rA, 38, 39 ); break;
    case 11: stacked3(  &rG, 37, 38 ); break;
    case 12: stacked3(  &rA, 36, 37 ); break;
    case 13: stacked3(  &rA, 35, 36 ); break;
    case 14: stacked3(  &rG, 34, 35 ); break;
    case 15: P_O3(      rCs, 32, 31 ); break;
    case 16: P_O3(      rUs, 33, 32 ); break;
    case 17: found_solution();
  }
}

int anticodon_constraint( v )
var v;
{
  if (v->i == 33)
  {
    var ref;
    struct struct_pt p, o3_;
    ref = get_var(34);
    atom_pos( p,   ref, &p );
    atom_pos( o3_, v,   &o3_ );
    return (pt_dist( &p, &o3_ ) < 3.0);
  }
  else
    return 1;
}

void anticodon()
{
  search( anticodon_domains, anticodon_constraint );
}

/* Pseudoknot problem */

void pseudoknot_domains()
{
  switch (partial_inst_length)
  {
    case  0: reference( &rA, 23     ); break;
    case  1: wc_dumas(  &rU,  8, 23 ); break;
    case  2: helix3(    &rG, 22, 23 ); break;
    case  3: wc_dumas(  &rC,  9, 22 ); break;
    case  4: helix3(    &rG, 21, 22 ); break;
    case  5: wc_dumas(  &rC, 10, 21 ); break;
    case  6: helix3(    &rC, 20, 21 ); break;
    case  7: wc_dumas(  &rG, 11, 20 ); break;
    case  8: helix3(    &rU_,19, 20 ); break;
    case  9: wc_dumas(  &rA, 12, 19 ); break;
    case 10: helix3(    &rC,  3, 19 ); break;
    case 11: wc_dumas(  &rG, 13,  3 ); break;
    case 12: helix3(    &rC,  2,  3 ); break;
    case 13: wc_dumas(  &rG, 14,  2 ); break;
    case 14: helix3(    &rC,  1,  2 ); break;
    case 15: wc_dumas(  &rG_,15,  1 ); break;
    case 16: P_O3(      rUs, 16, 15 ); break;
    case 17: P_O3(      rCs, 17, 16 ); break;
    case 18: P_O3(      rAs, 18, 17 ); break;
    case 19: helix3(    &rU,  7,  8 ); break;
    case 20: P_O3(      rCs,  4,  3 ); break;
    case 21: stacked5(  &rU,  5,  4 ); break;
    case 22: stacked5(  &rC,  6,  5 ); break;
    case 23: found_solution();
  }
}

int pseudoknot_constraint( v )
var v;
{
  switch (v->i)
  {
    case 18:
    {
      var ref;
      struct struct_pt p, o3_;
      ref = get_var(19);
      atom_pos( p,   ref, &p );
      atom_pos( o3_, v,   &o3_ );
      return (pt_dist( &p, &o3_ ) <= 4.0);
    }
    case 6:
    {
      var ref;
      struct struct_pt p, o3_;
      ref = get_var(7);
      atom_pos( p,   ref, &p );
      atom_pos( o3_, v,   &o3_ );
      return (pt_dist( &p, &o3_ ) <= 4.5);
    }
    default:
      return 1;
  }
}

void pseudoknot()
{
  search( pseudoknot_domains, pseudoknot_constraint );
}

/*---------------------------------------------------------------------------*/

/* TESTING */

pt point_list[50];

pt *list_of_atoms( n )
nuc n;
{
  pt *ptr;
  ptr = point_list;
  *ptr++ = &n->p;
  *ptr++ = &n->o1p;
  *ptr++ = &n->o2p;
  *ptr++ = &n->o5_;
  *ptr++ = &n->c5_;
  *ptr++ = &n->h5_;
  *ptr++ = &n->h5__;
  *ptr++ = &n->c4_;
  *ptr++ = &n->h4_;
  *ptr++ = &n->o4_;
  *ptr++ = &n->c1_;
  *ptr++ = &n->h1_;
  *ptr++ = &n->c2_;
  *ptr++ = &n->h2__;
  *ptr++ = &n->o2_;
  *ptr++ = &n->h2_;
  *ptr++ = &n->c3_;
  *ptr++ = &n->h3_;
  *ptr++ = &n->o3_;
  *ptr++ = &n->n1;
  *ptr++ = &n->n3;
  *ptr++ = &n->c2;
  *ptr++ = &n->c4;
  *ptr++ = &n->c5;
  *ptr++ = &n->c6;

  switch (n->type)
  {
    case 'A':
      *ptr++ = &n->_.A.n6;
      *ptr++ = &n->_.A.n7;
      *ptr++ = &n->_.A.n9;
      *ptr++ = &n->_.A.c8;
      *ptr++ = &n->_.A.h2;
      *ptr++ = &n->_.A.h61;
      *ptr++ = &n->_.A.h62;
      *ptr++ = &n->_.A.h8;
      break;

    case 'C':
      *ptr++ = &n->_.C.n4;
      *ptr++ = &n->_.C.o2;
      *ptr++ = &n->_.C.h41;
      *ptr++ = &n->_.C.h42;
      *ptr++ = &n->_.C.h5;
      *ptr++ = &n->_.C.h6;
      break;

    case 'G':
      *ptr++ = &n->_.G.n2;
      *ptr++ = &n->_.G.n7;
      *ptr++ = &n->_.G.n9;
      *ptr++ = &n->_.G.c8;
      *ptr++ = &n->_.G.o6;
      *ptr++ = &n->_.G.h1;
      *ptr++ = &n->_.G.h21;
      *ptr++ = &n->_.G.h22;
      *ptr++ = &n->_.G.h8;
      break;

    case 'U':
      *ptr++ = &n->_.U.o2;
      *ptr++ = &n->_.U.o4;
      *ptr++ = &n->_.U.h3;
      *ptr++ = &n->_.U.h5;
      *ptr++ = &n->_.U.h6;
      break;
  }
  *ptr = NULL;
  return point_list;
}

FLOAT distance( p )
pt p;
{
  FLOAT x = p->x, y = p->y, z = p->z;
  return sqrt( x*x + y*y + z*z );
}

FLOAT m; /* current maximal distance */

void var_most_distant_atom( v )
var v;
{
  pt *ptr;
  ptr = list_of_atoms( v->n );
  while (*ptr != NULL)
  {
    pt p;
    struct struct_pt abs_pos;
    FLOAT x;
    p = *ptr++;
    absolute_pos( v, p, &abs_pos );
    x = distance( &abs_pos );
    if (x > m) m = x;
  }
}

void sol_most_distant_atom( s )
var_list s;
{
  while (s->i >= 0)
  {
    var_most_distant_atom( s );
    s++;
  }
}

FLOAT most_distant_atom( sols )
sol_list sols;
{
  m = 0.0; /* set current maximum to 0 */

  while (sols!=NULL)
  {
    sol_most_distant_atom( sols->sol );
    sols = sols->next;
  }
  return m;
}

void check()
{
  int n;

  pseudoknot();

  n = 0;
  while (solutions!=NULL)
  {
    n++;
    solutions = solutions->next;
  }
  printf( "Number of solutions: %d\n", n );
}

FLOAT run()
{
  init_nucleotides();
  pseudoknot();
  return most_distant_atom(solutions);
}

int main (argc, argv)
int argc;
char *argv[];
{
  int i;
  FLOAT result;

  for (i=0; i<10; i++)
    result = run ();

  if ((result <=  33.797594890762724*0.999999) ||
      (result >=  33.797594890762724*1.000001))
    printf ("*** wrong result ***\n");

  return 0;
}

/*---------------------------------------------------------------------------*/
