;==============================================================================

; file: "_eval.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(##include "header.scm")

;------------------------------------------------------------------------------

(define ##self-var     (##make-uninterned-symbol "<self>"))
(define ##selector-var (##make-uninterned-symbol "<selector>"))
(define ##do-loop-var  (##make-uninterned-symbol "<do-loop>"))

(define (##mk-code* code-prc cte src stepper lst n)
  (let ((code (##make-vector (##fixnum.+ (##length lst) (##fixnum.+ n 5)) #f)))
    (##vector-set! code 0 #f)
    (##vector-set! code 1 code-prc)
    (##vector-set! code 2 cte)
    (##vector-set! code 3 src)
    (##vector-set! code 4 stepper)
    (let loop ((i 0) (l lst))
      (if (##pair? l)
        (let ((child (##car l)))
          (##vector-set! child 0 code)
          (code-set! code i child)
          (loop (##fixnum.+ i 1) (##cdr l)))
        code))))

(define ##step-handlers (make-step-handlers))

(define ##main-stepper (make-main-stepper))

(define (##current-stepper) ##main-stepper)

(define (##no-stepper) (make-no-stepper))

;------------------------------------------------------------------------------

; Structure representing source code.

(define ##source1-marker '#(source1)) ; source markers
(define ##source2-marker '#(source2))

(define (##make-source code loc)
  (##vector ##source1-marker
            code
            (if loc (##vector-ref loc 0) #f)
            (if loc (##vector-ref loc 1) #f)))

(define (##sourcify x src)
  (if (##source? x)
    x
    (##vector ##source2-marker
              x
              (##vector-ref src 2)
              (##vector-ref src 3))))

(define (##source? x)
  (and (##vector? x)
       (##fixnum.< 0 (##vector-length x))
       (let ((y (##vector-ref x 0)))
         (or (##eq? y ##source1-marker)
             (##eq? y ##source2-marker)))))

(define (##source-code src)
  (##vector-ref src 1))

(define (##source-locat src)
  (let ((file (##vector-ref src 2)))
    (if file
      (##make-locat file
                    (##vector-ref src 3))
      #f)))

(define (##desourcify src)

  (define (desourcify-list lst)
    (cond ((##pair? lst)
           (##cons (##desourcify (##car lst))
                   (desourcify-list (##cdr lst))))
          ((##null? lst)
           '())
          (else
           (##desourcify lst))))

  (define (desourcify-vector vect)
    (let* ((len (##vector-length vect))
           (x (##make-vector len #f)))
      (let loop ((i (##fixnum.- len 1)))
        (if (##fixnum.< i 0)
          x
          (begin
            (##vector-set! x i (##desourcify (##vector-ref vect i)))
            (loop (##fixnum.- i 1)))))))

  (if (##source? src)
    (let ((code (##source-code src)))
      (if (##eq? (##vector-ref src 0) ##source2-marker)
        code
        (cond ((##pair? code)
               (desourcify-list code))
              ((##vector? code)
               (desourcify-vector code))
              (else
               code))))
    src))

;------------------------------------------------------------------------------

; Structure representing a location in a source file.

(define (##make-locat file filepos)
  (##vector file filepos))

(define (##locat? x)
  (##vector? x))

(define (##locat-file locat)
  (##vector-ref locat 0))

(define (##locat-filepos locat)
  (##vector-ref locat 1))

;==============================================================================

; Compiler

;------------------------------------------------------------------------------

; Compile time environments

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Representation of local variables (up and over) and global variable.

(##define-macro (mk-loc-access up over) `(##cons ,up ,over))
(##define-macro (loc-access? x) `(##pair? ,x))
(##define-macro (loc-access-up x) `(##car ,x))
(##define-macro (loc-access-over x) `(##cdr ,x))

(##define-macro (mk-glo-access id)
  `(or (make-global-var ,id)
       (ct-error-global-env-overflow ,id)))

(##define-macro (glo-access? x)
  `(##not (##pair? ,x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Representation of compile time environments

; There are 4 types of structures in a compile time environment:
;
;    top        end of the environment and container for current state
;    frame      binding context for variables
;    macro      binding context for a macro
;    namespace  binding context for a namespace

(define (##cte-top top-cte)
  (##vector top-cte))

(define (##cte-top? cte)
  (##fixnum.= (##vector-length cte) 1))

(define (##cte-top-cte cte)
  (##vector-ref cte 0))

(define (##cte-top-cte-set! cte new-cte)
  (##vector-set! cte 0 new-cte))

(define (##cte-parent-cte cte)
  (##vector-ref cte 0))

(define (##cte-frame parent-cte vars)
  (##vector parent-cte vars))

(define (##cte-frame? cte)
  (##fixnum.= (##vector-length cte) 2))

(define (##cte-frame-vars cte)
  (##vector-ref cte 1))

(define (##cte-macro parent-cte name def)
  (##vector parent-cte name def))

(define (##cte-macro? cte)
  (and (##fixnum.= (##vector-length cte) 3)
       (##not (##string? (##vector-ref cte 1))))) ; distinguish from namespace

(define (##cte-macro-name cte)
  (##vector-ref cte 1))

(define (##cte-macro-def cte)
  (##vector-ref cte 2))

(define (##cte-namespace parent-cte prefix vars)
  (##vector parent-cte prefix vars))

(define (##cte-namespace? cte)
  (and (##fixnum.= (##vector-length cte) 3)
       (##string? (##vector-ref cte 1)))) ; distinguish from macro

(define (##cte-namespace-prefix cte)
  (##vector-ref cte 1))

(define (##cte-namespace-vars cte)
  (##vector-ref cte 2))

(define (##cte-relink cte new-parent-cte)
  (if new-parent-cte
    (cond ((##cte-frame? cte)
           (##cte-frame new-parent-cte
                        (##cte-frame-vars cte)))
          ((##cte-macro? cte)
           (##cte-macro new-parent-cte
                        (##cte-macro-name cte)
                        (##cte-macro-def cte)))
          ((##cte-namespace? cte)
           (##cte-namespace new-parent-cte
                            (##cte-namespace-prefix cte)
                            (##cte-namespace-vars cte))))
    #f))

(define (##cte-add-macro parent-cte name def)

  (define (replace cte)
    (cond ((##cte-top? cte)
           #f)
          ((and (##cte-macro? cte) (##eq? name (##cte-macro-name cte)))
           (##cte-macro (##cte-parent-cte cte) name def))
          ((##cte-namespace? cte)
           #f) ; don't go beyond a namespace declaration
          (else
           (##cte-relink cte (replace (##cte-parent-cte cte))))))

  (or (replace parent-cte)
      (##cte-macro parent-cte name def)))

(define (##cte-add-namespace parent-cte prefix vars)

  (define (replace cte)
    (cond ((##cte-top? cte)
           #f)
          ((##cte-namespace? cte)
           (if (##pair? (##cte-namespace-vars cte))
             (replace (##cte-parent-cte cte))
             (##cte-namespace (##cte-parent-cte cte) prefix vars)))
          (else
           #f))) ; don't go beyond a frame or macro definition

  (if (##pair? vars)
    (##cte-namespace parent-cte prefix vars)
    (or (replace parent-cte)
        (##cte-namespace parent-cte prefix vars))))

(define (##check-namespace src)
  (let ((code (##source-code src)))
    (let loop1 ((forms (##cdr code)))
      (if (##pair? forms)
        (let* ((form-src (##sourcify (##car forms) src))
               (form (##source-code form-src)))
          (if (##pair? form)
            (let* ((space-src (##sourcify (##car form) form-src))
                   (space (##source-code space-src)))
              (if (##string? space)
                (if (##valid-prefix? space)
                  (let loop2 ((lst (##cdr form)))
                    (cond ((##pair? lst)
                           (let* ((id-src
                                   (##sourcify (##car lst) form-src))
                                  (id
                                   (##source-code id-src)))
                             (if (##not (##symbol? id))
                               (ct-error-syntax
                                id-src
                                "Identifier expected"))
                             (loop2 (##cdr lst))))
                          ((##not (##null? lst))
                           (ct-error-syntax
                            form-src
                            "Ill-formed namespace"))))
                  (ct-error-syntax
                   space-src
                   "Ill-formed namespace prefix"))
                (ct-error-syntax
                 space-src
                 "Namespace prefix must be a string")))
            (ct-error-syntax
             form-src
             "Ill-formed namespace")))
        (loop1 (##cdr forms))))))

(define (##cte-process-declare parent-cte src)
  parent-cte) ; simply ignore ##declare

(define (##cte-process-namespace parent-cte src)
  (##check-namespace src)
  (let ((forms (##cdr (##desourcify src))))
    (let loop ((cte parent-cte) (forms forms))
      (if (##pair? forms)
        (let ((form (##car forms)))
          (loop (##cte-add-namespace cte (##car form) (##cdr form))
                (##cdr forms)))
        cte))))

(define (##cte-get-top-cte cte)
  (if (##cte-top? cte)
    cte
    (##cte-get-top-cte (##cte-parent-cte cte))))

(define (##cte-mutate-top-cte! cte proc)
  (let ((top-cte (##cte-get-top-cte cte)))
    (##cte-top-cte-set! top-cte (proc (##cte-top-cte top-cte)))))

(define (##make-top-cte)
  (let ((top-cte (##cte-top #f)))
    (##cte-top-cte-set! top-cte top-cte)
    top-cte))

(define (##top-cte-add-macro! top-cte name def)
  (let ((global-name (##cte-global-macro-name (##cte-top-cte top-cte) name)))
    (##cte-mutate-top-cte!
      top-cte
      (lambda (cte) (##cte-add-macro cte global-name def)))))

(define (##top-cte-process-declare! top-cte src)
  (##cte-mutate-top-cte!
    top-cte
    (lambda (cte) (##cte-process-declare cte src))))

(define (##top-cte-process-namespace! top-cte src)
  (##cte-mutate-top-cte!
    top-cte
    (lambda (cte) (##cte-process-namespace cte src))))

(define (##top-cte-clone top-cte)
  (let ((new-top-cte (##cte-top #f)))

    (define (clone cte)
      (if (##cte-top? cte)
        new-top-cte
        (##cte-relink cte (clone (##cte-parent-cte cte)))))

    (##cte-top-cte-set! new-top-cte (clone (##cte-top-cte top-cte)))
    new-top-cte))

(define (##cte-lookup cte name)
  (##declare (inlining-limit 500)) ; inline CTE access procedures
  (let loop ((name name) (full? (##full-name? name)) (cte cte) (up 0))
    (if (##cte-top? cte)
      (##vector 'not-found name)
      (let ((parent-cte (##cte-parent-cte cte)))
        (cond ((##cte-frame? cte)
               (let* ((vars (##cte-frame-vars cte))
                      (x (##memq name vars)))
                 (if x
                   (##vector
                     'var
                     name
                     up
                     (##fixnum.+ (##fixnum.- (##length vars) (##length x)) 1))
                   (loop name full? parent-cte (##fixnum.+ up 1)))))
              ((##cte-macro? cte)
               (if (##eq? name (##cte-macro-name cte))
                 (##vector 'macro name (##cte-macro-def cte))
                 (loop name full? parent-cte up)))
              ((and (##not full?) (##cte-namespace? cte))
               (let ((vars (##cte-namespace-vars cte)))
                 (if (or (##not (##pair? vars)) (##memq name vars))
                   (loop (##make-full-name (##cte-namespace-prefix cte) name)
                         #t
                         parent-cte
                         up)
                   (loop name full? parent-cte up))))
              (else
               (loop name full? parent-cte up)))))))

(define (##cte-global-macro-name cte name)
  (if (##full-name? name)
    name
    (let loop ((cte cte))
      (if (##cte-top? cte)
        name
        (let ((parent-cte (##cte-parent-cte cte)))
          (cond ((##cte-namespace? cte)
                 (let ((vars (##cte-namespace-vars cte)))
                   (if (or (##not (##null? vars)) (##memq name vars))
                     (##make-full-name (##cte-namespace-prefix cte) name)
                     (loop parent-cte))))
                (else
                 (loop parent-cte))))))))

(define (##full-name? sym) ; full name if it contains a "#"
  (let ((str (##symbol->string sym)))
    (let loop ((i (##fixnum.- (##string-length str) 1)))
      (if (##fixnum.< i 0)
        #f
        (if (##char=? (##string-ref str i) #\#)
          #t
          (loop (##fixnum.- i 1)))))))

(define (##make-full-name prefix sym)
  (if (##fixnum.= (##string-length prefix) 0)
    sym
    (##string->symbol (##string-append prefix (##symbol->string sym)))))

(define (##valid-prefix? str)      ; non-null name followed by a "#" at end is
  (let ((l (##string-length str))) ; valid as is the special prefix ""
    (or (##fixnum.= l 0)
        (and (##not (##fixnum.< l 2))
             (##char=? (##string-ref str (##fixnum.- l 1)) #\#)))))

(define (##var-lookup cte src)
  (let* ((name (##source-code src))
         (ind (##cte-lookup cte name)))
    (case (##vector-ref ind 0)
      ((not-found)
       (mk-glo-access (##vector-ref ind 1)))
      ((var)
       (mk-loc-access (##vector-ref ind 2) (##vector-ref ind 3)))
      (else
       (ct-error-syntax src "Macro name can't be used as a variable:" name)))))

(define ##macro-lookup #f)
(set! ##macro-lookup
  (lambda (cte name)
    (and (##symbol? name)
         (let ((ind (##cte-lookup cte name)))
           (case (##vector-ref ind 0)
             ((macro)
              (##vector-ref ind 2))
             (else
              #f))))))

(define ##macro-expand #f)
(set! ##macro-expand
  (lambda (cte src def)
    (##shape src src (##car def))
    (##apply (##cdr def) (##cdr (##desourcify src)))))

(define ##macro-def #f)
(set! ##macro-def
  (lambda (src)
    (let ((code (##source-code src)))
      (if (and (##pair? code)
               (##eq? (##source-code (##sourcify (##car code) src)) 'lambda))
        (begin
          (##shape src src -3)
          (##cons (##form-size (##source-code (##sourcify (##cadr code) src)))
                  (##eval-top src ##interaction-cte)))
      (ct-error-syntax src "Macro value must be a lambda expression")))))

(define (##form-size parms)
  (let loop ((lst parms) (n 1))
    (if (##pair? lst)
      (loop (##cdr lst) (##fixnum.+ n 1))
      (if (##null? lst) n (##fixnum.- 0 n)))))

(define ##interaction-cte
  (##make-top-cte))

;------------------------------------------------------------------------------

; Utilities

(define (##self-eval? val)
  (or (##complex? val)
      (##string? val)
      (##char? val)
      (##keyword? val)
      (##memq val '(#f #t #!eof #!optional #!rest #!key #!))))

(define (##variable src)
  (let ((code (##source-code src)))
    (if (##not (##symbol? code))
      (ct-error-syntax src "Identifier expected"))
    (if (##memq code
                '(quote quasiquote unquote unquote-splicing lambda if set!
                  cond => else and or case let let* letrec begin do define
                  delay future ##include ##define-macro ##declare ##namespace
                  c-define-type c-declare c-initialize c-lambda c-define))
      (ct-error-syntax
       src
       "Reserved identifier can't be used as a variable:"
       code))))

(define (##shape src x size)
  (let* ((code (##source-code x))
         (n (##proper-length code)))
    (if (or (##not n)
            (if (##fixnum.< 0 size)
              (##not (##fixnum.= n size))
              (##fixnum.< n (##fixnum.- 0 size))))
      (ct-error-syntax
       src
       "Ill-formed special form:"
       (##source-code (##sourcify (##car (##source-code src)) src))))))

(define (##proper-length lst)
  (let loop ((lst lst) (n 0))
    (cond ((##pair? lst) (loop (##cdr lst) (##fixnum.+ n 1)))
          ((##null? lst) n)
          (else          #f))))

(define (##include-file-as-a-begin-expr src)
  (let* ((code (##source-code src))
         (filename-src (##sourcify (##cadr code) src))
         (filename (##source-code filename-src)))
    (if (##string? filename)

      (let ((port (##open-input-file filename)))
        (if port
          (let* ((x
                  (##read-all-as-a-begin-expr-from-port
                   port
                   ##main-readtable
                   #t))
                 (expr
                  (##vector-ref x 1)))
            expr)
          (ct-error-syntax
           src
           "Can't find file"
           (or (##path-expand filename 'shortest)
               filename))))

      (ct-error-syntax filename-src "Filename expected"))))

;------------------------------------------------------------------------------

; Compiler's main entry

(define (##compile-top top-cte src)
  (##convert-source-to-locat!
    (let ((tail? #f))
      (##comp-top top-cte src tail?))))

(define (##compile-inner cte src)
  (##convert-source-to-locat!
    (let ((tail? #f))
      (gen ##gen-top src
        (##comp (##cte-frame cte (##list (self-var))) src tail?)))))

(define (##convert-source-to-locat! code)

  (define (convert! file code)
    (let ((locat (##source-locat (code-locat code)))) ; get the source location
      (if (##locat? locat)
        (let ((new-file (##locat-file locat)))
          (if (##equal? file new-file #f)
            (convert2! file (##locat-filepos locat) code)
            (convert2! new-file locat code)))
        (convert2! file #f code))))

  (define (convert2! file locat code)
    (code-locat-set! code locat)
    (let ((n (code-length code)))
      (let loop ((i 0))
        (if (##fixnum.< i n)
          (let ((x (code-ref code i)))
            (if (is-child-code? x code)
              (begin
                (convert! file x)
                (loop (##fixnum.+ i 1)))))))))

  (convert! #f code)
  code)

;------------------------------------------------------------------------------

(define (##comp-top top-cte src tail?)
  (let ((code (##source-code src))
        (cte (##cte-top-cte top-cte)))
    (if (##pair? code)
      (let* ((first-src (##sourcify (##car code) src))
             (first (##source-code first-src))
             (def (##macro-lookup cte first)))
        (if def
          (##comp-top top-cte
                      (##sourcify (##macro-expand cte src def) src)
                      tail?)
          (case first
            ((begin)          (##comp-top-begin top-cte src tail?))
            ((define)         (##comp-top-define top-cte src tail?))
            ((##include)      (##comp-top-include top-cte src tail?))
            ((##define-macro) (##comp-top-define-macro top-cte src tail?))
            ((##declare)      (##comp-top-declare top-cte src tail?))
            ((##namespace)    (##comp-top-namespace top-cte src tail?))
            (else             (##comp-aux cte src tail? first)))))
      (##comp-simple cte src tail?))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-top-begin top-cte src tail?)
  (##shape src src -1)
  (let ((code (##source-code src)))
    (##comp-top-seq top-cte src tail? (##cdr code))))

(define (##comp-top-seq top-cte src tail? seq)
  (if (##pair? seq)
    (##comp-top-seq-aux top-cte src tail? seq)
    (let ((cte (##cte-top-cte top-cte)))
      (gen ##gen-cst-no-step src
        (##void)))))

(define (##comp-top-seq-aux top-cte src tail? seq)
  (let ((first-src (##sourcify (##car seq) src))
        (rest (##cdr seq)))
    (if (##pair? rest)
      (let ((cte (##cte-top-cte top-cte)))
        (gen ##gen-seq first-src
          (##comp-top top-cte first-src #f)
          (##comp-top-seq-aux top-cte src tail? rest)))
      (##comp-top top-cte first-src tail?))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-top-define top-cte src tail?)
  (let ((name (##definition-name src)))
    (##variable name)
    (let* ((cte (##cte-top-cte top-cte))
           (ind (##var-lookup cte name))
           (val (##definition-value src)))
      (gen ##gen-glo-def src
        ind
        (##comp cte val #f)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-top-include top-cte src tail?)
  (##shape src src 2)
  (##comp-top top-cte
              (##include-file-as-a-begin-expr src)
              tail?))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-top-define-macro top-cte src tail?)
  (let* ((cte (##cte-top-cte top-cte))
         (name (##definition-name src))
         (val (##definition-value src)))
    (##top-cte-add-macro!
     top-cte
     (##source-code name)
     (##macro-def val))
    (gen ##gen-cst-no-step src
      (##void))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-top-declare top-cte src tail?)
  (##shape src src -2)
  (let ((cte (##cte-top-cte top-cte)))
    (##top-cte-process-declare! top-cte src)
    (gen ##gen-cst-no-step src
      (##void))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-top-namespace top-cte src tail?)
  (##shape src src -2)
  (let ((cte (##cte-top-cte top-cte)))
    (##top-cte-process-namespace! top-cte src)
    (gen ##gen-cst-no-step src
      (##void))))

;------------------------------------------------------------------------------

(define (##comp cte src tail?)
  (let ((code (##source-code src)))
    (if (##pair? code)
      (let* ((first-src (##sourcify (##car code) src))
             (first (##source-code first-src))
             (def (##macro-lookup cte first)))
        (if def
          (##comp cte
                  (##sourcify (##macro-expand cte src def) src)
                  tail?)
          (case first
            ((begin)          (##comp-begin cte src tail?))
            ((define)         (ct-error-syntax src "Ill-placed 'define'"))
            ((##include)      (ct-error-syntax src "Ill-placed '##include'"))
            ((##define-macro) (ct-error-syntax src "Ill-placed '##define-macro'"))
            ((##declare)      (ct-error-syntax src "Ill-placed '##declare'"))
            ((##namespace)    (ct-error-syntax src "Ill-placed '##namespace'"))
            (else             (##comp-aux cte src tail? first)))))
      (##comp-simple cte src tail?))))

(define (##comp-simple cte src tail?)
  (let ((code (##source-code src)))
    (cond ((##symbol? code)    (##comp-ref cte src tail?))
          ((##self-eval? code) (##comp-cst cte src tail?))
          (else                (ct-error-syntax src "Ill-formed expression")))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-begin cte src tail?)
  (##shape src src -2)
  (let ((code (##source-code src)))
    (##comp-seq cte src tail? (##cdr code))))

(define (##comp-seq cte src tail? seq)
  (if (##pair? seq)
    (##comp-seq-aux cte src tail? seq)
    (gen ##gen-cst-no-step src
      (##void))))

(define (##comp-seq-aux cte src tail? seq)
  (let ((first-src (##sourcify (##car seq) src))
        (rest (##cdr seq)))
    (if (##pair? rest)
      (let ((code (##source-code first-src)))
        (gen ##gen-seq first-src
          (##comp cte first-src #f)
          (##comp-seq-aux cte src tail? rest)))
      (##comp cte first-src tail?))))

;------------------------------------------------------------------------------

(define (##comp-aux cte src tail? first)

  (define (unsupported)
    (ct-error-syntax src "Interpreter does not support" first))

  (case first
    ((quote)            (##comp-quote cte src tail?))
    ((quasiquote)       (##comp-quasiquote cte src tail?))
    ((unquote)          (ct-error-syntax src "Ill-placed 'unquote'"))
    ((unquote-splicing) (ct-error-syntax src "Ill-placed 'unquote-splicing'"))
    ((set!)             (##comp-set! cte src tail?))
    ((lambda)           (##comp-lambda cte src tail?))
    ((if)               (##comp-if cte src tail?))
    ((cond)             (##comp-cond cte src tail?))
    ((and)              (##comp-and cte src tail?))
    ((or)               (##comp-or cte src tail?))
    ((case)             (##comp-case cte src tail?))
    ((let)              (##comp-let cte src tail?))
    ((let*)             (##comp-let* cte src tail?))
    ((letrec)           (##comp-letrec cte src tail?))
    ((do)               (##comp-do cte src tail?))
    ((delay)            (##comp-delay cte src tail?))
    ((future)           (##comp-future cte src tail?))
    ((c-define-type)    (unsupported))
    ((c-declare)        (unsupported))
    ((c-initialize)     (unsupported))
    ((c-lambda)         (unsupported))
    ((c-define)         (unsupported))
    (else               (##comp-app cte src tail?))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-ref cte src tail?)
  (##variable src)
  (let ((x (##var-lookup cte src)))
    (if (loc-access? x)
      (let ((up (loc-access-up x))
            (over (loc-access-over x)))
        (gen ##gen-loc-ref src
          up
          over))
      (gen ##gen-glo-ref src
        x))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-cst cte src tail?)
  (gen ##gen-cst src
    (##desourcify src)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-quote cte src tail?)
  (##shape src src 2)
  (let ((code (##source-code src)))
    (gen ##gen-cst src
      (##desourcify (##cadr code)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-quasiquote cte src tail?)
  (##shape src src 2)
  (let ((code (##source-code src)))
    (##comp-template cte
                     src
                     tail?
                     (##sourcify (##cadr code) src)
                     1)))

;*********************** fix me

(define (##comp-template cte src tail? form-src depth)
  (let ((form (##source-code form-src)))
    (cond ((##pair? form)
           (##comp-list-template cte
                                 src
                                 tail?
                                 form
                                 depth))
          ((##vector? form)
           (gen ##gen-quasi-list->vector src
             (##comp-list-template cte
                                   src
                                   #f
                                   (##vector->list form)
                                   depth)))
          (else
           (gen ##gen-cst-no-step src
             (##desourcify form-src))))))

(define (##comp-list-template cte src tail? lst depth)
  (cond ((##pair? lst)
         (let* ((first-src (##sourcify (##car lst) src))
                (first (##source-code first-src)))

           (define (non-special-list)
             (if (and (##pair? first)
                      (##eq? (##source-code
                              (##sourcify (##car first) first-src))
                             'unquote-splicing)
                      (##pair? (##cdr first)) ; proper list of length 2?
                      (##null? (##cddr first)))
               (if (##eq? depth 1)
                 (let ((second-src (##sourcify (##cadr first) src)))
                   (if (##null? (##cdr lst))
                     (##comp cte second-src tail?)
                     (gen ##gen-quasi-append src
                       (##comp cte second-src #f)
                       (##comp-list-template cte
                                             src
                                             #f
                                             (##cdr lst)
                                             depth))))
                 (gen ##gen-quasi-cons src
                   (##comp-template cte
                                    src
                                    #f
                                    first-src
                                    (##fixnum.- depth 1))
                   (##comp-list-template cte
                                         src
                                         #f
                                         (##cdr lst)
                                         depth)))
               (gen ##gen-quasi-cons src
                 (##comp-template cte
                                  src
                                  #f
                                  first-src
                                  depth)
                 (##comp-list-template cte
                                       src
                                       #f
                                       (##cdr lst)
                                       depth))))

           (if (and (##pair? (##cdr lst)) ; proper list of length 2?
                    (##null? (##cddr lst)))
             (case first
               ((quasiquote)
                (gen ##gen-quasi-cons src
                  (gen ##gen-cst-no-step first-src
                    first)
                  (##comp-list-template cte
                                        src
                                        #f
                                        (##cdr lst)
                                        (##fixnum.+ depth 1))))
               ((unquote)
                (if (##eq? depth 1)
                  (##comp cte (##sourcify (##cadr lst) first-src) tail?)
                  (gen ##gen-quasi-cons src
                    (gen ##gen-cst-no-step first-src
                      first)
                    (##comp-list-template cte
                                          src
                                          #f
                                          (##cdr lst)
                                          (##fixnum.- depth 1)))))
               (else
                (non-special-list)))
             (non-special-list))))

         ((##null? lst)
          (gen ##gen-cst-no-step src
            '()))

         (else
          (##comp-template cte
                           src
                           tail?
                           lst
                           depth))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-set! cte src tail?)
  (##shape src src 3)
  (let* ((code (##source-code src))
         (var-src (##sourcify (##cadr code) src))
         (val-src (##sourcify (##caddr code) src)))
    (##variable var-src)
    (let ((x (##var-lookup cte var-src)))
      (if (loc-access? x)
        (let ((up (loc-access-up x))
              (over (loc-access-over x)))
          (gen ##gen-loc-set src
            up
            over
            (##comp cte val-src #f)))
        (gen ##gen-glo-set src
          x
          (##comp cte val-src #f))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-lambda cte src tail?)
  (##shape src src -3)
  (let* ((code (##source-code src))
         (parms-src (##sourcify (##cadr code) src))
         (body (##cddr code)))
    (##comp-lambda-aux cte src tail? parms-src body)))

(define (##comp-lambda-aux cte src tail? parms-src body)
  (let* ((all-parms
          (##extract-parameters src parms-src))
         (required-parameters
          (##vector-ref all-parms 0))
         (optional-parameters
          (##vector-ref all-parms 1))
         (rest-parameter
          (##vector-ref all-parms 2))
         (key-parameters
          (##vector-ref all-parms 3)))
    (let loop1 ((frame required-parameters)
                (lst (or optional-parameters '()))
                (rev-inits '()))
      (if (##pair? lst)
        (let ((x (##car lst))
              (new-cte (##cte-frame cte (##cons (self-var) frame))))
          (loop1 (##append frame (##list (##car x)))
                 (##cdr lst)
                 (##cons (##comp new-cte (##cdr x) #f)
                         rev-inits)))
        (let loop2 ((frame (if rest-parameter
                             (##append frame (##list rest-parameter))
                             frame))
                    (lst (or key-parameters '()))
                    (rev-inits rev-inits)
                    (rev-keys '()))
          (if (##pair? lst)
            (let ((x (##car lst))
                  (new-cte (##cte-frame cte (##cons (self-var) frame))))
              (loop2 (##append frame (##list (##car x)))
                     (##cdr lst)
                     (##cons (##comp new-cte (##cdr x) #f)
                             rev-inits)
                     (##cons (##string->keyword (##symbol->string (##car x)))
                             rev-keys)))
            (let* ((new-cte (##cte-frame cte (##cons (self-var) frame)))
                   (c (##comp-body new-cte src #t body)))
              (cond ((or optional-parameters key-parameters)
                     (gen ##gen-prc src
                       frame
                       rest-parameter
                       (and key-parameters
                            (##list->vector (##reverse rev-keys)))
                       c
                       (##reverse rev-inits)))
                    (rest-parameter
                     (gen ##gen-prc-rest src
                       frame
                       c))
                    (else
                     (gen ##gen-prc-req src
                       frame
                       c))))))))))

(define (##extract-parameters src parms-src)

  (define (parm-expected-err src)
    (ct-error-syntax src "Parameter must be an identifier"))

  (define (parm-or-default-binding-expected-err src)
    (ct-error-syntax src "Parameter must be an identifier or default binding"))

  (define (duplicate-parm-err src)
    (ct-error-syntax src "Duplicate parameter in parameter list"))

  (define (dotted-rest-err src)
    (ct-error-syntax src "Ill-placed dotted rest parameter"))

  (define (rest-parm-expected-err src)
    (ct-error-syntax src "#!rest must be followed by a parameter"))

  (define (default-binding-err src)
    (ct-error-syntax src "Ill-formed default binding"))

  (define (optional-illegal-err src)
    (ct-error-syntax src "Ill-placed #!optional"))

  (define (rest-illegal-err src)
    (ct-error-syntax src "Ill-placed #!rest"))

  (define (key-illegal-err src)
    (ct-error-syntax src "Ill-placed #!key"))

  (define (key-expected-err src)
    (ct-error-syntax src "#!key expected after rest parameter"))

  (define (default-binding-illegal-err src)
    (ct-error-syntax src "Ill-placed default binding"))

  (let loop ((lst (##source->parms parms-src))
             (rev-required-parms '())
             (rev-optional-parms #f)
             (rest-parm #f)
             (rev-key-parms #f)
             (state 1)) ; 1 = required parms or #!optional or #!rest or #!key
                        ; 2 = optional parms or #!rest or #!key
                        ; 3 = #!key
                        ; 4 = key parms

    (define (done rest-parm)
      (##vector (##reverse rev-required-parms)
                (and rev-optional-parms (##reverse rev-optional-parms))
                rest-parm
                (if (or (##not rev-key-parms)
                        (and (##null? rev-key-parms) (##not rest-parm)))
                  #f
                  (##reverse rev-key-parms))))

    (define (check-if-duplicate parm-src)
      (let ((parm (##source-code parm-src)))
        (if (or (##memq parm rev-required-parms)
                (and rev-optional-parms (##assq parm rev-optional-parms))
                (and rest-parm (##eq? parm rest-parm))
                (and rev-key-parms (##assq parm rev-key-parms)))
          (duplicate-parm-err parm-src))))

    (cond ((##null? lst)
           (done rest-parm))
          ((##pair? lst)
           (let* ((parm-src (##sourcify (##car lst) src))
                  (parm (##source-code parm-src)))
             (cond ((##eq? #!optional parm)
                    (if (##fixnum.< 1 state)
                      (optional-illegal-err parm-src))
                    (loop (##cdr lst)
                          rev-required-parms
                          '()
                          #f
                          #f
                          2))
                   ((##eq? #!rest parm)
                    (if (##fixnum.< 2 state)
                      (rest-illegal-err parm-src))
                    (if (##pair? (##cdr lst))
                      (let* ((parm-src (##sourcify (##cadr lst) src))
                             (parm (##source-code parm-src)))
                        (##variable parm-src)
                        (check-if-duplicate parm-src)
                        (loop (##cddr lst)
                              rev-required-parms
                              rev-optional-parms
                              parm
                              #f
                              3))
                      (rest-parm-expected-err parm-src)))
                   ((##eq? #!key parm)
                    (if (##fixnum.< 3 state)
                      (key-illegal-err parm-src))
                    (loop (##cdr lst)
                          rev-required-parms
                          rev-optional-parms
                          rest-parm
                          '()
                          4))
                   ((##fixnum.= state 3)
                    (key-expected-err parm-src))
                   ((##symbol? parm)
                    (##variable parm-src)
                    (check-if-duplicate parm-src)
                    (case state
                      ((1)
                       (loop (##cdr lst)
                             (##cons parm
                                     rev-required-parms)
                             #f
                             #f
                             #f
                             1))
                      ((2)
                       (loop (##cdr lst)
                             rev-required-parms
                             (##cons (##cons parm
                                             (##sourcify #f parm-src))
                                     rev-optional-parms)
                             #f
                             #f
                             2))
                      (else
                       (loop (##cdr lst)
                             rev-required-parms
                             rev-optional-parms
                             rest-parm
                             (##cons (##cons parm
                                             (##sourcify #f parm-src))
                                     rev-key-parms)
                             4))))
                   ((##pair? parm)
                    (if (##not (or (##fixnum.= state 2) (##fixnum.= state 4)))
                      (default-binding-illegal-err parm-src))
                    (let ((len (##proper-length parm)))
                      (if (##not (##eq? len 2))
                        (default-binding-err parm-src)))
                    (let* ((val-src (##sourcify (##cadr parm) parm-src))
                           (parm-src (##sourcify (##car parm) parm-src))
                           (parm (##source-code parm-src)))
                      (##variable parm-src)
                      (check-if-duplicate parm-src)
                      (case state
                        ((2)
                         (loop (##cdr lst)
                               rev-required-parms
                               (##cons (##cons parm val-src)
                                       rev-optional-parms)
                               #f
                               #f
                               2))
                        (else
                         (loop (##cdr lst)
                               rev-required-parms
                               rev-optional-parms
                               rest-parm
                               (##cons (##cons parm val-src)
                                       rev-key-parms)
                               4)))))
                   (else
                    (if (##fixnum.< 1 state)
                      (parm-or-default-binding-expected-err parm-src)
                      (parm-expected-err parm-src))))))
          (else
           (let ((parm-src (##sourcify lst src)))
             (##variable parm-src)
             (if (or rev-optional-parms
                     rest-parm
                     rev-key-parms)
               (dotted-rest-err parm-src))
             (check-if-duplicate parm-src)
             (done (##source-code parm-src)))))))

(define (##source->parms src)
  (let ((x (##source-code src)))
    (if (or (##pair? x) (##null? x)) x src)))

(define (##comp-body cte src tail? body)

  (define (letrec-defines cte rev-vars rev-vals body)
    (if (##pair? body)

      (let* ((src (##sourcify (##car body) src))
             (code (##source-code src)))
        (if (##not (##pair? code))
          (letrec-defines* cte rev-vars rev-vals body)
          (let* ((first-src (##sourcify (##car code) src))
                 (first (##source-code first-src))
                 (def (##macro-lookup cte first)))
            (if def
              (letrec-defines cte
                              rev-vars
                              rev-vals
                              (##cons
                               (##sourcify (##macro-expand cte src def)
                                           src)
                               (##cdr body)))
              (case first
                ((begin)
                 (##shape src src -1)
                 (letrec-defines cte
                                 rev-vars
                                 rev-vals
                                 (##append (##cdr code) (##cdr body))))
                ((define)
                 (let* ((name-src (##definition-name src))
                        (name (##source-code name-src)))
                   (##variable name-src)
                   (if (##memq name rev-vars)
                     (ct-error-syntax name-src "Duplicate definition of a variable"))
                   (let ((val (##definition-value src)))
                     (letrec-defines cte
                                     (##cons name rev-vars)
                                     (##cons val rev-vals)
                                     (##cdr body)))))
                ((##include)
                 (##shape src src 2)
                 (letrec-defines cte
                                 rev-vars
                                 rev-vals
                                 (##cons
                                  (##include-file-as-a-begin-expr src)
                                  (##cdr body))))
                ((##define-macro)
                 (let* ((name-src (##definition-name src))
                        (name (##source-code name-src))
                        (val (##definition-value src)))
                   (letrec-defines (##cte-macro
                                    cte
                                    name
                                    (##macro-def val))
                                   rev-vars
                                   rev-vals
                                   (##cdr body))))
                ((##declare)
                 (##shape src src -2)
                 (letrec-defines (##cte-process-declare cte src)
                                 rev-vars
                                 rev-vals
                                 (##cdr body)))
                ((##namespace)
                 (##shape src src -2)
                 (letrec-defines (##cte-process-namespace cte src)
                                 rev-vars
                                 rev-vals
                                 (##cdr body)))
                (else
                 (letrec-defines* cte rev-vars rev-vals body)))))))

      (ct-error-syntax src "Body must contain at least one expression")))

  (define (letrec-defines* cte rev-vars rev-vals body)
    (if (##null? rev-vars)
      (##comp-seq cte src tail? body)
      (##comp-letrec-aux cte
                         src
                         tail?
                         (##reverse rev-vars)
                         (##reverse rev-vals)
                         body)))

  (letrec-defines cte '() '() body))

(define (##definition-name src)
  (##shape src src -3)
  (let* ((code (##source-code src))
         (pattern-src (##sourcify (##cadr code) src))
         (pattern (##source-code pattern-src)))
    (let ((name-src (if (##pair? pattern)
                      (##sourcify (##car pattern) src)
                      (begin
                        (##shape src src 3)
                        pattern-src))))
      (if (##not (##symbol? (##source-code name-src)))
        (ct-error-syntax name-src "Defined variable must be an identifier"))
      name-src)))

(define (##definition-value src)
  (let* ((code (##source-code src))
         (pattern-src (##sourcify (##cadr code) src))
         (pattern (##source-code pattern-src)))
    (if (##pair? pattern)
      (let ((locat (##source-locat src))
            (parms (##cdr pattern)))
        (##make-source
         (##cons (##make-source 'lambda locat)
                 (##cons (if (##source? parms) ; rest parameter?
                           parms
                           (##make-source parms locat))
                         (##cddr code)))
         locat))
      (##sourcify (##caddr code) src))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-if cte src tail?)
  (##shape src src -3)
  (let* ((code (##source-code src))
         (pre-src (##sourcify (##cadr code) src))
         (con-src (##sourcify (##caddr code) src)))
    (if (##pair? (##cdddr code))
      (let ((alt-src (##sourcify (##cadddr code) src)))
        (##shape src src 4)
        (gen ##gen-if3 src
          (##comp cte pre-src #f)
          (##comp cte con-src tail?)
          (##comp cte alt-src tail?)))
      (begin
        (##shape src src 3)
        (gen ##gen-if2 src
          (##comp cte pre-src #f)
          (##comp cte con-src tail?))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-cond cte src tail?)
  (##shape src src -2)
  (let* ((code (##source-code src))
         (clauses (##cdr code)))
    (##comp-cond-aux cte src tail? clauses)))

(define (##comp-cond-aux cte src tail? clauses)
  (if (##pair? clauses)
    (let* ((clause-src (##sourcify (##car clauses) src))
           (clause (##source-code clause-src)))
      (##shape src clause-src -1)
      (let* ((first-src (##sourcify (##car clause) clause-src))
             (first (##source-code first-src)))
        (cond ((##eq? first 'else)
               (##shape src clause-src -2)
               (if (##not (##null? (##cdr clauses)))
                 (ct-error-syntax clause-src "Else clause must be last"))
               (##comp-seq cte src tail? (##cdr clause)))
              ((##not (##pair? (##cdr clause)))
               (gen ##gen-cond-or src
                 (##comp cte first-src #f)
                 (##comp-cond-aux cte src tail? (##cdr clauses))))
              (else
               (let* ((second-src (##sourcify (##cadr clause) clause-src))
                      (second (##source-code second-src)))
                 (if (##eq? second '=>)
                   (begin
                     (##shape src clause-src 3)
                     (let ((third-src
                            (##sourcify (##caddr clause) clause-src)))
                       (gen ##gen-cond-send src
                         (##comp cte first-src #f)
                         (##comp cte third-src #f)
                         (##comp-cond-aux cte src tail? (##cdr clauses)))))
                   (gen ##gen-cond-if src
                     (##comp cte first-src #f)
                     (##comp-seq cte src tail? (##cdr clause))
                     (##comp-cond-aux cte src tail? (##cdr clauses)))))))))
    (gen ##gen-cst-no-step src
      (##void))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-and cte src tail?)
  (##shape src src -1)
  (let* ((code (##source-code src))
         (rest (##cdr code)))
    (if (##pair? rest)
      (##comp-and-aux cte src tail? rest)
      (gen ##gen-cst src
        #t))))

(define (##comp-and-aux cte src tail? lst)
  (let ((first-src (##sourcify (##car lst) src))
        (rest (##cdr lst)))
    (if (##pair? rest)
      (gen ##gen-and first-src
        (##comp cte first-src #f)
        (##comp-and-aux cte src tail? rest))
      (##comp cte first-src tail?))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-or cte src tail?)
  (##shape src src -1)
  (let* ((code (##source-code src))
         (rest (##cdr code)))
    (if (##pair? rest)
      (##comp-or-aux cte src tail? rest)
      (gen ##gen-cst src
        #f))))

(define (##comp-or-aux cte src tail? lst)
  (let ((first-src (##sourcify (##car lst) src))
        (rest (##cdr lst)))
    (if (##pair? rest)
      (gen ##gen-or first-src
        (##comp cte first-src #f)
        (##comp-or-aux cte src tail? rest))
      (##comp cte first-src tail?))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-case cte src tail?)
  (##shape src src -3)
  (let* ((code (##source-code src))
         (first-src (##sourcify (##cadr code) src))
         (clauses (##cddr code)))
    (gen ##gen-case first-src
      (##comp cte first-src #f)
      (let ((cte (##cte-frame cte (##list (selector-var)))))
        (##comp-case-aux cte src tail? clauses)))))

(define (##comp-case-aux cte src tail? clauses)
  (if (##pair? clauses)
    (let* ((clause-src (##sourcify (##car clauses) src))
           (clause (##source-code clause-src)))
      (##shape src clause-src -2)
      (let* ((first-src (##sourcify (##car clause) clause-src))
             (first (##source-code first-src)))
        (if (##eq? first 'else)
          (begin
            (if (##not (##null? (##cdr clauses)))
              (ct-error-syntax clause-src "Else clause must be last"))
            (gen ##gen-case-else clause-src
              (##comp-seq cte src tail? (##cdr clause))))
          (let ((n (##proper-length first)))
            (if (##not n)
              (ct-error-syntax first-src "Ill-formed selector list"))
            (gen ##gen-case-clause clause-src
              (##desourcify first-src)
              (##comp-seq cte src tail? (##cdr clause))
              (##comp-case-aux cte src tail? (##cdr clauses)))))))
    (gen ##gen-case-else src
      (gen ##gen-cst-no-step src
        (##void)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-let cte src tail?)
  (##shape src src -3)
  (let* ((code (##source-code src))
         (first-src (##sourcify (##cadr code) src))
         (first (##source-code first-src)))
    (if (##symbol? first)
      (begin
        (##shape src src -4)
        (let ((bindings-src (##sourcify (##caddr code) src)))
          (let* ((vars (##bindings->vars src bindings-src #t #f))
                 (vals (##bindings->vals src bindings-src)))
            (gen ##gen-app-no-step src
              (let ((inner-cte (##cte-frame cte (##list first))))
                (gen ##gen-letrec src
                  (##list first)
                  (let ((cte inner-cte)
                        (tail? #f))
                    (##list (gen ##gen-prc-req-no-step src
                              vars
                              (##comp-body (##cte-frame
                                             cte
                                             (##cons (self-var) vars))
                                           src
                                           #t
                                           (##cdddr code)))))
                  (let ((cte inner-cte)
                        (tail? #f))
                    (gen ##gen-loc-ref-no-step src ; fetch loop variable
                      0
                      1))))
              (##comp-vals cte src vals)))))
      (let* ((vars (##bindings->vars src first-src #t #f))
             (vals (##bindings->vals src first-src)))
        (if (##null? vars)
          (##comp-body cte src tail? (##cddr code))
          (let ((c
                 (##comp-body
                   (##cte-frame cte vars)
                   src
                   tail?
                   (##cddr code))))
            (gen ##gen-let src
              vars
              (##comp-vals cte src vals)
              c)))))))

(define (##comp-vals cte src lst)
  (if (##pair? lst)
    (##cons (##comp cte (##sourcify (##car lst) src) #f)
            (##comp-vals cte src (##cdr lst)))
    '()))

(define (##bindings->vars src bindings-src check-duplicates? allow-steps?)

  (define (bindings->vars lst rev-vars)
    (if (##pair? lst)
      (let* ((binding-src (##sourcify (##car lst) src))
             (binding (##source-code binding-src)))
        (if allow-steps?
          (begin
            (##shape src binding-src -2)
            (if (##pair? (##cddr binding)) (##shape src binding-src 3)))
          (##shape src binding-src 2))
        (let* ((first-src (##sourcify (##car binding) binding-src))
               (first (##source-code first-src)))
          (##variable first-src)
          (if (and check-duplicates? (##memq first rev-vars))
            (ct-error-syntax first-src "Duplicate variable in bindings"))
          (bindings->vars (##cdr lst)
                          (##cons first rev-vars))))
      (##reverse rev-vars)))

  (let* ((bindings (##source-code bindings-src))
         (len (##proper-length bindings)))
    (if len
      (bindings->vars bindings '())
      (ct-error-syntax bindings-src "Ill-formed binding list"))))

(define (##bindings->vals src bindings-src)

  (define (bindings->vals lst)
    (if (##pair? lst)
      (let* ((binding-src (##sourcify (##car lst) src))
             (binding (##source-code binding-src)))
        (##cons (##sourcify (##cadr binding) src)
                (bindings->vals (##cdr lst))))
      '()))

  (let ((bindings (##source-code bindings-src)))
    (bindings->vals bindings)))

(define (##bindings->steps src bindings-src)

  (define (bindings->steps lst)
    (if (##pair? lst)
      (let* ((binding-src (##sourcify (##car lst) src))
             (binding (##source-code binding-src)))
        (##cons (##sourcify (if (##pair? (##cddr binding))
                              (##caddr binding)
                              (##car binding))
                            src)
                (bindings->steps (##cdr lst))))
      '()))

  (let ((bindings (##source-code bindings-src)))
    (bindings->steps bindings)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-let* cte src tail?)
  (##shape src src -3)
  (let* ((code (##source-code src))
         (bindings-src (##sourcify (##cadr code) src))
         (vars (##bindings->vars src bindings-src #f #f))
         (vals (##bindings->vals src bindings-src)))
    (##comp-let*-aux cte src tail? vars vals (##cddr code))))

(define (##comp-let*-aux cte src tail? vars vals body)
  (if (##pair? vars)
    (let ((frame (##list (##car vars))))
      (let ((inner-cte (##cte-frame cte frame)))
        (gen ##gen-let src
          frame
          (##list (##comp cte (##car vals) #f))
          (##comp-let*-aux inner-cte
                           src
                           tail?
                           (##cdr vars)
                           (##cdr vals)
                           body))))
    (##comp-body cte src tail? body)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-letrec cte src tail?)
  (##shape src src -3)
  (let* ((code (##source-code src))
         (bindings-src (##sourcify (##cadr code) src))
         (vars (##bindings->vars src bindings-src #t #f))
         (vals (##bindings->vals src bindings-src)))
    (##comp-letrec-aux cte src tail? vars vals (##cddr code))))

(define (##comp-letrec-aux cte src tail? vars vals body)
  (if (##pair? vars)
    (let ((inner-cte (##cte-frame cte vars)))
      (gen ##gen-letrec src
        vars
        (##comp-vals inner-cte src vals)
        (##comp-body inner-cte src tail? body)))
    (##comp-body cte src tail? body)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-do cte src tail?)
  (##shape src src -3)
  (let* ((code (##source-code src))
         (bindings-src (##sourcify (##cadr code) src))
         (exit-src (##sourcify (##caddr code) src))
         (exit (##source-code exit-src)))
    (##shape src exit-src -1)
    (let* ((vars (##bindings->vars src bindings-src #t #t))
           (do-loop-vars (##list (do-loop-var)))
           (inner-cte (##cte-frame cte do-loop-vars)))
      (gen ##gen-letrec src
        do-loop-vars
        (##list
          (let ((cte inner-cte)
                (tail? #f))
            (gen ##gen-prc-req-no-step src
              vars
              (let ((cte (##cte-frame cte (##cons (self-var) vars)))
                    (tail? #t))
                (gen ##gen-if3 src
                  (##comp cte (##sourcify (##car exit) src) #f)
                  (##comp-seq cte src tail? (##cdr exit))
                  (let ((call
                          (gen ##gen-app-no-step src
                            (let ((tail? #f))
                              (gen ##gen-loc-ref-no-step src; fetch do-loop-var
                                1
                                1))
                            (##comp-vals cte
                                         src
                                         (##bindings->steps src
                                                            bindings-src)))))
                    (if (##null? (##cdddr code))
                      call
                      (gen ##gen-seq src
                        (##comp-seq cte src #f (##cdddr code))
                        call))))))))
        (let ((cte inner-cte))
          (gen ##gen-app-no-step src
            (let ((tail? #f))
              (gen ##gen-loc-ref-no-step src ; fetch do-loop-var
                0
                1))
            (##comp-vals cte src (##bindings->vals src bindings-src))))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-app cte src tail?)
  (let* ((code (##source-code src))
         (len (##proper-length code)))
    (if len
      (gen ##gen-app src
        (##comp cte (##sourcify (##car code) src) #f)
        (##comp-vals cte src (##cdr code)))
      (ct-error-syntax src "Ill-formed procedure call"))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-delay cte src tail?)
  (##shape src src 2)
  (let ((code (##source-code src)))
    (gen ##gen-delay src
      (##comp cte (##sourcify (##cadr code) src) #t))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##comp-future cte src tail?)
  (##shape src src 2)
  (let ((code (##source-code src)))
    (gen ##gen-future src
      (##comp cte (##sourcify (##cadr code) src) #t))))

;==============================================================================

; Code generation procedures

;------------------------------------------------------------------------------

(define ##cprc-top
  (mk-cprc
   (let* (($code (^ 0))
          (rte (mk-rte rte #f)))
     (##first-argument #f) ; make sure $code and rte are in environment-map
     (##check-heap-limit)
     (code-run $code))))

(define ##gen-top
  (mk-gen (val)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-top cte src stepper (val)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-cst
  (mk-cprc
   (constant-step! ()
    (^ 0))))

(define ##gen-cst
  (mk-gen (val)
    (let ((stepper (##current-stepper)))
      (mk-code ##cprc-cst cte src stepper () val))))

(define ##gen-cst-no-step
  (mk-gen (val)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-cst cte src stepper () val))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-loc-ref-0-1
  (mk-cprc
   (reference-step! ()
    (rte-ref rte 1))))

(define ##cprc-loc-ref-0-2
  (mk-cprc
   (reference-step! ()
    (rte-ref rte 2))))

(define ##cprc-loc-ref-0-3
  (mk-cprc
   (reference-step! ()
    (rte-ref rte 3))))

(define ##cprc-loc-ref-1-1
  (mk-cprc
   (reference-step! ()
    (rte-ref (rte-up rte) 1))))

(define ##cprc-loc-ref-1-2
  (mk-cprc
   (reference-step! ()
    (rte-ref (rte-up rte) 2))))

(define ##cprc-loc-ref-1-3
  (mk-cprc
   (reference-step! ()
    (rte-ref (rte-up rte) 3))))

(define ##cprc-loc-ref-2-1
  (mk-cprc
   (reference-step! ()
    (rte-ref (rte-up (rte-up rte)) 1))))

(define ##cprc-loc-ref-2-2
  (mk-cprc
   (reference-step! ()
    (rte-ref (rte-up (rte-up rte)) 2))))

(define ##cprc-loc-ref-2-3
  (mk-cprc
   (reference-step! ()
    (rte-ref (rte-up (rte-up rte)) 3))))

(define ##cprc-loc-ref
  (mk-cprc
   (reference-step! ()
    (let ((up (^ 0)))
      (let loop ((e rte) (i up))
        (if (##fixnum.< 0 i)
          (loop (rte-up e) (##fixnum.- i 1))
          (rte-ref e (^ 1))))))))

(define ##gen-loc-ref-aux
  (mk-gen (stepper up over)
    (case up
      ((0)
       (case over
         ((1)  (mk-code ##cprc-loc-ref-0-1 cte src stepper ()))
         ((2)  (mk-code ##cprc-loc-ref-0-2 cte src stepper ()))
         ((3)  (mk-code ##cprc-loc-ref-0-3 cte src stepper ()))
         (else (mk-code ##cprc-loc-ref     cte src stepper () up over))))
      ((1)
       (case over
         ((1)  (mk-code ##cprc-loc-ref-1-1 cte src stepper ()))
         ((2)  (mk-code ##cprc-loc-ref-1-2 cte src stepper ()))
         ((3)  (mk-code ##cprc-loc-ref-1-3 cte src stepper ()))
         (else (mk-code ##cprc-loc-ref     cte src stepper () up over))))
      ((2)
       (case over
         ((1)  (mk-code ##cprc-loc-ref-2-1 cte src stepper ()))
         ((2)  (mk-code ##cprc-loc-ref-2-2 cte src stepper ()))
         ((3)  (mk-code ##cprc-loc-ref-2-3 cte src stepper ()))
         (else (mk-code ##cprc-loc-ref     cte src stepper () up over))))
     (else
      (mk-code ##cprc-loc-ref cte src stepper () up over)))))

(define ##gen-loc-ref
  (mk-gen (up over)
    (let ((stepper (##current-stepper)))
      (gen ##gen-loc-ref-aux src stepper up over))))

(define ##gen-loc-ref-no-step
  (mk-gen (up over)
    (let ((stepper (##no-stepper)))
      (gen ##gen-loc-ref-aux src stepper up over))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-glo-ref
  (mk-cprc
   (reference-step! ()
    (let ((global-val (global-var-ref (^ 0))))
      (let loop ((val global-val))
        (if (unbound? val)
          (loop (rt-error-unbound-global-var $code rte))
          val))))))

(define ##gen-glo-ref
  (mk-gen (ind)
    (let ((stepper (##current-stepper)))
      (mk-code ##cprc-glo-ref cte src stepper () ind))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-loc-set
  (mk-cprc
   (let ((val (code-run (^ 0))))
     (set!-step! (val)
      (let ((up (^ 1)))
        (let loop ((e rte) (i up))
          (if (##fixnum.< 0 i)
            (loop (rte-up e) (##fixnum.- i 1))
            (begin
              (rte-set! e (^ 2) val)
              (##void)))))))))

(define ##gen-loc-set
  (mk-gen (up over val)
    (let ((stepper (##current-stepper)))
      (mk-code ##cprc-loc-set cte src stepper (val) up over))))

(define ##cprc-glo-set
  (mk-cprc
   (let ((val (code-run (^ 0))))
     (set!-step! (val)
      (global-var-set! (^ 1) val)
      (##void)))))

(define ##gen-glo-set
  (mk-gen (ind val)
    (let ((stepper (##current-stepper)))
      (mk-code ##cprc-glo-set cte src stepper (val) ind))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-glo-def
  (mk-cprc
   (let ((rte (##first-argument #f))) ; avoid constant propagation of #f
     (let ((val (code-run (^ 0))))
       (define-step! (val)
         (global-var-set! (^ 1) val)
         (##void))))))

(define ##gen-glo-def
  (mk-gen (ind val)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-glo-def cte src stepper (val) ind))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-if2
  (mk-cprc
   (let ((pred (code-run (^ 0))))
     (force-vars (pred)
       (if (true? pred)
         (code-run (^ 1))
         (##void))))))

(define ##gen-if2
  (mk-gen (pre con)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-if2 cte src stepper (pre con)))))

(define ##cprc-if3
  (mk-cprc
   (let ((pred (code-run (^ 0))))
     (force-vars (pred)
       (if (true? pred)
         (code-run (^ 1))
         (code-run (^ 2)))))))

(define ##gen-if3
  (mk-gen (pre con alt)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-if3 cte src stepper (pre con alt)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-seq
  (mk-cprc
   (begin
     (code-run (^ 0))
     (code-run (^ 1)))))

(define ##gen-seq
  (mk-gen (val1 val2)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-seq cte src stepper (val1 val2)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-quasi-list->vector
  (mk-cprc
   (quasi-list->vector
    (##first-argument ; keep $code and rte in environment-map
      (code-run (^ 0))
      $code
      rte))))

(define ##gen-quasi-list->vector
  (mk-gen (val)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-quasi-list->vector cte src stepper (val)))))

(define ##cprc-quasi-append
  (mk-cprc
   (let* ((val1
           (code-run (^ 0)))
          (val2
           (code-run (^ 1))))
     (##first-argument $code rte) ; keep $code and rte in environment-map
     (quasi-append val1 val2))))

(define ##gen-quasi-append
  (mk-gen (val1 val2)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-quasi-append cte src stepper (val1 val2)))))

(define ##cprc-quasi-cons
  (mk-cprc
   (let* ((val1
           (code-run (^ 0)))
          (val2
           (code-run (^ 1))))
     (##first-argument $code rte) ; keep $code and rte in environment-map
     (quasi-cons val1 val2))))

(define ##gen-quasi-cons
  (mk-gen (val1 val2)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-quasi-cons cte src stepper (val1 val2)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-cond-if
  (mk-cprc
   (let ((pred (code-run (^ 0))))
     (force-vars (pred)
       (if (true? pred)
         (code-run (^ 1))
         (code-run (^ 2)))))))

(define ##gen-cond-if
  (mk-gen (val1 val2 val3)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-cond-if cte src stepper (val1 val2 val3)))))

(define ##cprc-cond-or
  (mk-cprc
   (let ((pred (code-run (^ 0))))
     (force-vars (pred)
       (if (true? pred)
         pred
         (code-run (^ 1)))))))

(define ##gen-cond-or
  (mk-gen (val1 val2)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-cond-or cte src stepper (val1 val2)))))

(define ##cprc-cond-send-red
  (mk-cprc
   (let ((pred (code-run (^ 0))))
     (force-vars (pred)
       (if (true? pred)
         (let ((val (code-run (^ 1))))
           (let loop ((proc val))
             (force-vars (proc)
               (if (##not (##procedure? proc))
                 (loop (rt-error-non-procedure-send $code rte))
                 (call-step! (proc pred)
                   (proc pred))))))
         (code-run (^ 2)))))))

(define ##cprc-cond-send-sub
  (mk-cprc
   (let ((pred (code-run (^ 0))))
     (force-vars (pred)
       (if (true? pred)
         (let ((val (code-run (^ 1))))
           (let loop ((proc val))
             (force-vars (proc)
               (if (##not (##procedure? proc))
                 (loop (rt-error-non-procedure-send $code rte))
                 (##subproblem-apply1 $code rte proc pred)))))
         (code-run (^ 2)))))))

(define ##gen-cond-send
  (mk-gen (val1 val2 val3)
    (let ((stepper (##no-stepper)))
      (mk-code (if tail? ##cprc-cond-send-red ##cprc-cond-send-sub)
               cte
               src
               stepper
               (val1 val2 val3)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-or
  (mk-cprc
   (let ((pred (code-run (^ 0))))
     (force-vars (pred)
       (if (true? pred)
         pred
         (code-run (^ 1)))))))

(define ##gen-or
  (mk-gen (val1 val2)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-or cte src stepper (val1 val2)))))

(define ##cprc-and
  (mk-cprc
   (let ((pred (code-run (^ 0))))
     (force-vars (pred)
       (if (##not (true? pred))
         pred
         (code-run (^ 1)))))))

(define ##gen-and
  (mk-gen (val1 val2)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-and cte src stepper (val1 val2)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-case
  (mk-cprc
   (let ((selector (code-run (^ 0))))
     (force-vars (selector)
       (let* (($code (^ 1))
              (rte (mk-rte rte selector)))
         (##first-argument #f) ; make sure $code and rte are in environment-map
         (##check-heap-limit)
         (code-run $code))))))

(define ##gen-case
  (mk-gen (val1 val2)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-case cte src stepper (val1 val2)))))

(define ##cprc-case-clause
  (mk-cprc
   (if (##case-memv (rte-ref rte 1) (^ 2))
     (code-run (^ 0))
     (code-run (^ 1)))))

(define ##gen-case-clause
  (mk-gen (cases val1 val2)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-case-clause cte src stepper (val1 val2) cases))))

(define ##cprc-case-else
  (mk-cprc
   (code-run (^ 0))))

(define ##gen-case-else
  (mk-gen (val)
    (let ((stepper (##no-stepper)))
      (mk-code ##cprc-case-else cte src stepper (val)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-let
  (mk-cprc
   (let ((ns (##fixnum.- (code-length $code) 2)))
     (let loop1 ((i 1) (args '()))
       (if (##fixnum.< ns i)
         (let ((inner-rte (mk-rte* rte ns)))
           (##check-heap-limit)
           (let loop2 ((i ns) (args args))
             (if (##fixnum.< 0 i)
               (begin
                 (rte-set! inner-rte i (##car args))
                 (loop2 (##fixnum.- i 1) (##cdr args)))
               (let* (($code
                       (^ 0))
                      (rte
                       (##first-argument ; keep $code and rte in environment-map
                         inner-rte
                         rte)))
                 (code-run $code)))))
         (let ((new-args
                (##cons (code-run (code-ref $code i)) args)))
           (##check-heap-limit)
           (loop1 (##fixnum.+ i 1) new-args)))))))

(define ##gen-let
  (mk-gen (vars vals body)
    (let* ((stepper (##no-stepper))
           (c (##mk-code* ##cprc-let cte src stepper (##cons body vals) 1)))
      (code-set! c (##fixnum.+ (##length vals) 1) vars)
      c)))

(define ##cprc-letrec
  (mk-cprc
   (let ((ns (##fixnum.- (code-length $code) 2)))
     (let ((inner-rte (mk-rte* rte ns)))
       (let loop ((i 1))
         (if (##fixnum.< ns i)
           (let* (($code (^ 0))
                  (rte (##first-argument inner-rte rte)))
             (code-run $code))
           (begin
             (rte-set! inner-rte i
               (let* (($code (code-ref $code i))
                      (rte inner-rte))
                 (code-run $code)))
             (loop (##fixnum.+ i 1)))))))))

(define ##gen-letrec
  (mk-gen (vars vals body)
    (let* ((stepper (##no-stepper))
           (c (##mk-code* ##cprc-letrec cte src stepper (##cons body vals) 1)))
      (code-set! c (##fixnum.+ (##length vals) 1) vars)
      c)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-prc-req0
  (mk-cprc
   (lambda-step! ()
     (letrec ((proc
               (lambda arg1-and-up

                 (##define-macro (execute)
                   `(if (##not (##null? arg1-and-up))
                      (rt-error-wrong-nb-args proc arg1-and-up)
                      (let* (($code (^ 0))
                             (rte (mk-rte rte proc)))
                        (##first-argument #f) ; make sure $code and rte are in environment-map
                        (##check-heap-limit)
                        (code-run $code))))

                 (let ((entry-hook (^ 1)))
                   (if entry-hook
                     (let ((exec (lambda () (execute))))
                       (##check-heap-limit)
                       (entry-hook proc arg1-and-up exec))
                     (execute))))))

       (##check-heap-limit)
       (##first-argument ; keep $code and rte in environment-map
         proc
         $code
         rte)))))

(define ##cprc-prc-req1
  (mk-cprc
   (lambda-step! ()
     (letrec ((proc
               (lambda (#!optional (arg1 (absent-obj))
                        #!rest arg2-and-up)

                 (##define-macro (execute)
                   `(if (or (##eq? arg1 (absent-obj))
                            (##not (##null? arg2-and-up)))
                      (let ((args
                             (##cons arg1 arg2-and-up)))
                        (##check-heap-limit)
                        (##first-argument $code rte)
                        (rt-error-wrong-nb-args proc args))
                      (let* (($code (^ 0))
                             (rte (mk-rte rte proc arg1)))
                        (##first-argument #f) ; make sure $code and rte are in environment-map
                        (##check-heap-limit)
                        (code-run $code))))

                 (let ((entry-hook (^ 1)))
                   (if entry-hook
                     (let* ((args
                              (##cons arg1 arg2-and-up))
                            (exec
                             (lambda () (execute))))
                       (##check-heap-limit)
                       (##first-argument $code rte)
                       (entry-hook proc args exec))
                     (execute))))))

       (##check-heap-limit)
       (##first-argument ; keep $code and rte in environment-map
         proc
         $code
         rte)))))

(define ##cprc-prc-req2
  (mk-cprc
   (lambda-step! ()
     (letrec ((proc
               (lambda (#!optional (arg1 (absent-obj))
                                   (arg2 (absent-obj))
                        #!rest arg3-and-up)

                 (##define-macro (execute)
                   `(if (or (##eq? arg2 (absent-obj))
                            (##not (##null? arg3-and-up)))
                      (let ((args
                             (##cons arg1
                                     (##cons arg2 arg3-and-up))))
                        (##check-heap-limit)
                        (##first-argument $code rte)
                        (rt-error-wrong-nb-args proc args))
                      (let* (($code (^ 0))
                             (rte (mk-rte rte proc arg1 arg2)))
                        (##first-argument #f) ; make sure $code and rte are in environment-map
                        (##check-heap-limit)
                        (code-run $code))))

                 (let ((entry-hook (^ 1)))
                   (if entry-hook
                     (let* ((args
                              (##cons arg1
                                      (##cons arg2 arg3-and-up)))
                            (exec
                             (lambda () (execute))))
                       (##check-heap-limit)
                       (##first-argument $code rte)
                       (entry-hook proc args exec))
                     (execute))))))

       (##check-heap-limit)
       (##first-argument ; keep $code and rte in environment-map
         proc
         $code
         rte)))))

(define ##cprc-prc-req3
  (mk-cprc
   (lambda-step! ()
     (letrec ((proc
               (lambda (#!optional (arg1 (absent-obj))
                                   (arg2 (absent-obj))
                                   (arg3 (absent-obj))
                        #!rest arg4-and-up)

                 (##define-macro (execute)
                   `(if (or (##eq? arg3 (absent-obj))
                            (##not (##null? arg4-and-up)))
                      (let ((args
                             (##cons arg1
                                     (##cons arg2
                                             (##cons arg3 arg4-and-up)))))
                        (##check-heap-limit)
                        (##first-argument $code rte)
                        (rt-error-wrong-nb-args proc args))
                      (let* (($code (^ 0))
                             (rte (mk-rte rte proc arg1 arg2 arg3)))
                        (##first-argument #f) ; make sure $code and rte are in environment-map
                        (##check-heap-limit)
                        (code-run $code))))

                 (let ((entry-hook (^ 1)))
                   (if entry-hook
                     (let* ((args
                              (##cons arg1
                                      (##cons arg2
                                              (##cons arg3 arg4-and-up))))
                            (exec
                             (lambda () (execute))))
                       (##check-heap-limit)
                       (##first-argument $code rte)
                       (entry-hook proc args exec))
                     (execute))))))

       (##check-heap-limit)
       (##first-argument ; keep $code and rte in environment-map
         proc
         $code
         rte)))))

(define ##cprc-prc-req
  (mk-cprc
   (lambda-step! ()
     (letrec ((proc
               (lambda args

                 (define (execute)
                   (let ((ns (^ 1)))
                     (let ((inner-rte (mk-rte* rte ns)))
                       (##check-heap-limit)
                       (rte-set! inner-rte 1 proc)
                       (let loop ((i 2) (lst args))
                         (if (##fixnum.< ns i)
                           (if (##pair? lst)
                             (rt-error-wrong-nb-args proc args)
                             (let* (($code (^ 0))
                                    (rte (##first-argument inner-rte rte)))
                               (code-run $code)))
                           (if (##pair? lst)
                             (begin
                               (rte-set! inner-rte i (##car lst))
                               (loop (##fixnum.+ i 1) (##cdr lst)))
                             (rt-error-wrong-nb-args proc args)))))))

                 (let ((entry-hook (^ 2)))
                   (if entry-hook
                     (let ((exec
                            (lambda () (execute))))
                       (##check-heap-limit)
                       (##first-argument $code rte)
                       (entry-hook proc args exec))
                     (execute))))))

       (##check-heap-limit)
       (##first-argument ; keep $code and rte in environment-map
         proc
         $code
         rte)))))

(define ##gen-prc-req-aux
  (mk-gen (stepper frame body)
    (let ((n (##length frame)))
      (case n
        ((0)
         (mk-code ##cprc-prc-req0 cte src stepper (body) #f frame))
        ((1)
         (mk-code ##cprc-prc-req1 cte src stepper (body) #f frame))
        ((2)
         (mk-code ##cprc-prc-req2 cte src stepper (body) #f frame))
        ((3)
         (mk-code ##cprc-prc-req3 cte src stepper (body) #f frame))
        (else
         (let ((n+1 (##fixnum.+ n 1)))
           (mk-code ##cprc-prc-req  cte src stepper (body) n+1 #f frame)))))))

(define ##gen-prc-req
  (mk-gen (frame body)
    (let ((stepper (##current-stepper)))
      (gen ##gen-prc-req-aux src stepper frame body))))

(define ##gen-prc-req-no-step
  (mk-gen (frame body)
    (let ((stepper (##no-stepper)))
      (gen ##gen-prc-req-aux src stepper frame body))))

(define ##cprc-prc-rest
  (mk-cprc
   (lambda-step! ()
     (letrec ((proc
               (lambda args

                 (define (execute)
                   (let ((ns (^ 1)))
                     (let ((inner-rte (mk-rte* rte ns)))
                       (##check-heap-limit)
                       (rte-set! inner-rte 1 proc)
                       (let loop ((i 2) (lst args))
                         (if (##fixnum.< i ns)
                           (if (##pair? lst)
                             (begin
                               (rte-set! inner-rte i (##car lst))
                               (loop (##fixnum.+ i 1) (##cdr lst)))
                             (rt-error-wrong-nb-args proc args))
                           (begin
                             (rte-set! inner-rte i lst)
                             (let* (($code (^ 0))
                                    (rte (##first-argument inner-rte rte)))
                               (code-run $code))))))))

                 (let ((entry-hook (^ 2)))
                   (if entry-hook
                     (let ((exec
                            (lambda () (execute))))
                       (##check-heap-limit)
                       (##first-argument $code rte)
                       (entry-hook proc args exec))
                     (execute))))))

       (##check-heap-limit)
       (##first-argument ; keep $code and rte in environment-map
         proc
         $code
         rte)))))

(define ##gen-prc-rest
  (mk-gen (frame body)
    (let ((stepper (##current-stepper))
          (n+1 (##fixnum.+ (##length frame) 1)))
      (mk-code ##cprc-prc-rest cte src stepper (body) n+1 #f frame))))

(define ##cprc-prc
  (mk-cprc
   (lambda-step! ()
     (letrec ((proc
               (lambda args

                 (define (execute)
                   (let* ((n
                           (code-length $code))
                          (inner-rte
                           (mk-rte* rte (code-ref $code (##fixnum.- n 7)))))
  
                     (define (get-keys i j left rest? keys)
                       (let loop1 ((lst left))
                         (if (##pair? lst)
                           (let ((key (##car lst)) (lst (##cdr lst)))
                             (if (##keyword? key)
                               (if rest?
                                 (if (##pair? lst)
                                   (loop1 (##cdr lst))
                                   (rt-error-wrong-nb-args proc args))
                                 (let loop2 ((k (##fixnum.-
                                                 (##vector-length keys)
                                                 1)))
                                   (cond ((##fixnum.< k 0)
                                          (rt-error-unknown-keyword-arg
                                           proc
                                           args))
                                         ((##eq? key (##vector-ref keys k))
                                          (if (##pair? lst)
                                            (loop1 (##cdr lst))
                                            (rt-error-wrong-nb-args
                                             proc
                                             args)))
                                         (else
                                          (loop2 (##fixnum.- k 1))))))
                               (rt-error-keyword-expected proc args)))
                           (let loop3 ((i i) (j j) (k 0))
                             (if (##fixnum.< k (##vector-length keys))
                               (let ((key (##vector-ref keys k)))
                                 (let loop4 ((lst left))
                                   (if (##pair? lst)
                                     (if (##eq? (##car lst) key)
                                       (begin
                                         (rte-set! inner-rte i (##cadr lst))
                                         (loop3 (##fixnum.+ i 1)
                                                (##fixnum.+ j 1)
                                                (##fixnum.+ k 1)))
                                       (loop4 (##cddr lst)))
                                     (begin
                                       (rte-set! inner-rte i
                                         (let* (($code (code-ref $code j))
                                                (rte inner-rte))
                                           (code-run $code)))
                                       (loop3 (##fixnum.+ i 1)
                                              (##fixnum.+ j 1)
                                              (##fixnum.+ k 1))))))
                               (let* (($code (^ 0))
                                      (rte (##first-argument inner-rte rte)))
                                 (code-run $code)))))))

                     (##check-heap-limit)
                     (rte-set! inner-rte 1 proc)
                     (let loop1 ((i 2) (lst args))
                       (if (##fixnum.< i (code-ref $code (##fixnum.- n 6)))
                         (if (##pair? lst)
                           (begin
                             (rte-set! inner-rte i (##car lst))
                             (loop1 (##fixnum.+ i 1) (##cdr lst)))
                           (rt-error-wrong-nb-args proc args))
                         (let loop2 ((i i) (j 1) (lst lst))
                           (if (##fixnum.< i (code-ref $code (##fixnum.- n 5)))
                             (if (##pair? lst)
                               (begin
                                 (rte-set! inner-rte i (##car lst))
                                 (loop2 (##fixnum.+ i 1)
                                        (##fixnum.+ j 1)
                                        (##cdr lst)))
                               (begin
                                 (rte-set! inner-rte i
                                   (let* (($code (code-ref $code j))
                                          (rte inner-rte))
                                     (code-run $code)))
                                 (loop2 (##fixnum.+ i 1)
                                        (##fixnum.+ j 1)
                                        '())))
                             (let ((keys (code-ref $code (##fixnum.- n 3))))
                               (cond ((code-ref $code (##fixnum.- n 4))
                                      (rte-set! inner-rte i lst)
                                      (if keys
                                        (get-keys (##fixnum.+ i 1) j lst #t keys)
                                        (let* (($code
                                                (^ 0))
                                               (rte
                                                (##first-argument
                                                  inner-rte
                                                  rte)))
                                          (code-run $code))))
                                     (keys
                                      (get-keys i j lst #f keys))
                                     ((##null? lst)
                                      (let* (($code
                                              (^ 0))
                                             (rte
                                              (##first-argument
                                                inner-rte
                                                rte)))
                                        (code-run $code)))
                                     (else
                                      (rt-error-wrong-nb-args
                                       proc
                                       args))))))))))

                 (let ((entry-hook
                        (code-ref $code (##fixnum.- (code-length $code) 2))))
                   (if entry-hook
                     (let ((exec
                            (lambda () (execute))))
                       (##check-heap-limit)
                       (##first-argument $code rte)
                       (entry-hook proc args exec))
                     (execute))))))

       (##check-heap-limit)
       (##first-argument ; keep $code and rte in environment-map
         proc
         $code
         rte)))))

(define ##gen-prc
  (mk-gen (frame rest? keys body inits)
    (let* ((stepper (##current-stepper))
           (n (##length frame))
           (ni (##length inits))
           (nr (##fixnum.- (##fixnum.- n ni) (if rest? 1 0)))
           (no (##fixnum.- ni (if keys (##vector-length keys) 0)))
           (c (##mk-code* ##cprc-prc cte src stepper (##cons body inits) 7)))
      (code-set! c (##fixnum.+ ni 1) (##fixnum.+ n 1))
      (code-set! c (##fixnum.+ ni 2) (##fixnum.+ nr 2))
      (code-set! c (##fixnum.+ ni 3) (##fixnum.+ (##fixnum.+ nr 2) no))
      (code-set! c (##fixnum.+ ni 4) rest?)
      (code-set! c (##fixnum.+ ni 5) keys)
      (code-set! c (##fixnum.+ ni 6) #f)
      (code-set! c (##fixnum.+ ni 7) frame)
      c)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-app0-red
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (call-step! (proc)
           (proc)))))))

(define ##cprc-app1-red
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (let ((arg1 (code-run (^ 1))))
           (call-step! (proc arg1)
             (proc arg1))))))))

(define ##cprc-app2-red
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (let* ((arg1 (code-run (^ 1)))
                (arg2 (code-run (^ 2))))
           (call-step! (proc arg1 arg2)
             (proc arg1 arg2))))))))

(define ##cprc-app3-red
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (let* ((arg1 (code-run (^ 1)))
                (arg2 (code-run (^ 2)))
                (arg3 (code-run (^ 3))))
           (call-step! (proc arg1 arg2 arg3)
             (proc arg1 arg2 arg3))))))))

(define ##cprc-app-red
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (let loop ((i 1) (rev-args '()))
           (if (##fixnum.< i (code-length $code))
             (let ((new-rev-args
                    (##cons (code-run (code-ref $code i)) rev-args)))
               (##check-heap-limit)
               (loop (##fixnum.+ i 1) new-rev-args))
             (call-step! (proc rev-args)
               (let ((args (##reverse rev-args)))
                 (##first-argument $code rte)
                 (##apply proc args))))))))))

(define ##cprc-app0-sub
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (##subproblem-apply0 $code rte proc))))))

(define ##cprc-app1-sub
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (let ((arg1 (code-run (^ 1))))
           (##subproblem-apply1 $code rte proc arg1)))))))

(define ##cprc-app2-sub
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (let* ((arg1 (code-run (^ 1)))
                (arg2 (code-run (^ 2))))
           (##subproblem-apply2 $code rte proc arg1 arg2)))))))

(define ##cprc-app3-sub
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (let* ((arg1 (code-run (^ 1)))
                (arg2 (code-run (^ 2)))
                (arg3 (code-run (^ 3))))
           (##subproblem-apply3 $code rte proc arg1 arg2 arg3)))))))

(define ##cprc-app-sub
  (mk-cprc
   (let ((proc (code-run (^ 0))))
     (force-vars (proc)
       (if (##not (##procedure? proc))
         (rt-error-non-procedure-oper $code rte)
         (let loop ((i 1) (rev-args '()))
           (if (##fixnum.< i (code-length $code))
             (let ((new-rev-args
                    (##cons (code-run (code-ref $code i)) rev-args)))
               (##check-heap-limit)
               (loop (##fixnum.+ i 1) new-rev-args))
             (let ((args (##reverse rev-args)))
               (##first-argument $code rte)
               (##subproblem-apply $code rte proc args)))))))))

(define ##proper-tail-calls? #f)
(set! ##proper-tail-calls? #t)

(define ##gen-app-aux
  (mk-gen (stepper oper args)
    (if (and tail? ##proper-tail-calls?)
      (case (##length args)
        ((0)
         (mk-code ##cprc-app0-red
                  cte
                  src
                  stepper
                  (oper)))
        ((1)
         (mk-code ##cprc-app1-red
                  cte
                  src
                  stepper
                  (oper (##car args))))
        ((2)
         (mk-code ##cprc-app2-red
                  cte
                  src
                  stepper
                  (oper (##car args) (##cadr args))))
        ((3)
         (mk-code ##cprc-app3-red
                  cte
                  src
                  stepper
                  (oper (##car args) (##cadr args) (##caddr args))))
        (else
         (##mk-code* ##cprc-app-red
                     cte
                     src
                     stepper
                     (##cons oper args)
                     0)))
      (case (##length args)
        ((0)
         (mk-code ##cprc-app0-sub
                  cte
                  src
                  stepper
                  (oper)))
        ((1)
         (mk-code ##cprc-app1-sub
                  cte
                  src
                  stepper
                  (oper (##car args))))
        ((2)
         (mk-code ##cprc-app2-sub
                  cte
                  src
                  stepper
                  (oper (##car args) (##cadr args))))
        ((3)
         (mk-code ##cprc-app3-sub
                  cte
                  src
                  stepper
                  (oper (##car args) (##cadr args) (##caddr args))))
        (else
         (##mk-code* ##cprc-app-sub
                     cte
                     src
                     stepper
                     (##cons oper args)
                     0))))))

(define ##gen-app
  (mk-gen (oper args)
    (let ((stepper (##current-stepper)))
      (gen ##gen-app-aux src stepper oper args))))

(define ##gen-app-no-step
  (mk-gen (oper args)
    (let ((stepper (##no-stepper)))
      (gen ##gen-app-aux src stepper oper args))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##subproblem-apply0
  (let ()
    (##declare (not inline) (not interrupts-enabled) (environment-map))
    (lambda ($code rte proc)
      (##first-argument
       (call-step! (proc)
         (proc))
       $code
       rte))))

(define ##subproblem-apply1
  (let ()
    (##declare (not inline) (not interrupts-enabled) (environment-map))
    (lambda ($code rte proc arg1)
      (##first-argument
       (call-step! (proc arg1)
         (proc arg1))
       $code
       rte))))

(define ##subproblem-apply2
  (let ()
    (##declare (not inline) (not interrupts-enabled) (environment-map))
    (lambda ($code rte proc arg1 arg2)
      (##first-argument
       (call-step! (proc arg1 arg2)
         (proc arg1 arg2))
       $code
       rte))))

(define ##subproblem-apply3
  (let ()
    (##declare (not inline) (not interrupts-enabled) (environment-map))
    (lambda ($code rte proc arg1 arg2 arg3)
      (##first-argument
       (call-step! (proc arg1 arg2 arg3)
         (proc arg1 arg2 arg3))
       $code
       rte))))

(define ##subproblem-apply
  (let ()
    (##declare (not inline) (not interrupts-enabled) (environment-map))
    (lambda ($code rte proc args)
      (##first-argument
       (call-step! (proc args)
         (##apply proc args))
       $code
       rte))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-delay
  (mk-cprc
   (delay-step! ()
     (let ((promise (delay (code-run (^ 0)))))
       (##check-heap-limit)
       (##first-argument promise $code rte)))))

(define ##gen-delay
  (mk-gen (val)
    (let ((stepper (##current-stepper)))
      (mk-code ##cprc-delay cte src stepper (val)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##cprc-future
  (mk-cprc
   (future-step! ()
     (let ((promise (future (code-run (^ 0)))))
       (##first-argument promise $code rte)))))

(define ##gen-future
  (mk-gen (val)
    (let ((stepper (##current-stepper)))
      (mk-code ##cprc-future cte src stepper (val)))))

;==============================================================================

; Eval

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Evaluation in a top environment with the current dynamic environment.

(define ##eval-top #f)
(set! ##eval-top
  (lambda (src top-cte)
    (##eval-top-with-dyn-env
     src
     top-cte
     (##dynamic-env-ref))))

; Evaluation in a top environment with a specific dynamic environment.

(define ##eval-top-with-dyn-env #f)
(set! ##eval-top-with-dyn-env
  (lambda (src top-cte dyn-env)
    (let ((c (##compile-top top-cte (##sourcify src (##make-source #f #f)))))
      (##dynamic-env-bind
        dyn-env
        (lambda () (let ((rte #f)) (code-run c)))))))

; Evaluation inside a specific environment ('cte' describes the runtime
; environment 'rte').

(define ##eval-inner #f)
(set! ##eval-inner
  (lambda (src cte rte dyn-env)
    (let ((c (##compile-inner cte (##sourcify src (##make-source #f #f)))))
      (##dynamic-env-bind
        dyn-env
        (lambda () (let ((rte rte)) (code-run c)))))))

(define (eval expr #!optional env)
  (##eval-top (##sourcify expr (##make-source #f #f))
              ##interaction-cte))

;==============================================================================

(define (##wrap-datum re x)
  (##make-source x
                 (##make-locat (##port-name (##readenv-port re))
                               (##readenv-filepos re))))

(define (##unwrap-datum re x)
  (##source-code x))

(define (##read-expr-from-port port rt)
  (let ((re
         (##make-readenv port
                         rt
                         ##read-error
                         ##wrap-datum
                         ##unwrap-datum)))
    (##read-datum-or-eof re)))

(define (##read-all-as-a-begin-expr-from-port port rt close-port?)

  (define (read-exprs re)
    (let ((expr (##read-datum-or-eof re)))
      (if (##not (##eof-object? expr))
        (##cons expr (read-exprs re))
        '())))

  (##cleanup-on-signal
   (lambda ()
     (let* ((re
             (##make-readenv port
                             rt
                             ##read-error
                             ##wrap-datum
                             ##unwrap-datum))
            (head
             (##cons (##wrap-datum re 'begin)
                     '())) ; empty list will be replaced with expressions read
            (expr
             (##wrap-datum re head))
            (first
             (##read-datum-or-eof re))
            (script?
             (and (##not (##eof-object? first))
                  (##eq? (##unwrap-datum re first) '#!)
                  (let loop ()
                    (let ((c (##read-char port)))
                      (if (and (##char? c) (##not (##char=? c #\newline)))
                        (loop)
                        #t)))))
            (rest
             (if (##eof-object? first)
               '()
               (read-exprs re))))
       (if close-port?
         (##close-port port))
       (cond ((##eof-object? first)
              (##vector #f expr))
             (script?
              (##set-cdr! head rest)
              (##vector #t expr))
             (else
              (##set-cdr! head (##cons first rest))
              (##vector #f expr)))))
   (lambda ()
     (if close-port?
       (##close-port port)))))

(define (##load-source-from-port close-port? type-callback port)
  (let* ((x
          (##read-all-as-a-begin-expr-from-port
           port
           ##main-readtable
           close-port?))
         (script?
          (##vector-ref x 0))
         (expr
          (##vector-ref x 1)))
    (type-callback script?)
    (##eval-top expr
                (##top-cte-clone ##interaction-cte))))

(define (##load name type-callback trap-if-not-found? char-encoding)

  (define (load-source name name-with-ext)
    (let ((port (##open-input-file name-with-ext)))
      (if port
        (begin
          (##load-source-from-port #t type-callback port)
          (##port-name port))
        (if trap-if-not-found?
          (trap-open-file (load name))
          #f))))

  (define (load-binary name path)
    (let ((msg (##load-object-file path)))
      (if (##procedure? msg)
        (begin
          (msg)
          path)
        (trap-load (load name) msg))))
 
  (define (load-no-ext name)
    (let ((port (##open-input-file name)))
      (if port
        (begin
          (##load-source-from-port #t type-callback port)
          (##port-name port))
        (let loop ((version 1) (last #f))
          (let ((obj-port
                 (##open-input-file
                   (##string-append name
                                    ".o"
                                    (##number->string version 10)))))
            (if obj-port
              (begin
                (##close-port obj-port)
                (loop (##fixnum.+ version 1) (##port-name obj-port)))
              (if last
                (load-binary name last)
                (load-source name (##string-append name ".scm")))))))))

  (define (binary-extension? ext)
    (let ((len (##string-length ext)))
      (and (##fixnum.< 2 len)
           (##char=? (##string-ref ext 0) #\.)
           (##char=? (##string-ref ext 1) #\o)
           (let ((c (##string-ref ext 2)))
             (and (##char>=? c #\1) (##char<=? c #\9)
                  (let loop ((i (##fixnum.- len 1)))
                    (if (##fixnum.< i 3)
                      #t
                      (let ((c (##string-ref ext i)))
                        (and (##char>=? c #\0) (##char<=? c #\9)
                             (loop (##fixnum.- i 1)))))))))))

  (let ((ext (##path-extension name)))
    (cond ((##string=? ext "")
           (load-no-ext name))
          ((binary-extension? ext)
           (let ((path (##path-expand name 'absolute)))
             (if path
               (load-binary name path)
               (if trap-if-not-found?
                 (trap-open-file (load name))
                 #f))))
          (else
           (load-source name name)))))

(define (load path #!optional (char-encoding (absent-obj)))
  (force-vars (path char-encoding)
    (check-string path (load path char-encoding)
      (##load
       path
       (lambda (script?) #f)
       #t
       (if (##eq? char-encoding (absent-obj)) #f char-encoding)))))

;==============================================================================
