package biss.jde;

import biss.DirEnumerator;
import biss.DirFilter;
import biss.FileExtFilter;
import biss.FileLib;
import biss.ObserverSocket;
import biss.Set;
import biss.VectorLib;
import java.io.File;
import java.io.FilenameFilter;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Observer;
import java.util.Vector;

/**
 * singleton class which is responsible for cleaning up expanded
 * CompileUnits which are no longer required to be expanded
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
class CUShrinker
  extends Thread
{
	SourceLibrary Lib;
	long CheckInterval;
	long KeepInterval;

CUShrinker ( SourceLibrary lib ) {
	Lib = lib;
	CheckInterval = Jde.CacheCheck * 1000;
	KeepInterval  = Jde.CacheKeep * 1000;

	setPriority( MIN_PRIORITY);
	setDaemon( true);
	start();
}

public void run () {
	while ( true ) {
		try {
			sleep( CheckInterval);
			if ( Lib.CUs == null || Lib.CUs.size() == 0 ) continue;

			long timeNow = System.currentTimeMillis();
			for ( Enumeration e=Lib.CUs.elements(); e.hasMoreElements(); ) {
				CompileUnit cu = (CompileUnit) e.nextElement();
				if ( cu.IsExpanded && cu.Released > 0 ) {
					long delta = timeNow - cu.Released;
					if ( delta > KeepInterval ){
						cu.shrink();
					}
				}
			}
		}
		catch ( InterruptedException x ) {}
	}
}
}

class SourceLibScanner
  implements Runnable
{
	SourceLibrary Lib;

SourceLibScanner ( SourceLibrary lib ) {
	Lib = lib;
}

public void run () {
	Lib.scan();
}
}

/**
 * master (access) model to manage a set of CompileUnits (Java sources)
 * organized in directories and/or packages
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class SourceLibrary
{
	Hashtable CUs;
	Vector DirList;
	Set Excludes;
	Vector SrcRoots;
	int ScanState = SS_UNDEF;
	CUShrinker Shrinker = new CUShrinker( this);
	public ObserverSocket OsScanStart = new ObserverSocket( this);
	public ObserverSocket OsScanDirList = new ObserverSocket( this);
	public ObserverSocket OsScanDir = new ObserverSocket( this);
	public ObserverSocket OsScanFile = new ObserverSocket( this);
	public ObserverSocket OsScanCU = new ObserverSocket( this);
	public ObserverSocket OsScanReady = new ObserverSocket( this);
	public ObserverSocket OsChanged = new ObserverSocket( this);
	final static int SS_UNDEF = 0;
	final static int SS_SCANNING = 1;
	final static int SS_INITIALIZED = 2;
	public static SourceLibrary DefaultLib;

public void addAllCUsObserver ( Observer o ) {
	for ( Enumeration e=CUs.elements(); e.hasMoreElements(); ) {
		CompileUnit cu = (CompileUnit) e.nextElement();
		cu.addObserver( o);
	}
}

void addCompileUnit ( CompileUnit cu, Observer originator ) {
	CUs.put( cu.FileName, cu);
	OsChanged.notifyObservers( originator);
}

public void addObserver ( Observer o ) {
	OsScanStart.addObserver( o);
	OsScanDirList.addObserver( o);
	OsScanDir.addObserver( o);
	OsScanFile.addObserver( o);
	OsScanCU.addObserver( o);
	OsScanReady.addObserver( o);
	OsChanged.addObserver( o);
}

void buildTypeTree () {
}

public void deleteAllCUsObserver ( Observer o ) {
	for ( Enumeration e=CUs.elements(); e.hasMoreElements(); ) {
		CompileUnit cu = (CompileUnit) e.nextElement();
		cu.deleteObserver( o);
	}
}

void deleteCompileUnits ( Vector list, Observer originator ) {
	CompileUnit cu;

	for ( int i=list.size()-1; i>= 0; i-- ){
		cu = (CompileUnit)list.elementAt(i);
		cu.backup();
		CUs.remove( cu.FileName);
	}

	OsChanged.notifyObservers( originator);
}

public void deleteObserver ( Observer o ) {
	OsScanStart.deleteObserver( o);
	OsScanDirList.deleteObserver( o);
	OsScanDir.deleteObserver( o);
	OsScanFile.deleteObserver( o);
	OsScanCU.deleteObserver( o);
	OsScanReady.deleteObserver( o);
	OsChanged.deleteObserver( o);
}

static Set dirNameSet ( Vector dirList ) {
	int n = dirList.size();
	Set set = new Set( n);

	for ( int i =0; i<n; i++ )
		set.add( ((File)dirList.elementAt(i)).getAbsolutePath() );

	return set;
}

public static String expandHome ( String s ) {
	if ( s.charAt(0) == '~' )
		return Jde.HomeDir + s.substring( 1, s.length());
	else
		return s;
}

public void finalize () {
	Shrinker.stop();
}

public int getCUNumberOfDir ( File dir ) {
	int n;
	String dName = dir.getAbsolutePath();
	Enumeration e;

	for ( n=0, e=CUs.elements(); e.hasMoreElements();  ) {
		CompileUnit cu = (CompileUnit) e.nextElement();
		if ( cu.DirName.equals( dName) )
			n++;
	}

	return n;
}

public Vector getCUsOfAllDirs () {
	return getCUsOfDirs( null);
}

public synchronized Vector getCUsOfDirs ( Vector dirList ) {
	Vector list;
	CompileUnitSorter sorter = new CompileUnitSorter( false);

	if ( dirList == null ) {
		list = new Vector( CUs.size());
		VectorLib.sortInAll( list, CUs.elements(), sorter);
	}
	else {
		Set set = new Set( dirList.size());
		Enumeration e;

		for ( e=dirList.elements(); e.hasMoreElements(); )
			set.add( ((File)e.nextElement()).getAbsolutePath());

		list = new Vector( dirList.size() * 40);
		for ( e=CUs.elements(); e.hasMoreElements(); ){
			CompileUnit cu = (CompileUnit) e.nextElement();
			if ( set.contains( cu.DirName) )
				VectorLib.sortIn( list, cu, sorter);
		}
	}

	return list;
}

public Vector getChildTreeOfType ( TypeDecl td ) {
	Vector list = new Vector( 20);

	return list;
}

public static SourceLibrary getDefaultLib () {
	if ( DefaultLib == null )
		DefaultLib = new SourceLibrary();
	return DefaultLib;
}

public Vector getDirList () {
	return DirList;
}

public synchronized Vector getPackageList () {
	Set set = new Set(20);

	set.add( "<none>");
	for ( Enumeration e=CUs.elements(); e.hasMoreElements(); ){
		String s = ((CompileUnit)e.nextElement()).Package;
		if ( s != null )
			set.add( s);
	}

	return set.toSortedVector();
}

public Vector getParentListOfType ( TypeDecl td ) {
	Vector list = new Vector( 6);

	return list;
}

public String getQualifiedTypeId ( CompileUnit cu, String typeId ) {
	Enumeration e1, e2;
	CompileUnit u;

	if ( typeId.indexOf( '.') >= 0 )    // already qualified
		return typeId;

	/*
	  if ( cu.Package != null ) {         // check for package type
	  Vector v = (Vector) Packages.get( cu.Package);
	  for ( e1=v.elements(); e1.hasMoreElements(); ) {
	  u = (CompileUnit) e1.nextElement();
	  if ( u.typeWithName( typeId) != null )
	  return u.Package + '.' + typeId;
	  }
	  }
	 */
	String tid = "." + typeId;

	for ( e1 = cu.Imports.elements(); e1.hasMoreElements(); ) { // check imports
		Import imp = (Import) e1.nextElement();
		if ( imp.Spec.endsWith( tid) )
			return imp.Spec;
	}

	return typeId;
}

public int getTypeNumberOfPackage ( String pckg ) {
	int n;
	Enumeration e;

	for ( n=0, e=CUs.elements(); e.hasMoreElements(); ) {
		CompileUnit cu = (CompileUnit) e.nextElement();
		if ( cu.Package == null ) {
			if ( pckg.equals( "<none>") )
				n += cu.TypeDecls.size();
		}
		else {
			if ( cu.Package.equals( pckg) )
				n += cu.TypeDecls.size();
		} 
	}

	return n;
}

public Vector getTypesOfAllPackages () {
	return getTypesOfPackages( null);
}

public synchronized Vector getTypesOfPackages ( Vector pckgList ) {
	Vector list = new Vector(50);
	TypeDeclSorter sorter = new TypeDeclSorter( false, true);
	Set pckgSet = (pckgList == null) ? null : new Set( pckgList);

	for ( Enumeration e=CUs.elements(); e.hasMoreElements(); ) {
		CompileUnit cu = (CompileUnit) e.nextElement();
		String s = (cu.Package != null) ? cu.Package : "<none>";
		if ( pckgSet == null || pckgSet.contains( s) )
			VectorLib.sortInAll( list, cu.TypeDecls, sorter);
	}

	return list;
}

public boolean isInitialized () {
	return ScanState == SS_INITIALIZED;
}

public boolean isScanning () {
	return ScanState == SS_SCANNING;
}

public void scan () {
	ScanState = SS_SCANNING;
	OsScanStart.notifyObservers();

	setExcludes();
	setSrcRoots();
	setDirList();

	setCUs();

	ScanState = SS_INITIALIZED;
	OsScanReady.notifyObservers();
}

public void scanAsync () {
	Thread t = new Thread( new SourceLibScanner( this) );
	t.setPriority( Thread.MIN_PRIORITY);
	t.start();
}

public void setCUs () {
	String srcs[];
	FileExtFilter filt = new FileExtFilter( ".java");
	byte[] buf = new byte[50000];
	int i, j, l;

	CUs = new Hashtable( Jde.NCUs);

	for ( i=0; i<DirList.size(); i++) {
		File dir = (File) DirList.elementAt(i);
		OsScanDir.notifyObservers( dir);

		srcs = dir.list( filt);
		for ( j=0; j<srcs.length; j++ ) {
			File f = new File( dir, srcs[j]);
			try {
				String fn = f.getAbsolutePath();
				OsScanFile.notifyObservers( f);

				if ( (l = FileLib.copyFileData( f, buf)) < 0 ){
					buf = new byte[(int)f.length()+1];
					l = FileLib.copyFileData( f, buf);
				} 
				if ( l > 0 ) {
					buf[l] = JavaScanner.EOT;
					LibParser p = new LibParser( fn, buf);
					CompileUnit cu = p.parseCompileUnit();
					CUs.put( cu.FileName, cu);
					OsScanCU.notifyObservers( cu);
				}
			}
			catch ( ParseException x ) {
				System.out.println( "!!! " + f.getAbsolutePath() + " did not parse");
				x.printStackTrace();
			}
		}
	}
}

void setDirList () {
	SrcDirFilter filter = new SrcDirFilter( Excludes);

	DirList = null;
	Vector dirList = new Vector( Jde.NDirs);
	for ( Enumeration e=SrcRoots.elements(); e.hasMoreElements(); ){
		String dn = (String) e.nextElement();
		DirEnumerator de = new DirEnumerator( dn, filter);
		VectorLib.addAllElements( dirList, de.getDirList());
	}

	DirList = dirList;
	OsScanDirList.notifyObservers();
}

void setExcludes () {
	Excludes = new Set( 5);
	if ( Jde.Excludes != null ) {
		for ( int i=0; i< Jde.Excludes.length; i++ )
			Excludes.add( Jde.Excludes[i]);
	}
}

void setSrcRoots () {
	SrcRoots = new Vector( 5);
	if ( Jde.SrcRoots != null ) {
		for ( int i=0; i < Jde.SrcRoots.length; i++ ) {
			if ( !Excludes.contains( Jde.SrcRoots[i]) )
				SrcRoots.addElement( Jde.SrcRoots[i]);
		}
	}
}
}

class SrcDirFilter
  implements FilenameFilter
{
	Set Excludes;

public SrcDirFilter ( Set excludes ) {
	Excludes = excludes;
}

public boolean accept ( File dir, String name ) {
	File p = new File( dir, name);
	return (p.isDirectory() && !Excludes.contains( p.getAbsolutePath()));
}
}
