package biss.awt;

import biss.ObserverSocket;
import biss.awt.ClipBoard;
import biss.awt.Control;
import biss.awt.LineBuffer;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.util.Observable;

/**
 * replaces java.awt.TextField
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class TextField
  extends Control
  implements HotKeyProcessor, TextFieldPeer
{
	LineBuffer Contents = new LineBuffer( null);
	int CursorIdx;
	int Mode = 0;
/**
 * triggered by Enter key ( validate contents)
 * parameter: -
 */
	public ObserverSocket OsEnter = new ObserverSocket( this);
/**
 * triggered by ESC key ( clear contents)
 * parameter: -
 */
	public ObserverSocket OsReset = new ObserverSocket( this);
	boolean ReadOnly = false;
	int SelEnd;
	int SelStart;
	int XBorder = 3;
	int XOffs;
	int YOffs;
	boolean RightAlign = false;
	EntryValidator Validator = null;
	EntryKeyFilter KeyFilter = null;
	public static int DefaultBorderStyle = Border.BS_STEP;
	TextCursor Cursor;
	boolean Modified = false;
	static FontMetrics DefFm;
	boolean Validated = false;

static {
	DefFm = Awt.getFontMetrics( Awt.SysFont);
}

public TextField(){
	this( null, null);
}

public TextField( Font fnt) {
	this( null, fnt);
}

public TextField( String defCont){
	this( defCont, null);
}

public TextField( String defCont, Font fnt) {
	super();
	CursorIdx = 0;
	SelStart  = 0;
	SelEnd    = 0;

	Cursor = new TextCursor( Awt.EntryBackClr);
	setBackground( Awt.EntryBackClr);
	setForeground( Awt.EntryForeClr);

	setFont( fnt);
	FBorder.setBorder( DefaultBorderStyle);

	setContents( defCont);
	initEventHandler();
}

public TextField( int xBorder, Font fnt, Color cf, Color cb) {
	this( null, fnt);
	setForeground( cf);
	setBackground( cb);
	XBorder   = xBorder;
}

public TextField( java.awt.TextField peerTarget){
	// create peer object for java.awt.TextField
	this( peerTarget.getText() );
	setPeerTarget( peerTarget);
	addNotify();
}

void adjustContents() {

	if ( Height == 0)    //not yet initialized
		return;

	YOffs = (Height - Contents.CharHeight) / 2; // always vertically centered
	Cursor.setYPos( Height - YOffs);
	Cursor.setHeight( Contents.CharHeight);

	if ( ! RightAlign)
		XOffs = XBorder + FBorder.Ext;
	else
		XOffs = Width - 2*FBorder.Ext - Contents.charsWidth() - XBorder;
}

void adjustToCursor( boolean update) {

	int tw;

	if ( RightAlign) {
		tw = Contents.charsWidth();
		XOffs = Width - 2*FBorder.Ext - tw - XBorder;
		if ( update)
			redraw();
		return;
	}

	tw = Contents.charsWidth( 0, CursorIdx, 0);
	if ( tw + XOffs  >= Width - FBorder.Ext ){
		XOffs -= (tw - Width + 20 + XOffs - XBorder);
		if ( update)
			redraw();
	}	
	else if ( tw + XOffs < FBorder.Ext){
		XOffs -= (tw + XOffs - XBorder - FBorder.Ext);
		if ( update)
			redraw();
	}	
}

void backSpace() {
	if ( ReadOnly)
		return;

	if ( SelStart != SelEnd){
		Modified = true;
		deleteSelection();
	}
	else if ( CursorIdx > 0){
		CursorIdx--;
		Contents.remove( CursorIdx, CursorIdx + 1);
		updateSel();
		Modified = true;
		redraw();
	}
}

void  blankCursor( Graphics g) {

	if ( YOffs == 0)
		adjustContents();

	Cursor.clearAtOffs( g, XOffs, 0);

	if ( Contents.Len > CursorIdx){
		g.setColor( getForeground());
		g.drawChars( Contents.Text, CursorIdx, 1, XOffs + Cursor.XPos,
		             Height - YOffs - Contents.MaxDescent );
	}
}

public void charInput( char c, int mod) {

	if ( ReadOnly)
		return;

	if ( (KeyFilter != null) && ( ! KeyFilter.isValidKey( CursorIdx, c)) )
		return;

	if ( SelStart != SelEnd){
		replaceSelectionWith( c);
		return;
	}

	Contents.insert( c, CursorIdx);
	CursorIdx++;
	updateSel();

	Modified = true;
	redraw();
}

void cursorLeft( boolean extentSel) {

	if ( CursorIdx <= 0)
		return;

	Graphics g = getGraphics();
	if ( g == null )
		return;
	
	clip( g);

	blankCursor( g);
	CursorIdx--;
	adjustToCursor( true);
	if ( Mode == 1){
		updateSel();
		redraw( g);
	}	
	else if ( ! extentSel){
		if ( hasSelection() ){
			updateSel();
			redraw( g);
		}
		else {
			updateSel();
			drawCursor( g);
		}    
	}    
	else {
		SelEnd--;
		redraw( g);
	}

	g.dispose();
}

void cursorRight( boolean extentSel) {

	if ( CursorIdx >= Contents.Len)
		return;

	Graphics g = getGraphics();
	if ( g == null )
		return;
	
	clip( g);

	blankCursor( g);
	CursorIdx++;
	adjustToCursor(true);
	if ( Mode == 1){
		updateSel();
		redraw(g);
	}	
	else if ( ! extentSel){
		if ( hasSelection() ){
			updateSel();
			redraw( g);
		}
		else {
			updateSel();
			drawCursor( g);
		}    
	}    
	else {
		SelEnd++;
		redraw( g);
	}

	g.dispose();
}

void del() {
	if ( ReadOnly)
		return;

	if ( SelStart != SelEnd) {
		Modified = true;
		deleteSelection();
	}
	else if ( CursorIdx < Contents.Len) {        	
		updateSel();
		Contents.remove( CursorIdx, CursorIdx + 1);
		Modified = true;
		redraw();
	}
}

public void deleteSelection() {
	int sb  = Math.min( SelStart, SelEnd);
	int len = Math.abs( SelStart - SelEnd);

	if ( ReadOnly)
		return;

	Contents.remove( sb, sb + len);

	if ( SelEnd > SelStart)
		CursorIdx = Math.max( CursorIdx - len, 0);

	updateSel();
	redraw();
	Modified = true;
}

public synchronized void disable(){
	updateSel();
	super.disable();
}

void drawCursor ( Graphics g) {

	if ( YOffs == 0)
		adjustContents();

	Cursor.setXPos( Contents.posFor( CursorIdx));
	Cursor.drawAtOffs( g, XOffs, 0);
}

public String getContents() {
	// return copy of contents
	return Contents.toString();
}

public String getSelection() {
	int   sb    = Math.min( SelStart, SelEnd);
	int   len   = Math.abs( SelStart - SelEnd);
	return new String ( Contents.Text, sb, len);
}

public int getSelectionEnd() {
	//java.awt.peer.TextFieldPeer interface
	return Math.max( SelStart, SelEnd);
}

public int getSelectionStart() {
	//java.awt.peer.TextFieldPeer interface
	return Math.min( SelStart, SelEnd);
}

public String getText() {
	//java.awt.peer.TextFieldPeer interface
	return getContents();
}

public void gotFocus() {
	redraw();
}

public boolean hasSelection() {
	return ( SelStart != SelEnd);
}

void initEventHandler(){
	EventHandler.CursorType = TopWindow.TEXT_CURSOR;

	EventHandler.addHotKey( 'i', HotKey.Control, this, "imode" );
	EventHandler.addHotKey( 'o', HotKey.Control, this, "omode" );
	EventHandler.addHotKey( 'c', HotKey.Control, this, "copy" );
	EventHandler.addHotKey( 'x', HotKey.Control, this, "cut" );
	EventHandler.addHotKey( 'v', HotKey.Control, this, "paste" );
}

public boolean isModified() {
	return Modified;
}

public void lostFocus() {
	if ( hasSelection() )
		updateSel();
	redraw();
}

public Dimension minimumSize( int cols) {
	//java.awt.peer.TextFieldPeer interface
	return new Dimension( cols*Awt.SysFontWidth, Awt.SysFontHeight);
}

public void mouse1Down( Event evt) {
	setCursorPos( evt.x, true);
	SelStart = SelEnd = CursorIdx;
}

public void mouse1Drag( Event evt) {
	int cp = CursorIdx;
	setCursorPos( evt.x, false);
	SelEnd = CursorIdx;
	if ( cp != CursorIdx)
		redraw();
}

public void mouse1Up( Event evt){
	redraw();
}

protected void posChanged() {
	adjustContents();
}

public Dimension preferredSize() {
	return new Dimension( 6*Awt.SysFontWidth, 3*Awt.SysFontHeight/2);
}

public Dimension preferredSize( int cols) {
	//java.awt.peer.TextFieldPeer interface
	return new Dimension( cols*Awt.SysFontWidth, 3*Awt.SysFontHeight/2);
}

public void processKey( Object sel) {

	if ( sel.equals( "imode") ) {
		if ( Mode != 0){
			Mode = 0;
			updateSel();
			redraw();
		}	
	}
	else if ( sel.equals( "omode") ) {
		if ( Mode != 1) {
			Mode = 1;
			updateSel();
			redraw();
		}
	}
	else if ( sel.equals( "cut") ) {
		ClipBoard.set( getSelection() );
		deleteSelection();
	}
	else if ( sel.equals( "copy") ) {
		ClipBoard.set( getSelection() );
	}
	else if ( sel.equals( "paste") ) {
		replaceSelectionWith( ClipBoard.get() );
	}

}

protected void redraw( Graphics g){

	if ( YOffs == 0)
		adjustContents();

	adjustToCursor( false);

	if ( Contents.Len == 0) {
		blank( g);
		clip( g);
		if ( EventHandler.HasFocus)
			drawCursor( g);
		return;
	}	

	int bl = Height - YOffs - Contents.MaxDescent;

	blank( g);
	clip( g);

	if ( Validated && !Modified){
		g.setColor( Color.green);
		g.fillRect( Width - FBorder.Ext - 4, FBorder.Ext, 4, 4);
	}

	if ( SelStart == SelEnd) {
		g.setColor( EventHandler.IsDisabled ? Awt.InactiveClr : getForeground());
		Contents.draw( 0, Contents.Len, g, XOffs , bl, XOffs);
	}
	else {
		int offs = XOffs;
		int dx   = 0;
		int sb   = Math.min( SelStart, SelEnd);
		int se   = Math.max( SelStart, SelEnd);
		Color cf = getForeground();
		Color cb = getBackground();

		if ( sb > 0) {
			g.setColor( cb);
			dx = Contents.charsWidth( 0, sb, 0);
			g.fillRect( FBorder.Ext, FBorder.Ext, offs + dx,
			            Height - 2 * FBorder.Ext);
			g.setColor( cf);
			Contents.draw( 0, sb, g, offs , bl, XOffs);
			offs += dx;
		}

		g.setColor( Awt.SelBackClr);
		dx = Contents.charsWidth( sb, se - sb, 0);
		g.fillRect( offs,
		            Height - YOffs - Contents.CharHeight,
		            dx,
		            Contents.CharHeight + 1);
		g.setColor( Awt.SelForeClr);
		Contents.draw( sb, se - sb, g, offs , bl, XOffs);
		offs += dx;

		if ( se < Contents.Len) {
			g.setColor( cb );
			g.fillRect( offs, FBorder.Ext,
			            Width - offs,
			            Height - 2 * FBorder.Ext);
			g.setColor( cf );
			Contents.draw( se, Contents.Len - se, g, offs , bl, XOffs);
		}
	}

	if ( EventHandler.HasFocus)
		drawCursor( g);
}

public boolean replaceSelectionWith( String s) {
	int  sb   = Math.min( SelStart, SelEnd);
	int  len  = Math.abs( SelStart - SelEnd);

	if ( ReadOnly)
		return false;

	Contents.replace( sb, sb + len, s);
	if ( s != null)
		CursorIdx = sb + s.length();
	updateSel();

	redraw();
	Modified = true;
	return true;
}

public boolean replaceSelectionWith( char c) {
	return replaceSelectionWith( String.valueOf( c));
}

public void select( int selStart, int selEnd) {
	//java.awt.peer.TextFieldPeer interface
	SelStart = Math.min( selStart, Contents.Len);
	SelEnd   = Math.min( selEnd, Contents.Len);
	setCursorPos( 0, true);
}

public void selectContents() {
	SelStart = 0;
	SelEnd   = Contents.Len;
	setCursorPos( 0, true);
}

public void setChecker( Object chk) {
	if ( chk instanceof EntryKeyFilter)
		KeyFilter = (EntryKeyFilter) chk;
	if ( chk instanceof EntryValidator)
		Validator = (EntryValidator) chk;
}

public void setContents( String s) {
	Contents.free();
	Contents.append( s);
	SelStart = SelEnd = CursorIdx = 0;  
	XOffs = XBorder + FBorder.Ext;

	Modified = false;
	Validated = false;

	setCursorPos( 0, true);
}

boolean setCursorPos( int x, boolean upd) {
	int xo = XOffs;

	for( int idx = 0; idx < Contents.Len; idx++) {
		if ( x <= xo){
			CursorIdx = idx;
			if ( upd && isShowing() )
				redraw();
			return true;
		}
		xo += Contents.charsWidth( idx, 1, 0);
		//    System.out.println( Contents.Text[idx] + " " + w);
		//    xo += w;
	}

	CursorIdx = Contents.Len;

	if ( upd && isShowing() )
		redraw();

	return false;

}

public static void setDefaultBorderStyle( int style) {
	DefaultBorderStyle = style;
}

public void setEchoCharacter( char c) {
	//java.awt.peer.TextFieldPeer interface
}

public void setEditable( boolean editable){
	//java.awt.peer.TextFieldPeer interface
	enable( editable);
}

public synchronized void setFont( Font fnt) {
	Font uf = ( fnt != null) ? fnt : Awt.SysFont;
	super.setFont( uf);

	if ( uf.equals( Awt.SysFont))
		Contents.setMetrics( DefFm);
	else
		Contents.setMetrics( Awt.getFontMetrics( uf));

	adjustContents();
	if ( isShowing() )
		redraw();
}

public void setKeyFilter( EntryKeyFilter f) {
	KeyFilter = f;
}

public void setModified( boolean state) {
	Modified = state;
}

public void setReadOnly( boolean state){
	ReadOnly = state;
}

public void setRightAlign( boolean a) {
	RightAlign = a;
	if ( isShowing() )
		redraw();
}

public void setText( String l){
	//java.awt.peer.TextFieldPeer interface
	setContents( l);
}

public void setValidator( EntryValidator v) {
	Validator = v;
}

void updateSel() {
	SelStart = CursorIdx;
	SelEnd = Math.min( SelStart + Mode, Contents.Len);
}

public boolean vKeyInput( int k, int mod) {
	switch ( k) {
	case Key.Newline:
		Validated = validateContents();
		if ( Validated ){
			redraw();
			OsEnter.notifyObservers();
		}
		peerEvent( Event.ACTION_EVENT, getContents() );
		break;
	case Key.Escape:
		if ( ! ReadOnly) {
			Contents.free();
			CursorIdx = 0;
			updateSel();
			XOffs = XBorder + FBorder.Ext;
			redraw();
			OsReset.notifyObservers();
		}    
		break;
	case Key.Backspace:
		backSpace();
		break;                    
	case Key.Delete:
		del();
		break;
	case Key.CursorLeft:
		cursorLeft( (mod & Event.SHIFT_MASK) != 0 );
		break;
	case Key.CursorRight:
		cursorRight( (mod & Event.SHIFT_MASK) != 0 );
		break;
	case Key.Home:
		if ( (mod & Event.SHIFT_MASK) != 0) {
			SelStart = CursorIdx;
			SelEnd   = 0;
		}
		XOffs = XBorder + FBorder.Ext;
		setCursorPos( 0, true);
		break;
	case Key.End:
		if ( (mod & Event.SHIFT_MASK) != 0) {
			SelStart = CursorIdx;
			SelEnd   = Contents.Len;
		}
		setCursorPos( 10000, true);
		break;
	default:
		return false;
	}

	return true;
}

public boolean validateContents () {
	if ( Validator == null){
		Modified = false;
		return true;
	}

	String cont = getContents();
	String val = Validator.validateContents( cont);

	if ( val == cont){
		Modified = false;
		return true;
	}

	setContents( val);
	selectContents();

	return false;
}
}
