package biss.awt;

import biss.Command;
import biss.CommandQueue;
import biss.ObserverSocket;
import java.awt.Color;
import java.awt.Event;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Rectangle;

/**
 * scrollbar base ( HScrollBar / VScrollBar)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
abstract public class ScrollBar
  extends Control
  implements Command, ScrollbarPeer
{
	int DragOffs = 0;
	int LineInc = 1;
	int LoopDir = 0;
	int Max = 0;
	int Min = 0;
/**
 * triggered by any slider position change ( via slider or
 * positioning methods)
 * parameter: -
 */
	public ObserverSocket OsScroll = new ObserverSocket( this);
	int Pos = 0;
	int ScrollOp = 0;
	Rectangle SliderRect = new Rectangle();
	int StepPels = 1;
	int PageInc = 1;
	boolean AutoPageInc = true;
	boolean Delay = false;
	int SoftScrollPixDelay = -1;
	boolean IsSoftScroll = false;
	static int SoftScrollInitDelay = 0;

public ScrollBar( int min, int max, int pos) {
	Min = min;
	Max = max;
	Pos = pos;

	setBackground( Awt.ScrollBarClr);
	setBorder( Border.BS_INSET);
}

public boolean canGetFocus () {
	return false;
}

public void end( ) {
	setPosition( Max);
}

public boolean execute() {
	long tLast = 0, dt = 0;

	if ( Awt.SoftScroll && Mouse.is2Active() )
		IsSoftScroll = true;

	for ( int i = 0; true; i++ ) {
		switch( LoopDir) {
		case 1:
			if ( Pos == Min)
				return false;
			scrollDown( LineInc);
			break;
		case 2:
			if ( Pos == Max)
				return false;
			scrollUp( LineInc);
			break;
		}    

		if ( Mouse.isAnyActive() ){
			if ( IsSoftScroll ) {
				if ( SoftScrollPixDelay < 0 ) {
					if ( tLast == 0 ) 
						tLast = System.currentTimeMillis();
					else {
						dt = System.currentTimeMillis() - tLast;
						if ( dt < 100 )      { SoftScrollPixDelay = Awt.SoftScrollDelay; }
						else if ( dt > 400 ) { SoftScrollPixDelay = 0; }
						else {
							dt -= (SoftScrollInitDelay / Awt.RoundsPerMilli);
							SoftScrollPixDelay = Awt.SoftScrollDelay - (int)(dt * Awt.SoftScrollDelay/300);
						}
					}
				}
				Awt.DefToolkit.sync();	
			}
			else {                      // line scroll
				int delay = Awt.SpinDelay;
				try {
					if ( Delay ) {
						Delay = false;
						delay *= 4;
					}
				  Thread.sleep( delay);
				} catch ( Exception x ) {}
			}

			if ( !Mouse.isAnyActive() )
				break;
		}
		else {
			LoopDir = 0;
			break;
		}
	}

	IsSoftScroll = false;
	return true;  
}

public int getLineIncrement() {
	return LineInc;
}

public int getMaximum() {
	return Max;
}

public int getMinimum() {
	return Min;
}

public int getPageIncrement() {
	return PageInc;
}

public int getPosition() {
	return Pos;
}

public int getStepPels() {
	return StepPels;
}

public void home( ) {
	setPosition( Min);
}

public void mouse1Down (Event evt) {
	mouseDownScroll( evt);
}

public void mouse1Up ( Event evt) {
	mouseUp( evt);
}

public void mouse2Down (Event evt) {
	mouseDownScroll( evt);
}

public void mouse2Up ( Event evt) {
	mouseUp( evt);
}

void mouseDownScroll ( Event evt ) {
}

void mouseUp( Event evt) {
	int so = ScrollOp;
	ScrollOp = 0;
	DragOffs = 0;

	switch( so) {
	case 1:
	case 2: redrawSpins();
		break;
	case 3: scrollDown( PageInc);
		break;
	case 4: scrollUp( PageInc);
		break;
	case 5: setCursor( Frame.DEFAULT_CURSOR);
	}

	LoopDir = 0;
}

public void pageDown( ) {
	setPosition( Pos - PageInc);
}

public void pageUp( ) {
	setPosition( Pos + PageInc);
}

protected void posChanged( ) {
	updatePageInc();
	updateSliderRect();

	//after resizing contents may fit completely
	//into visible region, so reset to home position
	if ( PageInc >= Max - Min )
		home();
}

public void redraw( Graphics g) {
	redrawSlider( g);
	redrawSpins( g);
}

protected void redrawSlider () {
	Graphics g = getGraphics();
	if ( g != null ) {
		redrawSlider( g);
		g.dispose();
	}
}

abstract protected void redrawSlider( Graphics g);

protected void redrawSpins () {
	Graphics g = getGraphics();
	if ( g != null ) {
		redrawSpins( g);
		g.dispose();
	}
}

abstract protected void redrawSpins( Graphics g);

public void scrollDown( int steps) {
	setPosition( Pos - steps);
}

public void scrollUp( int steps) {
	setPosition( Pos + steps);
}

public void setLineIncrement( int inc) {
	//java.awt.peer.ScrollbarPeer
	LineInc = inc;
}

public void setMinMax( int min, int max) {
	Min = min;
	Max = max;

	if ( Pos > Max)       Pos = Max;
	else if ( Pos < Min)  Pos = Min;

	updateSlider();
}

public void setPageIncrement( int inc) {
	//java.awt.peer.ScrollbarPeer
	AutoPageInc = false;
	PageInc = inc;
	updateSlider();
}

public boolean setPosition( int sPos) {

	if ( sPos == Pos)
		return false;

	if ( sPos > Max){
		if ( Pos == Max)
			return false;
		sPos = Max;
	}
	else if ( sPos < Min){
		if ( Pos == Min)
			return false;
		sPos = Min;
	}

	Pos = sPos;
	updateSliderPos();
	redrawSlider();
	OsScroll.notifyObservers();

	return true;
}

public void setStepPels( int p) {
	StepPels = p;
	updatePageInc();
}

public void setValue( int value) {
	//java.awt.peer.ScrollbarPeer
	setPosition( value);
}

public void setValues( int pos, int min, int max) {
	Min = min;
	Max = max;

	if ( pos < Min)      Pos = Min;
	else if ( pos > Max) Pos = Max;
	else                 Pos = pos;

	updateSlider();
}

public void setValues( int value, int visible, int minimum, int maximum) {
	//java.awt.peer.ScrollbarPeer
	setValues( value, minimum, maximum);
	setPageIncrement( visible);
}

public String toString() {
	String s = "biss.awt.ScrollBar [Pos: " + Pos + " Min: " + Min +
	" Max: " + Max + " PageInc: " + PageInc + "]";
	return s;
}

abstract protected void updatePageInc();

public void updateSlider () {
	updateSliderRect();
	if ( isShowing() )
		redrawSlider();
}

abstract protected void updateSliderPos();

abstract protected void updateSliderRect();
}
