package biss.awt;

import biss.ObserverSocket;
import biss.VectorLib;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * replaces java.awt.List. Far more general because it can deal with
 * Vectors of arbitrary objects. All processing (drawing, selection,
 * processing, hotkey commands, popup menus etc.) is done via
 * ObserverSockets. In case of Vector items implementing the SelfDrawing
 * interface (or of Strings, of course), nothing has to be done to get
 * the items displayed. The List also does not make a distinction
 * between single or multiple select mode. Instead of this, a selection
 * will be "remembered" (marked) if the <ctrl> key was pressed in combination
 * with the mouse button. <esc> resets all selections, <F12> selects all
 * items.
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class List
  extends ScrollablePane
  implements ListPeer
{
	protected ListCanvas Cv;
	public Color DrawClrBack;
	public Color DrawClrFore;
	public Object DrawObject;
/**
 * triggered by an item paint request, just in case
 * of at least one registered observer.
 * This observer is responsible for all item painting
 * ( overrides the standard string representation)
 * parameter: list item
 */
	public ObserverSocket OsDrawObject = new ObserverSocket( this);
/**
 * triggered by an ACTION event (mouse1DblClick or Enter key).
 * parameter: selection
 */
	public ObserverSocket OsProcess = new ObserverSocket( this);
/**
 * triggered by a SELECT / DESELECT event (mouse1Up). Note that cleaning
 * the selection (e.g. ESC key) triggers the Socket with a null
 * parameter.
 * parameter: selection or null ( selection cleared)
 */
	public ObserverSocket OsSelection = new ObserverSocket( this);
	public boolean DrawingSelected = false;
	public boolean DrawingMarked = false;
/**
 * triggered by a mouse SELECT event( mouse1Up)
 * parameter: mouse selection point
 */
	public ObserverSocket OsMouseSelection = new ObserverSocket( this);

public List(){
	this( 0);
}

public List( String items){
	this(0);

	Vector v = new Vector();

	StringTokenizer tok = new StringTokenizer( items, ",:;");
	while( tok.hasMoreElements() )
		v.addElement( tok.nextToken() );
	setContents( v);
}

public List( int type) {
	super( type);

	Cv = new ListCanvas( Horz, Vert, this);
	setCanvas( Cv);
}

public List( int type, HScrollBar hb) {
	super( type);
	Horz = hb;
}

public List( java.awt.List list){
	this( 0);

	BaseCv.setPeerTarget( list);

	Vert.setPeerTarget( list);
	if ( Horz != null)
		Horz.PeerTarget = list;

	Vector v = new Vector( list.countItems() );
	for ( int i=0; i<list.countItems(); i++)
		v.addElement( list.getItem( i));

	addNotify();
	setContents( v);
}

public void addAllElements ( Vector list ) {
	if ( Cv.Lines == null) {
		Cv.Lines = new Vector();
		Cv.setLineDimensions( list.firstElement());
	}
	VectorLib.addAllElements( Cv.Lines, list);
	Cv.updateContents();
}

public void addElement ( Object obj ) {
	if ( Cv.Lines == null) {
		Cv.Lines = new Vector();
		Cv.setLineDimensions( obj);
	}
	Cv.Lines.addElement( obj);
	Cv.updateVScroll();
	if ( Cv.isShowing() )
		Cv.redrawObject( obj);
}

public void addItem ( String item, int index ) {
	//java.awt.peer.ListPeer interface
	if ( Cv.hasStringList() ){
		Cv.Lines.insertElementAt( item, index);
		Cv.updateContents();
	}
	else if ( Cv.Lines == null)
		addElement( item);
}

public void clear () {
	//java.awt.peer.ListPeer interface
	clearSelections( true);
}

public void clearMarked () {
	Cv.clearMarked();
}

public boolean clearMarked ( Object obj ) {
	return Cv.clearMarked( obj);
}

public void clearSelections ( boolean notify ) {
	Cv.clearMarked();
	setSelection( null, notify);
}

public void delItems ( int start, int end) {
	//java.awt.peer.ListPeer interface
	removeAllElements( VectorLib.elementsFromTo( Cv.Lines, start, end));
}

public void deselect ( int index) {
	//java.awt.peer.ListPeer interface
	Cv.deselect( index);
}

public Object elementAt( int index) {
	if ( Cv.Lines == null)
		return null;
	return Cv.Lines.elementAt( index);
}

public Vector getContents() {
	return Cv.Lines;
}

public Rectangle getDrawRectFor ( Object o ) {
	return Cv.drawRectFor( o);
}

public int getDrawWidth () {
	return Cv.Width;
}

public int[] getSelectedIndexes () {
	//java.awt.peer.ListPeer interface
	Vector v = getSelections();
	int[]  ia = null;
	if ( v != null){
		ia = new int[v.size()];
		for ( int i=0; i<v.size(); i++)
			ia[i] = Cv.Lines.indexOf( v.elementAt( i));
	}
	return ia;
}

public Object getSelection(){
	return Cv.Selection;
}

public Vector getSelections () {
	Vector list = null;

	if ( Cv.Marked != null )
		list = Cv.Marked;
	else {
		if ( Cv.Selection != null ) {
			list = new Vector( 1);
			list.addElement( Cv.Selection);
		}
	}
	return list;
}

public boolean hasSelection () {
	return (Cv.Selection != null);
}

public boolean hasSelections () {
	return ((Cv.Selection != null) || (Cv.Marked != null));
}

public void insertAllElementsBefore ( Vector v, Object nextObj ) {
	if ( Cv.Lines == null)
		Cv.Lines = new Vector();
	VectorLib.insertAllElementsBefore( Cv.Lines, v, nextObj);
	Cv.updateContents();
}

public void insertAllElementsBehind ( Vector v, Object obj ) {
	if ( Cv.Lines == null)
		Cv.Lines = new Vector();
	VectorLib.insertAllElementsBehind( Cv.Lines, v, obj);
	Cv.updateContents();
}

public void insertElementBefore ( Object insObj, Object prevObj ) {
	int insIdx;
	if ( Cv.Lines == null)
		Cv.Lines = new Vector();
	insIdx = Cv.Lines.indexOf( prevObj);
	if ( insIdx == -1)
		insIdx = 0;
	Cv.Lines.insertElementAt( insObj, insIdx);
	Cv.updateContents();
}

public void makeVisible (int index) {
	//java.awt.peer.ListPeer interface
	showObject( elementAt( index));
}

public Vector markedContents() {
	return Cv.Marked;
}

public Dimension minimumSize (int v) {
	//java.awt.peer.ListPeer interface
	int lh = Cv.LineHeight != 0 ? Cv.LineHeight : Awt.SysFontHeight;
	return new Dimension( 30*Awt.SysFontWidth, lh*v);
}

public boolean mouseSelection(){
	return (Cv.SelSource == ListCanvas.OP_SEL_MOUSE);
}

void notifyProcess () {
	if ( Cv.Selection != null) {
		Object arg;
		if ( Cv.Selection instanceof SelfDrawingObject)
			arg = ((SelfDrawingObject) Cv.Selection).label();
		else
			arg = Cv.Selection.toString();
		OsProcess.notifyObservers( Cv.Selection);
		peerEvent( Event.ACTION_EVENT, arg );
	}
}

void notifySelect () {
	Object arg = null;

	if ( Cv.Selection == null)
		peerEvent( Event.LIST_DESELECT, arg );
	else { 
		if ( Cv.Selection instanceof SelfDrawingObject)
			arg = ((SelfDrawingObject) Cv.Selection).label();
		else
			arg = Cv.Selection.toString();
	}

	OsSelection.notifyObservers( Cv.Selection);
	peerEvent( Event.LIST_SELECT, arg );
}

public int numberOfDisplayedItems () {
	return Cv.Height / Cv.LineHeight;
}

public void overrideDrawSocket ( boolean state) {
	Cv.OverrideDrawSocket = state;
}

public int position() {
	return Cv.IdxTop;
}

public Dimension preferredSize(){
	return new Dimension( 10*Awt.SysFontWidth, 10*Awt.SysFontHeight);
}

public Dimension preferredSize (int v) {
	//java.awt.peer.ListPeer interface
	int lh = Cv.LineHeight != 0 ? Cv.LineHeight : Awt.SysFontHeight;
	return new Dimension( 10*Awt.SysFontWidth, lh*v);
}

public void redrawObject ( Object obj ){
	Cv.redrawObject( obj);
}

public void removeAllElements ( Vector list ) {
	if ( list == null)
		return;
	if ( Cv.Lines != null)
		VectorLib.removeAllElements( Cv.Lines, list);
	if ( Cv.Marked != null)
		VectorLib.removeAllElements( Cv.Marked, list);
	Cv.updateContents();
}

public void removeElement ( Object obj ) {
	if ( Cv.Lines != null) {
		Cv.Lines.removeElement( obj);
		if ( Cv.Marked != null)
			Cv.Marked.removeElement( obj);
		Cv.updateContents();
	}
}

public void replaceElementWith ( Object oldObj, Object newObj ) {
	int insIdx;
	if ( Cv.Lines == null)
		return;
	insIdx = Cv.Lines.indexOf( oldObj);
	if ( insIdx == -1)
		return;
	Cv.Lines.setElementAt( newObj, insIdx);
	if ( oldObj.equals( Cv.Selection))
		Cv.Selection = newObj;
	Cv.redrawObject( newObj);
}

public void restoreLastSelection ( boolean fireSlot ) {
	setSelection( Cv.LastSelection, fireSlot);
}

public void scrollVertical ( int nLines ) {
	Cv.scrollVertical( nLines, true);
}

public void select (int index) {
	//java.awt.peer.ListPeer interface
	setSelection( elementAt( index), true);
}

public void selectAll ( boolean notify ) {
	Cv.selectAll( notify);
}

public synchronized void setBackground ( Color c ) {
	Cv.setBackground( c);
}

public void setColors( Color cFore, Color cBack, Color cMFore,
                Color cMBack, Color cSFore, Color cSBack) {
	Cv.setColors( cFore, cBack, cMFore, cMBack, cSFore, cSBack);
}

public int setContents( Vector data){
	return Cv.setContents( data);
}

public boolean setDrawEnvironment ( Object obj ) {
	return Cv.setDrawEnvironment( obj);
}

public void setDrawHeight ( int height ) {
	Cv.setLineHeight( height);
}

public void setDrawWidth ( int width ) {
	Cv.LineWidth = width;
}

public synchronized void setForeground ( Color c ) {
	Cv.setForeground( c);
}

public boolean setMarked ( Object obj ) {
	return Cv.setMarked( obj);
}

public void setMultipleSelections (boolean v) {
	//java.awt.peer.ListPeer interface
}

public boolean setSelection ( Object obj, boolean fireSlot ) {
	return Cv.setSelection( obj, fireSlot);
}

public boolean setSelection( int idx, boolean fireSlot){
	return Cv.setSelection( idx, fireSlot);
}

public boolean shiftDown () {
	int  selIdx;

	if ( Cv.Selection == null)
		return false;

	selIdx = Cv.Lines.indexOf( Cv.Selection);
	if ( selIdx < Cv.Lines.size()-1) {
		VectorLib.swap( Cv.Lines, selIdx, selIdx+1);
		Cv.updateContents();
		Cv.showArea( selIdx, selIdx+1);
		return true;
	}

	return false;
}

public boolean shiftUp () {
	int  selIdx;

	if ( (selIdx = Cv.Lines.indexOf( Cv.Selection)) > 0) {
		VectorLib.swap( Cv.Lines, selIdx-1, selIdx);
		Cv.updateContents();
		Cv.showArea( selIdx-1, selIdx);
		return true;
	}

	return false;
}

public void showLastObject () {
	Cv.showLastObject();
}

public boolean showObject ( Object obj ) {
	return Cv.showObject( obj);
}

public boolean toggleMarker ( Object obj ) {
	return Cv.toggleMarker( obj);
}

public void updateContents () {
	Cv.updateContents();
}

public void updateContents ( Vector newList) {
	Cv.updateContents( newList);
}
}

class ListCanvas
  extends ScrollableCanvas
{
	Color ColorBackMark = Color.lightGray;
	Color ColorBackSel = Awt.SelBackClr;
	Color ColorForeMark = Color.black;
	Color ColorForeSel = Awt.SelForeClr;
	List Master;
	StringBuffer FLNBuf = new StringBuffer();
	Object LastSelection;
	Vector Marked = null;
	final public static int OP_SEL_KBD = 1;
	final public static int OP_SEL_MOUSE = 2;
	public int SelSource;
	Object Selection;
	Rectangle WorkRect = new Rectangle();
	boolean OverrideDrawSocket = false;

ListCanvas( HScrollBar h, VScrollBar v, List cmpd) {
	super(h, v);
	setForeground( Color.black);
	setBackground( Color.white);
	Master = cmpd;
}

public void charInput( char c, int mod) {
	char lc = (char)( c | 32);

	if ( Key.isLoneKey( mod) )
		FLNBuf.setLength( 0);

	FLNBuf.append( lc);
	findFLN();

}

boolean clearMarked(){

	if (Marked == null)
		return false;

	Marked.removeAllElements();
	Marked = null;

	redraw();

	return true;
}

boolean clearMarked( Object obj) {
	if ( (obj == null) || (Marked == null) )
		return false;

	if ( Marked.contains( obj)){
		Marked.removeElement( obj);
		redrawObject( obj);
		return true;
	}

	return false;
}

void clearSelections( boolean redraw ){

	Object ls = Selection;

	Selection     = null;
	LastSelection = null;

	if ( ! clearMarked() && redraw ){
		if ( ls != null)
			redrawObject( ls);
	}
}

void deselect (int idx){
	if ( Lines == null)
		return;

	Object obj = Lines.elementAt( idx);

	if (Marked != null){
		if ( Marked.contains( obj))
			toggleMarker( obj);
	}
	if ( obj == Selection){
		Selection = null;
		drawObject( obj);
	}
}

synchronized boolean drawObject( Object obj) {
	if ( !setDrawEnvironment( obj) )
		return false;
	drawRenderedObject( obj);
	return true;
}

Rectangle drawRectFor( Object obj) {

	int s = Lines.size();

	WorkRect.reshape( XOffset, YOffset, Width - XOffset - FBorder.Ext, LineHeight);

	for ( int idx=IdxTop; idx<s; idx++) {
		if ( WorkRect.y > Height)
			return null;
		if ( Lines.elementAt( idx) == obj)
			return WorkRect;

		WorkRect.move( XOffset, WorkRect.y + LineHeight);
	}

	return null;
}

void drawRenderedObject( Object obj) {

	Rectangle dRect = Master.DrawRect;

	ResGraph.setColor( Master.DrawClrBack);
	ResGraph.fillRect( dRect.x, dRect.y, dRect.width, dRect.height);
	ResGraph.setColor( Master.DrawClrFore);

	if ( (! OverrideDrawSocket) && (! Master.OsDrawObject.isEmpty()) )
		Master.OsDrawObject.notifyObservers( Master);
	else if ( obj instanceof SelfDrawingObject)
		((SelfDrawingObject) obj).drawSelfIn( Master);
	else {
		ResGraph.setFont( Awt.SysFont);
		ResGraph.drawString( obj.toString(), dRect.x + 5,
		                     dRect.y + dRect.height -
		                     (dRect.height - Awt.SysFontHeight) / 2 -
		                     Awt.SysFontDesc );
	}
}

protected boolean findFLN(){
	return selectLabel( FLNBuf.toString());
}

boolean hasStringList(){
	if ( (Lines == null) || (Lines.size() == 0) )
		return false;
	return ( Lines.elementAt( 0) instanceof String);
}

public void mouse1DblClick( Event evt) {
	Master.notifyProcess();
}

public void mouse1Down( Event evt) {
	Object ns = objectAt( evt.x, evt.y);

	if ( evt.controlDown() )
		toggleMarker( ns);

	SelSource = OP_SEL_MOUSE;
	selectObject( ns, false);
}

public void mouse1Up( Event evt) {
	Object ns = objectAt( evt.x, evt.y);
	Rectangle r;
	Point pt;

	if ( ns == Selection)
		Master.notifySelect();

	if ( ns != null){
		r = drawRectFor( ns);
		if ( r != null ) {
			pt = new Point( evt.x - r.x, evt.y - r.y);
			Master.OsMouseSelection.notifyObservers( pt);
		}
	}
}

void redrawLine( int idx){
	Object obj = Lines.elementAt( idx);
	drawObject( obj );
}

void redrawObject ( Object obj ) {
	drawObject( obj);
}

void selectAll ( boolean notify ) {
	Marked = (Vector) Lines.clone();
	redraw();

	if ( notify )
		Master.notifySelect();
}

boolean selectLabel( String label) {
	Object obj;
	SelfDrawingObject sdo;
	String lStr;

	if ( (label == null) || (Lines == null) || (Lines.size() == 0))
		return false;

	label = label.toLowerCase();

	for ( Enumeration e = Lines.elements(); e.hasMoreElements(); ) {
		obj = e.nextElement();

		if ( obj instanceof SelfDrawingObject)
			lStr = ((SelfDrawingObject) obj).label();
		else
			lStr = obj.toString();

		if ( lStr.toLowerCase().startsWith( label) ){
			SelSource = OP_SEL_KBD;
			setSelection( Lines.indexOf( obj), true);
			return true;
		}
	}

	return false;						
}

protected void selectObject( Object obj, boolean notify ) {
	Object ps = Selection;

	LastSelection = Selection;
	Selection = obj;

	if ( ps != null)
		drawObject( ps);
	if ( Selection != null)
		drawObject( Selection);

	if ( notify )
		Master.notifySelect();
}

void setColors( Color cf, Color cb, Color cmf, Color cmb, Color csf, Color csb) {

	ColorForeMark = cmf;
	ColorBackMark = cmb;
	ColorForeSel  = csf;
	ColorBackSel  = csb;

	setBackground( cb);
	setForeground( cf);
}

synchronized int setContents( Vector data ) {
	clearSelections( false);

	Lines   = data;
	IdxTop  = 0;

	XOffset = XBorder + FBorder.Ext;
	YOffset = YBorder + FBorder.Ext;

	if ( (Lines == null) || (Lines.size() == 0)){
		setLineDimensions( null);
		Vert.setValues( 0, 0, 0);
		if ( ResGraph != null)
			redraw();
		return 0;
	}

	setLineDimensions( Lines.firstElement() );
	Vert.setValues( 0, 0, Lines.size() );

	if ( ResGraph != null )
		repaint();

	return Lines.size();
}

boolean setDrawEnvironment ( Object obj ) {
	if ( ResGraph == null) return false;
	return setDrawEnvironment( obj, drawRectFor( obj));
}

boolean setDrawEnvironment ( Object obj, Rectangle dRect ) {

	if ( dRect == null)
		return false;

	Master.DrawingSelected = Master.DrawingMarked = false;

	if ( obj == Selection){
		Master.DrawClrFore = ColorForeSel;
		Master.DrawClrBack = ColorBackSel;
		Master.DrawingSelected = true;
	}
	else if ( (Marked != null) && (Marked.contains( obj)) ){
		Master.DrawClrFore = ColorForeMark;
		Master.DrawClrBack = ColorBackMark;
		Master.DrawingMarked = true;
	}
	else {
		Master.DrawClrFore = getForeground();
		Master.DrawClrBack = getBackground();
	}

	Master.DrawObject = obj;
	Master.DrawRect = dRect;
	Master.DrawGraphics = ResGraph;

	return true;
}

void setLineDimensions ( Object refObj ) {
	if ( refObj != null && refObj instanceof SelfDrawingObject ){
		SelfDrawingObject fo = (SelfDrawingObject) refObj;
		if ( LineHeight == 0)
			LineHeight = fo.drawHeight();
		if ( LineWidth == 0)
			LineWidth = fo.maxDrawWidth();    
	}
	else {
		if ( LineHeight == 0)
			LineHeight = Awt.SysFontHeight;
		if ( LineWidth == 0)
			LineWidth = Awt.ScreenWidth;
	}

	Vert.setStepPels( LineHeight);
	if ( Horz != null){
		Horz.setValues( 0, 0, LineWidth);
		Horz.setLineIncrement( 10);
	}

}

boolean setMarked( Object obj) {
	if ( obj == null)
		return false;

	if ( Marked == null)
		Marked = new Vector();

	if ( ! Marked.contains( obj)){
		Marked.addElement( obj);
		redrawObject( obj);
		return true;
	}

	return false;
}

boolean setSelection( Object obj, boolean fire) {
	if ( Lines == null ){
		if ( obj == null ){
			if ( fire)
				Master.notifySelect();
			return true;
		}
		return false;
	}

	if ( obj == null ) {
		clearSelections( true);
		if ( fire)
			Master.notifySelect();
		return true;
	}

	return setSelection( Lines.indexOf( obj), fire);
}

boolean setSelection( int idx, boolean fire) {
	int lIdx;

	if ( Lines == null)
		return false;

	if ( (idx < 0) || (idx > Lines.size()) )
		return false;

	if ( idx < IdxTop) {
		selectObject (null, false);
		scrollVertical( idx - IdxTop, true);
	}
	else if ( idx > (lIdx = Vert.PageInc + IdxTop - 1)) {
		if ( idx < Lines.size() )
			selectObject( null, false);
		scrollVertical( idx - lIdx, true);
	}

	if ( idx < Lines.size() )
		selectObject( Lines.elementAt( idx), fire);

	return true;
}

boolean shiftSelection( int steps) {

	if ( Lines == null)
		return false;

	SelSource = OP_SEL_KBD;

	if ( Selection == null)
		setSelection( 0, true);
	else
		setSelection( Lines.indexOf( Selection) + steps, true);

	return true;
}

void showLastObject () {
	int n;

	if ( (Lines == null) || ( (n = Lines.size()) == 0))
		return;
	showArea( n, n);
}

boolean showObject ( Object obj ) {
	if ( (Lines == null) || ( ! Lines.contains( obj)) )
		return false;

	int oIdx = Lines.indexOf( obj);
	showArea( oIdx, oIdx);
	return true;
}

boolean toggleMarker( Object obj) {
	if ( obj == null)
		return false;

	if ( (Marked != null) && (Marked.contains( obj)) ){
		Marked.removeElement( obj);
		if ( Marked.isEmpty() )
			Marked = null;
	}
	else{
		if ( Marked == null)
			Marked = new Vector( 10);
		Marked.addElement( obj);
	}

	drawObject( obj);

	return true;
}

void updateContents () {
	Object mrk;

	if ( LineHeight == 0 ){  // first time
		setContents( Lines);
		return;
	}

	if ( Selection != null && ! Lines.contains( Selection) ){
		Selection = null;
		Master.notifySelect();
	}

	if ( Marked != null) {
		for ( int i=0; i<Marked.size(); i++  ) {
			mrk = Marked.elementAt( i);
			if ( ! Lines.contains( mrk) )
				Marked.removeElementAt( i);
		}
	}

	if ( Lines.size() < IdxTop)
		IdxTop = Lines.size();

	updateVScroll();
	redrawLines( IdxTop, -1);  
}

void updateContents ( Vector newList) {
	Lines = newList;
	updateContents();
}

void updateScrollImage( int sIdx){
	Graphics g = ResGraph;
	ResGraph = SoftGraph;

	int y0 = 0;
	for ( int i=0; i<2; i++) {
		WorkRect.reshape( XOffset, y0, Width, LineHeight);
		y0 += LineHeight;
		try {
			Object obj = Lines.elementAt( sIdx+i);
			if ( setDrawEnvironment( obj, WorkRect))
				drawRenderedObject( obj);
		}
		catch ( Throwable t) {
			ResGraph.setColor( getBackground() );
			ResGraph.fillRect( WorkRect.x, WorkRect.y,
			                   WorkRect.width, WorkRect.height);
		}
	}

	ResGraph = g;
}

public boolean vKeyInput ( int k, int mods) {

	if ( mods > 0){
		if ( Key.isCtrlDown(mods) ){
			if (k == 't'){
				toggleMarker( Selection);
				return true;
			}
		}
		return false;
	}

	switch( k) {
	case Key.Newline:
		if (Selection != null)
			Master.notifyProcess();
		break;
	case Key.Escape:
		LastSelection = Selection;
		Selection = null;
		if ( ! clearMarked() )
			redraw();
		scrollVertical( -IdxTop, true);
		Master.notifySelect();
		break;
	case Key.F12:
		selectAll( true);
		break;
	case Key.Home:
		Vert.home();
		break;
	case Key.End:
		Vert.end();
		break;
	case Key.PageUp:
		Vert.pageDown();
		break;
	case Key.PageDown:
		Vert.pageUp();
		break;
	case Key.CursorUp:
		shiftSelection( -1);
		break;
	case Key.CursorDown:
		shiftSelection( 1);
		break;
	case Key.CursorLeft:
		if ( Horz != null)
			Horz.pageDown();
		break;
	case Key.CursorRight:
		if ( Horz != null)
			Horz.pageUp();
		break;
	default:
		return false;
	}

	return true;
}
}
