package biss;


/**
 * Basic Text Scan class. Mostly useful as base for more specialized Scanners.
 * (otherwise not really orthogonal to java.util.StringTokenizer)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class Scanner
{
/**
     C[]:   [.....blabla.........]

	     ^    ^     ^       ^
	     0    I0    I       C.length-1         
  */
	public byte C[];
	public int I0;
	public int I;
	public int Line;
	public int Tok;
	public int CStart;
	public int CEnd;
	public boolean[] IsAN;
	public String[] Keywords;
	final public static byte EOT = 4;
	final public static int UNKNOWN = 0;
	final public static int CHAR = 1;
	final public static int STRING = 2;
	final public static boolean ANTable[] = {
    false, false, false, false, false, false, false, false,  // 0..7
    false, false, false, false, false, false, false, false,  // 8..15
    false, false, false, false, false, false, false, false,  // 16..23
    false, false, false, false, false, false, false, false,  // 24..31
    false, false, false, false, true,  false, false, false,  // 32..39
    false, false, false, false, false, false, false, false,  // 40..47
    true,  true,  true,  true,  true,  true,  true,  true,   // 48..55
    true,  true,  false, false, false, false, false, false,  // 56..63
    false, true,  true,  true,  true,  true,  true,  true,   // 64..71
    true,  true,  true,  true,  true,  true,  true,  true,   // 72..79
    true,  true,  true,  true,  true,  true,  true,  true,   // 80..87
    true,  true,  true,  false, false, false, false, true,   // 88..95
    false, true,  true,  true,  true,  true,  true,  true,   // 96..103
    true,  true,  true,  true,  true,  true,  true,  true,   // 104..111
    true,  true,  true,  true,  true,  true,  true,  true,   // 112..119
    true,  true,  true,  false, false, false, false, false   // 120..127
  };
	final public static String KeyNames[] = { "nothing", "char", "string" };

public Scanner ( byte source[] ) {
	C = source;
	IsAN = getANTable();
	Keywords = getKeywords();

	reset();
}

public char character () {
	return (char) C[I0];
}

public int copyTo ( char buf[], int iStart ) {
	int j = I0, k = iStart;
	try {
		for ( ; j<I; j++, k++ )
			buf[k] = (char)C[j];
	}
	catch ( ArrayIndexOutOfBoundsException x ){};

	return k - iStart;
}

public String currentLine () {
	int j, k;
	for ( j=I-1; j > 0 && C[j] != '\n'; j-- );
	if ( j > 0 ) j++;
	for ( k=I; k<C.length && C[k] != '\n'; k++);
	return new String( C, 0, j, (k-j));
}

protected boolean[] getANTable () {
	return Scanner.ANTable;
}

protected String[] getKeywords () {
	return Scanner.KeyNames;
}

public String keyword ( int id ) {
	if ( id < 0 ) id = -id;

	if ( id > Keywords.length )
		return "<" + String.valueOf(id) + '>';
	else
		return Keywords[id];
}

public String lastComment () {
	String s;

	if ( CEnd > CStart ) {
		s = new String( C, 0, CStart, (CEnd - CStart));
		CStart = 0; CEnd = 0;
	}
	else
		s = null;

	return s;
}

public String nextNonBlankString () {
	if ( nextNonBlankToken() != 0 )
		return string();
	else
		return null;
}

public String nextNonBlankStrings ( int nTokens ) {
	if ( nextNonBlankTokens( nTokens) != 0 )
		return string();
	else
		return null;
}

public int nextNonBlankToken () {
	byte b;
	I0 = I;

	try {
		while ( true ) {
			switch ( (b=C[I]) ) {
			case '\n':
				Line++;
				if ( I > I0 )  return (Tok = STRING);
				I0 = ++I;
				continue;
			case ' ': case '\t': case '\r':
						if ( I > I0 )  return (Tok = STRING);
						I0 = ++I;
						continue;
					default:
						I++;
					}
				}
			}
			catch ( ArrayIndexOutOfBoundsException x ) {
				if ( I > I0 )  return (Tok = STRING);
				return (Tok = UNKNOWN);
			}
		}

public int nextNonBlankTokens ( int nTokens ) {
	int j, i0, ret;

	ret = nextNonBlankToken();
	i0 = I0;

	for ( j=1; j<nTokens; j++ )
		ret = nextNonBlankToken();

	I0 = i0;
	return ret;
}

public int nextToken () {
	byte b;

	try {
		while ( true ){
			I0 = I;

			switch ( (b=C[I]) ) {
			case '\n':
				Line++;
				I++;
				continue;

			case ' ': case '\t': case '\r':
						I++;
						continue;

					default:
						if ( IsAN[b] ) {
							while ( IsAN[C[++I]] );
							return (Tok = STRING);
						}
						else {
							I++;
							return (Tok = b);
						}
					}
				}
			}
			catch ( ArrayIndexOutOfBoundsException x ) {
				if ( I > I0 )  return (Tok = STRING);
				return (Tok = UNKNOWN);
			}

		}

public int nextToken ( char separator ) {
	byte b;
	I0 = I;

	try {
		while ( true ){
			b = C[I];

			if ( b == '\n' )
				Line++;

			if ( b == (byte)separator ) {
				if ( I > I0 )  return (Tok = STRING);
				I0 = ++I;
			}
			else {
				I++;
			}
		}
	}
	catch ( ArrayIndexOutOfBoundsException x ) {
		if ( I > I0 )  return (Tok = STRING);
		return (Tok = UNKNOWN);
	}
}

public int nextToken ( char[] separator ) {
	byte b;
	int j;

	I0 = I;
	try {
		outer:
		while ( true ){
			b = C[I];

			if ( b == '\n' )
				Line++;

			for ( j=0; j<separator.length; j++ ) {
				if ( b == separator[j] ) {
					if ( I > I0 )  return (Tok = STRING);
					I0 = ++I;
					continue outer;
				}
			}
			I++;
		}
	}
	catch ( ArrayIndexOutOfBoundsException x ) {
		if ( I > I0 )  return (Tok = STRING);
		return (Tok = UNKNOWN);
	}

}

public void reset () {
	I0 = 0; I = 0;
	Line = 1;
	Tok = 0;
	CStart = 0;
	CEnd = 0;
}

public void reset ( byte[] src ) {
	C = src;
	reset();
}

public void resetComment() {
	CStart = 0;
	CEnd = 0;
}

public void setStartMark () {
	I0 = I;
}

public void setStartMark ( int i0 ) {
	I0 = i0;
}

public boolean skip () {
	try {
		if ( C[++I] == '\n' )
			Line++;

		return true;
	}
	catch ( ArrayIndexOutOfBoundsException x ) {
		return false;
	}
}

public boolean skip ( int nChars ) {
	int iEnd = I + nChars;

	try {
		for ( ; I < iEnd; I++ ) {
			if ( C[I] == '\n' )
				Line++;
		}
		return true;
	}
	catch ( ArrayIndexOutOfBoundsException x ) {
		return false;
	}
}

public void skipBlanks () {
	try {
		while ( C[I] <= ' ' ){
			if ( C[I] == '\n' ) Line++;
			I++;
		}
	}
	catch ( ArrayIndexOutOfBoundsException x ){};
}

public boolean skipBlock ( char delim1, char delim2 ) {  // { .. }
	int level = 1;
	byte a;

	try {
		while ( true ){

			if ( (a = C[I]) == (byte) delim1 ) {
				level++;
				I++;
			}
			else if ( a == (byte) delim2 ){
				I++;
				if ( --level == 0 )
					return true;
			}
			else if ( a == '\n' ) {
				I++;
				Line++;
			}
			else {
				I++;
			}
		}
	}
	catch ( ArrayIndexOutOfBoundsException x ){
		return false;    
	}
}

public boolean skipTo ( char delim) {
	byte[] c = C;
	byte b;

	try {
		while ( (b = c[I]) != delim ){
			I++;
			if ( b == '\n' ) Line++;
		}
		return true;
	}
	catch ( ArrayIndexOutOfBoundsException x ) {
		return false;
	}
}

public void skipToEOL () {
	try {
		while ( C[I] != '\n' ) I++;
		Line++;
	}
	catch ( ArrayIndexOutOfBoundsException x ){};  
}

public int skipToFirstOf ( char[] sep ) {
	try {
		while ( true ) {
			byte b = C[I];

			if ( b == '\n' )
				Line++;

			for ( int j=0; j<sep.length; j++ ) {
				if ( b == sep[j] ){
					I++;
					return b;
				}
			}
			I++;
		}
	}
	catch ( ArrayIndexOutOfBoundsException x ) {
		return 0;
	}
}

public String string () {
	return new String( C, 0, I0, (I-I0));
}

public String stringFrom( int iStart ) {
	return new String( C, 0, iStart, (I - iStart));
}
}
