static char statfile_c_rcsid[]="statfile.c,v 1.3 1994/12/01 14:46:02 kerce Exp";

/*-----------------------------------------------------------------------------
 * Kingsley Kerce
 *
 * Copyright (c) 1994
 *
 * Supercomputer Computations Research Institute (SCRI)
 * Florida State University
 *
 * SCRI representatives make no claims about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * statfile.c,v
 * Revision 1.3  1994/12/01  14:46:02  kerce
 * Solaris 2.x mods
 *
 * Revision 1.2  1994/06/22  15:48:23  kerce
 * For "CPU Usage Options" window, allow insertion/removal of defaults
 *   with the mere click of a button
 *
 * Better sanity checking for qusage/qacct's options
 *
 * Revision 1.1.1.1  1994/06/18  19:43:31  kerce
 * DQS X Distribution
 *
 *---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <math.h>
#include <sys/time.h>
#if defined(SYSV) || defined(SVR4)
#include <string.h>
#else
#include <strings.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include "h.h"
#include "def.h"
#include "dqs.h"
#include "struct.h"
#include "func.h"
#include "globals.h"
#include "proto.h"
#include "error.h"
#include "stdstuff.h"
#include "request.h"
#include "statfile.h"

STATFILE *
StatFileCreate (StatFileName, Result)
     char *StatFileName;
     int *Result;
{
  STATFILE *StatFile;

  DENTER ((DQS_EVENT, "StatFileCreate"));

  StatFile = (STATFILE *) calloc (1, sizeof (STATFILE));
  if (StatFile == (STATFILE *) NULL)
    {
      *Result = STATFILE_OUTOFMEM;
      DEXIT;
      return ((STATFILE *) NULL);
    }

  StatFile->StatFileName = StatFileName;

  StatFile->StatFilePtr = fopen (StatFile->StatFileName, "r");
  if (StatFile->StatFilePtr == (FILE *) NULL)
    {
      free (StatFile);
      *Result = STATFILE_NOFOPEN;
      DEXIT;
      return ((STATFILE *) NULL);
    }

  StatFile->Stats = (dqs_stats_type *) calloc (1, sizeof (dqs_stats_type));
  if (StatFile->Stats == (dqs_stats_type *) NULL)
    {
      fclose (StatFile->StatFilePtr);
      free (StatFile);
      *Result = STATFILE_OUTOFMEM;
      DEXIT;
      return ((STATFILE *) NULL);
    }

  *Result = STATFILE_SUCCESS;
  DEXIT;
  return ((STATFILE *) StatFile);
}

void
StatFileDestroy (StatFile)
     STATFILE *StatFile;
{
  DENTER ((DQS_EVENT, "StatFileDestroy"));

  free (StatFile->Stats);
  free (StatFile);
  fclose (StatFile->StatFilePtr);
  StatFile = (STATFILE *) NULL;

  DEXIT;
}

#define STATFILE_FSCANF_FORMAT_2_1 "%ld %s %s %s %s %ld %ld"

#define STATFILE_FSCANF_COUNT_2_1 7

#define STATFILE_FSCANF_VARS_2_1 \
             &Stats->time, \
	     Stats->hostname, \
  	     Stats->qname, \
	     qcomplex_str, \
	     qcomplex_str, \
	     &Stats->qty, \
	     &Stats->load_avg

/*******************************************************************************************/
int
dqs_read_stats_2_1 (f, Stats, qcomplex_str)
FILE *f;
dqs_stats_type *Stats;
char *qcomplex_str;
/*
  dqs_read_stats_2_1 - read stat info from file.
  Returns: -3 if f is NULL
           -2 if unexpected EOF reached
           -1 if the number of items read was incorrect
            0 if success
            1 if expected EOF reached
*/

{
  int fscanf_count;

  DENTER((DQS_EVENT,"dqs_read_stats_2_1"));

  if (f == NULL)
    return (-3);

  fscanf_count = fscanf (f, STATFILE_FSCANF_FORMAT_2_1,
			 STATFILE_FSCANF_VARS_2_1);
  if (fscanf_count == EOF)
    {
      /* Expected end of file. */
      DEXIT;
      return (1);
    }
  else if (fscanf_count != STATFILE_FSCANF_COUNT_2_1)
    {
      /* Bad news.  File must be malformed. */
      fscanf_count = fscanf(f, STATFILE_FSCANF_FORMAT_2_1,
			    STATFILE_FSCANF_VARS_2_1);
      if (fscanf_count != EOF)
	{
	  DEXIT;
	  return (-1);
	}
      else
	{
	  DEXIT;
	  return (-2);
	}
    }

  DEXIT;
  return (0);
}

static int
StatFileStatsRead (StatFile, qcomplex_str, Ver2_1)
     STATFILE *StatFile;
     char *qcomplex_str;
     BOOLEAN Ver2_1;
{
  DENTER ((DQS_EVENT, "StatFileStatsRead"));

  if (Ver2_1 == TRUE)
    {
      DEXIT;
      return (dqs_read_stats_2_1 (StatFile->StatFilePtr, StatFile->Stats,
				  qcomplex_str));
    }    				  
  else
    {
      DEXIT;
      return (dqs_read_stats (StatFile->StatFilePtr, StatFile->Stats,
			      qcomplex_str));
    }
}

/*
 *  Given an open file, start reading in order to find a newline.
 *  Returns: TRUE if newline found.  File pointer follows newline.
 *           FALSE if end-of-file reached.
 */
static BOOLEAN
NewLineSkip (FilePtr)
     FILE *FilePtr;
{
  BOOLEAN Done, Found;
  int fgetcResult;
  
  DENTER ((DQS_EVENT, "NewLineSkip"));

  Done = FALSE;
  Found = FALSE;
  while (Done == FALSE)
    {
      fgetcResult = fgetc (FilePtr);
      if (fgetcResult == '\n')
	Done = Found = TRUE;
      else if (fgetcResult == EOF)
	Done = TRUE;
    }

  DEXIT;
  return (Found);
}

/*
 *  For debugging...
 */
static void
StatsWrite (Stats, qcomplex_str)
     dqs_stats_type *Stats;
     char *qcomplex_str;
{
  DENTER ((DQS_EVENT, "StatsWrite"));

  DPRINTF ((DQS_EVENT, "%lu %s %s %s %lu %lu",
	    Stats->time,
	    Stats->hostname,
	    Stats->qname,
	    qcomplex_str,
	    Stats->qty,
	    Stats->load_avg));

  DEXIT;
}

static BOOLEAN
Match (Stats, QComplex, Req)
     dqs_stats_type *Stats;
     dqs_list_type *QComplex;
     REQUEST *Req;
{
  BOOLEAN QCMatch, QNameMatch, HostNameMatch;

  DENTER ((DQS_EVENT, "Match"));

  QCMatch = FALSE;
  QNameMatch = FALSE;
  HostNameMatch = FALSE;

  if ((Req->QComplex != NULL) &&
      (QComplex != NULL))
    /* Check for a queue complex match. */
    {
      int Suitable, NumSuitable;
      dqs_list_type *QC, *ReqQC;
      
      QC = QComplex->chain;
      ReqQC = Req->QComplex->chain;
      
      NumSuitable = 0;
      
      while (ReqQC != NULL)
	{
	  while (QC != NULL)
	    {
	      Suitable = dqs_complex_el_suitable (ReqQC, QC);
	      if (Suitable == 1)
		NumSuitable++;
	      QC = QC->next;
	    }
	  
	  ReqQC = ReqQC->next;
	}
      
      if (NumSuitable == Req->QComplex->int1)
	QCMatch = TRUE;
      else
	QCMatch = FALSE;
    }

  if (dqs_wildmat (Req->QName, Stats->qname) == TRUE)
    QNameMatch = TRUE;
    
  if (dqs_wildmat (Req->HostName, Stats->hostname) == TRUE)
    HostNameMatch = TRUE;
  
  if ((QCMatch == TRUE) || (QNameMatch == TRUE) || (HostNameMatch == TRUE))
    {
      DEXIT;
      return (TRUE);
    }
  else
    {
      DEXIT;
      return (FALSE);
    }
}

int
StatFileCalc (StatFile, IStop, Req, Ver2_1)
     STATFILE *StatFile;
     unsigned long IStop;
     REQUEST *Req;
     BOOLEAN Ver2_1;
{
  int dqs_read_statsResult;
  unsigned long int CurrentTime, LoadSum, ActiveSum, QueueCount;
  BOOLEAN FirstTime, Matched;
  char qcomplex_str[255];
  double ActiveAvg, LoadAvgAvg;
  
  DENTER ((DQS_EVENT, "StatFileCalc"));

  CurrentTime = 0;
  FirstTime = TRUE;

  do
    {
      dqs_read_statsResult =
	StatFileStatsRead (StatFile, qcomplex_str, Ver2_1);
      if (dqs_read_statsResult < 0)
	{
	  DEXIT;
	  return (dqs_read_statsResult);
	}
      
      if (dqs_read_statsResult == 1)
	{
	  /* end of file... */
	  if (QueueCount > 0)
	    {
	      ActiveAvg = (double) ActiveSum / (double) QueueCount;
	      LoadAvgAvg =
		(double) LoadSum / ((double) QueueCount * 100.0);
	    }
	  else
	    {
	      ActiveAvg = 0.0;
	      LoadAvgAvg = 0.0;
	    }

	  printf ("%lu\t%f\t%f\n",
		  StatFile->Stats->time, ActiveAvg, LoadAvgAvg);
		  
	  DEXIT;
	  return (STATFSRCH_SUCCESS);
	}

      Matched = Match (StatFile->Stats, NULL, Req);

      if (StatFile->Stats->time != CurrentTime)
	{
	  if (FirstTime == TRUE)
	    {
	      FirstTime = FALSE;
	    }
	  else
	    {
	      if (QueueCount > 0)
		{
		  ActiveAvg = (double) ActiveSum / (double) QueueCount;
		  LoadAvgAvg =
		    (double) LoadSum / ((double) QueueCount * 100.0);
		}
	      else
		{
		  ActiveAvg = 0.0;
		  LoadAvgAvg = 0.0;
		}
	      
	      printf ("%lu\t%f\t%f\n", CurrentTime, ActiveAvg, LoadAvgAvg);
	    }
	  
	  if (Matched == TRUE)
	    {
	      ActiveSum = StatFile->Stats->qty_active;
	      LoadSum = StatFile->Stats->load_avg;
	      QueueCount = 1;
	    }
	  else
	    {
	      ActiveSum = 0;
	      LoadSum = 0;
	      QueueCount = 0;
	    }

	  CurrentTime = StatFile->Stats->time;
	}
      else
	{
	  if (Matched == TRUE)
	    {
	      ActiveSum += StatFile->Stats->qty_active;
	      LoadSum += StatFile->Stats->load_avg;
	      QueueCount++;
	      DPRINTF ((DQS_EVENT,
			"ActiveSum = %lu, LoadSum = %lu, QueueCount = %lu",
			ActiveSum, LoadSum, QueueCount));
	    }
	}

      if (StatFile->Stats->time > IStop)
	{
	  DEXIT;
	  return (STATFSRCH_SUCCESS);
	}
    }
  while (1);

  DEXIT;
}

/* Find out the first and last times that the stats were logged. */
static int
FirstAndLastTimes (StatFile, Ver2_1, FirstTime, LastTime)
     STATFILE *StatFile;
     BOOLEAN Ver2_1;
     unsigned long *FirstTime;
     unsigned long *LastTime;
{
  char qcomplex_str[255];
  BOOLEAN FoundNewLine;
  int dqs_read_statsResult, fseekResult;

  DENTER ((DQS_EVENT, "FirstAndLastTimes"));

  dqs_read_statsResult =
    StatFileStatsRead (StatFile, qcomplex_str, Ver2_1);
  if (dqs_read_statsResult < 0)
    {
      DEXIT;
      return (dqs_read_statsResult);
    }
  
  *FirstTime = StatFile->Stats->time;
  
  /*
   *  If the file is long enough, seek 500 bytes back from the end and
   *  move to the next line.
   *  Otherwise, the file is short so we just start reading...
   */
  if (StatFile->FileStat.st_size > 500)
    {
      fseekResult =
	fseek (StatFile->StatFilePtr, (StatFile->FileStat.st_size - 500), 0);
      if (fseekResult != 0)
	{
	  Error (__FILE__, __LINE__, 0, "fseek() failed on stat file");
	  DEXIT;
	  return (STATFSRCH_FSEEK_FAIL);
	}
      
      FoundNewLine = NewLineSkip (StatFile->StatFilePtr);
      if (FoundNewLine == FALSE)
	{
	  DEXIT;
	  return (STATFSRCH_UNEXP_EOF);
	}
    }

  do
    {
      dqs_read_statsResult =
	StatFileStatsRead (StatFile, qcomplex_str, Ver2_1);
      if (dqs_read_statsResult < 0)
	{
	  DEXIT;
	  return (dqs_read_statsResult);
	}
    }
  while (dqs_read_statsResult != 1);
  
  *LastTime = StatFile->Stats->time;

  /* Rewind... */
  fseekResult = fseek (StatFile->StatFilePtr, 0, 0);
  if (fseekResult != 0)
    {
      Error (__FILE__, __LINE__, 0, "fseek() failed on stat file");
      DEXIT;
      return (STATFSRCH_FSEEK_FAIL);
    }

  DEXIT;
  return (STATFSRCH_SUCCESS);
}

static int
BinarySearch (StatFile, IStart, Ver2_1)
     STATFILE *StatFile;
     unsigned long IStart;
     BOOLEAN Ver2_1;
{
  off_t seekdest;
  unsigned long int power2, PrevTime, Offset, ftellResult, ftellResultPrev;
  int dqs_read_statsResult, fseekResult;
  BOOLEAN Found, FoundNewLine;
  char qcomplex_str[255];

  DENTER ((DQS_EVENT, "BinarySearch"));

  /* Start looking in the middle of the file. */
  seekdest = StatFile->FileStat.st_size / 2;

  power2 = 2;
  PrevTime = 0;
  ftellResultPrev = 0;
  ftellResult = 0;

  Found = FALSE;
  while (Found == FALSE)
    {
      DPRINTF ((DQS_EVENT, "seekdest = %ld | ", seekdest));
      fseekResult = fseek (StatFile->StatFilePtr, seekdest, 0);
      if (fseekResult != 0)
	{
	  Error (__FILE__, __LINE__, 0, "fseek() failed on stat file");
	  DEXIT;
	  return (STATFSRCH_FSEEK_FAIL);
	}

      /*
       *  We've most likely jumped into the middle of a line, so move to
       *  beginning of next line.
       */
      FoundNewLine = NewLineSkip (StatFile->StatFilePtr);
      if (FoundNewLine == FALSE)
	{
	  DEXIT;
	  return (STATFSRCH_UNEXP_EOF);
	}

      dqs_read_statsResult =
	StatFileStatsRead (StatFile, qcomplex_str, Ver2_1);
      if (dqs_read_statsResult < 0)
	{
	  DEXIT;
	  return (dqs_read_statsResult);
	}

      StatsWrite (StatFile->Stats, qcomplex_str);

      {
	unsigned long int Prevpower2;

	Prevpower2 = power2;
	power2 *= 2;

	/*
	 * Power2 should always increase, otherwise we've probably overflown.
	 */
	if (power2 < Prevpower2)
	  {
	    Error (__FILE__, __LINE__, 0, "INTERNAL ERROR");
	    DEXIT;
	    return (STATFSRCH_ARITH_OVERFLOW);
	  }
      }

      Offset = StatFile->FileStat.st_size / power2;

      if (StatFile->Stats->time == PrevTime)
	{
	  if (ftellResultPrev == 0)
	    {
	      DEXIT;
	      FatalError (__FILE__, __LINE__, 0, "INTERNAL ERROR");
	    }

	  fseekResult = fseek (StatFile->StatFilePtr, ftellResultPrev, 0);
	  if (fseekResult != 0)
	    {
	      Error (__FILE__, __LINE__, 0, "fseek() failed on stat file");
	      DEXIT;
	      return (STATFSRCH_FSEEK_FAIL);
	    }

	  Found = TRUE;
	}
      else
	{
	  if (StatFile->Stats->time < IStart)
	    {
	      seekdest = seekdest + Offset;
	      ftellResultPrev = ftellResult;
	      ftellResult = ftell (StatFile->StatFilePtr);
	    }
	  else if (StatFile->Stats->time > IStart)
	    {
	      seekdest = seekdest - Offset;
	      ftellResultPrev = ftellResult;
	      ftellResult = ftell (StatFile->StatFilePtr);
	    }
	  else /* StatFile->Stats->time == IStart */
	    {
	      Found = TRUE;
	    }
	}

      PrevTime = StatFile->Stats->time;
    }

  DEXIT;
  return (STATFSRCH_SUCCESS);
}

/*
 *  Perform binary search of StatFile, looking for the time IStart.
 *  If STATFSRCH_SUCCESS is returned, file pointer is sitting on the first
 *    statistics line having a time that's less than or equal to the IStart
 *    time.
 */
int
StatFileSearch (StatFile, IStart, IStop, Ver2_1)
     STATFILE *StatFile;
     unsigned long IStart;
     unsigned long IStop;
     BOOLEAN Ver2_1;
{
  int dqs_read_statsResult, fseekResult;
  char qcomplex_str[255];
  unsigned long int FirstTime, LastTime, ftellResult;

  DENTER ((DQS_EVENT, "StatFileSearch"));

  DPRINTF ((DQS_EVENT, "IStart = %ld", IStart));
  DPRINTF ((DQS_EVENT, "IStop = %ld", IStop));

  {
    int statResult;

    statResult = stat (StatFile->StatFileName, &StatFile->FileStat);
    if (statResult != 0)
      {
	DEXIT;
	return (STATFSRCH_STAT_FAIL);
      }
  }
  
  if (StatFile->FileStat.st_size <= 0)
    {
      DEXIT;
      return (STATFSRCH_EMPTY);
    }

  {
    int FirstAndLastTimesResult;

    FirstAndLastTimesResult =
      FirstAndLastTimes (StatFile, Ver2_1, &FirstTime, &LastTime);
    if (FirstAndLastTimesResult != STATFSRCH_SUCCESS)
      {
	DEXIT;
	return (FirstAndLastTimesResult);
      }
  }

  DPRINTF ((DQS_EVENT, "FirstTime = %ld", FirstTime));
  DPRINTF ((DQS_EVENT, "LastTime = %ld", LastTime));
  
  if (IStop < FirstTime)
    {
      DEXIT;
      return (STATFSRCH_NOT_FOUND);
    }
  
  if (IStart > LastTime)
    {
      DEXIT;
      return (STATFSRCH_NOT_FOUND);
    }

  {
    int BinarySearchResult;

    if (IStart > FirstTime)
      {
	BinarySearchResult = BinarySearch (StatFile, IStart, Ver2_1);
	if (BinarySearchResult != STATFSRCH_SUCCESS)
	  {
	    DEXIT;
	    return (BinarySearchResult);
	  }
      }
  }

  /*
   *  We're now sitting on the statistics "just before" the IStart
   *  (or greater) time.
   *  Read until the IStart time is reached, then back up a line.
   */
  do
    {
      ftellResult = ftell (StatFile->StatFilePtr);

      dqs_read_statsResult =
	StatFileStatsRead (StatFile, qcomplex_str, Ver2_1);
      if (dqs_read_statsResult < 0)
	{
	  DEXIT;
	  return (dqs_read_statsResult);
	}
      
      StatsWrite (StatFile->Stats, qcomplex_str);
    }
  while (StatFile->Stats->time < IStart);

  fseekResult = fseek (StatFile->StatFilePtr, ftellResult, 0);
  if (fseekResult != 0)
    {
      Error (__FILE__, __LINE__, 0, "fseek() failed on stat file");
      DEXIT;
      return (STATFSRCH_FSEEK_FAIL);
    }

  DEXIT;
  return (STATFSRCH_SUCCESS);
}

/* Set Emacs C-mode style.
   Local Variables:
   c-style: GNU
   End:
 */
