/*
 *  misc.c:		Miscellaneous functions
 *
 *  Written by:		Ullrich Hafner
 *		
 *  Copyright (C) 1998 Ullrich Hafner <hafner@bigfoot.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1999/09/28 17:40:36 $
 *  $Author: hafner $
 *  $Revision: 1.23 $
 *  $State: Exp $
 */

#include "config.h"

#if HAVE_STDLIB_H
#	include <stdlib.h>
#endif /* not HAVE_STDLIB_H */
#if HAVE_STRING_H
#	include <string.h>
#else /* not HAVE_STRING_H */
#	include <strings.h>
#endif /* not HAVE_STRING_H */
#if HAVE_SYS_TYPES_H
#	include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <stdio.h>
#include <gtk/gtk.h>
#include <pwd.h>
/* unistd.h defines _POSIX_VERSION on POSIX.1 systems. */
#if defined(HAVE_DIRENT_H) || defined(_POSIX_VERSION)
#   include <dirent.h>
#   define NLENGTH(dirent) (strlen ((dirent)->d_name))
#else
#   define dirent direct
#   define NLENGTH(dirent) ((dirent)->d_namlen)
#   ifdef HAVE_SYS_NDIR_H
#       include <sys/ndir.h>
#   endif /* HAVE_SYS_NDIR_H */
#   ifdef HAVE_SYS_DIR_H
#       include <sys/dir.h>
#   endif /* HAVE_SYS_DIR_H */
#   ifdef HAVE_NDIR_H
#       include <ndir.h>
#   endif /* HAVE_NDIR_H */
#endif /* not (HAVE_DIRENT_H or _POSIX_VERSION) */

#include "misc.h"
#include "load.h"
#include "error.h"

void
Free (void *ptr)
{
   g_free (ptr);
}

void *
Calloc (size_t n, size_t size)
/*
 *  Allocate memory like calloc ().
 *
 *  Return value: Pointer to the new block of memory on success,
 *		  else terminate the program.
 */
{
   void		*ptr;			/* pointer to the new memory block */

   if (n <= 0 || size <= 0)
      error ("Can't allocate memory for %d items of size %d",
	     (unsigned) n, (unsigned) size);

   ptr = g_malloc (n * size);
   if (ptr == NULL)
      error (_("Out of memory!"));
   memset (ptr, 0, n * size);
   return ptr;
}

char *
get_temporary_file_name (void)
{
#ifdef HAVE_TEMPNAM
   return (char *) tempnam (NULL, "wmcnf");
#else  /* not HAVE_TEMPNAM */
   return tmpnam (NULL);
#endif /* not HAVE_TEMPNAM */
}

char *
get_gnustep_path (const char *domain)
/*
 *  Generate path to file 'domain' in users GNUstep directory
 *
 *  E.g. domain = 'Defaults/WindowMaker', 'Defaults/WMRootMenu'
 *
 *  Return value:
 *	pointer to char array containing generated path
 */
{
   const char *gspath = getenv ("GNUSTEP_USER_ROOT");
   
   if (gspath)
      return g_strconcat (gspath, "/", domain, NULL);
   else
      return g_strconcat (g_get_home_dir (), "/GNUstep/", domain, NULL);
}

char *
expand_tilde (const char *name)
/*
 *  Try to expand tilde (~) in filename.
 *
 *  Return value:
 *	string with expanded path 
 */
{
   struct passwd *user = NULL;
   
   assert (name);

   if (name [0] != '~' && name [0] != '$')
      return g_strdup (name);		/* nothing to do */
   
   if (name [1] == '/' || !name [1]	/* users home directory */
       || strncmp (name, "$HOME", 5) == 0
       || strncmp (name, "$(HOME)", 7) == 0)
   {
      if (name [0] == '~')
	 name += 1;			/* skip ~ */
      else if (strncmp (name, "$HOME", 5) == 0)
	 name += 5;
      else if (strncmp (name, "$(HOME)", 7) == 0)
	 name += 7;
      
      return g_strconcat (g_get_home_dir (), name, NULL);
   }
   else if (name [0] == '$')		/* environment expansion */
   {
      const char *first = name + 1;
      const char *last;
      const char *rest;
      
      if (*first == '(')		/* $(ENV) */
      {
	 if (!(last = strrchr (name, ')')))
	    return g_strdup (name);	/* parse error */
	 first++;
	 rest = last + 1;
	 last--;
      }
      else
      {
	 if (!(rest = strchr (name, '/'))) /* "$ENV" */
	 {
	    last = first + strlen (first) - 1;
	    rest = last + 1;		/* empty */
	 }
	 else				/* "$ENV/rest" */
	    last = rest - 1;
      }
      {
	 char		*var = g_strndup (first, last - first + 1);
	 const char	*dir = getenv (var);

	 Free (var);
	 if (dir)
	    return g_strconcat (dir, rest, NULL);
	 else
	    return g_strdup (name);	/* parse error */
      }
   }
#ifdef HAVE_GETPWNAM   
   else					/* other user directory */
   {
      char *usrname = strchr (name, '/');

      if (usrname)
      {
	 char *tmp = g_strndup (name + 1, usrname - name - 1);
	 
	 user = getpwnam (tmp);
	 Free (tmp);
	 name = usrname;		/* first slash */
      }
      else
      {
	 user = getpwnam (name + 1);
	 name += strlen (name);		/* empty string */
      }
   }

   if (!user)
      warning (_("Can't find passwd entry to expand\n`~' in filename %s"));
   else
      return g_strconcat (user->pw_dir, name, NULL);
#endif /* HAVE_GETPWNAM */

   return g_strdup (name);		/* no success */
}

void
delete_file_or_dir (const char *filename)
{
#ifdef HAVE_REMOVE
   remove (filename);
#else  /* not HAVE_REMOVE */
   char *tmp;
   DIR  *dir = opendir (filename);

   if (!dir)				/* a file */
      tmp = g_strconcat ("rm -f ", filename, NULL);
   else					/* a directory */
   {
      tmp = g_strconcat ("rmdir ", filename, NULL);
      closedir (dir);
   }
   system (tmp);
   Free (tmp);
#endif /* not HAVE_REMOVE */
}

char *
preview_name (const char *name)
{
   char *str = g_strdup (name);
   char *ptr, *path;

   for (ptr = str; *ptr; ptr++)
      if (*ptr == '/')
	 *ptr = '-';
   path = g_strconcat (g_get_home_dir (), "/.wmakerconf/", str, NULL);
   Free (str);

   return path;
}

proplist_t *
read_proplist (const char *filename)
{
   DIR *dir = opendir (filename);

   if (dir)				/* directory */
   {
      closedir (dir);
      return NULL;
   }
   else
      return ReadProplistFromFile (filename);
}
