namespace eval mkdir {
    variable mkdir

    # Define the mkdir array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set mkdir {
	list.reset ""
	list.clear ""
	entry.dir ""
	entry.regexp ""
	dialog ""
	dir ""
	make_parent ""
    }
}

# mkdir::create --
#
#   Method to create the dialog box for the mkdir command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc mkdir::create { } {
    global tkWorld
    variable mkdir

    # Put the focus on the mkdir dialog if it is already open.
    if [winfo exists $mkdir(dialog)] {
	switch -- [wm state $mkdir(dialog)] {
	    normal {
		raise $mkdir(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $mkdir(dialog)
	    }
	}
	focus $mkdir(dialog)
	return
    } else {
	set mkdir(dialog) [dialog::create .mkdir Mkdir]
    }

    # The first tab has the dir information, along with the make
    # parent command line option.
    set tab1 [tabnotebook::page [dialog::interior \
	    $mkdir(dialog)] "Options"]

    # Use a frame to encapsulate the dir options so that
    # the frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    label $f1.dir_label \
	    -text "Directory" \
	    -width 10
    set mkdir(entry.dir) [entry $f1.entry_dir \
	    -width 35 \
	    -textvariable mkdir::mkdir(dir)]
    grid $f1.dir_label $f1.entry_dir \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Build the make parent option frame
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]
    checkbutton $f2.make_parent \
	    -text "Make Parent" \
	    -variable mkdir::mkdir(make_parent) \
	    -onvalue "p" \
	    -offvalue ""
    grid $f2.make_parent \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build the first tab.
    pack $f1 $f2 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # The 2nd tab has directory permission information.
    set tab2 [tabnotebook::page [dialog::interior \
	    $mkdir(dialog)] "Permissions"]
    set f3 [frame $tab2.f3 \
	    -class TabnotebookFrame]

    set col 0
    foreach t "All User Group Other" {
	set x [string tolower $t]
	set cb($x) [checkbox::create $f3.$x "$t"]
	grid $f3.$x \
		-row 0 \
		-column $col \
		-sticky w \
		-padx 5 \
		-pady 0
	incr col
    }
    foreach {a b} {Read 4 Write 2 Execute 1} {
	set t [string tolower $a]
	checkbox::add $cb(all) $a $b "0" mkdir::mkdir(all_perm_$t)
	checkbox::add $cb(user) $a $b "0" mkdir::mkdir(user_perm_$t)
	checkbox::add $cb(group) $a $b "0" mkdir::mkdir(group_perm_$t)
	checkbox::add $cb(other) $a $b "0" mkdir::mkdir(other_perm_$t)
    }
    unset a b col cb x t

    # Build the second tab.
    pack $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $mkdir(entry.dir)

    # Define the lists for the reset and clear methods
    set mkdir(list.reset) "make_parent all_perm_read all_perm_write \
	    all_perm_execute user_perm_read user_perm_write \
	    user_perm_execute group_perm_read group_perm_write \
	    group_perm_execute other_perm_read other_perm_write \
	    other_perm_execute"
    set mkdir(list.clear) "dir"
}

# mkdir::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc mkdir::ok { } {
    global tkWorld
    variable mkdir

    # Build the directory list with the working directory in
    # it so it works.
    set dlist ""
    foreach d $mkdir(dir) {
	if [regexp $tkWorld(working_dir) $d] {
	    append dlist "$d "
	} else {
	    append dlist "$tkWorld(working_dir)/$d "
	}
    }

    # Build the first argument if we should build the
    # parent directory.
    set cmd_arg ""
    if [string length $mkdir(make_parent)] {
	set cmd_arg "-$mkdir(make_parent) "
    }

    # Now define the permission stuff if they are turned on.
    foreach x "all user group other" {
	set total($x) [expr $mkdir($x\_perm_read) + \
		$mkdir($x\_perm_write) + \
		$mkdir($x\_perm_execute)]
    }
    if {$total(all) > 0} {
	append cmd_arg "-m \"$total(all)$total(all)$total(all)\" "
    } elseif {$total(user) > 0 || $total(group) > 0 || $total(other) > 0} {
	append cmd_arg "-m \"$total(user)$total(group)$total(other)\" "
    } else {
	append cmd_arg {"000"}
    }
    unset total

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    if [string length $cmd_arg] {
	set cmd_arg [string trimright $cmd_arg " "]
	$tkWorld(cmd_center) insert insert \
		"mkdir $cmd_arg $dlist"
    } else {
	$tkWorld(cmd_center) insert insert \
		"mkdir $dlist"
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled

    # Change to the default background/foreground of the
    # directory entry.
    $mkdir(entry.dir) configure -bg white -fg black

    # Give a warning if no directory name is supplied.
    if ![string length $mkdir(dir)] {
	$mkdir(entry.dir) configure -bg yellow
	stderr::log 3001
    }
}

# mkdir::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc mkdir::reset { } {
    variable mkdir

    # Allow for the permission elements to maintain an integer value.
    foreach x $mkdir(list.reset) {
	switch -glob $x {
	    all_perm* -
	    user_perm* -
	    group_perm* -
	    other_perm* {
		set mkdir($x) 0
	    }
	    default {
		set mkdir($x) ""
	    }
	}
    }
}

# mkdir::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc mkdir::clear { } {
    variable mkdir

    # Reset the data structure elements and bg/fg.
    foreach x $mkdir(list.clear) {
	set mkdir($x) ""
	$mkdir(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $mkdir(entry.dir)
}

# mkdir::help --
#
#   Method to invoke the Mkdir Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc mkdir::help { } {
    global tkWorld

    help::create "mkdir/index.html" "Mkdir Command Help"
}

# mkdir::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc mkdir::close { } {
    variable mkdir
    
    balloonhelp::cancel
    destroy $mkdir(dialog)
}
