/*
 * Copyright (C) 1998 Sasha Vasko <sashav@sprintmail.com>
 * Copyright (C) 1995 Bo Yang
 * Copyright (C) 1993 Robert Nation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../configure.h"

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>

#ifdef NeXT
#include <fcntl.h>
#endif

#include "Wharf.h"
#include "../../include/loadimg.h"

#ifdef XPM
#include <X11/xpm.h>
#endif /* XPM */

/****************************************************************************
 *
 * Loads an icon file into a pixmap
 *
 ****************************************************************************/
void
LoadIconFile (icon_info * icon)
{
#ifndef NO_ICONS
  /* First, check for a monochrome bitmap */
  if ((*icon).file != NULL)
    GetBitmapFile (icon);
  /* Next, check for a color pixmap */
  if (((*icon).file != NULL) &&
      ((*icon).w == 0) && ((*icon).h == 0))
    GetImageFile (icon);
#endif
}

/****************************************************************************
 *
 * Creates an Icon Window
 *
 ****************************************************************************/
void
CreateButtonIconWindow (button_info * button, Window * win)
{
#ifndef NO_ICONS
  unsigned long valuemask;	/* mask for create windows */
  XSetWindowAttributes attributes;	/* attributes for create windows */

  /* This used to make buttons without icons explode when pushed
     if(((*button).icon_w == 0)&&((*button).icon_h == 0))
     return;
   */
  attributes.background_pixel = back_pix;
  attributes.event_mask = ExposureMask;
  valuemask = CWEventMask | CWBackPixel;

  /* make sure the window geometry does not match the button, so 
   * place_buttons() is forced to configure it */
  (*button).IconWin =
    XCreateWindow (dpy, *win, 0, 0, 1, 1, 0, CopyFromParent,
		   CopyFromParent, CopyFromParent, valuemask, &attributes);

  return;
#endif
}

/****************************************************************************
 *
 * Combines icon shape masks after a resize
 *
 ****************************************************************************/
void
ConfigureIconWindow (button_info * button)
{
#ifndef NO_ICONS
  int i;
  GC ShapeGC;
/*#ifdef XPM */
  int w, h;
  int xoff, yoff;
  /*#endif *//* XPM */

  if ((*button).completeIcon.icon != None)
    XFreePixmap (dpy, (*button).completeIcon.icon);
  (*button).completeIcon.icon = XCreatePixmap (dpy, Scr.Root, (*button).width,
					     (*button).height, Scr.d_depth);

  /* if the background does not fill the whole button, shape */
  if (back_pixmap.mask != None)
    (*button).is_shaped = True;
  /* if an icon fills the whole button, don't shape */
  for (i = 0; i < (*button).num_icons; i++)
    if (((*button).icons[i].mask == None) &&
	((*button).icons[i].w >= (*button).width) &&
	((*button).icons[i].h >= (*button).height))
      (*button).is_shaped = False;

  /* create and clear the mask */
  if ((*button).completeIcon.mask != None)
    XFreePixmap (dpy, (*button).completeIcon.mask);
  (*button).completeIcon.mask = XCreatePixmap (dpy, Scr.Root, (*button).width,
					       (*button).height, 1);
  ShapeGC = XCreateGC (dpy, (*button).completeIcon.mask, 0, NULL);
  if ((*button).is_shaped == True)
    XSetForeground (dpy, ShapeGC, 0);
  else
    XSetForeground (dpy, ShapeGC, 1);
  XFillRectangle (dpy, (*button).completeIcon.mask, ShapeGC,
		  0, 0, (*button).width, (*button).height);

  /* tile with the background pixmap */
  if (back_pixmap.icon != None)
    {
      XSetFillStyle (dpy, Scr.NormalGC, FillTiled);
      XSetTile (dpy, Scr.NormalGC, back_pixmap.icon);
      XFillRectangle (dpy, (*button).completeIcon.icon, Scr.NormalGC,
		      0, 0, (*button).width, (*button).height);
      XSetFillStyle (dpy, Scr.NormalGC, FillSolid);

      if (back_pixmap.mask != None)
	{
	  XSetFillStyle (dpy, ShapeGC, FillTiled);
	  XSetTile (dpy, ShapeGC, back_pixmap.mask);
	  XFillRectangle (dpy, (*button).completeIcon.mask, ShapeGC,
			  0, 0, (*button).width, (*button).height);
	  XSetFillStyle (dpy, ShapeGC, FillSolid);
	}
    }

/*#ifdef XPM */
  /* overlay the icons */
  XSetFunction (dpy, ShapeGC, GXor);
  for (i = 0; i < (*button).num_icons; i++)
    {
      w = (*button).icons[i].w;
      h = (*button).icons[i].h;
      if (w < 1 || h < 1)
	continue;
      if (w > (*button).width)
	w = (*button).width;
      if (h > (*button).height)
	h = (*button).height;
      if (w < 1)
	w = 1;
      if (h < 1)
	h = 1;
      xoff = ((*button).width - w) / 2;
      yoff = ((*button).height - h) / 2;
      if (xoff < 0)
	xoff = 0;
      if (yoff < 0)
	yoff = 0;
      if ((*button).icons[i].mask != None)
	{
	  XSetClipOrigin (dpy, MaskGC, xoff, yoff);
	  XSetClipMask (dpy, MaskGC, (*button).icons[i].mask);
	}
      else
	{
	  XRectangle rect[1];
	  rect[0].x = 0;
	  rect[0].y = 0;
	  rect[0].width = w;
	  rect[0].height = h;

	  XSetClipRectangles (dpy, MaskGC, xoff, yoff, rect, 1, YSorted);
	}
      XCopyArea (dpy, (*button).icons[i].icon,
		 (*button).completeIcon.icon, MaskGC, 0, 0,
		 w, h, xoff, yoff);
      if ((*button).icons[i].mask != None)
	{
	  XCopyArea (dpy, (*button).icons[i].mask,
		     (*button).completeIcon.mask, ShapeGC, 0, 0,
		     w, h, xoff, yoff);
	}
      else
	{
	  XSetForeground (dpy, ShapeGC, 1);
	  XFillRectangle (dpy, (*button).completeIcon.mask, ShapeGC,
			  xoff, yoff, w, h);
	}

      if ((*button).icons[i].depth == -1)
	{
	  XCopyPlane (dpy, (*button).icons[i].icon,
		      (*button).completeIcon.icon, Scr.NormalGC,
		      0, 0, w, h, xoff, yoff, 1);
	  XFreePixmap (dpy, (*button).icons[i].icon);
	  (*button).icons[i].icon = None;
	}
    }
  /*#endif *//* XPM */
  XFreeGC (dpy, ShapeGC);
  XSetWindowBackgroundPixmap (dpy, (*button).IconWin,
			      (*button).completeIcon.icon);
  XClearWindow (dpy, (*button).IconWin);
#endif
}

/***************************************************************************
 *
 * Looks for a monochrome icon bitmap file
 *
 **************************************************************************/
void
GetBitmapFile (icon_info * icon)
{
#ifndef NO_ICONS
  char *path = NULL;
  int HotX, HotY;

  path = findIconFile ((*icon).file, iconPath, R_OK);
  if (path == NULL)
    return;

  if (XReadBitmapFile (dpy, Scr.Root, path, (unsigned int *) &(*icon).w,
		       (unsigned int *) &(*icon).h,
		       &(*icon).icon,
		       (int *) &HotX,
		       (int *) &HotY) != BitmapSuccess)
    {
      (*icon).w = 0;
      (*icon).h = 0;
    }
  else
    {
      (*icon).depth = -1;
    }
  (*icon).mask = None;
  free (path);
#endif
}


/****************************************************************************
 *
 * Looks for a color icon file
 *
 ****************************************************************************/
int
GetImageFile (icon_info * icon)
{
#ifndef NO_ICONS
  char *path = NULL;
  unsigned int dum;
  int dummy;
  int width, height;
  Window root;

  if ((path = findIconFile ((*icon).file, pixmapPath, R_OK)) == NULL)
    return 0;

  icon->icon = LoadImageWithMask (dpy, Scr.Root, 256, path, &(icon->mask));
  free (path);

  if (icon->icon == 0)
    return 0;

  XGetGeometry (dpy, icon->icon, &root, &dummy, &dummy,
		&width, &height, &dum, &dum);
  icon->w = width;
  icon->h = height;
  icon->depth = Scr.d_depth;

  if (icon == &back_pixmap)
    {
      if (ForceSize && (icon->w > 64 || icon->h > 64))
	{
	  Pixmap resized;

	  resized = XCreatePixmap (dpy, Scr.Root, 64, 64, Scr.d_depth);
	  XCopyArea (dpy, icon->icon, resized, Scr.NormalGC, 0, 0, 64, 64, 0, 0);
	  XFreePixmap (dpy, icon->icon);
	  icon->icon = resized;
	  icon->w = 64;
	  icon->h = 64;
	}
      DrawOutline ((*icon).icon, (*icon).w, (*icon).h);
    }
  return 1;
#endif
}

/****************************************************************************
 *
 * read background icons from data
 *
 ****************************************************************************/
int
GetXPMData (icon_info * icon, char **data)
{
#ifndef NO_ICONS
#ifdef XPM
  XWindowAttributes root_attr;
  XpmAttributes xpm_attributes;

  if (icon != &back_pixmap)
    return 0;
  XGetWindowAttributes (dpy, Scr.Root, &root_attr);
  xpm_attributes.colormap = root_attr.colormap;
  xpm_attributes.valuemask = XpmSize | XpmReturnPixels | XpmColormap;
  if (XpmCreatePixmapFromData (dpy, Scr.Root, data,
			       &(*icon).icon,
			       &(*icon).mask,
			       &xpm_attributes) == XpmSuccess)
    {
      (*icon).w = xpm_attributes.width;
      (*icon).h = xpm_attributes.height;
      (*icon).depth = Scr.d_depth;
    }
  else
    {
      return 0;
    }
  DrawOutline ((*icon).icon, (*icon).w, (*icon).h);

  return 1;
#else /* !XPM */
  (*icon).icon = None;
  return 0;
#endif /* !XPM */
#endif
}

/*******************************************************************
 * 
 * Make a gradient pixmap
 * 
 *******************************************************************/

int
GetXPMGradient (icon_info * icon, int from[3], int to[3], int maxcols,
		int type)
{
  (*icon).icon = XCreatePixmap (dpy, Scr.Root, 64, 64, Scr.d_depth);
  (*icon).mask = None;
  (*icon).w = 64;
  (*icon).h = 64;
  (*icon).depth = Scr.d_depth;
  switch (type)
    {
    case TEXTURE_GRADIENT:
      if (!DrawDegradeRelief (dpy, (*icon).icon, 0, 0, 64, 64,
			      from, to, 0, maxcols))
	{
	  XFreePixmap (dpy, (*icon).icon);
	  (*icon).icon = None;
	  return 0;
	}
      break;
    case TEXTURE_HGRADIENT:
    case TEXTURE_HCGRADIENT:
      if (!DrawHGradient (dpy, (*icon).icon, 0, 0, 64, 64,
			  from, to, 0, maxcols, type - TEXTURE_HGRADIENT))
	{
	  XFreePixmap (dpy, (*icon).icon);
	  (*icon).icon = None;
	  return 0;
	}
      break;
    case TEXTURE_VGRADIENT:
    case TEXTURE_VCGRADIENT:
      if (!DrawVGradient (dpy, (*icon).icon, 0, 0, 64, 64,
			  from, to, 0, maxcols, type - TEXTURE_VGRADIENT))
	{
	  XFreePixmap (dpy, (*icon).icon);
	  (*icon).icon = None;
	  return 0;
	}
      break;
    default:
      return 0;
    }
  DrawOutline ((*icon).icon, 64, 64);

  return 1;
}

/*******************************************************************
 * 
 * Make a solid color pixmap
 * 
 *******************************************************************/

int
GetSolidXPM (icon_info * icon, Pixel pixel)
{
  GC gc;
  XGCValues gcv;

  gcv.foreground = pixel;
  gc = XCreateGC (dpy, Scr.Root, GCForeground, &gcv);
  (*icon).icon = XCreatePixmap (dpy, Scr.Root, 64, 64, Scr.d_depth);
  (*icon).mask = None;
  XFillRectangle (dpy, (*icon).icon, gc, 0, 0, 64, 64);
  (*icon).w = 64;
  (*icon).h = 64;
  (*icon).depth = Scr.d_depth;
  XFreeGC (dpy, gc);
  DrawOutline ((*icon).icon, 64, 64);

  return 1;
}
