/* 
   sitecopy, for managing remote web sites. WebDAV client routines.
   Copyright (C) 1998-99, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: httpdav.c,v 1.46.2.18 1999/08/27 10:03:49 joe Exp $
*/

/* This file is a collection of routines to implement a basic WebDAV
 * client, including an HTTP/1.1 client. 
 * Transparently supports basic and digest authentication.
 */

/* HTTP method implementation:
 *   Call, in this order:
 *     1.  http_request_init()  - set up the request
 *     2.  http_request()       - make the request
 *     3.  http_request_end()   - clean up the request
 */

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#ifdef __EMX__
#include <sys/select.h>
#endif

#include <netinet/in.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifndef HAVE_SNPRINTF
#include <snprintf.h>
#endif

#include <dates.h>
#include <basename.h>
#include <dirname.h>
#include <strsplit.h>

#include "frontend.h"
#include "protocol.h"
#include "httpauth.h"
#include "httpdav.h"
#include "common.h"
#include "socket.h"
#include "base64.h"

/* Connection information... */
int http_sock;
bool http_connected;

bool http_disable_expect = false;

/* Time in seconds to wait for the server to give us a 100 Continue
 * after submitting a PUT with an "Expect: 100-continue" header.
 */
#define HTTP_EXPECT_TIMEOUT 15
/* 100-continue only used if size > HTTP_EXPECT_MINSIZ */
#define HTTP_EXPECT_MINSIZE 512

/* Whether the current server respects the Expect: 100-continue header */
int http_expect_works; /* == 0 if it we don't know, 1 if it does,
			    -1 if it doesn't support 100-continue */

bool http_webdav_server = false;
bool http_init_checks = true;
bool http_conn_limit = false;

/* Warning given out on spoof attack */
const char *http_warn_spoof = 
  "The server has switched to using basic authentication from digest\n"
  "authenticaion, which may be an attempt to discover your password.\n"
  "Basic auth will NOT be used.";

bool http_can_authenticate;

http_auth_session_t http_server_auth, http_proxy_auth;

unsigned int http_version_major, http_version_minor;

const char *http_quotes = "\"'";
const char *http_whitespace = " \r\n\t";

#ifdef HAVE_LIBEXPAT
/* WebDAV Fetch mode */

#include <xmlparse.h>

typedef enum {
    dav_xml_multistatus = 0,
    dav_xml_response,
    dav_xml_responsedescription,
    dav_xml_href,
    dav_xml_propstat,
    dav_xml_prop,
    dav_xml_status,
    dav_xml_getlastmodified,
    dav_xml_getcontentlength,
    dav_xml_resourcetype,
    dav_xml_collection,
    dav_xml_unknown,
    dav_xml_root
} dav_xml_tag_t;

/* We get the tag_t from the names array below */
const char *dav_xml_tagnames[] = {
    "DAV:multistatus",
    "DAV:response",
    "DAV:responsedescription",
    "DAV:href",
    "DAV:propstat",
    "DAV:prop",
    "DAV:status",
    "DAV:getlastmodified",
    "DAV:getcontentlength",
    "DAV:resourcetype",
    "DAV:collection",
    NULL, /* end-of-list marker */
    "@<root>@" /* filler, so we can index this by dav_xml_tag_t */
};

typedef struct dav_xml_ns_s dav_xml_ns;

/* Linked list of namespace scopes */
struct dav_xml_ns_s {
    char *name;
    char *value;
    dav_xml_ns *next;
};

struct dav_xml_state {
    dav_xml_tag_t tag;
    char *tag_name; /* The full tag name */
    char *default_ns; /* The current default namespace */
    dav_xml_ns *nspaces; /* List of other namespace scopes */
    struct dav_xml_state *parent; /* The parent in the tree */
};

/* We pass around a dav_xmldoc as the userdata using expat.  This
 * maintains the current state of the parse and various other bits and
 * bobs. Within the parse, we store the current branch of the tree,
 * i.e., the current element and all its parents, but nothing other
 * than that. 
 *
 * The files list is filled as we go (by dav_fetch_gotresource), but
 * always kept in sorted order, since no ordering of resources in the
 * PROPFIND response is given. 
 */
struct dav_xmldoc {
    /* Points to the root of the document */
    struct dav_xml_state *root;
    /* Points to the current element in the document */
    struct dav_xml_state *current;
    /* Whether we want to collect CDATA at the moment or not */
    bool want_cdata;
    /* The cdata we've collected so far - grows as we collect
     * more. */
    char *cdata;
    /* How big the buffer is atm */
    size_t cdata_buflen;
    /* How much cdata we've collected so far */
    size_t cdata_len; 
    /* Is it valid? */
    bool valid;
    /* Temporary store of file info */
    struct proto_file_t *file;
    /* The complete fetch list */
    struct proto_file_t *files;
    /* The collection we did a PROPFIND on */
    const char *fetch_root;
};

/* The initial size of the cdata buffer, and the minimum size by which
 * it grows each time we overflow it. For the PROPFIND requests
 * we do for sitecopy, the only cdata segments we collect are very small.
 */
#define CDATABUFSIZ 128
/* If the cdata buffer size is larger than this when we've finished
 * with its contents, then we reallocate it. Otherwise, we just
 * zero it out. Reallocation -> a free() and a malloc(). No realloc -> 
 * just a memset().
 */
#define CDATASHRINK 128

/* Prototypes */
static void dav_xml_startelm( void *userdata, const char *tag, const char **atts );
static void dav_xml_endelm( void *userdata, const char *tag );
static void dav_xml_cdata( void *userdata, const char *cdata, int len );
static bool dav_xml_parsetag( struct dav_xml_state *state,
		       const char *tag, const char **atts );

static int dav_fetch_getdepth( const char *href );
static bool dav_fetch_parse_href( struct dav_xmldoc *doc );
static void dav_fetch_gotresource( struct dav_xmldoc *doc );

static void dav_xml_parsebody( void *userdata, const char *buffer, const size_t len );
static void http_get_content_charset( const char *name, const char *value );

char *http_content_charset;

#endif /* HAVE_LIBEXPAT */

/* Handy macro to free things. */
#define DOFREE(x) if( x!=NULL ) free( x )

#define EOL "\r\n"
#define HTTP_PORT 80

const char *http_useragent = PACKAGE "/" VERSION;

/* We need to remember the remote host and port even after connecting
 * the socket e.g. so we can form the Destination: header */
struct proto_host_t http_server_host, http_proxy_host;

/* This, to store the address of the server we CONNECT to - i.e.,
 * the proxy if we have one, else the server */
struct in_addr http_remoteaddr;
int http_remoteport;

bool http_use_proxy; /* whether we are using the proxy or not */

int http_mkdir_works;

static int http_response_read( http_req_t *req, char *buffer, size_t buflen );

/* Sets up the body size for the given request */
static int http_req_bodysize( http_req_t *req );

/* The callback for GET requests (to allow signalling progress to the FE) */
static void http_get_callback( void *user, const char *buffer, const size_t len );

/* Do a dummy MKDIR with PUT */
static int http_mkdir_with_put( const char *realdir );

/* Holds the error message to be passed up */
char http_error[BUFSIZ];

/* Put the fixed headers into the request */
static void http_req_fixedheaders( http_req_t *req );

/* Concatenates the remote server name with :port if port!=80 on to 
 * the end of str. */
static void http_strcat_hostname( struct proto_host_t *host, char *str );

/* Header parser for OPTIONS requests. */
static void http_options_parsehdr( const char *name, const char *value );

static int http_parse_status( http_req_t *req, char *status_line );

/* Sends the body down the wire */
static int http_req_sendbody( http_req_t *req );

/* Encodes the absPath sectionf of a URI using %<hex><hex> encoding */
static char *uri_abspath_encode( const char *abs_path );

/* Decodes a URI */
static char *uri_decode( const char *uri );

/* Opens the connection to the remote server.
 * Returns:
 *  PROTO_OK       on success
 *  PROTO_CONNECT  if the socket couldn't be connected
 */
static int http_open( void );

/* Closes the connection.
 * Always returns PROTO_OK
 */
static int http_close( void );

/* This doesn't really connect to the server.
 * Returns
 *  PROTO_LOOKUP if the hostname of the server could not be resolved
 *  PROTO_LOCALHOST if the hostname of the local machine could not be
 *    found
 *  PROTO_CONNECT if the socket could not be connected
 *  PROTO_OK if the connection was made successfully.
 */
int http_init( const char *remote_root,
	       struct proto_host_t *server, struct proto_host_t *proxy ) {
    int ret;
    /* Take a copy of the server information */
    memcpy( &http_server_host, server, sizeof(struct proto_host_t) );
    fe_connection( fe_namelookup );
    if( proxy != NULL ) {
	memcpy( &http_proxy_host, proxy, sizeof(struct proto_host_t) );
	http_remoteport = proxy->port;
	http_use_proxy = true;
	if( host_lookup( http_proxy_host.hostname, &http_remoteaddr ) )
	    return PROTO_LOOKUP;
    } else {
	http_remoteport = server->port;
	http_use_proxy = false;
	if( host_lookup( http_server_host.hostname, &http_remoteaddr ) )
	    return PROTO_LOOKUP;
    }
    http_connected = false;
    http_expect_works = http_disable_expect?-1:0; /* we don't know yet */
    /* temporary workaround */
    http_expect_works = -1;
    http_mkdir_works = 0; /* we don't know yet */
    http_auth_init( &http_server_auth, server->username, server->password );
    if( http_use_proxy ) {
	/* TODO: Implement properly */
/*	http_auth_init( &http_proxy_auth, proxy->username, proxy->password );
 */
    }
    http_can_authenticate = false;
    ret = http_open();
    /* Drop out if that failed, or they don't want the OPTIONS */
    if( (!http_init_checks) || (ret != PROTO_OK) )
	return ret;
    /* Capability discovery... we don't care whether this
     * actually works or not, we just want http_webdav_server
     * set appropriately. */
    (void) http_options( remote_root );
    return PROTO_OK;
}

int http_finish( void ) {
    DEBUG( DEBUG_HTTP, "http_finish called.\n" );
    http_auth_finish( &http_server_auth );
    if( http_connected ) http_close( );
    return PROTO_OK;
}

/* Parse the HTTP-Version and Status-Code segments of the
 * given status_line. Sets http_version_* and req->class,status.
 * Returns: PROTO_OK on success, PROTO_ERROR otherwise */
int http_parse_status( http_req_t *req, char *status_line ) {
    char *part;
    DEBUG( DEBUG_HTTP, "HTTP response line: %s", status_line );
    /* Save the line for error codes later */
    memset( http_error, 0, BUFSIZ );
    strncpy( http_error, status_line, BUFSIZ );
    /* Strip off the CRLF for the status line */
    if( (part = strchr( http_error, '\r' )) != NULL )
	*part = '\0';
    /* Check they're speaking the right language */
    if( strncmp( status_line, "HTTP/", 5 ) != 0 )
	return PROTO_ERROR;
    /* And find out which dialect of this peculiar language
     * they can talk... */
    http_version_major = 0;
    http_version_minor = 0; 
    /* Note, we're good children, and accept leading zero's on the
     * version numbers */
    for( part = status_line + 5; *part != '\0' && isdigit(*part); part++ ) {
	http_version_major += http_version_major*10 + (*part-'0');
    }
    if( *part != '.' ) return PROTO_ERROR;
    for( part++ ; *part != '\0' && isdigit(*part); part++ ) {
	http_version_minor += http_version_minor*10 + (*part-'0');
    }
    DEBUG( DEBUG_HTTP, "HTTP Version Major: %d, Minor: %d\n", 
	   http_version_major, http_version_minor );
    if( *part != ' ' ) return PROTO_ERROR;
    /* Now for the Status-Code. part now points at the space
     * between "HTTP/x.x YYY". We want YYY... could use atoi, but
     * probably quicker this way. */
    req->status = 100*(part[1]-'0') + 10*(part[2]-'0') + (part[3]-'0');
    req->class = part[1]-'0';
    /* And we can ignore the Reason-Phrase */
    DEBUG( DEBUG_HTTP, "HTTP status code: %d\n", req->status );
    return PROTO_OK;
}

/* Sends the body down the socket.
 * Returns PROTO_OK on success, PROTO_ERROR otherwise */
int http_req_sendbody( http_req_t *req ) {
    int ret;
    switch( req->body ) {
    case http_body_file:
	ret = transfer( fileno(req->body_file), http_sock, req->body_size );
	DEBUG( DEBUG_HTTP, "Sent %d bytes.\n", ret );
	rewind( req->body_file ); /* since we may have to send it again */
	break;
    case http_body_buffer:
	DEBUG( DEBUG_HTTP, "Sending body:\n%s\n", req->body_buffer );
	ret = send_string( http_sock, req->body_buffer );
	break;
    default:
	DEBUG( DEBUG_HTTP, "Argh in http_req_sendbody!" );
	ret = -1;
    }
    if( ret == -1 ) { 
	/* transfer failed */
	return PROTO_ERROR;
    } else {
	return PROTO_OK;
    }
}

/* Deal with the body size */
int http_req_bodysize( http_req_t *req ) {
    struct stat bodyst;
    /* Do extra stuff if we have a body */
    switch( req->body ) {
    case http_body_file:
	/* Get file length */
	if( fstat( fileno(req->body_file), &bodyst ) < 0 ) {
	    /* Stat failed */
	    DEBUG( DEBUG_HTTP, "Stat failed: %s\n", strerror( errno ) );
	    return PROTO_ERROR;
	}
	req->body_size = bodyst.st_size;
	break;
    case http_body_buffer:
	req->body_size = strlen( req->body_buffer );
	break;
    default:
	/* No body, so no size. */
	return PROTO_OK;
    }
    if( req->body != http_body_none ) {
	char tmp[BUFSIZ];
	/* Add the body length header */
	snprintf( tmp, BUFSIZ, "Content-Length: %d" EOL, req->body_size );
	strcat( req->headers, tmp );
    }
    return PROTO_OK;
}

void http_strcat_hostname( struct proto_host_t *host, char *str ) {
    strcat( str, host->hostname );
    /* Only add the port if it isn't 80 */
    if( host->port != HTTP_PORT ) {
	static char buffer[128];
	snprintf( buffer, 128, ":%d", host->port );
	strcat( str, buffer );
    }
}

/* Lob the User-Agent, connection and host headers in to the request
 * headers */
void http_req_fixedheaders( http_req_t *req ) {
    strcat( req->headers, "User-Agent: " );
    strcat( req->headers, http_useragent );
    strcat( req->headers, EOL 
	    "Connection: Keep-Alive" EOL 
	    "Host: " );
    http_strcat_hostname( &http_server_host, req->headers );
    strcat( req->headers, EOL );
}

/* Decodes a URI */
char *uri_decode( const char *uri ) {
    const char *pnt;
    char *ret, *retpos, buf[5] = { "0x00\0" };
    retpos = ret = malloc( strlen( uri ) + 1 );
    for( pnt = uri; *pnt != '\0'; pnt++ ) {
	if( *pnt == '%' ) {
	    if( !isxdigit((unsigned char) pnt[1]) || 
		!isxdigit((unsigned char) pnt[2]) ) {
		/* Invalid URI */
		return NULL;
	    }
	    buf[2] = *++pnt; buf[3] = *++pnt; /* bit faster than memcpy */
	    *retpos++ = strtol( buf, NULL, 16 );
	} else {
	    *retpos++ = *pnt;
	}
    }
    *retpos = '\0';
    return ret;
}

/* RFC2396 spake:
 * "Data must be escaped if it does not have a representation 
 * using an unreserved character".
 * ...where...
 *  unreserved  = alphanum | mark
 *  mark = "-" | "_" | "." | "!" | "~" | "*" | "'" | "(" | ")"
 * 
 * We need also to skip reserved characters
 * reserved    = ";" | "/" | "?" | ":" | "@" | "&" |
 *               "=" | "+" | "$" | ","
 */

/* Lookup table:
 * 1 marks an RESERVED character. 2 marks a UNRESERVED character.
 * 0 marks everything else. 
 */

#define RE 1
#define UN 2 
const short uri_chars[128] = {
/* 0 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 16 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 32 */  0, UN, 0, 0, RE, 0, RE, UN, UN, UN, UN, RE, RE, UN, UN, RE,
/* 48 */ UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, RE, RE, 0, RE, 0, RE,
/* 64 */ RE, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,
/* 80 */ UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, 0, 0, 0, 0, UN,
/* 96 */ 0, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,
/* 112 */ UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, 0, 0, 0, UN, 0 
};
#undef RE
#undef UN

/* Encodes the abspath segment of a URI.
 * TODO: Make this parse a complete URI */
char *uri_abspath_encode( const char *abs_path ) {
    const char *pnt;
    char *ret, *retpos;
    /* Rather than mess about growing the buffer, allocate as much as
     * the URI could possibly need, i.e. every character gets %XX
     * escaped. Hence 3 times input size. 
     */
    retpos = ret = malloc( strlen( abs_path ) * 3 + 1 );
    for( pnt = abs_path; *pnt != '\0'; pnt++ ) {
	/* Escape it:
	 *  - if it isn't 7-bit
	 *  - if it is a reserved character (but ignore '/')
	 *  - otherwise, if it is not an unreserved character
	 * (note, there are many characters that are neither reserved
	 * nor unreserved)
	 */
	if( *pnt<0 || (uri_chars[(int) *pnt] < 2 && *pnt!='/' )) {
	    /* Escape it - %<hex><hex> */
	    sprintf( retpos, "%%%02x", (unsigned char) *pnt );
	    retpos += 3;
	} else {
	    /* It's cool */
	    *retpos++ = *pnt;
	}
    }
    *retpos = '\0';
    return ret;
}

#ifdef URITEST

void fe_transfer_progress( size_t progress, size_t total ) {}

int main( int argc, char *argv[] ) {
    char *tmp;
    if( argc!=2 ) {
	printf( "doh. usage:\nuritest a_uri_abspath_segment\n"
		"e.g. uritest \"/this/is/a silly<filename>/but/hey\"\n" );
	exit(-1);
    }
    printf( "Input URI: %s\n", argv[1] );
    tmp = uri_abspath_encode( argv[1] );
    printf( "Encoded: %s\n", tmp );
    printf( "Decoded: %s\n", uri_decode( tmp ) );
    return 0;
}

#endif /* URITEST */

/* Initializes the request with given method and URI.
 * URI must be abs_path - i.e., NO scheme+hostname. It will BREAK 
 * otherwise. */
void http_request_init( http_req_t *req, 
			const char *method, const char *uri ) {
    /* Clear it out */
    memset( req, 0, sizeof( http_req_t ) );

    DEBUG( DEBUG_HTTP, "Request starts.\n" );

    /* Add in the fixed headers */
    http_req_fixedheaders( req );

    /* Set the standard stuff */
    req->method = method;
    req->uri = uri_abspath_encode( uri );
    
    req->body_callback = NULL;
    req->body = http_body_none;
    
}

void http_request_end( http_req_t *req ) {
    if( req->uri != NULL ) {
	free( req->uri );
    }
    DEBUG( DEBUG_HTTP, "Request ends.\n" );
}

/* Reads a block of the response into buffer, which is of size buflen.
 * Returns number of bytes read, 0 on end-of-response, or -1 on error.
 */
int http_response_read( http_req_t *req, char *buffer, size_t buflen ) {
    int willread, readlen;
    if( req->resp_te==http_te_chunked ) {
	/* We are doing a chunked transfer-encoding.
	 * It goes:  `SIZE CRLF CHUNK CRLF SIZE CRLF CHUNK CRLF ...'
	 * ended by a `CHUNK CRLF 0 CRLF', a 0-sized chunk.
	 * The slight complication is that we have to cope with
	 * partial reads of chunks.
	 * For this reason, resp_chunk_left contains the number of
	 * bytes left to read in the current chunk.
	 */
	if( req->resp_chunk_left == 0 ) {
	    /* We are at the start of a new chunk. */
	    DEBUG( DEBUG_HTTP, "New chunk.\n" );
	    if( read_line( http_sock, buffer, buflen ) < 0 ) {
		DEBUG( DEBUG_HTTP, "Could not read chunk size.\n" );
		return -1;
	    }
	    DEBUG( DEBUG_HTTP, "[Chunk Size] < %s", buffer );
	    if( sscanf( buffer, "%x", &req->resp_chunk_left ) != 1 ) {
		DEBUG( DEBUG_HTTP, "Couldn't read chunk size.\n" );
		return -1;
	    }
	    DEBUG( DEBUG_HTTP, "Got chunk size: %d\n", req->resp_chunk_left );
	    if( req->resp_chunk_left == 0 ) {
		/* Zero-size chunk */
		DEBUG( DEBUG_HTTP, "Zero-size chunk.\n" );
		return 0;
	    }
	}
	willread = min( buflen - 1, req->resp_chunk_left );
    } else if( req->resp_length > 0 ) {
	/* Have we finished reading the body? */
	if( req->resp_left == 0 )
	    return 0;
	willread = min( buflen - 1, req->resp_left );
    } else {
	/* Read until socket-close */
	willread = buflen - 1;
    }
    DEBUG( DEBUG_HTTP, "Reading %d bytes of response body.\n", willread );
    readlen = sock_read( http_sock, buffer, willread );
    DEBUG( DEBUG_HTTP, "Got %d bytes.\n", readlen );
    if( readlen < 0 ) {
	/* It broke */
	DEBUG( DEBUG_HTTP, "Could not read block.\n" );
	return -1;
    } else if( (readlen == 0) && 
	       ( (req->resp_length > 0) ||
		 (req->resp_te==http_te_chunked) )) {
	/* Premature close before read all of body, or during chunk read. */
	DEBUG( DEBUG_HTTP, "Socket closed before end of body.\n" );
	return -1;
    }
    buffer[readlen] = '\0';
    DEBUG( DEBUG_HTTP, "Read block:\n%s\n", buffer );
    if( req->resp_te==http_te_chunked ) {
	req->resp_chunk_left -= readlen;
	if( req->resp_chunk_left == 0 ) {
	    char crlfbuf[2];
	    /* If we've read a whole chunk, read a CRLF */
	    if( read_data( http_sock, crlfbuf, 2 ) == 0 ) {
		DEBUG( DEBUG_HTTP, "Read CRLF bytes.\n" );
		if( strncmp( crlfbuf, EOL, 2 ) != 0 ) {
		    DEBUG( DEBUG_HTTP, "CRLF bytes didn't contain CRLF!\n" );
		    return -1;
		}
	    } else {
		return -1;
	    }
	}		
    } else if( req->resp_length > 0 ) {
	req->resp_left -= readlen;
    }
    return readlen;
}

/* The HTTP/1.x request/response mechanism 
 *
 * Returns:
 *   PROTO_OK if the request was made (not related to status code)
 *   PROTO_ERROR if the request could not be made
 * The STATUS CODE is placed in req->status. The error string is
 * placed in http_error.
 * 
 * TODO: This should be chopped up into smaller chunks, and get rid of
 * the horrid goto's.  
 */
int http_request( http_req_t *req ) {
    char buffer[BUFSIZ]; /* For reading from the socket */
    int ret, attempt, con_attempt;
    bool using_expect, /* whether we have sent a Expect: 100 header */
	close_connection,
	dead_connection,
	wants_body; /* whether the caller wants the response body
		     * callbacks */

#ifdef USE_BROKEN_PROPFIND
    bool is_propfind = (strcmp( req->method, "PROPFIND" ) == 0);
#endif

#define HTTP_FATAL_ERROR(a) {				\
	DEBUG( DEBUG_HTTP, a );					\
	ret = PROTO_ERROR;					\
	close_connection = true;				\
	goto http_request_finish_proc;				\
    }

    /* Initialization... */
    DEBUG( DEBUG_HTTP, "Request started...\n" );
    strcpy( http_error, "Unknown error." );
    ret = PROTO_OK;

    if( http_req_bodysize( req ) != PROTO_OK )
	return PROTO_ERROR;

    /* I shall try this only twice...
     * First time, with default authentication stuff (either, what we
     * did last time, or none at all), then with up-to-the-minute
     * what-the-server-requested authentication stuff. */

    attempt = con_attempt = 1;
    http_auth_new_request( &http_server_auth, req->method, req->uri,
			   req->body_buffer, req->body_file );

    do {
	char request[REQSIZ], *authinfo = NULL;

	/* Add in the Request-Line */
	strcpy( request, req->method );
	if( http_use_proxy ) {
	    strcat( request, " http://" );
	    http_strcat_hostname( &http_server_host, request );
	} else {
	    strcat( request, " " );
	}
	strcat( request, req->uri );
	strcat( request, " HTTP/1.1" EOL );

	/* The caller-supplied headers */
	strcat( request, req->headers );

	if( http_can_authenticate ) {
	    /* Add the authorization headers in */
	    char *val = http_auth_request( &http_server_auth );
	    if( val != NULL ) {
		strcat( request, "Authorization: " );
		strcat( request, val );
		free( val );
	    } else {
		DEBUG( DEBUG_HTTP, "auth_request returned NULL.\n" );
	    }
	}

	/* Now handle the body. */
	using_expect = false;
	if( req->body!=http_body_none ) {
	    if( (http_expect_works > -1) &&
		(req->body_size > HTTP_EXPECT_MINSIZE) 
#ifdef USE_BROKEN_PROPFIND
		/* ... definitely NOT if we're doing PROPFIND */
		&& (!is_propfind)
#endif /* USE_BROKEN_PROPFIND */
		) {
		/* Add Expect: 100-continue. */
		strcat( request, "Expect: 100-continue" EOL );
		using_expect = true;
	    }
	}

	/* Final CRLF */
	strcat( request, EOL );
	
	/* Now send the request */

	/* Open the connection if necessary */
	if( !http_connected ) {
	    if( (ret = http_open()) != PROTO_OK )
		return ret;
	}

	dead_connection = false;

#ifdef USE_BROKEN_PROPFIND
	if( !is_propfind ) {
#endif
	/* Send the headers */
#ifdef DEBUGGING
	if( (256&debug_mask) == 256 ) { 
	    /* Display everything mode */
	    DEBUG( DEBUG_HTTP, "Sending request headers:\n%s", request );
	} else {
	    /* Blank out the Authorization paramaters */
	    char reqdebug[REQSIZ], *pnt;
	    strcpy( reqdebug, request );
	    pnt = strstr( reqdebug, "Authorization: " );
	    if( pnt != NULL ) {
		for( pnt += 15; *pnt != '\r' && *pnt != '\0'; pnt++ ) {
		    *pnt = 'x';
		}
	    }
	    DEBUG( DEBUG_HTTP, "Sending request headers:\n%s", reqdebug );
	}
#endif DBEUGGING
	if( send_string( http_sock, request ) < 0 ) {
	    dead_connection = true;
	    HTTP_FATAL_ERROR( "Could not send request!\n" );
	}

	DEBUG( DEBUG_HTTP, "Request sent.\n" );
	
#ifdef USE_BROKEN_PROPFIND
	}
#endif /* USE_BROKEN_PROPFIND */

	/* Now, if we are doing a Expect: 100, hang around for a short
	 * amount of time, to see if the server actually cares about the 
	 * Expect and sends us a 100 Continue response if the request
	 * is valid, else an error code if it's not. This saves sending
	 * big files to the server when they will be rejected.
	 */
	
	if( using_expect ) {
	    DEBUG( DEBUG_HTTP, "Waiting for response...\n" );
	    ret = sock_block( http_sock, HTTP_EXPECT_TIMEOUT );
	    switch( ret ) {
	    case -1: /* error */
		HTTP_FATAL_ERROR( "Wait (select) failed.\n" );
		break;
	    case 1: /* we got us a response! */
		DEBUG( DEBUG_HTTP, "Wait got data.\n" );
		http_expect_works = 1; /* it works - use it again */
		break;
	    case 0: 
		/* Timed out - i.e. Expect: ignored. There is a danger
		 * here that the server DOES respect the Expect: header,
		 * but was going SO slowly that it didn't get time to
		 * respond within HTTP_EXPECT_TIMEOUT.
		 * TODO: while sending the body, check to see if the
		 * server has sent anything back - if it HAS, then
		 * stop sending - this is a spec compliance SHOULD */
		DEBUG( DEBUG_HTTP, "Wait timed out.\n" );
		http_expect_works = -1; /* don't try that again */
		/* and give them the body */
		if( http_req_sendbody( req ) != PROTO_OK )
		    HTTP_FATAL_ERROR( "Could not send body.\n" );
		break;
	    }
	} else if( req->body != http_body_none ) {
#ifdef USE_BROKEN_PROPFIND
	    if( is_propfind ) {
		char *pfbuf;
		DEBUG( DEBUG_HTTP, "Broken PROPFIND handler...\n" );
		pfbuf = malloc( strlen(request) + req->body_size + 1 );
		memset( pfbuf, 0, strlen(request) + req->body_size + 1 );
		strcpy( pfbuf, request );
		strcat( pfbuf, req->body_buffer );
		DEBUG( DEBUG_HTTP, "Sending PROPFIND request...\n" );
		if( send_string( http_sock, pfbuf ) < 0 ) {
		    HTTP_FATAL_ERROR( "Could not send request.\n" );
		}
		DEBUG( DEBUG_HTTP, "Sending newline.\n" );
		if( send_line( http_sock, "" ) < 0 ) {
		    HTTP_FATAL_ERROR( "Could not send newline.\n" );
		}
		free( pfbuf );
	    } else {
		
#endif /* USE_BROKEN_PROPFIND */
	    /* Just chuck the file down the socket */
	    DEBUG( DEBUG_HTTP, "Sending body...\n" );
	    if( http_req_sendbody( req ) == PROTO_ERROR )
		HTTP_FATAL_ERROR( "Could not send body.\n" );
	    /* FIXME: This. Should it be here? */
	    DEBUG( DEBUG_HTTP, "Sending newline.\n" );
	    if( send_line( http_sock, "" ) < 0 ) {
		HTTP_FATAL_ERROR( "Could not send newline.\n" );
	    }
	    DEBUG( DEBUG_HTTP, "Body sent.\n" );
#ifdef USE_BROKEN_PROPFIND
	    }
#endif /* USE_BROKEN_PROPFIND */
	}
	
	/* Now, we have either:
	 *   - Sent the header and body, or
	 *   - Sent the header incl. Expect: line, and got some response.
	 * In any case, we get the status line of the response.
	 */
	
	/* HTTP/1.1 says we MUST be able to accept any number of
	 * 100 (Continue) responses prior to the normal response.
	 * So loop while we get them.
	 */
	
	do {
	    if( read_line( http_sock, buffer, BUFSIZ ) < 0 ) {
		dead_connection = true;
		HTTP_FATAL_ERROR( "Could not read status line.\n" );
	    }
	    
	    DEBUG( DEBUG_HTTP, "[Status Line] < %s", buffer );
	    
	    /* Got the status line - parse it */
	    if( http_parse_status( req, buffer ) == PROTO_ERROR )
		HTTP_FATAL_ERROR( "Could not parse status line.\n" );

	    if( req->class == 1 ) {
		DEBUG( DEBUG_HTTP, "Got 1xx-class.\n" );
		/* Skip any headers, we don't need them */
		do {
		    if( read_line( http_sock, buffer, BUFSIZ ) < 0 )
			HTTP_FATAL_ERROR( "Could not read header.\n" );
		    DEBUG( DEBUG_HTTP, "[Ignored header] < %s", buffer );
		} while( strcmp( buffer, EOL ) != 0 );
	
		if( using_expect && (req->status == 100) ) {
		    /* We are using Expect: 100, and we got a 100-continue 
		     * return code... send the request body */
		    DEBUG( DEBUG_HTTP, "Got continue... sending body now.\n" );
		    if( http_req_sendbody( req ) != PROTO_OK )
			HTTP_FATAL_ERROR( "Could not send body.\n" );
		    DEBUG( DEBUG_HTTP, "Body sent.\n" );
		}
	    }
	} while( req->class == 1 );
	
	/* We've got the real status line... now get the headers */
	
	req->resp_length = -1;
	req->resp_te = http_te_unknown;
	close_connection = false;
	
	/* Now read the rest of the header... up to the next blank line */
	while( read_line( http_sock, buffer, BUFSIZ ) > 0 ) {
	    char extra[BUFSIZ], *pnt;
	    DEBUG( DEBUG_HTTP, "[Header:%d] < %s", strlen(buffer), buffer );
	    if( strcmp( buffer, EOL ) == 0 ) {
		DEBUG( DEBUG_HTTP, "CRLF: End of headers.\n" );
		break;
	    }
	    while(true) {
		/* Collect any extra lines into buffer */
		ret = sock_recv( http_sock, extra, 1, MSG_PEEK);
		if( ret <= 0 ) {
		    HTTP_FATAL_ERROR( "Couldn't peek at next line.\n" );
		}
		if( extra[0] != ' ' && extra[0] != '\t' ) {
		    /* No more headers */
		    break;
		}
		ret = read_line( http_sock, extra, BUFSIZ );
		if( ret == -2 ) {
		    /* No newline within BUFSIZ bytes. This is a 
		     * loooong header. */
		    DEBUG( DEBUG_HTTP, 
			   "Header line longer than buffer, skipped.\n" );
		    break;
		} else if( ret <= 0 ) { 
		    HTTP_FATAL_ERROR( "Couldn't read next header line.\n" );
		} else {
		    DEBUG( DEBUG_HTTP, "[Cont:%d] < %s", strlen(extra), extra);
		}
		/* Append a space to the end of the last header, in
		 * place of the CRLF. */
		pnt = strchr( buffer, '\r' );
		pnt[0] = ' '; pnt[1] = '\0';
		/* Skip leading whitespace off next line */
		for( pnt = extra; *pnt!='\0' && 
			 ( *pnt == ' ' || *pnt =='\t' ); pnt++ ) /*oneliner*/;
		DEBUG( DEBUG_HTTP, "[Continued] < %s", pnt );
		if( strlen(buffer) + strlen(pnt) >= BUFSIZ ) {
		    DEBUG( DEBUG_HTTP, "Exceeded header buffer space.\n" );
		    /* Note, we don't break out of the loop here, cos
		     * we need to collect all the continued lines */
		} else {
		    strcat( buffer, pnt );
		}
	    }
	    /* Now parse the header line. This is all a bit noddy. */
	    pnt = strchr( buffer, ':' );
	    if( pnt != NULL ) {
		char *name, *value, *part;
		/* Null-term name at the : */
		*pnt = '\0';
		name = buffer;
		/* Strip leading whitespace from the value */		
		for( value = pnt+1; *value!='\0' && *value==' '; value++ )
		    /* nullop */;
		if( (part = strchr( value, '\r' )) != NULL ) 
		    *part = '\0';
		if( (part = strchr( value, '\n' )) != NULL ) 
		    *part = '\0';
		DEBUG( DEBUG_HTTP, "Header Name: [%s], Value: [%s]\n",
		       name, value );
		if( strcasecmp( name, "Content-Length" ) == 0 ) {
		    /* TODO: 2068 says we MUST notify the user if this
		     * is not a real number. */
		    req->resp_length = atoi( value );
		} else if( strcasecmp( name, "Transfer-Encoding" ) == 0 ) {
		    if( strcasecmp( value, "chunked" ) == 0 ) {
			req->resp_te = http_te_chunked;
		    } else {
			req->resp_te = http_te_unknown;
		    }
		} else if( strcasecmp( name, "Connection" ) == 0 ) {
		    if( strcasecmp( value, "close" ) == 0 ) {
			close_connection = true;
		    }
		} else if( strcasecmp( name, "WWW-Authenticate" ) == 0 ) {
		    /* Parse the authentication challenge */
		    http_can_authenticate = 
			http_auth_challenge( &http_server_auth, value );
		} else if( strcasecmp( name, "Authentication-Info" ) == 0 ) {
		    /* Remember on the authentication reponse */
		    authinfo = strdup( value );
		} else if( req->hdrs_callback != NULL ) {
		    (*req->hdrs_callback)( name, value );
		}
	    }
	}

	/* Body length calculation, bit icky.
	 * Here, we set:
	 * length==-1 if we DO NOT know the exact body length
	 * length>=0 if we DO know the body length.
	 *
	 * RFC2068, section 4.3: 
	 * NO body is returned if the method is HEAD, or the resp status
	 * is 204 or 304
	 */
	if( (strcmp( req->method, "HEAD" ) == 0 ) ||
	    req->status==204 ||
	    req->status==304 ) {
	    req->resp_length = 0;
	} else {
	    /* RFC2068, section 4.4: if we have a transfer encoding
	     * and a content-length, then ignore the content-length. */
	    if( (req->resp_length>-1) && 
		(req->resp_te!=http_te_unknown) ) {
		req->resp_length = -1;
	    }
	}

	/* The caller only wants the body if this request
	 * was 2xx class. */
	/* FIXME: Let the caller decide when they want it*/
	wants_body = (req->class == 2);

	if( req->resp_length != 0 ) {
	    /* Now, read the body */
	    int readlen;
	    req->resp_left = req->resp_length;
	    req->resp_chunk_left = 0;
	    do {
		/* Read a block */
		readlen = http_response_read( req, buffer, BUFSIZ );
		DEBUG( DEBUG_HTTP, "Read %d bytes.\n", readlen );
		if( readlen > -1 ) {
		    /* What to do with the body block */
		    if( req->body_callback && wants_body ) 
			(*req->body_callback)( req->body_callback_userdata,
					       buffer, readlen );
		    http_auth_response_body( &http_server_auth, 
					     buffer, readlen );
		}
	    } while( readlen > 0 );
	    if( readlen < 0 ) {
		HTTP_FATAL_ERROR( "Block read error.\n" );
	    } else if( (readlen == 0) && (req->resp_te == http_te_chunked) ) {
		char *pnt;
		/* Read till CRLF - skip trailing headers */
		do {
		    if( read_line( http_sock, buffer, BUFSIZ ) < 0 ) {
			/* It broke */
			HTTP_FATAL_ERROR( "Could not read trailer.\n" );
		    }
		    DEBUG( DEBUG_HTTP, "[Chunk trailer] %s", buffer );
		    /* Now parse the header line. */
		    pnt = strchr( buffer, ':' );
		    if( pnt != NULL ) {
			char *name, *value, *part;
			/* Null-term name at the : */
			*pnt = '\0';
			name = buffer;
			/* Strip leading whitespace from the value */ 
			for( value = pnt+1; 
			     *value!='\0' && *value==' '; value++ )
			    /* nullop */;
			if( (part = strchr( value, '\r' )) != NULL ) 
			    *part = '\0';
			if( (part = strchr( value, '\n' )) != NULL ) 
			    *part = '\0';
			DEBUG( DEBUG_HTTP, "Header Name: [%s], Value: [%s]\n",
			       name, value );
			if( strcasecmp( name, "Authentication-Info" ) == 0 ) {
			    /* Save the authinfo value */
			    if( authinfo == NULL ) {
				authinfo = strdup( value );
			    } else {
				DEBUG( DEBUG_HTTP, "Recieved TWO auth-info headers... ignoring second.\n" );
			    }
			}
		    }
		} while( strcmp( buffer, EOL ) != 0 );
	    }
	}

	/* Now, if we are doing authentication, check the
	 * server's credentials. */
	if( http_can_authenticate && (authinfo != NULL) ) {
	    if( http_auth_verify_response( &http_server_auth, authinfo ) ) {
		ret = PROTO_OK;
		DEBUG( DEBUG_HTTP, "Response authenticated okay.\n" );
	    } else {
		DEBUG( DEBUG_HTTP, "Reponse authenticated as invalid.\n" );
		ret = PROTO_ERROR;
		sprintf( http_error,"Server was not authenticated correctly.");
	    }
	} else {
	    ret = PROTO_OK;
	}
	DOFREE( authinfo );
    
http_request_finish_proc:
	/* Now, do we close the connection? */
	if( close_connection ) {
	    DEBUG( DEBUG_HTTP, "Forced connection close.\n" );
	    http_close( );
	} else if( http_conn_limit ) {
	    DEBUG( DEBUG_HTTP, "Limited connection close.\n" );
	    http_close( );
	}	    
    
	/* Now, do that all *again* if it didn't work.
	 * Otherwise, give up */

    } while( (dead_connection && (++con_attempt<4)) || ((++attempt<3) && http_can_authenticate && (req->status == 401) ));

    DEBUG( DEBUG_HTTP, "Req ends, status %d class %dxx, status line:\n%s\n", 
	   req->status, req->class, http_error );

    return ret;
}

/* Simple HTTP put. 
 * local is the local filename. Remote is the destination URI (URI?)
 * Make it proper.
 * Returns:
 *   PROTO_FILE if no local file
 *   PROTO_ERROR if something general goes wrong
 *   PROTO_OK if it all works fine
 */
int http_put( const char *local, const char *remote, const bool ascii ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "PUT", remote );

    /* joe: ANSI C says the "b" will be ignored by platforms which
     * should ignore it... but, we'll play safe: */
#if defined (__EMX__) || defined(__CYGWIN__)
    req.body_file = fopen( local, "rb" );
#else
    req.body_file = fopen( local, "r" );
#endif
    if( req.body_file == NULL ) {
	strcpy( http_error, "Could not open file." );
	ret = PROTO_FILE;
    } else {
	req.body = http_body_file;
	
	ret = http_request( &req );
	fclose( req.body_file );
	
	if( ret == PROTO_OK && req.class != 2 )
	    ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

int http_get_fd;
bool http_get_working;
size_t http_get_total, http_get_progress;

void http_get_callback( void *user, const char *buffer, const size_t len ) {
    if( !http_get_working ) return;
    DEBUG( DEBUG_HTTP, "Got progress: %d out of %d\n", len, http_get_total );
    if( send_data( http_get_fd, buffer, len ) < 0 ) {
	http_get_working = false;
    } else {
	http_get_progress += len;
	fe_transfer_progress( http_get_progress, http_get_total ); 
    }
}

int http_get( const char *local, const char *remote, const int remotesize,
	      const bool ascii ) {
    http_req_t req;
    int ret;

#if defined (__EMX__) || defined (__CYGWIN__)
    /* We have to set O_BINARY, thus need open(). Otherwise it should be
       equivalent to creat(). */
    http_get_fd = open( local, O_WRONLY|O_TRUNC|O_CREAT|O_BINARY, 0644 );
#else
    http_get_fd = creat( local, 0644 );
#endif
    if( http_get_fd < 0 ) {
	snprintf( http_error, BUFSIZ, "Could not open local file: %s", 
		  strerror( errno ) );
	return PROTO_ERROR;
    }

    http_request_init( &req, "GET", remote );
    req.body_callback = http_get_callback;

    http_get_working = true;
    http_get_progress = 0;
    http_get_total = remotesize;

    DEBUG( DEBUG_HTTP, "Total remote size: %d\n", remotesize );

    ret = http_request( &req );
    
    if( close( http_get_fd ) < 0 ) {
	snprintf( http_error, BUFSIZ, "Error closing local file: %s",
		  strerror( errno ) );
	ret = PROTO_ERROR;
    } else if( ret == PROTO_OK && req.class != 2 ) {
	ret = PROTO_ERROR;
    }
    http_request_end( &req );
    return ret;
}

/* Perform the file operations */
int dav_move( const char *from, const char *to ) {
    http_req_t req;
    int ret;

    http_request_init( &req, "MOVE", from );

    strcat( req.headers, "Destination: http://" );
    http_strcat_hostname( &http_server_host, req.headers );
    strcat( req.headers, to );
    strcat( req.headers, EOL );

    if( ! http_webdav_server ) {
	/* For non-WebDAV servers */
	strcat( req.headers, "New-URI: " );
	strcat( req.headers, to );
	strcat( req.headers, EOL );
    }

    ret = http_request( &req );
    
    if( ret == PROTO_OK && req.class != 2 ) {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

/* Deletes the specified resource on the server */
int http_delete( const char *filename ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "DELETE", filename );
    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;

    http_request_end( &req );
    return ret;
}

/* Deletes the specified resource on the server.
 * I wish we could do Depth: 0 with DELETE. */
int dav_rmdir( const char *filename ) {
    http_req_t req;
    int ret;
    char *dirname;
    
    if( strlen( filename ) < 1 ) {
	return PROTO_ERROR;
    }

    if( *(filename+strlen(filename)) != '/' ) {
	dirname = malloc( strlen( filename ) + 2 );
	strcpy( dirname, filename );
	strcat( dirname, "/" );
    } else {
	dirname = strdup( filename );
    }

    http_request_init( &req, "DELETE", dirname );
    
    ret = http_request( &req );

    /* If we get a 207, something has gone wrong */
    if( ret == PROTO_OK && ( (req.status == 207) || (req.class != 2) ) )
	ret = PROTO_ERROR;

    free( dirname );
    http_request_end( &req );
    return ret;

}

int http_mkdir_with_put( const char *realdir ) {
    char filename[BUFSIZ];
    http_req_t req;
    int ret;

    strcpy( filename, realdir );
    strcat( filename, "SitecopyTempDirCreator.txt" );

    http_request_init( &req, "PUT", filename );
    /* Can't send Content-Type, would break digest auth entity
     * digest code
    strcat( req.headers, "Content-type: text/plain" EOL );
     */
    
    ret = http_request( &req );

    if( ret == PROTO_OK && req.class == 2 ) {
	http_request_end( &req );
	http_request_init( &req, "DELETE", filename );

	ret = http_request( &req );
	if( ret != PROTO_OK || req.class != 2 )
	    /* NEED an FE warning function here */
            DEBUG( DEBUG_HTTP,
		   "Can\'t delete the temporary file http://%s:%d%s",
		   http_server_host.hostname, http_server_host.port,
		   filename );
	ret = PROTO_OK;
    } else {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

int dav_mkcol( const char *dirname ) {
    http_req_t req;
    int ret;
    char *realdir;

    if( strlen( dirname ) < 1 ) {
	strcpy( http_error, "Invalid directory name." );
	return PROTO_ERROR;
    }
    if( *(dirname+strlen(dirname)-1) == '/' ) {
	realdir = strdup( dirname );
    } else {
	/* +2 since one for \0, one for / */
	realdir = malloc( strlen(dirname) + 2 );
	strcpy( realdir, dirname );
	strcat( realdir, "/" );
    }
    
    if( http_mkdir_works == 1 ) {
	/* Use MKDIR, since we know it works */

	http_request_init( &req, "MKDIR", realdir );
	ret = http_request( &req );

	if( ret == PROTO_OK && req.class != 2 )
	    ret = PROTO_ERROR;

    } else {
	/* Try MKCOL first */
	http_request_init( &req, "MKCOL", realdir );
	
	ret = http_request( &req );
	
	if( ret == PROTO_OK && req.class == 2 ) {
	    /* MKCOL works -> MKDIR doesn't */
	    http_mkdir_works = -1;
	} else if( (http_mkdir_works > -1) && !http_webdav_server ) { 
	    /* MKCOL failed, we're not on a DAV server, lets try MKDIR */
	    http_request_end( &req );
	    http_request_init( &req, "MKDIR", realdir );
	    ret = http_request( &req );
	    if( ret == PROTO_OK && req.class == 2 ) {
		/* MKDIR does work */
		http_mkdir_works = 1;
		ret = PROTO_OK;
	    } else { 
		/* MKDIR doesn't work */
		if( req.status == 501 ) {
		    /* MKDIR definitely isn't implemented */
		    http_mkdir_works = -1;
		}
		/* Try a dummy PUT/DELETE */
		return http_mkdir_with_put( realdir );
	    }
	} else {
	    ret = PROTO_ERROR;
	}
    }

    free( realdir );
    http_request_end( &req );
    return ret;
}

int http_open( void ) {
    if( http_use_proxy ) {
	DEBUG( DEBUG_SOCKET, "Connecting to proxy at %s:%d...\n", 
	       http_proxy_host.hostname, http_proxy_host.port );
    } else {
	DEBUG( DEBUG_SOCKET, "Connecting to server at %s:%d...\n", 
	       http_server_host.hostname, http_server_host.port );
    }
    fe_connection( fe_connecting );
    http_sock = socket_connect( http_remoteaddr, http_remoteport);
    if( http_sock < 0 ) {
	DEBUG( DEBUG_SOCKET, "Could not connect: %s\n", strerror( errno ) );
	return PROTO_CONNECT;
    }
    DEBUG( DEBUG_SOCKET, "Connected.\n" );
    fe_connection( fe_connected );
    http_connected = true;
    return PROTO_OK;
}

int http_close( void ) {
    DEBUG( DEBUG_SOCKET, "Closing socket.\n" );
    socket_close( http_sock );
    http_connected = false;
    DEBUG( DEBUG_SOCKET, "Socket closed.\n" );
    return PROTO_OK;
}

int dav_mkref( const char *resref, const char *target ) {
    http_req_t req;
    int ret;

    http_request_init( &req, "MKREF", resref );
    strcat( req.headers, "Ref-Target: <" );
    strcat( req.headers, target );
    strcat( req.headers, ">" EOL );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}

int dav_chref( const char *resref, const char *target ) {
    int ret;

    /* Delete it */
    ret = dav_rmref( resref );
    if( ret != PROTO_OK ) 
	return ret;
    /* Then create it again */
    ret = dav_mkref( resref, target );

    return ret;
}

int dav_rmref( const char *resref ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "DELETE", resref );
    strcat( req.headers, "No-Passthrough: 1" EOL );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}

#ifdef HAVE_LIBEXPAT

/* TODO: Go read the XML specs again and use the correct terminology
 * for everything.  
 */

XML_Parser dav_xml_parser;

/* Parses the tag attributes, and handles XML namespaces. 
 * With a little bit of luck.
 * Return true on success. */
static bool dav_xml_parsetag( struct dav_xml_state *state,
		       const char *tag, const char **atts ) {
    dav_xml_ns *ns;
    int attn;
    char *pnt;
    const char *tag_prefix, *tag_suffix;
    struct dav_xml_state *xmlt;

    DEBUG( DEBUG_XML, "Parsing tag of name: [%s]\n", tag );
    /* Parse the atts for namespace declarations */
    for( attn = 0; atts[attn]!=NULL; attn+=2 ) {
	DEBUG( DEBUG_XML, "Got attribute: [%s] = [%s]\n", atts[attn], atts[attn+1] );
	if( strcasecmp( atts[attn], "xmlns" ) == 0 ) {
	    /* New default namespace */
	    state->default_ns = strdup( atts[attn+1] );
	    DEBUG( DEBUG_XML, "New default namespace: %s\n", 
		   state->default_ns );
	} else if( strncasecmp( atts[attn], "xmlns:", 6 ) == 0 ) {
	    /* New namespace scope */
	    ns = malloc( sizeof( dav_xml_ns ) );
	    ns->next = state->nspaces;
	    state->nspaces = ns;
	    ns->name = strdup( atts[attn]+6 ); /* skip the xmlns= */
	    ns->value = strdup( atts[attn+1] );
	    DEBUG( DEBUG_XML, "New namespace scope: %s -> %s\n",
		   ns->name, ns->value );
	}
    }
    /* Now check the tag name for a namespace scope */
    pnt = strchr( tag, ':' );
    tag_prefix = NULL;
    tag_suffix = NULL;
    if( pnt == NULL ) {
	/* No scope - have we got a default? */
	DEBUG( DEBUG_XML, "No scope found, searching for default.\n" );
	for( xmlt = state; xmlt!=NULL; xmlt=xmlt->parent ) {
	    if( xmlt->default_ns != NULL ) {
		tag_prefix = xmlt->default_ns;
		break;
	    }
	}
	if( tag_prefix != NULL ) {
	    DEBUG( DEBUG_XML, "Found default namespace [%s]\n", tag_prefix );
	} else {
	    DEBUG( DEBUG_XML, "No default namespace, using empty.\n" );
	    tag_prefix = "";
	}
	tag_suffix = tag;
    } else {
	DEBUG( DEBUG_XML, "Got namespace scope. Trying to resolve..." );
	/* Have a scope - resolve it */
	for( xmlt = state; tag_prefix==NULL && xmlt!=NULL; xmlt=xmlt->parent ) {
	    for( ns = xmlt->nspaces; ns!=NULL; ns=ns->next ) {
		/* Just compare against the bit before the :
		 * pnt points to the colon. */
		if( strncasecmp( ns->name, tag, pnt-tag ) == 0 ) {
		    /* Scope matched! Hoorah */
		    tag_prefix = ns->value;
		    /* end the search */
		    break;
		}
	    }
	}
	if( tag_prefix != NULL ) {
	    DEBUG( DEBUG_XML, "Resolved scope to [%s]\n", tag_prefix );
	    /* The suffix is everything after the ':' */
	    tag_suffix = pnt+1;
	    if( *tag_suffix == '\0' ) {
		DEBUG( DEBUG_XML, "No element name after ':'. Failed.\n" );
		return false;
	    }
	} else {
	    DEBUG( DEBUG_XML, "Smeg. We lost it somewhere.\n" );
	    return false;
	}
    }
    /* here, we have tag_suffix and tag_prefix */
    DEBUG( DEBUG_XML, "prefix: [%s], suffix: [%s]\n", tag_prefix, tag_suffix );
    pnt = state->tag_name = malloc( strlen(tag_prefix) + strlen(tag_suffix) +
				    1 );
    strcpy( pnt, tag_prefix );
    strcat( pnt, tag_suffix );
    DEBUG( DEBUG_XML, "You gave me: %s, and I gave you this: %s\n", tag, pnt );
    return true;
}

/* This is the XML_StartElementHandler... called with the start of a
 * new element. */
static void dav_xml_startelm( void *userdata, const char *tag, const char **atts ) {
    struct dav_xmldoc *doc = (struct dav_xmldoc *)userdata;
    struct dav_xml_state *s;
    int n;

    if( !doc->valid ) {
	/* We've stopped parsing */
	DEBUG( DEBUG_XML, "Parse died. Ignoring start of element: %s\n", tag );
	return;
    }
    /* Set the new state */
    s = malloc( sizeof(struct dav_xml_state) );
    memset( s, 0, sizeof(struct dav_xml_state) );
    s->parent = doc->current;
    doc->current = s;
    if( dav_xml_parsetag( s, tag, atts ) == false ) {
	/* nooo, it bombed */
	doc->valid = false;
	return;
    }
    /* Map the name to a tag */
    DEBUG( DEBUG_XML, "Mapping tag name... " );
    s->tag = dav_xml_unknown;
    for( n = 0; dav_xml_tagnames[n] != NULL; n++ ) {
	if( strcasecmp( dav_xml_tagnames[n], s->tag_name ) == 0 ) {
	    s->tag = n;
	    break;
	}
    }
    if( s->tag == dav_xml_unknown ) {
	DEBUG( DEBUG_XML, "unknown tag, ignoring.\n" );
	return;
    }
    DEBUG( DEBUG_XML, "mapped.\n" );

    /* Normally, we don't want to collect cdata */
    doc->want_cdata = false;

    /* expat is not a validating parser - check the new tag
     * is valid in the current context.
     */
    DEBUG( DEBUG_XML, "Checking context of tag (parent: %s)\n",
	   dav_xml_tagnames[s->parent->tag] );
    switch( s->parent->tag ) {
    case dav_xml_root:
	switch( s->tag ) {
	case dav_xml_multistatus:
	case dav_xml_response:
	    break;
	default:
	    doc->valid = false;
	    break;
	}
	break;
    case dav_xml_multistatus:
	switch( s->tag ) {
	case dav_xml_response:
	case dav_xml_responsedescription:
	    break;
	default:
	    doc->valid = false;
	    break;
	}
	break;
    case dav_xml_response:
	switch( s->tag ) {
	case dav_xml_href:
	case dav_xml_propstat:
	case dav_xml_responsedescription:
	case dav_xml_status:
	    break;
	default:
	    doc->valid = false;
	}
	break;
    default:
	break;
    }
    if( doc->valid == false ) {
	DEBUG( DEBUG_XML, "Parse error - not good context for this tag.\n" );
	return;
    }
    /* Now actually do something about it */
    switch( s->tag ) {
    case dav_xml_response:
	/* new file information */
	DEBUG( DEBUG_XML, "New file context.\n" );
	memset( doc->file, 0, sizeof(struct proto_file_t) );
	break;

    case dav_xml_href:
    case dav_xml_getcontentlength:
    case dav_xml_resourcetype:
    case dav_xml_getlastmodified:
	/* For these elements, collect the CDATA */
	DEBUG( DEBUG_XML, "Collecting CDATA...\n" );
	doc->want_cdata = true;
	break;
    case dav_xml_prop:
    case dav_xml_propstat:
    case dav_xml_collection:
    default:
	/* donothing */
	break;
    }
    if( doc->want_cdata ) {
	/* Ready the cdata buffer for new cdata */
	if( doc->cdata_buflen == -1 ) {
	    /* Create a buffer */
	    DEBUG( DEBUG_XML, "Allocating new cdata buffer.\n" );
	    doc->cdata = malloc( CDATABUFSIZ );
	    doc->cdata_buflen = CDATABUFSIZ;
	}
	/* Now zero-out the buffer. */
	memset( doc->cdata, 0, doc->cdata_buflen );
	/* And we've got nothing in it */
	doc->cdata_len = 0;
    }
}

/* Returns the depth of the file in the directory heirarchy,
 * i.e. 1 for /foo.html, 2 for /bar/norm.html, 4 for /a/b/c/d.html
 */
static int dav_fetch_getdepth( const char *href ) {
    const char *pnt;
    int count = 0;
    for( pnt=href; *pnt != '\0'; pnt++ ) /* oneliner */
	if( *pnt == '/' ) count++;
    return count;
}

static bool dav_fetch_parse_href( struct dav_xmldoc *doc ) {
    const char *tmp;
    char *dec;
    bool ret;
    size_t rootlen;
    DEBUG( DEBUG_HTTP, "Parsing href [%s]\n", doc->cdata );
    if( strncmp( doc->cdata, "http://", 7 ) == 0 ) {
	/* Absolute URI.
	 * Look for the path bit */
	DEBUG( DEBUG_HTTP, "Got absolute URI.\n" );
	tmp = strchr( doc->cdata+7, '/' );
	if( tmp == NULL ) {
	    DEBUG( DEBUG_HTTP, "No path segment found.\n" );
	    return false;
	}
    } else {
	tmp = doc->cdata;
    }
    DEBUG( DEBUG_HTTP, "Using abspath segment: %s\n", tmp );
    dec = uri_decode( tmp );
    DEBUG( DEBUG_HTTP, "Decoded is: [%s]. Root is [%s]\n", tmp,
	   doc->fetch_root );
    /* Now, dec points to the absPath section of the URI.
     * We check whether this resource is actually in the
     * collection we have done the PROPFIND against.
     */
    rootlen = strlen( doc->fetch_root );
    if( strncmp( dec, doc->fetch_root, rootlen ) != 0 ) {
	DEBUG( DEBUG_HTTP, "parse_href failed: root collection not matched." );
	ret = false;
    } else {
	/* We're in the right place */
	DEBUG( DEBUG_HTTP, "parse_href: Got [%s]\n", dec + rootlen );
	/* Filename must be the basename */
	doc->file->filename = strdup( base_name( dec + rootlen ) );
	/* And fill in the directory while we're here */
	doc->file->directory = dir_name( dec + rootlen );
	ret = true;
	DEBUG( DEBUG_HTTP, "parse_href: Filename [%s], Directory [%s]\n",
	       doc->file->filename, doc->file->directory );
    }
    free( dec );
    return ret;
}

static void dav_fetch_gotresource( struct dav_xmldoc *doc ) {
    struct proto_file_t *current, *previous;

    DEBUG( DEBUG_HTTP, "Got resource:\n"
	   "filename = [%s] isdir = %s size = %d mtime = %s\n", 
	   doc->file->filename, doc->file->isdir?"true":"false", 
	   doc->file->size, rfc1123_date( doc->file->modtime ) );

    if( (strlen( doc->file->directory ) == 0) &&
	(strlen( doc->file->filename ) == 0 ) ) {
	DEBUG( DEBUG_HTTP, "Resource is root collection, ignoring.\n" );
	return;
    }
   
    if( doc->file->isdir ) {
	doc->file->directory[strlen(doc->file->directory)-1] = '\0';
    }
    DEBUG( DEBUG_HTTP, "Filename is really: %s\n", doc->file->filename );

    /* Depth in the hierarchy - i.e., how many directories deep the
     * resource is. */
    doc->file->depth = dav_fetch_getdepth( doc->file->directory );
    DEBUG( DEBUG_HTTP, "File is at depth: %d\n", doc->file->depth );

    /* Insert it into the list, keeping the list sorted by depth */
    for( current=doc->files, previous=NULL; current!=NULL; 
	 previous=current, current=current->next )
	/* one-liner */
	if( current->depth > doc->file->depth )
	    break;

    doc->file->next = current;

    if( previous == NULL ) {
	doc->files = doc->file;
    } else {
	previous->next = doc->file;
    }

    /* Create a new file, ready to be filled in */
    doc->file = malloc( sizeof( struct proto_file_t ) );
    memset( doc->file, 0, sizeof( struct proto_file_t ) );
}

/* End-of-element handler */
static void dav_xml_endelm( void *userdata, const char *tag ) {
    struct dav_xmldoc *doc = (struct dav_xmldoc *)userdata;
    struct dav_xml_state *s;
    dav_xml_ns *this_ns, *next_ns;
    if( !doc->valid ) {
	/* We've stopped parsing */
	DEBUG( DEBUG_XML, "Parse died. Ignoring end of element: %s\n", tag );
	return;
    }
    s = doc->current;
    DEBUG( DEBUG_XML, "End of element %s.\n", tag);
    switch( s->tag ) {
    case dav_xml_href:
	doc->valid = dav_fetch_parse_href( doc );
	break;
    case dav_xml_getlastmodified:
	DEBUG( DEBUG_HTTP, "Parsing date [%s]\n", doc->cdata );
	doc->file->modtime = rfc1123_parse( doc->cdata );
	if( doc->file->modtime == (time_t)-1 ) {
	    DEBUG( DEBUG_HTTP, "Date is not in RFC1123 format.\n" );
	    doc->valid = false;
	}
	break;
    case dav_xml_getcontentlength:
	doc->file->size = atoi( doc->cdata );
	break;
    case dav_xml_collection:
	doc->file->isdir = true;
	break;
    case dav_xml_response:
	dav_fetch_gotresource( doc );
	break;
    default:
	break;
    }
    /* Move the current pointer up the chain */
    doc->current = s->parent;
    DEBUG( DEBUG_XML, "Back in tag: %s\n", doc->current->tag_name );
    if( doc->want_cdata ) {
	/* Free the cdata buffer if it's grown to big for its boots. */
	if( doc->cdata_buflen > CDATASHRINK ) {
	    DEBUG( DEBUG_XML, "cdata buffer overgrown, freeing.\n" );
	    free( doc->cdata );
	    doc->cdata_buflen = -1;
	} 
	/* And we've stopped collecting it now, thanks */
	doc->want_cdata = false;
    }
    if( s->default_ns!=NULL ) free( s->default_ns );
    /* Free the namespaces */
    this_ns = s->nspaces;
    while( this_ns != NULL ) {
	next_ns = this_ns->next;
	free( this_ns );
	this_ns = next_ns;
    };
    free( s->tag_name );
    free( s );
    DEBUG( DEBUG_XML, "Cleanup okay.\n" );
}

/* CDATA handler. We keep the entire cdata for each element in
 * doc->cdata, and expand the buffer as necessary. */
static void dav_xml_cdata( void *userdata, const char *cdata, int len ) {
    struct dav_xmldoc *doc = (struct dav_xmldoc *)userdata;
    size_t newlen;
    
    if( !doc->want_cdata ) return;
    /* First, if this is the beginning of the CDATA, skip all
     * leading whitespace, we don't want it. */
    if( doc->cdata_buflen < 0  ) {
	DEBUG( DEBUG_XML, "ALERT: Shouldn't be collecting.\n" );
	return;
    }
    DEBUG( DEBUG_XML, "Given %d bytes of cdata.\n", len );
    if( doc->cdata_len == 0 ) {
	size_t wslen = 0;
	/* Ignore any leading whitespace */
	while( wslen < len && 
	       ( cdata[wslen] == ' ' || cdata[wslen] == '\r' ||
		 cdata[wslen] == '\n' || cdata[wslen] == '\t' ) ) {
	    wslen++;
	}
	cdata += wslen;
	len -= wslen;
	DEBUG( DEBUG_XML, "Skipped %d bytes of leading whitespace.\n", 
	       wslen );
	if( len == 0 ) {
	    DEBUG( DEBUG_XML, "Zero bytes of content.\n" );
	    return;
	}
    }
    /* Work out whether we need to expand the cdata buffer to
     * include the new incoming cdata. We always keep one more
     * byte in the buffer than we need, for the null-terminator */
    for( newlen = doc->cdata_buflen; 
	 newlen < (doc->cdata_len + len + 1);
	 newlen += CDATABUFSIZ ) /* nullop */ ;
    if( newlen > doc->cdata_buflen ) { 
	size_t oldbuflen = doc->cdata_buflen;
	/* Reallocate bigger buffer */
	DEBUG( DEBUG_XML, "Growing CDATA buffer from %d to %d.\n",
	       oldbuflen, newlen );
	doc->cdata = realloc( doc->cdata, newlen );
	doc->cdata_buflen = newlen;
	/* Zero-out the new bit of buffer */
	memset( doc->cdata+oldbuflen, 0, newlen-oldbuflen );
    }
    /* Now simply copy the new cdata onto the end of the buffer */
    memcpy( doc->cdata+doc->cdata_len, cdata, len );
    doc->cdata_len += len;
    DEBUG( DEBUG_XML, "Collected %d bytes of cdata, buffer now:\n%s\n",
	   len, doc->cdata );
}

static void http_get_content_charset( const char *name, const char *value ) {
    char **pairs;
    int n;
    if( strcasecmp( name, "Content-Type" ) == 0 ) {
	/* Get the charset so we can pass it on to expat */
	pairs = strpairs( value, ';', '=', http_quotes, http_whitespace );
	for( n = 0; pairs[n] != NULL; n+=2 ) {
	    if( (strcasecmp( pairs[n], "charset") == 0) &&
		pairs[n+1] != NULL ) {
		DOFREE( http_content_charset );
		/* Strip off the quotes */
		http_content_charset = strstrip( pairs[n+1], '\"' );
		DEBUG( DEBUG_HTTP, "Got content type charset: %s\n",
		       http_content_charset );
	    }
	}
	strpairs_free( pairs );
    }
    return;
}

static void dav_xml_parsebody( void *userdata, 
			const char *buffer, const size_t len ) {
    struct dav_xmldoc *doc = userdata;
    int ret;
    /* duck out if it's broken */
    if( !doc->valid ) {
	DEBUG( DEBUG_XML, "Not parsing %d bytes!\n", len );
	return;
    }
    if( len == 0 ) {
	DEBUG( DEBUG_XML, "Got 0-length buffer, end of response.\n" );
	ret = XML_Parse( dav_xml_parser, "", 0, -1 );
    } else {
	DEBUG( DEBUG_XML, "Got %d length buffer.\n", len );
	ret = XML_Parse( dav_xml_parser, buffer, len, 0 );
    }
    DEBUG( DEBUG_XML, "XML_Parse returned %d\n", ret );
    if( ret == 0 ) {
	doc->valid = false;
    }
}

/* WebDAV fetch mode handler. */
int dav_fetch( const char *dirname, struct proto_file_t **files ) {
    http_req_t req;
    struct dav_xmldoc doc = {0};
    int ret;
    const char *propfind_body =
	"<?xml version=\"1.0\"?>" EOL /* should we use encoding=? */
	"<propfind xmlns=\"DAV:\">" EOL
	"  <prop>" EOL
	"    <getcontentlength/>" EOL
	"    <getlastmodified/>" EOL
	"    <resourcetype/>" EOL
	"  </prop>" EOL
	"</propfind>" EOL;
    const char *myheaders =
	"Content-Type: text/xml" EOL /* should we use charset=? */
	"Depth: infinity" EOL;
       
    dav_xml_parser = XML_ParserCreate( NULL );
    XML_SetElementHandler( dav_xml_parser, dav_xml_startelm, dav_xml_endelm );
    XML_SetCharacterDataHandler( dav_xml_parser, dav_xml_cdata );
    XML_SetUserData( dav_xml_parser, (void *) &doc );
    /* Create a dummy state to act as the root element in the 
     * tree. Just makes things a little bit easier since we can 
     * then always presume we have a ->parent element.
     */
    doc.root = malloc( sizeof( struct dav_xml_state ) );
    memset( doc.root, 0, sizeof( struct dav_xml_state ) );
    doc.root->tag = dav_xml_root;
    doc.root->tag_name = "@root@";
    /* And set it to the current element */
    doc.current = doc.root;
    /* Init the document */
    doc.files = NULL;
    doc.fetch_root = dirname;
    doc.valid = true; /* so far... */
    doc.file = malloc( sizeof( struct proto_file_t ) );
    memset( doc.file, 0, sizeof( struct proto_file_t ) );

    http_request_init( &req, "PROPFIND", dirname );
    req.body_callback = dav_xml_parsebody;
    req.body_callback_userdata = &doc;
    req.body = http_body_buffer;
    req.body_buffer = propfind_body;
    /* Add in the content type header */
    strcat( req.headers, myheaders );
    
    ret = http_request( &req );

    XML_ParserFree( dav_xml_parser );

    free( doc.root );

    if( ret == PROTO_OK && req.class == 2 && doc.valid ) {
	*files = doc.files;
	ret = PROTO_OK;
    } else {
	*files = NULL;
	ret = PROTO_ERROR;
    }
    http_request_end( &req );
    return ret;
}

#endif /* HAVE_LIBEXPAT */

int http_head( const char *directory ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "HEAD", directory );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}

static void http_options_parsehdr( const char *name, const char *value ) {
    char **classes, **class;
    if( strcasecmp( name, "DAV" ) == 0 ) {
	DEBUG( DEBUG_HTTP, "Got OPTIONS header with value: %s\n", value );
	classes = strsplit( value, ',', http_quotes, http_whitespace );
	for( class = classes; *class!=NULL; class++ ) {
	    DEBUG( DEBUG_HTTP, "Got compliance class: [%s]\n", *class );
	    if( strncmp( *class, "1", 1 ) == 0 ) {
		DEBUG( DEBUG_HTTP, "Class 1 compliant server.\n" );
		http_webdav_server = true;
	    }
	}
	strsplit_free( classes );
    }
}

/* Performs an OPTIONS request.
 * Sets http_webdav_server appropriately.
 */
int http_options( const char *directory ) {
    http_req_t req;
    int ret;
    
    http_webdav_server = false;

    http_request_init( &req, "OPTIONS", directory );
    req.hdrs_callback = http_options_parsehdr;

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}
