/*  Screem:  linkView.c,
 *  A start at a site wide link view
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <gnome.h>

#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#include "linkview.h"
#include "preferences.h"
#include "site.h"
#include "siteTree.h"
#include "support.h"
#include "xml.h"

extern GtkWidget *app;
extern Site *current_site;
extern Preferences *cfg;

typedef struct _LinkIcon {
	Icons *icon;     /* the icon */
	const gchar *filename; /* the name of this file */
	GList *links;    /* a list of LinkIcons */
	gboolean is_page;
	GnomeCanvasGroup *group; /* the group for the icon */
} LinkIcon;

GList *icons = NULL;   /* a list of all link icons */

static LinkIcon *create_link_icon( Page *page, const gchar *filename );

static GnomeCanvasGroup *create_link_item( GnomeCanvasGroup *link_root,
					   Site *site,
					   LinkIcon *icon );

/**
 * screem_link_view_create_icons:
 *
 * Creates a list of canvas items for each file in the site
 *
 * return values: none
 */
void screem_link_view_create()
{
	GtkWidget *canvas;
	GnomeCanvasGroup *root;
	GtkType type;
	static GnomeCanvasGroup *link_root = NULL;

	Site *site;
	Page *page;

	gchar *filename;

	LinkIcon *icon;
	GList *list;

	site = current_site;

	g_return_if_fail( site != NULL );

	page = screem_site_get_current_page( site );
	
	g_return_if_fail( page != NULL );

	canvas = gtk_object_get_data( GTK_OBJECT( app ), "link_view" );

	gnome_canvas_set_scroll_region( GNOME_CANVAS( canvas ), 0.0, 0.0,
                                        (gdouble)640.0, (gdouble)480.0 );

	if( link_root ) {
		/* destroy old view */
		gtk_object_destroy( GTK_OBJECT( link_root ) );
		gnome_canvas_update_now( GNOME_CANVAS( canvas ) );
		g_print( "removed old icons\n" );
		for( list = icons; list; list = list->next ) {
			icon = list->data;
			g_list_free( icon->links );
			g_free( icon );
		}
		for( list = g_list_last( icons ); 
		     list != g_list_first( icons ); ) {
			list = list->prev;
			g_list_free( list->next );
		}
		g_list_free( icons );
		icons = NULL;
	}

	root = GNOME_CANVAS_GROUP( gnome_canvas_root( GNOME_CANVAS(canvas) ) );
	type = gnome_canvas_group_get_type();
	link_root = 
		GNOME_CANVAS_GROUP( gnome_canvas_item_new( root, type,
							   "x", 0.0,
							   "y", 0.0,
							   NULL ) );
	create_link_icon( page, screem_page_get_pathname( page ) );

	/* we have our list of icons, build canvas items for them */
	gtk_widget_hide( canvas );
	for( list = icons; list; list = list->next ) {
		create_link_item( link_root, site, list->data );
	}

	/* created, position and link with lines */
	for( list = icons; list; list = list->next ) {

	}

	gtk_widget_show( canvas );
}

static LinkIcon *create_link_icon( Page *page, const gchar *filename )
{
	LinkIcon *icon;

	icon = g_new( LinkIcon, 1 );
	icon->filename = filename;
	icon->icon = icon_from_mime_type( gnome_mime_type( filename ) );
	icon->links = NULL;
	icon->is_page = (gboolean)page;

	/* add to list of link icons */
	icons = g_list_append( icons, icon );

	if( page ) {
		/* we need to identify the links */
	}

	return icon;
}

static GnomeCanvasGroup *create_link_item( GnomeCanvasGroup *link_root,
					   Site *site,
					   LinkIcon *icon )
{
	GnomeCanvasGroup *group;
	GnomeCanvasItem *item1;
	GnomeCanvasItem *item2;
	GnomeCanvasItem *item3;
	GnomeCanvasItem *item4;
	gint width;
	gint c;
	GtkType type;
	const gchar *filename;

	/* create a group for the icon so we can move it all
	   around at once, eg the text, the border box and the
	   image */
	type = gnome_canvas_group_get_type();
	group = GNOME_CANVAS_GROUP( gnome_canvas_item_new( link_root,
							   type,
							   "x", 0.0,
							   "y", 320.0,
							   NULL ) );
	icon->group = group;
	
	filename = icon->filename;
	if( icon->is_page )
		filename += strlen( screem_site_get_pathname( site ) );
	for( c = 0, width = 0; filename[ c ]; c ++ ) {
		width += gdk_char_width( cfg->font, filename[ c ] );
		width += 2;
	}
	
	/* the box */
	type = gnome_canvas_rect_get_type();
	item3 = gnome_canvas_item_new( group, type,
				       "x1", 0.0,
				       "y1", 0.0,
				       "x2", 0.0 + (double)width + 
				       24.0 + 16.0 ,
				       "y2", 0.0 + 22,
				       "fill_color_rgba", 0x8080FFFF,
				       "outline_color", "black",
				       "width_units", 1.0,
				       NULL );
	/* the image */
	type = gnome_canvas_image_get_type();
	item1 = gnome_canvas_item_new( group, type,
				       "image", icon->icon->image,
				       "x", 2.0,
				       "y", 2.0,
				       "width", 20.0,
				       "height", 20.0,
				       "anchor", GTK_ANCHOR_NORTH_WEST,
				       NULL );
	/* the text */
	type = gnome_icon_text_item_get_type();
	item2 = gnome_canvas_item_new( group, type, NULL );
	gnome_icon_text_item_configure( GNOME_ICON_TEXT_ITEM( item2 ),
					22.0, 2.0, 
					(double)width,
					cfg->font_name,
					filename,
					FALSE, FALSE );
	/* the expand box */
	type = gnome_canvas_rect_get_type();
	item4 = gnome_canvas_item_new( group, type,
				       "x1", 0.0 + (double)width +22.0,
				       "y1", 6.0,
				       "x2", 0.0 + (double)width +
				       22.0 + 12.0,
				       "y2", 6.0 + 12,
				       "fill_color_rgba", 0xFFFFFF00,
				       "outline_color", "black",
				       "width_units", 1.0,
				       NULL );

	return group;
}
