/* Bluefish HTML Editor
 * callbacks.c - contains all callbacks
 *
 * Copyright (C) 1998-1999 Olivier Sessink and Chris Mazuc
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <locale.h>
#include <libintl.h>
#define _(STRING) gettext(STRING)

#include "config.h"

#if HAVE_GTKEDITOR
#include <gtkeditor/gtkeditor.h>
#endif

#include "bluefish.h"
#include "coloursel.h"
#include "widgets.h"
#include "interface.h"
#include "network.h"
#include "stringlist.h"
#include "init.h"
#include "html.h"
#include "gtk_easy.h"
#include "debug.h"

/* these should use the proper .h files instead of using them extern */
extern int closed;
extern int sockfd;
extern void return_files(void);
extern void create_dialog (int, int, gchar *);
extern void string_apply (gchar ** config_var, GtkWidget * entry);

/* this struct is used for the project editor */
typedef struct
  {
    GtkWidget *window;
    GtkWidget *base_entry;
    GtkWidget *web_entry;
    GtkWidget *templ_entry;
  }
EditProject;

/* this struct is used for the closing of modified files */
typedef struct
  {
    GtkWidget *dialog;
    gint still_continue;
  }
Warn_Close_Struct;

/* a global variable for reading and writing projects */
GList *project_list = NULL;

/****************************************************************/
void
file_save (GtkWidget * widget, gpointer data);
void
file_save_as (GtkWidget * widget, gpointer data);
void
files_open (GtkWidget * widget, gpointer data);
void
file_open (GtkWidget * widget, gpointer data);
void
file_insert (GtkWidget * widget, gpointer data);
void
file_new (GtkWidget * widget, gpointer data);
void
file_close (GtkWidget * widget, gpointer data);
void
file_close_all (GtkWidget * widget, gpointer data);
void
file_save_all (GtkWidget * widget, gpointer data);
void
project_open (GtkWidget * widget, gpointer data);
void
project_close (GtkWidget * widget, gpointer data);
void
project_save (GtkWidget * widget, gpointer data);
void
project_save_as (GtkWidget * widget, gpointer data);
void
project_add_document (GtkWidget * widget, gpointer data);
GtkWidget *
string_entry (gchar * labeltext, GtkWidget * table, gchar * which_config_string, gint left, gint right, gint top, gint bottom);
GtkWidget *button_list(gchar *but_title, gint which_list, gchar *tooltip_text, GtkTooltips *tooltips);
void
project_edit (GtkWidget * widget, gpointer data);
void
project_edit_list (GtkWidget * widget, gpointer data);
void
project_edit_ok (GtkWidget * widget, EditProject * data);
void
project_edit_cancel (GtkWidget * widget, EditProject * data);
GList *
create_project_list (GList * project_list);
void
project_from_file (void);
void
project_to_file (void);
void
view_in_netscape (GtkWidget * widget, gpointer data);
void
error_dialog (gchar * window_title, gchar * error_string);
void
open_from_web ();
void
open_from_web_callback ();
void
key_press_callback (GtkWidget * widget, GdkEventKey * event, gpointer data);
gint
warn_close_dialog (gchar * filename);
void
file_close_cancel (GtkWidget * button, Warn_Close_Struct * windowname);
void
file_close_ok (GtkWidget * button, Warn_Close_Struct * windowname);
void
file_close_save (GtkWidget * button, Warn_Close_Struct * windowname);
void
file_to_textbox (gchar * filename);
void
textbox_to_file (gchar * filename);
void
re_check (GtkWidget * widget, GtkWidget * window);
void go_to_line_win(GtkWidget *widget, gpointer data);
void go_to_line_cb(GtkWidget *widget, gpointer data);
void
selection_made (GtkWidget * clist, gint row, gint column,
		GdkEventButton * event, gpointer data);
void
run_weblint (GtkWidget * widget, gpointer data);
static void
scrollbar_update (GtkText * txt, gint tl, gint ln);
static void
go_to_line (gint linenum);

/****************************************************************/

/*
 * Function: file_save
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	This call back function is check and dispath if need
 * 	to save or save as
 */

void
file_save (GtkWidget * widget, gpointer data)
{
  /* This is the callback function for a save button */
  DEBUG_MSG ("file_save, file save started\n");
  if (main_v->current_document->modified)
    {
      if (main_v->current_document->filename == NULL)
	{
	  DEBUG_MSG ("file_save, No filename known --> save as..\n");
	  file_save_as (widget, data);
	}
      else
	{
	  DEBUG_MSG ("file_save, Saving to %s\n", main_v->current_document->filename);
	  textbox_to_file (main_v->current_document->filename);
	}
    }
}

/*
 * Function: file_save_as
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Get new file name and small check
 */
void
file_save_as (GtkWidget * widget, gpointer data)
{
gchar *oldfilename;

  oldfilename = main_v->current_document->filename;
  main_v->current_document->filename = return_file (NULL);
  if (main_v->current_document->filename == NULL)
    {
      main_v->current_document->filename = oldfilename;
      DEBUG_MSG ("file_save_as, returned file == NULL\n");
      error_dialog (_("Bluefish error"), _("No filename known"));
      DEBUG_MSG ("file_save_as, error dialog has happened? about to return;\n");
      return;
    }
  else
    {
      DEBUG_MSG ("file_save_as, returned file %s\n", main_v->current_document->filename);
      g_free (oldfilename);
      textbox_to_file (main_v->current_document->filename);
      gtk_label_set (GTK_LABEL (main_v->current_document->tab_label), strip_filename (main_v->current_document->filename));
    }
}

/*
 * Function: files_open
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Display file open dialog and load multiple files (if selected)
 */
void
files_open (GtkWidget * widget, gpointer data)
{
  return_files ();

}

/*
 * Function: file_open
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Display file open dialog and if file is entered load it.
 */
void
file_open (GtkWidget * widget, gpointer data)
{
  gchar *tmpfilename;

  tmpfilename = return_file (NULL);
  if (tmpfilename == NULL)
    {
      DEBUG_MSG ("file_open, returned file == NULL\n");
      error_dialog (_("Bluefish error"), _("No filename known"));
      return;
    }
  else
    {
      DEBUG_MSG ("file_open, returned file %s\n", tmpfilename);
      main_v->current_document = new_document ();
      main_v->current_document->filename = tmpfilename;
      file_to_textbox (main_v->current_document->filename);
      gtk_label_set (GTK_LABEL (main_v->current_document->tab_label), strip_filename (main_v->current_document->filename));
    }
}

/*
 * Function: file_insert
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Display dialog for file name and insert entered file in editor
 */
void
file_insert (GtkWidget * widget, gpointer data)
{
  gchar *tmpfilename;
  gint currentp;

  tmpfilename = return_file (NULL);
  if (tmpfilename == NULL)
    {
      error_dialog (_("Bluefish error"), _("No filename known"));
      return;
    }
  else
    {
#if HAVE_GTKEDITOR
      currentp = GTK_SCTEXT (main_v->current_document->textbox)->cursor_mark.index;
      GTK_EDITABLE (main_v->current_document->textbox)->current_pos = currentp;
      GTK_SCTEXT (main_v->current_document->textbox)->cursor_mark.index = currentp;
      gtk_sctext_set_point (GTK_SCTEXT (main_v->current_document->textbox), currentp);
#else
      currentp = GTK_TEXT (main_v->current_document->textbox)->cursor_mark.index;
      gtk_editable_set_position (GTK_EDITABLE (main_v->current_document->textbox), currentp);
      gtk_text_set_point (GTK_TEXT (main_v->current_document->textbox), currentp);
#endif
      file_to_textbox (tmpfilename);
      g_free (tmpfilename);
    }
}

/*
 * Function: file_new
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Create new editor notebook
 */
void
file_new (GtkWidget * widget, gpointer data)
{

  /* This is the callback function for a file new button */
  DEBUG_MSG ("file_new, started\n");
  main_v->current_document = new_document ();
  if (file_exists_and_readable (main_v->current_project.template) == 1)
    {
      file_to_textbox (main_v->current_project.template);
    }
  DEBUG_MSG ("file_new, end\n");
}

/*
 * Function: file_close
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Close current editor notebook, make some checking
 */
void
file_close (GtkWidget * widget, gpointer data)
{
  /* This is the callback function for a file close button */
  DEBUG_MSG ("file_close, begin\n");

  if (main_v->current_document != NULL)
    {
      if (!main_v->current_document->modified)
	{
	  DEBUG_MSG ("file_close, destroy it\n");
	  if (!(main_v->current_document->filename == NULL &&
		main_v->current_document->modified ==
		FALSE && g_list_length (main_v->documentlist) == 2))
	    {
	      DEBUG_MSG
		("file_close, not closing last unmodified, untitled document\n");
	      destroy_current_document ();
	    }
	}
      else
	{
	  if (warn_close_dialog (main_v->current_document->filename) == 1)
	    {
	      DEBUG_MSG ("file_close, warning dialog returned 1 --> destroy\n");
	      destroy_current_document ();
	    }
	  else
	    {
	      DEBUG_MSG ("file_close, warning dialog returned 0, ** STOP **\n");
	      return;
	    }
	}
    }
#ifdef DEBUG
  else
    {
      DEBUG_MSG ("file_close, cannot close NULL document\n");
    }
#endif
  if (main_v->current_document == NULL)
    {
      file_new (NULL, NULL);
    }
}

/*
 * Function: file_close_all
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Close all editor notebooks
 */
void
file_close_all (GtkWidget * widget, gpointer data)
{

  gint count;

  count = g_list_length (main_v->documentlist) - 1;
  while (count > 0)
    {
      main_v->current_document = NULL;
      main_v->current_document = g_list_nth_data (main_v->documentlist, count);
      if (main_v->current_document == NULL)
	{
	  g_print ("file_close_all, a NULL document in the documentlist\n");
	}
      else
	{
	  if (main_v->current_document->modified == 1)
	    {
	      if (warn_close_dialog (main_v->current_document->filename))
		{
		  DEBUG_MSG ("file_close_all, returned 1, so still continue\n");
		}
	      else
		{
		  DEBUG_MSG ("file_close_all, returned 0,  ** STOP **\n");
		  return;
		}
	    }
	}
      count--;
    }
  count = g_list_length (main_v->documentlist) - 1;
  while (count > 0)
    {
      main_v->current_document = g_list_nth_data (main_v->documentlist, count);
      if (main_v->current_document != NULL)
	{
	  destroy_current_document ();
	}
#ifdef DEBUG
      else
	{
	  DEBUG_MSG ("file_close_all, main_v->current_document==NULL, count=%d\n", count);
	}
#endif
      count--;
      DEBUG_MSG ("file_close_all, count=%d\n", count);
    }
  notebook_changed ();
  if (main_v->current_document == NULL)
    {
      /* File close all, keep one document open */
      file_new (NULL, NULL);
    }
}


/*
 * Function: file_save_all
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Save all editor notebooks
 */
void
file_save_all (GtkWidget * widget, gpointer data)
{

  gint count;

  count = g_list_length (main_v->documentlist) - 1;
  while (count > 0)
    {
      main_v->current_document = NULL;
      main_v->current_document = g_list_nth_data (main_v->documentlist, count);
      if (main_v->current_document != NULL)
	{
	  if (main_v->current_document->modified)
	    {
	      if (main_v->current_document->filename == NULL)
		{
		  DEBUG_MSG
		    ("file_save_all, No filename known --> save as..\n");
		  file_save_as (widget, data);
		}
	      else
		{
		  DEBUG_MSG ("file_save_all, Saving to %s\n", main_v->current_document->filename);
		  textbox_to_file (main_v->current_document->filename);
		}
	    }
	}
      else
	{
	  DEBUG_MSG ("file_save_all, main_v->current_document==NULL, count=%d\n", count);
	}
      count--;
    }
}

void
project_open (GtkWidget * widget, gpointer data)
{
  gchar *tmpfilename;

  /* change dir into project dir */
  tmpfilename = g_strconcat (g_get_home_dir(), "/.bluefish/projects/", NULL);
  DEBUG_MSG ("project_open, change_dir(%s)\n", tmpfilename);
  change_dir (tmpfilename);
  g_free (tmpfilename);
  DEBUG_MSG ("project_open, main_v->current_project.filename=%p\n", main_v->current_project.filename);
  tmpfilename = return_file (main_v->current_project.filename);
  if (file_exists_and_readable (tmpfilename) == 1)
    {

      /* close old project first */
      project_close (NULL, NULL);
      if (main_v->current_project.filename != NULL)
	{
	  DEBUG_MSG ("project_open, the old project isn't closed !!!!\n");
	  /* TODO: statusbar message needed here !!!!!!!!! */
	  goto out_free;
	}

      main_v->current_project.filename = g_strdup(tmpfilename);
      project_from_file ();
    }
  else
    {
      DEBUG_MSG ("project_open, filename not readable\n");
      /* TODO: statusbar message needed here !!!!!!!!! */
    }
out_free:
  if (tmpfilename != NULL)
	  g_free (tmpfilename);
}

void
project_close (GtkWidget * widget, gpointer data)
{
  GList *tmpdoclist, *tmpprolist;

/*  if (main_v->current_project.modified == 1)
 *    {
 *		if (warn_close_dialog (main_v->current_project.filename)) {
 *			DEBUG_MSG ("project_close, returned 1, so still continue\n");
 *		} else {
 *			DEBUG_MSG ("project_close, returned 0,   STOP \n");
 *	   	return;
 *		}
 *	}
 */
  g_free (main_v->current_project.basedir);
  main_v->current_project.basedir = NULL;
  g_free (main_v->current_project.webdir);
  main_v->current_project.webdir = NULL;
  g_free (main_v->current_project.template);
  main_v->current_project.template = NULL;
  g_free (main_v->current_project.filename);
  main_v->current_project.filename = NULL;
  free_stringlist (main_v->current_project.fontlist);
  main_v->current_project.fontlist = NULL;
  free_stringlist (main_v->current_project.urllist);
  main_v->current_project.urllist = NULL;
  free_stringlist (main_v->current_project.targetlist);
  main_v->current_project.targetlist = NULL;
  free_stringlist (main_v->current_project.colorlist);
  main_v->current_project.colorlist = NULL;
  free_stringlist (main_v->current_project.dtdlist);
  main_v->current_project.dtdlist = NULL;
  free_stringlist (main_v->current_project.classlist);
  main_v->current_project.classlist = NULL;
  free_stringlist (main_v->current_project.metalist);
  main_v->current_project.metalist = NULL;

  DEBUG_MSG ("project_close, closing files that are in current project\n");
  DEBUG_MSG ("project_close, project filenames(%p)\n", main_v->current_project.filenames);
  tmpdoclist = g_list_first (main_v->documentlist);
  while (tmpdoclist != NULL)
    {
      if (tmpdoclist->data != NULL)
	{
	  main_v->current_document = (filestruct *) tmpdoclist->data;
	  tmpprolist = g_list_first (main_v->current_project.filenames);
	  while (tmpprolist != NULL)
	    {
	      if ((main_v->current_document->filename != NULL) && (tmpprolist->data != NULL))
		{
		  DEBUG_MSG ("project_close, filename(%s) data(%s)\n", main_v->current_document->filename, (char *) tmpprolist->data);
		  if (strcmp (main_v->current_document->filename, (gchar *) tmpprolist->data) == 0)
		    {
		      DEBUG_MSG ("project_close, filename %s can be closed\n", main_v->current_document->filename);
		      tmpdoclist = g_list_previous (tmpdoclist);
		      file_close (NULL, NULL);
		      tmpprolist = g_list_last (tmpprolist);
		    }
		}
	      tmpprolist = g_list_next (tmpprolist);
	    }
	}
      tmpdoclist = g_list_next (tmpdoclist);
    }
  free_stringlist (main_v->current_project.filenames);
  main_v->current_project.modified = 0;
  main_v->current_project.filenames = NULL;
}

void
project_save (GtkWidget * widget, gpointer data)
{

  if (main_v->current_project.filename == NULL)
    {
      project_save_as (NULL, NULL);
    }
  else
    {
      project_to_file ();
    }
}

void
project_save_as (GtkWidget * widget, gpointer data)
{
  gchar *tmpfilename;

  /* change dir into project dir */
  tmpfilename = g_strconcat (g_get_home_dir(), "/.bluefish/projects/", NULL);
  DEBUG_MSG ("project_save_as, change_dir(%s)\n", tmpfilename);
  change_dir (tmpfilename);
  g_free (tmpfilename);

  tmpfilename = return_file (main_v->current_project.filename);
  DEBUG_MSG ("project_save_as, tmpfilename=%s\n", tmpfilename);
  if (tmpfilename != NULL)
    {
      g_free (main_v->current_project.filename);
      main_v->current_project.filename = tmpfilename;
      DEBUG_MSG ("project_save_as, main_v->current_project.filename=%s\n", main_v->current_project.filename);
      project_to_file ();
    }
  else
    {
      /*statusbar message needed !!!!!!!!!!!!!!!!!!!!! */
    }
}

void
project_add_document (GtkWidget * widget, gpointer data)
{

  GList *tmplist;
  filestruct *doc;

	if (GPOINTER_TO_INT (data) == 0) {
		DEBUG_MSG ("project_add_document, adding current file\n");
		if (main_v->current_document->filename != NULL) {
			main_v->current_project.filenames = add_to_stringlist (main_v->current_project.filenames, main_v->current_document->filename);
		} else{
	  /*statusbar message needed !!!!!!!!!!!!!!!!!!!!! */
		}
	} else {
      DEBUG_MSG ("project_add_document, adding ALL files\n");
      tmplist = g_list_first (main_v->documentlist);
      while (tmplist != NULL) {
			doc = (filestruct *) tmplist->data;
	   	if ((doc != NULL) && (doc->filename != NULL)){
				main_v->current_project.filenames = add_to_stringlist (main_v->current_project.filenames, doc->filename);
			}
			tmplist = g_list_next (tmplist);
		}
	}
	main_v->current_project.modified = 1;
}

/***************************************************** 
 *
 *           project edit functions
 *
 *****************************************************/

GtkWidget *
string_entry (gchar * labeltext, GtkWidget * table, gchar * which_config_string, gint left, gint right, gint top, gint bottom)
{

  GtkWidget *label, *tmpentry;

  label = gtk_label_new (labeltext);
  gtk_table_attach_defaults (GTK_TABLE (table), label, left, right, top, bottom - 1);
  tmpentry = gtk_entry_new_with_max_length (1023);
  gtk_table_attach_defaults (GTK_TABLE (table), tmpentry, left, right, top + 1, bottom);
  if (which_config_string != NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (tmpentry), which_config_string);
    }
  return tmpentry;
}

GtkWidget *button_list(gchar *but_title, gint which_list, gchar *tooltip_text, GtkTooltips *tooltips) {
	GtkWidget *return_widget;

  return_widget = gtk_button_new_with_label (but_title);
  GTK_WIDGET_SET_FLAGS (return_widget, GTK_CAN_DEFAULT);
  if (tooltips) gtk_tooltips_set_tip(tooltips, return_widget, tooltip_text, NULL);
  gtk_signal_connect (GTK_OBJECT (return_widget), "clicked",
		  GTK_SIGNAL_FUNC (project_edit_list), GINT_TO_POINTER (which_list));
  return return_widget;
}

void
project_edit (GtkWidget * widget, gpointer data)
{
  GtkWidget *table, *vbox2, *frame, *ok_b, *cancel_b;
  GtkWidget *but_templ;
  GtkTooltips *tooltips;
  EditProject *editproject;
  GList *tmplist;

  editproject = g_malloc (sizeof (EditProject));
  editproject->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
/*   gtk_widget_set_usize (editproject->window, 350, 300);
   gtk_window_set_policy( GTK_WINDOW(editproject->window), TRUE, TRUE, TRUE ); */
  gtk_container_border_width (GTK_CONTAINER (editproject->window), 10);
  gtk_window_set_title (GTK_WINDOW (editproject->window), main_v->current_project.filename);
  gtk_signal_connect (GTK_OBJECT (editproject->window), "destroy",
		      GTK_SIGNAL_FUNC (project_edit_cancel), editproject);
  table = gtk_table_new (14, 6, 0);
  gtk_container_add (GTK_CONTAINER (editproject->window), table);

  tooltips = gtk_tooltips_new();

  editproject->base_entry = string_entry (_("Basedir"), table, main_v->current_project.basedir, 0, 4, 0, 2);
  gtk_tooltips_set_tip(tooltips, editproject->base_entry, _("The root directory of your current website"), NULL);
  editproject->web_entry = string_entry (_("Webdir"), table, main_v->current_project.webdir, 0, 4, 2, 4);
  gtk_tooltips_set_tip(tooltips, editproject->web_entry, _("The URL which is the equivalent of the base directory"), NULL);
  editproject->templ_entry = string_entry ("Template", table, main_v->current_project.template, 0, 4, 4, 6);
  gtk_tooltips_set_tip(tooltips, editproject->templ_entry, _("The template to use when creating a new file"), NULL);

  /* print the filenames */
  frame = gtk_frame_new (_("Files"));
  gtk_container_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults (GTK_TABLE (table), frame, 0, 3, 6, 12);

  vbox2 = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  tmplist = g_list_first (main_v->current_project.filenames);
  while (tmplist != NULL)
    {
      label = gtk_label_new ((gchar *) tmplist->data);
      gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
      tmplist = g_list_next (tmplist);
    }
	 
  gtk_table_attach_defaults (GTK_TABLE (table), 
   			button_list(_("  Edit files  "), 1, _("Edit the files in this project"), tooltips)
				, 0, 3, 12, 13);
  gtk_table_attach_defaults (GTK_TABLE (table), 
   			button_list(_("Edit colors"), 2, _("Edit the colors in this project"), tooltips)
				, 3, 5, 6, 7);
  gtk_table_attach_defaults (GTK_TABLE (table), 
   			button_list(_("Edit fonts"), 3, _("Edit the fonts in this project"), tooltips)
				, 3, 5, 7, 8);
  gtk_table_attach_defaults (GTK_TABLE (table), 
   			button_list(_("Edit URL's"), 4, _("Edit the URL's in this project"), tooltips)
				, 3, 5, 8, 9);
  gtk_table_attach_defaults (GTK_TABLE (table), 
   			button_list(_("Edit targets"), 5, _("Edit the targets in this project"), tooltips)
				, 3, 5, 9, 10);
  gtk_table_attach_defaults (GTK_TABLE (table), 
   			button_list(_("Edit DTD's"), 6, _("Edit the Document Type Definitions in this project"), tooltips)
				, 3, 5, 10, 11);
  gtk_table_attach_defaults (GTK_TABLE (table), 
   			button_list(_("Edit CSS classes"), 7, _("Edit the Cascading Style Sheet classes in this project"), tooltips)
				, 3, 5, 11, 12);
  gtk_table_attach_defaults (GTK_TABLE (table), 
   			button_list(_("Edit META tags"), 8, _("Edit the meta tags in this project"), tooltips)
				, 3, 5, 12, 13);
  ok_b = bf_stock_ok_button (GTK_SIGNAL_FUNC (project_edit_ok), editproject);
  cancel_b = bf_stock_cancel_button ( GTK_SIGNAL_FUNC (project_edit_cancel), editproject);
  GTK_WIDGET_SET_FLAGS (ok_b, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS (cancel_b, GTK_CAN_DEFAULT);
  gtk_window_set_default (GTK_WINDOW (editproject->window), ok_b);
  gtk_table_attach_defaults (GTK_TABLE (table), ok_b, 3, 5, 13, 14);
  gtk_table_attach_defaults (GTK_TABLE (table), cancel_b, 0, 2, 13, 14);

  gtk_widget_realize (editproject->window);
  but_templ = file_but_new (editproject->templ_entry, editproject->window, 1);
  GTK_WIDGET_SET_FLAGS (but_templ, GTK_CAN_DEFAULT);
  gtk_table_attach_defaults (GTK_TABLE (table), but_templ, 4, 5, 5, 6);

  gtk_widget_show_all (editproject->window);
}

void
project_edit_list (GtkWidget * widget, gpointer data)
{
  switch (GPOINTER_TO_INT (data))
    {
    case 1:
      DEBUG_MSG ("project_edit_list, before, main_v->current_project.filenames(%p)\n", main_v->current_project.filenames);
      main_v->current_project.filenames = edit_stringlist (main_v->current_project.filenames, "Bluefish project - files", GPOINTER_TO_INT (data));
      DEBUG_MSG ("project_edit_list, after, main_v->current_project.filenames(%p)\n", main_v->current_project.filenames);
      break;
    case 2:
      main_v->current_project.colorlist = edit_stringlist (main_v->current_project.colorlist, "Bluefish project - colorlist", GPOINTER_TO_INT (data));
      break;
    case 3:
      main_v->current_project.fontlist = edit_stringlist (main_v->current_project.fontlist, "Bluefish project - fontlist", GPOINTER_TO_INT (data));
      break;
    case 4:
      main_v->current_project.urllist = edit_stringlist (main_v->current_project.urllist, "Bluefish project - urllist", GPOINTER_TO_INT (data));
      break;
    case 5:
      main_v->current_project.targetlist = add_to_stringlist (main_v->current_project.targetlist, "_top");
      main_v->current_project.targetlist = add_to_stringlist (main_v->current_project.targetlist, "_blank");
      main_v->current_project.targetlist = add_to_stringlist (main_v->current_project.targetlist, "_self");
      main_v->current_project.targetlist = add_to_stringlist (main_v->current_project.targetlist, "_parent");
      main_v->current_project.targetlist = edit_stringlist (main_v->current_project.targetlist, "Bluefish project - targetlist", GPOINTER_TO_INT (data));
      break;
    case 6:
      main_v->current_project.dtdlist = edit_stringlist (main_v->current_project.dtdlist, "Bluefish project - dtdlist", GPOINTER_TO_INT (data));
      break;
    case 7:
      main_v->current_project.classlist = edit_stringlist (main_v->current_project.classlist, "Bluefish project - classlist", GPOINTER_TO_INT (data));
      break;
    case 8:
      main_v->current_project.metalist = edit_stringlist (main_v->current_project.metalist, "Bluefish project - metalist", GPOINTER_TO_INT (data));
      break;
    default:

      break;
    }
}

void
project_edit_ok (GtkWidget * widget, EditProject * data)
{
  string_apply (&main_v->current_project.basedir, data->base_entry);
  string_apply (&main_v->current_project.webdir, data->web_entry);
  string_apply (&main_v->current_project.template, data->templ_entry);
  DEBUG_MSG ("project_edit_ok, main_v->current_project.webdir(%p)=%s\n",
	     main_v->current_project.webdir, main_v->current_project.webdir);
	main_v->current_project.modified = 1;
  project_edit_cancel (widget, data);
}

void
project_edit_cancel (GtkWidget * widget, EditProject * data)
{
  DEBUG_MSG ("project_edit_cancel, started\n");
  gtk_signal_handlers_destroy (GTK_OBJECT (data->window));
  gtk_widget_destroy (data->window);
  g_free (data);
  DEBUG_MSG ("project_edit_cancel, finished\n");
}


/*******************************************
 *    
 *        other project functions
 *
 ********************************************/
GList *
create_project_list (GList * project_list)
{
  if (project_list == NULL)
    {
      /* this should be done only once !!!!!!!!!!! */
      project_list = make_config_list_item (project_list, &main_v->current_project.basedir, 's', "basedir:");
      project_list = make_config_list_item (project_list, &main_v->current_project.webdir, 's', "webdir:");
      project_list = make_config_list_item (project_list, &main_v->current_project.template, 's', "template:");
      project_list = make_config_list_item (project_list, &main_v->current_project.fontlist, 'l', "fontlist:");
      project_list = make_config_list_item (project_list, &main_v->current_project.urllist, 'l', "urllist:");
      project_list = make_config_list_item (project_list, &main_v->current_project.targetlist, 'l', "targetlist:");
      project_list = make_config_list_item (project_list, &main_v->current_project.colorlist, 'l', "colorlist:");
      project_list = make_config_list_item (project_list, &main_v->current_project.dtdlist, 'l', "dtdlist:");
      project_list = make_config_list_item (project_list, &main_v->current_project.classlist, 'l', "classlist:");
      project_list = make_config_list_item (project_list, &main_v->current_project.metalist, 'l', "metalist:");
      project_list = make_config_list_item (project_list, &main_v->current_project.filenames, 'l', "filenames:");
    }
  return project_list;
}


void
project_from_file (void)
{
  GList *tmplist;
  gchar *tmpstring;

  DEBUG_MSG ("project_from_file, started, checking if project_list exists\n");
  project_list = create_project_list (project_list);

  DEBUG_MSG ("project_from_file, project_list is (or even was) loaded\n");
  if (main_v->current_project.filename == NULL)
    {
      /* can't load it ??!! */
      DEBUG_MSG ("project_from_file, current_project.filename == NULL!! error!!!\n");
      /* there should be a statusbar message here */
      return;
    }

  /* now we should close all open files or not? */
  DEBUG_MSG ("project_from_file, parse config files about to start\n");
  parse_config_file (project_list, main_v->current_project.filename);
  DEBUG_MSG ("project_from_file, parse config files ready\n");
  tmpstring = g_strconcat(CURRENT_VERSION_NAME, " ", main_v->current_project.filename, NULL);
  gtk_window_set_title (GTK_WINDOW (main_v->main_window), tmpstring);
  DEBUG_MSG ("project_from_file, setting title to %s\n", tmpstring);
  g_free(tmpstring);
  /* then we should open the files from the project */
  tmplist = g_list_first (main_v->current_project.filenames);
  while (tmplist != NULL)
    {
      DEBUG_MSG ("project_from_file, opening files in project, tmplist=%p, tmplist->data=%s\n", tmplist, (char *) tmplist->data);
      if (file_exists_and_readable ((char *) tmplist->data))
	{
	  main_v->current_document = new_document ();
	  main_v->current_document->filename = g_strdup ((char *) tmplist->data);
	  DEBUG_MSG ("project_from_file, filename(%p)=%s\n", main_v->current_document->filename, main_v->current_document->filename);
	  DEBUG_MSG ("project_from_file, tmplist=%p, tmplist->data=%s\n", tmplist, (char *) tmplist->data);
	  file_to_textbox (main_v->current_document->filename);
	  gtk_label_set (GTK_LABEL (main_v->current_document->tab_label), strip_filename (main_v->current_document->filename));
	}
      tmplist = g_list_next (tmplist);
    }
  DEBUG_MSG ("project_from_file, opening files in project finished\n");
  main_v->current_project.modified = 0;
  change_dir (main_v->current_project.basedir);
}

void
project_to_file (void)
{
  gchar *tmpstring;

  project_list = create_project_list (project_list);
  DEBUG_MSG ("project_to_file, main_v->current_project.filename(%p)=%s\n", main_v->current_project.filename, main_v->current_project.filename);
  DEBUG_MSG ("project_to_file, project_list(%p)\n", project_list);
  save_config_file (project_list, main_v->current_project.filename);
  tmpstring = g_strconcat(CURRENT_VERSION_NAME, " ", main_v->current_project.filename, NULL);
  gtk_window_set_title (GTK_WINDOW (main_v->main_window), tmpstring);
  g_free(tmpstring);  
	main_v->current_project.modified = 0;
}


/*************************************
 *
 *  end of project functions
 *
 *************************************/

/* view in netscape functions */

void
view_in_netscape (GtkWidget * widget, gpointer data)
{
  gchar *command;
  gint rcode;
  gchar *url, *url2;


  if (main_v->current_document->filename == NULL)
    {
      file_save_as (widget, data);
      if (main_v->current_document->filename == NULL)
	{
	  return;
	}
    }
  else if (main_v->current_document->modified == 1)
    {
      textbox_to_file (main_v->current_document->filename);
    }
  DEBUG_MSG ("view_in_netscape, main_v->current_project.basedir=%p\n", main_v->current_project.basedir);
  if ((main_v->current_project.basedir != NULL)
      && (strlen (main_v->current_project.basedir) > 1)
      && (strlen (main_v->current_project.webdir) > 1))
    {
      /* convert the local filename to an URL (in the project)
         so it works with dynamic content */
      DEBUG_MSG ("view_in_netscape, basedir=%s, webdir=%s, filename=%s\n", main_v->current_project.basedir, main_v->current_project.webdir, main_v->current_document->filename);
      url = strip_common_path (main_v->current_document->filename, main_v->current_project.basedir);
      DEBUG_MSG ("view_in_netscape, url=%s\n", url);
      url2 = g_strconcat (main_v->current_project.webdir, url, NULL);
    }
  else
    {
      url2 = g_strdup (main_v->current_document->filename);
      DEBUG_MSG ("view_in_netscape, filename=%s, url2=%s\n", main_v->current_document->filename, url2);
    }
  DEBUG_MSG ("view_in_netscape, url2=%s\n", url2);
  command = g_strdup_printf (cfg_browser_cline, url2);
  statusbar_message ("Sending URL to browser", 1000);
  rcode = system (command);
  g_free (command);
  DEBUG_MSG ("view_in_netscape, rcode = %d\n", rcode);
  if (rcode == 0)
    {
      goto out_free;
      /*everything was ok */
    }
  else
    {
      statusbar_message (_("Browser is not running, starting browser"), 2000);
      /* this part is now only valid for netscape .... */
      command = g_strdup_printf ("netscape %s &", url2);
      DEBUG_MSG ("view_in_netscape, command=%s\n", command);
      system (command);
      g_free (command);
    }
out_free:
  g_free (url2);
}


/* ---------------------------------------------------------------
 * the functions below are only declared in this file, so they are NOT known if you use callbacks.h 
 * ---------------------------------------------------------------- */

void
error_dialog (gchar * window_title, gchar * error_string)
{

  dialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), window_title);
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_container_border_width (GTK_CONTAINER (dialog), 15);
  /* gtk_widget_set_usize(dialog, xsize, ysize); */

  okbutton = gtk_button_new_with_label (_("OK"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		      okbutton, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (okbutton, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (okbutton);
  gtk_signal_connect_object (GTK_OBJECT (okbutton), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_widget_show (okbutton);

  label = gtk_label_new (error_string);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE,
		      TRUE, 0);
  gtk_widget_show (label);
  gtk_widget_show (dialog);
}

void
open_from_web ()
{
  create_dialog (450, 80, _("Open from Web"));
  dialogtable = gtk_table_new (1, 10, 0);
  gtk_container_border_width (GTK_CONTAINER (vbox), 0);
  gtk_box_pack_start (GTK_BOX (vbox), dialogtable, FALSE, FALSE, 0);
  gtk_widget_show (dialogtable);

  label = gtk_label_new (_("URL:"));
  combo1 = gtk_combo_new ();
  gtk_widget_set_usize (combo1, 350, 80);
  if (main_v->current_project.urllist)
    {
      gtk_combo_set_popdown_strings (GTK_COMBO (combo1), main_v->current_project.urllist);
    }
  gtk_table_attach_defaults (GTK_TABLE (dialogtable), label, 0, 1, 0, 1);
  gtk_table_attach_defaults (GTK_TABLE (dialogtable), combo1,
			     1, 10,
			     0, 1);
  gtk_widget_show (label);
  gtk_widget_show (combo1);

  create_buttons (GTK_SIGNAL_FUNC(open_from_web_callback), NULL);
}

void
open_from_web_callback ()
{
  gint return_code;
  gint signal_code;
  gchar *url_string;

  main_v->current_project.urllist = add_to_stringlist (main_v->current_project.urllist, gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (combo1)->entry)));
  url_string = g_strdup_printf ("%s", gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (combo1)->entry)));
  gtk_widget_destroy (dialog);

  /* Bind escape key to cancel d/l */

  signal_code = gtk_signal_connect (GTK_OBJECT (main_v->main_window),
				    "key_press_event",
				GTK_SIGNAL_FUNC (key_press_callback), NULL);

  return_code = open_web (url_string);
  switch (return_code)
    {
    case 0:
      break;
    case 1:
      error_dialog (_("Error"),
	 _("That looks like a secure server,\nBluefish doesn't support them"));
      break;
    case 2:
      error_dialog (_("Error"), _("Invalid address"));
      break;
    case 3:
      error_dialog (_("Error"), _("Host name lookup failure"));
      break;
    case 4:
      error_dialog (_("Error"), _("Couldn't create socket"));
      break;
    case 5:
      error_dialog (_("Error"), _("The network is unreachable"));
      break;
    case 6:
      error_dialog (_("Error"), _("Can't send data"));
      break;
    case 7:
      error_dialog (_("Error"), _("Didn't send all the data"));
      break;
    case 8:
      error_dialog (_("Error"), _("Can't open file"));
      break;
    case 9:
/* The used cancelled the d/l, do nothing */
      break;
    default:
      error_dialog (_("Error"), _("Unknown error"));
      break;
    }
  g_free (url_string);
  DEBUG_MSG ("open_from_web_callback, disconnecting key_press_callback()\n");
  gtk_signal_disconnect (GTK_OBJECT (main_v->main_window), signal_code);
}

void
key_press_callback (GtkWidget * widget, GdkEventKey * event, gpointer data)
{
  DEBUG_MSG ("key_press_callback, Value is .%d.\n", event->keyval);
  /* Escape */
  if (event->keyval == GDK_Escape)
    {
      closed = 1;
      close (sockfd);
#if HAVE_GTKEDITOR
      gtk_sctext_thaw (GTK_SCTEXT (main_v->current_document->textbox));
#else
      gtk_text_thaw (GTK_TEXT (main_v->current_document->textbox));
#endif
      DEBUG_MSG ("key_press_callback, socket closed and document thawed\n");
    }
}

/* warn if you really want to close a changed file 

 * the filename is only used for the text in the warning
 * it returns 1 if you can continue closing
 * it returns 0 if you have to stop closing
 */
gint
warn_close_dialog (gchar * filename)
{
  GtkWidget *label1, *button_ok, *button_cancel, *button_save;
  gint sigid, returnval;
  Warn_Close_Struct *warn_close;

  warn_close = g_malloc (sizeof (Warn_Close_Struct));
  warn_close->dialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (warn_close->dialog), _("Confirmation"));
  gtk_container_border_width (GTK_CONTAINER (warn_close->dialog), 10);
  gtk_grab_add (warn_close->dialog);
  sigid = gtk_signal_connect (GTK_OBJECT (warn_close->dialog), "delete_event",
			      GTK_SIGNAL_FUNC (file_close_cancel),
			      warn_close);
  gtk_window_position (GTK_WINDOW (warn_close->dialog), GTK_WIN_POS_MOUSE);
  button_ok = bf_stock_ok_button (GTK_SIGNAL_FUNC (file_close_ok),
		  (gpointer) warn_close);
  button_save = gtk_button_new_with_label (_("Save & close"));
  button_cancel = bf_stock_cancel_button (GTK_SIGNAL_FUNC (file_close_cancel),
		  (gpointer) warn_close);
  GTK_WIDGET_SET_FLAGS (button_ok, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS (button_save, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS (button_cancel, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (warn_close->dialog)->action_area), button_save,
		      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (warn_close->dialog)->action_area), button_ok,
		      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (warn_close->dialog)->action_area), button_cancel,
		      TRUE, TRUE, 0);
  gtk_widget_show (button_ok);
  gtk_widget_show (button_save);
  gtk_widget_show (button_cancel);
  gtk_signal_connect (GTK_OBJECT (button_save), "clicked",
		      GTK_SIGNAL_FUNC (file_close_save),
		      (gpointer) warn_close);
  if (filename != NULL)
    {
      label1 = gtk_label_new (filename);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (warn_close->dialog)->vbox), label1, TRUE,
			  TRUE, 0);
      gtk_widget_show (label1);
    }
  label1 = gtk_label_new
    (_("This file is modified, are you sure you want to close it?"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (warn_close->dialog)->vbox), label1, TRUE,
		      TRUE, 0);
  gtk_widget_show (label1);
  gtk_widget_show (warn_close->dialog);
  gtk_window_set_default (GTK_WINDOW (warn_close->dialog), button_save);

  warn_close->still_continue = 0;
  DEBUG_MSG ("warn_close_dialog, a new gtk_main started, still_continue=%d\n", warn_close->still_continue);
  gtk_main ();
  DEBUG_MSG ("warn_close_dialog, warn_close=%p, dialog=%p\n", warn_close, warn_close->dialog);
  DEBUG_MSG ("warn_close_dialog, the new gtk_main stopped, still_continue=%d\n", warn_close->still_continue);
  gtk_signal_disconnect (GTK_OBJECT (warn_close->dialog), sigid);
  gtk_grab_remove (warn_close->dialog);
  gtk_object_destroy (GTK_OBJECT (warn_close->dialog));
  returnval = warn_close->still_continue;
  g_free (warn_close);
  return returnval;
}

/* cancel the closing of files */
void
file_close_cancel (GtkWidget * button, Warn_Close_Struct * windowname)
{
  windowname->still_continue = 0;
  DEBUG_MSG ("file_close_cancel, still_continue=%d\n", windowname->still_continue);
  gtk_main_quit ();
}

/* close the file even if it's modified */
void
file_close_ok (GtkWidget * button, Warn_Close_Struct * windowname)
{
  DEBUG_MSG ("file_close_ok, begin\n");
  windowname->still_continue = 1;
  gtk_main_quit ();
  DEBUG_MSG ("file_close_ok, windowname=%p, dialog=%p\n", windowname, windowname->dialog);
  DEBUG_MSG ("file_close_ok, end, still_continue=%d\n", windowname->still_continue);
}

/* close the file even if it's modified, but save it first */
void
file_close_save (GtkWidget * button, Warn_Close_Struct * windowname)
{

  if (main_v->current_document->filename == NULL)
    {
      DEBUG_MSG ("file_close_save, No filename known --> return_file()..\n");
      main_v->current_document->filename = return_file (NULL);
    }
  if (main_v->current_document->filename == NULL)
    {
      windowname->still_continue = 0;
    }
  else
    {
      DEBUG_MSG ("file_close_save, Saving to %s\n", main_v->current_document->filename);
      textbox_to_file (main_v->current_document->filename);
      windowname->still_continue = 1;
    }
  gtk_main_quit ();
  DEBUG_MSG ("file_close_save, end, still_continue=%d\n", windowname->still_continue);
}

void
file_to_textbox (gchar * filename)
{
  FILE *fd;
  gchar *errmessage, line[512];

  statusbar_message (_("Opening file"), 1000);
  /* This opens the contents of a file to a textbox */
  change_dir (filename);
  fd = fopen (filename, "r");
  if (fd == NULL)
    {
      DEBUG_MSG ("file_to_textbox, cannot open file %s\n", filename);
      errmessage = g_strconcat (_("Could not open file:\n"), filename, NULL);
      error_dialog (_("Error"), errmessage);	/* 7 */
      g_free (errmessage);
      return;
    }
  else
    {

#if HAVE_GTKEDITOR
      gtk_sctext_freeze (GTK_SCTEXT (main_v->current_document->textbox));
#else
      gtk_text_freeze (GTK_TEXT (main_v->current_document->textbox));
#endif
      while (fgets (line, 512, fd) != NULL)
	{
#if HAVE_GTKEDITOR
	  gtk_sctext_insert (GTK_SCTEXT (main_v->current_document->textbox),
			     NULL, NULL, NULL, line, -1);
#else
	  gtk_text_insert (GTK_TEXT (main_v->current_document->textbox),
			   NULL, NULL, NULL, line, -1);
#endif
	}
#if HAVE_GTKEDITOR
      gtk_sctext_thaw (GTK_SCTEXT (main_v->current_document->textbox));
      gtk_editor_hilite_screen (GTK_EDITOR (main_v->current_document->textbox));
#else
      gtk_text_thaw (GTK_TEXT (main_v->current_document->textbox));
#endif
      fclose (fd);
      main_v->current_document->modified = FALSE;
    }
}

void
textbox_to_file (gchar * filename)
{
  FILE *fd;
  gchar *tmpchar, *errmessage = NULL;

  statusbar_message (_("Saving file"), 1000);
  /* This writes the contents of a textbox to a file */
  change_dir (filename);
  fd = fopen (filename, "w");
  if (fd == NULL)
    {
      DEBUG_MSG ("textbox_to_file, cannot open file %s\n", filename);
      errmessage = g_strconcat (_("Could not write file:\n"), filename, NULL);
      error_dialog (_("Error"), errmessage);
      g_free (errmessage);
      return;
    }
  else
    {
      DEBUG_MSG ("textbox_to_file, lenght=%d\n",
	gtk_text_get_length (GTK_TEXT (main_v->current_document->textbox)));
      DEBUG_MSG ("textbox_to_file, fd=%p\n", fd);
      DEBUG_MSG ("textbox_to_file, filename=%s\n", filename);

#if HAVE_GTKEDITOR
      tmpchar = gtk_editable_get_chars (GTK_EDITABLE (main_v->current_document->textbox),
					0, gtk_sctext_get_length (GTK_SCTEXT (main_v->current_document->textbox)));
#else
      tmpchar = gtk_editable_get_chars (GTK_EDITABLE (main_v->current_document->textbox),
      0, gtk_text_get_length (GTK_TEXT (main_v->current_document->textbox)));
#endif
      fputs (tmpchar, fd);
      g_free (tmpchar);
      fclose (fd);
      main_v->current_document->modified = 0;
      DEBUG_MSG ("textbox_to file, text saved to: %s\n", filename);
    }
}


void
re_check (GtkWidget * widget, GtkWidget * window)
{
  gtk_widget_destroy (window);
  run_weblint (NULL, NULL);
}

static void
scrollbar_update (GtkText * txt, gint tl, gint ln)
{
  gfloat value;

  if (tl < 3 || ln > tl)
    return;

  value = (ln * GTK_ADJUSTMENT (txt->vadj)->upper) /
    tl - GTK_ADJUSTMENT (txt->vadj)->page_increment;

  gtk_adjustment_set_value (GTK_ADJUSTMENT (txt->vadj), value);
}

void go_to_line_win(GtkWidget *widget, gpointer data) {
  create_dialog (450, 80, _("Goto line"));
  dialogtable = gtk_table_new (1, 5, 0);
  gtk_container_border_width (GTK_CONTAINER (vbox), 0);
  gtk_box_pack_start (GTK_BOX (vbox), dialogtable, FALSE, FALSE, 0);
  gtk_widget_show (dialogtable);
  
  textentry1 = gtk_entry_new();
  gtk_table_attach_defaults (GTK_TABLE (dialogtable), gtk_label_new(_("Line number: ")), 0, 1, 0, 1);
  gtk_table_attach_defaults (GTK_TABLE (dialogtable), textentry1, 1, 10, 0, 1);
  create_buttons (GTK_SIGNAL_FUNC(go_to_line_cb), NULL);  
}

void go_to_line_cb(GtkWidget *widget, gpointer data) {
	gchar *linestr;
	gint linenum;

  linestr = g_strdup_printf ("%s", (gchar *)gtk_entry_get_text (GTK_ENTRY(textentry1)));
  gtk_widget_destroy (dialog);
  
  linenum = atoi(linestr);
  DEBUG_MSG ("go_to_line_cb, going to line %d (string:%s)\n", linenum, linestr);  
  go_to_line(linenum);
}

static void
go_to_line (gint linenum)
{

/* Thanks to Andy Kahn for the amazing non-leaking go to line function
   It dances, it does tricks, AND you have memory left after... :) */

  gchar *buf, *haystack, *needle;
  gint numlines;
  gint a, b, len;

  numlines = 1;

#if HAVE_GTKEDITOR
  len = gtk_sctext_get_length (GTK_SCTEXT (main_v->current_document->textbox));
#else
  len = gtk_text_get_length (GTK_TEXT (main_v->current_document->textbox));
#endif
  buf = gtk_editable_get_chars (GTK_EDITABLE (main_v->current_document->textbox),
				1, len);

/*  If index is past the end, or 0 (Usually because line number is -) */

  if ((linenum > (len - 1)) || (linenum == 0))
    return;

  a = 1;
  b = len;
  haystack = buf;
  do
    {
      needle = strchr (haystack, '\n');
      if (needle)
	{
	  haystack = needle + 1;
	  if (linenum == numlines)
	    b = needle - buf + 1;
	  numlines++;
	  if (linenum == numlines)
	    a = needle - buf + 1;
	}
    }
  while (needle != NULL);

  g_free (buf);
  gtk_editable_select_region (GTK_EDITABLE (main_v->current_document->textbox),
			      a, b);
  scrollbar_update (GTK_TEXT (main_v->current_document->textbox), numlines, linenum);
  gtk_editable_insert_text (GTK_EDITABLE ((GtkText *)
					  main_v->current_document->textbox),
			    " ", 1, &b);
  gtk_editable_delete_text (GTK_EDITABLE ((GtkText *)
				     main_v->current_document->textbox), b -
			    1, b);
}


void
selection_made (GtkWidget * clist, gint row, gint column,
		GdkEventButton * event, gpointer data)
{

/*  Every time a list item is selected or unselected this checks
   if it was also a double or triple click, if so goToLine is called */

  gchar *text;
  gchar *bracket_ptr;
  gint line_number;
  gtk_clist_get_text (GTK_CLIST (clist), row, column, &text);

  if (event->type == GDK_2BUTTON_PRESS || event->type == GDK_3BUTTON_PRESS)
    {

/*  Find first bracket, increment, atoi rest of string, atoi will
   stop at second bracket (we hope) */

      bracket_ptr = strchr (text, '(');
      bracket_ptr++;

/* Mark that in the record books, Neil uses a pointer increment! */

      line_number = atoi (bracket_ptr);
      go_to_line (line_number);
    }
}

void
run_weblint (GtkWidget * widget, gpointer data)
{
  gint rcode, row, code;
  GtkWidget *ext_window, *list, *close_button, *recheck_button;
  GtkWidget *scrolwin;
  gchar *temp, *html_file, *output_file, *command;
  gchar *text[1];
  FILE *fd;
  struct stat st_output_file;

  output_file = g_strdup_printf ("%s", "weblint.out");

/* Current HTML file saved to filename 'untitled' */

  html_file = g_strdup_printf ("%s", "untitled");

  statusbar_message (_("Running Weblint"), 1000);

/* Main Window */

  ext_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect (GTK_OBJECT (ext_window), "destroy",
		      NULL, &ext_window);
  gtk_signal_connect (GTK_OBJECT (ext_window), "delete_event",
		      NULL, &ext_window);
  gtk_window_set_title (GTK_WINDOW (ext_window), _("Weblint output"));
  gtk_container_border_width (GTK_CONTAINER (ext_window), 10);
  gtk_window_set_policy (GTK_WINDOW (ext_window), TRUE, TRUE, FALSE);
  gtk_widget_set_usize (ext_window, 550, 350);

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (ext_window), vbox);

  /* since GTK 1.2 one should use a scrolledwindow for a clist */
  scrolwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolwin),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (vbox), scrolwin, TRUE, TRUE, 0);
  gtk_widget_show (scrolwin);
  list = gtk_clist_new (1);
  gtk_clist_set_shadow_type (GTK_CLIST (list), GTK_SHADOW_OUT);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scrolwin), list);
  gtk_widget_show (list);
  gtk_widget_show (vbox);
  flush_queue ();

  gtk_signal_connect (GTK_OBJECT (list), "select_row",
		      GTK_SIGNAL_FUNC (selection_made), NULL);

  gtk_signal_connect (GTK_OBJECT (list), "unselect_row", GTK_SIGNAL_FUNC
		      (selection_made), NULL);

/*      Button box */

  hbox = gtk_hbox_new (TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);

  recheck_button = gtk_button_new_with_label (_("Recheck"));
  gtk_box_pack_start (GTK_BOX (hbox), recheck_button, TRUE, TRUE, 0);
  gtk_widget_show (recheck_button);

  gtk_signal_connect (GTK_OBJECT (recheck_button), "clicked",
		      GTK_SIGNAL_FUNC (re_check), GTK_OBJECT (ext_window));

  close_button = gtk_button_new_with_label (_("Close"));
  gtk_box_pack_start (GTK_BOX (hbox), close_button, TRUE, TRUE, 0);

/*  GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
   gtk_widget_grab_default (close_button); */

  gtk_widget_show (close_button);

  gtk_signal_connect_object (GTK_OBJECT (close_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (ext_window));

  if (cfg_weblint_cline == NULL)
    {
      command = DEFAULT_WEBLINT;
    }
  else
    {
      command = cfg_weblint_cline;
    }

  /* Exec Weblint */

  textbox_to_file (html_file);
  temp = g_strdup_printf ("%s %s > %s 2>/dev/null", command, html_file, output_file);
  flush_queue ();

  DEBUG_MSG ("Weblint command: %s\n", temp);
  rcode = system (temp);
  flush_queue ();

  if (rcode == 32512 || rcode == 32256)
    {

/*  return code == 32512, seems to be 'no weblint.' Tell user to
   install brainModule */
      gtk_widget_destroy (ext_window);
      error_dialog (_("Weblint"),
		    _("You don't appear to have weblint installed \nTo use this option you will need to download weblint from:\nhttp://www.cre.canon.co.uk/~neilb/weblint/"));

    }
  else
    {
      code = stat (output_file, &st_output_file);
      if (code == -1) {
	DEBUG_MSG ("Error stat file: %s\n", output_file);
	error_dialog(_("Weblint"), _("Error stat file"));
      } else {
	if (st_output_file.st_size) {
	fd = fopen (output_file, "r");
	if (fd != NULL)	{
	  DEBUG_MSG ("Open output file: %s\n", output_file);
	  text[0] = g_malloc (512);
	  gtk_clist_freeze (GTK_CLIST (list));
	  while (fgets (text[0], 512, fd) != NULL) {
	    DEBUG_MSG ("Message: %s\n", text[0]);
	    row = gtk_clist_append (GTK_CLIST (list), text);
	  }
	  gtk_clist_thaw (GTK_CLIST (list));
	  g_free (text[0]);
	  fclose (fd);
	};

	code = unlink (output_file);
	if (code == -1)
	  {
		DEBUG_MSG ("Cannot delete file %s\n", output_file);
		error_dialog(_("Weblint"), _("Error delete stdout temporary file"));
	  }

	};
      };

      flush_queue ();
      gtk_widget_show (ext_window);
    }

  code = unlink (html_file);
  if (code == -1)
    {
	    DEBUG_MSG ("Cannot delete file %s\n", html_file);
	    error_dialog(_("Weblint"), _("Error delete temporary saved file"));
    }
  
  g_free (temp);
  g_free (html_file);
  g_free (output_file);
}
