/*
 * The contents of this file are subject to the AOLserver Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://aolserver.com/.
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is AOLserver Code and related documentation
 * distributed by AOL.
 * 
 * The Initial Developer of the Original Code is America Online,
 * Inc. Portions created by AOL are Copyright (C) 1999 America Online,
 * Inc. All Rights Reserved.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU General Public License (the "GPL"), in which case the
 * provisions of GPL are applicable instead of those above.  If you wish
 * to allow use of your version of this file only under the terms of the
 * GPL and not to allow others to use your version of this file under the
 * License, indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by the GPL.
 * If you do not delete the provisions above, a recipient may use your
 * version of this file under either the License or the GPL.
 */

/*
 * nssybpd.c --
 * 
 * This file contains the Sybase-specific routines to be linked with  
 * the 'nspdmain' program, found in the libnspd.a library.  The resulting 
 * "database proxy daemon" is contacted by the server's "external" driver 
 * for all database activities. The 'nspdmain' program handles all
 * communication with the server's 'external' driver, calling routines
 * in this file for database-specific functions.  The 'Ns_PdDbInit' and 
 * 'Ns_PdDbCleanup' routines are called once from 'nspdmain' at process 
 * startup and shutdown, respectively.
 *
 * Note that there is a single database proxy daemon associated
 * with each database connection, so all of the state for each connection
 * is encapsulated here.  Database connections are managed efficiently by the
 * server's database pool mechanism.
 *
 * Function naming conventions:
 *     Ns_PdDbxxx    -called directly from nspdmain
 *     Sybxxx	-called by Dbxxx routines
 *     syb_xxx	-support routines 
 */

static const char *RCSID = "@(#) $Header: /cvsroot/aolserver/aolserver3/nssybpd/nssybpd.c,v 1.5 2000/04/01 12:43:14 petej Exp $, compiled: " __DATE__ " " __TIME__;

#include "nsextmsg.h"
#include "nspd.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "ctpublic.h"
#include <string.h>
#include <errno.h>
#include <signal.h>
#include "exutils.h"

/*
 * Manifest constants
 */
#define SYBASE_TYPES	"text real datetime tinyint smallint integer bit"
#define IDENTITY_STRING "Sybase Proxy Daemon v1.1"
#define SYB_CTLIB_VERSION	CS_VERSION_100
#define SYB_CMDBUFSIZE		1024
#define MAXERRMSG 2048
#define OID_SIZE    50
#define INIT_SQLBUFSIZE 4096
#define ERRBUF_SIZE 4096

/*
 * This message code from the Sybase server is ignored
 */
#define DBCONTEXT_CHANGE 5701

/*
 * Structure definitions
 */

/*
 * Structure to hold the definition and value for a column of data
 * resulting from a Sybase query.
 */
typedef struct SybColumn {
    CS_DATAFMT	datafmt;
    CS_SMALLINT indicator;
    CS_CHAR	*value;
    CS_INT	valuelen;
    CS_INT	bufferlen;
} SybColumn;

/*
 * Structure to hold the data in a row resulting from a Sybase query
 */
typedef struct SybRow {
    int		nrColumnsAllocated;
    int		nrColumnsUsed;
    SybColumn	columns[1];
} SybRow;

/*
 * Structure to maintain the state of the proxy, as well as the state
 * of the Sybase connection.
 */
typedef struct SybState {
    CS_CONNECTION	*conn;
    CS_COMMAND		*cmd;
    int			connected;
    int			fetchingRows;
    int			resultsPending;
    CS_CONTEXT		*cs_context;
    char		*datasource;
    char		*user;
    char		*password;
    char		*param;
    char		*sqlbuf;
    int			sqlbufsize;
    SybRow		*dataRow;
    int			dataRowCount;
    int			dataRowsRead;
    int			spReturnCode;
    SybRow		*spParamRow;
    char		exceptionCode[EXCEPTION_CODE_MAX]; 
    char		exceptionMsg[EXCEPTION_MSG_MAX];
    int			squelchError;
    int			transactionState;
} SybState;

/*
 * Predeclarations of functions local to this file
 */

static Ns_PdRowInfo *
Ns_PdNewRowInfoFromColumnNames(SybRow *row);

static Ns_PdRowInfo *
Ns_PdNewRowInfoFromColumnValueRefs(SybRow *row);

static Ns_PdRowInfo *
Ns_PdNewRowInfoFromColumnValueCopy(SybRow *row);

static void
strTrimTrailingSpace(char *str);

static int
SybCancel(SybState *state);

static int
SybExec(SybState *state, char *sql);

static SybRow *
SybRow_dup (SybRow *row);

static void
SybRow_free (SybRow *row);

static SybRow *
SybRow_new (int columnCount);

static SybRow *
SybRow_newFromResult (CS_COMMAND *cmd, CS_INT forceDataType);

static int
syb_check_transaction_state (CS_COMMAND *cmd);

static CS_RETCODE CS_PUBLIC
syb_clientmsg_cb(CS_CONTEXT *context, CS_CONNECTION *connection, 
		 CS_CLIENTMSG	*errmsg);

static CS_RETCODE CS_PUBLIC
syb_con_cleanup(CS_CONNECTION *connection, CS_RETCODE status);

static CS_RETCODE CS_PUBLIC
syb_connect(CS_CONTEXT *context, CS_CONNECTION **connection, CS_CHAR *appname, 
	    CS_CHAR *username, CS_CHAR *password, CS_CHAR *server);

static CS_INT CS_PUBLIC
syb_display_dlen(CS_DATAFMT *column);

static void
syb_drain_results (CS_COMMAND *cmd);

static void
syb_drain_rows (CS_COMMAND *cmd);

static CS_RETCODE CS_PUBLIC
syb_execute_cmd(CS_CONNECTION *connection, CS_CHAR *cmdbuf);

static CS_RETCODE
syb_fetch_next_result (
		       CS_COMMAND		*cmd,
		       SybRow		**pDataRow,
		       int			*rowCount,
		       int			*spResult,
		       SybRow		**pSpParamRow,
		       int			*cmd_done_received,
		       int			*cmd_succeed_received,
		       int			*cmd_fail_received,
		       int			*transactionState,
		       int			*spResultsFetched
		       );

static CS_RETCODE
syb_fetch_row (CS_COMMAND *cmd, CS_INT *rowsFetched, SybRow *row);

static CS_RETCODE CS_PUBLIC
syb_init(SybState *state);

static CS_RETCODE CS_PUBLIC
syb_servermsg_cb(CS_CONTEXT *context, CS_CONNECTION *connection, 
		 CS_SERVERMSG	*srvmsg);

CS_RETCODE
syb_use_db(CS_CONNECTION *connection, char *dbname);

/*
 * Variables local to this file
 */

/*
 * There are no variables local to this file.
 */

/*
 * Variables exported by this file
 */

/*
 * This variable appears to be needed by the proxy support routines
 * with which this file is linked to produce the daemon.
 */
CS_CHAR *Appname  = "AOLserver Sybase Proxy Daemon";

/*
 *==========================================================================
 * API functions
 *==========================================================================
 */

/*
 * There are no API functions defined in this file.
 */

/* 
 *==========================================================================
 * Exported functions
 *==========================================================================
 */

/*
 *----------------------------------------------------------------------
 * main -
 *
 *  The main routine for the proxy daemon, which calls Ns_PdMain and,
 *  upon its termination, exits.
 *
 * Results:
 *  This function does not return. (The return value is there to prevent
 *  the compiler from complaining).
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
 
int
main(int argc, char **argv)
{
    int exitStatus = 0;
    /*
     * 'nspdmain' handles all communication with the external driver
     * and calls database-specific functions in this file to handle requests.
     */
    if (Ns_PdMain(argc, argv)) {
	exitStatus = 1;
    }
    Ns_PdLog(Notice, "Exiting.");
    exit(exitStatus);
    return 0;
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbBestRowID -
 *
 *  Determine the correct column to use as a row ID.  This is usually
 *  the single-column primary key, if one exists.  The current AOLserver
 *  database proxy protocol does not permit multi-column keys.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  This runs a complete query, so any pending-but-uncollected results from
 *  the Sybase server are cancelled.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbBestRowId(void *handle, char *tableName)
{
    SybState	*state = (SybState *)handle;
    int		sentColumn = 0;
    int		status = NS_ERROR;
    CS_RETCODE	returnCode;
    CS_INT	rowsFetched;
    char	*index_description;
    char	*index_keys;
    char	*index_name;

    Ns_PdLog(Trace, "bestrowid(%s):", tableName);

    /* 
     * allows the edit of tables with adhoc when primary and foreign keys are
     * defined.
     */
    sprintf(state->sqlbuf, "sp_helpindex %s", tableName);
    Ns_PdSendString(OK_STATUS);
    if ((status = SybExec(state, state->sqlbuf)) == DB_ROWS) {
	returnCode = CS_SUCCEED;
	index_name = state->dataRow->columns[0].value;
	index_description = state->dataRow->columns[1].value;
	index_keys = state->dataRow->columns[2].value;
	while (returnCode == CS_SUCCEED || returnCode == CS_ROW_FAIL) {
	    returnCode = syb_fetch_row(state->cmd,
				       &rowsFetched, state->dataRow);
	    if (returnCode == CS_SUCCEED || returnCode == CS_ROW_FAIL) {
		Ns_PdLog(Trace, "Ns_PdBestRowID: Checking index %s: %s: %s",
			 index_name, index_description, index_keys);
		/*
		 * Locate the first row for which column 1 (the
		 * index description) contains the word "unique", and
		 * for which column 2 (the index key list) contains no
		 * commas, indicating a single-column key (because
		 * the AOLserver db scripts cannot handle a
		 * multi-column key (yet))
		 */
		if (strstr(index_description, "unique") != NULL &&
		    strchr(index_keys, ',') == NULL) {
		    index_keys = state->dataRow->columns[2].value;
		    /* Trim leading space from column name */
		    while ((*index_keys == ' ') && (*index_keys != '\0')) {
			index_keys++;
		    }
		    /*
		     * Return this one and flush the rest of the rows.
		     */
		    Ns_PdSendString(index_keys);
		    sentColumn = 1;
		    syb_drain_rows(state->cmd);
		    break;
		}
		
	    }
	}
	state->fetchingRows = 0;
	syb_drain_results(state->cmd);
	state->resultsPending = 0;
    }
    if (sentColumn != 1) {
	Ns_PdSendString("__nobestrowid__");
    }

}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbBindRow -
 *
 *  Extract the names of the columns in the current Sybase result and
 *  send them to the AOLserver.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbBindRow(void *handle)
{
    SybState		*state = (SybState *)handle;
    Ns_PdRowInfo	*rowInfo = NULL;

    Ns_PdLog(Trace, "bindrow:");

    if (state->dataRow != NULL) {
	rowInfo = Ns_PdNewRowInfoFromColumnNames(state->dataRow);
    }
    if (rowInfo == NULL) {
	if (strlen(state->exceptionCode) == 0 
	    && strlen(state->exceptionMsg) == 0) {
	    strcpy(state->exceptionCode, SILENT_ERROR_STATUS);
	}
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
    } else {
	Ns_PdSendString(OK_STATUS);
	Ns_PdSendRowInfo(rowInfo);
	Ns_PdFreeRowInfo(rowInfo, 0);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbCancel -
 *
 *  Cancel pending-but-uncollected results from the Sybase server,
 *  and notify the AOLserver of the results of the cancellation.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Pending-but-uncollected results from the Sybase server are cancelled.
 *  The Sybase connection should then be in a state from which new commands
 *  can be issued.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbCancel(void *handle)
{
    SybState *state = (SybState *)handle;
    int status;
    Ns_PdLog(Trace, "cancel:");
    if (SybCancel(state) == NS_ERROR) {
	Ns_PdSendString("DbCancel error");
    } else {
	Ns_PdSendString(OK_STATUS);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbCleanup -
 *
 *  Perform cleanup of the Sybase 
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Any open Sybase connection is closed, and the state buffer is *
 *  destroyed.
 *
 *----------------------------------------------------------------------
 */
void
Ns_PdDbCleanup(void *handle)
{
    SybState *state = (SybState *)handle;
    if (handle != NULL) {
	if (state->connected) {
	    Ns_PdDbClose(handle);
	}
	free(state->sqlbuf);
	free(state);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbClose -
 *
 *  Terminate the open Sybase connection.  Cancel any pending-but-uncollected
 *  results and roll back any open transactions.  In theory, this should mean
 *  that the daemon could then re-open a Sybase connection, but I do
 *  not think all of the state is managed properly to permit that,
 *  so I think this process always terminates after closing a connection.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Any pending-but-uncollected results from the Sybase server are cancelled.
 *  Any open transactions are rolled back.  The Sybase connection is closed
 *  and dropped.  The state structure is cleared appropriately.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbClose(void *handle)
{
    SybState *state = (SybState *)handle;
    int status = NS_OK;
    CS_RETCODE	    retcode;
    char errbuf[ERRBUF_SIZE];
    int i;
    Ns_PdLog(Trace, "close:"); 

    errbuf[0] = '\0';
    if (state->connected) {
	Ns_PdLog(Notice, "Closing database connection: %s", state->datasource);

	if (SybCancel(state) != NS_OK) {
	    sprintf(errbuf, "DbClose: SybCancel failed");
	    status = NS_ERROR;
	} else if (ct_cmd_drop(state->cmd) != CS_SUCCEED) {
	    status = NS_ERROR;
	    sprintf(errbuf, "DbClose: "
		    "ct_cmd_drop() failed closing connection: %s",
		    state->datasource);
	    Ns_PdLog(Error, errbuf);
	} else if (state->conn != NULL) {
	    if (syb_con_cleanup(state->conn, CS_FORCE_CLOSE) != CS_SUCCEED) {
		status = NS_ERROR;
		sprintf(errbuf, "DbClose: connection cleanup failed"); 
		Ns_PdLog(Error, errbuf);
	    }
	}
	state->connected = 0;
	if (state->datasource != NULL) free(state->datasource);
	if (state->user != NULL) free(state->user);
	if (state->password != NULL) free(state->password);
	if (state->param != NULL) free(state->param);
	free(state->conn);
	state->conn = NULL;
	if (state->dataRow != NULL) {
	    SybRow_free(state->dataRow);
	    state->dataRow = NULL;
	}
	if (state->spParamRow != NULL) {
	    SybRow_free(state->spParamRow);
	    state->spParamRow = NULL;
	}
    } else {
	Ns_PdLog(Notice, "DbClose: not connected.");
    }
    status == NS_OK ? Ns_PdSendString(OK_STATUS) : Ns_PdSendString(errbuf);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbExec -
 *
 *  Run the currently prepared Sybase command.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  The command is executed appropriately.  If the command execution
 *  results in a problem, then an exception message will be
 *  transmitted to the AOLserver; if command executioin succeed, the
 *  AOLserver is sent a message indicating whether the result contains
 *  rows.  Sybase result processing is started, which will stop with a
 *  result with multiple rows is reached, or when there are no further
 *  results.  If there are rows to return, they must be collected by
 *  the AOLserver, or the results will be cancelled prior to execution
 *  of the next Sybase command.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbExec(void *handle, char *sql)
{
    SybState *state = (SybState *)handle;
    int execRet;

    Ns_PdLog(Trace, "exec(%s):", sql);

    execRet = SybExec(state, sql);

    if (execRet == NS_ERROR) {
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
    } else {
	Ns_PdSendString(OK_STATUS);
	(execRet == DB_ROWS) ? Ns_PdSendString(EXEC_RET_ROWS) :
	    Ns_PdSendString(EXEC_RET_DML);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbFlush -
 *
 *  Flush any results pending from the Sybase server.  This would typically
 *  be called in response to detecting a failure of execution.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Pending-but-uncollected results are cancelled at the Sybase server.
 *  The AOLserver should then be able to issue another command.	 Note
 *  that transaction state is not affected at this point.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbFlush(void *handle)
{
    SybState *state = (SybState *)handle;
    int status = NS_OK;

    Ns_PdLog(Trace, "flush (calling cancel):");

    Ns_PdDbCancel (state);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbGetRow -
 *
 *  Fetch the next row from the result at the Sybase server and return
 *  it to the AOLserver.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  The next row in the current Sybase result is fetched.  If it is the
 *  last row, then result processing is continued, and will continue until
 *  all results are fetched, or until a result returning multiple rows is
 *  encountered.  The state structure is appropriately updated.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbGetRow(void *handle, char *columnCount)
{
    SybState		*state = (SybState *)handle;
    Ns_PdRowInfo	*rowInfo;
    CS_RETCODE		returnCode;
    CS_INT		rowsFetched;
    char		*me = "Ns_PdDbGetRow";
    int			cmdDoneReceived = 0;
    int			cmdSucceedReceived = 0;
    int			cmdFailReceived = 0;
    int			spResultReceived = 0;

    Ns_PdLog(Trace, "getrow(%s):", columnCount);
    returnCode = syb_fetch_row(state->cmd, &rowsFetched, state->dataRow);
    if (returnCode == CS_SUCCEED || returnCode == CS_ROW_FAIL) {
	state->dataRowsRead += rowsFetched;
	rowInfo = Ns_PdNewRowInfoFromColumnValueRefs(state->dataRow);
	if (rowInfo != NULL) {
	    Ns_PdSendString(OK_STATUS);
	    Ns_PdSendRowInfo(rowInfo);
	    Ns_PdFreeRowInfo(rowInfo, 0);
	} else {
	    sprintf(state->exceptionCode, "%d", errno);
	    strcpy(state->exceptionMsg, strerror(errno));
	    Ns_PdLog(Error, "%s: Can't allocate memory: %d: %s", me,
		     errno, strerror(errno));
	    Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	}
    } else if (returnCode == CS_END_DATA) {
	Ns_PdSendString(OK_STATUS);
	Ns_PdSendData(END_DATA, strlen(END_DATA));
	state->fetchingRows = 0;
	returnCode = syb_fetch_next_result (state->cmd, &state->dataRow,
					    &(state->dataRowCount),
					    &(state->spReturnCode),
					    &(state->spParamRow),
					    &cmdDoneReceived,
					    &cmdSucceedReceived,
					    &cmdFailReceived,
					    &(state->transactionState),
					    &spResultReceived);
	if (returnCode == CS_END_RESULTS || returnCode == CS_FAIL) {
	    state->fetchingRows = 0;
	    state->resultsPending = 0;
	} else if (returnCode == CS_SUCCEED) {
	    state->fetchingRows = 1;
	    state->resultsPending = 1;
	}
    } else {
	sprintf(state->exceptionCode, "%d", errno);
	sprintf(state->exceptionMsg,
		"%s: unexpected return from syb_fetch_row: %ld",
		me, returnCode);
	Ns_PdLog(Error, "%s: Unexpected return from syb_fetch_row: %d", me,
		 returnCode);
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbGetTableInfo -
 *
 *  Obtain the column names and datatypes for the specified table and
 *  return the information to the AOLserver.
 *
 * Results:
 *  Returns a pointer the the state variable.
 *
 * Side effects:
 *  Since this requires a complete query from the Sybase database, any
 *  pending- but-uncollected results are cancelled.  You should be
 *  able to use the Sybase connection for new commands after
 *  completion of this operation.
 *
 *  It might be more portable to attempt a "select * from table" and decode
 *  the results that way, but I wasn't sure what that might break.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbGetTableInfo(void *handle, char *tableName)
{
    SybState		*state = (SybState *)handle;
    int			status = NS_ERROR;
    CS_RETCODE		returnCode;
    CS_INT		rowsFetched;

    Ns_PdLog(Trace, "gettableinfo(%s):", tableName);

    /*
     * NB: the external driver relies on the name, type, notnull
     * ordering below
     */
    sprintf(state->sqlbuf,
	    "select syscolumns.name, systypes.name, syscolumns.status"
	    " from syscolumns, systypes, sysobjects "
	    " where sysobjects.name = '%s'"
	    " and sysobjects.id = syscolumns.id"
	    " and syscolumns.usertype = systypes.usertype", tableName);
    status = SybExec(state, state->sqlbuf);
    if (status == DB_ROWS) {
	Ns_PdRowInfo	*rowInfo = NULL;

	Ns_PdSendString(OK_STATUS);
	rowInfo = Ns_PdNewRowInfoFromColumnNames(state->dataRow);
	Ns_PdSendRowInfo(rowInfo);	/* sends list of column names */
	Ns_PdFreeRowInfo(rowInfo, 0);

	returnCode = CS_SUCCEED;
	while (returnCode == CS_SUCCEED || returnCode == CS_FAIL) {
	    returnCode = syb_fetch_row(state->cmd, &rowsFetched,
				       state->dataRow);
	    if (returnCode == CS_SUCCEED || returnCode == CS_FAIL) {
		rowInfo = Ns_PdNewRowInfoFromColumnValueRefs(state->dataRow);
		if (rowInfo) {
		    Ns_PdSendRowInfo(rowInfo);
		}
		Ns_PdFreeRowInfo(rowInfo, 0);
	    }
	}
	Ns_PdSendData(END_DATA, strlen(END_DATA));
	state->fetchingRows = 0;
	syb_drain_results(state->cmd);
	state->resultsPending = 0;
    } else {
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbGetTypes -
 *
 *  Return to the AOLserver the list of datatypes supported by the
 *  database driver.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbGetTypes(void *handle)
{
    SybState *state = (SybState *)handle;
    Ns_PdLog(Trace, "gettypes:");
    Ns_PdSendString(SYBASE_TYPES);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbIdentify -
 *
 *  Return the current database driver identification to the AOLserver.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbIdentify(void *handle)
{
    SybState *state = (SybState *)handle;
    Ns_PdLog(Trace, "identify:");
    Ns_PdSendString(IDENTITY_STRING);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbInit -
 *
 *  Allocate and initialize the Sybase state structure, which is used
 *  everywhere in this file.
 *
 * Results:
 *  Returns a pointer the the state variable.
 *
 * Side effects:
 *  None, although the returned state structure is initialized appropriately.
 *
 *----------------------------------------------------------------------
 */

void *
Ns_PdDbInit()
{
    SybState *state;
    int status = NS_OK;
    if ((state = (SybState *)calloc(1, sizeof(SybState))) != NULL) {
	if ((state->sqlbuf = malloc(INIT_SQLBUFSIZE)) == NULL) {
	    Ns_PdLog(Error, "Cannot allocate sqlbuf memory (%s)",
		     strerror(errno));
	    status = NS_ERROR;
	} else {
	    state->sqlbufsize = INIT_SQLBUFSIZE;
	}
	if (status == NS_ERROR) {
	    free (state);
	    state = NULL;
	}
    }
    return (void *)state;
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbOpen -
 *
 *  Open a connection to the Sybase server.
 *
 * Results:
 *  No explicit returns.
 *
 * Side effects:
 *  A connection to the Sybase server is opened, and the state structure
 *  is filled-in appropriately.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbOpen(void *handle, char *openArgs)
{
    SybState *state = (SybState *)handle;
    CS_RETCODE	    retcode;
    CS_CONNECTION   *connection;
    CS_COMMAND	    *cmd;
    int status = NS_ERROR;
    char *errmsg;
    char errbuf[ERRBUF_SIZE];
    char envbuf[1024];
    char	   *sep;

    Ns_PdLog(Trace, "open:"); 

    Ns_PdParseOpenArgs(openArgs, &state->datasource,
		       &state->user, &state->password, &state->param);
    Ns_PdLog(Notice, 
	     "Connecting to datasource '%s' as user '%s' and param '%s'", 
	     state->datasource==NULL ? "NULL" : state->datasource, 
	     state->user==NULL ? "NULL" : state->user, 
	     state->param==NULL ? "NULL" : state->param);
    errbuf[0] = '\0';
    if (state->datasource == NULL) {
	sprintf (errbuf, "Sybase proxy daemon received NULL datasource");
	Ns_PdLog(Error, "NULL datasource in openargs");
    } else if ((sep = strchr(state->datasource, ':')) == NULL) {
	sprintf (errbuf,
		 "Sybase proxy daemon received malformed datasource: %s",
		 state->datasource);
	Ns_PdLog(Error, "Malformed datasource:	%s", state->datasource);
    } else {
	if (state->param != NULL) {
	    sprintf(envbuf, "SYBASE=%s", state->param);
	    putenv(envbuf);
	}
	if (syb_init(state) != CS_SUCCEED) {
	    sprintf(errbuf, "syb_init failure: "
		    "verify that \"Param\" is set to the SYBASE "
		    "environment variable value");
	} else {
	    *sep++ = '\0';
	    state->dataRowCount = 0;
	    state->conn = NULL;
	    state->cmd = NULL;
	    state->dataRow = NULL;
	    state->spReturnCode = -9999;
	    state->spParamRow = NULL;
	    connection = NULL;
	    /*
	     * Allocate a connection structure, set its properties, and
	     * establish a connection.
	     */
	    retcode = syb_connect(state->cs_context, &connection,
				  Appname, state->user, state->password,
				  state->datasource);
	    
	    /* Now use the specified database */
	    if ((retcode == CS_SUCCEED) && 
		((retcode = syb_use_db(connection, sep)) != CS_SUCCEED)) {
		errmsg = "DbOpen: syb_use_db() failed";
		Ns_PdLog(Error, errmsg);
		strcpy(errbuf, errmsg);
	    }
	    /*
	     * Allocate a command handle to send the compute query with
	     */
	    if ((retcode == CS_SUCCEED) &&
		((retcode = ct_cmd_alloc(connection, &cmd)) != CS_SUCCEED)) {
		errmsg = "DbOpen: ct_cmd_alloc() failed";
		Ns_PdLog(Error, errmsg);
		strcpy(errbuf, errmsg);
	    }
	    /*
	     * Restore the handle->datasource field
	     */
	    *--sep = ':';
	    if (retcode != CS_SUCCEED) {
		Ns_PdLog(Error, "Could not open connection to:	%s", 
			 state->datasource);
		sprintf(errbuf, "Sybase proxy daemon could not open "
			" connection to: %s with user: %s",
			state->datasource==NULL ? "NULL" : state->datasource, 
			state->user==NULL ? "NULL" : state->user);
		state->connected = 0;
		state->conn = NULL;
	    } else {
		state->conn = connection;
		state->cmd = cmd;
		state->connected = 1;
		Ns_PdLog(Notice, "Connected.");
		status = NS_OK;
	    }
	}
    }
    if (status == NS_OK) {
	Ns_PdSendString(OK_STATUS);
    } else {
	Ns_PdSendString(errbuf);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbResetHandle -
 *
 *  This is supposed to return the AOLserver database handle to a known
 *  state, but we don't implement it here for Sybase.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.  Really.
 *
 *----------------------------------------------------------------------
 */

void 
Ns_PdDbResetHandle (void *handle) {
    Ns_PdLog(Trace, "resethandle:");
    Ns_PdLog(Trace, "DbResetHandle function is undefined for Sybase.");
    Ns_PdSendString(OK_STATUS);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbResultID -
 *
 *  Determine the identity value resulting from the previous insert
 *  operation and return it to the AOLserver.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  We have to run a query to get the identity column value, so
 *  pending-but- uncollected results from the Sybase server will be
 *  cancelled.  Also, I haven't tested this, but if you insert several
 *  results in a stored procedure, I'll bet you only get the last ID
 *  using this routine.
 *
 *----------------------------------------------------------------------
 */

void 
Ns_PdDbResultId(void *handle)
{
    SybState *state = (SybState *)handle;
    char *sql = "select @@identity";
    CS_RETCODE getRowRet;
    int sentData = 0;
    CS_INT rowsFetched;

    Ns_PdLog(Trace, "resultid:");
    if (SybExec(state, sql) == DB_ROWS) {
	getRowRet = syb_fetch_row (state->cmd, &rowsFetched, state->dataRow);
	if (getRowRet == CS_SUCCEED || getRowRet == CS_ROW_FAIL) {
	    Ns_PdSendString(state->dataRow->columns[0].value);
	    sentData = 1;
	}
	syb_drain_rows(state->cmd);
	state->fetchingRows = 0;
	syb_drain_results(state->cmd);
	state->resultsPending = 0;
    }
    if (sentData != 1) {
	Ns_PdSendString("");
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbResultRows -
 *
 *  Send the AOLserver the number of rows in the current result.  Note
 *  that Sybase won't make this information available until the entire
 *  result has been processed, by which time you could have
 *  counted the rows anyway.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbResultRows(void *handle)
{
    SybState *state = (SybState *)handle;
    char rowcntbuf[128];
 
    Ns_PdLog(Trace, "resultrows:");
    sprintf(rowcntbuf, "%d", state->dataRowCount);
    Ns_PdSendString(rowcntbuf);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbSetMaxRows -
 *
 *  Set the maximum number of rows the Sybase server will return.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  The maximum number of rows the Sybase server will return is set.
 *  This is done using a query, so any pending-but-uncollected results
 *  will be cancelled.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbSetMaxRows(void *handle, char *maxRows)
{
    SybState *state = (SybState *)handle;
    Ns_PdLog(Trace, "setmaxrows:");
    sprintf(state->sqlbuf, "set rowcount %s", maxRows);
    if (SybExec(state, state->sqlbuf) != DB_DML) {
	Ns_PdLog(Error, "DbSetMaxRows: SybExec(%s) failed", state->sqlbuf);
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
    }
    Ns_PdSendString(OK_STATUS);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbSpExec -
 *
 *  Send the stored procedure execution command to the Sybase server.
 *  Report the execution result to the AOLserver, and begin processing
 *  results.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Result processing begins at the Sybase server.
 *
 *----------------------------------------------------------------------
 */
void
Ns_PdDbSpExec (void *handle) {
    SybState *state = (SybState *)handle;
    CS_RETCODE retcode;
    int morep = 1;
    int status = NS_ERROR;
    CS_INT res_type;
    int cmdDoneReceived = 0;
    int cmdSucceedReceived = 0;
    int cmdFailReceived = 0;
    int spResultReceived = 0;

    Ns_PdLog(Trace, "spexec:");

    retcode = ct_send(state->cmd);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "DbSpExec: ct_send() failed.");
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	return;
    }

    if (state->dataRow != NULL) {
	SybRow_free(state->dataRow);
	state->dataRow = NULL;
    }
    state->fetchingRows = 0;
    state->resultsPending = 1;
    state->dataRowsRead = state->dataRowCount = state->spReturnCode = 0;
    if (state->spParamRow != NULL) {
	SybRow_free(state->spParamRow);
	state->spParamRow = NULL;
    }
    retcode = syb_fetch_next_result (state->cmd, &state->dataRow,
				     &(state->dataRowCount),
				     &(state->spReturnCode),
				     &(state->spParamRow),
				     &cmdDoneReceived,
				     &cmdSucceedReceived,
				     &cmdFailReceived,
				     &(state->transactionState),
				     &spResultReceived);
    if (retcode == CS_SUCCEED) {
	status = DB_ROWS;
	state->fetchingRows = 1;
    } else {
	if (cmdDoneReceived > 0 || cmdSucceedReceived > 0) {
	    status = DB_DML;
	}
	state->resultsPending = 0;
    }
    if (status == NS_ERROR) {
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
    } else {
	Ns_PdSendString(OK_STATUS);
	status == DB_ROWS ? Ns_PdSendString(EXEC_RET_ROWS) :
	    Ns_PdSendString(EXEC_RET_DML);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbSpGetParams -
 *
 *  Extract the returned parameters from the stored procedure and send
 *  them to the AOLserver.  If no return parameters have been yet
 *  collected, this routine will return an empty row to the AOLserver.
 *  You can call this routine more than once, but you'll get the same
 *  answer until you do another stored procedure or SQL command.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbSpGetParams(void *handle)
{
    SybState *state = (SybState *)handle;
    Ns_PdRowInfo	*paramNames = NULL;
    Ns_PdRowInfo	*paramValues = NULL;
    int			nrParams = state->spParamRow->nrColumnsUsed;
    CS_DATAFMT		*datafmt;
    char		*me = "Ns_PdDbSpGetParams";

    Ns_PdLog(Trace, "spgetparams:");
    Ns_PdSendString(OK_STATUS);
    if (state->spParamRow != NULL) {
	paramNames = Ns_PdNewRowInfoFromColumnNames(state->spParamRow);
	paramValues = Ns_PdNewRowInfoFromColumnValueRefs(state->spParamRow);
    }
    if (paramNames != NULL && paramValues != NULL) {
	Ns_PdSendRowInfo(paramNames);
	Ns_PdSendRowInfo(paramValues);
    } else {
	Ns_PdSendData(END_DATA, strlen(END_DATA));
    }
    if (paramNames != NULL) {
	Ns_PdFreeRowInfo(paramNames, 0);
    }
    if (paramValues != NULL) {
	Ns_PdFreeRowInfo(paramValues, 0);
    }
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbSpReturnCode -
 *
 *  Send the AOLserver the most-recently-collected stored procedure
 *  return code.  Note that if no stored procedure return code has
 *  been collected, you'll get a zero.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbSpReturnCode (void *handle) {
    SybState *state = (SybState *)handle;
    char rc[15];

    sprintf(rc, "%d", state->spReturnCode);
    Ns_PdLog(Trace, "spreturncode: %s", rc);
    Ns_PdSendString(rc);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbSpSetParam -
 *
 *  Set the passed parameter as a stored procedure parameter.
 *  Reviewing this function, I'm not sure that you really can't do
 *  this for DbExec as well, which would allow fast retriggering of
 *  queries, but I haven't tested it.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbSpSetParam (void *handle, char *args) {
    SybState   *state = (SybState *)handle;
    static char delim = ' ';
    CS_CONTEXT *context;
    CS_RETCODE	retcode;
    CS_DATAFMT	datafmt, srcfmt, destfmt;
    CS_INT	datatype, maxlength, status;
    CS_DATETIME dtbuffer;
    CS_INT	intbuffer;
    void       *destbuf;
    CS_INT	destbuflen;
    char       *paramname, *paramtype, *inout, *value;
    int		converting = 0;

    if (state->cmd == NULL) {
	Ns_PdLog(Error, "DbSpSetParam: spsetparam called before spstart");
	Ns_PdSendString("spsetparam called before spstart");
	return;
    }

    /*
     * Rip apart the arguments that are separated by a delimeter.
     * Replace the delimieter with a null so we can use the strings
     * in-place.  Can't use strtok() because the last argument can
     * have spaces inside itself.
     */
    paramname = args;
    paramtype = strchr(paramname, delim);
    *paramtype = '\0';	paramtype++;
    inout     = strchr(paramtype, delim);
    *inout = '\0';  inout++;
    value     = strchr(inout, delim);
    *value = '\0';  value++;

    Ns_PdLog(Trace, "spsetparam: %s, %s, %s, %s",
	     paramname, paramtype, inout, value);

    if (!strcmp(inout, "out")) {
	status = CS_RETURN;
    }
    else {
	status = CS_INPUTVALUE;
    }

    if (!strcmp(paramtype, "int") || !strcmp(paramtype, "smallint") ||
	!strcmp(paramtype, "tinyint")) {
	datatype = CS_INT_TYPE;
	maxlength = CS_UNUSED;
	destbuf = &intbuffer;
	destbuflen = CS_UNUSED;
	converting = 1;
    }
    else {
	datatype = CS_CHAR_TYPE;
	destbuf = value;
	destbuflen = strlen(value);
    }

    if (converting) {
	/* convert data types */
	memset(&srcfmt, 0, sizeof(srcfmt));
	srcfmt.datatype = CS_CHAR_TYPE;
	srcfmt.maxlength = strlen(value);
	srcfmt.locale = NULL;

	memset(&destfmt, 0, sizeof(destfmt));
	destfmt.datatype = datatype;
	destfmt.maxlength = maxlength;
	destfmt.locale = NULL;

	retcode = ct_cmd_props(state->cmd, CS_GET, CS_PARENT_HANDLE,
			       &state->conn, CS_UNUSED, NULL);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "DbSpSetParam: ct_cmd_props() failed.");
	    Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	    return;
	}

	retcode = ct_con_props(state->conn, CS_GET, CS_PARENT_HANDLE,
			       &context, CS_UNUSED, NULL);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "DbSpSetParam: ct_con_props() failed.");
	    Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	    return;
	}

	retcode = cs_convert(context, &srcfmt, value, &destfmt, destbuf,
			     NULL);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "DbSpSetParam: cs_convert() failed.");
	    Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	    return;
	}
    }

    /*
     * set up and add the parameter
     */
    strcpy(datafmt.name, paramname);
    datafmt.namelen = CS_NULLTERM;
    datafmt.datatype = datatype;
    datafmt.status = status;

    retcode = ct_param(state->cmd, &datafmt, destbuf, destbuflen, NULL);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "DbSpSetParam: ct_param() failed for arg %s.",
		 paramname);
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	return;
    }

    Ns_PdSendString(OK_STATUS);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbSpStart -
 *
 *  Prepare the Sybase state structure for execution of a stored procedure
 *  by cancelling any pending-but-uncollected results at the server, and
 *  remembering the name of the stored procedure.  (In fact, in reviewing
 *  this code, I can't see where there's any advantage to using
 *  SpStart/SpExec over DbExec -- at one time, DbExec wasn't capable of
 *  returning the stored procedure return codes and multiple results, but
 *  that's been fixed by now.)
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Cancels any pending-but-uncollected results, and sets up a new Sybase
 *  command structure to handle the stored procedure.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbSpStart (void *handle, char *procname) {
    SybState *state = (SybState *)handle;
    CS_RETCODE retcode;

    Ns_PdLog(Trace, "spstart: %s", procname);

    /*
     * Create a new stored procedure command if it hasn't been allocated yet.
     */
    if (state->cmd == NULL) {
	retcode = ct_cmd_alloc(state->conn, &state->cmd);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "DbSpStart: ct_cmd_alloc() failed.");
	    Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	    return;
	}
    }

    /* clear out any existing command structure */
    retcode = ct_cancel(NULL, state->cmd, CS_CANCEL_ALL);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "DbSpStart: ct_cancel() failed.");
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	return;
    }

    /* set up a new command structure for this stored procedure */
    retcode = ct_command(state->cmd, CS_RPC_CMD, procname, CS_NULLTERM,
			 CS_UNUSED);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "DbSpStart: ct_command() failed.");
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
	return;
    }

    Ns_PdSendString(OK_STATUS);
}


/*
 *----------------------------------------------------------------------
 * Ns_PdDbTableList -
 *
 *  Obtain a list of tables from the Sybase server and return it to the
 *  AOLserver. Note this procedure uses the Sybase stored procedure
 *  "sp_tables", which is definitely in Sybase 11, but you need to verify
 *  that it is in the version of Sybase _you_ are using.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  If there are pending-but-uncollected results from an earlier query to
 *  the Sybase server, they will be cancelled.	This is a complete query,
 *  and will send appropriate status notices to the AOLserver.	The Sybase
 *  connection should be in a state from which new queries can be issued,
 *  if all goes well.
 *
 *----------------------------------------------------------------------
 */

void
Ns_PdDbTableList(void *handle, char *includeSystem)
{
    SybState *state = (SybState *)handle;
    static char tableColName[] = "name";
    int fSystemTables;
    Ns_PdRowInfo *bindRowInfo, *getRowInfo;
    int status = NS_ERROR;
    int getRowRet;

    Ns_PdLog(Trace, "tablelist(%s):", includeSystem);
    fSystemTables = atoi(includeSystem);
    /* canned SQL below is smaller that initial SQL buf size (4k) */
    if (fSystemTables) {
	strcpy(state->sqlbuf,
	       "sp_tables @table_type = \"'TABLE', 'VIEW', 'SYSTEM TABLE'\"");
    } else {
	strcpy(state->sqlbuf, "sp_tables @table_type = \"'TABLE', 'VIEW'\"");
    }
    if ((status = SybExec(state, state->sqlbuf)) == DB_ROWS) {
	char countbuf[64];
	CS_INT rowsFetched;

	status = NS_OK;
	Ns_PdSendString(OK_STATUS);
	getRowRet = CS_SUCCEED;
	while (getRowRet == CS_SUCCEED
	       || getRowRet == CS_ROW_FAIL) {
	    getRowRet = syb_fetch_row (state->cmd, &rowsFetched,
				       state->dataRow);
	    if (getRowRet == CS_SUCCEED || getRowRet == CS_ROW_FAIL) {
		sprintf(countbuf, "%ld",
			state->dataRow->columns[2].valuelen-1);
		Ns_PdSendString(countbuf);
		Ns_PdSendData(state->dataRow->columns[2].value,
			      state->dataRow->columns[2].valuelen-1);
	    }
	}
	state->fetchingRows = 0;
	syb_drain_results(state->cmd);
	state->resultsPending = 0;
	Ns_PdSendData(END_DATA, strlen(END_DATA));
    }
    if (status == NS_ERROR) {
	Ns_PdSendException(state->exceptionCode, state->exceptionMsg);
    }
}



/*
 *==========================================================================
 * Static functions
 *==========================================================================
 */

/*
 *----------------------------------------------------------------------
 * Ns_PdNewRowInfoFromColumnNames -
 *
 *  This routine creates the structure we need to pass back to AOLserver
 *  in response to a bind row request (which requests the column names
 *  for the current result set.
 *
 * Results:
 *  This function returns a row info structure which can be passed back
 *  to AOLserver.
 *
 * Side effects:
 *  None
 *
 *----------------------------------------------------------------------
 */

static Ns_PdRowInfo *
Ns_PdNewRowInfoFromColumnNames(SybRow *row)
{
    Ns_PdRowInfo	*rowInfo = NULL;
    int			i0 = 0;
    char		*me = "Ns_PdNewRowInfoFromColumnNames";
    CS_DATAFMT		*datafmt;

    if ((rowInfo = Ns_PdNewRowInfo(row->nrColumnsUsed)) != NULL) {
	for (i0 = 0; i0 < row->nrColumnsUsed; i0++) {
	    datafmt = &(row->columns[i0].datafmt);
	    Ns_PdSetRowInfoItem(rowInfo, i0, datafmt->name, datafmt->namelen);
	}
    } else {
	Ns_PdLog(Error, "%s: Couldn't allocate row info for names: %d: %s",
		 me, errno, strerror(errno));
    }
    return rowInfo;
}


/*
 *----------------------------------------------------------------------
 * Ns_PdNewRowInfoFromColumnValues -
 *
 *  This routine extracts the values from a Sybase row structure and
 *  puts them in the format required to pass back to AOLserver.
 *
 * Results:
 *  This function returns a row info structure which can be passed back
 *  to AOLserver. The row structure will contain pointers to the data
 *  values, so if those pointers are freed, the corresponding Sybase row
 *  structure should be discarded, because it will no longer have valid
 *  data buffers.
 *
 * Side effects:
 *  None
 *
 *----------------------------------------------------------------------
 */

static Ns_PdRowInfo *
Ns_PdNewRowInfoFromColumnValueRefs(SybRow *row)
{
    Ns_PdRowInfo	*rowInfo = NULL;
    int			i0 = 0;
    char		*me = "Ns_PdNewRowInfoFromColumnValueRefs";
    CS_DATAFMT		*datafmt;

    if ((rowInfo = Ns_PdNewRowInfo(row->nrColumnsUsed)) != NULL) {
	for (i0 = 0; i0 < row->nrColumnsUsed; ++i0) {
	    Ns_PdSetRowInfoItem(rowInfo, i0,
				row->columns[i0].value,
				strlen(row->columns[i0].value));
	}
    } else {
	Ns_PdLog(Error, "%s: Couldn't allocate row info for valuerefs: %d: %s",
		 me, errno, strerror(errno));
    }
    return rowInfo;
}


/*
 *----------------------------------------------------------------------
 * Ns_PdNewRowInfoFromColumnValues -
 *
 *  This routine extracts the values from a Sybase row structure and
 *  puts them in the format required to pass back to AOLserver.
 *
 * Results:
 *  This function returns a row info structure which can be passed back
 *  to AOLserver. The row structure will contain copies of the data
 *  values, so that freeing these buffers will not have impact on the Sybase
 *  row structure.
 *
 * Side effects:
 *  None
 *
 *----------------------------------------------------------------------
 */

static Ns_PdRowInfo *
Ns_PdNewRowInfoFromColumnValueCopy(SybRow *row)
{
    Ns_PdRowInfo	*rowInfo = NULL;
    int			i0 = 0;
    char		*me = "Ns_PdNewRowInfoFromColumnValueCopy";
    CS_DATAFMT		*datafmt;

    if ((rowInfo = Ns_PdNewRowInfo(row->nrColumnsUsed)) != NULL) {
	for (i0 = 0; i0 < row->nrColumnsUsed; ++i0) {
	    Ns_PdSetRowInfoItem(rowInfo, i0,
				strdup(row->columns[i0].value),
				strlen(row->columns[i0].value));
	}
    } else {
	Ns_PdLog(Error, "%s: Couldn't allocate row info for valuecopy: %d: %s",
		 me, errno, strerror(errno));
    }
    return rowInfo;
}


/*
 *----------------------------------------------------------------------
 * strTrimTrailingSpace -
 *
 *  Trim the trailing spaces from a string.  This is because Sybase
 *  has funny rules about using spaces to concatenate strings, and
 *  sometimes returns a space for a NULL-valued column (and we don't
 *  have mechanism in the AOLserver protocol to communicate the status
 *  of NULL-valued columns.)
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

static void
strTrimTrailingSpace(char *str)
{
    int		len;
    char	ch;

    len = strlen(str);
    while (len-- > 0) {
	ch = *(str + len);
	if (ch == ' ' || ch == '\t' || ch == '\r' || ch == '\n') {
	    *(str + len) = '\0';
	} else {
	    break;
	}
    }
}


/*
 *----------------------------------------------------------------------
 * SybCancel -
 *
 *  Terminate any command which hasn't been totally processed, and roll back
 *  any open transaction.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  The Sybase connection is reset so that further commands can be issued.
 *
 *----------------------------------------------------------------------
 */

static int
SybCancel(SybState *state)
{
    CS_RETCODE	retcode;
    int		status = NS_OK;
    int		transactionPending = 0;
    int		rollbackReturn = 0;
    CS_INT	transactionState = 0;
    CS_INT	transactionStateLength;

    Ns_PdLog(Trace, "SybCancel:");

    /*
     * Terminate results processing if any
     */
    if (state->resultsPending) {
	retcode = ct_cancel(state->conn, NULL, CS_CANCEL_ALL);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "Ns_SybCloseDb: ct_cancel() failed");
	    status = NS_ERROR;
	} else {
	    state->fetchingRows = 0;
	    state->resultsPending = 0;
	    syb_drain_results(state->cmd);
	}
    }
    /*
     * rollback any transactions pending
     */
    if (state->transactionState) {
	Ns_PdLog(Trace, "Rolling back in-progress transaction");
	rollbackReturn = SybExec (state, "rollback transaction");
	if (rollbackReturn != DB_DML) {
	    Ns_PdLog(Error, "could not roll back open transactions on cancel");
	}
    }

    return status;
}


/*
 *----------------------------------------------------------------------
 * SybExec -
 *
 *  Begin execution of a Sybase language command.  If there are pending-
 *  but-uncollected results at the server, cancel them.
 *
 * Results:
 *  NS_ERROR - The command execution failed.
 *  DB_ROWS - the command execution succeeded, and there are rows available
 *  DB_DML - the command execution succeeded, and there are no rows available.
 *     Any stored procedure results (return codes or output parameters) are
 *     available in the state structure.
 *
 * Side effects:
 *  result processing commences.
 *
 *----------------------------------------------------------------------
 */

static int
SybExec(SybState *state, char *sql)
{
    int			status = NS_ERROR;
    CS_RETCODE		retcode;
    int			cmdDoneReceived = 0;
    int			cmdSucceedReceived = 0;
    int			cmdFailReceived = 0;
    int			spResultReceived = 0;

    Ns_PdLog(Trace, "SybExec(%s):", sql);
    if (!state->connected) {
	Ns_PdLog(Error, "SybExec: Not connected.");
    } else {
	int sqlTrimmedLen;
	sql = Ns_PdStringTrim(sql);
	/* remove trailing ; for Illustra interoperability */
	sqlTrimmedLen = strlen(sql);
	if (sql[sqlTrimmedLen - 1] == ';') {
	    sql[sqlTrimmedLen - 1] = '\0';
	}
	if (state->fetchingRows || state->resultsPending) {
	    Ns_PdLog(Trace, "SybExec: fetchingRows: %d; resultsPending: %d",
		     state->fetchingRows, state->resultsPending);
	    retcode = ct_cancel(NULL, state->cmd, CS_CANCEL_ALL);
	    if (retcode == CS_SUCCEED) {
		syb_drain_results(state->cmd);
		state->fetchingRows = 0;
		state->resultsPending = 0;
	    } else {
		Ns_PdLog(Error, "SybExec: pre-flight ct_cancel failed: %d",
			 retcode);
	    }
	}
	Ns_PdLog(Trace, "SybExec: Sending query: \"%s\"", sql);
	if (ct_command(state->cmd, CS_LANG_CMD, sql, CS_NULLTERM, CS_UNUSED) 
	    != CS_SUCCEED) {
	    Ns_PdLog(Error, "SybExec: ct_command() failed");
	} else if (ct_send(state->cmd) != CS_SUCCEED) {
	    Ns_PdLog(Error, "SybExec: ct_send() failed");
	} else {
	    if (state->dataRow != NULL) {
		SybRow_free(state->dataRow);
		state->dataRow = NULL;
	    }
	    state->fetchingRows = 0;
	    state->resultsPending = 1;
	    state->dataRowsRead = state->dataRowCount = state->spReturnCode = 0;
	    if (state->spParamRow != NULL) {
		SybRow_free(state->spParamRow);
		state->spParamRow = NULL;
	    }
	    retcode = syb_fetch_next_result(state->cmd, &state->dataRow,
					     &(state->dataRowCount),
					     &(state->spReturnCode),
					     &(state->spParamRow),
					     &cmdDoneReceived,
					     &cmdSucceedReceived,
					     &cmdFailReceived,
					     &(state->transactionState),
					     &spResultReceived);
	    if (retcode == CS_SUCCEED) {
		status = DB_ROWS;
		state->fetchingRows = 1;
	    } else {
		state->resultsPending = 0;
	    }
	    if (cmdDoneReceived > 0 || cmdSucceedReceived > 0) {
		status = DB_DML;
	    }
	}
    }
    return status;
} 


/*
 *----------------------------------------------------------------------
 * SybRow_dup -
 *
 *  Create a newly-allocated Sybase row structure cloned from an existing
 *  one.
 *
 * Results:
 *  A new Sybase row structure.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

static SybRow *
SybRow_dup (SybRow *row)
{
    int			i0 = 0;
    int			keep_going = 1;
    SybRow		*newRow = NULL;
    char		*me = "SybRow_dup";
    SybColumn		*col;
    CS_DATAFMT		*datafmt;
    char		*columnValue;

    newRow = SybRow_new(row->nrColumnsAllocated);
    if (newRow == NULL) {
	Ns_PdLog(Error, "%s: Could not allocate %d-column SybRow",
		 me, row->nrColumnsAllocated);
	keep_going = 0;
    }

    for (i0 = 0; keep_going && (i0 < row->nrColumnsAllocated); i0++) {
	newRow->columns[i0] = row->columns[i0];
	newRow->columns[i0].value = 
	    malloc (row->columns[i0].datafmt.maxlength);
	if (newRow->columns[i0].value == NULL) {
	    Ns_PdLog(Error,
		     "%s: Can't allocate %d bytes for column %d: (%d) %s",
		     me, row->columns[i0].datafmt.maxlength, i0,
		     errno, strerror(errno));
	    keep_going = 0;
	} else {
	    memcpy (newRow->columns[i0].value,
		    row->columns[i0].value,
		    row->columns[i0].datafmt.maxlength);
	}
    }

    if (!keep_going && newRow) {
	SybRow_free(newRow);
	newRow = NULL;
    }

    return newRow;
}


/*
 *----------------------------------------------------------------------
 * SybRow_free -
 *
 *  Free a Sybase row structure.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

static void
SybRow_free (SybRow *row)
{
    int i0;

    for (i0 = 0; i0 < row->nrColumnsAllocated; i0++) {
	if (row->columns[i0].value != NULL) {
	    if (row->columns[i0].datafmt.maxlength > 0) {
		memset(row->columns[i0].value, 0,
		       row->columns[i0].datafmt.maxlength);
	    }
	    free(row->columns[i0].value);
	    row->columns[i0].value = NULL;
	}
	row->columns[i0].datafmt.maxlength = 0;
    }
    memset((char *)row, 0, sizeof(SybRow) +
	   ((row->nrColumnsAllocated - 1) * sizeof(SybColumn)));
    free((char *)row);
}


/*
 *----------------------------------------------------------------------
 * SybRow_new -
 *
 *  Allocate a new Sybase row structure capable of handling the
 *  specified number of columns.
 *
 * Results:
 *  A Sybase row structure initialized to zero, or NULL if unable to allocate.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

static SybRow *
SybRow_new (int columnCount)
{
    SybRow	*tmp;
    size_t	struct_size;


    /*
     * This is slightly larger than necessary, but it should take
     * care of alignment issues
     */
    struct_size = sizeof(SybRow) + (columnCount * sizeof(SybColumn));
    tmp = (SybRow *)malloc(struct_size);
    if (tmp == NULL) {
	Ns_PdLog(Error, "SybRow_new: Can't calloc %d: (%d) %s",
		 struct_size, errno, strerror(errno));
    } else {
	memset((char *)tmp, 0, struct_size);
	tmp->nrColumnsAllocated = columnCount;
	tmp->nrColumnsUsed = columnCount;
    }

    return tmp;
}


/*
 *----------------------------------------------------------------------
 * SybRow_newFromResult -
 *
 *  Allocate a new Sybase row structure capable of handling the
 *  columns in the current result.
 *
 * Results:
 *  A Sybase row structure initialized to zero, or NULL if unable to allocate.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */

static SybRow *
SybRow_newFromResult (CS_COMMAND *cmd, CS_INT forceDataType)
{
    int			keep_going = 1;
    int			i0;
    int			columnCount = 0;
    SybRow		*row = NULL;
    char		*me = "SybRow_newFromResult";
    CS_RETCODE		returnCode;
    SybColumn		*col;
    CS_DATAFMT		*datafmt;
    char		*columnValue;

    returnCode = ct_res_info(cmd, CS_NUMDATA, &columnCount,
			     CS_UNUSED, NULL);
    if (returnCode != CS_SUCCEED) {
	Ns_PdLog(Error, "%s: Failed to get column count: %d", me,
		 returnCode);
	keep_going = 0;
    } else if (columnCount <= 0) {
	Ns_PdLog(Error, "%s: Invalid column count: %d", me, columnCount);
	keep_going = 0;
    }

    if (keep_going) {
	row = SybRow_new(columnCount);
	if (row == NULL) {
	    Ns_PdLog(Error, "%s: Could not allocate %d-column SybRow",
		     me, columnCount);
	    keep_going = 0;
	}
    }

    for (i0 = 0; keep_going && (i0 < columnCount); i0++) {
	col = &(row->columns[i0]);
	datafmt = &(col->datafmt);
	memset(datafmt, 0, sizeof(*datafmt));
	returnCode = ct_describe (cmd, (i0 + 1), datafmt);
	if (returnCode != CS_SUCCEED) {
	    Ns_PdLog(Error, "%s: ct_describe failed on column %d: %d\n",
		     me, i0, returnCode);
	    keep_going = 0;
	} else {
	    if (forceDataType == CS_CHAR_TYPE) {
		datafmt->maxlength = syb_display_dlen(datafmt) + 1;
		datafmt->datatype = forceDataType;
		datafmt->format = CS_FMT_NULLTERM;
	    }
	    col->value = malloc (datafmt->maxlength);
	    if (col->value == NULL) {
		Ns_PdLog(Error, "%s: Can't allocate %d bytes for column "
			 "%d (%.*s): (%d) %s", me, datafmt->maxlength, i0,
			 datafmt->namelen, datafmt->name,
			 errno, strerror(errno));
		keep_going = 0;
	    } else {
		memset (col->value, 0, datafmt->maxlength);
	    }
	    returnCode = ct_bind (cmd, (i0 + 1), datafmt, col->value,
				  &(col->valuelen), &(col->indicator));
	    if (returnCode != CS_SUCCEED) {
		Ns_PdLog(Error, "%d: ct_bind on column %d failed: %d", me,
			 i0, returnCode);
		keep_going = 0;
	    }
	}
    }

    if (!keep_going && row) {
	SybRow_free(row);
	row = NULL;
    }

    return row;
}


/*
 *----------------------------------------------------------------------
 * syb_check_transaction_state -
 *
 *  Determine whether the current Sybase connection has an open
 *  transaction.  The nice thing about this implementation (as opposed
 *  to one where we do "select @@transtate, @@trancount") is that
 *  this can be done without disturbing the current result.
 *
 * Results:
 *  1 - A transaction is in progress, so a rollback is necessary
 *  0 - no transaction is in progress.
 *
 *  (I know that these should be NS_TRUE and NS_FALSE, or a count of
 *  open transactions; that will have to wait for the next release.)
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------
 */
 
static int
syb_check_transaction_state (CS_COMMAND *cmd)
{
    CS_INT		transactionState;
    CS_INT		transactionStateLen;
    CS_INT		retcode;
    CS_INT		returnValue = -1;

    retcode = ct_res_info(cmd, CS_TRANS_STATE, &transactionState,
			  CS_UNUSED, &transactionStateLen);
    if (retcode == CS_SUCCEED) {
	switch (transactionState) {
	case CS_TRAN_IN_PROGRESS:
	    Ns_PdLog(Trace, "syb_check_transaction_state: In progress");
	    returnValue = 1;
	    break;
	case CS_TRAN_COMPLETED:
	    Ns_PdLog(Trace, "syb_check_transaction_state: Completed");
	    returnValue = 0;
	    break;
	case CS_TRAN_FAIL:
	    Ns_PdLog(Trace, "syb_check_transaction_state: Failed");
	    returnValue = 0;
	    break;
	default:
	    Ns_PdLog(Trace, "syb_check_transaction_state: ??: %d",
		     transactionState);
	    break;
	}
    } else {
	Ns_PdLog(Error, "syb_check_transaction_state: retcode: %d", retcode);
    }
    return returnValue;
}


/*
 *----------------------------------------------------------------------
 * syb_clientmsg_cb -
 *
 *  Callback for processing client messages.  Since the AOLserver proxy
 *  protocol only allows a single exception result, store the message
 *  in the state structure for later transmission to the AOLserver, unless
 *  the message is one we can ignore ("Command has been aborted" -- we
 *  will already have received a better server message).  All messages are
 *  logged via syslog.
 *
 * Results:
 *  Returns results consistent with what a Sybase Client Message Callback
 *  procedure should return.
 *
 * Side effects:
 *  Message is logged, and probably recorded in the state structure.
 *
 *----------------------------------------------------------------------
 */
 
static CS_RETCODE CS_PUBLIC
syb_clientmsg_cb(CS_CONTEXT *context, CS_CONNECTION *connection, 
		 CS_CLIENTMSG	*errmsg)
{
    SybState *state = NULL;

    if (cs_config(context, CS_GET, CS_USERDATA, (CS_VOID *)&state,
		  sizeof(SybState *), NULL) != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_clientmsg_cb: cannot get SybState pointer");
    }

    if (state == NULL || !state->squelchError) {
	Ns_PdLog(Error,	"Client callback msg#: "
		 "LAYER = (%ld) ORIGIN = (%ld) "
		 "SEVERITY = (%ld) NUMBER = (%ld)",
		 CS_LAYER(errmsg->msgnumber), CS_ORIGIN(errmsg->msgnumber),
		 CS_SEVERITY(errmsg->msgnumber), CS_NUMBER(errmsg->msgnumber));
	Ns_PdLog(Error, "Client callback msg: %s", errmsg->msgstring);
	Ns_PdLog(Trace,	 "Client callback msg#: "
		 "LAYER = (%ld) ORIGIN = (%ld) "
		 "SEVERITY = (%ld) NUMBER = (%ld)",
		 CS_LAYER(errmsg->msgnumber), CS_ORIGIN(errmsg->msgnumber),
		 CS_SEVERITY(errmsg->msgnumber), CS_NUMBER(errmsg->msgnumber));
	Ns_PdLog(Trace, "Client callback msg: %s", errmsg->msgstring);
	if (errmsg->osstringlen > 0) {
	    Ns_PdLog(Error, "Operating System Error: %s",
		     errmsg->osstring);
	}
    }
    
    strTrimTrailingSpace(errmsg->msgstring);
    if (strcmp (errmsg->msgstring, "Command has been aborted.") != 0) {
	if (state != NULL && !state->squelchError) {
	    strcpy(state->exceptionCode, "ERROR");
	    strcpy(state->exceptionMsg, errmsg->msgstring);
	    if (errmsg->osstringlen > 0) {
		strcat(state->exceptionMsg, ">>OS: ");
		strcat(state->exceptionMsg, errmsg->osstring);
	    }
	}
    }

    return CS_SUCCEED;
}


/*
 *----------------------------------------------------------------------
 * syb_con_cleanup -
 *
 *  Terminate the Sybase server connection
 *
 * Results:
 *  Return code from closing and dropping connection.
 *
 * Side effects:
 *  Sybase server connection is terminated and dropped.
 *
 *----------------------------------------------------------------------
 */
 
static CS_RETCODE CS_PUBLIC
syb_con_cleanup(CS_CONNECTION *connection, CS_RETCODE status)
{
    CS_RETCODE	retcode;
    CS_INT		close_option;
    
    close_option = (status != CS_SUCCEED) ? CS_FORCE_CLOSE : CS_UNUSED;
    retcode = ct_close(connection, close_option);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_con_cleanup: ct_close() failed");
	return retcode;
    }
    retcode = ct_con_drop(connection);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_con_cleanup: ct_con_drop() failed");
	return retcode;
    }

    return retcode;
}


/*
 *----------------------------------------------------------------------
 * syb_connect -
 *
 *  Actually connect to a Sybase server
 *
 * Results:
 *  Return code from connection.
 *
 * Side effects:
 *  Sybase server connection is opened.
 *
 *----------------------------------------------------------------------
 */
 
static CS_RETCODE CS_PUBLIC
syb_connect(CS_CONTEXT *context, CS_CONNECTION **connection, CS_CHAR *appname, 
	    CS_CHAR *username, CS_CHAR *password, CS_CHAR *server)
{
    CS_INT		len;
    CS_RETCODE	retcode;

    /* 
     * Allocate a connection structure. 
     */
    retcode = ct_con_alloc(context, connection);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "ct_con_alloc failed");
	return retcode;
    }

    /*	
     * If a username is defined, set the CS_USERNAME property.
     */
    if (retcode == CS_SUCCEED && username != NULL) {
	if ((retcode = ct_con_props(*connection,
				    CS_SET, CS_USERNAME, username,
				    CS_NULLTERM, NULL)) != CS_SUCCEED) {
	    Ns_PdLog(Error, "ct_con_props(username) failed");
	}
    }
    
    /*	
     * If a password is defined, set the CS_PASSWORD property.
     */
    if (retcode == CS_SUCCEED && password != NULL) {
	if ((retcode = ct_con_props(*connection,
				    CS_SET, CS_PASSWORD, password,
				    CS_NULLTERM, NULL)) != CS_SUCCEED) {
	    Ns_PdLog(Error, "ct_con_props(password) failed");
	}
    }
    
    /*	
     * Set the CS_APPNAME property.
     */
    if (retcode == CS_SUCCEED && appname != NULL) {
	if ((retcode = ct_con_props(*connection,
				    CS_SET, CS_APPNAME, appname,
				    CS_NULLTERM, NULL)) != CS_SUCCEED) {
	    Ns_PdLog(Error, "ct_con_props(appname) failed");
	}
    }

    /*	
     * Open a Server connection.
     */
    if (retcode == CS_SUCCEED) {
	len = (server == NULL) ? 0 : CS_NULLTERM;
	retcode = ct_connect(*connection, server, len);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "ct_connect failed");
	} else {
	    Ns_PdLog(Trace, "ct_connect succeeded");
	}
    }
    if (retcode != CS_SUCCEED) {
	ct_con_drop(*connection);
	*connection = NULL;
    }
    return retcode;
}


/*
 *----------------------------------------------------------------------
 * syb_display_dlen -
 *
 *  return the maximum length of the specified column type
 *
 * Results:
 *  maximum length of the column
 *
 * Side effects:
 *  None
 *
 *----------------------------------------------------------------------
 */
 
static CS_INT CS_PUBLIC
syb_display_dlen(CS_DATAFMT *column)
{
    CS_INT		len;

    switch ((int) column->datatype) {
    case CS_TEXT_TYPE:
	len = MAX_CHAR_BUF;
    case CS_CHAR_TYPE:
    case CS_VARCHAR_TYPE:
    case CS_IMAGE_TYPE:
	len = MIN(column->maxlength, MAX_CHAR_BUF);
	break;

    case CS_BINARY_TYPE:
    case CS_VARBINARY_TYPE:
	len = MIN((2 * column->maxlength) + 2, MAX_CHAR_BUF);
	break;

    case CS_BIT_TYPE:
    case CS_TINYINT_TYPE:
	len = 3;
	break;

    case CS_SMALLINT_TYPE:
	len = 6;
	break;

    case CS_INT_TYPE:
	len = 11;
	break;

    case CS_REAL_TYPE:
    case CS_FLOAT_TYPE:
	len = 20;
	break;

    case CS_MONEY_TYPE:
    case CS_MONEY4_TYPE:
	len = 24;
	break;

    case CS_DATETIME_TYPE:
    case CS_DATETIME4_TYPE:
	len = 30;
	break;

    case CS_NUMERIC_TYPE:
    case CS_DECIMAL_TYPE:
	len = (CS_MAX_PREC + 2);
	break;

    default:
	len = 12;
	break;
    }

    return MAX(strlen(column->name) + 1, len);
}


/*
 *----------------------------------------------------------------------
 * syb_drain_results -
 *
 *  Drain any pending-but-uncollected results from the Sybase connection.
 *  Typically, this would be done after issuing a cancel, to make sure
 *  we don't leave anything behind, and we'd hope that, after a cancel,
 *  there aren't too many results pending.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Once the current result is drained, processing of subsequent results
 *  commences.
 *
 *----------------------------------------------------------------------
 */
static void
syb_drain_results (CS_COMMAND *cmd)
{
    CS_RETCODE	fetchResult = CS_SUCCEED;
    SybRow	*dataRow = NULL;
    int		rowCount = 0;
    SybRow	*paramRow = NULL;
    int		spResult = 0;
    int		cmd_done_received = 0;
    int		cmd_succeed_received = 0;
    int		cmd_fail_received = 0;
    int		spResultReceived = 0;
    int		transactionState = 0;

    while (fetchResult == CS_SUCCEED) {
	fetchResult = syb_fetch_next_result (cmd, &dataRow, &rowCount,
	     &spResult, &paramRow, &cmd_done_received, &cmd_succeed_received,
	     &cmd_fail_received, &transactionState, &spResultReceived);
	if (fetchResult == CS_SUCCEED) {
	    Ns_PdLog(Trace, "syb_drain_results: "
		     "Draining rows from row result");
	    syb_drain_rows (cmd);
	}
    }
    if (dataRow != NULL) {
	SybRow_free(dataRow);
	dataRow = NULL;
    }
    if (paramRow != NULL) {
	SybRow_free(paramRow);
	paramRow = NULL;
    }
}
 

/*
 *----------------------------------------------------------------------
 * syb_drain_rows -
 *
 *  Drain the rows from the current result.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Once the current result is drained, processing of subsequent results
 *  commences.
 *
 *----------------------------------------------------------------------
 */
 
static void
syb_drain_rows (CS_COMMAND *cmd)
{
    CS_INT		rowsFetched;
    CS_RETCODE		fetchReturnCode = CS_SUCCEED;

    while (fetchReturnCode == CS_SUCCEED ||
	   fetchReturnCode == CS_ROW_FAIL) {
	rowsFetched = 0;
	fetchReturnCode = syb_fetch_row(cmd, &rowsFetched, NULL);
    }
}


/*
 *----------------------------------------------------------------------
 * syb_execute -
 *
 *  Execute the specified Sybase command on the specified Sybase server
 *  connection.
 *
 * Results:
 *  Return code from command execution.
 *
 * Side effects:
 *  Result processing is initiated.
 *
 *----------------------------------------------------------------------
 */
 
static CS_RETCODE CS_PUBLIC
syb_execute_cmd(CS_CONNECTION *connection, CS_CHAR *cmdbuf)
{
    CS_RETCODE	    retcode;
    CS_INT	    restype;
    CS_COMMAND	    *cmd;
    CS_RETCODE	    query_code;
    SybRow	*dataRow = NULL;
    int		rowCount = 0;
    SybRow	*paramRow = NULL;
    int		spResult = 0;
    int		cmdDoneReceived = 0;
    int		cmdSucceedReceived = 0;
    int		cmdFailReceived = 0;
    int		spResultReceived = 0;
    int		transactionState = 0;
    
    /*
     * Get a command handle, store the command string in it, and
     * send it to the server.
     */
    if ((retcode = ct_cmd_alloc(connection, &cmd)) != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_execute_cmd: ct_cmd_alloc() failed %d", retcode);
	return retcode;
    }
    
    if ((retcode = ct_command(cmd, CS_LANG_CMD, cmdbuf, CS_NULLTERM,
			      CS_UNUSED)) != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_execute_cmd: ct_command() failed: %d", retcode);
	(void)ct_cmd_drop(cmd);
	return retcode;
    }
    
    if ((retcode = ct_send(cmd)) != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_execute_cmd: ct_send() failed: %d", retcode);
	(void)ct_cmd_drop(cmd);
	return retcode;
    }

    /*
     * Examine the results coming back. If any errors are seen, the query
     * result code (which we will return from this function) will be
     * set to FAIL.
     */
    query_code = CS_SUCCEED;

    while ((retcode = syb_fetch_next_result(cmd, &dataRow, &rowCount,
					    &spResult, &paramRow,
					    &cmdDoneReceived,
					    &cmdSucceedReceived,
					    &cmdFailReceived,
					    &transactionState,
					    &spResultReceived))
	   == CS_SUCCEED) {
	if (cmdFailReceived) {
	    query_code = CS_FAIL;
	}
    }
    
    /*
     * Clean up the command handle used
     */
    Ns_PdLog(Trace, "syb_execute_cmd: query_code after all results is %d",
	     query_code);
    if (retcode == CS_END_RESULTS) {
	Ns_PdLog(Trace, "syb_execute_cmd: last retcode is CS_END_RESULTS");
	retcode = ct_cmd_drop(cmd);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Trace, "syb_execute_cmd: "
		     "normal ct_cmd_drop failed: %d", retcode);
	    query_code = CS_FAIL;
	}
    } else {
	Ns_PdLog(Trace, "syb_execute_cmd: last retcode unknown; dropping");
	(void)ct_cmd_drop(cmd);
	query_code = CS_FAIL;
    }
    return query_code;
}


/*
 *----------------------------------------------------------------------
 * syb_fetch_next_result -
 *
 *  Fetch the next result from the Sybase server, yielding one of
 *  three possible return states:
 *    - we get a row result, and we have to tell the AOLserver
 *	  to start binding and fetching rows
 *    - We get the end of the results.
 *    - Something goes horribly wrong with the connection, and we
 *	  need to bail (we'll tear down the connection).
 *  This is the most complicated routine in the entire file, but
 *  form follows function, in this case.
 *
 * Results:
 *  Return code from fetch.
 *
 * Side effects:
 *  Sybase server connection is opened.
 *
 *----------------------------------------------------------------------
 */
 
static CS_RETCODE
syb_fetch_next_result (
		       CS_COMMAND		*cmd,
		       SybRow		**pDataRow,
		       int			*rowCount,
		       int			*spResult,
		       SybRow		**pSpParamRow,
		       int			*cmd_done_received,
		       int			*cmd_succeed_received,
		       int			*cmd_fail_received,
		       int			*transactionState,
		       int			*spResultsFetched
		       )
{
    int			keepFetching = 1;
    CS_RETCODE		returnCode;
    CS_RETCODE		rowCountResultCode;
    CS_RETCODE		cancelReturnCode;
    CS_RETCODE		fetchReturnCode;
    CS_INT		resultType;
    char		*me = "syb_fetch_next_result";
    SybRow		*tmpRow;
    CS_INT		rowsFetched;
    int			spParamsFetched = 0;
    int			transtate;

    *cmd_done_received = 0;
    *cmd_succeed_received = 0;
    *cmd_fail_received = 0;
    *spResultsFetched = 0;

    Ns_PdLog(Trace, "%s: entry.", me);
    while (keepFetching) {
	returnCode = ct_results(cmd, &resultType);
	if (returnCode == CS_SUCCEED) {
	    if (resultType == CS_ROW_RESULT) {
		Ns_PdLog(Trace, "%s: ct_results set: CS_ROW_RESULT", me);
		if (*pDataRow != NULL) {
		    SybRow_free(*pDataRow);
		    *pDataRow = NULL;
		}
		*pDataRow = SybRow_newFromResult(cmd, CS_CHAR_TYPE);
		keepFetching = 0;
	    } else if (resultType == CS_COMPUTE_RESULT) {
		Ns_PdLog(Trace,
			 "%s: ct_results set: CS_COMPUTE_RESULT", me);
		if (*pDataRow != NULL) {
		    SybRow_free(*pDataRow);
		    *pDataRow = NULL;
		}
		*pDataRow = SybRow_newFromResult(cmd, CS_CHAR_TYPE);
		keepFetching = 0;
	    } else if (resultType == CS_CMD_FAIL) {
		(*cmd_fail_received)++;
		Ns_PdLog(Trace, "%s: ct_results set: CS_CMD_FAIL", me);
		transtate = syb_check_transaction_state(cmd);
		if (transtate >= 0) {
		    *transactionState = transtate;
		}
		/* We need to return the failure to the application here,
		   so cancel the rest of the results, and then drain
		   off what we have to. */
		returnCode = ct_cancel(NULL, cmd, CS_CANCEL_ALL);
		if (returnCode != CS_SUCCEED) {
		    Ns_PdLog(Error, "%s: "
			     "ct_cancel failed after CS_CMD_FAIL: %d.", me,
			     returnCode);
		} else {
		    syb_drain_results (cmd);
		}
		/*
		 * Strictly speaking this return code is a lie, but
		 * it maps to what AOLserver can handle.  This is
		 * also a little tricky -- if the actual returnCode was
		 * CS_FAIL, then we'd have some mandatory handling to
		 * do; by injecting the returnCode in here, and by
		 * handling the returnCode with an if, rather than a
		 * switch, we can avoid the handling (which would not be
		 * good to do here).
		 */
		keepFetching = 0;
		returnCode = CS_FAIL;
	    } else if (resultType == CS_STATUS_RESULT) {
		Ns_PdLog(Trace, "%s: ct_results set: CS_STATUS_RESULT", me);
		/* Process a stored procedure return code. */
		if ((*spResultsFetched)++ > 0) {
		    /* 
		     * In theory, there's only one stored procedure result,
		     * so if we encounter a second one, it's an error
		     */
		    Ns_PdLog(Error, "%s: Fetching SP result number %d", me,
			     *spResultsFetched);
		}
		tmpRow = SybRow_newFromResult(cmd, CS_ILLEGAL_TYPE);
		fetchReturnCode = syb_fetch_row(cmd, &rowsFetched, NULL);
		if (fetchReturnCode == CS_SUCCEED) {
		    memcpy (spResult, tmpRow->columns[0].value,
			    sizeof(*spResult));
		}
		syb_drain_rows(cmd);
		SybRow_free(tmpRow);
	    } else if (resultType == CS_PARAM_RESULT) {
		Ns_PdLog(Trace, "%s: ct_results set: CS_PARAM_RESULT", me);
		/*
		 * Process stored procedure return parameters.
		 */
		if (spParamsFetched++ > 0) {
		    /*
		     * There should only be one stored procedure output
		     * parameter set, too
		     */
		    Ns_PdLog(Error, "%s: Fetching SP param number %d", me,
			     spParamsFetched);
		}
		tmpRow = SybRow_newFromResult(cmd, CS_CHAR_TYPE);
		fetchReturnCode = syb_fetch_row (cmd, &rowsFetched, NULL);
		if (fetchReturnCode == CS_SUCCEED) {
		    if (*pSpParamRow != NULL) {
			SybRow_free(*pSpParamRow);
			*pSpParamRow = NULL;
		    }
		    *pSpParamRow = SybRow_dup(tmpRow);
		}
		syb_drain_rows(cmd);
		SybRow_free(tmpRow);
	    } else if (resultType == CS_CMD_SUCCEED) {
		Ns_PdLog(Trace, "%s: ct_results set: CS_CMD_SUCCEED",
			 me);
		(*cmd_succeed_received)++;
		transtate = syb_check_transaction_state(cmd);
		if (transtate >= 0) {
		    *transactionState = transtate;
		}
	    } else if (resultType == CS_CMD_DONE) {
		Ns_PdLog(Trace, "%s: ct_results set: CS_CMD_DONE", me);
		(*cmd_done_received)++;
		transtate = syb_check_transaction_state(cmd);
		if (transtate >= 0) {
		    *transactionState = transtate;
		}
		/*
		 * Fetch result row count and go on
		 */
		rowCountResultCode = ct_res_info(cmd, CS_ROW_COUNT,
						 rowCount, CS_UNUSED, NULL);
		if (rowCountResultCode != CS_SUCCEED) {
		    Ns_PdLog(Error,
			     "%s: ct_res_info to get row count failed: %d.", me,
			     rowCountResultCode);
		} else {
		    Ns_PdLog(Trace, "%s: Number of affected rows: %d", me,
			     *rowCount);
		}
	    } else {
		Ns_PdLog(Error, "%s: "
			 "ct_results returned unexpected result type: %d", me,
			 resultType);
	    }
	} else {
	    keepFetching = 0;
	    if (returnCode == CS_END_RESULTS) {
		Ns_PdLog(Trace, "%s: "
			 "ct_results returnCode: CS_END_RESULTS", me);
		/*
		 * No action required
		 */
	    } else if (returnCode == CS_CANCELED) {
		Ns_PdLog(Trace, "%s: "
			 "ct_results returnCode: CS_CANCELED", me);
		/*
		 * No action required
		 */
	    } else if (returnCode == CS_FAIL) {
		/*
		 * The command failed. No more results are available.
		 * We are required to call ct_cancel with CS_CANCEL_ALL.
		 */
		Ns_PdLog(Error, "%s: ct_results() returned CS_FAIL", me);
		cancelReturnCode = ct_cancel(NULL, cmd, CS_CANCEL_ALL);
		if (cancelReturnCode != CS_SUCCEED) {
		    Ns_PdLog(Error, "%s: "
			     "after ct_results CS_FAIL, ct_cancel returned %d",
			     me, cancelReturnCode);
		    /*
		     * We're supposed to call ct_close at this point,
		     * but I haven't figured out what else I'd need to
		     * do to prevent the whole mess from exploding.
		     */
		}
	    } else {
		Ns_PdLog(Error, "%s: ct_results: unexpected return: %d",
			 me, returnCode);
	    }
	}
    }
    
    Ns_PdLog(Trace, "%s: Returning %d", me, returnCode);
    return returnCode;
}


/*
 *----------------------------------------------------------------------
 * syb_fetch_row -
 *
 *  Fetch the next row from the Sybase result.	If there are no more
 *  results, begin processing further results.
 *
 * Results:
 *  result code from fetch.
 *
 * Side effects:
 *  If this was the last row in a result, processing of subsequent
 *  results commences.	If there are no subsequent results with rows
 *  to return, then all results will be processed.
 *
 *----------------------------------------------------------------------
 */
 
static CS_RETCODE
syb_fetch_row (CS_COMMAND *cmd, CS_INT *rowsFetched, SybRow *row)
{
    char		*me = "syb_fetch_row";
    CS_RETCODE		fetchReturnCode;
    int			i0;

    fetchReturnCode = ct_fetch(cmd, CS_UNUSED, CS_UNUSED, CS_UNUSED,
			       rowsFetched);
    if (fetchReturnCode == CS_SUCCEED) {
	Ns_PdLog(Trace, "%s: ct_fetch Fetched %d rows", me, *rowsFetched);
    } else if (fetchReturnCode == CS_ROW_FAIL) {
	Ns_PdLog(Trace, "%s: ct_fetch Fetch %d rows failed", me, *rowsFetched);
	if (row != NULL) {
	    for (i0 = 0; i0 < row->nrColumnsAllocated; i0++) {
		if (row->columns[i0].indicator > 0) {
		    Ns_PdLog(Trace, "%s: "
			     "Column %d (%.*s) truncated from %d to %d", me,
			     i0,
			     row->columns[i0].datafmt.namelen,
			     row->columns[i0].datafmt.name,
			     row->columns[i0].indicator,
			     row->columns[i0].datafmt.maxlength);
		}
	    }
	}
    } else if (fetchReturnCode == CS_END_DATA) {
	Ns_PdLog(Trace, "%s: ct_fetch returns end of data", me);
    } else {
	Ns_PdLog(Error, "%s: ct_fetch unexpected result: %d", me,
		 fetchReturnCode);
    }

    return fetchReturnCode;
}


/*
 *----------------------------------------------------------------------
 * syb_init -
 *
 *  Open a connection to the Sybase server and record the necessary
 *  values in the state structure.
 *
 * Results:
 *  NS_ERROR - The command execution failed.
 *  DB_ROWS - the command execution succeeded, and there are rows available
 *  DB_DML - the command execution succeeded, and there are no rows available.
 *     Any stored procedure results (return codes or output parameters) are
 *     available in the state structure.
 *
 * Side effects:
 *  Sybase server connection is opened.
 *
 *----------------------------------------------------------------------
 */
 
static CS_RETCODE CS_PUBLIC
syb_init(SybState *state)
{
    CS_RETCODE	retcode;
    CS_INT		netio_type = CS_SYNC_IO;
    CS_INT		con_temp = CS_TRUE;
    CS_CONTEXT	**context = &state->cs_context;
    /*
     * Get a context handle to use.
     */
    retcode = cs_ctx_alloc(SYB_CTLIB_VERSION, context);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_init: cs_ctx_alloc() failed");
	return retcode;
    }
    /*
     * Initialize Open Client.
     */
    retcode = ct_init(*context, SYB_CTLIB_VERSION);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_init: ct_init() failed");
	cs_ctx_drop(*context);
	*context = NULL;
	return retcode;
    }

#ifdef SYB_API_DEBUG
    /*
     * ct_debug stuff. Enable this function right before any call to
     * OC-Lib that is returning failure.
     */
    retcode = ct_debug(*context, NULL, CS_SET_FLAG, CS_DBG_API_STATES,
		       NULL, CS_UNUSED);
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_init: ct_debug() failed");
    }
#endif

    /* 
     * Associate the (SybState *) state with the context for retrieval
     * in callbacks.  This will be used to set exceptionMsg (and possibly
     * exceptionCode)
     */
    if ((retcode = cs_config(*context, CS_SET, CS_USERDATA, (CS_VOID *)&state,
			     (sizeof (SybState *)), NULL)) != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_init: cs_config(USERDATA) failed");
    }
    /*
     * Install client and server message handlers.
     */
    if (retcode == CS_SUCCEED) {
	retcode = ct_callback(*context, NULL, CS_SET, CS_CLIENTMSG_CB,
			      (CS_VOID *)syb_clientmsg_cb);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "syb_init: ct_callback(clientmsg) failed");
	}
    }
    if (retcode == CS_SUCCEED) {
	retcode = ct_callback(*context, NULL, CS_SET, CS_SERVERMSG_CB,
			      (CS_VOID *)syb_servermsg_cb);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "syb_init: ct_callback(servermsg) failed");
	}
    }
    /* 
     * This is an synchronous example so set the input/output type
     * to synchronous (This is the default setting, but show an
     * example anyway).
     */
    if (retcode == CS_SUCCEED) {
	retcode = ct_config(*context, CS_SET, CS_NETIO, &netio_type, 
			    CS_UNUSED, NULL);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "syb_init: ct_config(netio) failed");
	}
    }
    if (retcode == CS_SUCCEED) {
	retcode = ct_config(*context, CS_SET, CS_NOINTERRUPT, &con_temp,
			    CS_UNUSED, NULL);
	if (retcode != CS_SUCCEED) {
	    Ns_PdLog(Error, "syb_init: ct_config(nointerrupt) failed");
	}
    }
    if (retcode != CS_SUCCEED) {
	Ns_PdLog(Trace, "syb_init: Initialization failed; forcing close");
	ct_exit(*context, CS_FORCE_EXIT);
	cs_ctx_drop(*context);
	*context = NULL;
    } else {
	Ns_PdLog(Trace, "syb_init: Initialization succeeded");
    }
    return retcode;
}


/*
 *----------------------------------------------------------------------
 * syb_servermsg_cb -
 *
 *  Callback for processing server messages.  Since the AOLserver proxy
 *  protocol only allows a single exception result, store the message
 *  in the state structure for later transmission to the AOLserver.  It's
 *  possible for a Sybase command to result in several messages, both
 *  client and server.	Since we can only store one for transmission to
 *  the AOLserver, all messages are logged via syslog.	Note that the server
 *  and client message callbacks share the same storage buffer in the state
 *  structure, so the messages can overwrite each other.
 *
 * Results:
 *  CS_SUCCEED - everything worked.
 *
 * Side effects:
 *  Message is logged, and recorded in the state structure.
 *
 *----------------------------------------------------------------------
 */
 
static CS_RETCODE CS_PUBLIC
syb_servermsg_cb(CS_CONTEXT *context, CS_CONNECTION *connection, 
		 CS_SERVERMSG	*srvmsg)
{
    SybState *state;

    /*
     * ignore database context change messages
     */
    if (srvmsg->msgnumber != DBCONTEXT_CHANGE) {
	Ns_PdLog(Error, "Server callback msg#: "
		 "%ld, Severity %ld, State %ld, Line %ld",
		 srvmsg->msgnumber, srvmsg->severity,
		 srvmsg->state, srvmsg->line);
	if (srvmsg->svrnlen > 0) {
	    Ns_PdLog(Error, "Server '%s'", srvmsg->svrname);
	}
	if (srvmsg->proclen > 0) {
	    Ns_PdLog(Error, " Procedure '%s'", srvmsg->proc);
	}
	Ns_PdLog(Error, "Server callback msg: \"%s\"", srvmsg->text);
	strTrimTrailingSpace(srvmsg->text);
	if (strcmp(srvmsg->text, "Command has been aborted.") != 0) {
	    if (cs_config(context, CS_GET, CS_USERDATA, (CS_VOID *)&state,
			  sizeof(SybState *), NULL) != CS_SUCCEED) {
		Ns_PdLog(Error, "syb_servermsg_cb: "
			 "cannot get SybState pointer");
	    } else {
		strcpy(state->exceptionCode, "ERROR");
		strcpy(state->exceptionMsg, srvmsg->text);
	    }
	}
    }
    return CS_SUCCEED;
}


/*
 *----------------------------------------------------------------------
 * syb_use_db -
 *
 *  Set the database to use.  This is accomplished by executing a
 *  Sybase language command.
 *
 * Results:
 *  Result code from command execution.
 *
 * Side effects:
 *  Sybase connection is now attached to indicated database
 *
 *----------------------------------------------------------------------
 */
CS_RETCODE
syb_use_db(CS_CONNECTION *connection, char *dbname)
{
    CS_RETCODE	    retcode;
    CS_CHAR	    *cmdbuf;
    
    /*
     * Allocate the buffer for the command string.
     */
    cmdbuf = (CS_CHAR *) malloc(SYB_CMDBUFSIZE);
    if (cmdbuf == (CS_CHAR *)NULL) {
	Ns_PdLog(Error, "syb_use_db: malloc() failed");
	return CS_FAIL;
    }
    
    /*
     * Set up and send the command to use the database
     */
    sprintf(cmdbuf, "use %s\n", dbname);

    if ((retcode = syb_execute_cmd(connection, cmdbuf)) != CS_SUCCEED) {
	Ns_PdLog(Error, "syb_use_db: syb_execute_cmd(use db) failed");
    }
    free(cmdbuf);
    return retcode;
}
