/*
    ras - Redundant Archive System
    Copyright (C) 1999  Nick Cleaton

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    Nick Cleaton <nick@cleaton.net>
*/
/*

applysum.h

This module provides a function which reads input data from a set of input
files, computes output data and writes it to a set of output files.  This
is the underlying RAS transform that is used both to generate 1 or more
sumfiles from 2 or more segfiles and to regenerate 1 or more missing
segfiles from a mixture of segfiles and sumfiles.

The guts of the transform is in field.c, this module just sorts out the
boring details like handling segfiles of different lengths correctly.

In order to be able to handle segfiles of different lengths, a 1 is
appended to each segfile when it gets read in and underlength ones are
then padded with 0s.   When writing the segfiles out again the last 1
(followed by 0 or more 0s) is pruned away. This is the behaviour that 
is turned off by the -F option.

For example, given some (very short) segfiles ...

xaa contains 3 bytes : 12, 0, 129
xab contains 1 bytes : 45
xac contains 2 bytes : 0,  0

When generating sumfiles the 3 segfiles are read in, 1s are appended
and they are padded to the same length.

xaa: 12, 0,  129, 1
xab: 45, 1,  0,   0
xac: 0,  0,  1,   0

Now sumfile 0 is generated (see field.c for how) 

sum0: X, X, X, X

... and these 4 bytes are written out to file sum0.

now segment xac is lost. We have ...

xaa:  12, 0,  129
xab:  45
sum0: X,  X,  X,  X

We pad the input segfiles to length again ...

xaa:  12, 0,  129, 1
xab:  45, 1,  0,   0
sum0: X,  X,  X,   X

We use the method in field.c to regenerate xac ...

xac: 0, 0, 1, 0

... and now we strip off the trailing 1,0 and write the bytes 0,0 out
to file xac.

*/ 



/* An inblkstream is a file open for reading. It may be either a segfile
   or a sumfile.
*/
struct inblkstream
{  FILE *f;         /*  the open file  */
   char *name;      /*  its name  */
   int is_sum;      /*  1 for a sumfile, 0 for a segfile  */
   int x_val;       /*  The x value associated with this data block  */
};


/* An outblkstream is a file open for writing.  It may either be a segfile
   or a sumfile.
*/
struct outblkstream
{  FILE *f;         /*  the open file  */
   char *name;      /*  its name  */
   int is_sum;      /*  1 for a sumfile, 0 for a segfile  */
   int x_val;       /*  The x value associated with this data block  */
};


void process_data(size_t incount,  struct inblkstream *inblks, 
                  size_t outcount, struct outblkstream *outblks,
		  size_t blocksize, int footer);
/* 
   Reads data from incount inblkstreams (pointed to by inblks) and, for each
   of the outcount outblkstreams pointed to by outblks, computes output data
   and writes it out.  blocksize is the number of bytes read and written at
   a time, and footer is a flag to enable the algorithm described above for
   dealing with segfiles of different lengths.
*/



