/******************************************************************************\
 gnofin/history.c   $Revision: 1.2 $
 Copyright (C) 1999 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include <stdio.h>
#include <gtk/gtksignal.h>
#include "gnofin.h"
#include "history.h"

enum {
  FREE_ITEM,
  UNDO_ITEM,
  REDO_ITEM,
  LAST_SIGNAL
};

static void fin_history_class_init	(FinHistoryClass * klass);
static void fin_history_init		(FinHistory      * history);
static void fin_history_real_destroy	(GtkObject	 * object);
static void fin_history_real_free_item	(FinHistory	 * history,
					 gpointer	   item);
static void fin_history_real_undo_item	(FinHistory	 * history,
					 gpointer	   item);
static void fin_history_real_redo_item	(FinHistory	 * history,
					 gpointer	   item);

static GtkObjectClass * parent_class = NULL;
static guint history_signals[LAST_SIGNAL] = {0};

GtkType
fin_history_get_type (void)
{
  static GtkType type = 0;

  if (!type)
  {
    static const GtkTypeInfo info =
    {
      "FinHistory",
      sizeof (FinHistory),
      sizeof (FinHistoryClass),
      (GtkClassInitFunc) fin_history_class_init,
      (GtkObjectInitFunc) fin_history_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc) NULL,
    };
    type = gtk_type_unique(GTK_TYPE_OBJECT, &info);
  }
  return type;
}

static void
fin_history_class_init (FinHistoryClass * klass)
{
  GtkObjectClass * object_class;

  fin_trace("");

  object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  history_signals[FREE_ITEM] =
    gtk_signal_new("free_item",
    		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(FinHistoryClass, free_item),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);
  history_signals[UNDO_ITEM] =
    gtk_signal_new("undo_item",
    		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(FinHistoryClass, undo_item),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);
  history_signals[REDO_ITEM] =
    gtk_signal_new("redo_item",
    		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(FinHistoryClass, redo_item),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);

  gtk_object_class_add_signals(object_class, history_signals, LAST_SIGNAL);

  object_class->destroy = fin_history_real_destroy;

  klass->free_item = fin_history_real_free_item;
  klass->undo_item = fin_history_real_undo_item;
  klass->redo_item = fin_history_real_redo_item;
}

static void
fin_history_init (FinHistory * history)
{
  history->items = NULL;
  history->current = NULL;
  history->can_undo = 0;
  history->can_redo = 0;
}

static void
fin_history_real_destroy (GtkObject * object)
{
  fin_trace("");

  fin_history_clear(FIN_HISTORY(object));

  parent_class->destroy(object);
}

static void
fin_history_real_free_item (FinHistory * history,
			    gpointer     item)
{
  fin_trace("");

  g_return_if_fail(history != NULL);
  g_return_if_fail(FIN_IS_HISTORY(history));

  /* do nothing */
}

static void
fin_history_real_undo_item (FinHistory * history,
			    gpointer     item)
{
  fin_trace("");

  g_return_if_fail(history);
  g_return_if_fail(FIN_IS_HISTORY(history));
  g_return_if_fail(history->items);
  g_return_if_fail(history->current);

  history->current = history->current->prev;

  if (history->current)
    history->can_undo = 1;
  else
    history->can_undo = 0;

  history->can_redo = 1;
}

static void
fin_history_real_redo_item (FinHistory * history,
			    gpointer     item)
{
  fin_trace("");

  g_return_if_fail(history);
  g_return_if_fail(FIN_IS_HISTORY(history));
  g_return_if_fail(history->items);

  if (history->current)
  {
    g_return_if_fail(history->current->next != NULL);
    history->current = history->current->next;
  }
  else
    history->current = history->items;

  g_assert(history->current != NULL);

  if (history->current->next)
    history->can_redo = 1;
  else
    history->can_redo = 0;
  history->can_undo = 1;
}

FinHistory *
fin_history_new (void)
{
  return FIN_HISTORY(gtk_type_new(FIN_TYPE_HISTORY));
}

void
fin_history_undo (FinHistory * history)
{
  fin_trace("");
  
  g_return_if_fail(history != NULL);
  g_return_if_fail(FIN_IS_HISTORY(history));

  if (history->can_undo)
  {
    g_assert(history->current != NULL);
    gtk_signal_emit(GTK_OBJECT(history), history_signals[UNDO_ITEM], history->current->data);
  }
}

void
fin_history_redo (FinHistory * history)
{
  fin_trace("");
  
  g_return_if_fail(history != NULL);
  g_return_if_fail(FIN_IS_HISTORY(history));

  if (history->can_redo)
  {
    gpointer item;

    if (history->current)
    {
      g_assert(history->current->next != NULL);
      item = history->current->next->data;
    }
    else
    {
      g_assert(history->items != NULL);
      item = history->items->data;
    }

    gtk_signal_emit(GTK_OBJECT(history), history_signals[REDO_ITEM], item);
  }
}

void
fin_history_remember (FinHistory * history,
		      gpointer     item)
{
  GList * it, * head;

  fin_trace("");
  
  g_return_if_fail(history != NULL);
  g_return_if_fail(FIN_IS_HISTORY(history));

  /* free all above current */
  if (history->current)
  {
    head = history->current->next;
    history->current->next = NULL;
  }
  else
  {
    head = history->items;
    history->items = NULL;
  }
  if (head)
  {
    for (it=head; it; it=it->next)
    {
      gtk_signal_emit(GTK_OBJECT(history), history_signals[FREE_ITEM], it->data);
      it->data = NULL;
    }
    g_list_free(head);
  }

  if (history->current)
    g_assert(history->current->next == NULL);

  /* append to history -- could be more efficient */
  history->items = g_list_first(g_list_append(history->current, item));
  g_assert(history->items->prev == NULL);
  history->current = g_list_last(history->items);

  /* set flags */
  history->can_redo = 0;
  history->can_undo = 1;
}

void
fin_history_clear (FinHistory * history)
{
  GList * it;

  fin_trace("");
  
  g_return_if_fail(history != NULL);
  g_return_if_fail(FIN_IS_HISTORY(history));

  for (it=history->items; it; it=it->next)
    gtk_signal_emit(GTK_OBJECT(history), history_signals[FREE_ITEM], it->data);

  g_list_free(history->items);

  history->items = NULL;
  history->current = NULL;
  history->can_undo = 0;
  history->can_redo = 0;
}

void
fin_history_dump (FinHistory * history)
{
  GList * it, * head;

  g_return_if_fail(history != NULL);

  printf("history->can_undo\t%d\n", (int) history->can_undo);
  printf("history->can_redo\t%d\n", (int) history->can_redo);

  printf("history->items (length)\t%d\n", g_list_length(history->items));
  printf("history->items\t%p\n", history->items);
  printf("history->current\t%p\n", history->current);

  printf("undo items:\n");
  head = history->current;
  for (it=head; it; it=it->prev)
    printf("  item\t%p\n", it->data);

  printf("redo items:\n");
  head = history->current ? history->current->next : history->items;
  for (it=head; it; it=it->next)
    printf("  item\t%p\n", it->data);
}
