/*
 * Copyright (c) 1998  Kazushi (Jam) Marukawa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice in the documentation and/or other materials provided with 
 *    the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * Routines to unify a multi bytes character.
 */

#include "defines.h"
#include "multi.h"


#if ISO

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

typedef struct {
	char* input;
	char* output;
	CHARSET charset;
} convtab;

typedef struct {
	int num;
	convtab tab[1];
} sortedconvtab;

typedef struct {
	convtab* ctab;
	sortedconvtab* sctab;
} convtable;

static int comp_convtab(p, q)
void* p;
void* q;
{
	return strcmp(((convtab*)p)->input, ((convtab*)q)->input);
}

static sortedconvtab* make_sortedconvtab(tab)
convtab tab[];
{
	int i;
	sortedconvtab* sctab;

	for (i = 0; tab[i].input != NULL; i++)
		;
	sctab = (sortedconvtab*)malloc(sizeof(sortedconvtab) +
				       sizeof(convtab) * i - 1);
	if (sctab == NULL)
		return NULL;
	sctab->num = i;
	for (i = 0; i < sctab->num; i++)
		sctab->tab[i] = tab[i];
	qsort(sctab->tab, sctab->num, sizeof(convtab), comp_convtab);
	return sctab;
}

static convtab* find_convtab_from_sctab(sctab, input)
sortedconvtab* sctab;
char* input;
{
	int from = 0;
	int to = sctab->num;
	int cur;
	int cmp;

	if (to == 0)
		return NULL;
	while (1) {
		cur = (from + to) / 2;
		cmp = strcmp(input, sctab->tab[cur].input);
		if (cmp == 0)
			return &sctab->tab[cur];
		if (to - from == 1)
			return NULL;
		if (cmp < 0)
			to = cur;
		if (0 < cmp)
			from = cur;
	}
}

static void init_convtable(ctable)
convtable* ctable;
{
	if (ctable->sctab == NULL)
		ctable->sctab = make_sortedconvtab(ctable->ctab);
}

static convtab* find_convtab(ctable, input)
convtable* ctable;
char* input;
{
	convtab* ptab;

	if (ctable->sctab == NULL) {
		init_convtable(ctable);
	}
	if (ctable->sctab != NULL)
		return find_convtab_from_sctab(ctable->sctab, input);

	for (ptab = ctable->ctab; ptab->input; ptab++)
		if (strcmp(input, ptab->input) == 0)
			return ptab;
}

static convtab conv_jisx0208_78_90[] = {
	/* 0x3646($@6F(B) -> 0x7421(&@$Bt!(B) */
	{ "6F", "t!", JISX0208_90KANJI },
	/* 0x4B6A($@Kj(B) -> 0x7422(&@$Bt"(B) */
	{ "Kj", "t\"", JISX0208_90KANJI },
	/* 0x4D5A($@MZ(B) -> 0x7423(&@$Bt#(B) */
	{ "MZ", "t#", JISX0208_90KANJI },
	/* 0x6076($@`v(B) -> 0x7424(&@$Bt$(B) */
	{ "`v", "t$", JISX0208_90KANJI },
	/* 0x3033($@03(B) -> 0x724D(&@$BrM(B) */
	{ "03", "rM", JISX0208_90KANJI },
	/* 0x724D($@rM(B) -> 0x3033(&@$B03(B) */
	{ "rM", "03", JISX0208_90KANJI },
	/* 0x3229($@2)(B) -> 0x7274(&@$Brt(B) */
	{ "2)", "rt", JISX0208_90KANJI },
	/* 0x7274($@rt(B) -> 0x3229(&@$B2)(B) */
	{ "rt", "2)", JISX0208_90KANJI },
	/* 0x3342($@3B(B) -> 0x695A(&@$BiZ(B) */
	{ "3B", "iZ", JISX0208_90KANJI },
	/* 0x695A($@iZ(B) -> 0x3342(&@$B3B(B) */
	{ "iZ", "3B", JISX0208_90KANJI },
	/* 0x3349($@3I(B) -> 0x5978(&@$BYx(B) */
	{ "3I", "Yx", JISX0208_90KANJI },
	/* 0x5978($@Yx(B) -> 0x3349(&@$B3I(B) */
	{ "Yx", "3I", JISX0208_90KANJI },
	/* 0x3376($@3v(B) -> 0x635E(&@$Bc^(B) */
	{ "3v", "c^", JISX0208_90KANJI },
	/* 0x635E($@c^(B) -> 0x3376(&@$B3v(B) */
	{ "c^", "3v", JISX0208_90KANJI },
	/* 0x3443($@4C(B) -> 0x5E75(&@$B^u(B) */
	{ "4C", "^u", JISX0208_90KANJI },
	/* 0x5E75($@^u(B) -> 0x3443(&@$B4C(B) */
	{ "^u", "4C", JISX0208_90KANJI },
	/* 0x3452($@4R(B) -> 0x6B5D(&@$Bk](B) */
	{ "4R", "k]", JISX0208_90KANJI },
	/* 0x6B5D($@k](B) -> 0x3452(&@$B4R(B) */
	{ "k]", "4R", JISX0208_90KANJI },
	/* 0x375B($@7[(B) -> 0x7074(&@$Bpt(B) */
	{ "7[", "pt", JISX0208_90KANJI },
	/* 0x7074($@pt(B) -> 0x375B(&@$B7[(B) */
	{ "pt", "7[", JISX0208_90KANJI },
	/* 0x395C($@9\(B) -> 0x6268(&@$Bbh(B) */
	{ "9\\", "bh", JISX0208_90KANJI },
	/* 0x6268($@bh(B) -> 0x395C(&@$B9\(B) */
	{ "bh", "9\\", JISX0208_90KANJI },
	/* 0x3C49($@<I(B) -> 0x6922(&@$Bi"(B) */
	{ "<I", "i\"", JISX0208_90KANJI },
	/* 0x6922($@i"(B) -> 0x3C49(&@$B<I(B) */
	{ "i\"", "<I", JISX0208_90KANJI },
	/* 0x3F59($@?Y(B) -> 0x7057(&@$BpW(B) */
	{ "?Y", "pW", JISX0208_90KANJI },
	/* 0x7057($@pW(B) -> 0x3F59(&@$B?Y(B) */
	{ "pW", "?Y", JISX0208_90KANJI },
	/* 0x4128($@A((B) -> 0x6C4D(&@$BlM(B) */
	{ "A(", "lM", JISX0208_90KANJI },
	/* 0x6C4D($@lM(B) -> 0x4128(&@$BA((B) */
	{ "lM", "A(", JISX0208_90KANJI },
	/* 0x445B($@D[(B) -> 0x5464(&@$BTd(B) */
	{ "D[", "Td", JISX0208_90KANJI },
	/* 0x5464($@Td(B) -> 0x445B(&@$BD[(B) */
	{ "Td", "D[", JISX0208_90KANJI },
	/* 0x4557($@EW(B) -> 0x626A(&@$Bbj(B) */
	{ "EW", "bj", JISX0208_90KANJI },
	/* 0x626A($@bj(B) -> 0x4557(&@$BEW(B) */
	{ "bj", "EW", JISX0208_90KANJI },
	/* 0x456E($@En(B) -> 0x5B6D(&@$B[m(B) */
	{ "En", "[m", JISX0208_90KANJI },
	/* 0x5B6D($@[m(B) -> 0x456E(&@$BEn(B) */
	{ "[m", "En", JISX0208_90KANJI },
	/* 0x4573($@Es(B) -> 0x5E39(&@$B^9(B) */
	{ "Es", "^9", JISX0208_90KANJI },
	/* 0x5E39($@^9(B) -> 0x4573(&@$BEs(B) */
	{ "^9", "Es", JISX0208_90KANJI },
	/* 0x4676($@Fv(B) -> 0x6D6E(&@$Bmn(B) */
	{ "Fv", "mn", JISX0208_90KANJI },
	/* 0x6D6E($@mn(B) -> 0x4676(&@$BFv(B) */
	{ "mn", "Fv", JISX0208_90KANJI },
	/* 0x4768($@Gh(B) -> 0x6A24(&@$Bj$(B) */
	{ "Gh", "j$", JISX0208_90KANJI },
	/* 0x6A24($@j$(B) -> 0x4768(&@$BGh(B) */
	{ "j$", "Gh", JISX0208_90KANJI },
	/* 0x4930($@I0(B) -> 0x5B58(&@$B[X(B) */
	{ "I0", "[X", JISX0208_90KANJI },
	/* 0x5B58($@[X(B) -> 0x4930(&@$BI0(B) */
	{ "[X", "I0", JISX0208_90KANJI },
	/* 0x4B79($@Ky(B) -> 0x5056(&@$BPV(B) */
	{ "Ky", "PV", JISX0208_90KANJI },
	/* 0x5056($@PV(B) -> 0x4B79(&@$BKy(B) */
	{ "PV", "Ky", JISX0208_90KANJI },
	/* 0x4C79($@Ly(B) -> 0x692E(&@$Bi.(B) */
	{ "Ly", "i.", JISX0208_90KANJI },
	/* 0x692E($@i.(B) -> 0x4C79(&@$BLy(B) */
	{ "i.", "Ly", JISX0208_90KANJI },
	/* 0x4F36($@O6(B) -> 0x6446(&@$BdF(B) */
	{ "O6", "dF", JISX0208_90KANJI },
	/* 0x6446($@dF(B) -> 0x4F36(&@$BO6(B) */
	{ "dF", "O6", JISX0208_90KANJI },
	/* NULL */
	{ 0, 0, 0 }
};
static convtable ctable_jisx0208_78_90 = { conv_jisx0208_78_90, NULL };

static convtab unify_jisx0208[] = {
	/* 0x2121(&@$B!!(B) -> 0x20( ) */
	{ "!!", " ", ASCII },
	/* 0x2122(&@$B!"(B) -> 0x2C(,) */
	{ "!\"", ",", ASCII },
	/* 0x2123(&@$B!#(B) -> 0x2E(.) */
	{ "!#", ".", ASCII },
	/* 0x2124(&@$B!$(B) -> 0x2C(,) */
	{ "!$", ",", ASCII },
	/* 0x2125(&@$B!%(B) -> 0x2E(.) */
	{ "!%", ".", ASCII },
	/* 0x2127(&@$B!'(B) -> 0x3A(:) */
	{ "!'", ":", ASCII },
	/* 0x2128(&@$B!((B) -> 0x3B(;) */
	{ "!(", ";", ASCII },
	/* 0x2129(&@$B!)(B) -> 0x3F(?) */
	{ "!)", "?", ASCII },
	/* 0x212A(&@$B!*(B) -> 0x21(!) */
	{ "!*", "!", ASCII },
	/* 0x2130(&@$B!0(B) -> 0x5E(^) */
	{ "!0", "^", ASCII },
	/* 0x2132(&@$B!2(B) -> 0x5F(_) */
	{ "!2", "_", ASCII },
	/* 0x213D(&@$B!=(B) -> 0x2D(-) */
	{ "!=", "-", ASCII },
	/* 0x213E(&@$B!>(B) -> 0x2D(-) */
	{ "!>", "-", ASCII },
	/* 0x213F(&@$B!?(B) -> 0x2F(/) */
	{ "!?", "/", ASCII },
	/* 0x2140(&@$B!@(B) -> 0x5C(\) */
	{ "!@", "\\", ASCII },
	/* 0x2141(&@$B!A(B) -> 0x2D(-) */
	{ "!A", "-", ASCII },
	/* 0x2143(&@$B!C(B) -> 0x7C(|) */
	{ "!C", "|", ASCII },
	/* 0x2146(&@$B!F(B) -> 0x27(') */
	{ "!F", "'", ASCII },
	/* 0x2147(&@$B!G(B) -> 0x27(') */
	{ "!G", "'", ASCII },
	/* 0x2148(&@$B!H(B) -> 0x22(") */
	{ "!H", "\"", ASCII },
	/* 0x2149(&@$B!I(B) -> 0x22(") */
	{ "!I", "\"", ASCII },
	/* 0x214A(&@$B!J(B) -> 0x28(() */
	{ "!J", "(", ASCII },
	/* 0x214B(&@$B!K(B) -> 0x29()) */
	{ "!K", ")", ASCII },
	/* 0x214C(&@$B!L(B) -> 0x5B([) */
	{ "!L", "[", ASCII },
	/* 0x214D(&@$B!M(B) -> 0x5D(]) */
	{ "!M", "]", ASCII },
	/* 0x214E(&@$B!N(B) -> 0x5B([) */
	{ "!N", "[", ASCII },
	/* 0x214F(&@$B!O(B) -> 0x5D(]) */
	{ "!O", "]", ASCII },
	/* 0x2150(&@$B!P(B) -> 0x7B({) */
	{ "!P", "{", ASCII },
	/* 0x2151(&@$B!Q(B) -> 0x7D(}) */
	{ "!Q", "}", ASCII },
	/* 0x2152(&@$B!R(B) -> 0x5B([) */
	{ "!R", "[", ASCII },
	/* 0x2153(&@$B!S(B) -> 0x5D(]) */
	{ "!S", "]", ASCII },
	/* 0x2154(&@$B!T(B) -> 0x5B([) */
	{ "!T", "[", ASCII },
	/* 0x2155(&@$B!U(B) -> 0x5D(]) */
	{ "!U", "]", ASCII },
	/* 0x2156(&@$B!V(B) -> 0x5B([) */
	{ "!V", "[", ASCII },
	/* 0x2157(&@$B!W(B) -> 0x5D(]) */
	{ "!W", "]", ASCII },
	/* 0x2158(&@$B!X(B) -> 0x5B([) */
	{ "!X", "[", ASCII },
	/* 0x2159(&@$B!Y(B) -> 0x5D(]) */
	{ "!Y", "]", ASCII },
	/* 0x215A(&@$B!Z(B) -> 0x5B([) */
	{ "!Z", "[", ASCII },
	/* 0x215B(&@$B![(B) -> 0x5D(]) */
	{ "![", "]", ASCII },
	/* 0x215C(&@$B!\(B) -> 0x2B(+) */
	{ "!\\", "+", ASCII },
	/* 0x215D(&@$B!](B) -> 0x2D(-) */
	{ "!]", "-", ASCII },
	/* 0x215F(&@$B!_(B) -> 0x2A(*) */
	{ "!_", "*", ASCII },
	/* 0x2160(&@$B!`(B) -> 0x2F(/) */
	{ "!`", "/", ASCII },
	/* 0x2161(&@$B!a(B) -> 0x3D(=) */
	{ "!a", "=", ASCII },
	/* 0x2163(&@$B!c(B) -> 0x3C(<) */
	{ "!c", "<", ASCII },
	/* 0x2164(&@$B!d(B) -> 0x3E(>) */
	{ "!d", ">", ASCII },
	/* 0x216C(&@$B!l(B) -> 0x27(') */
	{ "!l", "'", ASCII },
	/* 0x216D(&@$B!m(B) -> 0x22(") */
	{ "!m", "\"", ASCII },
	/* 0x2170(&@$B!p(B) -> 0x24($) */
	{ "!p", "$", ASCII },
	/* 0x2173(&@$B!s(B) -> 0x25(%) */
	{ "!s", "%", ASCII },
	/* 0x2174(&@$B!t(B) -> 0x23(#) */
	{ "!t", "#", ASCII },
	/* 0x2175(&@$B!u(B) -> 0x26(&) */
	{ "!u", "&", ASCII },
	/* 0x2176(&@$B!v(B) -> 0x2A(*) */
	{ "!v", "*", ASCII },
	/* 0x2177(&@$B!w(B) -> 0x40(@) */
	{ "!w", "@", ASCII },
	/* 0x2330(&@$B#0(B) -> 0x30(0) */
	{ "#0", "0", ASCII },
	/* 0x2331(&@$B#1(B) -> 0x31(1) */
	{ "#1", "1", ASCII },
	/* 0x2332(&@$B#2(B) -> 0x32(2) */
	{ "#2", "2", ASCII },
	/* 0x2333(&@$B#3(B) -> 0x33(3) */
	{ "#3", "3", ASCII },
	/* 0x2334(&@$B#4(B) -> 0x34(4) */
	{ "#4", "4", ASCII },
	/* 0x2335(&@$B#5(B) -> 0x35(5) */
	{ "#5", "5", ASCII },
	/* 0x2336(&@$B#6(B) -> 0x36(6) */
	{ "#6", "6", ASCII },
	/* 0x2337(&@$B#7(B) -> 0x37(7) */
	{ "#7", "7", ASCII },
	/* 0x2338(&@$B#8(B) -> 0x38(8) */
	{ "#8", "8", ASCII },
	/* 0x2339(&@$B#9(B) -> 0x39(9) */
	{ "#9", "9", ASCII },
	/* 0x2341(&@$B#A(B) -> 0x41(A) */
	{ "#A", "A", ASCII },
	/* 0x2342(&@$B#B(B) -> 0x42(B) */
	{ "#B", "B", ASCII },
	/* 0x2343(&@$B#C(B) -> 0x43(C) */
	{ "#C", "C", ASCII },
	/* 0x2344(&@$B#D(B) -> 0x44(D) */
	{ "#D", "D", ASCII },
	/* 0x2345(&@$B#E(B) -> 0x45(E) */
	{ "#E", "E", ASCII },
	/* 0x2346(&@$B#F(B) -> 0x46(F) */
	{ "#F", "F", ASCII },
	/* 0x2347(&@$B#G(B) -> 0x47(G) */
	{ "#G", "G", ASCII },
	/* 0x2348(&@$B#H(B) -> 0x48(H) */
	{ "#H", "H", ASCII },
	/* 0x2349(&@$B#I(B) -> 0x49(I) */
	{ "#I", "I", ASCII },
	/* 0x234A(&@$B#J(B) -> 0x4A(J) */
	{ "#J", "J", ASCII },
	/* 0x234B(&@$B#K(B) -> 0x4B(K) */
	{ "#K", "K", ASCII },
	/* 0x234C(&@$B#L(B) -> 0x4C(L) */
	{ "#L", "L", ASCII },
	/* 0x234D(&@$B#M(B) -> 0x4D(M) */
	{ "#M", "M", ASCII },
	/* 0x234E(&@$B#N(B) -> 0x4E(N) */
	{ "#N", "N", ASCII },
	/* 0x234F(&@$B#O(B) -> 0x4F(O) */
	{ "#O", "O", ASCII },
	/* 0x2350(&@$B#P(B) -> 0x50(P) */
	{ "#P", "P", ASCII },
	/* 0x2351(&@$B#Q(B) -> 0x51(Q) */
	{ "#Q", "Q", ASCII },
	/* 0x2352(&@$B#R(B) -> 0x52(R) */
	{ "#R", "R", ASCII },
	/* 0x2353(&@$B#S(B) -> 0x53(S) */
	{ "#S", "S", ASCII },
	/* 0x2354(&@$B#T(B) -> 0x54(T) */
	{ "#T", "T", ASCII },
	/* 0x2355(&@$B#U(B) -> 0x55(U) */
	{ "#U", "U", ASCII },
	/* 0x2356(&@$B#V(B) -> 0x56(V) */
	{ "#V", "V", ASCII },
	/* 0x2357(&@$B#W(B) -> 0x57(W) */
	{ "#W", "W", ASCII },
	/* 0x2358(&@$B#X(B) -> 0x58(X) */
	{ "#X", "X", ASCII },
	/* 0x2359(&@$B#Y(B) -> 0x59(Y) */
	{ "#Y", "Y", ASCII },
	/* 0x235A(&@$B#Z(B) -> 0x5A(Z) */
	{ "#Z", "Z", ASCII },
	/* 0x2361(&@$B#a(B) -> 0x61(a) */
	{ "#a", "a", ASCII },
	/* 0x2362(&@$B#b(B) -> 0x62(b) */
	{ "#b", "b", ASCII },
	/* 0x2363(&@$B#c(B) -> 0x63(c) */
	{ "#c", "c", ASCII },
	/* 0x2364(&@$B#d(B) -> 0x64(d) */
	{ "#d", "d", ASCII },
	/* 0x2365(&@$B#e(B) -> 0x65(e) */
	{ "#e", "e", ASCII },
	/* 0x2366(&@$B#f(B) -> 0x66(f) */
	{ "#f", "f", ASCII },
	/* 0x2367(&@$B#g(B) -> 0x67(g) */
	{ "#g", "g", ASCII },
	/* 0x2368(&@$B#h(B) -> 0x68(h) */
	{ "#h", "h", ASCII },
	/* 0x2369(&@$B#i(B) -> 0x69(i) */
	{ "#i", "i", ASCII },
	/* 0x236A(&@$B#j(B) -> 0x6A(j) */
	{ "#j", "j", ASCII },
	/* 0x236B(&@$B#k(B) -> 0x6B(k) */
	{ "#k", "k", ASCII },
	/* 0x236C(&@$B#l(B) -> 0x6C(l) */
	{ "#l", "l", ASCII },
	/* 0x236D(&@$B#m(B) -> 0x6D(m) */
	{ "#m", "m", ASCII },
	/* 0x236E(&@$B#n(B) -> 0x6E(n) */
	{ "#n", "n", ASCII },
	/* 0x236F(&@$B#o(B) -> 0x6F(o) */
	{ "#o", "o", ASCII },
	/* 0x2370(&@$B#p(B) -> 0x70(p) */
	{ "#p", "p", ASCII },
	/* 0x2371(&@$B#q(B) -> 0x71(q) */
	{ "#q", "q", ASCII },
	/* 0x2372(&@$B#r(B) -> 0x72(r) */
	{ "#r", "r", ASCII },
	/* 0x2373(&@$B#s(B) -> 0x73(s) */
	{ "#s", "s", ASCII },
	/* 0x2374(&@$B#t(B) -> 0x74(t) */
	{ "#t", "t", ASCII },
	/* 0x2375(&@$B#u(B) -> 0x75(u) */
	{ "#u", "u", ASCII },
	/* 0x2376(&@$B#v(B) -> 0x76(v) */
	{ "#v", "v", ASCII },
	/* 0x2377(&@$B#w(B) -> 0x77(w) */
	{ "#w", "w", ASCII },
	/* 0x2378(&@$B#x(B) -> 0x78(x) */
	{ "#x", "x", ASCII },
	/* 0x2379(&@$B#y(B) -> 0x79(y) */
	{ "#y", "y", ASCII },
	/* 0x237a(&@$B#z(B) -> 0x7A(z) */
	{ "#z", "z", ASCII },
	/* 0x2621(&@$B&!(B) -> 0x41(-FA) */
	{ "&!", "A", GREEK },
	/* 0x2622(&@$B&"(B) -> 0x42(-FB) */
	{ "&\"", "B", GREEK },
	/* 0x2623(&@$B&#(B) -> 0x43(-FC) */
	{ "&#", "C", GREEK },
	/* 0x2624(&@$B&$(B) -> 0x44(-FD) */
	{ "&$", "D", GREEK },
	/* 0x2625(&@$B&%(B) -> 0x45(-FE) */
	{ "&%", "E", GREEK },
	/* 0x2626(&@$B&&(B) -> 0x46(-FF) */
	{ "&&", "F", GREEK },
	/* 0x2627(&@$B&'(B) -> 0x47(-FG) */
	{ "&'", "G", GREEK },
	/* 0x2628(&@$B&((B) -> 0x48(-FH) */
	{ "&(", "H", GREEK },
	/* 0x2629(&@$B&)(B) -> 0x49(-FI) */
	{ "&)", "I", GREEK },
	/* 0x262A(&@$B&*(B) -> 0x4A(-FJ) */
	{ "&*", "J", GREEK },
	/* 0x262B(&@$B&+(B) -> 0x4B(-FK) */
	{ "&+", "K", GREEK },
	/* 0x262C(&@$B&,(B) -> 0x4C(-FL) */
	{ "&,", "L", GREEK },
	/* 0x262D(&@$B&-(B) -> 0x4D(-FM) */
	{ "&-", "M", GREEK },
	/* 0x262E(&@$B&.(B) -> 0x4E(-FN) */
	{ "&.", "N", GREEK },
	/* 0x262F(&@$B&/(B) -> 0x4F(-FO) */
	{ "&/", "O", GREEK },
	/* 0x2630(&@$B&0(B) -> 0x50(-FP) */
	{ "&0", "P", GREEK },
	/* 0x2631(&@$B&1(B) -> 0x51(-FQ) */
	{ "&1", "Q", GREEK },
	/* 0x2632(&@$B&2(B) -> 0x53(-FS) */
	{ "&2", "S", GREEK },
	/* 0x2633(&@$B&3(B) -> 0x54(-FT) */
	{ "&3", "T", GREEK },
	/* 0x2634(&@$B&4(B) -> 0x55(-FU) */
	{ "&4", "U", GREEK },
	/* 0x2635(&@$B&5(B) -> 0x56(-FV) */
	{ "&5", "V", GREEK },
	/* 0x2636(&@$B&6(B) -> 0x57(-FW) */
	{ "&6", "W", GREEK },
	/* 0x2637(&@$B&7(B) -> 0x58(-FX) */
	{ "&7", "X", GREEK },
	/* 0x2638(&@$B&8(B) -> 0x59(-FY) */
	{ "&8", "Y", GREEK },
	/* 0x2641(&@$B&A(B) -> 0x61(-Fa) */
	{ "&A", "a", GREEK },
	/* 0x2642(&@$B&B(B) -> 0x62(-Fb) */
	{ "&B", "b", GREEK },
	/* 0x2643(&@$B&C(B) -> 0x63(-Fc) */
	{ "&C", "c", GREEK },
	/* 0x2644(&@$B&D(B) -> 0x64(-Fd) */
	{ "&D", "d", GREEK },
	/* 0x2645(&@$B&E(B) -> 0x65(-Fe) */
	{ "&E", "e", GREEK },
	/* 0x2646(&@$B&F(B) -> 0x66(-Ff) */
	{ "&F", "f", GREEK },
	/* 0x2647(&@$B&G(B) -> 0x67(-Fg) */
	{ "&G", "g", GREEK },
	/* 0x2648(&@$B&H(B) -> 0x68(-Fh) */
	{ "&H", "h", GREEK },
	/* 0x2649(&@$B&I(B) -> 0x69(-Fi) */
	{ "&I", "i", GREEK },
	/* 0x264A(&@$B&J(B) -> 0x6A(-Fj) */
	{ "&J", "j", GREEK },
	/* 0x264B(&@$B&K(B) -> 0x6B(-Fk) */
	{ "&K", "k", GREEK },
	/* 0x264C(&@$B&L(B) -> 0x6C(-Fl) */
	{ "&L", "l", GREEK },
	/* 0x264D(&@$B&M(B) -> 0x6D(-Fm) */
	{ "&M", "m", GREEK },
	/* 0x264E(&@$B&N(B) -> 0x6E(-Fn) */
	{ "&N", "n", GREEK },
	/* 0x264F(&@$B&O(B) -> 0x6F(-Fo) */
	{ "&O", "o", GREEK },
	/* 0x2650(&@$B&P(B) -> 0x70(-Fp) */
	{ "&P", "p", GREEK },
	/* 0x2651(&@$B&Q(B) -> 0x71(-Fq) */
	{ "&Q", "q", GREEK },
	/* 0x2652(&@$B&R(B) -> 0x73(-Fs) */
	{ "&R", "s", GREEK },
	/* 0x2653(&@$B&S(B) -> 0x74(-Ft) */
	{ "&S", "t", GREEK },
	/* 0x2654(&@$B&T(B) -> 0x75(-Fu) */
	{ "&T", "u", GREEK },
	/* 0x2655(&@$B&U(B) -> 0x76(-Fv) */
	{ "&U", "v", GREEK },
	/* 0x2656(&@$B&V(B) -> 0x77(-Fw) */
	{ "&V", "w", GREEK },
	/* 0x2657(&@$B&W(B) -> 0x78(-Fx) */
	{ "&W", "x", GREEK },
	/* 0x2658(&@$B&X(B) -> 0x79(-Fy) */
	{ "&X", "y", GREEK },
	/* 0x2721(&@$B'!(B) -> 0x30(-L0) */
	{ "'!", "0", CYRILLIC },
	/* 0x2722(&@$B'"(B) -> 0x31(-L1) */
	{ "'\"", "1", CYRILLIC },
	/* 0x2723(&@$B'#(B) -> 0x32(-L2) */
	{ "'#", "2", CYRILLIC },
	/* 0x2724(&@$B'$(B) -> 0x33(-L3) */
	{ "'$", "3", CYRILLIC },
	/* 0x2725(&@$B'%(B) -> 0x34(-L4) */
	{ "'%", "4", CYRILLIC },
	/* 0x2726(&@$B'&(B) -> 0x35(-L5) */
	{ "'&", "5", CYRILLIC },
	/* 0x2727(&@$B''(B) -> 0x21(-L!) */
	{ "''", "!", CYRILLIC },
	/* 0x2728(&@$B'((B) -> 0x36(-L6) */
	{ "'(", "6", CYRILLIC },
	/* 0x2729(&@$B')(B) -> 0x37(-L7) */
	{ "')", "7", CYRILLIC },
	/* 0x272A(&@$B'*(B) -> 0x38(-L8) */
	{ "'*", "8", CYRILLIC },
	/* 0x272B(&@$B'+(B) -> 0x39(-L9) */
	{ "'+", "9", CYRILLIC },
	/* 0x272C(&@$B',(B) -> 0x3A(-L:) */
	{ "',", ":", CYRILLIC },
	/* 0x272D(&@$B'-(B) -> 0x3B(-L;) */
	{ "'-", ";", CYRILLIC },
	/* 0x272E(&@$B'.(B) -> 0x3C(-L<) */
	{ "'.", "<", CYRILLIC },
	/* 0x272F(&@$B'/(B) -> 0x3D(-L=) */
	{ "'/", "=", CYRILLIC },
	/* 0x2730(&@$B'0(B) -> 0x3E(-L>) */
	{ "'0", ">", CYRILLIC },
	/* 0x2731(&@$B'1(B) -> 0x3F(-L?) */
	{ "'1", "?", CYRILLIC },
	/* 0x2732(&@$B'2(B) -> 0x40(-L@) */
	{ "'2", "@", CYRILLIC },
	/* 0x2733(&@$B'3(B) -> 0x41(-LA) */
	{ "'3", "A", CYRILLIC },
	/* 0x2734(&@$B'4(B) -> 0x42(-LB) */
	{ "'4", "B", CYRILLIC },
	/* 0x2735(&@$B'5(B) -> 0x43(-LC) */
	{ "'5", "C", CYRILLIC },
	/* 0x2736(&@$B'6(B) -> 0x44(-LD) */
	{ "'6", "D", CYRILLIC },
	/* 0x2737(&@$B'7(B) -> 0x45(-LE) */
	{ "'7", "E", CYRILLIC },
	/* 0x2738(&@$B'8(B) -> 0x46(-LF) */
	{ "'8", "F", CYRILLIC },
	/* 0x2739(&@$B'9(B) -> 0x47(-LG) */
	{ "'9", "G", CYRILLIC },
	/* 0x273A(&@$B':(B) -> 0x48(-LH) */
	{ "':", "H", CYRILLIC },
	/* 0x273B(&@$B';(B) -> 0x49(-LI) */
	{ "';", "I", CYRILLIC },
	/* 0x273C(&@$B'<(B) -> 0x4A(-LJ) */
	{ "'<", "J", CYRILLIC },
	/* 0x273D(&@$B'=(B) -> 0x4B(-LK) */
	{ "'=", "K", CYRILLIC },
	/* 0x273E(&@$B'>(B) -> 0x4C(-LL) */
	{ "'>", "L", CYRILLIC },
	/* 0x273F(&@$B'?(B) -> 0x4D(-LM) */
	{ "'?", "M", CYRILLIC },
	/* 0x2740(&@$B'@(B) -> 0x4E(-LN) */
	{ "'@", "N", CYRILLIC },
	/* 0x2741(&@$B'A(B) -> 0x4F(-LO) */
	{ "'A", "O", CYRILLIC },
	/* 0x2751(&@$B'Q(B) -> 0x50(-LP) */
	{ "'Q", "P", CYRILLIC },
	/* 0x2752(&@$B'R(B) -> 0x51(-LQ) */
	{ "'R", "Q", CYRILLIC },
	/* 0x2753(&@$B'S(B) -> 0x52(-LR) */
	{ "'S", "R", CYRILLIC },
	/* 0x2754(&@$B'T(B) -> 0x53(-LS) */
	{ "'T", "S", CYRILLIC },
	/* 0x2755(&@$B'U(B) -> 0x54(-LT) */
	{ "'U", "T", CYRILLIC },
	/* 0x2756(&@$B'V(B) -> 0x55(-LU) */
	{ "'V", "U", CYRILLIC },
	/* 0x2757(&@$B'W(B) -> 0x71(-Lq) */
	{ "'W", "q", CYRILLIC },
	/* 0x2758(&@$B'X(B) -> 0x56(-LV) */
	{ "'X", "V", CYRILLIC },
	/* 0x2759(&@$B'Y(B) -> 0x57(-LW) */
	{ "'Y", "W", CYRILLIC },
	/* 0x275A(&@$B'Z(B) -> 0x58(-LX) */
	{ "'Z", "X", CYRILLIC },
	/* 0x275B(&@$B'[(B) -> 0x59(-LY) */
	{ "'[", "Y", CYRILLIC },
	/* 0x275C(&@$B'\(B) -> 0x5A(-LZ) */
	{ "'\\", "Z", CYRILLIC },
	/* 0x275D(&@$B'](B) -> 0x5B(-L[) */
	{ "']", "[", CYRILLIC },
	/* 0x275E(&@$B'^(B) -> 0x5C(-L\) */
	{ "'^", "\\", CYRILLIC },
	/* 0x275F(&@$B'_(B) -> 0x5D(-L]) */
	{ "'_", "]", CYRILLIC },
	/* 0x2760(&@$B'`(B) -> 0x5E(-L^) */
	{ "'`", "^", CYRILLIC },
	/* 0x2761(&@$B'a(B) -> 0x5F(-L_) */
	{ "'a", "_", CYRILLIC },
	/* 0x2762(&@$B'b(B) -> 0x60(-L`) */
	{ "'b", "`", CYRILLIC },
	/* 0x2763(&@$B'c(B) -> 0x61(-La) */
	{ "'c", "a", CYRILLIC },
	/* 0x2764(&@$B'd(B) -> 0x62(-Lb) */
	{ "'d", "b", CYRILLIC },
	/* 0x2765(&@$B'e(B) -> 0x63(-Lc) */
	{ "'e", "c", CYRILLIC },
	/* 0x2766(&@$B'f(B) -> 0x64(-Ld) */
	{ "'f", "d", CYRILLIC },
	/* 0x2767(&@$B'g(B) -> 0x65(-Le) */
	{ "'g", "e", CYRILLIC },
	/* 0x2768(&@$B'h(B) -> 0x66(-Lf) */
	{ "'h", "f", CYRILLIC },
	/* 0x2769(&@$B'i(B) -> 0x67(-Lg) */
	{ "'i", "g", CYRILLIC },
	/* 0x276A(&@$B'j(B) -> 0x68(-Lh) */
	{ "'j", "h", CYRILLIC },
	/* 0x276B(&@$B'k(B) -> 0x69(-Li) */
	{ "'k", "i", CYRILLIC },
	/* 0x276C(&@$B'l(B) -> 0x6A(-Lj) */
	{ "'l", "j", CYRILLIC },
	/* 0x276D(&@$B'm(B) -> 0x6B(-Lk) */
	{ "'m", "k", CYRILLIC },
	/* 0x276E(&@$B'n(B) -> 0x6C(-Ll) */
	{ "'n", "l", CYRILLIC },
	/* 0x276F(&@$B'o(B) -> 0x6D(-Lm) */
	{ "'o", "m", CYRILLIC },
	/* 0x2770(&@$B'p(B) -> 0x6E(-Ln) */
	{ "'p", "n", CYRILLIC },
	/* 0x2771(&@$B'q(B) -> 0x6F(-Lo) */
	{ "'q", "o", CYRILLIC },
	/* NULL */
	{ 0, 0, 0 }
};
static convtable utable_jisx0208 = { unify_jisx0208, NULL };

static convtab unify_n_jisx0201roman[] = {
	/* 0x5C((J\(B) -X 0x5C(\) */
	{ "\\", "\\", ASCII },
	/* 0x7E((J~(B) -X 0x7E(~) */
	{ "~", "~", ASCII },
	/* NULL */
	{ 0, 0, 0 }
};
static convtable utable_n_jisx0201roman = { unify_n_jisx0201roman, NULL };

static convtab unify_n_iso646[] = {
	/* 0x23((@#(B) -X 0x23(#) */
	{ "#", "#", ASCII },
	/* 0x24((@$(B) -X 0x24($) */
	{ "$", "$", ASCII },
	/* 0x40((@@(B) -X 0x40(@) */
	{ "@", "@", ASCII },
	/* 0x5B((@[(B) -X 0x5B([) */
	{ "[", "[", ASCII },
	/* 0x5C((@\(B) -X 0x5C(\) */
	{ "\\", "\\", ASCII },
	/* 0x5D((@](B) -X 0x5D(]) */
	{ "]", "]", ASCII },
	/* 0x5E((@^(B) -X 0x5E(^) */
	{ "^", "^", ASCII },
	/* 0x60((@`(B) -X 0x60(`) */
	{ "`", "`", ASCII },
	/* 0x7B((@{(B) -X 0x7B({) */
	{ "{", "{", ASCII },
	/* 0x7C((@|(B) -X 0x7C(|) */
	{ "|", "|", ASCII },
	/* 0x7D((@}(B) -X 0x7D(}) */
	{ "}", "}", ASCII },
	/* 0x7E((@~(B) -X 0x7E(~) */
	{ "~", "~", ASCII },
	/* NULL */
	{ 0, 0, 0 }
};
static convtable utable_n_iso646 = { unify_n_iso646, NULL };

static int iso646p(cs)
CHARSET cs;
{
	if (CS2TYPE(cs) != TYPE_94_CHARSET)
		return 0;
	switch (CS2CHARSET(cs)) {
	case TYPE_94_CHARSET | FT2CS('@'): /* ISO 646 IRV 1983 */
	case TYPE_94_CHARSET | FT2CS('A'): /* BSI 4730 United Kingdom */
	case TYPE_94_CHARSET | FT2CS('C'): /* NATS Standard Swedish/Finish */
	case TYPE_94_CHARSET | FT2CS('G'): /* ISO 646 Swedish */
					   /* (SEN 850200 Ann. B) */
	case TYPE_94_CHARSET | FT2CS('H'): /* ISO 646 Swedish Name */
					   /* (SEN 850200 Ann. C) */
	case JISX0201ROMAN:		   /* JIS X 0201-1976 Roman */
	case TYPE_94_CHARSET | FT2CS('K'): /* ISO 646 German (DIN 66083) */
	case TYPE_94_CHARSET | FT2CS('L'): /* ISO 646 Portuguese (ECMA) */
	case TYPE_94_CHARSET | FT2CS('R'): /* French */
	case TYPE_94_CHARSET | FT2CS('T'): /* China */
	case TYPE_94_CHARSET | FT2CS('Y'): /* Italian */
	case TYPE_94_CHARSET | FT2CS('Z'): /* Spanish */
	case TYPE_94_CHARSET | FT2CS('`'): /* NS 4551 Version 1 */
	case TYPE_94_CHARSET | FT2CS('a'): /* NS 4551 Version 2 */
	case TYPE_94_CHARSET | FT2CS('f'): /* NF Z 62-010-1982 */
	case TYPE_94_CHARSET | FT2CS('g'): /* IBM Portuguese */
	case TYPE_94_CHARSET | FT2CS('h'): /* IBM Spanish */
	case TYPE_94_CHARSET | FT2CS('i'): /* MS Z 7795/3 [Hungary] */
	case TYPE_94_CHARSET | FT2CS('n'): /* JIS C 6229-1984 OCR-B [Japan] */
	case TYPE_94_CHARSET | FT2CS('u'): /* CCITT Recommendation T.61 */
					   /* Teletex Primary Set */
	case TYPE_94_CHARSET | FT2CS('w'): /* CSA Z 243.4-1985 Alternate */
					   /* Primary Set No.1 [Canada] */
	case TYPE_94_CHARSET | FT2CS('x'): /* CSA Z 243.4-1985 Alternate */
					   /* Primary Set No.2 [Canada] */
	case TYPE_94_CHARSET | FT2CS('z'): /* JUS I.B1.002 [Yugoslavia] */
		return 1;
	default:
		return 0;
	}
}

void chconvert_cs(istr, ics, ostr, ocs)
char* istr;
CHARSET* ics;
char* ostr;
CHARSET* ocs;
{
	int i;
	convtab* ptab;

	if (istr[0] == NULCH && CSISNULLCS(ics[0])) {
		ostr[0] = NULCH;
		ocs[0] = NULLCS;
		return;
	}
	/* initialize output buffer */
	i = 0;
	do {
#if MSB_ENABLE
		if (CSISASCII(ics[i]) || CSISWRONG(ics[i]))
			ostr[i] = istr[i];
		else
			ostr[i] = istr[i] & ~0x80;
#else
		ostr[i] = istr[i];
#endif
		ocs[i] = ics[i];
		i++;
	} while (CSISREST(ics[i]));
	ostr[i] = NULCH;
	ocs[i] = NULLCS;
	/* convert codes into some traditional codes */
	if (CS2CHARSET(*ocs) == JISX0208_78KANJI) {
		/* convert JIS C 6226-1978 into JIS X 0208-1990 */
		ptab = find_convtab(&ctable_jisx0208_78_90, ostr);
		if (ptab) {
			ostr[0] = ptab->output[0];
			ostr[1] = ptab->output[1];
			ocs[0] = ptab->charset;
			ocs[1] = ptab->charset | REST_MASK;
		} else {
			ocs[0] = JISX0208_90KANJI;
			ocs[1] = JISX0208_90KANJI | REST_MASK;
		}
	} else if (CS2CHARSET(*ocs) == JISX0208KANJI) {
		/* convert JIS X 0208-1983 into JIS X 0208-1990 */
		ocs[0] = JISX0208_90KANJI;
		ocs[1] = JISX0208_90KANJI | REST_MASK;

		/*
		 * Both 0x7425 and 0x7426 is add.
		 * So here is nothing to do.
		 */
	} else if (CS2CHARSET(*ocs) == JISX0201ROMAN) {
		/* convert JIS X 0201-1976 into ASCII */
		ptab = find_convtab(&utable_n_jisx0201roman, ostr);
		if (!ptab) {
			ocs[0] = ASCII;
		}
	} else if (iso646p(*ocs)) {
		/* convert domestic ISO 646 into ASCII */
		ptab = find_convtab(&utable_n_iso646, ostr);
		if (!ptab) {
			ocs[0] = ASCII;
		}
	}
}

void chunify_cs(istr, ics, ostr, ocs)
char* istr;
CHARSET* ics;
char* ostr;
CHARSET* ocs;
{
	int i;
	convtab* ptab;

	chconvert_cs(istr, ics, ostr, ocs);
	/* unify codes */
	if (CS2CHARSET(*ocs) == JISX0208_90KANJI) {
		/*
		 * convert ASCII, GREEK and CYRILLIC character in
		 * JIS X 0208-1990 into ASCII, ISO 8859-7 and ISO 8859-5
		 * respectively.
		 */
		ptab = find_convtab(&utable_jisx0208, ostr);
		if (ptab) {
			int len = strlen(ptab->output);
			assert(len <= strlen(ostr));
			ostr[0] = ptab->output[0];
			ocs[0] = ptab->charset;
			for (i = 1; i < len; i++) {
				ostr[i] = ptab->output[i];
				ocs[i] = ptab->charset | REST_MASK;
			}
			ostr[i] = NULCH;
			ocs[i] = NULLCS;
		}
	}
}

int chcmp_cs(str1, cs1, str2, cs2)
char* str1;
CHARSET* cs1;
char* str2;
CHARSET* cs2;
{
	char buf1[32];
	CHARSET bcs1[32];
	char buf2[32];
	CHARSET bcs2[32];

	/* if there is no character set, compare them as ASCII */
	if (cs1 == NULL && cs2 == NULL)
		return *str1 - *str2;
	if (cs1 == NULL)
		return chcmp_cs(str2, cs2, str1, cs1);
	if (cs2 == NULL)
		return MAKECV(*str1, *cs1) - MAKECV(*str2, ASCII);

	/* unify both of inputs */
	chunify_cs(str1, cs1, buf1, bcs1);
	str1 = buf1;
	cs1 = bcs1;
	chunify_cs(str2, cs2, buf2, bcs2);
	str2 = buf2;
	cs2 = bcs2;
	/* compare them */
	if ((*str1 == NULCH && CSISNULLCS(*cs1)) ||
	    (*str2 == NULCH && CSISNULLCS(*cs2)))
		return MAKECV(*str1, *cs1) - MAKECV(*str2, *cs2);
	do {
		if (*str1 != *str2 || *cs1 != *cs2) {
			return MAKECV(*str1, *cs1) - MAKECV(*str2, *cs2);
		}
		str1++;
		cs1++;
		str2++;
		cs2++;
	} while (CSISREST(*cs1));
	return 0;
}

#endif
