/*
 * Module almost for main() function and some initializations.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <gnome.h>
#include "gdiff.h"
#include "misc.h"
#include "gui.h"
#include "guiwin.h"
#include "dirview.h"
#include "properties.h"
#include "style.h"


/* Private function declarations */
static void gtkdiff_init(void);
static void argument_init(GDiffWindow *gdwin, char **arg_files);
#if !defined(DEBUG) || defined(MSG_DIALOG)
static void msg_handler(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data);
#endif


int
main(int argc, char *argv[])
{
	GDiffWindow *gdiffwin;
	char **arg_files;
	static char *geometry_string = NULL;
	poptContext ctx;
	static const struct poptOption options[] = {
		{"geometry", 'g', POPT_ARG_STRING, &geometry_string, 0,
		 N_("Specify geometry"), N_("GEOMETRY")},
		{NULL, '\0', 0, NULL, 0, NULL, NULL}
	};
	
	/* Initialize the i18n stuff */
	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	textdomain(PACKAGE);

	/*
	 * gnome_init() is always called at the beginning of a program.  it
	 * takes care of initializing both Gtk and GNOME.  It also parses
	 * the command-line arguments.
	 */
	gnome_init_with_popt_table(APPNAME, VERSION, argc, argv,
							   options, 0, &ctx);

#if !defined(DEBUG) || defined(MSG_DIALOG)
	g_log_set_handler(G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE, msg_handler, NULL);
#endif
	style_init();
	gtkdiff_init();
	properties_init();

	gdiffwin = gdiff_window_new(geometry_string);

	arg_files = poptGetArgs(ctx);
	argument_init(gdiffwin, arg_files);
	poptFreeContext(ctx);
	
	gtk_main();

	return 0;
}


/**
 * gtkdiff_exit:
 * Whenever gtkdiff exits, this is called. (GUI level)
 * The arguments are written to be called from signal callback, so they don't matter.
 **/
void
gtkdiff_exit(GtkObject *obj, gpointer data)
{
	properties_exit();

	/* really exit */
	gtk_exit(0);
}


/* ---The followings are private functions--- */
/**
 * gtkdiff_init:
 * Later, it will do more.
 **/
static void
gtkdiff_init(void)
{
	signal(SIGCHLD, def_signal_handler);
}


/**
 * argument_init:
 * Just take care of the specified two files by command option.
 **/
static void
argument_init(GDiffWindow *gdwin, char **arg_files)
{
	GDiffDirViews *gdirviews;
	gboolean is_dir;
	const char *file1;
	const char *file2;
	FileType ftype1, ftype2;

	if (arg_files && arg_files[0] && arg_files[1]) {
		file1 = arg_files[0];
		file2 = arg_files[1];
	} else
		return;

	ftype1 = check_filetype(file1);
	ftype2 = check_filetype(file2);
	if (ftype1 != ftype2) {
		g_message("you can't compare file and directory.\n");
		return;
	}
	if (ftype1 == REGULARFILE)
		is_dir = FALSE;
	else if (ftype1 == DIRECTORY)
		is_dir = TRUE;
	else
		return;

	gdirviews = gdiff_dirviews_new(gdwin, file1, file2, is_dir);
	dirview_display(gdirviews);
}


#if !defined(DEBUG) || defined(MSG_DIALOG)
/**
 * msg_handler:
 * Handler for g_message().
 **/
static void
msg_handler(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data)
{
	GtkWidget *dialog;

	dialog = gnome_message_box_new(message, GNOME_MESSAGE_BOX_WARNING, GNOME_STOCK_BUTTON_OK, NULL);
	/* I can't use this, because it internally calls gtk_widget_show(), so set_position doesn't work.
	  dialog = gnome_warning_dialog(message); */
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
	/*
	gtk_signal_connect(GTK_OBJECT(dialog), "destroy", 
					   GTK_SIGNAL_FUNC(gtk_widget_destroyed),
					   &dialog);
	*/
	gtk_widget_show(dialog);
}
#endif
