# Perl module TeX::Kpsewhich
# $Id: Kpsewhich.pm,v 1.7 1999/07/20 22:52:12 jdg Exp $
# CVS version tag name: $Name: debian_version_3_3_1_4 $
#
# Copyright 1999, Julian Gilbey <jdg@debian.org>
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file defines the TeX::Kpsewhich module, which reimplements
# the functionality of the kpsewhich program in Perl using the
# TeX::Kpathsea module.  It does not implement the -help or -interactive
# features of kpsewhich, as these are not relevent in a scripting context.

package TeX::Kpsewhich;

# In case use strict is being used.
use vars qw(@EXPORT @EXPORT_OK $VERSION @ISA $program_name $debug);
use vars qw(@all_formats @format_info);

# If we have a CVS version name, prefer that as it's more likely to be
# up-to-date.  There's a fallback default.  Use this ghastly mess so
# that MakeMaker can extract a version number automatically.

$VERSION = do { my @r=(q$Name: debian_version_3_3_1_4 $=~/\d+/g); @r ? sprintf "%d."."%02d"x$#r,@r : 3.030101 };

require Carp;
import Carp ':DEFAULT';

require Exporter;
@ISA = qw(Exporter);
@EXPORT = ();
@EXPORT_OK = qw(kpsewhich);

require TeX::Kpathsea;
import TeX::Kpathsea qw(:short_formats :short_functions :short_misc);

sub version {
	return <<EOF;
TeX::Kpsewhich module, version $VERSION
Copyright (C) 1999 Julian Gilbey
TeX::Kpathsea module, version $TeX::Kpathsea::VERSION
Copyright (C) 1999 Julian Gilbey
Kpathsea[rch] library: $kpse_version
Copyright (C) 1997 K. Berry
There is NO warranty.  You may redistribute this software
under the terms of the GNU General Public License, version 2 or later.
For more information about these matters, see the file named COPYING.
EOF
}

#
# ------------------ CONSTRUCTOR STUFF--------------------
#

# Default options
my %default_options;
%default_options = (
	'progname' => $program_name,
	'format' => '',
	'path' => '',
	'must-exist' => 0,
	'find-all' => 0,
	'mktex' => [],
	'no-mktex' => [],
	'mode' => 0,
	'dpi' => 600,
	'debug' => 0,
);

# Syntax for constructor:
# $kpse1 = TeX::Kpsewhich(opt1 => var1, opt2 => var2, ...)
sub new {
	my ($pkg, %opt_hash, $opt, $optname, $optval);
	$pkg = shift;
	%opt_hash = %default_options;
	while (@_) {
		$opt = shift;
		($optname = $opt) =~ s/^-+//;
		$optname="dpi" if $optname eq "D";
		if (!defined($optval = shift)) {
			carp("Warning: TeX::Kpsewhich::new:  Option '$opt' has no value after it; ignoring option");
		}
		elsif ($optname =~ /^(no-)?mktex$/ and ref($optval) ne 'ARRAY') {
			carp("Warning: TeX::Kpsewhich::new:  Option '$opt' expects an array ref as value; ignoring option");
		}
		elsif (exists $opt_hash{$optname}) {
			$opt_hash{$optname} = $optval;
		}
		else {
			carp("Warning: TeX::Kpsewhich::new:  Unrecognised option '$opt' will be ignored");
		}
	}
	croak("TeX::Kpsewhich::new:  Can't supply both -path and -format options!")
		if $opt_hash{'path'} && $opt_hash{'format'};

	if ($opt_hash{'format'}) {
		$opt_hash{'format-type'} = find_cmd_format($opt_hash{'format'});
		croak("TeX::Kpsewhich::new: Unrecognised format type '$opt_hash{'format'}'")
			if (!defined $opt_hash{'format-type'});
	}

	bless \%opt_hash, $pkg;
}

# Here we are given a command-line-type option by the user such as
# '.tex' or 'mfpool' denoting the format type, which we must deduce.
# This function will only be called once per constructor.

sub find_cmd_format {
	my $fmt_name = shift;

	foreach my $fmt (@all_formats) {
		init_format($fmt);
		return $fmt if $fmt_name eq $format_info[$fmt]{'type'};
		foreach (@{$format_info[$fmt]{'suffix'}}) {
			return $fmt if $fmt_name eq $_;
		}
		foreach (@{$format_info[$fmt]{'alt_suffix'}}) {
			return $fmt if $fmt_name eq $_;
		}
	}
	return undef;
}

#
# -------------------- OBJECT METHODS ----------------------------
#

sub expand_var {
	if (@_ != 2) {
		croak(<<'EOF');
Bad syntax, use:
  $obj->expand_var($string)
or
  kpsewhich(opt1 => val1, ..., expand_var => $string)
EOF
	}
	my ($obj,$var) = @_;
	if (!ref($obj)) {
		croak("TeX::Kpsewhich::expand_var:  My first parameter is not a reference -- why not?");
	}
	if (ref($obj) eq 'HASH') {
		croak("TeX::Kpsewhich::expand_var:  My first parameter is not blessed -- why not?");
	}
	
	return unless defined wantarray();
	$obj->do_opts();
	return var_expand($var);
}

sub expand_braces {
	if (@_ != 2) {
		croak(<<'EOF');
Bad syntax, use:
  $obj->expand_braces($string)
or
  kpsewhich(opt1 => val1, ..., expand_braces => $string)
EOF
	}
	my ($obj,$var) = @_;
	if (!ref($obj)) {
		croak("TeX::Kpsewhich::expand_braces:  My first parameter is not a reference -- why not?");
	}
	if (ref($obj) eq 'HASH') {
		croak("TeX::Kpsewhich::expand_braces:  My first parameter is not blessed -- why not?");
	}
	
	return unless defined wantarray();
	$obj->do_opts();
	return brace_expand($var);
}

sub expand_path {
	if (@_ != 2) {
		croak(<<'EOF');
Bad syntax, use:
  $obj->expand_path($string)
or
  kpsewhich(opt1 => val1, ..., expand_path => $string)
EOF
	}
	my ($obj,$var) = @_;
	if (!ref($obj)) {
		croak("TeX::Kpsewhich::expand_path:  My first parameter is not a reference -- why not?");
	}
	if (ref($obj) eq 'HASH') {
		croak("TeX::Kpsewhich::expand_path:  My first parameter is not blessed -- why not?");
	}
	
	return unless defined wantarray();
	$obj->do_opts();
	return path_expand($var);
}

sub show_path {
	if (@_ != 2) {
		croak(<<'EOF');
Bad syntax, use:
  $obj->show_path($string)
or
  kpsewhich(opt1 => val1, ..., show_path => $string)
EOF
	}
	my ($obj,$fmt_name) = @_;
	if (!ref($obj)) {
		croak("TeX::Kpsewhich::show_path:  My first parameter is not a reference -- why not?");
	}
	if (ref($obj) eq 'HASH') {
		croak("TeX::Kpsewhich::show_path:  My first parameter is not blessed -- why not?");
	}
	carp("Warning: TeX::Kpsewhich::show_path:  Have already specified a format; will ignore for showing format path")
		if $obj->{'format'};
	carp("Warning: TeX::Kpsewhich::show_path:  Have already specified a search path; will ignore for showing format path")
		if $obj->{'path'};

	return unless defined wantarray();
	$obj->do_opts();
	my $fmt = find_cmd_format($fmt_name);
	if (defined $fmt) {
		return $format_info[$fmt]->{'path'};
	}
	else { return ''; }
}

sub find {
	my ($obj,$name,@ret);
	$obj = shift;
	if (!ref($obj)) {
		croak("TeX::Kpsewhich::find:  My first parameter is not a reference -- why not?");
	}
	if (ref($obj) eq 'HASH') {
		croak("TeX::Kpsewhich::find:  My first parameter is not blessed -- why not?");
	}
	
	return unless defined wantarray();
	return unless @_;
	$obj->do_opts();
	if (!wantarray && ( @_>1 || $obj->{'find-all'})) {
		carp("Warning: TeX::Kpsewhich::find:  asking for multiple finds in scalar context; will only return first result");
		$#_=0;
		$obj->{'must-exist'}=1, $obj->{'find-all'}=0 if $obj->{'find-all'};
	}

	# Have we been given a specific search path?
	if ($obj->{'path'}) {
		if ($obj->{'find-all'}) {
			foreach $name (@_) {
				push @ret, @{all_path_search($obj->{'path'},$name)};
			}
		} else {
			foreach $name (@_) {
				my $ret = path_search($obj->{'path'},$name,$obj->{'must-exist'});
				length $ret and push @ret, $ret;
			}
		}
		return wantarray() ? @ret : $ret[0];
	}
			
	# No specific search path, so we'll have to figure out the format
	# type for each file.  We will use the same heuristics as in kpsewhich.c
	# We also create a glyph_file_type struct in case we need it.
	my ($fmt,$dpi,$glyph_struct);
	$glyph_struct=kpse_glyph_file_type->new();

	foreach $name (@_) {
		$fmt = $obj->{'format-type'} || find_filename_format($name);
		init_format($fmt);  # No harm in doing this

		if ($fmt == pk_format() || $fmt == gf_format() || 
				$fmt == any_glyph_format()) {
			$name =~ m/(.*)\.(\d*)\w+$/;
			$dpi = $2 || $obj->{'dpi'};
			my $ret = find_glyph($1, $dpi, $fmt, $glyph_struct);
			length $ret and push @ret, $ret;
		}
		else {
			if ($obj->{'find-all'}) {
				push @ret, @{all_path_search($format_info[$fmt]{'path'},$name)};
			} else {
				my $ret = find_file($name, $fmt, $obj->{'must-exist'});
				length $ret and push @ret, $ret;
			}
		}
	}
	return wantarray() ? @ret : $ret[0];
}

sub find_filename_format {
	my $name = shift;
	return dvips_config_format() if ($name eq 'psfonts.map');

	my $suffix;
	foreach my $fmt (@all_formats) {
		init_format($fmt);
		foreach $suffix (@{$format_info[$fmt]{'suffix'}}) {
			return $fmt if $name =~ m/$suffix$/;
		}
		foreach $suffix (@{$format_info[$fmt]{'alt_suffix'}}) {
			return $fmt if $name =~ m/$suffix$/;
		}
	}
	# Default: assume tex format
	return tex_format();
}

sub do_opts {
	my ($ropts,$fmt);
	$ropts = shift;

	$debug=$ropts->{'debug'};
	reset_program_name($ropts->{'progname'});

	init_prog(uc($program_name), $ropts->{'dpi'}, $ropts->{'mode'}, 0);
	init_format($ropts->{'format-type'}) if exists $ropts->{'format-type'};
	for $fmt (@{$ropts->{'mktex'}}) {
		maketex_option($fmt,1);
	}
	for $fmt (@{$ropts->{'no-mktex'}}) {
		maketex_option($fmt,0);
	}
	# We don't deal with must-exist or find-all here, but rather
   # in the relevant functions above
}

#
# -------------- NON-OBJECT-ORIENTED FUNCTION ----------------
#

# Usage: kpsewhich({ opt1 => val1, ...}, name, ...)
# to search for name, ... using opts as given.  Only of of the
# options expand_* or show_path may be given, and in such a case,
# no names may be given.  If there are multiple names given, we will
# either search for each one and return an array
sub kpsewhich {
	my (@optarray, $ropts, $opt, $opt_name, $val, $kpse_obj, $method, $methopt);
	$method=$methopt='';
	if (ref($_[0]) eq 'HASH') {
		$ropts = shift;
		while (($opt, $val) = each %$ropts) {
			($opt_name = $opt) =~ s/^-+//;
			if ($opt_name =~ /^(expand-(var|braces|path)|show-path)$/) {
				croak("TeX::Kpsewhich::kpsewhich:  may only supply one of -expand-* or -show-path options")
					if ($method);
				($method = $opt_name) =~ tr/-/_/;
				$methopt = $val;
			}
			else {
				push @optarray, $opt_name, $val;
			}
		}
		croak("TeX::Kpsewhich::kpsewhich:  Cannot provide filenames in addition to -expand-* or -show-path options")
			if ($method and @_ > 0);
	}
	else { @optarray=(); }
	$kpse_obj = new TeX::Kpsewhich(@optarray);
	if ($method) { return $kpse_obj->$method($methopt); }
	else  { return $kpse_obj->find(@_); }
}

1;
