
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "mpg123.h"

static GtkWidget *window = NULL, *hbox, *vbox, *id3_frame, *table, *mpeg_frame, *mpeg_box;
static GtkWidget *title_label, *title_entry;
static GtkWidget *artist_label, *artist_entry;
static GtkWidget *album_label, *album_entry;
static GtkWidget *year_label, *year_entry;
static GtkWidget *comment_label, *comment_entry;
static GtkWidget *genre_label, *genre_combo;
static GtkWidget *bbox, *save, *remove_id3, *cancel;
static GtkWidget *mpeg_level, *mpeg_bitrate, *mpeg_samplerate, *mpeg_flags;
static GtkWidget *mpeg_fileinfo;

static GList *genre_list = NULL;
static gchar *current_filename = NULL;

#define MAX_STR_LEN 100

static void show_dialog(gchar * title, gchar * text)
{
	GtkWidget *dialog, *label, *button;

	dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog), title);
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
	gtk_container_border_width(GTK_CONTAINER(dialog), 5);
	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	button = gtk_button_new_with_label("Close");
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(button, 75, -1);
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, FALSE, FALSE, 0);
	gtk_widget_show(button);
	gtk_widget_grab_default(button);

	gtk_widget_show(dialog);
}

static void set_entry_tag(GtkEntry * entry, gchar * tag, gint length)
{
	gint stripped_len;
	gchar *text;

	stripped_len = mpg123_strip_spaces(tag, length);
	text = g_strdup_printf("%-*.*s", stripped_len, stripped_len, tag);
	gtk_entry_set_text(entry, text);
	g_free(text);
}

static void get_entry_tag(GtkEntry * entry, gchar * tag, gint length)
{
	gchar *text;

	text = gtk_entry_get_text(entry);
	memset(tag, ' ', length);
	memcpy(tag, text, strlen(text) > length ? length : strlen(text));
}

static gint find_genre_id(gchar * text)
{
	gint i;

	for (i = 0; i < GENRE_MAX; i++)
	{
		if (!strcmp(mpg123_id3_genres[i], text))
			return i;
	}
	if (text[0] == '\0')
		return 0xff;
	return 0;
}

static gint genre_comp_func(gconstpointer a, gconstpointer b)
{
	return strcasecmp(a, b);
}

static void save_cb(GtkWidget * w, gpointer data)
{
	gint fd;
	struct id3v1tag_t tag;

	if ((fd = open(current_filename, O_RDWR)) != -1)
	{
		lseek(fd, -128, SEEK_END);
		read(fd, &tag, sizeof (struct id3v1tag_t));

		if (!strncmp(tag.tag, "TAG", 3))
			lseek(fd, -128, SEEK_END);
		else
			lseek(fd, 0, SEEK_END);
		tag.tag[0] = 'T';
		tag.tag[1] = 'A';
		tag.tag[2] = 'G';
		get_entry_tag(GTK_ENTRY(title_entry), tag.title, 30);
		get_entry_tag(GTK_ENTRY(artist_entry), tag.artist, 30);
		get_entry_tag(GTK_ENTRY(album_entry), tag.album, 30);
		get_entry_tag(GTK_ENTRY(year_entry), tag.year, 4);
		get_entry_tag(GTK_ENTRY(comment_entry), tag.comment, 30);
		tag.genre = find_genre_id(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(genre_combo)->entry)));
		if (write(fd, &tag, sizeof (struct id3v1tag_t)) != sizeof (struct id3v1tag_t))
			       show_dialog("File Info", "\n    Couldn't write tag!    \n");

		close(fd);
	}
	else
		show_dialog("File Info", "\n    Couldn't write tag!    \n");
	gtk_widget_destroy(window);
}

static void remove_id3_cb(GtkWidget * w, gpointer data)
{
	gint fd, len;
	struct id3v1tag_t tag;

	if ((fd = open(current_filename, O_RDWR)) != -1)
	{
		len = lseek(fd, -128, SEEK_END);
		read(fd, &tag, sizeof (struct id3v1tag_t));

		if (!strncmp(tag.tag, "TAG", 3))
		{
			if (ftruncate(fd, len))
				show_dialog("File Info", "\n    Couldn't remove tag!    \n");
		}
		else
			show_dialog("File Info", "\n    No tag to remove!    \n");
		close(fd);
	}
	else
		show_dialog("File Info", "\n    Couldn't remove tag!    \n");
	gtk_widget_destroy(window);
}

void mpg123_file_info_box(char *filename)
{
	gint i;
	struct id3v1tag_t tag;
	FILE *fh;
	gchar *tmp, *name, *title;
	const gchar *emphasis[] = {"None", "50/15 ms", "", "CCIT J.17"};
	const gchar *channel_mode[] = {"Stereo", "Joint stereo", "Dual channel", "Single channel"};
	const gchar *bool_label[] = {"No", "Yes"};

	if (!strncasecmp(filename, "http://", 7))
		return;

	if (!window)
	{
		window = gtk_window_new(GTK_WINDOW_DIALOG);
		gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
		gtk_signal_connect(GTK_OBJECT(window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);
		gtk_container_set_border_width(GTK_CONTAINER(window), 10);

		hbox = gtk_hbox_new(FALSE, 10);
		gtk_container_add(GTK_CONTAINER(window), hbox);

		vbox = gtk_vbox_new(FALSE, 10);
		gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

		id3_frame = gtk_frame_new("ID3 Tag:");
		gtk_box_pack_start(GTK_BOX(vbox), id3_frame, FALSE, FALSE, 0);

		table = gtk_table_new(4, 5, FALSE);
		gtk_container_set_border_width(GTK_CONTAINER(table), 5);
		gtk_container_add(GTK_CONTAINER(id3_frame), table);

		title_label = gtk_label_new("Title:");
		gtk_misc_set_alignment(GTK_MISC(title_label), 1, 0.5);
		gtk_table_attach(GTK_TABLE(table), title_label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 5);
		gtk_widget_show(title_label);

		title_entry = gtk_entry_new_with_max_length(30);
		gtk_table_attach(GTK_TABLE(table), title_entry, 1, 4, 0, 1, GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);
		gtk_widget_show(title_entry);

		artist_label = gtk_label_new("Artist:");
		gtk_misc_set_alignment(GTK_MISC(artist_label), 1, 0.5);
		gtk_table_attach(GTK_TABLE(table), artist_label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 5, 5);
		gtk_widget_show(artist_label);

		artist_entry = gtk_entry_new_with_max_length(30);
		gtk_table_attach(GTK_TABLE(table), artist_entry, 1, 4, 1, 2, GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);
		gtk_widget_show(artist_entry);

		album_label = gtk_label_new("Album:");
		gtk_misc_set_alignment(GTK_MISC(album_label), 1, 0.5);
		gtk_table_attach(GTK_TABLE(table), album_label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 5, 5);
		gtk_widget_show(album_label);

		album_entry = gtk_entry_new_with_max_length(30);
		gtk_table_attach(GTK_TABLE(table), album_entry, 1, 4, 2, 3, GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);
		gtk_widget_show(album_entry);

		comment_label = gtk_label_new("Comment:");
		gtk_misc_set_alignment(GTK_MISC(comment_label), 1, 0.5);
		gtk_table_attach(GTK_TABLE(table), comment_label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 5, 5);
		gtk_widget_show(comment_label);

		comment_entry = gtk_entry_new_with_max_length(30);
		gtk_table_attach(GTK_TABLE(table), comment_entry, 1, 4, 3, 4, GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);
		gtk_widget_show(comment_entry);

		year_label = gtk_label_new("Year:");
		gtk_misc_set_alignment(GTK_MISC(year_label), 1, 0.5);
		gtk_table_attach(GTK_TABLE(table), year_label, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 5, 5);
		gtk_widget_show(year_label);

		year_entry = gtk_entry_new_with_max_length(4);
		gtk_widget_set_usize(year_entry, 40, -1);
		gtk_table_attach(GTK_TABLE(table), year_entry, 1, 2, 4, 5, GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);
		gtk_widget_show(year_entry);

		genre_label = gtk_label_new("Genre:");
		gtk_misc_set_alignment(GTK_MISC(genre_label), 1, 0.5);
		gtk_table_attach(GTK_TABLE(table), genre_label, 2, 3, 4, 5, GTK_FILL, GTK_FILL, 5, 5);
		gtk_widget_show(genre_label);

		genre_combo = gtk_combo_new();
		gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(genre_combo)->entry), FALSE);
		if (!genre_list)
		{
			for (i = 0; i < GENRE_MAX; i++)
				genre_list = g_list_append(genre_list, (gchar *) mpg123_id3_genres[i]);
			genre_list = g_list_append(genre_list, "");
			genre_list = g_list_sort(genre_list, genre_comp_func);
		}
		gtk_combo_set_popdown_strings(GTK_COMBO(genre_combo), genre_list);

		gtk_table_attach(GTK_TABLE(table), genre_combo, 3, 4, 4, 5, GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 5);
		gtk_widget_show(genre_combo);

		gtk_widget_show(id3_frame);
		gtk_widget_show(table);

		bbox = gtk_hbutton_box_new();
		gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
		gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
		gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);

		save = gtk_button_new_with_label("Save");
		gtk_signal_connect(GTK_OBJECT(save), "clicked", GTK_SIGNAL_FUNC(save_cb), NULL);
		GTK_WIDGET_SET_FLAGS(save, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(bbox), save, TRUE, TRUE, 0);
		gtk_widget_show(save);
		gtk_widget_grab_default(save);

		remove_id3 = gtk_button_new_with_label("Remove ID3");
		gtk_signal_connect(GTK_OBJECT(remove_id3), "clicked", GTK_SIGNAL_FUNC(remove_id3_cb), NULL);
		GTK_WIDGET_SET_FLAGS(remove_id3, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(bbox), remove_id3, TRUE, TRUE, 0);
		gtk_widget_show(remove_id3);

		cancel = gtk_button_new_with_label("Cancel");
		gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(window));
		GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
		gtk_box_pack_start(GTK_BOX(bbox), cancel, TRUE, TRUE, 0);
		gtk_widget_show(cancel);

		mpeg_frame = gtk_frame_new("MPEG Info:");
		gtk_box_pack_start(GTK_BOX(hbox), mpeg_frame, FALSE, FALSE, 0);

		mpeg_box = gtk_vbox_new(FALSE, 5);
		gtk_container_add(GTK_CONTAINER(mpeg_frame), mpeg_box);
		gtk_container_set_border_width(GTK_CONTAINER(mpeg_box), 10);
		gtk_box_set_spacing(GTK_BOX(mpeg_box), 0);

		mpeg_level = gtk_label_new("");
		gtk_widget_set_usize(mpeg_level, 120, -2);
		gtk_misc_set_alignment(GTK_MISC(mpeg_level), 0, 0);
		gtk_box_pack_start(GTK_BOX(mpeg_box), mpeg_level, FALSE, FALSE, 0);
		gtk_widget_show(mpeg_level);

		mpeg_bitrate = gtk_label_new("");
		gtk_misc_set_alignment(GTK_MISC(mpeg_bitrate), 0, 0);
		gtk_label_set_justify(GTK_LABEL(mpeg_bitrate), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(mpeg_box), mpeg_bitrate, FALSE, FALSE, 0);
		gtk_widget_show(mpeg_bitrate);

		mpeg_samplerate = gtk_label_new("");
		gtk_misc_set_alignment(GTK_MISC(mpeg_samplerate), 0, 0);
		gtk_box_pack_start(GTK_BOX(mpeg_box), mpeg_samplerate, FALSE, FALSE, 0);
		gtk_widget_show(mpeg_samplerate);

		mpeg_flags = gtk_label_new("");
		gtk_misc_set_alignment(GTK_MISC(mpeg_flags), 0, 0);
		gtk_label_set_justify(GTK_LABEL(mpeg_flags), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(mpeg_box), mpeg_flags, FALSE, FALSE, 0);
		gtk_widget_show(mpeg_flags);

		mpeg_fileinfo = gtk_label_new("");
		gtk_misc_set_alignment(GTK_MISC(mpeg_fileinfo), 0, 0);
		gtk_label_set_justify(GTK_LABEL(mpeg_fileinfo), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(mpeg_box), mpeg_fileinfo, FALSE, FALSE, 0);
		gtk_widget_show(mpeg_fileinfo);

		gtk_widget_show(mpeg_box);
		gtk_widget_show(mpeg_frame);


		gtk_widget_show(bbox);
		gtk_widget_show(vbox);
		gtk_widget_show(hbox);
		gtk_widget_show(window);
	}

	if (current_filename)
		g_free(current_filename);
	current_filename = g_strdup(filename);

	name = strrchr(filename, '/');
	if (name)
		name++;
	else
		name = filename;

	title = g_strdup_printf("File Info - %s", name);
	gtk_window_set_title(GTK_WINDOW(window), title);
	g_free(title);

	title = g_strdup(name);
	if ((tmp = strrchr(title, '.')) != NULL)
		*tmp = '\0';
	gtk_entry_set_text(GTK_ENTRY(title_entry), title);
	g_free(title);

	gtk_entry_set_text(GTK_ENTRY(artist_entry), "");
	gtk_entry_set_text(GTK_ENTRY(album_entry), "");
	gtk_entry_set_text(GTK_ENTRY(year_entry), "");
	gtk_entry_set_text(GTK_ENTRY(comment_entry), "");
	gtk_list_select_item(GTK_LIST(GTK_COMBO(genre_combo)->list), g_list_index(genre_list, ""));
	gtk_label_set_text(GTK_LABEL(mpeg_level), "MPEG ?, layer ?");
	gtk_label_set_text(GTK_LABEL(mpeg_bitrate), "");
	gtk_label_set_text(GTK_LABEL(mpeg_samplerate), "");
	gtk_label_set_text(GTK_LABEL(mpeg_flags), "");
	gtk_label_set_text(GTK_LABEL(mpeg_fileinfo), "");

	if ((fh = fopen(current_filename, "rb")) != NULL)
	{
		guint32 head;
		unsigned char tmp[4];
		struct frame frm;
		gboolean id3_found = FALSE;
		gchar tempstr[MAX_STR_LEN];

		fseek(fh, -sizeof (tag), SEEK_END);
		if (fread(&tag, 1, sizeof (tag), fh) == sizeof (tag))
		{
			if (!strncmp(tag.tag, "TAG", 3))
			{
				id3_found = TRUE;
				set_entry_tag(GTK_ENTRY(title_entry), tag.title, 30);
				set_entry_tag(GTK_ENTRY(artist_entry), tag.artist, 30);
				set_entry_tag(GTK_ENTRY(album_entry), tag.album, 30);
				set_entry_tag(GTK_ENTRY(year_entry), tag.year, 4);
				set_entry_tag(GTK_ENTRY(comment_entry), tag.comment, 30);
				gtk_list_select_item(GTK_LIST(GTK_COMBO(genre_combo)->list), g_list_index(genre_list, (gchar *) mpg123_id3_genres[tag.genre]));
			}
		}
		rewind(fh);
		if (fread(tmp, 1, 4, fh) != 4)
		{
			fclose(fh);
			return;
		}
		head = ((guint32) tmp[0] << 24) | ((guint32) tmp[1] << 16) | ((guint32) tmp[2] << 8) | (guint32) tmp[3];
		while (!mpg123_head_check(head))
		{
			head <<= 8;
			if (fread(tmp, 1, 1, fh) != 1)
			{
				fclose(fh);
				return;
			}
			head |= tmp[0];
		}
		if (mpg123_decode_header(&frm, head))
		{
			guchar *buf;
			gdouble tpf;
			gint pos;
			XHEADDATA xing_header;
			guint32 num_frames;

			buf = g_malloc(frm.framesize + 4);
			fseek(fh, -4, SEEK_CUR);
			fread(buf, 1, frm.framesize + 4, fh);
			xing_header.toc = NULL;
			tpf = mpg123_compute_tpf(&frm);
			if (frm.mpeg25)
				g_snprintf(tempstr, MAX_STR_LEN, "MPEG 2.5, layer %d", frm.lay);
			else
				g_snprintf(tempstr, MAX_STR_LEN, "MPEG %d, layer %d", frm.lsf + 1, frm.lay);
			gtk_label_set_text(GTK_LABEL(mpeg_level), tempstr);
			pos = ftell(fh);
			fseek(fh, 0, SEEK_END);
			if (mpg123_get_xing_header(&xing_header, buf))
			{
				num_frames = xing_header.frames;
				g_snprintf(tempstr, MAX_STR_LEN, "Bitrate: Variable,\navg. bitrate: %d kb/s", (gint) ((xing_header.bytes * 8) / (tpf * xing_header.frames * 1000)));
			}
			else
			{
				num_frames = (int) ((ftell(fh) - pos - (id3_found ? 128 : 0)) / mpg123_compute_bpf(&frm)) + 1;
				g_snprintf(tempstr, MAX_STR_LEN, "Bitrate: %d kb/s", tabsel_123[frm.lsf][frm.lay - 1][frm.bitrate_index]);
			}
			gtk_label_set_text(GTK_LABEL(mpeg_bitrate), tempstr);
			g_snprintf(tempstr, MAX_STR_LEN, "Samplerate: %ld Hz", mpg123_freqs[frm.sampling_frequency]);
			gtk_label_set_text(GTK_LABEL(mpeg_samplerate), tempstr);
			g_snprintf(tempstr, MAX_STR_LEN, "%s\nError protection: %s\nCopyright: %s\nOriginal: %s\nEmphasis: %s", channel_mode[frm.mode], bool_label[frm.error_protection], bool_label[frm.copyright], bool_label[frm.original], emphasis[frm.emphasis]);
			gtk_label_set_text(GTK_LABEL(mpeg_flags), tempstr);
			g_snprintf(tempstr, MAX_STR_LEN, "%d frames\nFilesize: %lu B", num_frames, ftell(fh));
			gtk_label_set_text(GTK_LABEL(mpeg_fileinfo), tempstr);
			g_free(buf);
		}
		fclose(fh);
	}
}
