
/*
 * The Real SoundTracker - GUI (menu bar)
 *
 * Copyright (C) 1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "i18n.h"
#include "menubar.h"
#include "gui.h"
#include "gui-subs.h"
#include "main.h"
#include "st-subs.h"
#include "keys.h"
#include "module-info.h"
#include "preferences.h"
#include "scope-group.h"
#include "track-editor.h"
#include "audioconfig.h"
#include "effectsconfig.h"
#include "gui-settings.h"
#include "tips-dialog.h"
#include "transposition.h"
#include "cheat-sheet.h"
#include "file-operations.h"
#include "instrument-editor.h"

#ifdef USE_GNOME
#include <gnome.h>
#else
static GtkItemFactory *item_factory;
static GtkAccelGroup *accel_group;
static int all_translated;
#endif

extern ScopeGroup *scopegroup;

#ifndef USE_GNOME
static GtkWidget *about = NULL;

static void
about_close (void)
{
    gtk_widget_destroy(about);
    about = NULL;
}
#endif

static void
about_dialog (void)
{
#ifdef USE_GNOME
    const gchar *authors[] = {"Michael Krause <m.krause@tu-harburg.de>", NULL};
    GtkWidget *about = gnome_about_new("SoundTracker",
				       VERSION,
				       "Copyright (C) 1998-1999 Michael Krause",
				       authors,
				       "Includes OpenCP player from Niklas Beisert and Tammo Hinrichs.",
				       NULL);
    gtk_widget_show (about);
#else
    GtkWidget *label, *button;

    if(about)
	gtk_widget_destroy(about);

    about = gtk_dialog_new();
    gtk_window_position (GTK_WINDOW(about), GTK_WIN_POS_CENTER);
    gtk_window_set_title(GTK_WINDOW(about), "About SoundTracker");

    label = gtk_label_new("SoundTracker " VERSION "\n\nCopyright (C) 1998-1999 Michael Krause\n<m.krause@tu-harburg.de>"
			  "\n\nIncludes OpenCP player from Niklas Beisert and Tammo Hinrichs.");
    gtk_container_border_width(GTK_CONTAINER(GTK_DIALOG(about)->vbox), 10);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(about)->vbox), label, TRUE, TRUE, 10);
    gtk_widget_show(label);
    
    button = gtk_button_new_with_label (_("Ok"));
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(about)->action_area), button, TRUE, TRUE, 10);
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                               GTK_SIGNAL_FUNC (about_close), 0);
    gtk_widget_grab_default (button);
    gtk_widget_show (button);
    
    gtk_widget_show(about);
#endif
}

static void
menubar_clear_callback (gint reply,
			gpointer data)
{
    if(reply == 0) {
	if((gint)data == 0) {
	    gui_free_xm();
	    gui_new_xm();
	    xm->modified = 0;
	} else {
	    gui_play_stop();
	    st_clean_song(xm);
	    gui_init_xm(1);
	    xm->modified = 0;
	}
    }
}

static void
menubar_clear_clicked (void *a,
		       gpointer b)
{
    if(xm->modified) {
	gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				  _("Are you sure you want to do this?\nAll changes will be lost!"),
				  menubar_clear_callback,
				  b);
    } else {
	menubar_clear_callback(0, b);
    }
}

static void
menubar_quit_requested_callback (gint reply,
				 gpointer data)
{
    if(reply == 0) {
	gtk_main_quit();
    }
}

void
menubar_quit_requested (void)
{
    if(xm->modified) {
	gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				  _("Are you sure you want to quit?\nAll changes will be lost!"),
				  menubar_quit_requested_callback,
				  0);
    } else {
	menubar_quit_requested_callback(0, NULL);
    }
}

/* Arghl. When will I be able to make GNOME a requirement such as GTK+? */

#ifdef USE_GNOME
#define _FUNCMACRO_(name) name (GtkWidget *widget)
#else
#define _FUNCMACRO_(name) name (void *dummy, void *dummy2, GtkWidget *widget)
#endif

static void
_FUNCMACRO_(menubar_backing_store_toggled)
{
    gui_settings.gui_use_backing_store = GTK_CHECK_MENU_ITEM(widget)->active;
    tracker_set_backing_store(tracker, gui_settings.gui_use_backing_store);
}

static void
_FUNCMACRO_(menubar_scopes_toggled)
{
    gui_settings.gui_display_scopes = GTK_CHECK_MENU_ITEM(widget)->active;
    scope_group_enable_scopes(scopegroup, gui_settings.gui_display_scopes);
}

static void
_FUNCMACRO_(menubar_save_settings_on_exit_toggled)
{
    gui_settings.save_settings_on_exit = GTK_CHECK_MENU_ITEM(widget)->active;
}

static void
menubar_save_settings_now (void)
{
    gui_settings_save_config();
    keys_save_config();
    audioconfig_save_config();
}

static void
menubar_handle_edit_menu (void *p,
			  guint a)
{
    Tracker *t = tracker;

    static void (* const functions[]) (Tracker *) = {
	track_editor_cut_pattern,
	track_editor_copy_pattern,
	track_editor_paste_pattern,
	track_editor_cut_track,
	track_editor_copy_track,
	track_editor_paste_track,
	track_editor_mark_selection,
	track_editor_cut_selection,
	track_editor_copy_selection,
	track_editor_paste_selection,
	track_editor_insert_track,
	track_editor_delete_track
    };

    functions[a](t);
}

#ifdef USE_GNOME

static GnomeUIInfo file_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_Open..."), NULL, fileops_open_dialog, (gpointer)DIALOG_LOAD_MOD, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 'O', GDK_CONTROL_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("Save _as..."), NULL, fileops_open_dialog, (gpointer)DIALOG_SAVE_MOD, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE_AS, 'A', GDK_CONTROL_MASK, NULL },

    GNOMEUIINFO_SEPARATOR,

#define SAVE_MOD_AS_WAV_POSITION 3
    { GNOME_APP_UI_ITEM, N_("Save Module as _WAV..."), NULL, fileops_open_dialog, (gpointer)DIALOG_SAVE_MOD_AS_WAV, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE_AS, 0, 0, NULL },

    GNOMEUIINFO_SEPARATOR,

    { GNOME_APP_UI_ITEM, N_("_Quit"), NULL, menubar_quit_requested, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_QUIT, 'Q', GDK_CONTROL_MASK, NULL },

    GNOMEUIINFO_END
};

static GnomeUIInfo module_menu[] = {
    { GNOME_APP_UI_ITEM, N_("Clear _All"), NULL, menubar_clear_clicked, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Clear _Patterns Only"), NULL, menubar_clear_clicked, (gpointer)1, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Optimize Module"), NULL, modinfo_optimize_module, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },

    GNOMEUIINFO_END
};

static GnomeUIInfo edit_pattern_menu[] = {
    { GNOME_APP_UI_ITEM, N_("C_ut"), NULL, menubar_handle_edit_menu, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CUT, GDK_F3, GDK_MOD1_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("_Copy"), NULL, menubar_handle_edit_menu, (gpointer)1, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_COPY, GDK_F4, GDK_MOD1_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("_Paste"), NULL, menubar_handle_edit_menu, (gpointer)2, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PASTE, GDK_F5, GDK_MOD1_MASK, NULL },

    GNOMEUIINFO_END
};

static GnomeUIInfo edit_track_menu[] = {
    { GNOME_APP_UI_ITEM, N_("C_ut"), NULL, menubar_handle_edit_menu, (gpointer)3, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CUT, GDK_F3, GDK_SHIFT_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("_Copy"), NULL, menubar_handle_edit_menu, (gpointer)4, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_COPY, GDK_F4, GDK_SHIFT_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("_Paste"), NULL, menubar_handle_edit_menu, (gpointer)5, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PASTE, GDK_F5, GDK_SHIFT_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("_Insert"), NULL, menubar_handle_edit_menu, (gpointer)10, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Delete"), NULL, menubar_handle_edit_menu, (gpointer)11, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },


    GNOMEUIINFO_END
};

static GnomeUIInfo edit_selection_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_Mark"), NULL, menubar_handle_edit_menu, (gpointer)6, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 'B', GDK_CONTROL_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("C_ut"), NULL, menubar_handle_edit_menu, (gpointer)7, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CUT, 'X', GDK_CONTROL_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("_Copy"), NULL, menubar_handle_edit_menu, (gpointer)8, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_COPY, 'C', GDK_CONTROL_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("_Paste"), NULL, menubar_handle_edit_menu, (gpointer)9, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PASTE, 'V', GDK_CONTROL_MASK, NULL },

    GNOMEUIINFO_END
};

static GnomeUIInfo edit_menu[] = {
    { GNOME_APP_UI_TOGGLEITEM, N_("_Jazz Edit Mode"), NULL, track_editor_toggle_jazz_edit, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_NONE, 0, ' ', GDK_SHIFT_MASK, NULL },

    GNOMEUIINFO_SEPARATOR,

    { GNOME_APP_UI_ITEM, N_("Transp_osition..."), NULL, transposition_dialog, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 'T', GDK_CONTROL_MASK, NULL },

    GNOMEUIINFO_SEPARATOR,

    GNOMEUIINFO_SUBTREE (N_("_Pattern"), edit_pattern_menu),
    GNOMEUIINFO_SUBTREE (N_("_Track"), edit_track_menu),
    GNOMEUIINFO_SUBTREE (N_("_Selection"), edit_selection_menu),

    GNOMEUIINFO_END
};

static GnomeUIInfo pattern_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_Find Unused Pattern"), NULL, modinfo_find_unused_pattern, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 'F', GDK_CONTROL_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("_Copy Current to Unused Pattern"), NULL, modinfo_copy_to_unused_pattern, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 'G', GDK_CONTROL_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_("C_lear Unused Patterns"), NULL, modinfo_clear_unused_patterns, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Pack Patterns"), NULL, modinfo_pack_patterns, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },

    GNOMEUIINFO_END
};

static GnomeUIInfo instrument_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_Load XI..."), NULL, fileops_open_dialog, (gpointer)DIALOG_LOAD_INSTRUMENT, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Save XI..."), NULL, fileops_open_dialog, (gpointer)DIALOG_SAVE_INSTRUMENT, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE_AS, 0, 0, NULL },

    GNOMEUIINFO_SEPARATOR,

    { GNOME_APP_UI_ITEM, N_("_Clear Current"), NULL, instrument_editor_clear_current_instrument, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },

    GNOMEUIINFO_SEPARATOR,

    { GNOME_APP_UI_ITEM, N_("_Delete Unused Instruments"), NULL, modinfo_delete_unused_instruments, (gpointer)0, NULL,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },

    GNOMEUIINFO_END
};

// Take care to adjust the indices in menubar_init_prefs() when reordering stuff here!
static GnomeUIInfo settings_menu[] = {
    { GNOME_APP_UI_TOGGLEITEM, N_("Display _Oscilloscopes"), 0, menubar_scopes_toggled, 0, 0,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },
    { GNOME_APP_UI_TOGGLEITEM, N_("Use _Backing Store"), 0, menubar_backing_store_toggled, 0, 0,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },

    GNOMEUIINFO_SEPARATOR,

    { GNOME_APP_UI_ITEM, N_("_Keyboard Configuration..."), NULL, keys_dialog, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Audio Configuration..."), NULL, audioconfig_dialog, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Playback Effects..."), NULL, effectsconfig_dialog, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_GUI Configuration..."), NULL, gui_settings_dialog, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0, NULL },

    GNOMEUIINFO_SEPARATOR,

    { GNOME_APP_UI_ITEM, N_("_Save Settings now"), NULL, menubar_save_settings_now, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
    { GNOME_APP_UI_TOGGLEITEM, N_("Save Settings on _Exit"), 0, menubar_save_settings_on_exit_toggled, 0, 0,
      GNOME_APP_PIXMAP_NONE, 0, 0, 0, NULL },

    GNOMEUIINFO_END
};

static GnomeUIInfo help_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_About..."), NULL, about_dialog, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT, 0, 0, NULL },

    GNOMEUIINFO_SEPARATOR,

    { GNOME_APP_UI_ITEM, N_("Show _Tips..."), NULL, tips_dialog_create, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_RED, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_XM Effects..."), NULL, cheat_sheet_dialog, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_RED, 'H', GDK_CONTROL_MASK, NULL },

    GNOMEUIINFO_END
};

static GnomeUIInfo main_menu[] = {
    GNOMEUIINFO_SUBTREE (N_("_File"), file_menu),
    GNOMEUIINFO_SUBTREE (N_("_Module"), module_menu),
    GNOMEUIINFO_SUBTREE (N_("_Edit"), edit_menu),
    GNOMEUIINFO_SUBTREE (N_("_Pattern"), pattern_menu),
    GNOMEUIINFO_SUBTREE (N_("_Instrument"), instrument_menu),
    GNOMEUIINFO_SUBTREE (N_("_Settings"), settings_menu),
    GNOMEUIINFO_SUBTREE (N_("_Help"), help_menu),
    GNOMEUIINFO_END
};

void
menubar_create (GtkWidget *window, GtkWidget *destbox)
{
    gnome_app_create_menus(GNOME_APP(window), main_menu);
#ifdef NO_AUDIOFILE
    gtk_widget_set_sensitive(file_menu[SAVE_MOD_AS_WAV_POSITION].widget, FALSE);
#endif
}

void
menubar_init_prefs ()
{
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(settings_menu[0].widget), gui_settings.gui_display_scopes);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(settings_menu[1].widget), gui_settings.gui_use_backing_store);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(settings_menu[9].widget), gui_settings.save_settings_on_exit);
}

void
menubar_write_accels ()
{
    // GNOME does this for us
}

#else

static GtkItemFactoryEntry menu_items[] = {
    { N_("/_File"),                                     NULL,          NULL, 0, "<Branch>" },
    { N_("/File/_Open..."),                             "<control>O",  fileops_open_dialog, DIALOG_LOAD_MOD },
    { N_("/File/Save _as..."),                          "<control>A",  fileops_open_dialog, DIALOG_SAVE_MOD },
    { N_("/File/-"),                                    NULL,          NULL, 0, "<Separator>" },
    { N_("/File/Save Module as _WAV..."),               NULL,          fileops_open_dialog, DIALOG_SAVE_MOD_AS_WAV },
    { N_("/File/-"),                                    NULL,          NULL, 0, "<Separator>" },
    { N_("/File/_Quit"),                                "<control>Q",  menubar_quit_requested, 0 },
    { N_("/_Module"),                                   NULL,          NULL, 0, "<Branch>" },
    { N_("/Module/Clear _All"),                         NULL,          menubar_clear_clicked, 0 },
    { N_("/Module/Clear _Patterns Only"),               NULL,          menubar_clear_clicked, 1 },
    { N_("/Module/_Optimize Module"),                   NULL,          modinfo_optimize_module, 0 },
    { N_("/_Edit"),                                     NULL,          NULL, 0, "<Branch>" },
    { N_("/Edit/_Jazz Edit Mode"),                      "<shift>space",track_editor_toggle_jazz_edit, 0 },
    { N_("/Edit/-"),                                    NULL,          NULL, 0, "<Separator>" },
    { N_("/Edit/_Transposition..."),                    "<control>T",  transposition_dialog, 0 },
    { N_("/Edit/-"),                                    NULL,          NULL, 0, "<Separator>" },
    { N_("/Edit/_Pattern"),                             NULL,          NULL, 0, "<Branch>" },
    { N_("/Edit/Pattern/C_ut"),                         "<alt>F3",     menubar_handle_edit_menu, 0 },
    { N_("/Edit/Pattern/_Copy"),                        "<alt>F4",     menubar_handle_edit_menu, 1 },
    { N_("/Edit/Pattern/_Paste"),                       "<alt>F5",     menubar_handle_edit_menu, 2 },
    { N_("/Edit/_Track"),                               NULL,          NULL, 0, "<Branch>" },
    { N_("/Edit/Track/C_ut"),                           "<shift>F3",   menubar_handle_edit_menu, 3 },
    { N_("/Edit/Track/_Copy"),                          "<shift>F4",   menubar_handle_edit_menu, 4 },
    { N_("/Edit/Track/_Paste"),                         "<shift>F5",   menubar_handle_edit_menu, 5 },
    { N_("/Edit/Track/_Insert"),                        NULL,          menubar_handle_edit_menu, 10 },
    { N_("/Edit/Track/_Delete"),                        NULL,          menubar_handle_edit_menu, 11 },
    { N_("/Edit/_Selection"),                           NULL,          NULL, 0, "<Branch>" },
    { N_("/Edit/Selection/_Mark"),                      "<control>B",  menubar_handle_edit_menu, 6 },
    { N_("/Edit/Selection/C_ut"),                       "<control>X",  menubar_handle_edit_menu, 7 },
    { N_("/Edit/Selection/_Copy"),                      "<control>C",  menubar_handle_edit_menu, 8 },
    { N_("/Edit/Selection/_Paste"),                     "<control>V",  menubar_handle_edit_menu, 9 },
    { N_("/_Pattern"),                                  NULL,          NULL, 0, "<Branch>" },
    { N_("/Pattern/_Find Unused Pattern"),              "<control>F",  modinfo_find_unused_pattern, 0 },
    { N_("/Pattern/_Copy Current to Unused Pattern"),   "<control>G",  modinfo_copy_to_unused_pattern, 0 },
    { N_("/Pattern/C_lear Unused Patterns"),            NULL,          modinfo_clear_unused_patterns, 0 },
    { N_("/Pattern/_Pack Patterns"),                    NULL,          modinfo_pack_patterns, 0 },
    { N_("/_Instrument"),                               NULL,          NULL, 0, "<Branch>" },
    { N_("/Instrument/_Load XI..."),                    NULL,          fileops_open_dialog, DIALOG_LOAD_INSTRUMENT },
    { N_("/Instrument/_Save XI..."),                    NULL,          fileops_open_dialog, DIALOG_SAVE_INSTRUMENT },
    { N_("/Instrument/-"),                              NULL,          NULL, 0, "<Separator>" },
    { N_("/Instrument/_Clear Current"),                 NULL,          instrument_editor_clear_current_instrument, 0 },
    { N_("/Instrument/-"),                              NULL,          NULL, 0, "<Separator>" },
    { N_("/Instrument/_Delete Unused Instruments"),     NULL,          modinfo_delete_unused_instruments, 0 },
    { N_("/_Settings"),                                 NULL,          NULL, 0, "<Branch>" },
    { N_("/Settings/Display _Oscilloscopes"),           NULL,          menubar_scopes_toggled, 0, "<ToggleItem>" },
    { N_("/Settings/Use _Backing Store"),               NULL,          menubar_backing_store_toggled, 0, "<ToggleItem>" },
    { N_("/Settings/-"),                                NULL,          NULL, 0, "<Separator>" },
    { N_("/Settings/_Keyboard Configuration..."),       NULL,          keys_dialog, 0 },
    { N_("/Settings/_Audio Configuration..."),          NULL,          audioconfig_dialog, 0 },
    { N_("/Settings/_Playback Effects..."),             NULL,          effectsconfig_dialog, 0 },
    { N_("/Settings/_GUI Configuration..."),            NULL,          gui_settings_dialog, 0 },
    { N_("/Settings/-"),                                NULL,          NULL, 0, "<Separator>" },
    { N_("/Settings/_Save Settings now"),               NULL,          menubar_save_settings_now, 0 },
    { N_("/Settings/Save Settings on _Exit"),           NULL,          menubar_save_settings_on_exit_toggled, 0, "<ToggleItem>" },
    { N_("/_Help"),                                     NULL,          NULL, 0, "<Branch>" },
    { N_("/Help/_About..."),                            NULL,          about_dialog, 0 },
    { N_("/Help/-"),                                    NULL,          NULL, 0, "<Separator>" },
    { N_("/Help/Show _Tips..."),                        NULL,          tips_dialog_create, 0 },
    { N_("/Help/_XM Effects..."),                       "<control>H",  cheat_sheet_dialog, 0 },
};

static char *
id (const char *p)
{
    return (char*)p;
}

void
menubar_init_prefs ()
{
    GtkWidget *w;
    char buf[256];
#ifdef ENABLE_NLS
    char * (*tl)(const char *) = all_translated ? gettext : id;
#else
    char * (*tl)(const char *) = id;
#endif

    prefs_check_prefs_dir();
    sprintf(buf, "%s/non-gnome-accels", prefs_get_prefsdir());

    gtk_item_factory_parse_rc(buf);

    w = gtk_item_factory_get_widget(item_factory, tl("/Settings/Display Oscilloscopes"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), gui_settings.gui_display_scopes);
    w = gtk_item_factory_get_widget(item_factory, tl("/Settings/Use Backing Store"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), gui_settings.gui_use_backing_store);
    w = gtk_item_factory_get_widget(item_factory, tl("/Settings/Save Settings on Exit"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), gui_settings.save_settings_on_exit);
}

void
menubar_create (GtkWidget *window, GtkWidget *destbox)
{
    GtkWidget *thing;
    int i;

    accel_group = gtk_accel_group_new ();

    item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR,
					 "<main>",
					 accel_group);

    for(i = 0, all_translated = TRUE;
	all_translated && i < sizeof(menu_items) / sizeof(menu_items[0]); i++) {
	all_translated &= !strcmp(menu_items[i].path, _(menu_items[i].path));
    }

    if(all_translated) {
	for(i = 0; i < sizeof(menu_items) / sizeof(menu_items[0]); i++) {
	    menu_items[i].path = _(menu_items[i].path);
	}
    } else {
#ifdef ENABLE_NLS
	fprintf(stderr, "*** SoundTracker Non-GNOME: Disabling menu bar translation because\n*** not all strings have been translated.\n");
#endif
    }

    gtk_item_factory_create_items (item_factory,
				   sizeof(menu_items) / sizeof(menu_items[0]),
				   menu_items, NULL);

    gtk_accel_group_attach (accel_group, GTK_OBJECT (window));

    thing = gtk_item_factory_get_widget (item_factory, "<main>");
    gtk_box_pack_start(GTK_BOX(destbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
}

void
menubar_write_accels ()
{
    char buf[256];

    prefs_check_prefs_dir();
    sprintf(buf, "%s/non-gnome-accels", prefs_get_prefsdir());

    gtk_item_factory_dump_rc(buf, NULL, TRUE);
}

#endif
