#include "snd.h"

#if defined(NEXT) || defined(HAVE_SYS_DIR_H)
  #include <sys/dir.h>
  #include <sys/dirent.h>
#else
  #if defined(WINDOZE) && (!(defined(__CYGWIN__)))
    #include <direct.h>
  #else
    #include <dirent.h>
  #endif
#endif

#include <X11/cursorfont.h>

#ifndef WINDOZE
  #define INIT_FILE_NAME "~/.snd"
#else
  #define INIT_FILE_NAME "snd-init"
#endif
#define EPS_FILE_NAME "snd.eps"
#define FALLBACK_FONT "fixed"

/* our default basic colors (resource file can override these): */
#define HIGHLIGHT_COLOR      "ivory1"
#define BASIC_COLOR          "ivory2"
#define POSITION_COLOR       "ivory3"
#define ZOOM_COLOR           "ivory4"
#define CURSOR_COLOR         "red"
#define SELECTION_COLOR      "lightsteelblue1"
#define MIX_COLOR            "lightgreen"
#define ENVED_WAVEFORM_COLOR "blue"
#define MIX_WAVEFORM_COLOR   "darkgray"
#define GRAPH_COLOR          "white"
#define SELECTED_GRAPH_COLOR "white"
#define DATA_COLOR           "black"
#define SELECTED_DATA_COLOR  "black"
#define MARK_COLOR           "red"
#define LISTENER_COLOR       "AliceBlue"
#define LIGHT_BLUE_COLOR     "lightsteelblue1"
#define LIGHTER_BLUE_COLOR   "AliceBlue"
#define MIX_FOCUS_COLOR      "green2"
#define WHITE_COLOR          "white"
#define BLACK_COLOR          "black"
#define GREEN_COLOR          "green2"
#define RED_COLOR            "red"
#define YELLOW_COLOR         "yellow"
#define TEXT_FOCUS_COLOR     "white"
#define FILTER_WAVEFORM_COLOR "blue"
#define PUSHED_BUTTON_COLOR  "lightsteelblue1"
#define SASH_COLOR           "lightgreen"

#define DEFAULT_SPECTROGRAM_COLOR -1 /* -1 = just lines or index */
#define MIXER_GROUPS 6
#define CHANNEL_SASH_INDENT -10
#define CHANNEL_SASH_SIZE 0
/* 0 means: use Motif default size */
#define ENVED_POINT_SIZE 10
#define NOTEBOOK_BINDING_WIDTH 20

#if HAVE_XmHTML
/* XmHTML defaults */
#define HTML_WIDTH 600
#define HTML_HEIGHT 400
#define HTML_DIR "."
#define HTML_FONT_SIZE_LIST "14,10,24,24,18,14,12"
#define HTML_FIXED_FONT_SIZE_LIST "14,10"
#endif

#define NO_ICON 0
#define PLAIN_ICON 1
#define XPM_ICON 2

#define TINY_FONT "6x12"

/* we assume later that we can always find these fonts (if resource file gives bogus entry, we fall back on these) */
#ifdef SGI
  #define BUTTON_FONT "-*-times-medium-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define BOLD_BUTTON_FONT "-*-times-bold-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define AXIS_LABEL_FONT "-*-times-medium-r-normal-*-20-*-*-*-*-*-iso8859-1"
  #define AXIS_NUMBERS_FONT "-*-courier-medium-r-normal-*-14-*-*-*-*-*-iso8859-1"
  #define HELP_TEXT_FONT "9x15"
  #define ICON_TYPE PLAIN_ICON
#else
#if defined(LINUX) || defined(SCO5) || defined(UW2) || defined(SOLARIS) || defined(HPUX) || defined(ALPHA)
  #define BUTTON_FONT "-*-times-medium-r-*-*-12-*-*-*-*-*-iso8859-1"
  #define BOLD_BUTTON_FONT "-*-times-bold-r-*-*-12-*-*-*-*-*-iso8859-1"
  #define AXIS_LABEL_FONT "-*-times-medium-r-*-*-16-*-*-*-*-*-iso8859-1"
  #define AXIS_NUMBERS_FONT "-*-courier-medium-r-*-*-12-*-*-*-*-*-iso8859-1"
  #define HELP_TEXT_FONT "8x13"
  #define ICON_TYPE XPM_ICON
#else
  #define BUTTON_FONT "-*-times-medium-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define BOLD_BUTTON_FONT "-*-times-bold-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define AXIS_LABEL_FONT "-*-times-medium-r-*-*-20-*-*-*-*-*-iso8859-1"
  #define AXIS_NUMBERS_FONT "-*-courier-medium-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define HELP_TEXT_FONT "9x15"
  #define ICON_TYPE NO_ICON
#endif
#endif

#define POSITION_SLIDER_WIDTH 13
#define ZOOM_SLIDER_WIDTH 10
#define TOGGLE_SIZE 0
#define CHANNEL_MIN_HEIGHT 150  /* open size (set to 5 immediately thereafter) */
                                /* paned window default setup is not very smart, so we force these to be this size to begin with */
                                /* this number is only a first approximation -- we try not to expand below the screen */
                                /* if too small (i.e. 100), the scrollbars are sometimes messed up on the initial layout */

#define SASH_SIZE 14
#define SASH_INDENT -6
#define AUTO_RESIZE_DEFAULT 1

/* /usr/lib/X11/rgb.txt, /usr/lib/X11/fonts/Type1/fonts.dir, /usr/lib/X11/fonts/misc */
/* schemes are in /usr/lib/X11/schemes */

#define HELP_ROWS 12
#define HELP_COLUMNS 56
/* these set the initial size of the (non XmHTML) help dialog text area */

typedef struct {
  char *highlight_color;
  char *basic_color;
  char *position_color;
  char *zoom_color;
  char *cursor_color;
  char *selection_color;
  char *mix_color;
  char *mix_focus_color;
  char *text_focus_color;
  char *graph_color;
  char *selected_graph_color;
  char *data_color;
  char *selected_data_color;
  char *listener_color;
  char *mark_color;
  char *pushed_button_color;
  char *enved_waveform_color;
  char *mix_waveform_color;
  char *filter_waveform_color;
  char *sash_color;
  char *white_color;
  char *black_color;
  char *red_color;
  char *yellow_color;
  char *green_color;
  char *light_blue_color;
  char *lighter_blue_color;

  char *use_schemes;
  char *button_font;
  char *listener_font;
  char *bold_button_font;
  char *axis_label_font;
  char *axis_numbers_font;
  char *help_text_font;
  char *init_file_name;
  char *eps_file_name;
  int def_output_type;
  int spectrogram_color;
  int overwrite_check;
  int ngroups;
  int auto_resize;
  int group_out_chans;
  int horizontal_panes;
  int zoom_slider_width;
  int position_slider_width;
  int toggle_size;
  int enved_point_size;
  int channel_sash_indent;
  int channel_sash_size;
  int sash_size;
  int sash_indent;
#if HAVE_XmHTML
  int html_width;
  int html_height;
  char *html_dir;
  char *html_font_size_list;
  char *html_fixed_font_size_list;
#endif
} sndres;

static XtResource resources[] = {
  {"highlightcolor","Highlightcolor",XmRString,sizeof(char *),XtOffset(sndres *,highlight_color),XmRString,HIGHLIGHT_COLOR},
  {"basiccolor","Basiccolor",XmRString,sizeof(char *),XtOffset(sndres *,basic_color),XmRString,BASIC_COLOR},
  {"positioncolor","Positioncolor",XmRString,sizeof(char *),XtOffset(sndres *,position_color),XmRString,POSITION_COLOR},
  {"zoomcolor","Zoomcolor",XmRString,sizeof(char *),XtOffset(sndres *,zoom_color),XmRString,ZOOM_COLOR},
  {"listenercolor","Listenercolor",XmRString,sizeof(char *),XtOffset(sndres *,listener_color),XmRString,LISTENER_COLOR},
  {"cursorcolor","Cursorcolor",XmRString,sizeof(char *),XtOffset(sndres *,cursor_color),XmRString,CURSOR_COLOR},
  {"selectioncolor","Selectioncolor",XmRString,sizeof(char *),XtOffset(sndres *,selection_color),XmRString,SELECTION_COLOR},
  {"mixcolor","Mixcolor",XmRString,sizeof(char *),XtOffset(sndres *,mix_color),XmRString,MIX_COLOR},
  {"mixfocuscolor","Mixfocuscolor",XmRString,sizeof(char *),XtOffset(sndres *,mix_focus_color),XmRString,MIX_FOCUS_COLOR},
  {"textfocuscolor","Textfocuscolor",XmRString,sizeof(char *),XtOffset(sndres *,text_focus_color),XmRString,TEXT_FOCUS_COLOR},
  {"redcolor","Redcolor",XmRString,sizeof(char *),XtOffset(sndres *,red_color),XmRString,RED_COLOR},
  {"greencolor","Greencolor",XmRString,sizeof(char *),XtOffset(sndres *,green_color),XmRString,GREEN_COLOR},
  {"whitecolor","Whitecolor",XmRString,sizeof(char *),XtOffset(sndres *,white_color),XmRString,WHITE_COLOR},
  {"blackcolor","Blackcolor",XmRString,sizeof(char *),XtOffset(sndres *,black_color),XmRString,BLACK_COLOR},
  {"redcolor","Redcolor",XmRString,sizeof(char *),XtOffset(sndres *,red_color),XmRString,RED_COLOR},
  {"lightbluecolor","Lightbluecolor",XmRString,sizeof(char *),XtOffset(sndres *,light_blue_color),XmRString,LIGHT_BLUE_COLOR},
  {"lighterbluecolor","Ligterbluecolor",XmRString,sizeof(char *),XtOffset(sndres *,lighter_blue_color),XmRString,LIGHTER_BLUE_COLOR},
  {"yellowcolor","Yellowcolor",XmRString,sizeof(char *),XtOffset(sndres *,yellow_color),XmRString,YELLOW_COLOR},
  {"envedwaveformcolor","Envedwaveformcolor",XmRString,sizeof(char *),XtOffset(sndres *,enved_waveform_color),XmRString,ENVED_WAVEFORM_COLOR},
  {"mixwaveformcolor","Mixwaveformcolor",XmRString,sizeof(char *),XtOffset(sndres *,mix_waveform_color),XmRString,MIX_WAVEFORM_COLOR},
  {"filterwaveformcolor","Filterwaveformcolor",XmRString,sizeof(char *),XtOffset(sndres *,filter_waveform_color),XmRString,FILTER_WAVEFORM_COLOR},
  {"graphcolor","Graphcolor",XmRString,sizeof(char *),XtOffset(sndres *,graph_color),XmRString,GRAPH_COLOR},
  {"selectedgraphcolor","Selectedgraphcolor",XmRString,sizeof(char *),XtOffset(sndres *,selected_graph_color),XmRString,SELECTED_GRAPH_COLOR},
  {"datacolor","Datacolor",XmRString,sizeof(char *),XtOffset(sndres *,data_color),XmRString,DATA_COLOR},
  {"selecteddatacolor","Selecteddatacolor",XmRString,sizeof(char *),XtOffset(sndres *,selected_data_color),XmRString,SELECTED_DATA_COLOR},
  {"markcolor","Markcolor",XmRString,sizeof(char *),XtOffset(sndres *,mark_color),XmRString,MARK_COLOR},
  {"sashcolor","Sashcolor",XmRString,sizeof(char *),XtOffset(sndres *,sash_color),XmRString,SASH_COLOR},
  {"pushedbuttoncolor","Pushedbuttoncolor",XmRString,sizeof(char *),XtOffset(sndres *,pushed_button_color),XmRString,PUSHED_BUTTON_COLOR},
  {"useSchemes","UseSchemes",XmRString,sizeof(char *),XtOffset(sndres *,use_schemes),XmRString,"none"},
  {"buttonFont","ButtonFont",XmRString,sizeof(char *),XtOffset(sndres *,button_font),XmRString,BUTTON_FONT},
  {"listenerFont","ListenerFont",XmRString,sizeof(char *),XtOffset(sndres *,listener_font),XmRString,NULL},
  {"boldbuttonFont","BoldbuttonFont",XmRString,sizeof(char *),XtOffset(sndres *,bold_button_font),XmRString,BOLD_BUTTON_FONT},
  {"axisLabelFont","AxisLabelFont",XmRString,sizeof(char *),XtOffset(sndres *,axis_label_font),XmRString,AXIS_LABEL_FONT},
  {"axisNumbersFont","AxisNumbersFont",XmRString,sizeof(char *),XtOffset(sndres *,axis_numbers_font),XmRString,AXIS_NUMBERS_FONT},
  {"helpTextFont","HelpTextFont",XmRString,sizeof(char *),XtOffset(sndres *,help_text_font),XmRString,HELP_TEXT_FONT},
  {"initFile","InitFile",XmRString,sizeof(char *),XtOffset(sndres *,init_file_name),XmRString,INIT_FILE_NAME},
  {"epsFile","EpsFile",XmRString,sizeof(char *),XtOffset(sndres *,eps_file_name),XmRString,EPS_FILE_NAME},
  {"defaultOutputType","DefaultOutputType",XmRInt,sizeof(int),XtOffset(sndres *,def_output_type),XmRImmediate,(XtPointer)DEFAULT_OUTPUT_TYPE},
  {"spectrogramColor","SpectrogramColor",XmRInt,sizeof(int),XtOffset(sndres *,spectrogram_color),XmRImmediate,(XtPointer)DEFAULT_SPECTROGRAM_COLOR},
  {"overwriteCheck","OverwriteCheck",XmRInt,sizeof(int),XtOffset(sndres *,overwrite_check),XmRImmediate,(XtPointer)0},
  {"mixerGroups","MixerGroups",XmRInt,sizeof(int),XtOffset(sndres *,ngroups),XmRImmediate,(XtPointer)MIXER_GROUPS},
  {"autoResize","AutoResize",XmRInt,sizeof(int),XtOffset(sndres *,auto_resize),XmRImmediate,(XtPointer)AUTO_RESIZE_DEFAULT},
  {"groupOutChans","GroupOutChans",XmRInt,sizeof(int),XtOffset(sndres *,group_out_chans),XmRImmediate,(XtPointer)4},
  {"horizontalPanes","HorizontalPanes",XmRInt,sizeof(int),XtOffset(sndres *,horizontal_panes),XmRImmediate,(XtPointer)SOUNDS_VERTICAL},
  {"zoomSliderWidth","ZoomSliderWidth",XmRInt,sizeof(int),XtOffset(sndres *,zoom_slider_width),XmRImmediate,(XtPointer)ZOOM_SLIDER_WIDTH},
  {"positionSliderWidth","PositionSliderWidth",XmRInt,sizeof(int),XtOffset(sndres *,position_slider_width),XmRImmediate,(XtPointer)POSITION_SLIDER_WIDTH},
  {"toggleSize","ToggleSize",XmRInt,sizeof(int),XtOffset(sndres *,toggle_size),XmRImmediate,(XtPointer)TOGGLE_SIZE},
  {"envedPointSize","EnvedPointSize",XmRInt,sizeof(int),XtOffset(sndres *,enved_point_size),XmRImmediate,(XtPointer)ENVED_POINT_SIZE},
  {"channelSashIndent","ChannelSashIndent",XmRInt,sizeof(int),XtOffset(sndres *,channel_sash_indent),XmRImmediate,(XtPointer)CHANNEL_SASH_INDENT},
  {"channelSashSize","ChannelSashSize",XmRInt,sizeof(int),XtOffset(sndres *,channel_sash_size),XmRImmediate,(XtPointer)CHANNEL_SASH_SIZE},
  {"sashSize","SashSize",XmRInt,sizeof(int),XtOffset(sndres *,sash_size),XmRImmediate,(XtPointer)SASH_SIZE},
  {"sashIndent","SashIndent",XmRInt,sizeof(int),XtOffset(sndres *,sash_indent),XmRImmediate,(XtPointer)SASH_INDENT},
#if HAVE_XmHTML
  {"htmlWidth","HtmlWidth",XmRInt,sizeof(int),XtOffset(sndres *,html_width),XmRImmediate,(XtPointer)HTML_WIDTH},
  {"htmlHeight","HtmlHeight",XmRInt,sizeof(int),XtOffset(sndres *,html_height),XmRImmediate,(XtPointer)HTML_HEIGHT},
  {"htmldir","HtmlDir",XmRString,sizeof(char *),XtOffset(sndres *,html_dir),XmRString,HTML_DIR},
  {"htmlfontsizelist","Htmlfontsizelist",XmRString,sizeof(char *),XtOffset(sndres *,html_font_size_list),XmRString,HTML_FONT_SIZE_LIST},
  {"htmlfixedfontsizelist","Htmlfixedfontsizelist",XmRString,sizeof(char *),XtOffset(sndres *,html_fixed_font_size_list),XmRString,HTML_FIXED_FONT_SIZE_LIST}
#endif
};

static task_manager *make_task_manager(snd_state *ss, Widget shell, Display *dpy)
{
  task_manager *tm;
  tm = (task_manager *)CALLOC(1,sizeof(task_manager));
  tm->slice = 0;
  tm->ss = ss;
  tm->shell = shell;
  tm->dpy = dpy;
  return(tm);
}

static void Window_Close(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_exit_cleanly((snd_state *)clientData);
}

int snd_not_current(snd_info *sp, void *dat)
{
  /* check for change in update status */
  int needs_update;
  snd_state *ss;
  needs_update = (file_write_date(sp->fullname) != sp->write_date);
  if (needs_update != sp->need_update)
    {
      ss = sp->state;
      sp->need_update = needs_update;
      if ((needs_update) && (auto_update(ss)))
	snd_update(ss,sp);
      else snd_file_bomb_icon(sp,needs_update);
    }
  return(0);
}

static void corruption_check(XtPointer clientData, XtIntervalId *id)
{
  snd_state *ss = (snd_state *)clientData;
  if (corruption_time(ss) > 0.0)
    {
      if ((!(play_in_progress())) && (!(record_in_progress())))
	{
	  map_over_sounds(ss,snd_not_current,NULL);
	}
      XtAppAddTimeOut((ss->sgx)->mainapp,(unsigned long)(corruption_time(ss)*1000),(XtTimerCallbackProc)corruption_check,clientData);
    }
}

void add_dialog(snd_state *ss, Widget dialog)
{
  state_context *sx;
  int i;
  sx = ss->sgx;
  if (sx->dialog_list_size == 0)
    {
      sx->dialog_list_size = 8;
      sx->dialogs = (Widget *)CALLOC(sx->dialog_list_size,sizeof(Widget));
      sx->ndialogs = 0;
    }
  else
    {
      if (sx->ndialogs == sx->dialog_list_size)
	{
	  sx->dialog_list_size *= 2;
	  sx->dialogs = (Widget *)REALLOC(sx->dialogs,sx->dialog_list_size * sizeof(Widget));
	  for (i=sx->ndialogs;i<sx->dialog_list_size;i++) sx->dialogs[i] = NULL;
	}
    }
  sx->dialogs[sx->ndialogs] = dialog;
  sx->ndialogs++;
}

void dismiss_all(snd_state *ss)
{
  state_context *sx;
  int i;
  sx = ss->sgx;
  if (sx->dialog_list_size > 0)
    {
      for (i=0;i<sx->ndialogs;i++)
	{
	  if (sx->dialogs[i])
	    {
	      if (XtIsManaged(sx->dialogs[i])) XtUnmanageChild(sx->dialogs[i]);
	    }
	}
    }
}

static void minify_maxify_window(Widget w,XtPointer clientData,XEvent *event,Boolean *cont) 
{
  XMapEvent *ev = (XMapEvent *)event;
  snd_state *ss = (snd_state *)clientData;
  /* ev->type can be several things, but the ones we care about here are
   * MapNotify and UnmapNotify.  Snd dialogs are "windows" in X-jargon, so
   * when the main window is minimized (iconified), other active dialogs
   * aren't also closed unless we mess with them explicitly here.  We keep
   * a list of created dialogs, and depend on XtIsManaged to tell us which
   * ones need to be unmanaged. 
   */
  if (ev->type == UnmapNotify) dismiss_all(ss);
}

#if TRAP_SEGFAULT
#include <setjmp.h>
/* stolen from scwm.c */
static jmp_buf envHandleEventsLoop;

static RETSIGTYPE segv(int ignored)
{
  snd_error("Caught seg fault. trying to continue...");
  siglongjmp(envHandleEventsLoop,1);
}
#endif

static char **auto_open_file_names = NULL;
static int auto_open_files = 0;
static int noglob = 0, noinit = 0;
static char *startup_filename = NULL;

static Boolean startup_funcs(XtPointer clientData)
{
  task_manager *tm = (task_manager *)clientData;
  Atom wm_delete_window;
  snd_info *sp;
  snd_state *ss;
  chan_info *cp;
  axis_info *ap;
  float apsx,apzx;
  char *argname;
#ifndef LESSTIF_VERSION
#ifndef _MSC_VER
  DIR *dp;
#endif
#endif
  int i,files;
  static int auto_open_ctr = 0;
  ss = tm->ss;
  switch (tm->slice)
    {
    case 0:
      create_popup_menu(ss);
      break;
    case 1:
      intern_atoms(ss);
      break;
    case 2:
#ifndef SND_AS_WIDGET
#ifndef __alpha__
      InitializeDrop(ss);
#endif
#endif
      break;
    case 3:  
#ifndef SND_AS_WIDGET
#ifndef NEXT
      /* trap outer-level Close for cleanup check */
      wm_delete_window = XmInternAtom(tm->dpy,"WM_DELETE_WINDOW",FALSE);
      XmAddWMProtocolCallback(tm->shell,wm_delete_window,Window_Close,(XtPointer)ss);
#endif
      XtAddEventHandler(tm->shell,StructureNotifyMask,FALSE,minify_maxify_window,(XtPointer)ss);
#endif
      break;
    case 4:
      make_graph_cursor(ss);
      make_mix_cursor(ss);
      break;
    case 5:
      CLM_connect(ss); 
      break;
    case 6: 
      snd_load_init_file(ss,noglob,noinit);
      break;
    case 7: 
      if (auto_open_files > 0)
	{
	  argname = auto_open_file_names[auto_open_ctr];
	  if (argname)
	    { /* wanted to use "-d" and "-i" (or "-s") but they're in use */
	      if ((strcmp("-h",argname) == 0) || 
		  (strcmp("-horizontal",argname) == 0) ||
		  (strcmp("-v",argname) == 0) || 
		  (strcmp("-vertical",argname) == 0) ||
		  (strcmp("-notebook",argname) == 0) ||
		  (strcmp("-scroller",argname) == 0) ||
		  (strcmp("-separate",argname) == 0) ||
		  (strcmp("-noglob",argname) == 0) ||
		  (strcmp("-noinit",argname) == 0))
		auto_open_ctr++; 
	      else
		{
		  if ((strcmp("-p",argname) == 0) ||
		      (strcmp("-preload",argname) == 0))
		    {
		      /* preload sound files in dir (can be ., should be unquoted) */
		      auto_open_ctr++;
		      add_directory_to_prevlist(ss,auto_open_file_names[auto_open_ctr]);
		      auto_open_ctr++;
		    }
		  else
		    {
		      if ((strcmp("-l",argname) == 0) ||
			  (strcmp("-load",argname) == 0))
			{
			  /* grab session name -- if arg is "." grab latest on this directory */
			  auto_open_ctr++;
			  snd_load_file(ss,auto_open_file_names[auto_open_ctr]);
			  auto_open_ctr++;
			}
		      else
			{
			  if ((strcmp("-e",argname) == 0) ||
			      (strcmp("-eval",argname) == 0))
			    {
			      /* evaluate expression */
			      auto_open_ctr++;
			      clm_doit(ss,auto_open_file_names[auto_open_ctr],FALSE);
			      auto_open_ctr++;
			    }
			  else
			    {
			      if (ss->pending_open)
				{
				  /* we're still waiting for a previous no-header initial file to be opened */
				  /* go into an event-loop waiting for the previous open to be completed */
				  while (ss->pending_open)
				    check_for_event(ss);
				}
			      if (startup_filename == NULL) startup_filename = copy_string(argname);
			      sp = snd_open_file_unselected(argname,ss);
			      auto_open_ctr++;
			      if ((sp) && (auto_open_ctr < auto_open_files))
				{
				  if (strcmp("-s",auto_open_file_names[auto_open_ctr]) == 0)
				    {
				      /* start up info follows -- [sx,sy,zx,zy] ... (per chan) */
				      auto_open_ctr++;
				      for (i=0;i<sp->nchans;i++)
					{
					  cp = sp->chans[i];
					  ap = cp->axis;
					  sscanf(auto_open_file_names[auto_open_ctr],"%f,%f,%f,%f",&apsx,&ap->sy,&apzx,&ap->zy);
					  ap->sx = apsx; ap->zx = apzx;
					  set_xy_bounds(cp,ap);
					  auto_open_ctr++;
					}}}}}}}}
	  if (auto_open_ctr < auto_open_files) return(FALSE); /* i.e. come back to this branch */
	}
      break;
    case 8: 
      /* this stupid thing (which I can't customize without major hassles) takes forever on large directories */
#ifndef LESSTIF_VERSION
#ifndef _MSC_VER
      files = 0;
      if ((dp=opendir(".")) != NULL)
	{
	  while ((files < 400) && (readdir(dp) != NULL)) files++;
	}
      closedir(dp);
      if (files < 400) CreateOpenDialog(tm->shell,(XtPointer)ss);
#endif
#endif
      /* in Lesstif, we can't set up the dialog in advance in the background
       * because it requires that the dialog be activated ("managed") before
       * various things are done to it as it is initialized.  But that requires
       * the dumb thing to popup unexpectedly.
       */
      break;
    case 9:
      if (ss->init_window_width > 0) set_snd_window_width(ss,ss->init_window_width);
      if (ss->init_window_height > 0) set_snd_window_height(ss,ss->init_window_height);
      if (ss->init_window_x != -1) set_snd_window_x(ss,ss->init_window_x);
      if (ss->init_window_y != -1) set_snd_window_y(ss,ss->init_window_y);
      break;
    case 10: 
      XtAppAddTimeOut((ss->sgx)->mainapp,(unsigned long)(corruption_time(ss)*1000),corruption_check,(XtPointer)ss);
      break;
    case 11: 
#if HAVE_GUILE
      if (dont_start(ss,startup_filename)) snd_exit(1);
#endif
#if TRAP_SEGFAULT
      if (trap_segfault(ss)) signal(SIGSEGV,segv);
#endif
      if (ss->sounds[0]) select_channel(ss->sounds[0],0);
      FREE(tm);
      return(TRUE); 
      break;
    }
  tm->slice++;
  return(FALSE);
}

#if (ICON_TYPE != NO_ICON)
static void SetupIcon(Widget shell);
#endif

#ifndef __cplusplus
/* 'String' is trouble in C++ -- easiest fix is probably to use 'char *' */
static void muffle_warning(String name, String type, String class, String defaultp, String *params, Cardinal *num_params)
{
  /* these warnings are occurring when they should not, and they are of no interest to anyone, so shove a sock in Xt */
#ifdef DEBUGGING
  fprintf(stderr,"ignoring: %s: %s\n",name,defaultp);
#endif
}
#endif

static Pixel get_color(Widget shell,
		       char *rs_color, char *defined_color, char *fallback_color, char *second_fallback_color,
		       int use_white)
{
  Colormap cmap;
  Display *dpy;
  int scr;
  XColor tmp_color,ignore;
  dpy=XtDisplay(shell);
  scr = DefaultScreen(dpy);
  cmap=DefaultColormap(dpy,scr);
  if ((!XAllocNamedColor(dpy,cmap,rs_color,&tmp_color,&ignore)) &&
      (!XAllocNamedColor(dpy,cmap,defined_color,&tmp_color,&ignore)) &&
      ((!fallback_color) || (!XAllocNamedColor(dpy,cmap,fallback_color,&tmp_color,&ignore))) &&
      ((!second_fallback_color) || (!XAllocNamedColor(dpy,cmap,second_fallback_color,&tmp_color,&ignore))))
    {
      if (use_white)
	{
	  snd_error(STR_no_color_use_white,rs_color);
	  return(WhitePixel(dpy,scr));
	}
      else
	{
	  snd_error(STR_no_color_use_black,rs_color);
	  return(BlackPixel(dpy,scr));
	}
    }
  return(tmp_color.pixel);
}

static XFontStruct *get_font(Display *dpy, char *font, char *backupfont, char *fallbackfont)
{
  XFontStruct *fs = NULL;
  fs = XLoadQueryFont(dpy,font);
  if ((!fs) && (backupfont))
    {
      snd_error(STR_cant_find_font,font);
      fs = XLoadQueryFont(dpy,backupfont);
      if ((!fs) && (fallbackfont))
	{
	  fs = XLoadQueryFont(dpy,fallbackfont);
	  if (!fs) snd_error(STR_serious_font_trouble,fallbackfont);
	}
    }
  return(fs);
}

static XmFontList get_xm_font(XFontStruct *fs, char *set, char *font)
{
  XmFontList fl = NULL;
  fl = XmFontListCreate(fs,set);
  XmFontListEntryCreate(font,XmFONT_IS_FONT,fs);
  return(fl);
}

#ifdef SND_AS_WIDGET
void snd_as_widget(int argc, char **argv, XtAppContext app, Widget parent, Arg *caller_args, int caller_argn)
{
  snd_state *state;
#else

void snd_doit(snd_state *state,int argc, char **argv)
{
  XtAppContext app;     
#endif

  Widget shell;
  Display *dpy;
  Drawable wn;
  Arg args[32];
  int i,n,err;
  sndres snd_rs;
  state_context *sx;
  Widget menu;
  XGCValues gv;
  char **Fallback = NULL;
  char *app_title = NULL;

#ifdef SND_AS_WIDGET
  state = snd_main(argc,argv);
#else
  XtSetLanguageProc(NULL,NULL,NULL);
#endif

  state->ctrls_height = CLOSED_CTRLS_HEIGHT;
  state->channel_min_height = CHANNEL_MIN_HEIGHT;
  state->Graph_Cursor = XC_crosshair;

#ifdef LESSTIF_VERSION
  Fallback = (char **)CALLOC(2,sizeof(char *));
  Fallback[0] = (char *)CALLOC(64,sizeof(char));
  sprintf(Fallback[0],"*.background: %s",BASIC_COLOR);
  Fallback[1] = (char *)CALLOC(64,sizeof(char));
  sprintf(Fallback[1],"*XmList.background: white");
  /* attempt to make sashes green here died with segfault (*XmSash.background: %s",MIXER_COLOR) */
#endif

#ifndef SND_AS_WIDGET
#if defined(SCO5) || defined(UW2) || defined(SOLARIS) || defined(HPUX) || defined(ALPHA)
  /*
  ** Use of the lower level Xt routines does not work.
  */
  XtSetArg(args[0],XtNwidth,640);
  XtSetArg(args[1],XtNheight,256);
  shell = XtAppInitialize( &app, "Snd", NULL, 0, &argc, argv, NULL, args, 2 );
#else

 #if defined(NEXT)
  XtInitialize(argv[0],"Snd",NULL,0,&argc,argv);
  app = XtCreateApplicationContext();
  n=0;
  XtSetArg(args[n],XmNallowShellResize,TRUE); n++;
  shell = XtCreateApplicationShell("Snd",shellWidgetClass,args,n);
 #else

  shell = XtVaOpenApplication(&app,"Snd",NULL,0,&argc,argv,Fallback,applicationShellWidgetClass,
			      XmNallowShellResize,AUTO_RESIZE_DEFAULT,
  #ifdef LESSTIF_VERSION
			      XmNwidth,250,XmNheight,100,
  #endif
			      NULL);
 #endif
#endif

  /* if user has *keyboardFocusPolicy: Pointer in .Xdefaults, it can screw up Snd's attempt to
   * keep the channel graphics window as the active widget in case of keyboard input.  So,
   * we try to force Snd's focus policy to be XmEXPLICIT
   */
  XtVaGetValues(shell,XmNkeyboardFocusPolicy,&err,NULL);
  if (err != XmEXPLICIT)
    XtVaSetValues(shell,XmNkeyboardFocusPolicy,XmEXPLICIT,NULL);

#else 
  /* SND_AS_WIDGET */
  shell = parent;
#endif

  auto_open_files = argc-1;
  if (argc > 1) auto_open_file_names = (char **)(argv+1);

  dpy=XtDisplay(shell);
  XtGetApplicationResources(shell,&snd_rs,resources,XtNumber(resources),NULL,0);
  XtVaGetValues(shell,XmNtitle,&app_title,NULL);  /* perhaps caller included -title arg */
  if (app_title) state->startup_title = copy_string(app_title); else state->startup_title = copy_string("snd");

  set_sound_style(state,snd_rs.horizontal_panes);
  for (i=1;i<argc;i++)
    {
      if ((strcmp(argv[i],"-h") == 0) || (strcmp(argv[i],"-horizontal") == 0))
	set_sound_style(state,SOUNDS_HORIZONTAL);
      else
	if ((strcmp(argv[i],"-v") == 0) || (strcmp(argv[i],"-vertical") == 0))
	  set_sound_style(state,SOUNDS_VERTICAL);
	else
	  if (strcmp(argv[i],"-notebook") == 0)
	    {
	      set_sound_style(state,SOUNDS_IN_NOTEBOOK);
		snd_rs.auto_resize = 0;
	      }
	  else
	    if (strcmp(argv[i],"-scroller") == 0)
	      {
		set_sound_style(state,SOUNDS_IN_SCROLLER);
		snd_rs.auto_resize = 0;
	      }
	    else
	      if (strcmp(argv[i],"-separate") == 0)
		set_sound_style(state,SOUNDS_IN_SEPARATE_WINDOWS);
	      else
		if (strcmp(argv[i],"-noglob") == 0)
		  noglob = 1;
		else
		  if (strcmp(argv[i],"-noinit") == 0)
		    noinit = 1;
		  else
		    if (strcmp(argv[i],"--version") == 0)
		      {
			fprintf(stdout,version_info());
		      }
		    else
		      if (strcmp(argv[i],"--help") == 0)
			{
			  fprintf(stdout,"Snd is a sound editor.  Execute it and peruse the 'help' menu for details.\n");
			  fprintf(stdout,version_info());
			  snd_exit(0);
			}
    }

#if HAVE_XmHTML
  set_html_width(state,snd_rs.html_width);
  set_html_height(state,snd_rs.html_height);
  set_html_dir(state,snd_rs.html_dir);
  set_html_font_size_list(state,snd_rs.html_font_size_list);
  set_html_fixed_font_size_list(state,snd_rs.html_fixed_font_size_list);
#endif

#ifdef SGI
  state->using_schemes = (strcmp(snd_rs.use_schemes,"all") == 0); /* so *useSchemes: All still gets our color scheme */
#else
  state->using_schemes = 0;
#endif 

  set_auto_resize(state,snd_rs.auto_resize);
  state->zoom_slider_width = snd_rs.zoom_slider_width;
  state->position_slider_width = snd_rs.position_slider_width;
  state->channel_sash_indent = snd_rs.channel_sash_indent;
  state->channel_sash_size = snd_rs.channel_sash_size;
  state->sash_size = snd_rs.sash_size;
  state->sash_indent = snd_rs.sash_indent;
  state->toggle_size = snd_rs.toggle_size;
  state->enved_point_size = snd_rs.enved_point_size;

  state->sgx = (state_context *)CALLOC(1,sizeof(state_context));
  sx = state->sgx;
  sx->dialog_list_size = 0;

#ifndef SND_AS_WIDGET
#ifndef __cplusplus
  XtAppSetWarningMsgHandler(app,muffle_warning);
#endif
#endif

  sx->mainapp = app;
  sx->mainshell = shell;
  sx->mdpy = dpy;

  /* the gray shades are an attempt to get around Netscape which hogs all the colors */
  sx->white = get_color(shell,snd_rs.white_color,WHITE_COLOR,NULL,NULL,TRUE);
  sx->black = get_color(shell,snd_rs.black_color,BLACK_COLOR,NULL,NULL,FALSE);
  sx->light_blue = get_color(shell,snd_rs.light_blue_color,LIGHT_BLUE_COLOR,NULL,NULL,TRUE);
  sx->lighter_blue = get_color(shell,snd_rs.lighter_blue_color,LIGHTER_BLUE_COLOR,NULL,NULL,TRUE);
  sx->red = get_color(shell,snd_rs.red_color,RED_COLOR,NULL,NULL,FALSE);
  sx->green = get_color(shell,snd_rs.green_color,GREEN_COLOR,NULL,NULL,FALSE);
  sx->yellow = get_color(shell,snd_rs.yellow_color,YELLOW_COLOR,NULL,NULL,TRUE);
  sx->highlight_color = get_color(shell,snd_rs.highlight_color,HIGHLIGHT_COLOR,"gray90",NULL,TRUE);
  sx->basic_color = get_color(shell,snd_rs.basic_color,BASIC_COLOR,"gray80","gray",TRUE);
  sx->position_color = get_color(shell,snd_rs.position_color,POSITION_COLOR,"gray60","gray",FALSE);
  sx->zoom_color = get_color(shell,snd_rs.zoom_color,ZOOM_COLOR,"gray20","gray",FALSE);
  sx->cursor_color = get_color(shell,snd_rs.cursor_color,CURSOR_COLOR,NULL,NULL,FALSE);
  sx->selection_color = get_color(shell,snd_rs.selection_color,SELECTION_COLOR,"gray80",NULL,FALSE);
  sx->mix_color = get_color(shell,snd_rs.mix_color,MIX_COLOR,NULL,NULL,FALSE);
  sx->mix_focus_color = get_color(shell,snd_rs.mix_focus_color,MIX_FOCUS_COLOR,NULL,NULL,FALSE);
  sx->mix_waveform_color = get_color(shell,snd_rs.mix_waveform_color,MIX_WAVEFORM_COLOR,NULL,NULL,FALSE);
  sx->enved_waveform_color = get_color(shell,snd_rs.enved_waveform_color,ENVED_WAVEFORM_COLOR,NULL,NULL,FALSE);
  sx->filter_waveform_color = get_color(shell,snd_rs.filter_waveform_color,FILTER_WAVEFORM_COLOR,NULL,NULL,FALSE);
  sx->listener_color = get_color(shell,snd_rs.listener_color,LISTENER_COLOR,NULL,NULL,TRUE);
  sx->graph_color = get_color(shell,snd_rs.graph_color,GRAPH_COLOR,NULL,NULL,TRUE);
  sx->selected_graph_color = get_color(shell,snd_rs.selected_graph_color,SELECTED_GRAPH_COLOR,NULL,NULL,TRUE);
  sx->data_color = get_color(shell,snd_rs.data_color,DATA_COLOR,NULL,NULL,FALSE);
  sx->selected_data_color = get_color(shell,snd_rs.selected_data_color,SELECTED_DATA_COLOR,NULL,NULL,FALSE);
  sx->mark_color = get_color(shell,snd_rs.mark_color,MARK_COLOR,NULL,NULL,FALSE);
  sx->sash_color = get_color(shell,snd_rs.sash_color,SASH_COLOR,NULL,NULL,FALSE);
  sx->pushed_button_color = get_color(shell,snd_rs.pushed_button_color,PUSHED_BUTTON_COLOR,NULL,NULL,FALSE);
  sx->text_focus_color = get_color(shell,snd_rs.text_focus_color,TEXT_FOCUS_COLOR,NULL,NULL,FALSE);

  sx->button_fontstruct = get_font(dpy,snd_rs.button_font,BUTTON_FONT,FALLBACK_FONT);
  sx->button_fontlist = get_xm_font(sx->button_fontstruct,"buttonset","button_font");
  sx->tiny_fontstruct = get_font(dpy,TINY_FONT,FALLBACK_FONT,NULL);
  sx->tiny_fontlist = get_xm_font(sx->tiny_fontstruct,"tinyset","tiny_font");
  sx->listener_fontlist = NULL;
  if (snd_rs.listener_font)
    {
      sx->listener_fontstruct = get_font(dpy,snd_rs.listener_font,NULL,NULL);
      if (sx->listener_fontstruct) sx->listener_fontlist = get_xm_font(sx->listener_fontstruct,"listenerset","listener_font");
    }
  sx->bold_button_fontstruct = get_font(dpy,snd_rs.bold_button_font,BOLD_BUTTON_FONT,FALLBACK_FONT);
  sx->bold_button_fontlist = get_xm_font(sx->bold_button_fontstruct,"boldbuttonset","bold_button_font");
  sx->axis_label_fontstruct = get_font(dpy,snd_rs.axis_label_font,AXIS_LABEL_FONT,FALLBACK_FONT);
  sx->axis_numbers_fontstruct = get_font(dpy,snd_rs.axis_numbers_font,AXIS_NUMBERS_FONT,FALLBACK_FONT);
  sx->help_text_fontstruct = get_font(dpy,snd_rs.help_text_font,HELP_TEXT_FONT,FALLBACK_FONT);
  sx->help_text_fontlist = get_xm_font(sx->help_text_fontstruct,"helptextset","help_text_font");

  if (!(state->using_schemes)) XtVaSetValues(shell,XmNbackground,sx->basic_color,NULL);
  state->init_file = snd_rs.init_file_name; /* doesn't make any sense to pass this out to the user -- what can he do? */
  set_eps_file(state,snd_rs.eps_file_name);
  set_default_output_type(state,snd_rs.def_output_type);
  set_color_map(state,snd_rs.spectrogram_color);
  set_ask_before_overwrite(state,snd_rs.overwrite_check);
  set_mixer_group_max_out_chans(state,snd_rs.group_out_chans);
  set_mixer_groups(state,snd_rs.ngroups);
  if (mixer_groups(state) <= 0) set_mixer_groups(state,1);

#ifndef SND_AS_WIDGET
  n=0;
  if (!(state->using_schemes)) n = background_basic_color(args,n,(snd_state *)state);
#ifdef UW2
  XtSetArg(args[n],XmNforeground,sx->black); n++;
#endif
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNallowResize,TRUE); n++;
  sx->mainpane = sndCreateFormWidget("mainpane",shell,args,n);
  XtAddEventHandler(sx->mainpane,KeyPressMask,FALSE,graph_key_press,(XtPointer)state);
#else
  sx->mainpane = sndCreateFormWidget("mainpane",parent,caller_args,caller_argn);
#endif
  menu = add_menu(state);

  n=0;
  if (!(state->using_schemes)) n = background_basic_color(args,n,(snd_state *)state);
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,menu); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNallowResize,TRUE); n++;
  if (sound_style(state) == SOUNDS_IN_SEPARATE_WINDOWS)
    sx->soundpane = sndCreateFormWidget("soundpane",sx->mainpane,args,n);
  else
    {
      if (sound_style(state) == SOUNDS_IN_SCROLLER)
	{
	  Widget scroller;
	  XtSetArg(args[n],XmNscrollingPolicy,XmAUTOMATIC); n++;
	  XtSetArg(args[n],XmNscrollBarDisplayPolicy,XmSTATIC); n++;
	  scroller = XmCreateScrolledWindow(sx->mainpane,"sb",args,n);
	  
	  n=0;
	  if (!(state->using_schemes)) n = background_basic_color(args,n,(snd_state *)state);
	  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
	  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
	  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
	  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
	  XtSetArg(args[n],XmNsashHeight,state->sash_size); n++;
	  XtSetArg(args[n],XmNsashWidth,state->sash_size); n++;
	  XtSetArg(args[n],XmNallowResize,TRUE); n++;
	  XtSetArg(args[n],XmNsashIndent,state->sash_indent); n++;
	  sx->soundpane = sndCreatePanedWindowWidget("soundpane",scroller,args,n);
	  XtVaSetValues(scroller,XmNworkWindow,sx->soundpane,NULL);

	  state->init_window_width = 500; /* need a first guess to get things rolling */
	  state->init_window_height = 400;

	  XtManageChild(scroller);
	}
      else
	{
#if (XmVERSION > 1)
	  if (sound_style(state) != SOUNDS_IN_NOTEBOOK)
	    {
#endif
	      XtSetArg(args[n],XmNsashHeight,state->sash_size); n++;
	      XtSetArg(args[n],XmNsashWidth,state->sash_size); n++;
	      if (sound_style(state) == SOUNDS_HORIZONTAL) {XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;}
	      XtSetArg(args[n],XmNsashIndent,state->sash_indent); n++;
	      sx->soundpane = sndCreatePanedWindowWidget("soundpane",sx->mainpane,args,n);
#if (XmVERSION > 1)
	    }
	  else
	    {
	      XtSetArg(args[n],XmNframeBackground,sx->zoom_color); n++;
	      XtSetArg(args[n],XmNbindingWidth,NOTEBOOK_BINDING_WIDTH); n++;
	      sx->soundpane = XtCreateManagedWidget("nb", xmNotebookWidgetClass, sx->mainpane, args,n);
	      map_over_children(sx->soundpane,set_main_color_of_widget,(void *)state); /* appears to be a no-op */
	    }
#endif
	}
      /* might want a second layer here so that the listener is underneath the horizontally placed sounds */
      XtAddEventHandler(sx->soundpane,KeyPressMask,FALSE,graph_key_press,(XtPointer)state);
    }

#ifndef SND_AS_WIDGET
  #if ICON_TYPE
    SetupIcon(shell);
  #endif

  XtRealizeWidget(shell);
  if (auto_resize(state) != AUTO_RESIZE_DEFAULT) XtVaSetValues(shell,XmNallowShellResize,auto_resize(state),NULL);
#endif

  wn = XtWindow(shell);
  gv.background = sx->graph_color;
  gv.foreground = sx->data_color;
  sx->basic_gc = XCreateGC(dpy,wn, GCForeground | GCBackground, &gv);

  gv.background = sx->graph_color;
  gv.foreground = sx->data_color;
  sx->combined_basic_gc = XCreateGC(dpy,wn, GCForeground | GCBackground, &gv);

  gv.background = sx->graph_color;
  gv.foreground = sx->mix_waveform_color;
  sx->mix_gc = XCreateGC(dpy,wn, GCForeground | GCBackground, &gv);

  gv.function = GXxor;
  gv.background = sx->graph_color;
  gv.foreground = (Pixel)(XOR(sx->cursor_color,gv.background));
  sx->cursor_gc = XCreateGC(dpy,wn,GCForeground | GCFunction, &gv);

  gv.function = GXxor;
  gv.background = sx->graph_color;
  gv.foreground = (Pixel)(XOR(sx->selection_color,gv.background));
  sx->selection_gc = XCreateGC(dpy,wn,GCForeground | GCFunction, &gv);

  gv.function = GXxor;
  gv.background = sx->graph_color;
  gv.foreground = (Pixel)(XOR(sx->mark_color,gv.background));
  sx->mark_gc = XCreateGC(dpy,wn,GCForeground | GCFunction, &gv);

  gv.function = GXcopy;
  gv.background = sx->data_color;
  gv.foreground = sx->graph_color;
  sx->erase_gc = XCreateGC(dpy,wn, GCForeground | GCBackground | GCFunction, &gv);

  gv.background = sx->selected_graph_color;
  gv.foreground = sx->selected_data_color;
  sx->selected_basic_gc = XCreateGC(dpy,wn, GCForeground | GCBackground, &gv);

  gv.function = GXxor;
  gv.background = sx->selected_graph_color;
  gv.foreground = (Pixel)(XOR(sx->cursor_color,gv.background));
  sx->selected_cursor_gc = XCreateGC(dpy,wn,GCForeground | GCFunction, &gv);

  gv.function = GXxor;
  gv.background = sx->selected_graph_color;
  gv.foreground = (Pixel)(XOR(sx->selection_color,gv.background));
  sx->selected_selection_gc = XCreateGC(dpy,wn,GCForeground | GCFunction, &gv);

  gv.function = GXxor;
  gv.background = sx->selected_graph_color;
  gv.foreground = (Pixel)(XOR(sx->mark_color,gv.background));
  sx->selected_mark_gc = XCreateGC(dpy,wn,GCForeground | GCFunction, &gv);

  gv.function = GXcopy;
  gv.background = sx->selected_data_color;
  gv.foreground = sx->selected_graph_color;
  sx->selected_erase_gc = XCreateGC(dpy,wn, GCForeground | GCBackground | GCFunction, &gv);

  gv.function = GXcopy;
  gv.background = sx->basic_color;
  gv.foreground = sx->black;
  sx->fltenv_basic_gc = XCreateGC(dpy,wn, GCForeground | GCFunction, &gv);

  gv.function = GXcopy;
  gv.background = sx->basic_color;
  gv.foreground = sx->filter_waveform_color;
  sx->fltenv_data_gc = XCreateGC(dpy,wn, GCBackground | GCForeground | GCFunction, &gv);

  gv.function = GXcopy;
  gv.background = sx->basic_color;
  gv.foreground = sx->black;
  sx->speed_gc = XCreateGC(dpy,wn,GCForeground | GCBackground,&gv);

  XtAppAddWorkProc(app,startup_funcs,make_task_manager(state,shell,dpy));

#if TRAP_SEGFAULT
  sigsetjmp(envHandleEventsLoop,1);
#endif

#ifndef SND_AS_WIDGET
  XtAppMainLoop(app);
#endif
}

void reflect_resize(snd_state *ss)
{
  XtVaSetValues(main_SHELL(ss),XmNallowShellResize,auto_resize(ss),NULL);
}


/* ---------------- HELP MONOLOG ---------------- */

static Widget help_dialog = NULL;
static Widget help_text = NULL;
static char help_window_label[64];

#if HAVE_XmHTML

#include <XmHTML/XmHTML.h>
/* CFLAGS = -g -Wall -DLINUX -DUSR_LIB_OSS=1 -DHAVE_GUILE -DHAVE_XmHTML=1 -I/home/bil/test/XmHTML-1.1.4/include */
/* LIBS = /home/bil/test/XmHTML-1.1.4/src/libXmHTML.a -L/usr/X11R6/lib -lMrm -lXp -lXm -lXpm -lXmu -lXt -lXext -lX11 /usr/local/lib/libguile.a -lm -ldl */

static char *Load_HTML_File(char *filename)
{ /* from XmHTML/examples/example_1.c */
  FILE *file;
  int size;
  char *content;
  if ((file = fopen(filename, "r")) == NULL)
    {
      perror(filename);
      return(NULL);
    }
  fseek(file, 0, SEEK_END);
  size = ftell(file);
  rewind(file);
  content = (char *)CALLOC(size,sizeof(char));
  if (content == NULL) return(NULL);
  if ((fread(content, 1, size, file)) != size)
    snd_error("%s[%d] %s: did not read entire file!\n",__FILE__,__LINE__,__FUNCTION__);
  fclose(file);
  return(content);
}

static char *snd_html = NULL;
static int snd_html_loaded = 0;
static char *extsnd_html = NULL;
static int extsnd_html_loaded = 0;

static void load_snd_html(snd_state *ss, char *anchor)
{
  char *buf,*temp = NULL;
  if (anchor) temp = copy_string(anchor);
  if (snd_html == NULL) 
    {
      if (html_dir(ss) && (*(html_dir(ss))))
	{
	  buf = (char *)CALLOC(256,sizeof(char));
	  sprintf(buf,"%s/snd.html",html_dir(ss));
	  snd_html = Load_HTML_File(buf);
	  FREE(buf);
	}
      else snd_html = Load_HTML_File("snd.html");
    }
  if (snd_html_loaded == 0) 
    {
      XmHTMLTextSetString(help_text,snd_html);
      snd_html_loaded = 1;
      extsnd_html_loaded = 0;
    }
  if (temp)
    {
      XmHTMLAnchorScrollToName(help_text,temp);
      FREE(temp);
    }
}

static void load_extsnd_html(snd_state *ss, char *anchor)
{
  char *buf,*temp = NULL;
  if (anchor) temp = copy_string(anchor);
  if (extsnd_html == NULL) 
    {
      if (html_dir(ss) && (*(html_dir(ss))))
	{
	  buf = (char *)CALLOC(256,sizeof(char));
	  sprintf(buf,"%s/extsnd.html",html_dir(ss));
	  extsnd_html = Load_HTML_File(buf);
	  FREE(buf);
	}
      else extsnd_html = Load_HTML_File("extsnd.html");
    }
  if (extsnd_html_loaded == 0) 
    {
      XmHTMLTextSetString(help_text,extsnd_html);
      extsnd_html_loaded = 1;
      snd_html_loaded = 0;
    }
  if (temp) 
    {
      XmHTMLAnchorScrollToName(help_text,temp);
      FREE(temp);
    }
}

static void anchorCB(Widget widget, XtPointer client_data, XmHTMLAnchorCallbackStruct *cbs)
{
  if (cbs->reason != XmCR_ACTIVATE) return;
  if (cbs->url_type == ANCHOR_FILE_LOCAL)
    {
      /* must be either ext->snd or snd->ext or something we can't handle in this context */
      /* all contents of cbs struct are pointers into current text, so we have to save them before changing that text */
      if (strncmp(cbs->href,"extsnd",6) == 0)
	load_extsnd_html(get_global_state(),(char *)(cbs->href+11)); /* "extsnd.html#" */
      else 
	if (strncmp(cbs->href,"snd",3) == 0)
	  load_snd_html(get_global_state(),(char *)(cbs->href+8));
    }
  else
    {
      cbs->doit = True;
      cbs->visited = True;
    }
}

#endif

static void help_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Help",
"You can get help within Snd either from\n\
the Help Menu items, or by clicking on\n\
some portion of the Snd display while the\n\
cursor is '?'.  See Click for Help in the\n\
Help Menu.\n\
");
}

static void create_help_monolog(snd_state *ss)
{
  /* create scrollable but not editable text window */
  Arg args[20];
  int n;
  XmString titlestr;
#if HAVE_XmHTML
  Widget ww;
#endif

  titlestr = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
  /* this window should be resizable by the user (i.e. have the resize bars), but not resize itself */
  XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
#if RESIZE_DIALOG
  XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
  XtSetArg(args[n],XmNtransient,FALSE); n++;
  help_dialog = XmCreateMessageDialog(main_PANE(ss),"snd-help",args,n);
  add_dialog(ss,help_dialog);
#if OVERRIDE_TOGGLE
  override_form_translation(help_dialog);
#endif

  XtUnmanageChild(XmMessageBoxGetChild(help_dialog,XmDIALOG_SYMBOL_LABEL));
  XtUnmanageChild(XmMessageBoxGetChild(help_dialog,XmDIALOG_CANCEL_BUTTON));
  XtAddCallback(help_dialog,XmNhelpCallback,help_help_callback,ss);
  XmStringFree(titlestr);

  n=0;
#if HAVE_XmHTML
  XtSetArg(args[n],XmNwidth,html_width(ss)); n++;
  XtSetArg(args[n],XmNheight,html_height(ss)); n++;
  XtSetArg(args[n],XmNfontSizeList,html_font_size_list(ss)); n++;
  XtSetArg(args[n],XmNfontSizeFixedList,html_fixed_font_size_list(ss)); n++;
  help_text = XtCreateManagedWidget("html",xmHTMLWidgetClass,help_dialog,args,n);
  XtAddCallback(help_text,XmNactivateCallback,(XtCallbackProc)anchorCB, NULL);
#else
  XtSetArg(args[n],XmNeditMode,XmMULTI_LINE_EDIT); n++;
  XtSetArg(args[n],XmNeditable,FALSE); n++;
  XtSetArg(args[n],XmNcolumns,HELP_COLUMNS); n++;
  XtSetArg(args[n],XmNrows,HELP_ROWS); n++;
  XtSetArg(args[n],XmNfontList,help_text_FONT(ss)); n++;
  if (!(ss->using_schemes))
    {
      XtSetArg(args[n],XmNforeground,(ss->sgx)->black); n++; /* needed if color allocation fails completely */
      XtSetArg(args[n],XmNbackground,(ss->sgx)->white); n++;
    }
  help_text = XmCreateScrolledText(help_dialog,"help-text",args,n);
  XtManageChild(help_text);
#endif

#if MANAGE_DIALOG
  XtManageChild(help_dialog);
#endif

  if (!(ss->using_schemes))
    {
      map_over_children(help_dialog,set_main_color_of_widget,(void *)ss);
      XtVaSetValues(help_text,XmNbackground,(ss->sgx)->white,NULL);
      XtVaSetValues(XmMessageBoxGetChild(help_dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XmMessageBoxGetChild(help_dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
#if HAVE_XmHTML
      XtVaGetValues(help_text,XmNworkWindow,&ww,NULL);
      XtVaSetValues(ww,XmNbackground,(ss->sgx)->white,NULL);
#endif
    }
}

void snd_help(snd_state *ss, char *subject, char *helpstr)
{
  /* place help string in scrollable help window */
  /* if window is already active, add this help at the top and reposition */
  XmString xstr1;
#if HAVE_XmHTML
  char *newhelp;
#endif
  if (!(help_dialog)) create_help_monolog(ss); else raise_dialog(help_dialog);
  sprintf(help_window_label,"%s help",subject);
  xstr1 = XmStringCreate(help_window_label,XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(help_dialog,XmNmessageString,xstr1,NULL);
#if HAVE_XmHTML
  if (helpstr[0] == '#')
    load_snd_html(ss,helpstr);
  else
    {
      if (strcmp(helpstr,"extsnd.html") == 0)
	load_extsnd_html(ss,NULL);
      else
	{
	  newhelp = (char *)CALLOC(strlen(helpstr) + 64,sizeof(char));
	  sprintf(newhelp,"<html><body><pre>%s</pre></body></html>",helpstr);
	  XmHTMLTextSetString(help_text,newhelp);
	  snd_html_loaded = 0;
	  FREE(newhelp);
	}
    }
#else
  XmTextSetString(help_text,helpstr);
#endif
  if (!XtIsManaged(help_dialog)) XtManageChild(help_dialog);
  XmStringFree(xstr1);
}

void ssnd_help(snd_state *ss, char *subject, ...)
{
  va_list ap;
  char *helpstr,*newstr;
  int len,size;
  va_start(ap,subject);
  size = 1024;
  newstr = (char *)CALLOC(size,sizeof(char));
  len = 0;
  while ((helpstr = va_arg(ap,char *)))
    {
      len += strlen(helpstr);
      if (len >= size)
	{
	  size = len+1024;
	  newstr = (char *)REALLOC(newstr,size * sizeof(char));
	}
      strcat(newstr,helpstr);
    }
  va_end(ap);
  snd_help(ss,subject,newstr);
  FREE(newstr);
}  

Widget help_shell(snd_state *ss)
{
  return(help_dialog);
}

int snd_window_width(snd_state *ss)
{
  Dimension width;
  XtVaGetValues(main_SHELL(ss),XmNwidth,&width,NULL);
  return(width);
}

int snd_window_height(snd_state *ss)
{
  Dimension height;
  XtVaGetValues(main_SHELL(ss),XmNheight,&height,NULL);
  return(height);
}

void set_snd_window_width(snd_state *ss,int width)
{
  XtVaSetValues(main_SHELL(ss),XmNwidth,(Dimension)width,NULL);
  ss->init_window_width = width;
}

void set_snd_window_height(snd_state *ss,int height)
{
  XtVaSetValues(main_SHELL(ss),XmNheight,(Dimension)height,NULL);
  ss->init_window_height = height;
  normalize_all_sounds(ss);
}

void set_snd_window_y(snd_state *ss, int y)
{
  XtVaSetValues(main_SHELL(ss),XmNy,y,NULL);
  ss->init_window_y = y;
}

int snd_window_y(snd_state *ss)
{
  Dimension y;
  XtVaGetValues(main_SHELL(ss),XmNy,&y,NULL);
  return((int)y);
}

void set_snd_window_x(snd_state *ss, int x)
{
  XtVaSetValues(main_SHELL(ss),XmNx,x,NULL);
  ss->init_window_x = x;
}

int snd_window_x(snd_state *ss)
{
  Dimension x;
  XtVaGetValues(main_SHELL(ss),XmNx,&x,NULL);
  return((int)x);
}

void sound_show_ctrls(snd_info *sp)
{
  snd_state *ss;
  ss = sp->state;
  XtUnmanageChild(w_snd_ctrls(sp));
  XtVaSetValues(w_snd_ctrls(sp),XmNpaneMinimum,ss->open_ctrls_height,XmNpaneMaximum,ss->open_ctrls_height,NULL);
  XtManageChild(w_snd_ctrls(sp));
  XtVaSetValues(w_snd_ctrls(sp),XmNpaneMinimum,1,XmNpaneMaximum,1000,NULL);
}

void sound_hide_ctrls(snd_info *sp)
{
  XtUnmanageChild(w_snd_ctrls(sp));
  XtVaSetValues(w_snd_ctrls(sp),XmNpaneMaximum,CLOSED_CTRLS_HEIGHT,XmNpaneMinimum,CLOSED_CTRLS_HEIGHT,NULL);
  XtManageChild(w_snd_ctrls(sp));
  XtVaSetValues(w_snd_ctrls(sp),XmNpaneMinimum,1,XmNpaneMaximum,1000,NULL);
}

int control_panel_open(snd_info *sp)
{
  Dimension hgt;
  XtVaGetValues(w_snd_ctrls(sp),XmNheight,&hgt,NULL);
  return(hgt > CLOSED_CTRLS_HEIGHT);
}

void show_controls(snd_state *ss)
{
  snd_info *sp;
  int i;
  ss->ctrls_height = ss->open_ctrls_height;
  set_view_ctrls_label(STR_Hide_controls);
  for (i=0;i<ss->max_sounds;i++)
    {
      sp = ss->sounds[i];
      if ((sp) && (sp->inuse)) sound_show_ctrls(sp);
    }
}

void hide_controls(snd_state *ss)
{
  snd_info *sp;
  int i;
  ss->ctrls_height = CLOSED_CTRLS_HEIGHT;
  set_view_ctrls_label(STR_Show_controls);
  for (i=0;i<ss->max_sounds;i++)
    {
      sp = ss->sounds[i];
      if ((sp) && (sp->inuse)) sound_hide_ctrls(sp);
    }
}

#if (ICON_TYPE == 1) || ((ICON_TYPE == 2) && (!HAVE_XPM))
/* use plain 48x48 bitmap */
#define snd_width 48
#define snd_height 48
static unsigned char snd_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0xf0, 0x01, 0x00,
   0x32, 0x00, 0x00, 0x40, 0x00, 0x00, 0x32, 0x0c, 0x00, 0x40, 0x00, 0x00,
   0x32, 0x0f, 0x10, 0x40, 0x00, 0x00, 0xb2, 0x19, 0x38, 0x40, 0x00, 0x00,
   0xb0, 0x10, 0x68, 0x40, 0x00, 0x00, 0xb0, 0x30, 0x88, 0x40, 0x00, 0x00,
   0xb0, 0x20, 0x88, 0x40, 0x00, 0x00, 0xf0, 0xe0, 0x89, 0x41, 0x00, 0x00,
   0x30, 0x00, 0x09, 0x43, 0x3c, 0x00, 0x30, 0x00, 0x0f, 0x4e, 0xe6, 0x00,
   0x30, 0x00, 0x00, 0xf8, 0x83, 0x7f, 0x30, 0x00, 0x00, 0x40, 0x00, 0xf8,
   0x30, 0x00, 0x00, 0x40, 0x00, 0x0e, 0x30, 0x00, 0x07, 0xe0, 0x01, 0x03,
   0x30, 0x80, 0x09, 0x50, 0x03, 0x01, 0x70, 0xc0, 0x08, 0x58, 0x82, 0x01,
   0xb0, 0x60, 0x18, 0x44, 0xc6, 0x00, 0xb0, 0x31, 0x10, 0x44, 0x7c, 0x00,
   0x30, 0x13, 0x10, 0x44, 0x00, 0x00, 0x30, 0x1e, 0x30, 0x44, 0x00, 0x00,
   0x30, 0x00, 0x60, 0x42, 0x00, 0x00, 0x30, 0x00, 0xc0, 0xc3, 0x00, 0x00,
   0x37, 0x00, 0x00, 0xc0, 0x01, 0x00, 0x35, 0x00, 0x00, 0xc0, 0x03, 0x00,
   0x35, 0x00, 0x00, 0xc0, 0x01, 0x00, 0x35, 0x00, 0x00, 0xc0, 0x00, 0x00,
   0xf7, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x7f,
   0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x70, 0x77, 0x17, 0x00, 0xde, 0x03,
   0x50, 0x55, 0x11, 0x00, 0xd6, 0x02, 0x50, 0x57, 0x13, 0x00, 0xd6, 0x02,
   0x50, 0x55, 0x11, 0x00, 0xd6, 0x02, 0x70, 0x77, 0x17, 0x00, 0xde, 0x03,
   0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x77,
   0x08, 0x00, 0x80, 0x0d, 0x00, 0x54, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x77,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf8, 0xff, 0xff, 0xff, 0xff, 0x77, 0x88, 0x0d, 0x00, 0x00, 0x00, 0x54,
   0xf8, 0xff, 0xff, 0xff, 0xff, 0x77, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static void SetupIcon(Widget shell)
{
  Display *dpy;
  Window root;
  Pixmap bitmap;
  dpy = XtDisplay (shell);
  root = DefaultRootWindow(dpy);
  bitmap = XCreateBitmapFromData(dpy,root,snd_bits,snd_width,snd_height);
  XtVaSetValues(shell,XmNiconPixmap,bitmap,NULL);
}
#endif

#if (ICON_TYPE == 2) && (HAVE_XPM)
/* use XPM-style 48x48 bitmap */
#include <X11/xpm.h>
static char *snd_bits[] = {
"48 48 9 1",
". c white m white",
"B c black m black",
"l c ivory1 m white s lightestcolor",
"a c ivory2 m white s basiccolor",
"d c ivory3 m black s darkcolor",
"X c ivory4 m black s darkestcolor",
"b c lightsteelblue1 m white s textcolor",
"r c red m black s cursorcolor",
"g c lightgreen m black s mixercolor",
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
"aaXXda.........................................a",
"aaXXda..BB........................rrrrrrr......a",
"aaXXda..BB...........................r.........a",
"aaXXda..BB...........................r.........a",
"aaXXda..BB...........................r.........a",
"aaXXda..BB...........................r.........a",
"aaXXda..BB........BBB................r.........a",
"aaXXda..BB......B.....B..............r.........a",
"aadBda..BB....B........B.............r.........a",
"aalBda..BB..B...........B............r.........a",
"aalBda..BB.B.............B...........r.........a",
"aalBda..BBB..............B...........r.........a",
"aaXXda..BB...............B...........r.........a",
"aaXXda..BB................B..........r.........a",
"aaXXda..BB................B..........r.........a",
"aaXXda..BB.................B.........r.........a",
"aaXXda..BB.................B.........r........Ba",
"aaXXda..BB..................B........r.......B.a",
"aaXXda..BB...................B.......r......B..a",
"aaXXda..BB.....................B.....r.....B...a",
"aaXXda..BB......................B....r...B.....a",
"aaXXda..BB.......................B...r.B.......a",
"aaXXda..BB.........................BBr.........a",
"aaXXda..BB...........................rr........a",
"aaXXda..BB...........................rrr.......a",
"aaaaaa..BB...........................rrrr......a",
"aaaaaa..BB...........................rrr.......a",
"aa.lda..BB...........................rr........a",
"aa.lda..BB...........................r.........a",
"aa.lda..BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB.a",
"aaddda..BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB.a",
"aaaaaa.........................................a",
"aaaaaa.........................................a",
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
"aBbbaaaddddd....ddddddddddddddddddddddddddddddda",
"adbbaaadddddllllddddddddddddddddddddddddddddddda",
"adbbaaadddddllllddddddddddddddddddddddddddddddda",
"adbbaaaXXXXXXXXXXXXXXXXXXXXddddXXXXXXXXXXXXXXXXa",
"aaaaaaaXXXXXXXXXXXXXXXXXXXXddddXXXXXXXXXXXXXXXXa",
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaggg",
"aaaaBBBlBBBlBBBlBBBlBaaaaaaaaaaaaaaa....a....ggg",
"aaaaBlBlBlBlBlBlBlllBaaaaaaaaaaaaaaallldallldggg",
"aaaaBlBlBBBlBlBlBBllBaaaaaaaaaaaaaaallldallldaaa",
"aaaaBlBlBlBlBlBlBlllBaaaaaaaaaaaaaaaddddaddddaaa",
"aaaaBBBlBBBlBBBlBBBlBaaaaaaaaaaaaaaaaaaaaaaaaaaa",
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"};

static void SetupIcon(Widget shell)
{
  Display *dpy;
  Window root;
  int status,scr;
  Pixmap pix,mask;
  XpmAttributes attributes;
  dpy = XtDisplay(shell);
  root = DefaultRootWindow(dpy);
  scr = DefaultScreen(dpy);
  XtVaGetValues(shell,XmNdepth,&attributes.depth,XmNcolormap,&attributes.colormap,NULL);
  attributes.visual = DefaultVisual(dpy,scr);
  attributes.valuemask = XpmDepth | XpmColormap | XpmVisual;
  status = XpmCreatePixmapFromData(dpy,root,snd_bits,&pix,&mask,&attributes);
  if (mask) XFreePixmap(dpy,mask);
  XtVaSetValues(shell,XmNiconPixmap,pix,NULL);
}
#endif
