/*
 * CDTOOL is a program to play and catalog audio compact discs on
 * a cdrom drive
 *
 * Copyright 1994 Thomas Insel.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 *
 * For information on CDTOOL, alternate licensing, or whatever, 
 * contact Thomas Insel via email (tinsel@uiuc.edu) or write to
 * Thomas Insel, 210 Parkview Drive, Bloomington, IL 61701-2038.
 * 
 * Multiple cdrom drive support
 * cdinfo command
 * skip track option
 * some minor enhanchments
 * and some bugs fixed
 * by
 * Sven Oliver Moll
 * 
 * Copyright 1995,96 Sven Oliver Moll
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#ifdef sun
#  include <sundev/srreg.h>
#elif defined linux
#  include <linux/cdrom.h>
#else
#  errror Please fix includes for your system in hardware.c
#endif

#include "config.h"
#include "commands.h"
#include "shuffle.h"
#include "hardware.h"
#include "info.h"
#include "main.h"

static char *rcsid = "$Id: main.c,v 1.6 1998/07/15 14:00:28 wadeh Exp wadeh $";
void do_usage(char *progname);

/* nasty -- made these globals... */
char            *cd_device;

/************************************************************************/
/* Procedure:  main
 * Purpose:    C code main function
 * 
 * Inputs:     command line args
 * Outputs:    to stdout
 * Returns:    error code (0 if exit is OK)
 * Notes:  
 *   1.
 */
/************************************************************************/
int main(int argc, char *argv[]) 
{
  int		info=I_AUDIOSTATUS, error=0, opt, cdfile, trk0=0, trk1=0;
  extern int	optind, getopt();
  extern char	*optarg;

  char		*progname;
  int           iVerbose = FALSE;

  int  usedb = 1;
  char p_format = P_LONG;
  int  iRepeat = 1;
  int  iTracks = 100;
  int  iCDNum = 0;

  if ( (progname=rindex(argv[0],'/')) != 0)
    progname++;
  else
    progname = argv[0];

  if ( (cd_device=getenv("CDTOOLDEV")) == NULL) 
    cd_device = strdup(CD_DEVICE);

  while((opt = getopt(argc,argv,"0123456789?ad:hi:nrstvVp:R:T:")) != EOF)
    {
      switch(opt)
	{    
	 case '0':
	 case '1':
	 case '2':
	 case '3':
	 case '4':
	 case '5':
	 case '6':
	 case '7':
	 case '8':
	 case '9':
	  cd_device=strdup(CD_DEVICE_BASE "?");
	  *(cd_device+sizeof(CD_DEVICE_BASE)-1)=opt;
	  iCDNum = opt - '0';
	  break;
	 case 'a':
	  error = 1;
	  if ( strcmp(progname, "cdinfo") == 0 )
	    {
	      error = 0;
	      info = I_DISP_ABS;
	    }
	  break;
	 case 'd':
	  cd_device=optarg;
	  break;
	 case 'n':
	  error = 1;
	  if ( strcmp(progname, "cdir") == 0 )
	    {
	      error = 0;
	      usedb = 0;
	    }
	  break;
	 case 'r':
	  error = 1;
	  if ( strcmp(progname, "cdir") == 0 )
	    {
	      error = 0;
	      p_format = P_RAW; 
	    }
	  if ( strcmp(progname, "cdinfo") == 0 )
	    {
	      error = 0;
	      info = I_DISP_REL;
	    }
	  break;
	 case 's':
	  error = 1;
	  if ( strcmp(progname, "cdir") == 0 )
	    {
	      error = 0;
	      p_format = P_QUICK;
	    }
	  if ( strcmp(progname, "cdinfo") == 0 )
	    {
	      error = 0;
	      info = I_AUDIOSTATUS;
	    }
	  break;
	 case 't':
	  error = 1;
	  if ( strcmp(progname, "cdir") == 0 )
	    {
	      error = 0;
	      p_format = P_TEMPL;
	    }
	  if ( strcmp(progname, "cdinfo") == 0 )
	    {
	      error = 0;
	      info = I_TRACK;
	    }
	  break;

	  /* verbose info printout */
	 case 'v':
	  iVerbose = TRUE;
	  error = 1;
	  if (( strcmp(progname, "cdshuffle") == 0 ) ||
	      ( strcmp(progname, "cdplay") == 0 ) ||
	      ( strcmp(progname, "cdstart") == 0 )) {
	      error = 0;
	  }
	  else if ( strcmp(progname, "cdir") == 0 ) {
	      error = 0;
	      usedb = 1;
	  }
	  else if ( strcmp(progname, "cdinfo") == 0 ) {
	      error = 0;
	      info = I_DISP_ALL;
	  }
	  break;

	  /* for debug, overload the program name, e.g., */
	  /* cdtool -p info is the same as cdinfo */
	case 'p':
	  progname = optarg;
	  error = 0;
	  break;

	case 'R':
	  if (( strcmp(progname, "cdshuffle") == 0 ) ||
	      ( strcmp(progname, "cdstart") == 0 ) ||
	      ( strcmp(progname, "cdplay") == 0 )) {
	      error = 0;
	      iRepeat = atoi(optarg);
	  }
	  break;

	case 'T':
	  if (( strcmp(progname, "cdshuffle") == 0 ) ||
	      ( strcmp(progname, "cdstart") == 0 ) ||
	      ( strcmp(progname, "cdplay") == 0 )) {
	      error = 0;
	      iTracks = atoi(optarg);
	  }
	  break;
	  
	 /* print verison number and then standard HELP */
	 case 'V':
	     printf ("%s\n  ID=%s\n", progname, rcsid);
	     printf ("  Build:  %s %s\n",__DATE__,__TIME__);

	  /* general options */
	 case '?':
	 case 'h':
	 default:
	  error = 1;
	  break;
	}
    }

  if (error)
    {
      do_usage(progname);
      exit(0);
    }

  if ( (cdfile=open(cd_device, O_RDONLY)) == -1 ) 
    {
      cdhw_t hw;
      int    iResult;
      fprintf(stderr, "%s: can't open cdrom (%s)\n", progname, cd_device);
      iResult = cd_status(progname, -1, cd_device, &hw);
      fprintf (stderr, "%s:  CD status is %s\n", progname, cd_status_text(iResult));
      exit(1);
    }

#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", device=%s\n",
	 progname, cd_device);
#endif

  if (( strcmp(progname, "cdplay") == 0 ) ||
      ( strcmp(progname, "cdstart") == 0 ))
    {
	if ( optind < argc ) {
	    trk0=atoi(argv[optind++]);
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", trk0=%d\n",
	 progname, trk0);
#endif
	}
	if ( optind < argc ) {
	    trk1=atoi(argv[optind]);
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", trk1=%d\n",
	 progname, trk0);
#endif
	}
	optind--;
	if (strcmp(argv[optind],"-") == 0 ) {
	    trk0=SKIPBACK;
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", trk0=SKIPBACK(%d)\n",
	 progname, trk0);
#endif
	}
	if (strcmp(argv[optind],"+") == 0 ) {
	    trk0=SKIPFORWARD;
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", trk0=SKIPFORWARD(%d)\n",
	 progname, trk0);
#endif
	}
	if (strcmp(argv[optind],"?") == 0 ) {
	    do_usage(progname);
	    exit(0);
	}
	if ((strcmp(argv[optind],"start") == 0) ||
	     (strcmp(argv[optind],"x") == 0 ) ||
	     (strcmp(argv[optind],"0") == 0 )) {
	    trk0=SKIPSTART;
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", trk0=SKIPSTART(%d)\n",
	 progname, trk0);
#endif
        }

	if (strcmp(argv[optind],"stop") == 0) {
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", stop\n",
	 progname);
#endif
            do_stop(progname, cdfile, iCDNum, iVerbose);
	    close(cdfile);
	    exit(0);
        }
	if (strcmp(argv[optind],"pause") == 0) {
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", pause\n",
	 progname);
#endif
            do_pause(progname, cdfile, iVerbose);
	    close(cdfile);
	    exit(0);
        }
	if (strcmp(argv[optind],"eject") == 0) {
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", eject\n",
	 progname);
#endif
            do_eject(progname, cdfile, iVerbose);
	    close(cdfile);
	    exit(0);
        }
	if (strcmp(argv[optind],"info") == 0) {
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", info\n",
	 progname);
#endif
            do_info(progname, cdfile, I_DISP_ALL, cd_device, DONADA);
	    fputc ('\n', stdout);
	    close(cdfile);
	    exit(0);
        }

	if (strcmp(argv[optind],"dir") == 0) {
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", dir\n",
	 progname);
#endif
            do_dir(progname, cdfile,P_RAW,0, cd_device, DOLF);
	    close(cdfile);
	    exit(0);
        }
	if (strcmp(argv[optind],"resume") == 0) {
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", resume\n",
	 progname);
#endif
	    do_play(progname, cdfile, 0, 0, iVerbose);
        }

	if (strcmp(argv[optind],"shuffle") == 0) {
#ifdef DEBUG
fprintf (stderr,"main:     called as \"%s\", shuffle\n",
	 progname);
#endif
             do_shuffle(progname, cdfile, iCDNum, iTracks, 
			iRepeat, iVerbose, cd_device, DOLF);
        }

	/* block to play a CD if it is an audio cd */
	{
	    cdhw_t *hw;
	    int    iTemp;
	    hw = check_disc (progname, cdfile, DOLF, &iTemp, cd_device);
	    free_hw_buf(hw);
	    if (iTemp == TRUE)
		do_play(progname, cdfile, trk0, trk1, iVerbose);
	}
    }

  else if ( strcmp(progname, "cdstop") == 0 )
    do_stop(progname, cdfile, iCDNum, iVerbose);
  else if ( strcmp(progname, "cdeject") == 0 )
    do_eject(progname, cdfile, iVerbose);
  else if ( strcmp(progname, "cdreset") == 0 )
    do_reset(progname, cdfile, iVerbose);
  else if ( strcmp(progname, "cdpause") == 0 )
    do_pause(progname, cdfile, iVerbose);
  else if ( strcmp(progname, "cdshuffle") == 0 )
    do_shuffle(progname, cdfile, iCDNum, iTracks, iRepeat, iVerbose, 
	       cd_device, DOLF);
  else if ( strcmp(progname, "cdinfo") == 0 ) {
    do_info(progname, cdfile, info, cd_device, DONADA);
    fputc ('\n', stdout);
  }
  else if ( strcmp(progname, "cdir") == 0 )
    do_dir(progname, cdfile, p_format, usedb, cd_device, DOLF);
  else do_usage(progname);

  close(cdfile);

  exit(0);
}

/************************************************************************/
/* Procedure:  do_usage
 * Purpose:    to print usage info (help)
 * 
 * Inputs:     program name
 * Outputs:    to stderr
 * Returns:    n/a
 * Notes:  
 *   1.
 */
/************************************************************************/
void do_usage(char *progname)
{
  fprintf(stderr,VERSION_STRING); fprintf(stderr,"\n");
  
  if (( strcmp(progname, "cdplay") == 0 ) ||
      ( strcmp(progname, "cdstart") == 0 )) {

    if ( strcmp(progname, "cdplay") == 0 )
      fprintf(stderr, "usage: cdplay [start [stop]]\n");
    else
      fprintf(stderr, "usage: cdstart [start [stop]]\n");

    fprintf(stderr, " +             next track\n");
    fprintf(stderr, " -             previous track\n");
    fprintf(stderr, " x             restart at track 1\n");
    fprintf(stderr, " S             play track S\n");
    fprintf(stderr, " S E           play track S..E\n");
    fprintf(stderr, " start         play from track 1\n");
    fprintf(stderr, " stop          stop playing\n");
    fprintf(stderr, " resume        resume playing\n");
    fprintf(stderr, " eject         eject CD-ROM\n");
    fprintf(stderr, " info          display info\n");
    fprintf(stderr, " dir           display directory\n");
    fprintf(stderr, " shuffle       shuffle play\n");
    fprintf(stderr, " -v            verbose output\n");
    fprintf(stderr, " -R rr         times to repeat the CDROM (shuffle only)\n");
    fprintf(stderr, " -T tt         max tracks to play (shuffle only)\n");
  }
  
  else if ( strcmp(progname, "cdshuffle") == 0 ) {
    fprintf(stderr, "usage: cdshuffle [options]\n");
    fprintf(stderr, " -R rr         times to repeat the CDROM\n");
    fprintf(stderr, " -T tt         max tracks to play\n");
    fprintf(stderr, " -v            verbose output\n");
  }
  else if ( strcmp(progname, "cdstop") == 0 )
    fprintf(stderr, "usage: cdstop  [options]\n");
  else if ( strcmp(progname, "cdeject") == 0 )
    fprintf(stderr, "usage: cdeject  [options]\n");
  else if ( strcmp(progname, "cdpause") == 0 )
    {
      fprintf(stderr, "usage: cdpause  [options]\n");
      fprintf(stderr, "  resume play with cdplay/cdstart\n");
    } 
  else if ( strcmp(progname, "cdir") == 0 ) 
    {
      fprintf(stderr, "usage: cdir [ [-n] {-s|-t} ]\n");
      fprintf(stderr, " -n            listing without titles\n");
      fprintf(stderr, " -r            raw read of toc\n");
      fprintf(stderr, " -s            short listing (title/artist/track)\n");
      fprintf(stderr, " -t            print template for database entry\n");
    } 
  else if ( strcmp(progname, "cdinfo") == 0 ) 
    {
      fprintf(stderr, "usage: cdinfo [ {-a|-r|-s|-t} ]\n");
      fprintf(stderr, " -a            print absolute disc time\n");
      fprintf(stderr, " -r            print relative track time\n");
      fprintf(stderr, " -s            print audiostatus\n");
      fprintf(stderr, " -t            print current track\n");
      fprintf(stderr, " -v            print status, track, absolute, relative\n");
    }
  else 
    fprintf(stderr, "Please call as cdplay/cdstart, cdstop, cdeject, \n\
cdpause, cdir, cdinfo, or cdshuffle\n");

  /* standard options below here: */
  fprintf(stderr, " -V            build, version, and this help\n");
  fprintf(stderr, " -? -h         print this message\n");
  fprintf(stderr, " -[0-9]        use device "CD_DEVICE_BASE"[0-9]\n");
  fprintf(stderr, " -d dev        use device dev\n");
}
