/*
 * main routine
 *
 * Virtual Tiny Keyboard
 *
 * Copyright (c) 1997-1999 by Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "vkb.h"
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <tk.h>

/*
 * prototypes
 */

static int seq_device(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_on(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_off(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_start_note(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_stop_note(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_control(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_program(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_bender(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_chorus_mode(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int seq_reverb_mode(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int vkb_app_init(Tcl_Interp *interp);


/*
 * local common variables
 */
static void *private;
static int seq_devno = 0;
static int seq_opened = 0;
static int seq_bank = 0, seq_preset = 0;
static int seq_bend = 0;
static vkb_oper_t *oper;

/*
 * main routine
 */

void main(int argc, char **argv)
{
	char **nargv;
	int c, nargc;

	nargc = argc + 1;
	if ((nargv = (char**)malloc(sizeof(char*) * nargc)) == NULL) {
		fprintf(stderr, "vkeybd: can't malloc\n");
		exit(1);
	}
	nargv[0] = "-f";
	nargv[1] = VKB_TCLFILE;
	for (c = 1; c < argc; c++)
		nargv[c+1] = argv[c];

	/* copy the default device operator */
	if (vkb_num_devices <= 0) {
		fprintf(stderr, "vkeybd: no device is defined\n");
		exit(1);
	}
	oper = vkb_device[seq_devno]->oper;

	/* call Tk main routine */
	Tk_Main(nargc, nargv, vkb_app_init);

	exit(0);
}


/*
 * Tcl commands to sequencer device
 */

/* select device */
static int
seq_device(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int i;

	if (argc < 2)
		return TCL_ERROR;
	for (i = 0; i < vkb_num_devices && vkb_device[i]; i++) {
		if (strcmp(vkb_device[i]->name, argv[1]) == 0) {
			seq_devno = i;
			oper = vkb_device[i]->oper;
			return TCL_OK;
		}
	}
	vkb_error(interp, "can't find device '%s'", argv[1]);
	return TCL_ERROR;
}

/* open device */
static int
seq_on(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	if (! seq_opened && oper->open(interp, &private)) {
		seq_opened = 1;
		Tcl_Eval(interp, "ToggleSwitch normal");
		if (oper->program)
			oper->program(private, seq_bank, seq_preset);
		if (oper->bender)
			oper->bender(private, seq_bend);
	}
	return TCL_OK;
}

/* close device */
static int
seq_off(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	if (seq_opened) {
		oper->close(private);
		seq_opened = 0;
		Tcl_Eval(interp, "ToggleSwitch disabled");
	}
	return TCL_OK;
}

static int
seq_start_note(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int note, vel;
	if (argc < 3)
		return TCL_ERROR;
	if (! seq_opened)
		return TCL_OK;
	note = atoi(argv[1]);
	vel = atoi(argv[2]);
	if (oper->noteon)
		oper->noteon(private, note, vel);
	return TCL_OK;
}

static int
seq_stop_note(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int note, vel;
	if (argc < 3)
		return TCL_ERROR;
	if (! seq_opened)
		return TCL_OK;
	note = atoi(argv[1]);
	vel = atoi(argv[2]);
	if (oper->noteoff)
		oper->noteoff(private, note, vel);
	return TCL_OK;
}

static int
seq_control(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int type, val;
	if (argc < 3)
		return TCL_ERROR;
	if (! seq_opened)
		return TCL_OK;
	type = atoi(argv[1]);
	val = atoi(argv[2]);
	if (oper->control)
		oper->control(private, type, val);
	return TCL_OK;
}

static int
seq_program(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	if (argc < 3)
		return TCL_ERROR;
	seq_bank = atoi(argv[1]);
	seq_preset = atoi(argv[2]);
	if (! seq_opened)
		return TCL_OK;
	if (oper->program)
		oper->program(private, seq_bank, seq_preset);
	return TCL_OK;
}

static int
seq_bender(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	if (argc < 2)
		return TCL_ERROR;
	seq_bend = atoi(argv[1]);
	if (! seq_opened)
		return TCL_OK;
	if (oper->bender)
		oper->bender(private, seq_bend);
	return TCL_OK;
}

static int
seq_chorus_mode(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	if (argc < 2)
		return TCL_ERROR;
	if (! oper->chorus_mode)
		return TCL_OK;
	if (seq_opened && oper->open(interp, &private)) {
		int mode = atoi(argv[1]);
		Tcl_Eval(interp, "ToggleSwitch normal");
		oper->chorus_mode(private, mode);
	}
	return TCL_OK;
}

static int
seq_reverb_mode(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	if (argc < 2)
		return TCL_ERROR;
	if (! oper->reverb_mode)
		return TCL_OK;
	if (seq_opened && oper->open(interp, &private)) {
		int mode = atoi(argv[1]);
		Tcl_Eval(interp, "ToggleSwitch normal");
		oper->reverb_mode(private, mode);
	}
	return TCL_OK;
}


/*
 * Misc. functions
 */

void
vkb_error(Tcl_Interp *ip, char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	fputs("ERROR: ", stderr);
	vfprintf(stderr, fmt, ap);
	putc('\n', stderr);
	va_end(ap);
}


/*
 * Initialize Tcl/Tk components
 */
static int
vkb_app_init(Tcl_Interp *interp)
{
	if (Tcl_Init(interp) == TCL_ERROR) {
		return TCL_ERROR;
	}
	if (Tk_Init(interp) == TCL_ERROR) {
		return TCL_ERROR;
	}

	Tcl_CreateCommand(interp, "SetDevice", seq_device,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqOn", seq_on,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqOff", seq_off,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqStartNote", seq_start_note,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqStopNote", seq_stop_note,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqControl", seq_control,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqProgram", seq_program,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqBender", seq_bender,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqChorusMode", seq_chorus_mode,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqReverbMode", seq_reverb_mode,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	return TCL_OK;
}

