/* handling functions for cdrom drives installed in Gnome Toaster */

#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include "config.h"
#include "int.h"

#include "cddrives.h"
#include "datatrack.h"
#include "audiotrack.h"
#include "streams.h"
#include "preferences.h"
#include "varman.h"
#include "varmanwidgets.h"
#include "tracks.h"
#include "record.h"
#include "cdromlow.h"
#include "sems.h"

#ifdef HAVE_PTHREADS
#include "pthread.h"
#endif

/* uncomment for debugging */
/* #define DEBUG */

cddrives_drives_def cddrives_drives;
int cddrives_update=1;
int cddrives_updatethreadrunning=0;

void cddrives_updatedrivelist();
int cddrives_checkcdchange(gpointer);

/* see,if update process is active. wait for another second if this was the
 * case to make sure the cdrom drive has regained control of itself */
#define LOCKDB sems_lock(cddrives_drives.locked)
#define UNLOCKDB sems_unlock(cddrives_drives.locked)

void cddrives_updatemediachangeflags()
{
   int x;

   if (cddrives_update>0)
     {
	LOCKDB;
	for (x=0;x<cddrives_drives.drivecount;x++)
	  {
	     if ((!cddrives_drives.drive[x]->mediachange)&&
		 (cddrives_drives.drive[x]->cddb!=
		  cdromlow_cddbnumber((char*)&cddrives_drives.drive[x]->device)))
	       cddrives_drives.drive[x]->mediachange=1;
	  }
	;
	UNLOCKDB;
     };
}
;

#ifdef HAVE_PTHREADS
void *cddrives_mthreadhandler(void *dummy)
{
   while (1)
     {
	usleep(3000000);
	cddrives_updatemediachangeflags();
     };
};	
#endif

void cddrives_enablecdchangedetection()
{
   
/* spawn cd change detect in a secondary thread to prevent a laggy cdrom
 * from blocking gnometoaster's user interface */   
#ifdef HAVE_PTHREADS
   pthread_t *thread;
   
   thread=(pthread_t*)malloc(sizeof(pthread_t));

# ifdef DEBUG
   printf ("cddrives_enablecdchangedetection: spawning media change thread.\n");
# endif
   cddrives_updatethreadrunning=
     (pthread_create(thread,NULL,cddrives_mthreadhandler,NULL)==0);
#else
   cddrives_updatethreadrunning=0;
#endif
   
   gtk_timeout_add(5000,
		   cddrives_checkcdchange,
		   NULL);
}
;
	
/* install a preferences page in setup to enter a list of valid cd drives */
void cddrives_init()
{
   GtkWidget *cddrives_prefs;
   GtkWidget *label;
   
   varmanwidgets_widget *dlist;
   
   cddrives_prefs=gtk_vbox_new(0,0);
   gtk_widget_show(cddrives_prefs);
   label=gtk_label_new(_("CDROM Drive Setup"));
   gtk_widget_show(label);
   gtk_notebook_append_page(GTK_NOTEBOOK(preferences_editpad),
			    cddrives_prefs,
			    label);
   
   dlist=varmanwidgets_entry_new(_("valid drives:"),
				 "cddrives_drives",
				 global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(cddrives_prefs),dlist->visual,0,0,0);
   
   /* init the drive database lock by getting a semaphore for it */
   cddrives_drives.locked=sems_allocate();
   
   cddrives_createdrivelist();
	
   varman_install_handler(global_defs,"cddrives_drives",
			  cddrives_updatedrivelist,NULL);
   
   cddrives_enablecdchangedetection();
}
;

void cddrives_cdinfo_update(cddrives_cdinfo *info)
{
   int tracksoncd;
   int trackcount;
   int isrecorder;
   
   isrecorder=(strcmp((char*)&info->device,RECORDER)==0);

   /* media change detect during the content update could stale-lock
    * some cd drives */
   LOCKDB;
   
   info->cddb=cdromlow_cddbnumber((char*)&info->device);
   info->tracks=0;
   tracksoncd=cdromlow_tracks((char*)&info->device);
   if (tracksoncd>0)
     {		  
	if (cdromlow_hasdatatrack((char*)&info->device))
	  {
	     info->track[0]=datatrack_create((char*)&info->device);
	     if (isrecorder)
	       info->track[0]->precacherequired=atadd;
	     info->tracks++;
	  }
	;
	for (trackcount=info->tracks;trackcount<tracksoncd;trackcount++)
	  {
	     info->track[trackcount]=audiotrack_create((char*)&info->device,trackcount+1);
	     /* if the track is located on the current cd writer
	      * force precaching */
	     if (isrecorder)
	       info->track[trackcount]->precacherequired=atadd;
	     info->tracks++;
	  }
	;
     }
   ;
    
   UNLOCKDB;
}
;

/* wrapper creating a new device */
cddrives_cdinfo *cddrives_cdinfo_create(char *device,char *scsiid)
{
	cddrives_cdinfo *info;
	
	info=(cddrives_cdinfo*)malloc(sizeof(cddrives_cdinfo));
	strcpy((char*)&info->device,device);
	strcpy((char*)&info->scsiid,scsiid);
        cddrives_cdinfo_update(info);
	
	return info;
}
;

void cddrives_cdinfo_destroycontent(cddrives_cdinfo *info)
{
	int x;

	for (x=0;x<info->tracks;x++)
	  {
		  tracks_unclaim(info->track[x]);
	  }
	;
}
;

void cddrives_cdinfo_destroy(cddrives_cdinfo *info)
{
	cddrives_cdinfo_destroycontent(info);
        free(info);
}
;

void  cddrives_createdrivelist()
{
   cddrives_cdinfo *info;
   char *drives,*cdrive,*ndrive;
   char *scsiid;
   
   /* get drive list */
   drives=varman_getvar_copy(global_defs,"cddrives_drives");
   /* this is done to make the while loop easier */
   if (drives[strlen(drives)]!=" "[0]) strcat(drives," ");
   
   cdrive=drives;
   cddrives_drives.drivecount=0;
   
   while (strchr(cdrive," "[0])!=NULL)
     {		  
	ndrive=strchr(cdrive," "[0]);
	*ndrive=0;		  
	
	if (strchr(cdrive,"("[0])!=NULL)
	  {			    
	     scsiid=strchr(cdrive,"("[0]);
	     *scsiid=0;
	     scsiid++;
	     if (strchr(scsiid,")"[0])!=NULL)
	       *strchr(scsiid,")"[0])=0;
	  }
	else
	  scsiid="";		  		  
	
	info=cddrives_cdinfo_create(cdrive,scsiid);
	cddrives_drives.drive[cddrives_drives.drivecount]=info;
	
	cddrives_drives.drivecount++;
	cdrive=ndrive+1;		  
     };
   free(drives);
   
}
;
	
void  cddrives_destroydrivelist()
{
   int x;

   LOCKDB;
   
   for (x=0;x<cddrives_drives.drivecount;x++)
     {
	cddrives_cdinfo_destroy(cddrives_drives.drive[x]);
     }
   ;
   cddrives_drives.drivecount=0;
   
   UNLOCKDB;
}
;

void cddrives_updatedrivelist()
{
	cddrives_destroydrivelist();
	cddrives_createdrivelist();
}
;

int cddrives_checkcdchange(gpointer d)
{
   int x;
   
   /* update flags here if for some reason the media change thread
    * couldn't be spawned,e.g. if there's no support for phreads on your
    * system */
   if (!cddrives_updatethreadrunning)
     cddrives_updatemediachangeflags();
   
   if (cddrives_update>0)
     {	
	
#ifdef DEBUG
	printf ("cddrives_checkcdchange: checking for a cd change\n");
#endif
	for (x=0;x<cddrives_drives.drivecount;x++)
	  {
	     if (cddrives_drives.drive[x]->mediachange)
	       {
		  cddrives_cdinfo_destroycontent(cddrives_drives.drive[x]);
		  cddrives_cdinfo_update(cddrives_drives.drive[x]);
		  /* mark this particular media change as processed */
		  cddrives_drives.drive[x]->mediachange=0;
	       }
	     ;
	  }
	;
     }
   ;
   return 1; /* keep the timeout running */
}
;

/* return rom containing cd with cddb code */
cddrives_cdinfo *cddrives_getrombycddb(int cddb)
{
	int x;
	cddrives_cdinfo *f=NULL;	
	for (x=0;(x<cddrives_drives.drivecount)&&(f==NULL);x++)
	  {
		  if (cddrives_drives.drive[x]->cddb==cddb)
		      f=cddrives_drives.drive[x];
	  }
	;
	return f;
}
;

/* return rom with given device name */
cddrives_cdinfo *cddrives_getrombydevicename(char *device)
{
	int x;
	cddrives_cdinfo *f=NULL;	
	for (x=0;(x<cddrives_drives.drivecount)&&(f==NULL);x++)
	  {
		  if (!strcmp(cddrives_drives.drive[x]->device,device))
		      f=cddrives_drives.drive[x];
	  }
	;
	return f;
}
;

void cddrives_updatedisable()
{
   LOCKDB;
   cddrives_update--;
   UNLOCKDB;
};
   
void cddrives_updateenable()
{
   cddrives_update++;
};
   

