/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <gtk/gtk.h>

#include "debug.h"
#include "gui.h"
#include "status-item-view.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static void status_item_view_class_init (StatusItemViewClass *klass);
static void status_item_view_init (StatusItemView *object);
static void status_item_view_destroy (GtkObject *object);
static void size_allocated (GtkWidget*, GtkAllocation*, gpointer);

static gchar* string_shorten (GdkFont* font, int width, const gchar* original);

static void status_item_view_unregister (StatusItemView *view);

static int init_steps_cb    (gpointer, gpointer, gpointer);
static int next_step_cb     (gpointer, gpointer, gpointer);
static int progress_cb      (gpointer, gpointer, gpointer);
static int set_activity_cb  (gpointer, gpointer, gpointer);
static int status_cb        (gpointer, gpointer, gpointer);
static int error_cb         (gpointer, gpointer, gpointer);
static int done_cb          (gpointer, gpointer, gpointer);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static GtkObjectClass *parent_class = NULL;

static GtkTooltips *tips = NULL;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

GtkType
status_item_view_get_type (void)
{
	static GtkType status_item_view_type = 0;

	if (!status_item_view_type)
	{
		static const GtkTypeInfo status_item_view_info =
		{
			"StatusItemView",
			sizeof(StatusItemView),
			sizeof(StatusItemViewClass),
			(GtkClassInitFunc) status_item_view_class_init,
			(GtkObjectInitFunc) status_item_view_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		status_item_view_type = gtk_type_unique (
			GTK_TYPE_VBOX,
			&status_item_view_info);
	}

	return status_item_view_type;
}

/*****
******
*****/

static void
count_children (GtkWidget* w, gpointer data)
{
	*((int*)data) += 1;
}


static void
status_item_view_class_init (StatusItemViewClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class (GTK_TYPE_OBJECT);
	object_class->destroy = status_item_view_destroy;
	tips = gtk_tooltips_new ();
}

static void
status_item_view_init (StatusItemView *view)
{
	view->progress_major = NULL;
	view->progress_minor = NULL;
	view->description = NULL;

	view->steps = 0;
	view->step = 0;
	view->last_status = NULL;

	view->item = NULL;
}


/*****
******
*****/

static void
status_item_view_destroy (GtkObject* object)
{
	StatusItemView *item = NULL;
	debug(DEBUG_QUEUE, "Object '%p' in status_item_view_destroy", object);

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_STATUS_ITEM_VIEW (object));
	g_return_if_fail (GTK_OBJECT_CONSTRUCTED (object));

	item = STATUS_ITEM_VIEW(object);

	status_item_view_unregister (STATUS_ITEM_VIEW(object));

	gtk_signal_disconnect_by_func (
		GTK_OBJECT(item->progress_major), size_allocated, item);

	/* cleanup last_status */
	g_free (item->last_status);
	item->last_status = NULL;

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

GtkWidget*
status_item_view_new (void)
{
        /* gtk instantiation */
	StatusItemView *view = STATUS_ITEM_VIEW(
		gtk_type_new(TYPE_STATUS_ITEM_VIEW));

	/* parent setup */
	GTK_CONTAINER(view)->border_width = 0;
	GTK_BOX(view)->spacing = 0;
	GTK_BOX(view)->homogeneous = FALSE;

	/* status_item_view setup */
	view->progress_major = gtk_progress_bar_new ();
	gtk_signal_connect_after (
		GTK_OBJECT(view->progress_major), "size_allocate", size_allocated, view);
	gtk_progress_set_format_string (GTK_PROGRESS(view->progress_major), "");
	gtk_progress_set_show_text (GTK_PROGRESS(view->progress_major), TRUE);
	gtk_widget_show (view->progress_major);
	view->eventbox = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(view->eventbox), view->progress_major);
	view->description = gtk_label_new ("hello world");
#if 0
	gtk_widget_show (view->description);
	gtk_box_pack_start(GTK_BOX(view), view->description, FALSE, TRUE, 0);
#endif
	gtk_box_pack_start(GTK_BOX(view), view->eventbox, FALSE, TRUE, 0);

	return GTK_WIDGET(view);
}

GtkWidget*
status_item_view_new_with_item (StatusItem* item)
{
	StatusItemView* l = STATUS_ITEM_VIEW(status_item_view_new());
	status_item_view_set_item (l, item);
	return GTK_WIDGET(l);
}

/*****
******
*****/

static void
status_item_view_unregister (StatusItemView *view)
{
	StatusItem *item = NULL;

	item = STATUS_ITEM(view->item);
	if (!item)
		return;

	/* stop listening to the item... */
	pan_callback_remove (item->init_steps, init_steps_cb, view);
	pan_callback_remove (item->next_step, next_step_cb, view);
	pan_callback_remove (item->progress, progress_cb, view);
	pan_callback_remove (item->set_activity, set_activity_cb, view);
	pan_callback_remove (item->status, status_cb, view);
	pan_callback_remove (item->error, error_cb, view);
	pan_callback_remove (item->done, done_cb, view);

	/* unref the item... */
	pan_object_unref(PAN_OBJECT(item));
	view->item = NULL;
}

void
status_item_view_set_item (StatusItemView* view, StatusItem* item)
{
	gchar *description = NULL;

	/* skip the no-op */
	if (item == view->item)
		return;

	description = status_item_describe (item);

	status_item_view_unregister (view);

	pan_object_ref(PAN_OBJECT(item));

	gtk_tooltips_set_tip(tips,
		GTK_WIDGET(view->eventbox), description, NULL);

	gtk_label_set_text (GTK_LABEL(view->description), description);

	g_free (description);

	pan_callback_add (item->init_steps, init_steps_cb, view);
	pan_callback_add (item->next_step, next_step_cb, view);
	pan_callback_add (item->progress, progress_cb, view);
	pan_callback_add (item->set_activity, set_activity_cb, view);
	pan_callback_add (item->status, status_cb, view);
	pan_callback_add (item->error, error_cb, view);
	pan_callback_add (item->done, done_cb, view);

	view->item = item;
}

/*****
******
*****/

static int
progress_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	int of_100 = GPOINTER_TO_INT(call_arg);
	StatusItemView *view = (StatusItemView*) user_data;

	view->progress_value = of_100;
	if (view->progress_last_value != view->progress_value)
	{
		GtkProgress *prog = GTK_PROGRESS(view->progress_major);


		/* make sure we're in percentage mode */
		if (prog->activity_mode) {
			pan_lock();
			gtk_progress_set_activity_mode (prog, FALSE);
			pan_unlock();
		}

		/* update the percentage */
		view->progress_last_value = view->progress_value;
		if (0<=view->progress_value && view->progress_value<=100) {
			pan_lock();
			gtk_progress_set_percentage (prog, view->progress_value/100.0);
			pan_unlock();
		}
	}

	return 0;
}


static int
init_steps_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	int steps = GPOINTER_TO_INT(call_arg);
	StatusItem* item = (StatusItem*) call_object;
	StatusItemView *view = (StatusItemView*) user_data;

	view->step = 0;
	view->steps = steps;
	view->progress_value = 0;
	view->progress_last_value = 0;

	progress_cb (item, GINT_TO_POINTER(0), view);

	return 0;
}

static int
next_step_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	StatusItem* item = (StatusItem*) call_object;
	StatusItemView *view = (StatusItemView*) user_data;

	if (view->steps) {
		gint of_100 = (int)((100.0 * ++view->step)/view->steps);
		progress_cb (item, GINT_TO_POINTER(of_100), view);
	}

	return 0;
}

/**
***
**/

static int
set_activity_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	StatusItemView *view = (StatusItemView*) user_data;
	GtkProgress *progress = GTK_PROGRESS(view->progress_major);
	view->progress_value = GPOINTER_TO_INT(call_arg);

	/* make sure we're in activity mode */
	if (!progress->activity_mode)
	{
		pan_lock();
		gtk_progress_set_activity_mode (progress, TRUE);
		gtk_progress_bar_set_discrete_blocks(
			GTK_PROGRESS_BAR(progress), 20);
		gtk_progress_bar_set_activity_step (
			GTK_PROGRESS_BAR(progress), 1);
		pan_unlock();
	}

	/* set the value */
	pan_lock();
	gtk_progress_set_value (progress, view->step%19);
	pan_unlock();

	return 0;
}

/**
***
**/


static int
status_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	StatusItemView *v = STATUS_ITEM_VIEW(user_data);
	GtkProgress* prog = GTK_PROGRESS(v->progress_major);
	const gchar* str = (const gchar*)call_arg;
	gchar *s = string_shorten (
		gtk_widget_get_style(v->progress_major)->font,
		v->progress_major->allocation.width,
		str);

	/* update last_status */
	g_free (v->last_status);
	v->last_status = g_strdup(str);

	/* update progressbar w/o a resize */
	pan_lock();
	if (prog->format)
		g_free (prog->format);
	prog->format = s;
	GTK_PROGRESS_CLASS (GTK_OBJECT (prog)->klass)->update (prog);
	pan_unlock();

	/* cleanup */
	return 0;
}

static void
size_allocated (GtkWidget* progress_major, GtkAllocation* a, gpointer data)
{
	StatusItemView *v = STATUS_ITEM_VIEW(data);
	gchar *status = g_strdup(v->last_status);
	status_cb (NULL, status, v);
	g_free (status);
}

static int
error_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	const gchar *error = (const gchar*)call_arg;
	StatusItemView *v = STATUS_ITEM_VIEW(user_data);

	g_warning ("%s",error);
	debug(DEBUG_QUEUE,"Error: %s", error);
	status_cb (NULL, (gpointer)error, v);

	return 0;
}

static int
done_cb (
	gpointer call_obj,
	gpointer call_arg,
	gpointer user_data)
{
	/* FIXME */
	return 0;
}

/**
***
**/

static void
string_zotz_char (gchar* str, int i) {
	gchar *p = str+i;
	for (;;) {
		++p;
		p[-1] = *p;
		if (!*p)
			break;
	}
}

static gchar*
string_shorten (GdkFont* font, int max_width, const gchar* original)
{
	gchar *clone = NULL;
	size_t len = 0;
	int i, j;
	gboolean i_turn = TRUE;
	int target_width = max_width - 4;

	/* if original is NULL, so is shortened */
	if (!original)
		return NULL;

	/* if original fits, use it */
	len = strlen(original);
	if (gdk_text_width (font, original, len) < target_width)
		return g_strdup (original);

	/* add in the middle "..." */
	clone = g_malloc (len + strlen("...") + 1);
	i = len/2;
	memcpy (clone, original, i);
	memcpy (clone+i, "...", 3);
	j = i+3;
	memcpy (clone+j, original+i, len+1-i);
	original = NULL;

	/* start trimming out text until it fits */	
	while (gdk_text_width(font, clone, strlen(clone)) >= target_width)
	{
		int *pi = i_turn ? &i : &j;
		*pi -= 1;
		if (*pi<0 || *pi>=strlen(clone))
			break;
		string_zotz_char (clone, *pi);
		i_turn = !i_turn;
	}

	return clone;
}
