/*
 *  Copyright (C) 1998-99 Luca Deri <deri@unipi.it>
 *                      
 *  			  Centro SERRA, University of Pisa
 *  			  http://www-serra.unipi.it/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ntop.h"


typedef struct arpEntries {
  HostTraffic* host;
  unsigned long sentPkts, rcvdPkts;
} ArpEntries;

struct	_arphdr {
  u_short	ar_hrd;		/* format of hardware address */
#define ARPHRD_ETHER 	1	/* ethernet hardware address */
#define ARPHRD_802 	6	/* any 802 network */
  u_short	ar_pro;		/* format of protocol address */
  u_char	ar_hln;		/* length of hardware address */
  u_char	ar_pln;		/* length of protocol address */
  u_short	ar_op;		/* one of: */
#define	ARPOP_REQUEST	1	/* request to resolve address */
#define	ARPOP_REPLY	2	/* response to previous request */
};

struct	_ether_arp {
  struct	_arphdr ea_hdr;	/* fixed-size header */
  u_char	arp_sha[6];	/* sender hardware address */
  u_char	arp_spa[4];	/* sender protocol address */
  u_char	arp_tha[6];	/* target hardware address */
  u_char	arp_tpa[4];	/* target protocol address */
};

#ifndef arp_hrd
#define	arp_hrd	ea_hdr.ar_hrd
#endif

#ifndef arp_pro
#define	arp_pro	ea_hdr.ar_pro
#endif

#ifndef arp_hln
#define	arp_hln	ea_hdr.ar_hln
#endif

#ifndef arp_pln
#define	arp_pln	ea_hdr.ar_pln
#endif

#ifndef arp_op
#define	arp_op	ea_hdr.ar_op
#endif

extern char* intoa(struct in_addr addr);
extern char* etheraddr_string(const u_char *ep);
extern void printHTTPheader();
extern void printHTTPtrailer();
extern HostTraffic* findHostByNumIP(char* numIPaddr);
extern char* makeHostLink(HostTraffic *el, short mode, 
			  short cutName, short addCountryFlag);
extern void sendString(char *theString);
extern void quicksort(void*, size_t, size_t, int(*cmp)());
extern char* getRowColor();

#ifdef MULTITHREADED
extern pthread_mutex_t gdbmMutex;
#endif

extern int webMode;
static int dumpArpPackets = 0;
static GDBM_FILE arpDB;
static int columnSort = 0;

#define MAX_NUM_ARP_ENTRIES  512
static ArpEntries theArpEntries[MAX_NUM_ARP_ENTRIES];


static void handleArpPacket(const struct pcap_pkthdr *h, const u_char *p) {
  struct _ether_arp *arpPkt;
  struct in_addr addr;
  u_short op;
  datum key_data, data_data;
  char *ipAddr, *macAddr, tmpStr[32], tmpStr1[32];
  unsigned long numPkts;

  if(arpDB == NULL)
    return;

#ifdef MULTITHREADED
    accessMutex(&gdbmMutex);
#endif 

  arpPkt = (struct _ether_arp*)(p+sizeof(struct ether_header));
  op = (u_short)ntohs(*(u_short *)(&arpPkt->arp_pro));
  if (!((op != ETHERTYPE_IP && op != ETHERTYPE_TRAIL)
	|| arpPkt->arp_hln != sizeof(arpPkt->arp_sha)
	|| arpPkt->arp_pln != sizeof(arpPkt->arp_spa))) {
    op = (u_short)ntohs(*(u_short *)(&arpPkt->arp_op));

    if(op == ARPOP_REQUEST) {      
      memcpy(&addr.s_addr, arpPkt->arp_tpa, sizeof(addr.s_addr));
      NTOHL(addr.s_addr);
      if(dumpArpPackets) printf("ARP Request: who-has %s ", intoa(addr));
      memcpy(&addr.s_addr, arpPkt->arp_spa, sizeof(addr.s_addr));
      NTOHL(addr.s_addr);
      ipAddr = intoa(addr);
      if(dumpArpPackets) printf("tell %s\n", ipAddr);

      /* ******** */
      sprintf(tmpStr, "s%s", ipAddr);
      key_data.dptr = tmpStr; key_data.dsize = strlen(key_data.dptr)+1;
      data_data = gdbm_fetch(arpDB, key_data);
      if(data_data.dptr != NULL) {
	numPkts = atol(data_data.dptr)+1;
	free(data_data.dptr);
      } else
	numPkts = 1;
	
      sprintf(tmpStr1, "%u", (unsigned long)numPkts);
      data_data.dptr = tmpStr1; data_data.dsize = strlen(data_data.dptr)+1;
      gdbm_store(arpDB, key_data, data_data, GDBM_REPLACE);	
      /* ******** */
    } else {
      memcpy(&addr.s_addr, arpPkt->arp_spa, sizeof(addr.s_addr));
      NTOHL(addr.s_addr);
      ipAddr = intoa(addr);
      macAddr = etheraddr_string(arpPkt->arp_sha);

      if(dumpArpPackets)
	printf("ARP Reply: %s is-at %s\n", ipAddr, macAddr);
      
      /* ******** */
      key_data.dptr = ipAddr; key_data.dsize = strlen(key_data.dptr)+1;
      data_data.dptr = macAddr; data_data.dsize = strlen(data_data.dptr)+1;
      gdbm_store(arpDB, key_data, data_data, GDBM_REPLACE);

      /* ******** */
      sprintf(tmpStr, "r%s", ipAddr);
      key_data.dptr = tmpStr; key_data.dsize = strlen(key_data.dptr)+1;
      data_data = gdbm_fetch(arpDB, key_data);
      if(data_data.dptr != NULL) {
	numPkts = atol(data_data.dptr)+1;
	free(data_data.dptr);
      } else
	numPkts = 1;

      sprintf(tmpStr1, "%u", (unsigned long)numPkts);
      data_data.dptr = tmpStr1; data_data.dsize = strlen(data_data.dptr)+1;
      gdbm_store(arpDB, key_data, data_data, GDBM_REPLACE);
      /* ******** */
    }
  }

#ifdef MULTITHREADED
  releaseMutex(&gdbmMutex);
#endif 

}

/* ******************************* */

static int sortARPhosts(const void *_a, const void *_b) {
  ArpEntries *a = (ArpEntries *)_a;
  ArpEntries *b = (ArpEntries *)_b;

  if((a == NULL) && (b != NULL)) {
    printf("WARNING (1)\n");
    return(1);
  } else if((a != NULL) && (b == NULL)) {
    printf("WARNING (2)\n");
    return(-1);
  } else if((a == NULL) && (b == NULL)) {
    printf("WARNING (3)\n");
    return(0);
  }

  switch(columnSort) {
  default:
  case 1:
    if(a->host->hostSymIpAddress == NULL) a->host->hostSymIpAddress = a->host->hostNumIpAddress;
    if(b->host->hostSymIpAddress == NULL) b->host->hostSymIpAddress = b->host->hostNumIpAddress;		
    return(strcasecmp(a->host->hostSymIpAddress, b->host->hostSymIpAddress));
    break;
  case 2:
    if(a->host->hostIpAddress.s_addr > b->host->hostIpAddress.s_addr)
      return(1);
    else if(a->host->hostIpAddress.s_addr < b->host->hostIpAddress.s_addr)
      return(-1);
    else
      return(0);
    break;
  case 3:
    return(strcasecmp(a->host->ethAddressString, b->host->ethAddressString));
    break;

  case 4:
    if(a->sentPkts < b->sentPkts)
      return(1);
    else if (a->sentPkts > b->sentPkts)
      return(-1);
    else
      return(0);
    break;
  case 5:
    if(a->rcvdPkts < b->rcvdPkts)
      return(1);
    else if (a->rcvdPkts > b->rcvdPkts)
      return(-1);
    else
      return(0);
    break;
  }  
}

/* ****************************** */

static void handleArpWatchHTTPrequest(char* url) {
  datum data_data, key_data, fkey_data, return_data = gdbm_firstkey (arpDB);
  char tmpStr[BUF_SIZE], macAddr[32];
  int numEntries = 0, i, revertOrder=0;
  char *pluginName = "<A HREF=/plugins/arpWatch";
  char *sign = "-";

  if(url[0] =='\0')
    columnSort = 0;
  else {
    if(url[0] == '-') {
      sign = "";
      revertOrder = 1;
      columnSort = atoi(&url[1]);
    } else
      columnSort = atoi(url);
  }

  printHTTPheader();
  sendString("<HTML><BODY BGCOLOR=#FFFFFF><FONT FACE=Helvetica>"
	     "<CENTER><H1>Welcome to arpWatch</H1>\n<p>"
	     "<TABLE BORDER><TR>");

  sprintf(tmpStr, "<TH>%s?%s1>Host</A></TH>"
	 "<TH>%s?%s2>IP&nbsp;Address</A></TH>"
	 "<TH>%s?%s3>MAC</A></TH>"
	 "<TH>%s?%s4>#&nbsp;ARP&nbsp;Req.&nbsp;Sent</A></TH>"
	 "<TH>%s?%s5>#&nbsp;ARP&nbsp;Resp.&nbsp;Sent</A></TH></TR>\n",
	 pluginName, sign,
	 pluginName, sign,
	 pluginName, sign,
	 pluginName, sign,
	 pluginName, sign);

  sendString(tmpStr);
 
#ifdef MULTITHREADED
  accessMutex(&gdbmMutex);
#endif 
 
  while (return_data.dptr != NULL) {
    key_data = return_data;

    if((key_data.dptr[0] != 'r')
       && (key_data.dptr[0] != 's')) {
      unsigned long sentPkts, rcvdPkts;
      HostTraffic* host;
      
      data_data = gdbm_fetch(arpDB, key_data);
      if(data_data.dptr != NULL) {
	strcpy(macAddr, data_data.dptr);
	free(data_data.dptr);
      } else
	strcpy(macAddr, "???");

      sprintf(tmpStr, "s%s", key_data.dptr);
      fkey_data.dptr = tmpStr; fkey_data.dsize = strlen(fkey_data.dptr)+1;
      data_data = gdbm_fetch(arpDB, fkey_data);
      if(data_data.dptr != NULL) {
	sentPkts = atol(data_data.dptr)+1;
	free(data_data.dptr);
      } else
	sentPkts = 0;

      sprintf(tmpStr, "r%s", key_data.dptr);
      fkey_data.dptr = tmpStr; fkey_data.dsize = strlen(fkey_data.dptr)+1;
      data_data = gdbm_fetch(arpDB, fkey_data);
      if(data_data.dptr != NULL) {
	rcvdPkts = atol(data_data.dptr)+1;
	free(data_data.dptr);
      } else
	rcvdPkts = 0;


      host = findHostByNumIP(key_data.dptr);

      if((host != NULL)
	 && (numEntries < MAX_NUM_ARP_ENTRIES)) {
	theArpEntries[numEntries].host = host;
	theArpEntries[numEntries].sentPkts = sentPkts;
	theArpEntries[numEntries++].rcvdPkts = rcvdPkts;
      }
    }

    return_data = gdbm_nextkey(arpDB, key_data);    
    free(key_data.dptr);
  }

#ifdef MULTITHREADED
  releaseMutex(&gdbmMutex);
#endif 

  /* printf("---> %d\n", numEntries); */

  quicksort(theArpEntries, numEntries, sizeof(ArpEntries), sortARPhosts);

  for(i=0; i<numEntries; i++) {
    ArpEntries *theEntry;

    if(revertOrder)
      theEntry = &theArpEntries[numEntries-i-1];
    else
      theEntry = &theArpEntries[i];

    sprintf(tmpStr, "<TR %s>%s"
	    "<TD ALIGN=RIGHT>%s</TD>"
	    "<TD ALIGN=RIGHT>%s</TD>"
	    "<TD ALIGN=RIGHT>%u</TD>"
	    "<TD ALIGN=RIGHT>%u</TD></TR>\n",
	    getRowColor(),
	    makeHostLink(theEntry->host, 1, 1, 0),
	    theEntry->host->hostNumIpAddress,
	    theEntry->host->ethAddressString,
	    theEntry->sentPkts,
	    theEntry->rcvdPkts);

    sendString(tmpStr);
  }


  sendString("</TABLE></CENTER><p>\n");
  printHTTPtrailer();
}

/* ****************************** */

static void termArpFunct() {
  if(webMode)
    printf("Thanks for using arpWatch.\n");
  
  if(arpDB != NULL) {
    gdbm_close(arpDB);
    arpDB = NULL;
  }
}

/* ****************************** */

static PluginInfo arpPluginInfo[] = {
  "arpWatchPlugin",
  "This plugin handles ARP packets",
  "1.0", /* version */
  "<A HREF=http://jake.unipi.it/~deri/>Luca Deri</A>", 
  "arpWatch", /* http://<host>:<port>/plugins/arpWatch */
  termArpFunct, /* TermFunc   */
  handleArpPacket, /* PluginFunc */
  handleArpWatchHTTPrequest,
  "arp" /* BPF filter: filter all the ARP packets */
};

/* Plugin entry fctn */
PluginInfo* PluginEntryFctn() {

  if(webMode)
    printf("Welcome to %s. (C) 1999 by Luca Deri.\n", arpPluginInfo->pluginName);

  arpDB = gdbm_open ("arpWatch.db", 0, GDBM_NEWDB, 00664, NULL);

  if(arpDB == NULL) 
    printf("Unable to open arpWatch database. This plugin will be disabled.\n");

  return(arpPluginInfo);
}
