/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    Significant bug-fixes and redesigns by Han Holl <jeholl@euronet.nl>
*/

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <netdb.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include "mserver.h"

extern int errno;

struct msrv_client *clients[MAXCLIENTS];
extern int nclients;

static void advertize()
{
	int i, len, totlen = 0;
	char *name, *dot;
	char buf[MAXCLIENTNAMES];
	
	
	for (i = 0 ; i < MAXCLIENTS ; i++)
	{
		if (clients[i] == (struct msrv_client *) NULL)
		{
			continue;
		}

		name = clients[i]->name;
		dot = strchr(name, '.');

		len = dot ? dot - name : strlen(name);

		if (len > 12)
		{
			len = 12;
		}

		if (totlen + len > MAXCLIENTNAMES -1)
		{
			len = MAXCLIENTNAMES - 1 - totlen;
		}
	
		if (len == 0) 
		{
			syslog(LOG_WARNING, "MAXCLIENTNAMES too small");
			break;
		}

		strncpy(buf + totlen, name, len);
		totlen += len;
		buf[totlen++] = ':';
	}
	if (totlen > 0)
		totlen--;
	buf[totlen] = 0;
	syslog(LOG_DEBUG, "names: %s\n", buf);
	shmem_set_who(buf);
}

static struct msrv_client *_add_client(pid_t pid, int sock)
{
	char error[1024];

	struct hostent *hostp;
	struct sockaddr_in peer;

	struct msrv_client *mcl = malloc(sizeof(struct msrv_client));

	int len = sizeof(peer);

	mcl->pid = pid;
	mcl->name = NULL;
	mcl->start = time(0);

	nclients++;

	shmem_set_clients(nclients);

	if (getpeername(sock, (struct sockaddr *) &peer, &len) == 0)
	{
		hostp = gethostbyaddr((const char *) &peer.sin_addr, sizeof(struct in_addr), AF_INET);

		if (hostp)
		{
			mcl->name = strdup(hostp->h_name);
		}
	}
	else
	{
		strncpy (error, strerror(errno), 1024);
		syslog(LOG_WARNING, "Could not determine peername for client: %s!\n", error);
	}

	if (mcl->name == NULL)
	{
		mcl->name = strdup("unknown");
	}

	syslog(LOG_INFO, "Signon: %s", mcl->name);
	return mcl;
}

void client_connect_script (pid_t pid, int sock)
{
	char error[1024];
	char command[1024];
	char temp[1024];
	
	int retval;

	struct hostent *clienthost;

	struct sockaddr_in peer;

	size_t len = sizeof(peer);

	if (getpeername(sock, (struct sockaddr *) &peer, &len) == 0)
	{
		clienthost = gethostbyaddr((const char *) &peer.sin_addr, sizeof(struct in_addr), AF_INET);

		config_getvalue("clientup", temp, 1024);

		if (strncmp(temp, "", 1024))
		{
			if (clienthost)
			{
				snprintf(command, 1024, "%s \"%s\"", temp, clienthost->h_name);
			}
			else
			{
				snprintf(command, 1024, "%s \"\"", temp);
			}

			syslog(LOG_DEBUG, "Executing: %s", command);
			retval = util_system_wait(command);
			syslog(LOG_DEBUG, "Return code: %d", retval);
		}
	}
	else
	{
		strncpy (error, strerror(errno), 1024);
		syslog(LOG_WARNING, "Could not determine peername for client: %s!\n", error);
	}
}

void add_client(pid_t pid, int sock)
{
	int i;
	
	syslog(LOG_DEBUG, "Adding client %d", pid);

	for (i = 0 ; i < MAXCLIENTS ; i++)
	{
		if (clients[i] == 0)
		{
			clients[i] = _add_client(pid, sock);
			advertize();
			client_connect_script(pid, sock);
			return;
		}
	}
	syslog(LOG_ERR, "Too many clients");
}

static void _remove_client(struct msrv_client *mcl)
{
	time_t connected = util_time(0) - mcl->start;
	int hr,mn,sec;

	hr = connected /3600;
	connected -= hr * 3600;
	mn = connected / 60;
	sec = connected - mn * 60;
	mcl->pid = 0;
	nclients--;

	syslog(LOG_INFO, "Signoff: %s (%d:%02d:%02d)", mcl->name, hr, mn, sec);

	free(mcl->name);
	free(mcl);

	shmem_set_clients(nclients);
}

static void client_disconnect_script(struct msrv_client *mcl)
{
	char temp[1024];
	char command[1024];

	int retval;

	auth_downscript();

	config_getvalue("clientdown", temp, 1024);

	if (strncmp(temp, "", 1024))
	{
		snprintf(command, 1024, "%s \"%s\"", temp, mcl->name);

		syslog(LOG_DEBUG, "Executing: %s", command);
		retval = util_system_wait(command);
		syslog(LOG_DEBUG, "Return code: %d", retval);
	}
}

void child_handler(int sig)
{
	pid_t pid;
	int i, status;
	
	for (;;)
	{
		pid = wait3(&status, WNOHANG, (struct rusage *) 0);

		if (pid <= 0)
		{
			break;
		}

		for (i = 0 ; i < MAXCLIENTS ; i++)
		{
			if (clients[i] && clients[i]->pid == pid)
			{
				client_disconnect_script(clients[i]);
				_remove_client(clients[i]);
				clients[i] = (struct msrv_client *) NULL;
				advertize();
				break;
			}
		}

		if (i == MAXCLIENTS)
		{
			syslog(LOG_ERR, "Unknown client %d", pid);
		}

		syslog(LOG_DEBUG, "Client %d exits, %d", pid, nclients);
	}
}

int child_count(void)
{
	return shmem_get_clients();
}

bool child_alive(pid_t checkpid)
{
	int i;
	
	if (checkpid <= 0)
	{
		return false;	
	}

	for (i = 0 ; i < MAXCLIENTS ; i++)
	{
		if (clients[i] && clients[i]->pid == checkpid)
		{
			return true;
		}
	}

	return false;
}
