/*
 itext.c : Irssi text widget

           This code has nothing irssi specific so if someone really wants to
	   it can be used in other projects too.. Unlike rest of irssi, this
	   is distributed under LGPL.


    Copyright (C) 1999 Timo Sirainen

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xos.h>

#ifdef HAVE_IMLIB
#include <gdk_imlib.h>
#endif

#include "itext.h"

#define iswordcut(a) (isspace((gint) a) || \
    (a) == '\n' || (a) == '\0' || (a) == '"' || \
    (a) == '(' || (a) == ')' || (a) == '[' || (a) == ']' || \
    (a) == '<' || (a) == '>')

#define d(x)

#define TRANSPARENCY

enum
{
    WORD_CLICKED,
    LAST_SIGNAL
};
static guint itext_signals[LAST_SIGNAL] = { 0 };

/* GTK parent class */
static GtkWidgetClass *parent_class = NULL;
#ifdef TRANSPARENCY
static Window desktop_window = None;
#endif

static void gtk_itext_class_init(GtkITextClass *class);
static void gtk_itext_init(GtkIText *itext);
static void gtk_itext_destroy(GtkObject *object);

static void gtk_itext_init(GtkIText *itext);
static void gtk_itext_realize(GtkWidget *widget);
static void gtk_itext_unrealize (GtkWidget *widget);
static void gtk_itext_size_request(GtkWidget *widget, GtkRequisition *requisition);
static void gtk_itext_size_allocate(GtkWidget *widget, GtkAllocation *allocation);
static gint gtk_itext_button_press(GtkWidget *widget, GdkEventButton *event);
static gint gtk_itext_button_release(GtkWidget *widget, GdkEventButton *event);
static gint gtk_itext_motion_notify(GtkWidget *widget, GdkEventMotion *event);
static gint gtk_itext_expose(GtkWidget *widget, GdkEventExpose *event);
static void gtk_itext_draw(GtkWidget *widget, GdkRectangle *area);

static gint gtk_itext_selection_clear(GtkWidget *widget, GdkEventSelection *event);
static void gtk_itext_selection_get(GtkWidget *widget, GtkSelectionData *selection_data_ptr, guint info, guint time);

static void gtk_itext_set_origin(GtkIText *itext);
static void gtk_itext_redraw_moved(GtkIText *itext, gint diff);
static void gtk_itext_scrollbar_moved(GtkAdjustment *adj, GtkWidget *widget);
static void gtk_itext_refresh(GtkIText *itext);
static void gtk_itext_init_background(GtkIText *itext);

static GdkPixmap *get_background(GtkIText *itext);
static void itext_transparent_free(GtkIText *itext);

guint gtk_itext_get_type(void)
{
    static guint itext_type = 0;

    if (!itext_type)
    {
	GtkTypeInfo itext_info =
	{
	    "GtkIText",
	    sizeof (GtkIText),
	    sizeof (GtkITextClass),
	    (GtkClassInitFunc) gtk_itext_class_init,
	    (GtkObjectInitFunc) gtk_itext_init,
	    (GtkArgSetFunc) NULL,
	    (GtkArgGetFunc) NULL,
	};

        itext_type = gtk_type_unique (gtk_widget_get_type (), &itext_info);
    }

    return itext_type;
}

static void gtk_itext_class_init(GtkITextClass *class)
{
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;
    GtkITextClass *itext_class;

    object_class = (GtkObjectClass *) class;
    widget_class = (GtkWidgetClass *) class;
    itext_class = (GtkITextClass *) class;

    parent_class = gtk_type_class (gtk_widget_get_type ());

    itext_signals[WORD_CLICKED] =
	gtk_signal_new ("word_clicked",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET(GtkITextClass, word_clicked),
			gtk_marshal_NONE__INT_POINTER,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_INT,
			GTK_TYPE_STRING);
    gtk_object_class_add_signals(object_class, itext_signals, LAST_SIGNAL);

    object_class->destroy = gtk_itext_destroy;

    widget_class->realize = gtk_itext_realize;
    widget_class->unrealize = gtk_itext_unrealize;
    widget_class->expose_event = gtk_itext_expose;
    widget_class->draw = gtk_itext_draw;
    widget_class->size_request = gtk_itext_size_request;
    widget_class->size_allocate = gtk_itext_size_allocate;
    widget_class->button_press_event = gtk_itext_button_press;
    widget_class->button_release_event = gtk_itext_button_release;
    widget_class->motion_notify_event = gtk_itext_motion_notify;

    widget_class->selection_clear_event = gtk_itext_selection_clear;
    widget_class->selection_get = gtk_itext_selection_get;

    itext_class->word_clicked = NULL;
}

static void gtk_itext_init(GtkIText *itext)
{
    GdkFont *font, *font_bold;

    GTK_WIDGET_SET_FLAGS (itext, GTK_CAN_FOCUS);

    d(printf("gtk_itext_init()\n"));

    itext->fore_gc = NULL;
    itext->back_gc = NULL;
    itext->tback_gc = NULL;

    /* scrollback position adjustment */
    itext->adjustment =
	GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 1.0, 1.0, 1.0, 1.0));
    gtk_object_ref(GTK_OBJECT(itext->adjustment));
    gtk_object_sink(GTK_OBJECT(itext->adjustment));

    gtk_signal_connect(GTK_OBJECT(itext->adjustment), "value_changed",
		       GTK_SIGNAL_FUNC(gtk_itext_scrollbar_moved), itext);
    itext->select_start = -1;

    gtk_selection_add_target(GTK_WIDGET(itext), GDK_SELECTION_PRIMARY,
			     GDK_SELECTION_TYPE_STRING, 0);

    font = gdk_font_load("-adobe-helvetica-medium-r-normal-*-*-120-*-*-p-*-iso8859-1");
    font_bold = gdk_font_load("-adobe-helvetica-bold-r-normal-*-*-120-*-*-p-*-iso8859-1");
    gtk_itext_set_fonts(itext, font, font_bold);
}

static void gtk_itext_destroy(GtkObject *object)
{
    GtkIText *itext;
    GList *tmp;

    g_return_if_fail(object != NULL);
    g_return_if_fail(GTK_IS_ITEXT(object));

    itext = GTK_ITEXT(object);

    /* release fonts */
    gdk_font_unref(itext->font);
    gdk_font_unref(itext->font_bold);
    itext->font = itext->font_bold = NULL;

    /* release the adjustment */
    if (itext->adjustment)
    {
	gtk_signal_disconnect_by_data(GTK_OBJECT(itext->adjustment), itext);
	gtk_object_unref(GTK_OBJECT(itext->adjustment));
	itext->adjustment = NULL;
    }

    if (GTK_OBJECT_CLASS(parent_class)->destroy)
	(*GTK_OBJECT_CLASS(parent_class)->destroy)(object);

    /* free memory used by text lines */
    for (tmp = itext->lines; tmp != NULL; tmp = tmp->next)
    {
	GtkITextLine *line = tmp->data;

	g_list_foreach(line->colors, (GFunc) g_free, NULL);
	g_list_free(line->colors);

	g_free(line->str);
	g_free(line);
    }
    g_list_free(itext->lines);

    if (itext->pixmap_path != NULL) g_free(itext->pixmap_path);
}

static void gtk_itext_realize(GtkWidget *widget)
{
    GtkIText *itext;
    GdkWindowAttr attributes;
    gint attributes_mask;

    d(printf("gtk_itext_realize()\n"));
    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_ITEXT(widget));

    GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
    itext = GTK_ITEXT(widget);

    attributes.x = widget->allocation.x;
    attributes.y = widget->allocation.y;
    attributes.width = widget->allocation.width;
    attributes.height = widget->allocation.height;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.event_mask = gtk_widget_get_events(widget) |
	GDK_EXPOSURE_MASK |
	GDK_BUTTON_PRESS_MASK |
	GDK_BUTTON_MOTION_MASK |
	GDK_POINTER_MOTION_MASK |
	GDK_BUTTON_RELEASE_MASK |
	GDK_KEY_PRESS_MASK;
    attributes.visual = gtk_widget_get_visual (widget);
    attributes.colormap = gtk_widget_get_colormap (widget);

    attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

    /* main window */
    widget->window = gdk_window_new(gtk_widget_get_parent_window(widget),
				    &attributes, attributes_mask);
    widget->style = gtk_style_attach(widget->style, widget->window);
    gdk_window_set_user_data(widget->window, widget);
    gtk_style_set_background(widget->style, widget->window, GTK_STATE_ACTIVE);

    gdk_window_set_background(widget->window, &widget->style->black);
    gdk_window_clear(widget->window);

    /* colors */
    itext->fore_gc = gdk_gc_new(widget->window);
    itext->back_gc = gdk_gc_new(widget->window);
    itext->tback_gc = gdk_gc_new(widget->window);
    gdk_gc_set_fill(itext->tback_gc, GDK_SOLID);
    gdk_gc_set_exposures(itext->back_gc, 1);

    /* foreground selection color - white */
    memcpy(&itext->select_fg, &widget->style->white, sizeof(GdkColor));
    itext->select_fg.pixel = 0x00ffffff;
    gdk_color_alloc(gtk_widget_get_colormap(widget), &itext->select_fg);

    /* background selection color - dark blue */
    itext->select_bg.red = itext->select_bg.green = 0;
    itext->select_bg.blue = 0x9000;
    itext->select_bg.pixel = 0x00000090;
    gdk_color_alloc(gtk_widget_get_colormap(widget), &itext->select_bg);

    /* foreground selection color when we don't own selection - black */
    memcpy(&itext->nselect_fg, &widget->style->black, sizeof(GdkColor));
    itext->nselect_fg.pixel = 0x00000000;
    gdk_color_alloc(gtk_widget_get_colormap(widget), &itext->nselect_fg);

    /* background selection color when we don't own selection - grey */
    itext->nselect_bg.red = itext->nselect_bg.green = itext->nselect_bg.blue = 0x9000;
    itext->nselect_bg.pixel = 0x00909090;
    gdk_color_alloc(gtk_widget_get_colormap(widget), &itext->nselect_bg);

    gtk_itext_init_background(itext);
}

static void gtk_itext_unrealize (GtkWidget *widget)
{
    GtkIText *itext;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_ITEXT(widget));

    itext = GTK_ITEXT(widget);

    gdk_gc_destroy(itext->back_gc);
    gdk_gc_destroy(itext->tback_gc);
    gdk_gc_destroy(itext->fore_gc);
    itext->back_gc = itext->tback_gc = itext->fore_gc = NULL;

    if (itext->pixmap != NULL)
    {
	if (itext->bg_flags & ITEXT_BG_TRANSPARENT)
	    itext_transparent_free(itext);
	else
	    gdk_pixmap_unref(itext->pixmap);
	itext->pixmap = NULL;
    }

    if (GTK_WIDGET_CLASS(parent_class)->unrealize)
	(*GTK_WIDGET_CLASS(parent_class)->unrealize)(widget);
}

static void gtk_itext_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
    GtkIText *itext;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_ITEXT(widget));
    g_return_if_fail(requisition != NULL);

    itext = GTK_ITEXT(widget);

    d(printf("gtk_itext_size_request()\n"));
    requisition->width = 300;
    requisition->height = 300;
}

static void gtk_itext_resize_text(GtkIText *itext)
{
    GtkITextLine *line, *dline;
    GtkITextLineCol *col, *c2;
    GString *str;
    GList *tmp, *cols;

    /* move all lines to lines_add buffer and redraw them */
    dline = NULL; line = NULL;
    str = g_string_new(NULL);
    for (tmp = itext->lines; tmp != NULL; tmp = tmp->next)
    {
	line = tmp->data;

	if (line->first)
	{
	    if (dline != NULL)
	    {
		/* save text string to line */
		g_string_append_c(str, '\n');
		g_free(dline->str);
		dline->str = g_strdup(str->str);
		dline->length = str->len;
	    }
	    g_string_assign(str, line->str);

	    dline = line;
	    itext->lines_add = g_list_append(itext->lines_add, dline);
	}
	else
	{
	    /* continue next line */
	    for (cols = line->colors; cols != NULL; cols = cols->next)
	    {
		col = cols->data;

		col->pos += str->len;
		c2 = g_list_last(dline->colors)->data;
                dline->colors = g_list_append(dline->colors, col);
	    }
	    g_string_append(str, line->str);

	    g_list_free(line->colors);
	    g_free(line->str);
	    g_free(line);
	}
    }

    if (dline != NULL)
    {
	/* save text string to line */
	if (!line->cont) g_string_append_c(str, '\n');
	g_free(dline->str);
	dline->str = g_strdup(str->str);
	dline->length = str->len;
    }

    g_string_free(str, TRUE);

    /* Remove all lines .. */
    g_list_free(itext->lines);
    itext->lines = NULL;
    itext->last_line = NULL;
    itext->linecount = 0;
}

static void gtk_itext_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
    GtkIText *itext;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_ITEXT(widget));
    g_return_if_fail(allocation != NULL);

    d(printf("gtk_itext_size_allocate()\n"));

    itext = GTK_ITEXT(widget);

    if (GTK_WIDGET_REALIZED(widget))
    {
	gdk_window_move_resize(widget->window,
			       allocation->x, allocation->y,
			       allocation->width, allocation->height);

	if (itext->width != allocation->width || itext->height != allocation->height)
	{
	    if (itext->width != allocation->width)
	    {
		d(printf(" - resize\n"));
		gtk_itext_resize_text(itext);
	    }

	    itext->width = allocation->width;
	    itext->height = allocation->height;
	    gtk_itext_init_background(itext);
	}
    }
    widget->allocation = *allocation;
}

static GList *gtk_itext_get_line(GtkIText *itext, gint findline)
{
    GList *pos;

    g_return_val_if_fail(GTK_IS_ITEXT(itext), NULL);
    if (findline < 0 || findline >= itext->linecount) return NULL;

    pos = NULL;
    if (itext->last_line != NULL)
    {
	/* try to find the position as fast as possible,
	   last saved line might help us .. */
	if (itext->last_line_num <= findline)
	{
	    /* 0 .. last_line_num .. findline */
            pos = g_list_nth(itext->last_line, findline-itext->last_line_num);
	}
	else if (itext->last_line_num/2 < findline)
	{
	    /* 0 .. findline .. lastline - faster to find it via last_line */
	    gint n;

	    n = itext->last_line_num-findline;
	    for (pos = itext->last_line; pos != NULL && n != 0; pos = pos->prev)
		n--;
	}
    }

    if (pos == NULL)
	pos = g_list_nth(itext->lines, findline);

    if (pos != NULL)
    {
	/* save this line number/position */
	itext->last_line = pos;
	itext->last_line_num = findline;
    }

    return pos;
}

static gint gtk_itext_select_start(GtkIText *itext, GtkITextLine *line, gint xpos, gboolean word)
{
    GtkITextLineCol *linecol;
    GdkFont *font;
    GList *cpos, *npos;
    gint size, fsize, spos, len;
    gchar *p;

    g_return_val_if_fail(GTK_IS_ITEXT(itext), 0);
    g_return_val_if_fail(line != NULL, 0);

    if (!line->first)
	xpos -= line->indent_pix;

    size = 0; font = NULL;
    cpos = line->colors; linecol = cpos->data;
    for (; cpos != NULL; cpos = npos)
    {
	spos = linecol->pos;
	font = (linecol->flags & ITEXT_LINECOL_BOLD) == 0 ?
	    itext->font : itext->font_bold;

	npos = cpos->next;
	linecol = npos == NULL ? NULL : npos->data;
	len = (npos == NULL ? line->length : linecol->pos)-spos;

	fsize = gdk_text_width(font, line->str+spos, len);

	if (size+fsize > xpos)
	{
	    /* right block. */
	    break;
	}

	size += fsize;
    }

    if (cpos == NULL)
    {
        /* clicked at the end of the line */
	xpos = line->length;
    }
    else
    {
	/* find the character clicked */
	linecol = cpos->data;
	for (p = line->str+linecol->pos; *p != '\0'; p++)
	{
	    fsize = gdk_char_width(font, *p);

	    if (size+fsize > xpos)
                break;
	    size += fsize;
	}

	xpos = (gint) (p-line->str);
    }

    if (xpos != line->length && word)
    {
	/* select the whole word.. */
        while (xpos > 0 && !isspace(line->str[xpos-1])) xpos--;
    }

    return xpos;
}

static void gtk_itext_select(GtkIText *itext, gint ypos, gint xpos, gint style)
{
    GtkITextLine *line;
    GList *tmp;
    gint n, start, end;

    g_return_if_fail(GTK_IS_ITEXT(itext));

    if (itext->select_start >= 0)
    {
	/* remove the old selection */
	if (itext->select_start < itext->select_end)
	{
	    start = itext->select_start;
	    end = itext->select_end;
	}
	else
	{
	    start = itext->select_end;
	    end = itext->select_start;
	}
	tmp = gtk_itext_get_line(itext, start);
        for (n = start; n <= end && tmp != NULL; n++, tmp = tmp->next)
            ((GtkITextLine *) (tmp->data))->select_start = -1;
	itext->select_start = -1;

	/* remove selection from screen */
	itext->refresh_first = start;
	itext->refresh_last = end+1;
	gtk_itext_refresh(itext);
    }

    /* find the line */
    ypos = ypos/itext->font_height + itext->starty;
    tmp = gtk_itext_get_line(itext, ypos);
    if (tmp == NULL)
    {
	tmp = g_list_last(itext->lines);
	if (tmp == NULL) return;
    }
    line = tmp->data;

    /* get the first word */
    line->select_start = gtk_itext_select_start(itext, line, xpos, style != 0);

    if (style == 0)
    {
	/* just mark the selection start, don't really select anything. */
        line->select_end = line->select_start;
    }
    else if (style == 1)
    {
	/* select word */
	xpos = line->select_start;
	while (xpos < line->length && !isspace(line->str[xpos])) xpos++;
	line->select_end = xpos;
    }
    else
    {
	/* select word + the rest of the line */
	line->select_end = line->length;
    }

    itext->select_start = ypos;
    itext->select_end = ypos;
    itext->selecting = TRUE;

    if (line->select_start != line->select_end)
    {
	/* draw the selection */
	itext->refresh_first = ypos;
	itext->refresh_last = ypos+1;
	gtk_itext_refresh(itext);
    }
}

static gint gtk_itext_button_press(GtkWidget *widget, GdkEventButton *event)
{
    GtkIText *itext;
    GtkITextLine *line;
    GdkModifierType mask;
    gint x, y, val, max, start, end;
    GString *word;
    GList *lines;
    gchar c;

    g_return_val_if_fail(GTK_IS_ITEXT(widget), FALSE);
    g_return_val_if_fail(event != NULL, FALSE);

    itext = GTK_ITEXT(widget);

    switch (event->button)
    {
	case 4:
	    /* wheel mouse scroll up */
	    val = itext->adjustment->value - 5;
	    gtk_adjustment_set_value(itext->adjustment, val > 0 ? val : 0);
	    break;
	case 5:
	    /* wheel mouse scroll down */
	    val = itext->adjustment->value + 5;
	    max = itext->adjustment->upper - itext->adjustment->lower - itext->adjustment->page_size;
	    gtk_adjustment_set_value(itext->adjustment, val <= max ? val : max);
	    break;
    }

    /* get mouse position */
    gdk_window_get_pointer(widget->window, &x, &y, &mask);

    itext->own_selection = TRUE;
    if (event->button == 1)
    {
	gtk_grab_add(widget);
	gdk_pointer_grab(widget->window, TRUE,
			 GDK_BUTTON_RELEASE_MASK |
			 GDK_BUTTON_MOTION_MASK |
			 GDK_POINTER_MOTION_HINT_MASK,
			 NULL, NULL, 0);

	switch (event->type)
	{
	    case GDK_BUTTON_PRESS:
		/* start selection */
		gtk_itext_select(itext, y, x, 0);
		break;
	    case GDK_2BUTTON_PRESS:
		/* select the clicked word */
		gtk_itext_select(itext, y, x, 1);
		break;
	    case GDK_3BUTTON_PRESS:
		/* select the clicked word + the rest of the line after it */
		gtk_itext_select(itext, y, x, 2);
		break;
	    default:
		break;
	}
    }

    /* get the clicked character.. */
    y = y/itext->font_height + itext->starty;
    lines = gtk_itext_get_line(itext, y);
    if (lines == NULL)
    {
	/* missed */
	gtk_signal_emit(GTK_OBJECT(itext), itext_signals[WORD_CLICKED], NULL, event);
	return FALSE;
    }

    line = lines->data;
    start = gtk_itext_select_start(itext, line, x, FALSE);

    /* get the clicked word */
    word = g_string_new(NULL);

    /* get the word start position/line */
    for (;;)
    {
	while (start > 0 && !iswordcut(line->str[start-1])) start--;

	if (start > 0 || line->first || lines->prev == NULL)
	    break;

	lines = lines->prev;
	line = lines->data;
	start = line->length;
    }

    /* get the word end position/line */
    end = start;
    for (;;)
    {
	while (end < line->length && !iswordcut(line->str[end])) end++;

	c = line->str[end];
	line->str[end] = '\0';
	g_string_append(word, line->str+start);
	line->str[end] = c;

	lines = lines->next;
	if (end < line->length || !line->cont || lines == NULL)
	    break;

	line = lines->data;
	start = end = 0;
    }

    gtk_signal_emit(GTK_OBJECT(itext), itext_signals[WORD_CLICKED], word->str, event);
    g_string_free(word, TRUE);

    return FALSE;
}

static gint gtk_itext_button_release(GtkWidget *widget, GdkEventButton *event)
{
    GtkIText *itext;
    GtkITextLine *line;
    GList *tmp;

    g_return_val_if_fail(GTK_IS_ITEXT(widget), FALSE);
    g_return_val_if_fail(event != NULL, FALSE);

    itext = GTK_ITEXT(widget);
    itext->selecting = FALSE;

    if (event->button != 1 || itext->select_start == -1)
	return FALSE;

    /* release mouse pointer */
    gtk_grab_remove(widget);
    gdk_pointer_ungrab(0);

    /* text selected .. maybe */
    if (itext->select_start == itext->select_end)
    {
	tmp = gtk_itext_get_line(itext, itext->select_start);
	line = tmp == NULL ? NULL : tmp->data;

	if (line != NULL && line->select_start == line->select_end)
	{
	    line->select_start = -1;
	    line = NULL;
	}
	if (line == NULL)
	{
	    /* no text selected */
	    itext->own_selection = FALSE;
	    return FALSE;
	}
    }

    itext->own_selection = TRUE;
    gtk_selection_owner_set(widget, GDK_SELECTION_PRIMARY, event->time);
    return FALSE;
}

static gint gtk_itext_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
    GtkIText *itext;
    GtkITextLine *line, *tmpline;
    GdkModifierType mask;
    gint xpos, ypos, pos, start, end;
    GList *tmp;

    g_return_val_if_fail(GTK_IS_ITEXT(widget), FALSE);
    g_return_val_if_fail(event != NULL, FALSE);

    itext = GTK_ITEXT(widget);
    if (!itext->selecting)
    {
	/* not selecting text, ignore. */
	return FALSE;
    }

    /* get mouse position */
    gdk_window_get_pointer(widget->window, &xpos, &ypos, &mask);
    if (ypos < 0) ypos = 0;

    if (ypos <= 0)
    {
	pos = itext->adjustment->value-1;
	gtk_adjustment_set_value(itext->adjustment, pos > 0 ? pos : 0);
    }
    else if (ypos >= itext->height)
    {
	gint max;

	pos = itext->adjustment->value+1;
	max = itext->adjustment->upper - itext->adjustment->lower - itext->adjustment->page_size;
	gtk_adjustment_set_value(itext->adjustment, pos <= max ? pos : max);
    }

    /* get line */
    ypos = ypos/itext->font_height + itext->starty;
    tmp = ypos >= itext->linecount ? NULL : gtk_itext_get_line(itext, ypos);
    if (tmp == NULL)
    {
	tmp = g_list_last(itext->lines);
	if (tmp == NULL) return FALSE;
	ypos = itext->linecount-1;
	xpos = itext->width;
    }
    line = tmp->data;

    /* set the initial refresh area.. */
    if (itext->select_start < itext->select_end)
    {
        start = itext->select_start;
        end = itext->select_end;
    }
    else
    {
        start = itext->select_end;
        end = itext->select_start;
    }

    if (start < ypos)
    {
        itext->refresh_first = start;
        itext->refresh_last = ypos+1;
    }
    else if (itext->select_start > ypos)
    {
        itext->refresh_first = ypos;
        itext->refresh_last = end+1;
    }
    else
    {
        itext->refresh_first = start;
        itext->refresh_last = end+1;
    }

    if (itext->select_end > ypos && itext->select_end > itext->select_start)
    {
	/* selected less lines than last time (from end) */
	tmp = gtk_itext_get_line(itext, ypos+1);
	for (pos = ypos+1; pos <= itext->select_end && tmp != NULL; pos++, tmp = tmp->next)
	{
	    tmpline = tmp->data;
	    tmpline->select_start = -1;
	}
	itext->refresh_last = itext->select_end+1;
    }

    if (itext->select_end < ypos && itext->select_end < itext->select_start)
    {
	/* selected less lines than last time (from start) */
	tmp = gtk_itext_get_line(itext, itext->select_end);
	for (pos = itext->select_end; pos < ypos && tmp != NULL; pos++, tmp = tmp->next)
	{
	    tmpline = tmp->data;
	    tmpline->select_start = -1;
	}
	itext->refresh_first = itext->select_end;
    }

    itext->select_end = ypos;
    if (line->select_start == -1)
	line->select_end = -1;

    if (itext->select_start < itext->select_end)
    {
	/* update the selection between start and end */
	ypos = itext->select_start+1;
	tmp = gtk_itext_get_line(itext, ypos);
	for (; ypos < itext->select_end && tmp != NULL; ypos++, tmp = tmp->next)
	{
	    tmpline = tmp->data;
	    tmpline->select_start = 0;
	    tmpline->select_end = tmpline->length;
	}

	/* last line */
	line->select_start = 0;

	/* first line */
	tmp = gtk_itext_get_line(itext, itext->select_start);
	if (tmp != NULL)
	{
	    tmpline = tmp->data;
            tmpline->select_end = tmpline->length;
	}
    }
    else if (itext->select_start > itext->select_end)
    {
	/* update the selection between start and end */
	ypos = itext->select_end+1;
	tmp = gtk_itext_get_line(itext, ypos);
	for (; ypos < itext->select_start && tmp != NULL; ypos++, tmp = tmp->next)
	{
	    tmpline = tmp->data;
	    tmpline->select_start = 0;
	    tmpline->select_end = tmpline->length;
	}

	/* last line */
	line->select_start = line->length;

	/* first line */
	tmp = gtk_itext_get_line(itext, itext->select_start);
	if (tmp != NULL)
	{
	    tmpline = tmp->data;
	    tmpline->select_end = 0;
	}
    }

    /* get character position where mouse is */
    xpos = gtk_itext_select_start(itext, line, xpos, FALSE);
    if (xpos < line->length && line->select_start <= xpos) xpos++;
    if (line->select_end != xpos)
	line->select_end = xpos;

    if (itext->refresh_first != -1)
	gtk_itext_refresh(itext);

    return FALSE;
}

static gint gtk_itext_selection_clear(GtkWidget *widget, GdkEventSelection *event)
{
    GtkIText *itext;

    g_return_val_if_fail(GTK_IS_ITEXT(widget), FALSE);
    g_return_val_if_fail(event != NULL, FALSE);

    /* Let the selection handling code know that the selection
       has been changed, since we've overriden the default handler */
    if (!gtk_selection_clear (widget, event))
	return FALSE;

    itext = GTK_ITEXT(widget);
    itext->own_selection = FALSE;

    if (itext->select_start == -1)
	return TRUE;

    /* redraw the selection with the non-owned selection color */
    if (itext->select_start < itext->select_end)
    {
	itext->refresh_first = itext->select_start;
	itext->refresh_last = itext->select_end+1;
    }
    else if (itext->select_start >= itext->select_end)
    {
	itext->refresh_first = itext->select_end;
	itext->refresh_last = itext->select_start+1;
    }

    gtk_itext_refresh(itext);
    return TRUE;
}

static void gtk_itext_selection_get(GtkWidget *widget, GtkSelectionData *selection_data_ptr, guint info, guint time)
{
    GtkIText *itext;
    GtkITextLine *line;
    GString *str;
    GList *lines;
    gint n, lstart, lend, start, end;
    gboolean lf;
    gchar c;

    g_return_if_fail(GTK_IS_ITEXT(widget));
    g_return_if_fail(selection_data_ptr != NULL);

    itext = GTK_ITEXT(widget);

    str = g_string_new(NULL);

    if (itext->select_start < itext->select_end)
    {
        lstart = itext->select_start;
        lend = itext->select_end;
    }
    else
    {
        lstart = itext->select_end;
        lend = itext->select_start;
    }

    lines = gtk_itext_get_line(itext, lstart); lf = FALSE;
    for (n = lstart; n <= lend && lines != NULL; n++, lines = lines->next)
    {
	line = lines->data;

	if (lf)
	{
	    /* add line feed */
	    g_string_append_c(str, '\n');
	}

	if (line->select_start == -1)
	    continue;

	if (line->select_start < line->select_end)
	{
            start = line->select_start;
            end = line->select_end;
	}
	else
	{
            start = line->select_end;
            end = line->select_start;
	}

	/* append selected text to string .. */
	c = line->str[end];
	line->str[end] = '\0';
	g_string_append(str, line->str+start);
	line->str[end] = c;

	if (!line->cont)
	    lf = TRUE;
    }

    gtk_selection_data_set(selection_data_ptr, GDK_SELECTION_TYPE_STRING,
			    8, str->str, str->len);
    g_string_free(str, TRUE);
}

static void gtk_itext_append_lines(GtkIText *itext)
{
    GtkITextLine *line;
    GtkITextLineCol *linecol, *linecol2;
    GdkColor *bg;
    GList *lines;
    gint len, pos;
    gchar *s;

    g_return_if_fail(GTK_IS_ITEXT(itext));

    gtk_itext_freeze(itext);

    lines = itext->lines_add;
    itext->lines_add = NULL;
    while (lines != NULL)
    {
	line = lines->data;

	pos = 0; s = g_malloc(line->length+1);
	while (line->colors != NULL)
	{
	    linecol = line->colors->data;
	    linecol2 = line->colors->next == NULL ? NULL : line->colors->next->data;

	    len = (linecol2 == NULL ? line->length : linecol2->pos)-linecol->pos;
	    memcpy(s, line->str+pos, len); s[len] = '\0';
	    bg = (linecol->flags & ITEXT_LINECOL_BG) ? &linecol->bg : NULL;
	    gtk_itext_append(itext, s, &linecol->fg, bg, linecol->flags, line->indent_chars);

	    g_free(linecol);
	    line->colors = g_list_remove(line->colors, linecol);
	    pos += len;
	}
	g_free(s);

	g_free(line->str);
	g_free(line);

	lines = g_list_remove(lines, line);
    }

    gtk_itext_thaw(itext);
}

static GList *line_get_selection_colors(GtkIText *itext, GtkITextLine *line)
{
    GtkITextLineCol *col, *col2, *newcol;
    GList *tmp, *list;
    gint start, end;
    gint selfont;

    g_return_val_if_fail(GTK_IS_ITEXT(itext), NULL);
    g_return_val_if_fail(line != NULL, NULL);

    if (line->select_start < line->select_end)
    {
	start = line->select_start;
	end = line->select_end;
    }
    else
    {
	start = line->select_end;
	end = line->select_start;
    }

    list = NULL; col = col2 = NULL;
    selfont = -1; newcol = NULL;
    for (tmp = line->colors; ; tmp = tmp->next)
    {
	col = tmp == NULL ? NULL : tmp->data;

        if (selfont == -1 && (col == NULL || col->pos >= start))
	{
            /* inside selection */
	    newcol = g_new(GtkITextLineCol, 1);
	    newcol->fg = itext->own_selection ? itext->select_fg : itext->nselect_fg;
	    newcol->bg = itext->own_selection ? itext->select_bg : itext->nselect_bg;
	    newcol->flags = ITEXT_LINECOL_BG |
		(col != NULL && (col->pos == start || col2 == NULL) ? col->flags : col2->flags);
	    newcol->pos = start;

	    selfont = newcol->flags & ITEXT_LINECOL_BOLD;
	    list = g_list_append(list, newcol);
	}

	if (tmp == NULL)
	    break;

	/* outside selection, just print the color */
	if (col->pos != start)
	{
	    if (selfont != -2 && col->pos > end && col2 != NULL)
	    {
		newcol = g_new(GtkITextLineCol, 1);
		memcpy(newcol, col2, sizeof(GtkITextLineCol));
		newcol->pos = end;
		list = g_list_append(list, newcol);
		selfont = -2;
	    }

	    newcol = g_new(GtkITextLineCol, 1);
	    memcpy(newcol, col, sizeof(GtkITextLineCol));

	    if (col->pos > start && col->pos < end)
	    {
		newcol->fg = itext->own_selection ? itext->select_fg : itext->nselect_fg;
		newcol->bg = itext->own_selection ? itext->select_bg : itext->nselect_bg;
		newcol->flags |= ITEXT_LINECOL_BG;
	    }
	    list = g_list_append(list, newcol);
	}

	col2 = col;
    }

    if (selfont != -2 && col2 != NULL)
    {
	newcol = g_new(GtkITextLineCol, 1);
	memcpy(newcol, col2, sizeof(GtkITextLineCol));
	newcol->pos = end;
	list = g_list_append(list, newcol);
    }

    return list;
}

static void draw(GtkWidget *widget, gint xp1, gint yp1, gint xp2, gint yp2)
{
    GtkIText *itext;
    GtkITextLine *line;
    GtkITextLineCol *linecol, *linecol2;
    GList *pos, *cpos, *clist;
    gint xpos, y, y2, len, size;
    GdkFont *font;

    g_return_if_fail(GTK_IS_ITEXT(widget));

    itext = GTK_ITEXT(widget);

    if (!GTK_WIDGET_MAPPED(widget) || !GTK_WIDGET_DRAWABLE(widget))
	return;

    /* first add the missing lines .. */
    if (itext->width >= itext->min_width && itext->lines_add != NULL)
	gtk_itext_append_lines(itext);

    /* clear area */
    gtk_itext_set_origin(itext);
    gdk_draw_rectangle(GTK_WIDGET(itext)->window, itext->back_gc, 1,
		       xp1, yp1, xp2-xp1+1, yp2-yp1+1);

    y = yp1 < itext->font_height ? 0 : yp1/itext->font_height-1;
    y2 = yp2/itext->font_height;
    if (yp2 >= itext->height) y2--;
    d(printf("draw %d..%d\n", y, y2));

    pos = gtk_itext_get_line(itext, itext->starty+y);
    for (; pos != NULL && y <= y2; pos = pos->next, y++)
    {
	line = pos->data;

	if (line->select_start == -1 || line->select_start == line->select_end)
	{
	    clist = NULL;
	    cpos = line->colors;
	}
	else
	{
	    /* selection in line - get the combined selection + normal colors */
            clist = cpos = line_get_selection_colors(itext, line);
	}

	xpos = line->first ? 0 : line->indent_pix;
	linecol2 = cpos->data;
	while (cpos != NULL)
	{
	    linecol = linecol2;

	    font = (linecol->flags & ITEXT_LINECOL_BOLD) == 0 ?
		itext->font : itext->font_bold;

	    /* get text width */
	    cpos = cpos->next;
	    linecol2 = cpos == NULL ? NULL : cpos->data;
	    len = (cpos == NULL ? line->length : linecol2->pos)-linecol->pos;
	    size = gdk_text_width(font, line->str+linecol->pos, len);

	    /* check if we need to draw this */
	    if (xpos > xp2)
		break;
	    if (xpos+size >= xp1)
	    {
		/* set foreground color */
		gdk_gc_set_foreground(itext->fore_gc, &linecol->fg);

		/* draw background */
		if (linecol->flags & ITEXT_LINECOL_BG)
		{
		    gdk_gc_set_foreground(itext->tback_gc, &linecol->bg);
		    gdk_draw_rectangle(widget->window, itext->tback_gc, 1,
				       xpos, itext->font_height*y, size, itext->font_height);
		}

		/* draw the text */
		gdk_draw_text(widget->window, font, itext->fore_gc,
			      xpos, itext->font_height*(y+1)-2,
			      line->str+linecol->pos, len);

		if (linecol->flags & ITEXT_LINECOL_UNDERLINE)
		{
		    /* underline */
		    gdk_draw_line(widget->window, itext->fore_gc,
				  xpos, itext->font_height*(y+1)-1,
				  xpos+size, itext->font_height*(y+1)-1);
		}
	    }
	    xpos += size;
	}

	if (clist != NULL)
	{
	    /* free memory used by selection colors list */
	    g_list_foreach(clist, (GFunc) g_free, NULL);
	    g_list_free(clist);
	}
    }
}

static void gtk_itext_set_origin(GtkIText *itext)
{
    Window childret;
    gint x, y;

    g_return_if_fail(GTK_IS_ITEXT(itext));

    if (itext->bg_flags & ITEXT_BG_SCROLLABLE)
    {
	gdk_gc_set_ts_origin(itext->back_gc, 0, -itext->starty*itext->font_height);
	return;
    }

    if ((itext->bg_flags & ITEXT_BG_TRANSPARENT) == 0)
	return;

    XTranslateCoordinates(GDK_WINDOW_XDISPLAY(GTK_WIDGET(itext)->window),
			  GDK_WINDOW_XWINDOW(GTK_WIDGET(itext)->window),
			  GDK_ROOT_WINDOW(), 0, 0, &x, &y, &childret);

    gdk_gc_set_ts_origin(itext->back_gc, -x, -y);
}

static gint gtk_itext_expose(GtkWidget *widget, GdkEventExpose *event)
{
    GtkIText *itext;

    g_return_val_if_fail(widget != NULL, FALSE);
    g_return_val_if_fail(GTK_IS_ITEXT(widget), FALSE);
    g_return_val_if_fail(event != NULL, FALSE);

    if (!GTK_WIDGET_DRAWABLE(widget))
	return FALSE;

    itext = GTK_ITEXT(widget);

    d(printf("exposed!  (%d,%d) %dx%d\n",
	   event->area.x,
	   event->area.y,
	   event->area.width,
	   event->area.height));

    draw(widget, event->area.x, event->area.y, event->area.x+event->area.width, event->area.y+event->area.height);

    return FALSE;
}

static void gtk_itext_draw(GtkWidget *widget, GdkRectangle *area)
{
    GtkIText *itext;
    Window childret;
    gint x, y;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_ITEXT(widget));
    g_return_if_fail(area != NULL);

    if (!GTK_WIDGET_DRAWABLE(widget))
	return;

    itext = GTK_ITEXT(widget);

    d(printf("draw!  (%d,%d) %dx%d\n", area->x, area->y, area->width, area->height));

    if (itext->bg_flags & ITEXT_BG_TRANSPARENT)
    {
	itext_transparent_free(itext);
	itext->pixmap = get_background(itext);

	XTranslateCoordinates (GDK_WINDOW_XDISPLAY(widget->window),
			       GDK_WINDOW_XWINDOW(widget->window),
			       GDK_ROOT_WINDOW(),
			       0, 0,
			       &x, &y,
			       &childret);
	gdk_gc_set_ts_origin(itext->back_gc, -x, -y);
	gdk_gc_set_tile(itext->back_gc, itext->pixmap);
	gdk_gc_set_fill(itext->back_gc, GDK_TILED);
    }
    draw(widget, area->x, area->y, area->x+area->width, area->y+area->height);
}

GtkWidget *gtk_itext_new(void)
{
    GtkIText *itext;

    itext = gtk_type_new(gtk_itext_get_type());
    return GTK_WIDGET(itext);
}

static gint get_text_size(GtkIText *itext, gchar *str, gint len, gboolean first, gint width, gint avg_width, gint min_chars, GdkFont *font)
{
    gint pos, guess;

    g_return_val_if_fail(GTK_IS_ITEXT(itext), 0);
    g_return_val_if_fail(str != NULL, 0);
    g_return_val_if_fail(font != NULL, 0);

    /* guess some good value */
    guess = width/avg_width;
    if (guess < min_chars) guess = min_chars;
    if (guess > len) guess = len;

    /* go to last character of the word */
    while (guess > min_chars && isspace(str[guess])) guess--;
    while (guess < len && !isspace(str[guess])) guess++;

    pos = 0;
    while (pos < len && ((first && guess < min_chars) || gdk_text_width(font, str, guess) < width))
    {
	/* we need more text */
	pos = guess;
	if (pos == len)
	{
	    /* all fits! */
	    return len;
	}

        while (isspace(str[pos])) pos++;

	guess = pos;
        while (guess < len && !isspace(str[guess])) guess++;
    }

    if (pos == 0)
    {
	/* we need less text */

	/* if this is the start of line, default to first word. if not,
	   default to zero so we can add this text to next line .. */
	pos = first || min_chars > 1 ? guess : 0;
	for (;;)
	{
	    if (gdk_text_width(font, str, guess) <= width)
	    {
		pos = guess;
		break;
	    }

	    while (guess > min_chars && !isspace(str[guess-1])) guess--;
	    while (guess > min_chars && isspace(str[guess-1])) guess--;
	    if (guess <= min_chars) break;
	}
    }

    /* check if there was only one word and it was longer than what
       fits to screen.. */
    guess = pos;
    while (guess > 0 && isspace(str[guess])) guess--;
    if (guess > 0 && width >= itext->min_width && gdk_text_width(font, str, guess) > width)
    {
	/* check if it's the only word in this line? */
	for (; guess > 0; guess--)
	    if (isspace(str[guess])) break;

	if (guess == 0 || (first && guess < min_chars))
	{
	    /* split the word .. */
	    for (guess = pos; guess > 1; guess--)
	    {
		if (gdk_text_width(font, str, guess) < width)
		    break;
	    }
	    if (guess > 1) pos = guess;
	}
    }

    /* trailing spaces always fit */
    while (pos > 0 && pos < len && isspace(str[pos])) pos++;

    return pos;
}

void gtk_itext_append_str(GtkIText *itext, gchar *str, GdkColor *fg, GdkColor *bg, gint flags, gint indent)
{
    GtkITextLine *line, *oldline;
    GtkITextLineCol *col, *col2;
    GdkFont *font;
    GList *tmp;
    gint pos, len, slen, width, pixels, avg_width, min_chars;
    gboolean first, linestart, last_lf;
    gchar *pstr;

    g_return_if_fail(GTK_IS_ITEXT(itext));
    g_return_if_fail(str != NULL);
    g_return_if_fail(fg != NULL);

    if (*str == '\0') return;

    /* Get font to use */
    font = (flags & ITEXT_LINECOL_BOLD) == 0 ?
	itext->font : itext->font_bold;
    avg_width = (flags & ITEXT_LINECOL_BOLD) == 0 ?
	itext->font_width : itext->bfont_width;

    /* Check if we can continue to last line */
    line = itext->lines == NULL ? NULL :
	g_list_last(itext->lines)->data;
    if (line != NULL && !line->cont) line = NULL;
    first = line == NULL;
    linestart = first;

    /* how much still fits to line */
    width = itext->width;
    if (line != NULL)
    {
	width -= line->pixels;
	if (!line->first)
	    width -= line->indent_pix;
    }

    /* get text length, check if it ends with LF */
    slen = strlen(str);
    last_lf = str[slen-1] == '\n';

    oldline = line;
    for (; *str != '\0'; str += len, slen -= len)
    {
	if (last_lf && str[slen-1] == '\n')
	{
	    slen--;
	    str[slen] = '\0';
	}

	/* find out how much text we can add here.. */
	if (line == NULL || !line->first || line->length >= line->indent_chars)
	    min_chars = 1;
	else
	    min_chars = line->indent_chars-line->length;
	len = (slen == 0 || (width <= 1 && !first)) ? 0 :
	    get_text_size(itext, str, slen, first, width, avg_width, min_chars, font);
	pixels = len == 0 ? 0 : gdk_text_width(font, str, len);

	width = itext->width;
	first = TRUE;

	/* create the color for this line */
	col = g_new0(GtkITextLineCol, 1);
	memcpy(&col->fg, fg, sizeof(GdkColor));
	col->flags = flags;
	if (bg != NULL)
	{
            col->flags |= ITEXT_LINECOL_BG;
	    memcpy(&col->bg, bg, sizeof(GdkColor));
	}

	if (len == 0 && slen != 0)
	{
	    /* need to split to next line */
	    line->cont = TRUE;

	    if (line->last_space_pixels == 0)
	    {
		/* last line ended to space, just create a new line. */
		oldline = NULL;
		linestart = FALSE;
		width -= line->indent_pix;
		g_free(col);
		continue;
	    }

            /* move the last word to start of next line */
	    oldline = line;

	    line = g_new0(GtkITextLine, 1);
	    itext->lines = g_list_append(itext->lines, line);
	    itext->linecount++;

	    line->str = g_strdup(oldline->str+oldline->last_space+1);
	    line->length = oldline->length - oldline->last_space - 1;

	    line->first = FALSE;
	    line->cont = TRUE;
	    line->pixels = oldline->last_space_pixels;

	    line->indent_chars = oldline->indent_chars;
	    line->indent_pix = oldline->indent_pix;
	    line->select_start = -1;

	    col->pos = line->length;

	    /* truncate old line */
	    oldline->length = oldline->last_space+1;
	    oldline->str[oldline->length] = '\0';

	    /* move colors to next line */
	    for (tmp = g_list_last(oldline->colors); tmp != NULL; tmp = tmp->prev)
	    {
		col2 = tmp->data;

		if (col2->pos < oldline->length)
		    break;
	    }

	    if (tmp != NULL && tmp->next != NULL)
	    {
                /* split the colors */
                tmp = tmp->next;
		tmp->prev->next = NULL;
		tmp->prev = NULL;
		line->colors = tmp;

                /* move the colors' position */
		for (tmp = line->colors; tmp != NULL; tmp = tmp->next)
		{
		    col2 = tmp->data;

		    col2->pos -= oldline->length;
		}
	    }

	    line->colors = g_list_append(line->colors, col);

	    col = line->colors->data;
	    if (col->pos != 0)
	    {
		/* we need a color at the start of the line */
		col2 = g_new(GtkITextLineCol, 1);
		memcpy(col2, g_list_last(oldline->colors)->data, sizeof(GtkITextLineCol));
		col2->pos = 0;
		line->colors = g_list_prepend(line->colors, col2);
	    }

	    width -= oldline->last_space_pixels;

	    oldline = line;
	    first = FALSE;
	}
	else if (oldline != NULL)
	{
	    /* append text after the last line */
	    line = oldline;

	    if (line->first && line->indent_chars > line->length)
	    {
		gint max;

		max = line->indent_chars-line->length;
		if (max > len) max = len;
		line->indent_pix += gdk_text_width(font, str, max);
	    }

	    col->pos = line->length;
	    line->colors = g_list_append(line->colors, col);

	    pstr = g_malloc(line->length+len+1);
	    memcpy(pstr, line->str, line->length);
	    memcpy(pstr+line->length, str, len);

	    line->pixels += pixels;
	    line->length += len;

	    pstr[line->length] = '\0';
	    g_free(line->str);
	    line->str = pstr;

	    oldline = NULL;
	}
	else
	{
	    /* create new line, add the text after it. */
            line = g_new0(GtkITextLine, 1);
	    itext->lines = g_list_append(itext->lines, line);
	    itext->linecount++;

	    line->str = g_malloc(len+1);
	    memcpy(line->str, str, len); line->str[len] = '\0';
	    line->length = len;

	    line->first = linestart;
	    line->cont = slen != 0 || !last_lf;
	    line->pixels = pixels;
	    line->select_start = -1;

	    if (slen == 0) last_lf = FALSE;

	    if (linestart)
	    {
		line->indent_chars = indent;
		line->indent_pix = gdk_text_width(font, str, indent > len ? len : indent);
	    }
	    else
	    {
		oldline = g_list_last(itext->lines)->prev->data;
		line->indent_chars = oldline->indent_chars;
		line->indent_pix = oldline->indent_pix;
	    }

	    col->pos = 0;
	    line->colors = g_list_append(line->colors, col);
	    oldline = NULL;
	}

	width -= line->indent_pix;
	linestart = FALSE;

	/* get how much text fits after the last space */
	line->last_space_pixels = 0;
	line->last_space = 0;

	min_chars = line->first ? line->indent_chars : 1;
	for (pos = line->length-1; pos > min_chars; pos--)
	{
	    if (isspace(line->str[pos]))
	    {
		if (line->last_space != pos)
		{
		    line->last_space = pos;
		    pos++;
		    line->last_space_pixels =
			pos == line->length ? 0 : gdk_text_width(font, line->str+pos, line->length-pos);
		}
		break;
	    }
	}
    }

    if (last_lf)
	line->cont = FALSE;
}

void gtk_itext_append(GtkIText *itext, gchar *str, GdkColor *fg, GdkColor *bg, gint flags, gint indent)
{
    GtkITextLine *line;
    GtkITextLineCol *linecol;
    gchar *old, *p, c;

    g_return_if_fail(GTK_IS_ITEXT(itext));
    g_return_if_fail(str != NULL);
    g_return_if_fail(fg != NULL);

    if (!GTK_WIDGET_REALIZED(GTK_WIDGET(itext)) || itext->lines_add != NULL || itext->width < itext->min_width)
    {
	/* can't append the lines now, do it later. */
	linecol = g_new0(GtkITextLineCol, 1);
	memcpy(&linecol->fg, fg, sizeof(GdkColor));
        linecol->flags = flags;
	linecol->pos = 0;

	if (bg != NULL)
	{
	    linecol->flags |= ITEXT_LINECOL_BG;
	    memcpy(&linecol->bg, bg, sizeof(GdkColor));
	}

	line = g_new0(GtkITextLine, 1);
	line->str = g_strdup(str);
	line->length = strlen(str);
	line->indent_chars = indent;
	line->colors = g_list_append(NULL, linecol);
	line->select_start = -1;
	itext->lines_add = g_list_append(itext->lines_add, line);
	return;
    }

    /* get the first line that need refreshing */
    itext->refresh_first = itext->linecount;
    if (itext->lines != NULL)
    {
	line = g_list_last(itext->lines)->data;
        if (line->cont) itext->refresh_first--;
    }

    gtk_itext_freeze(itext);
    old = str = g_strdup(str); c = 0;
    do
    {
	p = strchr(str, '\n');
	if (p != NULL)
	{
	    c = p[1];
	    p[1] = '\0';
	}

	gtk_itext_append_str(itext, str, fg, bg, flags, indent);

	if (p != NULL)
	    p[1] = c;
	str = p+1;
    }
    while (p != NULL && *str != '\0');
    g_free(old);
    itext->refresh_last = itext->linecount;
    gtk_itext_thaw(itext);
}

void gtk_itext_remove_first_line(GtkIText *itext)
{
    GtkITextLine *line;
    gboolean cont;

    g_return_if_fail(GTK_IS_ITEXT(itext));

    cont = TRUE;
    if (itext->linecount == 0)
    {
        /* remove first line from buffer.. */
        while (itext->lines_add != NULL && cont)
        {
            line = itext->lines_add->data;
            cont = strchr(line->str, '\n') == NULL;

	    itext->lines_add = g_list_remove(itext->lines_add, line);

            while (line->colors != NULL)
            {
                g_free(line->colors->data);
                line->colors = g_list_remove(line->colors, line->colors->data);
            }
            g_free(line->str);
            g_free(line);
        }
	return;
    }

    while (itext->lines != NULL && cont)
    {
	line = itext->lines->data;
	cont = line->cont;

	if (itext->last_line == itext->lines)
	    itext->last_line = NULL;
	else
	    itext->last_line_num--;

	/* Destroy line record */
	itext->lines = g_list_remove(itext->lines, line);
	itext->linecount--;

	g_list_foreach(line->colors, (GFunc) g_free, NULL);
	g_list_free(line->colors);
	g_free(line->str);
	g_free(line);

	if (itext->select_start >= 0)
	{
	    /* update selection */
	    if (itext->select_end > 0)
		itext->select_end--;

	    if (--itext->select_start < 0)
		itext->select_start = itext->select_end;
	}

	if (itext->starty == 0)
	{
	    /* first line is displayed, need to redraw the screen.. */
	    itext->refresh_first = 0;
	    itext->refresh_last = itext->height/itext->font_height+1;
	}
	itext->fixy++;
    }

    if (itext->freeze == 0)
	gtk_itext_refresh(itext);
}

void gtk_itext_clear(GtkIText *itext)
{
    gtk_itext_freeze(itext);
    while (itext->lines != NULL)
	gtk_itext_remove_first_line(itext);
    gtk_itext_thaw(itext);
}

void gtk_itext_freeze(GtkIText *itext)
{
    g_return_if_fail(GTK_IS_ITEXT(itext));

    itext->freeze++;
}

void gtk_itext_thaw(GtkIText *itext)
{
    g_return_if_fail(GTK_IS_ITEXT(itext));

    itext->freeze--;
    if (itext->freeze == 0)
	gtk_itext_refresh(itext);
}

static void gtk_itext_redraw_moved(GtkIText *itext, gint diff)
{
    gint y, y2, ysize;

    g_return_if_fail(itext != NULL);

    itext->refresh_first = -1;

    if (diff == 0)
	return;

    ysize = (itext->height/itext->font_height);
    if (itext->pixmap != NULL && (itext->bg_flags & ITEXT_BG_SCROLLABLE) == 0)
    {
	/* background pixmap used, need to redraw the whole thing. */
	diff = ysize+1;
    }

    if (diff > ysize || diff < -ysize)
    {
	/* just redraw it entirely. */
	draw(GTK_WIDGET(itext), 0, 0, itext->width, itext->height);
	return;
    }

    /* scroll the window and redraw only the missing parts */
    if (diff > 0)
    {
	y = itext->height - diff*itext->font_height;
	y2 = itext->height-y;
	gdk_window_copy_area(GTK_WIDGET(itext)->window, itext->back_gc, 0, 0,
			     GTK_WIDGET(itext)->window,
			     0, itext->height-y, itext->width, y);
    }
    else
    {
	y = 0;
	y2 = -diff*itext->font_height;
	gdk_window_copy_area(GTK_WIDGET(itext)->window, itext->back_gc, 0, y2,
			     GTK_WIDGET(itext)->window,
			     0, 0, itext->width, itext->height-y2);
    }

    draw(GTK_WIDGET(itext), 0, y, itext->width, y+y2);
}

static gboolean gtk_itext_fix_scrollbar(GtkIText *itext)
{
    gint ysize, old, oldupper;

    if (!GTK_WIDGET_REALIZED(itext))
	return FALSE;

    old = itext->starty-itext->fixy;
    oldupper = itext->adjustment->upper;

    ysize = (itext->height/itext->font_height);
    itext->fixy = 0;

    if (ysize < itext->linecount &&
	itext->starty >= (gint) itext->adjustment->upper+1-ysize)
    {
	/* scrollbar at the lowest position, scroll down.. */
	itext->starty = itext->linecount-ysize;
    }
    else
    {
	if (ysize > itext->linecount || itext->starty < 0)
	    itext->starty = 0;
	else if (itext->starty > itext->linecount-ysize)
	    itext->starty = itext->linecount-ysize;
    }

    itext->adjustment->upper = itext->linecount-1;
    itext->adjustment->value = itext->starty;
    itext->adjustment->page_increment = ysize-1;
    itext->adjustment->page_size = ysize-1;

    if (itext->starty != old || (itext->linecount-1 != oldupper && ysize < itext->linecount))
    {
	/* update scrollbar if it's position/size is changed */
	gtk_signal_emit_by_name (GTK_OBJECT (itext->adjustment), "changed");
    }

    if (itext->starty == old)
	return FALSE;

    /* update window */
    d(printf("redraw_moved, new %d, old %d\n", itext->starty, old));
    gtk_itext_redraw_moved(itext, itext->starty-old);
    return TRUE;
}

static void gtk_itext_scrollbar_moved(GtkAdjustment *adj, GtkWidget *widget)
{
    GtkIText *itext;
    gint old;

    g_return_if_fail(adj != NULL);
    g_return_if_fail(GTK_IS_ITEXT(widget));

    itext = GTK_ITEXT(widget);

    if ((itext->starty != (gint)adj->value && adj->value >= 0) || itext->refresh_first != -1)
    {
	d(printf("scrollbar moved %d vs %d\n", itext->starty, (gint) adj->value));
        old = itext->starty;
	itext->starty = (gint) adj->value;
	gtk_itext_redraw_moved(itext, itext->starty-old);
    }
}

static void gtk_itext_refresh(GtkIText *itext)
{
    gint y, ypix, y2pix;

    g_return_if_fail(GTK_IS_ITEXT(itext));

    if (itext->freeze != 0)
	return;

    if (gtk_itext_fix_scrollbar(itext))
	return;

    if (itext->refresh_first != -1)
    {
	d(printf("refresh %d..%d\n", itext->refresh_first, itext->refresh_last));
	y = itext->refresh_first-itext->starty;
	ypix = y*itext->font_height;
	y2pix = itext->refresh_last == -1 ? itext->height :
	    (itext->refresh_last-itext->starty)*itext->font_height;

	if (ypix < y2pix)
	{
	    draw(GTK_WIDGET(itext), 0, ypix/* == 0 ? 0 : ypix-itext->font_height*/, itext->width, y2pix);
	    itext->refresh_first = -1;
	}
    }
}

void gtk_itext_set_fonts(GtkIText *itext, GdkFont *font_normal, GdkFont *font_bold)
{
    g_return_if_fail(GTK_IS_ITEXT(itext));
    g_return_if_fail(font_normal != NULL);

    if (itext->font != NULL)
    {
        gdk_font_unref(itext->font);
        gdk_font_unref(itext->font_bold);
    }

    itext->font = font_normal;
    if (font_bold == NULL) font_bold = font_normal;
    itext->font_bold = font_bold;

    gdk_font_ref(itext->font);
    gdk_font_ref(itext->font_bold);

    itext->font_width = gdk_char_width(itext->font, 'o');
    itext->bfont_width = gdk_char_width(itext->font_bold, 'o');
    itext->font_height = itext->font->ascent + itext->font->descent;
    itext->min_width = 30;

    if (itext->font_width <= 0) itext->font_width = 1;
    if (itext->bfont_width <= 0) itext->bfont_width = 1;
    if (itext->font_height <= 0) itext->font_height = 1;

    if (GTK_WIDGET_REALIZED(itext))
    {
	/* redraw the text after font is changed */
	gtk_itext_resize_text(itext);
    }
}

static GdkPixmap *load_pixmap(GtkIText *itext)
{
#ifdef HAVE_IMLIB
    GdkPixmap *pix;
    GdkImlibColorModifier mod;
    GdkImlibImage *iim;

    g_return_val_if_fail(GTK_IS_ITEXT(itext), NULL);

    iim = gdk_imlib_load_image(itext->pixmap_path);
    if (iim == NULL) return NULL;

    mod.contrast = (itext->bg_flags & ITEXT_BG_SHADED) ? 190 : 256;
    mod.gamma = 256;
    mod.brightness = (itext->bg_flags & ITEXT_BG_SHADED) ? 190 : 256;
    gdk_imlib_set_image_modifier(iim, &mod);

    if (itext->bg_flags & ITEXT_BG_SCALED)
	gdk_imlib_render(iim, itext->width, itext->height);
    else
	gdk_imlib_render(iim, iim->rgb_width, iim->rgb_height);
    pix = gdk_imlib_move_image(iim);
    gdk_imlib_destroy_image(iim);
    return pix;
#else
    return gdk_pixmap_create_from_xpm(GTK_WIDGET(itext)->window, NULL, NULL, itext->pixmap_path);
#endif
}

#ifdef TRANSPARENCY
static Window get_desktop_window (Window the_window)
{
  Atom prop, type, prop2;
  int format;
  unsigned long length, after;
  unsigned char *data;
  unsigned int nchildren;
  Window w, root, *children, parent;
  
  prop = XInternAtom(GDK_DISPLAY(), "_XROOTPMAP_ID", True);
  prop2 = XInternAtom(GDK_DISPLAY(), "_XROOTCOLOR_PIXEL", True);
  
  if (prop == None && prop2 == None)
    return None;
  
#ifdef WATCH_DESKTOP_OPTION
  if (Options & Opt_watchDesktop)  {
    if (TermWin.wm_parent != None)
      XSelectInput(GDK_DISPLAY(), TermWin.wm_parent, None);
    
    if (TermWin.wm_grandparent != None)
      XSelectInput(GDK_DISPLAY(), TermWin.wm_grandparent, None);
  }
#endif
  
  for (w = the_window; w; w = parent) {
    if ((XQueryTree(GDK_DISPLAY(), w, &root, &parent, &children, &nchildren)) == False) 
      return None;
      
    if (nchildren) 
      XFree(children);

#ifdef WATCH_DESKTOP_OPTION
    if (Options & Opt_watchDesktop) {
      if (w == TermWin.parent) {
	TermWin.wm_parent = parent;
	XSelectInput(GDK_DISPLAY(), TermWin.wm_parent, StructureNotifyMask);
      } else if (w == TermWin.wm_parent) {
	TermWin.wm_grandparent = parent;
	XSelectInput(GDK_DISPLAY(), TermWin.wm_grandparent, StructureNotifyMask);
      }
    }
#endif
    
    if (prop != None) {
      XGetWindowProperty(GDK_DISPLAY(), w, prop, 0L, 1L, False, AnyPropertyType,
			 &type, &format, &length, &after, &data);
    } else if (prop2 != None) {
      XGetWindowProperty(GDK_DISPLAY(), w, prop2, 0L, 1L, False, AnyPropertyType,
			 &type, &format, &length, &after, &data);
    } else  {
      continue;
    }
    
    if (type != None) {
      return (desktop_window = w);
    }
  }
  
  return (desktop_window = None);
}

static GdkPixmap *get_background(GtkIText *itext)
{
#ifdef HAVE_IMLIB
    GdkImlibColorModifier mod;
    GdkImlibImage *iim;
#endif
    GdkPixmap *pixmap;
    Atom prop, type;
    int format;
    unsigned long length, after;
    unsigned char *data;
    int width, height;

    g_return_val_if_fail(GTK_IS_ITEXT(itext), NULL);

    /*this should be changed when desktop changes I guess*/
    if(desktop_window == None)
	desktop_window = get_desktop_window(GDK_WINDOW_XWINDOW(GTK_WIDGET(itext)->window));
    if(desktop_window == None)
	desktop_window = GDK_ROOT_WINDOW();

    prop = XInternAtom(GDK_DISPLAY(), "_XROOTPMAP_ID", True);

    if (prop == None)
	return NULL;

    XGetWindowProperty(GDK_DISPLAY(), desktop_window, prop, 0L, 1L, False,
		       AnyPropertyType, &type, &format, &length, &after,
		       &data);

    if (type != XA_PIXMAP)
	return NULL;

    pixmap = gdk_pixmap_foreign_new(*((Pixmap *)data));
#ifdef HAVE_IMLIB
    if (itext->bg_flags & ITEXT_BG_SHADED)
    {
	gdk_window_get_size(pixmap, &width, &height);
	iim = gdk_imlib_create_image_from_drawable(pixmap, NULL, 0, 0, width, height);
	if (iim == NULL)
	{
	    itext->bg_flags &= ~ITEXT_BG_SHADED;
	    return pixmap;
	}

	gdk_xid_table_remove(GDK_WINDOW_XWINDOW(pixmap));
	g_dataset_destroy(pixmap);
	g_free(pixmap);

	mod.contrast = 190;
	mod.gamma = 256;
	mod.brightness = 190;
	gdk_imlib_set_image_modifier(iim, &mod);

	gdk_imlib_render(iim, iim->rgb_width, iim->rgb_height);
	pixmap = gdk_imlib_move_image(iim);
	gdk_imlib_destroy_image(iim);
    }
#endif
    return pixmap;
}
#endif

static void itext_transparent_free(GtkIText *itext)
{
    if (itext->pixmap == NULL) return;

#ifdef HAVE_IMLIB
    if (itext->bg_flags & ITEXT_BG_SHADED)
    {
	gdk_pixmap_unref(itext->pixmap);
	return;
    }
#endif

    gdk_xid_table_remove(GDK_WINDOW_XWINDOW(itext->pixmap));
    g_dataset_destroy(itext->pixmap);
    g_free(itext->pixmap);
}

static void gtk_itext_init_background(GtkIText *itext)
{
    if (!GTK_WIDGET_REALIZED(itext))
	return;

    if (itext->pixmap != NULL)
    {
	/* destroy old pixmap */
	if (itext->bg_flags & ITEXT_BG_TRANSPARENT)
	    itext_transparent_free(itext);
	else
	    gdk_pixmap_unref(itext->pixmap);
	itext->pixmap = NULL;
    }

    gdk_gc_set_fill(itext->back_gc, GDK_SOLID);
    if (itext->bg_flags & ITEXT_BG_TRANSPARENT)
    {
#ifdef TRANSPARENCY
	itext->pixmap = get_background(itext);
#endif
    }
    else if (itext->pixmap_path != NULL)
    {
	itext->pixmap = load_pixmap(itext);
	if (itext->pixmap != NULL)
	    gdk_gc_set_fill(itext->back_gc, GDK_TILED);
    }

    if (itext->pixmap != NULL)
    {
	gdk_gc_set_tile(itext->back_gc, itext->pixmap);
	gdk_gc_set_ts_origin(itext->back_gc, 0, 0);
    }

    gdk_gc_set_foreground(itext->back_gc, &itext->back_color);
    gtk_widget_queue_draw(GTK_WIDGET(itext));
}

void gtk_itext_set_background(GtkIText *itext, GdkColor *color, gchar *pixmap, gint flags)
{
    g_return_if_fail(GTK_IS_ITEXT(itext));

    if (itext->pixmap_path != NULL)
    {
	g_free(itext->pixmap_path);
	itext->pixmap_path = NULL;
    }

    if (itext->bg_flags & ITEXT_BG_TRANSPARENT)
    {
	itext_transparent_free(itext);
	itext->pixmap = NULL;
    }

    itext->bg_flags = flags;

    if (flags & ITEXT_BG_TRANSPARENT)
	itext->bg_flags &= ~(ITEXT_BG_SCROLLABLE|ITEXT_BG_SCALED);
    else
    {
	if (pixmap != NULL && *pixmap != '\0')
	    itext->pixmap_path = g_strdup(pixmap);
	else if (color != NULL)
	    memcpy(&itext->back_color, color, sizeof(GdkColor));
    }

    gtk_itext_init_background(itext);
}

void gtk_itext_get_selection(GtkIText *itext)
{
    g_return_if_fail(GTK_IS_ITEXT(itext));

    if (itext->select_start == -1)
	return;

    itext->own_selection = TRUE;
    gtk_selection_owner_set(GTK_WIDGET(itext), GDK_SELECTION_PRIMARY, 0);

    /* redraw the selection with the owned selection color */
    if (itext->select_start < itext->select_end)
    {
	itext->refresh_first = itext->select_start;
	itext->refresh_last = itext->select_end+1;
    }
    else if (itext->select_start >= itext->select_end)
    {
	itext->refresh_first = itext->select_end;
	itext->refresh_last = itext->select_start+1;
    }

    gtk_itext_refresh(itext);
}
