/*
 irssi.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#ifdef HAVE_POPT_H
#include <popt.h>
#endif
#ifdef HAVE_POPT_GNOME_H
#include <popt-gnome.h>
#endif

#include "channels-newdata.h"
#include "gui-dcc.h"
#include "gui-nicklist.h"
#include "gui-nicklist-popup.h"
#include "setup-int.h"

#ifdef HAVE_GNOME_PANEL
#include <applet-widget.h>
#include "panel.h"

static gboolean panel_applet_running;
#endif

static void irssi_init(void)
{
#ifdef HAVE_GNOME
    irssi_gui = IRSSI_GUI_GNOME;
#else
    irssi_gui = IRSSI_GUI_GTK;
#endif
    irc_base_init();
    irc_extra_init();
    ui_common_init();
    setup_init();
}

static void irssi_finish_init(void)
{
    CHANNEL_REC *channel;
#ifdef GTK_HEBREW
    gchar *rc_file;

    rc_file = convert_home(GTK_HEBREW_RC);
    gtk_rc_parse(rc_file);
    g_free(rc_file);
#endif

    gui_init();
    gui_server_init();
    gui_mainwindows_init();
    gui_windows_init();
    gui_channels_init();
    gui_printtext_init();
    gui_dcc_init();
    gui_events_init();
    gui_statusbar_init();
    channels_newdata_init();
    gui_nicklist_init();
    gui_nicklist_popup_init();
    gui_menus_init();
    dialogs_init();

    ui_common_finish_init();
    channel = channel_find_level(MSGLEVEL_MSGS);
    if (channel != NULL && setup_get_bool("toggle_autoraise_msgs_window"))
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(WINDOW_GUI(CHANNEL_PARENT(channel))->parent->autoraise), TRUE);

#ifdef HAVE_GNOME_PANEL
    if (panel_applet_running) panel_init();
#endif

    signal_emit("irssi init finished", 0);
}

static void irssi_deinit(void)
{
    setup_deinit();
    gui_deinit();
#ifdef HAVE_GNOME_PANEL
    panel_deinit();
#endif
    dialogs_deinit();
    gui_menus_deinit();
    gui_nicklist_deinit();
    gui_nicklist_popup_deinit();
    channels_newdata_deinit();
    gui_dcc_deinit();
    gui_events_deinit();
    gui_server_deinit();
    gui_statusbar_deinit();
    gui_mainwindows_deinit();
    gui_windows_deinit();
    gui_channels_deinit(); /* channels after gui_mainwindows_deinit()! */
    gui_printtext_deinit();

    ui_common_deinit();
    irc_extra_deinit();
    settings_deinit();
    irc_base_deinit();
}

int main(int argc, char **argv)
{
    static struct poptOption options[] = {
	POPT_AUTOHELP
        { NULL, '\0', 0, NULL }
    };
#ifdef HAVE_GNOME_PANEL
    static struct poptOption panel_options[] = {
        { "no-applet", '\0', POPT_ARG_NONE, NULL, 0, N_("Don't run as applet"), NULL },
        { NULL, '\0', 0, NULL }
    };

    gint num;
#endif

#ifdef HAVE_SOCKS_H
    SOCKSinit(argv[0]);
#endif

#ifdef ENABLE_NLS
    /* initialize the i18n stuff */
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
#endif

    irssi_init();

    args_register(options);
#ifdef HAVE_GNOME_PANEL
    args_register(panel_options);

    panel_applet_running = TRUE;
    for (num = 1; num < argc; num++)
        if (strcmp(argv[num], "--no-applet") == 0 || strcmp(argv[num], "--help") == 0)
        {
            panel_applet_running = FALSE;
            break;
        }
#endif

#ifdef HAVE_GNOME_PANEL
    if (panel_applet_running)
        applet_widget_init(PACKAGE, VERSION, argc, argv, (struct poptOption *) (iopt_tables->data), 0, NULL);
    else
#endif
#ifdef HAVE_GNOME
	gnome_init_with_popt_table(PACKAGE, VERSION, argc, argv, (struct poptOption *) (iopt_tables->data), 0, NULL);
    signal(SIGSEGV, SIG_DFL); /* disable gnome's stupid sigsegv handler */
#else
    gtk_init(&argc, &argv);
    args_execute(argc, argv);
#ifdef HAVE_IMLIB
    gdk_imlib_init();
#endif
#endif

    irssi_finish_init();

#ifdef HAVE_GNOME_PANEL
    if (panel_applet_running)
        applet_widget_gtk_main ();
    else
#endif
        gtk_main();

    irssi_deinit();
#ifdef MEM_DEBUG
    ig_mem_profile();
#endif
    return 0;
}
