/* gnome-napster, a GNOME Napster client.
 * Copyright (C) 1999 Evan Martin <eeyem@u.washington.edu>
 */
#include <stdio.h>
#include <gnome.h>
#include "lnap.h"
#include "search.h"
#include "download.h"
#include "util.h"
#include "ui.h"
#include "gtkflist.h"
#include "gconfig.h"
#include "songinfo.h"
#include "host.h"

#include "../pixmaps/speedgreen.xpm"
#include "../pixmaps/speedyellow.xpm"
#include "../pixmaps/speedred.xpm"

static GtkWidget *entry, *clist, *cmdbox;
static GdkPixmap *speedgreen = NULL, *speedyellow, *speedred;
static GdkBitmap *speedgreenb, *speedyellowb, *speedredb;
static GList *songlist = NULL;
static int songcount = 0;

static void popup_add_cb(GtkWidget *w, gpointer d);
static void selall_cb(GtkWidget *w, gpointer d);

enum {
	COL_NAME, COL_SIZE, COL_BITRATE, COL_TIME, COL_HOST, 
	COL_CONNECTION, COL_COUNT
};
static gchar* clist_titles[] = { "Song Name", "Size", "kbps", "Length",
	"Host", "Connection" };

static GnomeUIInfo popup_menu[] = {
	GNOMEUIINFO_ITEM_NONE(N_("_Add to host list"), 
			N_("Add this host to the Host tab"), 
			popup_add_cb),
	GNOMEUIINFO_ITEM_NONE(N_("_Select All"), 
			N_("Select all items in the list"), 
			selall_cb),
	GNOMEUIINFO_END
};

void search_load_pixmaps(void) {
	GtkStyle *style;

	if (speedgreen != NULL) return;
	style = gtk_widget_get_style(app);
	speedgreen = gdk_pixmap_create_from_xpm_d(app->window, 
			&speedgreenb, &style->bg[GTK_STATE_NORMAL], speedgreen_xpm);
	speedyellow = gdk_pixmap_create_from_xpm_d(app->window, 
			&speedyellowb, &style->bg[GTK_STATE_NORMAL], speedyellow_xpm);
	speedred = gdk_pixmap_create_from_xpm_d(app->window, 
			&speedredb, &style->bg[GTK_STATE_NORMAL], speedred_xpm);
}

/* implement column sorting when selecting column headings */
static void 
sort_col(GtkCList *clist, gint column, gint width, gpointer user_data)
{
	switch (column) {
		case COL_SIZE:
			gtk_clist_set_compare_func(clist, songinfo_compare_size);
			break;
		case COL_BITRATE:
			gtk_clist_set_compare_func(clist, songinfo_compare_bitrate);
			break;
		case COL_CONNECTION:
			gtk_clist_set_compare_func(clist, songinfo_compare_connection);
			break;
		case COL_TIME:
			gtk_clist_set_compare_func(clist, songinfo_compare_time);
			break;
		case COL_NAME:
		case COL_HOST:
		default:
			gtk_clist_set_compare_func(clist, NULL); /* default compare func */
	}

	if (column == clist->sort_column) {
		/* switch sort type */
		gtk_clist_set_sort_type(clist, GTK_SORT_DESCENDING + GTK_SORT_ASCENDING
				- clist->sort_type);
	} else {
		/* select column to sort */
		gtk_clist_set_sort_column(clist, column);
	}

	/* go for it */
	gtk_clist_sort(clist);
}

static void list_update(int row) {
	GdkPixmap *pix = NULL; GdkBitmap *bit = NULL;
	char *text;
	ssong_info *i = gtk_clist_get_row_data(GTK_CLIST(clist), row);

	/*FUNCPRINT("row=%d", row);*/
	if (i->conn <= NAP_CONNTYPE_57_6) {
		pix = speedred; bit = speedredb;
	} else if (i->conn <= NAP_CONNTYPE_ISDN_128K) {
		pix = speedyellow; bit = speedyellowb;
	} else if (i->conn <= NAP_CONNTYPE_T3_PLUS) {
		pix = speedgreen; bit = speedgreenb;
	}

	text = short_path(i->title, i->pathshift);
 
	if (pix)
		gtk_clist_set_pixtext(GTK_CLIST(clist), row, COL_NAME, text, 2,
				pix, bit);
	else 
		gtk_clist_set_text(GTK_CLIST(clist), row, COL_NAME, text);
}

static void 
list_add(ssong_info *i) {
	gchar* add[COL_COUNT];
	int row;

	FUNCPRINT("i=%p", i);
	/* make a pointer to the name of the file */
	add[COL_NAME] = g_strdup_printf("%s", short_path(i->title, i->pathshift));
	
	/* then the size of the file */
	add[COL_SIZE] = g_strdup_printf("%lu", i->size);
	
	/* now the host */
	add[COL_HOST] = g_strdup_printf("%s", i->host);
	
	/* print out the length of the file */
	add[COL_TIME] = g_strdup_printf("%u:%02u", i->time/60, i->time % 60);
	
	/* figure out what the connection type is */
	add[COL_CONNECTION] = g_strdup_printf("%s", nap_conntypes[i->conn]);

	/* then make the bitrate into a string */
	add[COL_BITRATE] = g_strdup_printf("%d", i->bitrate);

	/* now append the info to then end of the list */
	row = gtk_clist_append(GTK_CLIST(clist), add);

	/* associate the data in the ssong_info struct with this row 
	 * of the list */
	gtk_clist_set_row_data(GTK_CLIST(clist), row, i);

	list_update(row);

	/* clean up */
	for(row=0; row < COL_COUNT; row++)
		g_free(add[row]);
	FUNCPRINT("Leaving");
}

void search_result(char *text) {
	ssong_info *i;
	char buf[50];
	FUNCPRINT("%s", text);
	/* allocate a new struct to hold the info */
	i = songinfo_new();
	
	/* break it up into its parts, FIXME: it seems like there should be a better
	 * more robust way to do this */
	sscanf(text, "\"%[^\"]\" %s %lu %d %d %d %s %lu %d",
		i->title, i->checksum, &i->size, &i->bitrate, &i->samprate, &i->time, 
		i->host, &i->ip, &i->conn);

	/* get the pathshift values from gnome */
	i->pathshift = gnome_config_get_int(CONFIG_PREFIX "/Interface/PathDepth=1");

	/* make sure that we count this song */
	songcount++;

	/* say how many songs we have got */
	snprintf(buf, 50, "Got %d results...", songcount);
	gtk_clist_set_text(GTK_CLIST(clist), 0, COL_NAME, buf);

	/* tack the song info onto the end of the list */
	songlist = g_list_append(songlist, i);
}

void search_complete(void) {
	static gchar* add[COL_COUNT];
	GList *l = songlist;

	FUNCPRINT("<no args>");

	/* clear out add and initialize the first element */
	memset(add, 0, sizeof(add));
	add[0] = g_strdup_printf("%s", "File not found.");

	/* stop the screen updates and clear the list */
	gtk_clist_freeze(GTK_CLIST(clist));
	gtk_clist_clear(GTK_CLIST(clist));

	if (songcount == 0) { /* not found */
		gtk_clist_append(GTK_CLIST(clist), add);
	} else {
		/* add the songlist data into the list view */
		do {
			list_add(l->data);
		} while((l=g_list_next(l)) != NULL);
	}

	/* let the list update the screen again */
	gtk_clist_thaw(GTK_CLIST(clist));
	gtk_widget_set_sensitive(cmdbox, TRUE);
	
	printcn("Search complete.");
	g_free(add[0]);
	/* perhaps ping hosts? */
}

static void getfiles_cb() {
	GList *sel;
	ssong_info *i;

	sel = GTK_CLIST(clist)->selection;
	if (sel == NULL) return;
	while (sel != NULL) {
		i = gtk_clist_get_row_data(GTK_CLIST(clist), (int)sel->data);
		if (i != NULL) /* NULL possible if they select "no matches found" */
			download_enqueue(i->host, i->title, i->pathshift);
		sel = sel->next;
	}
}

static void search_cb(GtkWidget* w, gpointer d) {
	char buf[1024];
	char *search = gtk_entry_get_text(GTK_ENTRY(
				gnome_entry_gtk_entry(GNOME_ENTRY(entry))));
	int len;
	gchar* add[COL_COUNT];

	gnome_entry_save_history(GNOME_ENTRY(entry));

	if (search[0] == 0) return;
	sprintf(buf, "Searching for '%s'...", search);
	memset(add, 0, sizeof(add)); /* initialize all columns to null */
	add[0] = g_strdup_printf("%s", buf);
	
	printcn(buf);

	/* dissable the buttons */
	gtk_widget_set_sensitive(cmdbox, FALSE);
	/* clear the list */
	gtk_clist_clear(GTK_CLIST(clist));
	/* and write out the searching string */
	gtk_clist_append(GTK_CLIST(clist), add);
	/* tell it that nothing is associated with this row */
	gtk_clist_set_row_data(GTK_CLIST(clist), 0, 0);
	len = sprintf(buf, "FILENAME CONTAINS \"%s\" MAX_RESULTS 100", search);

	songlist = g_list_free_all(songlist);
	songcount = 0;

	nap_sendpacket(nap_socknap, NM_SEARCH_REQUEST, len, buf);

	g_free(add[0]);
}

/* emit the entry's "changed" signal so history gets updated properly 
 * FIXME: this is a hack of sorts. */
static void search_btn_cb(GtkWidget *w, gpointer d) {
	gtk_signal_emit_by_name(GTK_OBJECT(gnome_entry_gtk_entry(GNOME_ENTRY(entry))),
			"activate", entry);
}

static void selall_cb(GtkWidget *w, gpointer d) {
	gtk_clist_select_all(GTK_CLIST(clist));
}

static void shift_foreach_cb(gpointer data, gpointer d) {
	int row = (int)data;
	int shift = (int)d;
	ssong_info *i = gtk_clist_get_row_data(GTK_CLIST(clist), row);

	if (i == NULL) return;

	if (i->pathshift + shift < 0) i->pathshift = 0;
	else i->pathshift += shift;
	list_update(row);
}

static void shift_cb(GtkWidget *w, gpointer d) {
	if (GTK_CLIST(clist)->selection)
		g_list_foreach(GTK_CLIST(clist)->selection, shift_foreach_cb, d);
}

static gint button_press_cb(GtkWidget *w, GdkEventButton *e, gpointer d) {
	GtkWidget *menu;
	ssong_info *i;
	int row, col;

	if (e->type == GDK_2BUTTON_PRESS) {
		getfiles_cb();
		return TRUE;
	} else if (e->type == GDK_BUTTON_PRESS && e->button == 3) {
		gtk_clist_get_selection_info(GTK_CLIST(clist), e->x, e->y,
				&row, &col);
		i = gtk_clist_get_row_data(GTK_CLIST(clist), row);
		if (i == NULL) return FALSE;
		menu = gnome_popup_menu_new(popup_menu);
		gnome_popup_menu_do_popup_modal(menu, NULL, NULL, e, i->host);
		gtk_widget_destroy(menu);
		return TRUE;
	}

	return FALSE;
}

static void popup_add_cb(GtkWidget *w, gpointer d) {
	host_addhost((char*)d);
}
static void button_addhost_cb(GtkWidget *w, gpointer d) {
	ssong_info *i;
	GList *sel;
	if ((sel = GTK_CLIST(clist)->selection) == NULL) return;
	i = gtk_clist_get_row_data(GTK_CLIST(clist), (int)sel->data);
	host_addhost(i->host);
}

GtkWidget* create_search(void) {
	GtkWidget *vbox, *hbox;
	GtkWidget *label, *bsearch, *bget, *bhost;
	GtkWidget *bselall;
	GtkWidget *scroll;
	GtkWidget *bshiftleft, *bshiftright;
	
	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
		cmdbox = gtk_hbox_new(FALSE, 10);
		gtk_widget_set_sensitive(cmdbox, FALSE);
			label = gtk_label_new("Search for:");
		gtk_box_pack_start(GTK_BOX(cmdbox), label, FALSE, FALSE, 0);
			entry = gnome_entry_new("search");
			gnome_entry_load_history(GNOME_ENTRY(entry));
			/* TESTING gtk_entry_set_text(GTK_ENTRY(entry), "aphex");*/
			gtk_signal_connect(GTK_OBJECT(gnome_entry_gtk_entry(GNOME_ENTRY(entry))), 
						"activate", GTK_SIGNAL_FUNC(search_cb), NULL);
		gtk_box_pack_start(GTK_BOX(cmdbox), entry, FALSE, FALSE, 0);
			bsearch = gtk_button_new_with_label("Search");
			gtk_signal_connect(GTK_OBJECT(bsearch), "clicked",
					GTK_SIGNAL_FUNC(search_btn_cb), NULL);
		gtk_box_pack_start(GTK_BOX(cmdbox), bsearch, FALSE, FALSE, 0);
			bget = gtk_button_new_with_label("Get File(s)");
			gtk_signal_connect(GTK_OBJECT(bget), "clicked",
					GTK_SIGNAL_FUNC(getfiles_cb), NULL);
		gtk_box_pack_start(GTK_BOX(cmdbox), bget, FALSE, FALSE, 0);
			bselall = gtk_button_new_with_label("Select All");
			gtk_signal_connect(GTK_OBJECT(bselall), "clicked",
					GTK_SIGNAL_FUNC(selall_cb), NULL);
		gtk_box_pack_start(GTK_BOX(cmdbox), bselall, FALSE, FALSE, 0);
			bhost = gtk_button_new_with_label("Add Host");
			gtk_signal_connect(GTK_OBJECT(bhost), "clicked",
					GTK_SIGNAL_FUNC(button_addhost_cb), NULL);
		gtk_box_pack_start(GTK_BOX(cmdbox), bhost, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), cmdbox, FALSE, FALSE, 0);
		scroll = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), 
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
			clist = gtk_flist_new_with_titles(sizeof(clist_titles)/sizeof(char*), 
					clist_titles);
				hbox = gtk_hbox_new(FALSE, 0);
					bshiftleft = songinfo_arrow_button_new(app, TRUE);
					gtk_signal_connect(GTK_OBJECT(bshiftleft), "clicked",
							GTK_SIGNAL_FUNC(shift_cb), (gpointer)-1);
				gtk_box_pack_start(GTK_BOX(hbox), bshiftleft, FALSE, FALSE, 0);
					bshiftright = songinfo_arrow_button_new(app, FALSE);
					gtk_signal_connect(GTK_OBJECT(bshiftright), "clicked",
							GTK_SIGNAL_FUNC(shift_cb), (gpointer)1);
				gtk_box_pack_start(GTK_BOX(hbox), bshiftright, FALSE, FALSE, 0);
				gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(clist_titles[0]), 
						FALSE, FALSE, 5);
			gtk_widget_show_all(hbox);
			gtk_clist_set_column_widget(GTK_CLIST(clist), 0, hbox);
			gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_MULTIPLE);
			gtk_clist_set_column_justification(GTK_CLIST(clist), COL_SIZE,
					GTK_JUSTIFY_RIGHT);
			gtk_clist_set_column_justification(GTK_CLIST(clist), COL_BITRATE,
					GTK_JUSTIFY_RIGHT);
			gtk_clist_set_column_justification(GTK_CLIST(clist), COL_TIME,
					GTK_JUSTIFY_RIGHT);
			gtk_clist_set_column_width(GTK_CLIST(clist), COL_NAME, 200);
			gtk_clist_set_column_width(GTK_CLIST(clist), COL_SIZE, 50);
			gtk_clist_set_column_width(GTK_CLIST(clist), COL_HOST, 60);
			gtk_clist_set_column_width(GTK_CLIST(clist), COL_CONNECTION, 65);
			/*gtk_clist_set_column_width(GTK_CLIST(clist), COL_BITRATE, 25);*/
			gtk_signal_connect(GTK_OBJECT(clist), "button_press_event",
					GTK_SIGNAL_FUNC(button_press_cb), NULL);
			gtk_signal_connect(GTK_OBJECT(clist), "click-column",
					GTK_SIGNAL_FUNC(sort_col), NULL);
			gtk_clist_set_sort_column(GTK_CLIST(clist), 0);	/* sort on title by default */
			gtk_clist_set_auto_sort(GTK_CLIST(clist), TRUE);	
		gtk_container_add(GTK_CONTAINER(scroll), clist);
	gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 0);

	search_load_pixmaps();
	return vbox;
}

void search_login_status(int loggedin) {
	if (loggedin) {
		gtk_widget_set_sensitive(cmdbox, TRUE);
		gtk_clist_clear(GTK_CLIST(clist));
	} else {
		gtk_widget_set_sensitive(cmdbox, FALSE);
		gtk_clist_clear(GTK_CLIST(clist));
	}
}
