// File for Darxite daemon handling lowest-level FTP/HTTP commands

// REMEMBER that sockets aren't freed by Disconnect() calls, you have to
// do so with FreeSocket afterwards!

#include <includes.h>
#include "sockets.h"
#include "global.h"

#include <arpa/telnet.h>
#include <netdb.h>

#ifndef O_NONBLOCK
#  define O_NONBLOCK 04000
#endif

int SocketCount = 0;
static int default_fd;

void InitNetworking(void)
{
    DefaultFtpPort = getservbyname("ftp", "tcp")->s_port;
    DefaultHttpPort = getservbyname("http", "tcp")->s_port;
}

int sock_CreateNetSocket(int port)
{
    struct sockaddr_in addr;
    int net_fd, old_flags;
    
    if (port <= 0)
    {
        error(E_TRACE, "Not creating net socket on port %d", port);
        setuid(getuid());
        return -1;
    }
    error(E_TRACE, "Creating net socket on port %d", port);
    
    net_fd = socket(PF_INET, SOCK_STREAM, 0);
    if (net_fd < 0)
    {
        error(E_WARN, "Couldn't create net socket on port %d: %s",
                 port, strerror(errno));
        return -1;
    }

    old_flags = fcntl(net_fd, F_GETFL, 0);
    if (old_flags == -1)
    {
        DX_errno = errno;
        error(E_WARN, "Couldn't read socket flags: %s", strerror(errno));
        close(net_fd);
        return -1;
    }
    old_flags |= O_ASYNC;
    if (fcntl(net_fd, F_SETFL, old_flags) < 0)
    {
        DX_errno = errno;
        error(E_WARN, "Couldn't set socket flags: %s", strerror(errno));
        close(net_fd);
        return -1;
    }
    if (fcntl(net_fd, F_SETOWN, getpid()) < 0)
    {
        DX_errno = errno;
        error(E_WARN, "Couldn't set IO owner: %s", strerror(errno));
        close(net_fd);
        return -1;
    }
    
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    if (bind(net_fd, (struct sockaddr *)&addr, sizeof(addr)) < 0)
    {
        DX_errno = errno;
        error(E_WARN, "Couldn't bind net control socket: %s",
                 strerror(errno));
        close(net_fd);
        return -1;
    }
    if (listen(net_fd, 1) < 0)
    {
        DX_errno = errno;
        error(E_WARN, "Couldn't listen on net control socket: %s",
                 strerror(errno));
        close(net_fd);
        return -1;
    }
    // restore the original uid so ppl don't r00t the box
    // this probably doesn't belong here
    setuid(getuid());
    return net_fd;
}

/* Set an fd to our pid; return 0 on success, -1+errno on fail.
 * Be sure to do this after forking. */
int sock_SetSocketPid(int fd)
{
    return fcntl(fd, F_SETOWN, getpid());
}

int sock_CreateDefaultSocket(void)
{
    struct sockaddr_un addr;
    size_t size;
    int old_flags, old_uid;
    char buffer[256];

    // if we're suided, we need to temporarily drop root privileges to
    // create the socket
    old_uid = geteuid();
    seteuid(getuid());
    sprintf(buffer, "%s.%s", DEFAULT_SOCKET_FILE,
            getpwuid(geteuid())->pw_name);
    if (access(buffer, R_OK) != -1)
    {
        if (DX_QuietExit)
        {
            exit(1);
        }
        else
        {
            error(E_FAIL, "The client control socket %s already exists - "
                     "are you running another copy of Darxite? If not, "
                     "delete the file and try running Darxite again.",
                     buffer);
        }
    }
    
    default_fd = socket(PF_UNIX, SOCK_STREAM, 0);
    if (default_fd < 0)
    {
        DX_errno = errno;
        seteuid(old_uid);
        error(E_WARN, "Couldn't create default socket: %s",
                 strerror(errno));
        return -1;
    }
    sock_SetSocketPid(default_fd);
    old_flags = fcntl(default_fd, F_GETFL, 0);
    if (old_flags == -1)
    {
        DX_errno = errno;
        seteuid(old_uid);
        error(E_WARN, "Couldn't read socket flags: %s", strerror(errno));
        close(default_fd);
        return -1;
    }
    old_flags |= O_ASYNC;
    if (fcntl(default_fd, F_SETFL, old_flags) < 0)
    {
        DX_errno = errno;
        seteuid(old_uid);
        error(E_WARN, "Couldn't set socket flags: %s", strerror(errno));
        close(default_fd);
        return -1;
    }
    
    if (sock_SetSocketPid(default_fd))
    {
        DX_errno = errno;
        seteuid(old_uid);
        error(E_WARN, "Couldn't set IO owner: %s", strerror(errno));
        close(default_fd);
        return -1;
    }
    addr.sun_family = AF_FILE;
    strcpy(addr.sun_path, buffer);
    size = sizeof(addr.sun_family) + strlen(addr.sun_path) + 1;
    if (bind(default_fd, (struct sockaddr *)&addr, size) < 0)
    {
        DX_errno = errno;
        seteuid(old_uid);
        error(E_WARN, "Couldn't bind client control socket: %s",
                 strerror(errno));
        close(default_fd);
        return -1;
    }
    // For security, make it only read-write to this user
    if (chmod(buffer, S_IRUSR | S_IWUSR) < 0)
    {
        DX_errno = errno;
        seteuid(old_uid);
        error(E_WARN,
                 "Couldn't set permissions of client control socket: %s",
                 strerror(errno));
        close(default_fd);
        return -1;
    }
    
    if (listen(default_fd, 1) < 0)
    {
        DX_errno = errno;
        seteuid(old_uid);
        error(E_WARN, "Couldn't listen on client control socket: %s",
                 strerror(errno));
        close(default_fd);
        return -1;
    }
    seteuid(old_uid);
    error(E_TRACE, "Client control socket created OK");
    return default_fd;
}

int sock_DestroyDefaultSocket(void)
{
    char buffer[256];

    sprintf(buffer, "%s.%s", DEFAULT_SOCKET_FILE, getpwuid(getuid())->pw_name);
    if (shutdown(default_fd, 2) < 0)
    {
        error(E_WARN, "Couldn't shut down default socket: %s",
                 strerror(errno));
        return DX_LIB_ERROR;
    }
    remove(buffer);
    error(E_TRACE, "Default socket destroyed OK");
    return DX_LIB_OK;
}

int sock_DestroyNetSocket(int net_fd)
{
    if ((net_fd > 0) && (close(net_fd) != 0))
    {
        error(E_WARN, "Couldn't close net socket: %s", strerror(errno));
        return DX_LIB_ERROR;
    }
    return DX_LIB_OK;
}

// Arguments: host to connect to, port to use, string to write current
// activity to, socket to return. Returns an error value or DX_OK.
int Connect(const char *host, int port, char *activity, SockInfo **new_socket)
{
    struct hostent *hostinfo = NULL;
    SockInfo *sock;
    fd_set write_fds;
    struct timeval tv;
    int rc;
    ServerInfo *server = FirstServer;
    
    error(E_TRACE, "  Connect(%s)", host);
    
    if (!host)
        return DX_ERROR;
    
    sock = CreateSocket();
    if (sock == NULL)
        return DX_E_SOCKET;
    
    error(E_TRACE, "socket created");
    if (activity)
        sprintf(activity, "Looking up host \"%s\"...", host);
    
    // As gethostbyname() can take ages and needs to be locked, we save
    // the IP address for use next time.
    sock->Name.sin_addr.s_addr = 0;
    while (server)
    {
        if (!strcmp(host, server->Name))
        {
            if (server->IpAddress > 0)
            {
                sock->Name.sin_addr.s_addr = server->IpAddress;
                error(E_TRACE, "Found IP %d for %s", server->IpAddress, host);
            }
            break;
        }
        server = server->Next;
    }
    // back to the old code - 3/11/99
    if (sock->Name.sin_addr.s_addr == 0)
    {
        error(E_TRACE, "Looking up %s...", host);
        // As gethostbyname() returns a pointer to a static data
        // structure, we have to be careful that no other threads
        // will access it before we've copied it.
        LOCK_MUTEX(ConnectMutex);
        ConnectMutexLocked = TRUE;
        //Sleep(10000);
        hostinfo = gethostbyname(host);
        if (hostinfo == NULL)
        {
            error(E_TRACE, "Unknown host %s: %s (%d)", host,
                  hstrerror(h_errno), h_errno);
            UNLOCK_MUTEX(ConnectMutex);
            ConnectMutexLocked = FALSE;
            if (server)
                server->IpAddress = 0;
            close(sock->FileDes);
            sock->Connecting = FALSE;
            dxfree(sock);
            return DX_E_LOOKUP;
        }
        sock->Name.sin_addr = *((struct in_addr *)(hostinfo->h_addr));
        if (server)
            server->IpAddress = sock->Name.sin_addr.s_addr;
        UNLOCK_MUTEX(ConnectMutex);
        ConnectMutexLocked = FALSE;
    }
    error(E_TRACE, "Looked up OK");
    if (activity)
        sprintf(activity, "Connecting to host...");
    strcpy(sock->HostName, host);
    sock->Name.sin_family = AF_INET;
    sock->Name.sin_port = port;

    //error(E_TRACE, "About to connect: fd=%i", sock->FileDes);
    rc = connect(sock->FileDes, (struct sockaddr *) &sock->Name,
                 sizeof(sock->Name));
    // since it's non-blocking, there is a good chance we'll get this error
    if ((rc == -1) && (errno == EINPROGRESS))
    {
        //error(E_TRACE, "connect() returned EINPROGRESS, selecting");
        tv.tv_sec = 30;
        tv.tv_usec = 0;
        FD_ZERO(&write_fds);
        FD_SET(sock->FileDes, &write_fds);
        rc = select(FD_SETSIZE, NULL, &write_fds, NULL, &tv);
        //error(E_TRACE, "select returned %i", rc);
        // pretend it's an error for the bit below
        if (rc == 0)
            rc = -1;
    }
    if (rc == -1)
    {
        error(E_TRACE, "connect returned %i: %s", rc, strerror(errno));
        // if we have a dodgy IP address, reset it
        if (server && !strcmp(host, server->Name))
            server->IpAddress = 0;
        close(sock->FileDes);
        dxfree(sock);
        return DX_E_CONNECT;
    }
    
    error(E_TRACE, "Host %s connected OK", host);
    strcpy(sock->LastCmd, "");
    sock->Connected = TRUE;
    sock->Connecting = FALSE;
    SocketCount++;
    
    *new_socket = sock;
    return DX_OK;
}

SockInfo *DataConnect(SockInfo *sock)
{
    int rc, addr[6];
    char buffer[256], *buf_ptr;
    SockInfo *data_sock;
    struct timeval tv;
    fd_set write_fds;

    if (sock == NULL)
        return NULL;
    error(E_TRACE, "  DataConnect(%s)", sock->HostName);
    
    data_sock = CreateSocket();
    if (data_sock == NULL)
        return NULL;
    
    rc = NetWrite(sock, "PASV");
    if (rc >= FTP_ERROR_START)
    {
        error(E_WARN, "Couldn't execute PASV command: server response was %s",
              sock->CmdResponse);
        close(data_sock->FileDes);
        dxfree(data_sock);
        return NULL;
    }
    else if (rc == -1) // ie. the write itself failed
    {
        close(data_sock->FileDes);
        dxfree(data_sock);
        return NULL;
    }
    strncpy(buffer, sock->CmdResponse, sizeof(buffer));
    
    // skip the error code and then all the text following
    buf_ptr = buffer + 3;
    while (*buf_ptr && !isdigit(*buf_ptr))
        buf_ptr++;
    
    // This string represents a four-byte IP address followed by
    // two bytes for the port no. Scanning should now work with Roxen
    // Challenger, and all other FTP servers...
    if (sscanf(buf_ptr, "%d,%d,%d,%d,%d,%d", &addr[2], &addr[3], &addr[4],
           &addr[5], &addr[0], &addr[1]) != 6)
    {
        error(E_WARN, "Got invalid PASV response of %s", buffer);
        close(data_sock->FileDes);
        dxfree(data_sock);
        return NULL;
    }
    strcpy(data_sock->HostName, sock->HostName);
    data_sock->Name.sin_family = AF_INET;
    data_sock->Name.sin_addr.s_addr = htonl((addr[2] << 24) | (addr[3] << 16) |
                                      (addr[4] << 8) | addr[5]);
    data_sock->Name.sin_port = htons((addr[0] << 8) | addr[1]);
    
    rc = connect(data_sock->FileDes, (struct sockaddr *) &data_sock->Name,
                 sizeof(data_sock->Name));
    
    // since it's non-blocking, there is a good chance we'll get this error
    if ((rc == -1) && (errno == EINPROGRESS))
    {
        tv.tv_sec = 10;
        tv.tv_usec = 0;
        FD_ZERO(&write_fds);
        FD_SET(data_sock->FileDes, &write_fds);
        rc = select(FD_SETSIZE, NULL, &write_fds, NULL, &tv);
        // pretend it's an error for the bit below
        if (rc == 0)
            rc = -1;
    }
    if (rc == -1)
    {
        error(E_WARN, "Couldn't connect data socket: %s", strerror(errno));
        close(data_sock->FileDes);
        dxfree(data_sock);
        return NULL;
    }
    
    strcpy(data_sock->LastCmd, "");
    // allow space for a null terminator when we need it
    data_sock->DataBuf = (char *)dxmalloc(DX_BufferSize + 1);
    data_sock->Connected = TRUE;
    data_sock->Connecting = FALSE;
    SocketCount++;

    return data_sock;
}

SockInfo *CreateSocket(void)
{
    struct linger lng = {0, 0};
    int old_flags, on = TRUE;
    SockInfo *sock;

    //error(E_TRACE, "trying to create socket");
    
    sock = dxcalloc(sizeof(SockInfo));
    sock->Connecting = TRUE;
    sock->Transferring = FALSE;
    sock->Connected = FALSE;
    memclr(sock->HostName);
    memclr(sock->LastCmd);
    memset(&sock->Name, 0, sizeof(sock->Name));
    sock->LastError = 0;
    sock->Transferring = FALSE;
    sock->MoreResponse = FALSE;
    sock->Eof = FALSE;
    sock->FileDes = socket(PF_INET, SOCK_STREAM, 0);
    sock->DataLength = 0;
    if (sock->FileDes == -1)
    {
        error(E_WARN, "Couldn't create socket: %s", strerror(errno));
        dxfree(sock);
        return NULL;
    }
    
    // we make the socket non-blocking to avoid the possibility
    // of any lock-ups
    old_flags = fcntl(sock->FileDes, F_GETFL, 0);
    if (old_flags == -1)
    {
        error(E_WARN, "Couldn't read socket flags: %s", strerror(errno));
        close(sock->FileDes);
        dxfree(sock);
        return NULL;
    }
    old_flags |= O_NONBLOCK;
    if (fcntl(sock->FileDes, F_SETFL, old_flags) < 0)
    {
        error(E_WARN, "Couldn't set socket flags: %s", strerror(errno));
        close(sock->FileDes);
        dxfree(sock);
        return NULL;
    }
    
    if (setsockopt(sock->FileDes, SOL_SOCKET, SO_REUSEADDR,
		   &on, sizeof(on)) == -1)
    {
        error(E_WARN, "Couldn't set socket options: %s", strerror(errno));
        close(sock->FileDes);
        dxfree(sock);
        return NULL;
    }
    if (setsockopt(sock->FileDes, SOL_SOCKET, SO_LINGER, &lng,
                   sizeof(lng)) == -1)
    {
        error(E_WARN, "Couldn't set socket options: %s", strerror(errno));
        close(sock->FileDes);
        dxfree(sock);
        return NULL;
    }
    
    return sock;
}

void FreeSocket(SockInfo **sock)
{
    /* We should probably just silently return */
    if (sock == NULL || *sock == NULL)
    {
        error(E_TRACE, "Tried to FreeSocket a NULL socket");
        return;
    }
    // We don't want to call Disconnect() or anything here because we don't
    // know what kind of socket it is.
    if((*sock)->DataBuf)
        dxfree((*sock)->DataBuf);
    if((*sock)->CmdResponse)
        dxfree((*sock)->CmdResponse);
    
    dxfree(*sock);
}

/* If a socket is open, close it. */
static int CloseSocket(SockInfo *sock)
{
    int ret;
    error(E_TRACE, "Closing %i", sock->FileDes);

    if (sock->FileDes == -1) return 1;
    ret = close(sock->FileDes);
    
    /* Is it even worth returning this error to caller?  It can only
     * happen if our fd got closed by someone else. */
    if (ret == -1)
    {
	/* Should never happen. */
        error(E_WARN, "Couldn't shut down socket %i: %s",
		sock->FileDes, strerror(errno));
    }

    sock->FileDes = -1;
    sock->Connected = FALSE;

    return ret == 0;
}

// "quick" disconnect doesn't send a QUIT command
BOOL Disconnect(SockInfo *sock, BOOL quick)
{
    int rc;
    
    if (sock == NULL)
    {
        error(E_TRACE, "Tried to disconnect a NULL socket");
        return FALSE;
    }
    
    error(E_TRACE, "  Disconnect(%s, %s)", sock->HostName, quick? "TRUE" :
          "FALSE");
    if (!sock->Connected)
    {
        error(E_TRACE, "Tried to disconnect a non-connected socket");
        return FALSE;
    }
    
    if (!quick)
    {
        rc = NetWrite(sock, "QUIT");
        if (rc >= FTP_ERROR_START)
        {
            error(E_WARN, "Couldn't close connection: server response was "
                  "\"%s\"", sock->CmdResponse);
        }
        else if (rc == -1)
        {
	    /* The connection may have been closed, and we may have
	     * just noticed now; don't whine about that */
	    if(!sock->Connected) return TRUE;
            
	    error(E_TRACE, "Couldn't write QUIT command");
        }
    }
    
    // Command response CAN'T be freed here because someone
    // may be interested in it - free it later!
    
    return CloseSocket(sock);
}

BOOL DataDisconnect(SockInfo *sock)
{
    if (sock == NULL)
    {
        error(E_TRACE, "Tried to disconnect a NULL data socket");
        return FALSE;
    }

    error(E_TRACE, "  DataDisconnect(%s)", sock->HostName);
    
    if (!sock->Connected)
    {
        error(E_TRACE, "Tried to disconnect a non-connected data socket");
        return FALSE;
    }
    
    return CloseSocket(sock);
}

BOOL HttpDisconnect(SockInfo *sock)
{
    if (sock == NULL)
    {
        error(E_TRACE, "Tried to disconnect a NULL http socket");
        return FALSE;
    }
    
    error(E_TRACE, "  HttpDisconnect(%s)", sock->HostName);
    
    if (!sock->Connected)
    {
        error(E_TRACE, "Tried to disconnect a non-connected http socket");
        return FALSE;
    }
    
    return CloseSocket(sock);
}

BOOL CancelTransfer(SockInfo *sock, const char *protocol)
{
    //char buffer[10];
    //int rc;
    
    if (!sock || !(sock->Transferring))
    {
        error(E_TRACE, "Didn't cancel transfer because it isn't happening");
        return FALSE;
    }
    error(E_TRACE, "CancelTransfer(%s)", sock->HostName);
    return TRUE;

    // This stuff is NOT USED - doing the FTP abort sequence may be the
    // Right Thing, but it causes huge slowdowns and other problems.
    /*if (!strcasecmp(protocol, "ftp"))
    {
        sprintf(buffer, "%c%c", IAC, IP);
        if (write(sock->FileDes, buffer, 2) != 2)
        {
            error(E_TRACE, "Couldn't send a Telnet interrupt: %s",
                  strerror(errno));
            return FALSE;
        }
        sprintf(buffer, "%c%c", IAC, DM);
        if (send(sock->FileDes, buffer, 2, MSG_OOB) != 2)
        {
            error(E_TRACE, "Couldn't send a Telnet sync: %s", strerror(errno));
            return FALSE;
        }
        rc = NetWrite(sock, "ABOR");
        // If there's a second response to come
        if (rc == 426) // 426 = FTP "Data connection closed"
            GetResponse(sock, NULL, 0);
    }
    sock->Transferring = FALSE;
    return TRUE;*/
}

// Tries to read and write data from/to a number of sockets
// The lists of sockets passed are arrays of pointers to sockets.
int TransferData(SockInfo *read_socks[], int read_sock_count,
                 SockInfo *write_socks[], int write_sock_count, int timeout)
{
    int i, rc, chars_read, chars_written, read_write_count = 0;
    fd_set read_set, write_set;
    time_t start_time;
    struct timeval tv;
    
    if ((read_sock_count == 0) && (write_sock_count == 0))
        return 0;
    
    start_time = time(NULL);
    
    error(E_TRACE, "TransferData(%d, %d, %d, %d, %d)", read_socks,
          read_sock_count, write_socks, write_sock_count, timeout);

    FD_ZERO(&read_set);
    // set up to read data
    if (read_socks)
    {
        for (i = 0; i < read_sock_count; i++)
        {
            // if a socket we're interested in has been disconnected then
            // all bets are off
            if (!read_socks[i] || !(read_socks[i]->Connected))
            {
                error(E_TRACE, "Whoops! A socket's been disconnected");
                return 0;
            }
            FD_SET(read_socks[i]->FileDes, &read_set);
        }
    }
    
    FD_ZERO(&write_set);
    // set up to write data
    if (write_socks)
    {
        for (i = 0; i < write_sock_count; i++)
        {
            // if a socket we're interested in has been disconnected then
            // all bets are off
            if (!write_socks[i] || !(write_socks[i]->Connected))
            {
                error(E_TRACE, "Whoops! A socket's been disconnected");
                return 0;
            }
            error(E_TRACE, "Watching fd %d", write_socks[i]->FileDes);
            FD_SET(write_socks[i]->FileDes, &write_set);
        }
    }
    
    // see if there's any data to read or write
    //error(E_TRACE, "Selecting I/O, timeout is %ds...", tv.tv_sec);
    // we may only have to read, or only have to write
    /* glenn: that's ok */
    do {
        /* We need to set this here--we don't want to reset the timeout
         * on EINTR, and we can't depend on the return value of tv. */
        tv.tv_sec = time(NULL) - start_time + timeout;
        tv.tv_usec = 0;
        rc = select(FD_SETSIZE, &read_set, &write_set, NULL, &tv);
    } while ((rc == -1) && (errno == EINTR));
    //error(E_TRACE, "select() returned %d", rc);
    
    if (rc < 0)
    {
        error(E_WARN, "Couldn't select I/O: %s", strerror(errno));
        return -1;
    }
    else if (rc == 0) // ie. there's no I/O from any of the files
    {
        error(E_TRACE, "No network I/O in %d seconds", timeout);
        return 0;
    }
    
    // first check for read socks
    for (i = 0; i < read_sock_count; i++)
    {
        //error(E_TRACE, "sock count: %d, sock[%d] = %d->%d", sock_count, i,
        //      socks[i], *socks[i]);
        if (!read_socks[i] || !read_socks[i]->Connected ||
            !FD_ISSET(read_socks[i]->FileDes, &read_set))
            continue;
        
        read_write_count++;
        //error(E_TRACE, "Data is available on data socket for %s",
        //      read_socks[i]->HostName);
        //flags = fcntl(socks[i]->FileDes, F_GETFL, 0);
        //error(E_TRACE, ", socket flags are %o", flags);
        chars_read = buf_read(read_socks[i]->FileDes, read_socks[i]->DataBuf,
                              DX_BufferSize);
        if (chars_read > 0)
        {
            read_socks[i]->DataLength = chars_read;
            //error(E_TRACE, "TransferData: read %d bytes", chars_read);
            //error(E_TRACE, "%d:%s", chars_read, socks[i]->DataBuf);
        }
        else if (chars_read == 0)
        {
            read_socks[i]->Eof = TRUE;
        }
        else
        {
            error(E_WARN, "Couldn't read input: %s", strerror(errno));
            return -1;
        }
        //error(E_TRACE, "sock %d has %d bytes", i, read_socks[i]->DataLength);
    }
    
    // now check for write socks
    for (i = 0; i < write_sock_count; i++)
    {
        error(E_TRACE, "Is fd %d set?", write_socks[i]->FileDes);
        if (!write_socks[i] || !write_socks[i]->Connected ||
            !FD_ISSET(write_socks[i]->FileDes, &write_set))
            continue;
        
        read_write_count++;
        error(E_TRACE, "Writing %d bytes to fd %d:",
              write_socks[i]->DataLength, write_socks[i]->FileDes);
        chars_written = buf_write(write_socks[i]->FileDes,
                                  write_socks[i]->DataBuf,
                                  write_socks[i]->DataLength);
        // we need this line to update the current size!
        write_socks[i]->DataWritten = chars_written;
        if (chars_written > 0)
        {
            error(E_TRACE, "Wrote %d bytes", chars_written);
            //error(E_TRACE, "%d:%s", chars_read, socks[i]->DataBuf);
        }
        else if (chars_written == 0)
        {
            error(E_TRACE, "Writing data: EOF");
            write_socks[i]->Eof = TRUE;
        }
        else
        {
            error(E_WARN, "Couldn't write output: %s", strerror(errno));
            return -1;
        }
    }
    //error(E_TRACE, "TransferData() returning: %d socks", read_write_count);
    return read_write_count;
}

// Sends an FTP command and gets the response
int NetWrite(SockInfo *sock, const char *string, ...)
{
    int rc;
    va_list ap;
    char write_text[2048];

    if ((sock == NULL) || (string == NULL) || (!sock->Connected))
        return -1;
    
    va_start(ap, string);
    vsnprintf(write_text, sizeof(write_text) - 2, string, ap);
    strcat(write_text, "\r\n");
    //error(E_TRACE, "Writing to socket for host %s", sock->HostName);
    
    if (write(sock->FileDes, write_text, strlen(write_text)) == -1)
    {
        error(E_WARN, "Couldn't write to socket: %s", strerror(errno));
        return -1;
    }
    strncpy(sock->LastCmd, write_text, sizeof(sock->LastCmd));
    
    //error(E_TRACE, "Written \"%s\"", write_text);
    rc = GetResponse(sock, sock->CmdResponse, RESPONSE_MAX);
    if (rc == DX_E_MORE_DATA)
    {
        sock->MoreResponse = TRUE;
        error(E_TRACE, "Response was truncated");
    }
    else
    {
        sock->MoreResponse = FALSE;
    }
    //error(E_TRACE, "NetWrite returning %d", atoi(sock->CmdResponse));
    if ((rc == DX_OK) || (rc == DX_E_MORE_DATA))
        return atoi(sock->CmdResponse);
    else
        return -1;
}

// Gets a line of response text
BOOL GetResponseLine(SockInfo *sock, char *buffer, int buffer_size)
{
    int rc, chars_read, total_chars = 0;
    fd_set read_socks;
    time_t start_time;
    struct timeval tv;
    char c;

    error(E_TRACE, "Trying to get the response line");
    start_time = time(NULL);
select_again:
    if (sock == NULL)
        return FALSE;
    memset(buffer, 0, buffer_size);
    
    FD_ZERO(&read_socks);
    FD_SET(sock->FileDes, &read_socks);
    
    do {
        /* 30 second timeout on reading the line by default.
         * We need to set this here--we don't want to reset the timeout
         * on EINTR, and we can't depend on the return value of tv. */
        tv.tv_sec = time(NULL) - start_time + RESP_TIMEOUT;
        tv.tv_usec = 0;
        rc = select(FD_SETSIZE, &read_socks, NULL, NULL, &tv);
    } while(rc == -1 && errno == EINTR);
    
    if (rc < 0)
    {
        error(E_WARN, "Couldn't select response line: %s",
              strerror(errno));
        return FALSE;
    }
    else if (rc == 0)
    {
        error(E_TRACE, "Timeout when selecting response line");
        return FALSE;
    }
    
    if (!FD_ISSET(sock->FileDes, &read_socks))
    {
        /* huh? */
        error(E_WARN, "GetResponseLine(): select() > 0 && !FD_ISSET(sock->FileDes), confused");
        return FALSE;
    }
    
    chars_read = buf_read(sock->FileDes, &c, 1);
    // keep reading until we run out of data
    while (chars_read > 0)
    {
        //error(E_TRACE, "Read char '%c', %d chars", c, chars_read);
        if (c == '\r')
        {
            chars_read = read(sock->FileDes, &c, 1);
            continue;
        }
        
        if ((c == '\n') || (c == '\0') || (total_chars >= buffer_size))
        {
            buffer[total_chars] = '\0';
            //error(E_TRACE, "Got response line: \"%s\"", buffer);
            return TRUE;
        }
        
        buffer[total_chars++] = c;
        do {
            chars_read = read(sock->FileDes, &c, 1);
        } while(chars_read == -1 && errno == EINTR);
        //error(E_TRACE, "%d:%s", chars_read, socks[j]->DataBuf);
    }
    
    if (chars_read == 0)
    {
        // chars_read == 0, the socket's closed; mark it as such
        error(E_TRACE, "Socket closed (for %s)", sock->HostName);
        CloseSocket(sock);
        return FALSE;
    }
    
    // chars_read == 0 or -1; we've run out of data for the moment.
    if (errno == EAGAIN || errno == EINPROGRESS)
    {
        error(E_TRACE, "Response line: selecting again");
        goto select_again;
    }
    
    if(chars_read == -1)
    {
        error(E_WARN, "Couldn't get response line: %s (for %s)",
              strerror(errno), sock->HostName);
        return FALSE;
    }
    /* NOTREACHED */
    return FALSE;
}

// Get an FTP response -
// Handles multi-line responses correctly. Probably.
int GetResponse(SockInfo *sock, char *response_buffer, int response_max)
{
    int begin_line, response = -1;
    char read_buffer[256], full_buffer[RESPONSE_MAX];
    BOOL multiline = FALSE;
    BOOL response_complete = FALSE;
    int bufleft = RESPONSE_MAX-1;

    memset(read_buffer, 0, sizeof(read_buffer));
    memset(full_buffer, 0, sizeof(full_buffer));
    if (response_buffer)
        memset(response_buffer, 0, response_max);

    error(E_TRACE, "Trying to get response for fd %d", sock->FileDes);
    
    while (GetResponseLine(sock, read_buffer, sizeof(read_buffer)))
    {
	scnprintf(full_buffer, bufleft, "%s\r\n", read_buffer);
	bufleft -= strlen(read_buffer)-2;
        
	if (response == -1)
        {
	    /* First line received. Grab the response code. */
            response = atoi(read_buffer);
	    /* Is this a multiline response? */
	    if(isdigit(read_buffer[0]) &&
	       isdigit(read_buffer[1]) &&
	       isdigit(read_buffer[2]) && read_buffer[3] == '-')
		multiline = TRUE;
	    else /* No, so we're done. */
		response_complete = TRUE;
        }
	else if (multiline)
        {
	    begin_line = atoi(read_buffer);
	    // It's the end of the multiline response if the line begins
	    // with the same number as it started NOT followed by a -, eg.
	    // "220 Complete." would end the response, "220-Incomplete"
	    // would not.
	    if (isdigit(read_buffer[0]) && isdigit(read_buffer[1]) &&
		isdigit(read_buffer[2]) && read_buffer[3] != '-' &&
		begin_line == response)
	    {
		response_complete = TRUE;
	    }
        }
	
        if (response_complete)
        {
	    strip(full_buffer);
            if (response_buffer)
                strncpy(response_buffer, full_buffer, response_max);

            error(E_TRACE, "    Got response (host %s): \"%s\"",
                  sock->HostName, full_buffer);
            if (strlen(full_buffer) > response_max)
                return DX_E_MORE_DATA;
            else
                return DX_OK;
        }
    }

    // If we got here then there's been an error (or a timeout)
    //error(E_WARN, "Couldn't get the response: %s", strerror(errno));
    return DX_ERROR;
}
