/*
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "eventlog.h"
#include "bn_type.h"
#include "packet.h"


extern t_packet * packet_create(t_packet_class class)
{
    t_packet * temp;
    
    if (class!=packet_class_normal && class!=packet_class_file && class!=packet_class_raw)
    {
	eventlog(eventlog_level_error,"packet_create","invalid packet class %d",(int)class);
        return NULL;
    }
    
    if (!(temp = malloc(sizeof(t_packet))))
    {
	eventlog(eventlog_level_error,"packet_create","unable to allocate memory for temp");
	return NULL;
    }
    
    temp->ref   = 1;
    temp->class = class;
    temp->flags = 0;
    packet_set_size(temp,0);
    
    return temp;
}


extern void packet_destroy(t_packet const * packet)
{
    if (packet)
	pfree((void *)packet,sizeof(t_packet)); /* avoid warning */
    else
	eventlog(eventlog_level_error,"packet_destroy","got NULL packet");
}


extern t_packet * packet_add_ref(t_packet * packet)
{
    if (packet)
	packet->ref++;
    else
	eventlog(eventlog_level_error,"packet_destroy","got NULL packet");
    
    return packet;
}


extern void packet_del_ref(t_packet * packet)
{
    if (packet)
	if (packet->ref<2) /* if would go to zero */
	    packet_destroy(packet);
	else
	    packet->ref--;
    else
	eventlog(eventlog_level_error,"packet_destroy","got NULL packet");
}


extern t_packet_class packet_get_class(t_packet const * packet)
{
    if (packet)
        switch (packet->class)
        {
        case packet_class_normal:
            return packet_class_normal;
        case packet_class_file:
            return packet_class_file;
        case packet_class_raw:
            return packet_class_raw;
        case packet_class_none:
	    return packet_class_none;
        default:
	    eventlog(eventlog_level_error,"packet_get_class","packet has invalid class %d",(int)packet->class);
	    return packet_class_none;
        }
    eventlog(eventlog_level_error,"packet_get_class","got NULL packet");
    return packet_class_none;
}


extern char const * packet_get_class_str(t_packet const * packet)
{
    if (packet)
        switch (packet->class)
        {
        case packet_class_normal:
            return "normal";
        case packet_class_file:
            return "file";
        case packet_class_raw:
            return "raw";
        case packet_class_none:
	    return "none";
        default:
	    eventlog(eventlog_level_error,"packet_get_class_str","packet has invalid class %d",(int)packet->class);
	    return "unknown";
        }
    eventlog(eventlog_level_error,"packet_get_class_str","got NULL packet");
    return "unknown";
}


extern unsigned short packet_get_type(t_packet const * packet)
{
    if (!packet)
    {
	eventlog(eventlog_level_error,"packet_get_type","got NULL packet");
	return 0;
    }
    
    switch (packet->class)
    {
    case packet_class_normal:
	if (packet_get_size(packet)<sizeof(t_normal_header))
	{
	    eventlog(eventlog_level_error,"packet_get_type","packet is shorter than header (len=%u)",packet_get_size(packet));
	    return 0;
	}
	return bn_short_get(packet->u.normal.h.type);
    case packet_class_file:
	if (packet_get_size(packet)<sizeof(t_file_header))
	{
	    eventlog(eventlog_level_error,"packet_get_type","packet is shorter than header (len=%u)",packet_get_size(packet));
	    return 0;
	}
	return bn_short_get(packet->u.file.h.type);
    case packet_class_raw:
	return 0; /* raw packets don't have a type, but don't warn because the packet dump tries anyway */
    default:
	eventlog(eventlog_level_error,"packet_get_type","packet has invalid class %d",(int)packet->class);
	return 0;
    }
}


extern char const * packet_get_type_str(t_packet const * packet, t_packet_dir dir)
{
    if (!packet)
    {
	eventlog(eventlog_level_error,"packet_get_type_str","got NULL packet");
	return "unknown";
    }
    
    switch (dir)
    {
    case packet_dir_from_client:
	switch (packet->class)
	{
	case packet_class_normal:
	    if (packet_get_size(packet)<sizeof(t_normal_header))
	    {
		eventlog(eventlog_level_error,"packet_get_type_str","packet is shorter than header (len=%u)",packet_get_size(packet));
		return "unknown";
	    }
	    switch (bn_short_get(packet->u.normal.h.type))
	    {
	    case CLIENT_COMPINFO1:
		return "CLIENT_COMPINFO1";
	    case CLIENT_COMPINFO2:
		return "CLIENT_COMPINFO2";
	    case CLIENT_COUNTRYINFO:
		return "CLIENT_COUNTRYINFO";
	    case CLIENT_CREATEACCTREQ:
		return "CLIENT_CREATEACCTREQ";
	    case CLIENT_UNKNOWN_2B:
		return "CLIENT_UNKNOWN_2B";
	    case CLIENT_PROGIDENT:
		return "CLIENT_PROGIDENT";
	    case CLIENT_AUTHREQ:
		return "CLIENT_AUTHREQ";
	    case CLIENT_ICONREQ:
		return "CLIENT_ICONREQ";
	    case CLIENT_LADDERSEARCHREQ:
		return "CLIENT_LADDERSEARCHREQ";
	    case CLIENT_CDKEY:
		return "CLIENT_CDKEY";
	    case CLIENT_UNKNOWN_14:
		return "CLIENT_UNKNOWN_14";
	    case CLIENT_TOSREQ:
		return "CLIENT_TOSREQ";
	    case CLIENT_STATSREQ:
		return "CLIENT_STATSREQ";
	    case CLIENT_LOGINREQ:
		return "CLIENT_LOGINREQ";
	    case CLIENT_CDKEY2:
		return "CLIENT_CDKEY2";
	    case CLIENT_CHANGEPASSREQ:
		return "CLIENT_CHANGEPASSREQ";
	    case CLIENT_PLAYERINFOREQ:
		return "CLIENT_PLAYERINFOREQ";
	    case CLIENT_PROGIDENT2:
		return "CLIENT_PROGIDENT2";
	    case CLIENT_JOINCHANNEL:
		return "CLIENT_JOINCHANNEL";
	    case CLIENT_MESSAGE:
		return "CLIENT_MESSAGE";
	    case CLIENT_GAMELISTREQ:
		return "CLIENT_GAMELISTREQ";
	    case CLIENT_STARTGAME1:
		return "CLIENT_STARTGAME1";
	    case CLIENT_UNKNOWN_1B:
		return "CLIENT_UNKNOWN_1B";
	    case CLIENT_STARTGAME3:
		return "CLIENT_STARTGAME3";
	    case CLIENT_STARTGAME4:
		return "CLIENT_STARTGAME4";
	    case CLIENT_CLOSEGAME:
		return "CLIENT_CLOSEGAME";
	    case CLIENT_LEAVECHANNEL:
		return "CLIENT_LEAVECHANNEL";
	    case CLIENT_MAPAUTHREQ:
		return "CLIENT_MAPAUTHREQ";
	    case CLIENT_ADREQ:
		return "CLIENT_ADREQ";
	    case CLIENT_ADACK:
		return "CLIENT_ADACK";
	    case CLIENT_ADCLICK:
		return "CLIENT_ADCLICK";
	    case CLIENT_LADDERREQ:
		return "CLIENT_LADDERREQ";
	    case CLIENT_ECHOREPLY:
		return "CLIENT_ECHOREPLY";
	    case CLIENT_PINGREQ:
		return "CLIENT_PINGREQ";
	    case CLIENT_GAME_REPORT:
		return "CLIENT_GAME_REPORT";
	    case CLIENT_JOIN_GAME:
		return "CLIENT_JOIN_GAME";
	    case CLIENT_STATSUPDATE:
		return "CLIENT_STATSUPDATE";
	    }
	    return "unknown";
	    
	case packet_class_file:
	    if (packet_get_size(packet)<sizeof(t_file_header))
	    {
		eventlog(eventlog_level_error,"packet_get_type_str","packet is shorter than header (len=%u)",packet_get_size(packet));
		return "unknown";
	    }
	    switch (bn_short_get(packet->u.file.h.type))
	    {
	    case CLIENT_FILE_REQ:
		return "CLIENT_FILE_REQ";
	    }
	    return "unknown";
	    
	case packet_class_raw:
	    return "CLIENT_RAW";
	case packet_class_none:
	    return "unknown";
	}
	
	eventlog(eventlog_level_error,"packet_get_type_str","packet has invalid class %d",(int)packet->class);
	return "unknown";
	
    case packet_dir_from_server:
	switch (packet->class)
	{
	case packet_class_normal:
	    if (packet_get_size(packet)<sizeof(t_normal_header))
	    {
		eventlog(eventlog_level_error,"packet_get_type_str","packet is shorter than header (len=%u)",packet_get_size(packet));
		return "unknown";
	    }
	    switch (bn_short_get(packet->u.normal.h.type))
	    {
	    case SERVER_COMPREPLY:
		return "SERVER_COMPREPLY";
	    case SERVER_SESSIONKEY1:
		return "SERVER_SESSIONKEY1";
	    case SERVER_SESSIONKEY2:
		return "SERVER_SESSIONKEY2";
	    case SERVER_CREATEACCTREPLY:
		return "SERVER_CREATEACCTREPLY";
	    case SERVER_AUTHREQ:
		return "SERVER_AUTHREQ";
	    case SERVER_AUTHREPLY:
		return "SERVER_AUTHREPLY";
	    case SERVER_ICONREPLY:
		return "SERVER_ICONREPLY";
	    case SERVER_LADDERSEARCHREPLY:
		return "SERVER_LADDERSEARCHREPLY";
	    case SERVER_CDKEYREPLY:
		return "SERVER_CDKEYREPLY";
	    case SERVER_TOSREPLY:
		return "SERVER_TOSREPLY";
	    case SERVER_STATSREPLY:
		return "SERVER_STATSREPLY";
	    case SERVER_LOGINREPLY:
		return "SERVER_LOGINREPLY";
	    case SERVER_CHANGEPASSACK:
		return "SERVER_CHANGEPASSACK";
	    case SERVER_CDKEYREPLY2:
		return "SERVER_CDKEYREPLY2";
	    case SERVER_PLAYERINFOREPLY:
		return "SERVER_PLAYERINFOREPLY";
	    case SERVER_CHANNELLIST:
		return "SERVER_CHANNELLIST";
	    case SERVER_SERVERLIST:
		return "SERVER_SERVERLIST";
	    case SERVER_MESSAGE:
		return "SERVER_MESSAGE";
	    case SERVER_GAMELISTREPLY:
		return "SERVER_GAMELISTREPLY";
	    case SERVER_STARTGAME1_ACK:
		return "SERVER_STARTGAME1_ACK";
	    case SERVER_STARTGAME3_ACK:
		return "SERVER_STARTGAME3_ACK";
	    case SERVER_STARTGAME4_ACK:
		return "SERVER_STARTGAME4_ACK";
	    case SERVER_MAPAUTHREPLY:
		return "SERVER_MAPAUTHREPLY";
	    case SERVER_ADREPLY:
		return "SERVER_ADREPLY";
	    case SERVER_LADDERREPLY:
		return "SERVER_LADDERREPLY";
	    case SERVER_ECHOREQ:
		return "SERVER_ECHOREQ";
	    case SERVER_PINGREPLY:
		return "SERVER_PINGREPLY";
	    }
	    return "unknown";
	    
	case packet_class_file:
	    if (packet_get_size(packet)<sizeof(t_file_header))
	    {
		eventlog(eventlog_level_error,"packet_get_type_str","packet is shorter than header (len=%u)",packet_get_size(packet));
		return "unknown";
	    }
	    switch (bn_short_get(packet->u.file.h.type))
	    {
	    case SERVER_FILE_REPLY:
		return "SERVER_FILE_REPLY";
	    }
	    return "unknown";
	    
	case packet_class_raw:
	    return "SERVER_RAW";
	    
	case packet_class_none:
	    return "unknown";
	}
	
	eventlog(eventlog_level_error,"packet_get_type_str","packet has invalid class %d",(int)packet->class);
	return "unknown";
    }
    
    eventlog(eventlog_level_error,"packet_get_type_str","got unknown direction %d",(int)dir);
    return "unknown";
}


extern int packet_set_type(t_packet * packet, unsigned short type)
{
    if (!packet)
    {
	eventlog(eventlog_level_error,"packet_set_type","got NULL packet");
	return -1;
    }
    
    switch (packet->class)
    {
    case packet_class_normal:
	if (packet_get_size(packet)<sizeof(t_normal_header))
	{
	    eventlog(eventlog_level_error,"packet_set_type","packet is shorter than header (len=%u)",packet_get_size(packet));
	    return -1;
	}
	bn_short_set(&packet->u.normal.h.type,type);
	return 0;
    case packet_class_file:
	if (packet_get_size(packet)<sizeof(t_file_header))
	{
	    eventlog(eventlog_level_error,"packet_set_type","packet is shorter than header (len=%u)",packet_get_size(packet));
	    return -1;
	}
	bn_short_set(&packet->u.file.h.type,type);
	return 0;
    case packet_class_raw:
	eventlog(eventlog_level_error,"packet_set_type","can not set packet type for raw packet");
	return 0;
    default:
	eventlog(eventlog_level_error,"packet_set_type","packet has invalid class %d",(int)packet->class);
	return -1;
    }
}


extern unsigned short packet_get_size(t_packet const * packet)
{
    unsigned short size;
    
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_get_size","got NULL packet");
	return 0;
    }
    
    switch (packet->class)
    {
    case packet_class_normal:
        size = bn_short_get(packet->u.normal.h.size);
	break;
    case packet_class_file:
        size = bn_short_get(packet->u.file.h.size);
	break;
    case packet_class_raw:
	size = packet->len;
	break;
    default:
	eventlog(eventlog_level_error,"packet_get_size","packet has invalid class %d",(int)packet->class);
	return 0;
    }
    
    if (size>MAX_PACKET_SIZE)
    {
        eventlog(eventlog_level_error,"packet_get_size","packet has bad size %hu",size);
	return 0;
    }
    return size;
}


extern int packet_set_size(t_packet * packet, unsigned short size)
{
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_set_size","got NULL packet");
	return -1;
    }
    if (size>MAX_PACKET_SIZE)
    {
        eventlog(eventlog_level_error,"packet_set_size","got bad size %hu",size);
	return -1;
    }
    
    switch (packet->class)
    {
    case packet_class_normal:
        bn_short_set(&packet->u.normal.h.size,size);
        return 0;
    case packet_class_file:
        bn_short_set(&packet->u.file.h.size,size);
        return 0;
    case packet_class_raw:
	packet->len = size;
	return 0;
    default:
	eventlog(eventlog_level_error,"packet_set_size","packet has invalid class %d",(int)packet->class);
	return -1;
    }
}


extern unsigned int packet_get_flags(t_packet * packet)
{
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_get_flags","got NULL packet");
        return -1;
    }
    
    return packet->flags;
}


extern int packet_set_flags(t_packet * packet, unsigned int flags)
{
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_set_flags","got NULL packet");
        return -1;
    }
    
    packet->flags = flags;
    return 0;
}


extern int packet_append_string(t_packet * packet, char const * str)
{
    unsigned int   len;
    unsigned short addlen;
    unsigned short size;
    
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_append_string","got NULL packet");
        return -1;
    }
    if (!str)
    {
        eventlog(eventlog_level_error,"packet_append_string","got NULL string");
        return -1;
    }
    
    len = strlen(str)+1;
    size = packet_get_size(packet);
    if (size>=MAX_PACKET_SIZE)
        return -1;
    
    addlen = (MAX_PACKET_SIZE-size>len)?len:MAX_PACKET_SIZE-size;
    if (addlen<1)
	return -1;
    
    memcpy(packet->u.data+size,str,addlen-1);
    packet->u.data[size+addlen-1] = '\0';
    packet_set_size(packet,size+addlen);
    
    return 0;
}


extern int packet_append_data(t_packet * packet, void const * data, unsigned int len)
{
    unsigned short addlen;
    unsigned short size;
    
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_append_data","got NULL packet");
        return -1;
    }
    if (!data)
    {
        eventlog(eventlog_level_error,"packet_append_data","got NULL data");
        return -1;
    }
    
    size = packet_get_size(packet);
    if (size>=MAX_PACKET_SIZE)
        return -1;
    
    addlen = (MAX_PACKET_SIZE-size>len)?len:MAX_PACKET_SIZE-size;
    if (addlen<1)
	return -1;
    
    memcpy(packet->u.data+size,data,addlen);
    packet_set_size(packet,size+addlen);
    
    return 0;
}


extern void const * packet_get_raw_data_const(t_packet const * packet, unsigned int offset)
{
    unsigned int size;
    
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_get_raw_data_const","got NULL packet");
        return NULL;
    }
    size = (unsigned int)packet_get_size(packet);
    if (offset>=size || offset>=MAX_PACKET_SIZE)
    {
        eventlog(eventlog_level_error,"packet_get_raw_data_const","got bad offset %u for packet size %u",offset,size);
        return NULL;
    }
    
    return packet->u.data+offset;
}


extern void * packet_get_raw_data(t_packet * packet, unsigned int offset)
{
    unsigned int size;
    
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_get_raw_data","got NULL packet");
        return NULL;
    }
    size = (unsigned int)packet_get_size(packet);
    if (offset>=size || offset>=MAX_PACKET_SIZE)
    {
        eventlog(eventlog_level_error,"packet_get_raw_data","got bad offset %u for packet size %u",offset,size);
        return NULL;
    }
    
    return packet->u.data+offset;
}


extern void * packet_get_raw_data_build(t_packet * packet, unsigned int offset)
{
    unsigned int size;
    
    if (!packet)
    {
        eventlog(eventlog_level_error,"packet_get_raw_data_build","got NULL packet");
        return NULL;
    }
    size = (unsigned int)packet_get_size(packet);
    if (offset>size || offset>=MAX_PACKET_SIZE) /* allow one past the end */
    {
        eventlog(eventlog_level_error,"packet_get_raw_data_build","got bad offset %u for packet size %u",offset,size);
        return NULL;
    }
    
    return packet->u.data+offset;
}


/* maxlen includes room for NUL char */
extern char const * packet_get_str_const(t_packet const * packet, unsigned int offset, unsigned int maxlen)
{
    unsigned int size;
    unsigned int pos;
    
    if (!packet)
    {   
        eventlog(eventlog_level_error,"packet_get_str_const","got NULL packet");
        return NULL;
    }
    size = (unsigned int)packet_get_size(packet);
    if (offset>=size)
    {
        eventlog(eventlog_level_error,"packet_get_str_const","got bad offset %u for packet size %u",offset,size);
        return NULL;
    }
    
    for (pos=offset; packet->u.data[pos]!='\0'; pos++)
	if (pos>=size || pos-offset>=maxlen)
	    return NULL;
    if (pos>=size || pos-offset>=maxlen) /* NUL must be inside too */
	return NULL;
    return packet->u.data+offset;
}


extern void const * packet_get_data_const(t_packet const * packet, unsigned int offset, unsigned int len)
{
    unsigned int size;
    
    if (!packet)
    {   
        eventlog(eventlog_level_error,"packet_get_data_const","got NULL packet");
        return NULL;
    }
    if (len<1)
    {
        eventlog(eventlog_level_error,"packet_get_data_const","got zero length");
	return NULL;
    }
    size = (unsigned int)packet_get_size(packet);
    if (offset+len>size)
    {
        eventlog(eventlog_level_error,"packet_get_data_const","got bad offset %u and length %u for packet size %u",offset,len,size);
        return NULL;
    }
    
    return packet->u.data+offset;
}
