#ifndef lint
static char rcsId[]="$Header: /home/linas/cvsroot/xacc/lib/XmHTML-1.1.0/src/fonts.c,v 1.1 1997/11/30 05:13:09 linas Exp $";
#endif
/*****
* fonts.c : XmHTML font loading & caching routines.
*
* This file Version	$Revision: 1.1 $
*
* Creation date:		Mon Sep 22 09:46:00 GMT+0100 1997
* Last modification: 	$Date: 1997/11/30 05:13:09 $
* By:					$Author: linas $
* Current State:		$State: Exp $
*
* Author:				newt
*
* Copyright (C) 1994-1997 by Ripley Software Development 
* All Rights Reserved
*
* This file is part of the XmHTML Widget Library
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the Free
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/
/*****
* ChangeLog 
* $Log: fonts.c,v $
* Revision 1.1  1997/11/30 05:13:09  linas
* import XmHTML source for the help widget
*
*****/ 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <XmHTMLP.h>
#include <XmHTMLfuncs.h>

/*** External Function Prototype Declarations ***/

/*** Public Variable Declarations ***/
int xmhtml_fn_sizes[8], xmhtml_basefont_sizes[7], xmhtml_fn_fixed_sizes[2];

/*** Private Datatype Declarations ****/
/*****
* A single font cache entry.
* The font cache is a binary tree, sorted according to the name of a font.
* The name field points to a field in the font field.
*****/
typedef struct _fontCacheEntry{
	XmHTMLfont *font;				/* font data */
	String name;					/* font name */
	Boolean is_map;					/* true when this is a font mapping */
	XmHTMLfont *map_to;				/* ptr to real font data */
	struct _fontCacheEntry *left;
	struct _fontCacheEntry *right;
}fontCacheEntry;

/*****
* Definition of a display-bound font cache.
* This structure contains the ID of the display this cache is used for,
* the actual font cache, the default font for this display and a list of
* widgets referencing this cache.
*****/
typedef struct _fontCache{
	Display *dpy;					/* display were fonts come from */
	int res_x;						/* horizontal screen resolution */
	int res_y;						/* vertical screen resolution */
	fontCacheEntry *cache;			/* cached fonts for this display */
	XmHTMLfont *default_font;		/* default font */
	int nwidgets;					/* no of widgets referring this cache */
	WidgetList widgets;				/* array of widget referring this cache */
	struct _fontCache *next;		/* ptr to next cache */
#ifdef DEBUG
	int nentries;					/* no of cached fonts */
	int nmaps;						/* no of mapped fonts */
	int nlookups;					/* no of search actions */
	int requests;					/* no of requests made */
	int hits;						/* no of hits */
	int misses;						/* no of missed requests */
#endif
}fontCache;

/*** Private Function Prototype Declarations ****/
/* create a fully valid XLFD */
static String makeFontName(String name, String foundry, String family,
	String weight, String slant, int points, String charset, String fam_return);

static fontCacheEntry *insertFont(fontCacheEntry *entry, String name,
	XmHTMLfont *font, XmHTMLfont *map_to);

/* get a font from the cache */
static XmHTMLfont *getFont(fontCacheEntry *entry, String name, Byte style);

/* create a map for the given font */
static XmHTMLfont *mapFont(XmHTMLfont *font, String name);

/* allocate a new XmHTMLfont entry */
static XmHTMLfont *allocFont(XFontStruct *xfont, String name, String family,
	Byte style);

/* load or get a font from the cache */
static XmHTMLfont *loadAndCacheFont(Widget w, String name, String family,
	Byte style);

/* free all cached fonts for the given display */
static void freeFontEntries(Display *dpy, fontCacheEntry *fonts);

/* (re-)initialize a font cache */
static void initializeFontSizeLists(XmHTMLWidget html);

/*** Private Variable Declarations ***/
static fontCache *master_cache;		/* master font cache */
static fontCache *curr_cache;		/* current font cache */

/* Backup lists when sizes are not specified */
static int def_fn_sizes[8] = {140,80,240,180,160,140,120,100};
static int def_fn_fixed_sizes[2] = {120,80};

/*****
* Name: 		makeFontName
* Return Type: 	String
* Description: 	creates a full 14 field font name from the given args.
* In: 
*	name:		font base name, required
*	foundry:	font foundry, optional
*	family:		font family, optional
*	weight:		font weight, required
*	slant:		font slant, required
*	points:		font pointsize (tenths of a point), required
*	charset:	current ISO character set encoding
*	fam_return: XmHTML fontFamily spec, updated upon return.
* Returns:
*	the composed font name
* Note:
*	The name argument contains the following string:
*		foundry-family-width-spacing
*	The charset contains the last two fields of a valid font name:
*		iso9959-1
*	When family is specified, foundry will be replaced by a wildcard
*
*	this routine will compose the following fontname given the basename:
*	-foundry-family-weight-slant-width--*-points-res_x-res_y-spacing-*-charset
*****/
static String 
makeFontName(String name, String foundry, String family, String weight, 
	String slant, int points, String charset,
	String fam_return)
{
	int i;
	static char fontfam[512], new_name[1024];
	String fndry, fam, wd, sp;

	strncpy(fontfam, name, 511);
	fontfam[strlen(name)] = '\0';

	/* foundry */
	fndry = &fontfam[0];

	/* family */
	for(i = 0; fontfam[i] != '\0' && fontfam[i] != '-'; i++);
	fontfam[i++] = '\0';
	fam = &fontfam[i];

	/* set width */
	for(; fontfam[i] != '\0' && fontfam[i] != '-'; i++);
	fontfam[i++] = '\0';
	wd = &fontfam[i];

	/* spacing */
	for(; fontfam[i] != '\0' && fontfam[i] != '-'; i++);
	fontfam[i++] = '\0';
	sp = &fontfam[i];

	_XmHTMLFullDebug(8, ("fonts.c: makeFontName, split fontFamily "
		"value %s:\nfoundry : %s\nfamily : %s\nwidth : %s\nspacing : %s\n",
		name, fndry, fam, wd, sp));

	/* screen resolutions are stored in the display-bound font cache */
	sprintf(new_name, "-%s-%s-%s-%s-%s--*-%i-%i-%i-%s-*-%s",
		(foundry != NULL ? foundry : fndry), (family != NULL ? family : fam), 
		weight, slant, wd, points, curr_cache->res_x, curr_cache->res_y,
		sp, charset);

	/* create XmHTML fontFamily spec for this font */
	sprintf(fam_return, "%s-%s-%s-%s", (foundry != NULL ? foundry : fndry),
		(family != NULL ? family : fam), wd, sp);

	/* make it all lowercase */
	my_locase(new_name);

	return(new_name);
}

/*****
* Name: 		insertFont
* Return Type: 	fontCacheEntry
* Description: 	inserts the given font in the given font cache.
* In: 
*	entry:		current font cache;
*	name:		name of font to insert;
*	font:		font data to be inserted;
*	map_to:		original font data if this is a mapping;
* Returns:
*	updated cache entry;
*****/
static fontCacheEntry*
insertFont(fontCacheEntry *entry, String name, XmHTMLfont *font, 
	XmHTMLfont *map_to)
{
	if(entry == NULL)
	{
		/* allocate new font entry */
		entry = (fontCacheEntry*)malloc(sizeof(fontCacheEntry));
		entry->name   = font->font_name;
		entry->font   = font;
		entry->is_map = map_to != NULL;
		entry->map_to = map_to;
		entry->left   = (fontCacheEntry*)NULL;
		entry->right  = (fontCacheEntry*)NULL;
	}
	else
	{
		int ret_val = strncmp(name, entry->name, strlen(name));

		/* duplicate font entries can exist, so check the style as well */
		if(ret_val == 0 && entry->font->style == font->style)
			return(entry);
		if(ret_val < 0)
			entry->left = insertFont(entry->left, name, font, map_to);
		else
			entry->right = insertFont(entry->right, name, font, map_to);
	}
	return(entry);
}

/*****
* Name:			getFont
* Return Type: 	XmHTMLfont
* Description: 	looks for a font in the fontcache;
* In: 
*	entry:		current font cache;
*	name:		name of font to locat;
* Returns:
*	a valid font if name is found, NULL if not found.
*****/
static XmHTMLfont*
getFont(fontCacheEntry *entry, String name, Byte style)
{
	if(entry != NULL)
	{
		int ret_val = strncmp(name, entry->name, strlen(name));
#ifdef DEBUG
		curr_cache->nlookups++;
#endif
		/*****
		* We want the styles to match as well, _XmHTMLloadQueryFont is
		* a bit too smart sometimes.
		*****/
		if(ret_val == 0 && style == entry->font->style)
		{
			_XmHTMLDebug(8,("already cached.\n"));

			if(entry->map_to)
			{
				_XmHTMLDebug(8, ("\t(mapped to %s)\n",
					entry->map_to->font_name));
				return(entry->map_to);
			}
			else
				return(entry->font);
		}
		if(ret_val < 0)
			return(getFont(entry->left, name, style));
		else
			return(getFont(entry->right, name, style));
	}
	return((XmHTMLfont*)NULL);
}

/*****
* Name: 		mapFont
* Return Type:	XmHTMLfont*
* Description: 	creates a font mapping;
* In: 
*	font:		data to which ``name'' is mapped;
*	name:		name of font
* Returns:
*	a new font entry;
*****/
static XmHTMLfont*
mapFont(XmHTMLfont *font, String name)
{
	static XmHTMLfont *map;

	map = (XmHTMLfont*)malloc(sizeof(XmHTMLfont));

	/* copy everything */
	memcpy(map, font, sizeof(XmHTMLfont));

	/* override name */
	map->font_name = strdup(name);
	return(map);
}

/*****
* Name: 		allocFont
* Return Type: 	XmHTMLfont
* Description: 	allocates a new font entry and retrieves all required
*				font properties;
* In: 
*	xfont:		ptr to an X font;
*	name:		name of this font;
*	family:		family to which this font belongs;
*	style:		style of this font, see the FONT_ defines in XmHTMLP.h
* Returns:
*	a new font entry;
*****/
static XmHTMLfont*
allocFont(XFontStruct *xfont, String name, String family, Byte style)
{
	static XmHTMLfont *font;
	unsigned long value = 0;

	font = (XmHTMLfont*)malloc(sizeof(XmHTMLfont));

	/* default items */
	font->xfont = xfont;
	font->font_name = strdup(name);
	font->font_family = strdup(family);
	font->style = style;

	/* size of largest character */
	font->height = xfont->max_bounds.ascent + xfont->max_bounds.descent;

	/* suggested lineheight */
	font->lineheight = xfont->ascent + xfont->descent;

	/* now go get a bunch of properties */

	/* normal interword spacing */
	if((XGetFontProperty(xfont, XA_NORM_SPACE, &value)) == True)
		font->isp = (Cardinal)value;
	else
	{
		/* use width of a single space */
		int dir, ascent, descent;
		XCharStruct sp;
		XTextExtents(font->xfont, " ", 1, &dir, &ascent, &descent, &sp);
		font->isp = sp.width;
	}

	/* additional end-of-line spacing */
	if((XGetFontProperty(xfont, XA_END_SPACE, &value)) == True)
		font->eol_sp = (Cardinal)value;
	else
		font->eol_sp = 0;

	/* superscript x-offset */
	if((XGetFontProperty(xfont, XA_SUPERSCRIPT_X, &value)) == True)
		font->sup_xoffset = (int)value;
	else
		font->sup_xoffset = 0;

	/* superscript y-offset */
	if((XGetFontProperty(xfont, XA_SUPERSCRIPT_Y, &value)) == True)
		font->sup_yoffset = (int)value;
	else
		font->sup_yoffset = (int)(xfont->max_bounds.ascent  * -.4);

	/* subscript x-offset */
	if((XGetFontProperty(xfont, XA_SUBSCRIPT_X, &value)) == True)
		font->sub_xoffset = (int)value;
	else
		font->sub_xoffset = 0;

	/* subscript y-offset */
	if((XGetFontProperty(xfont, XA_SUBSCRIPT_Y, &value)) == True)
		font->sub_yoffset = (int)value;
	else
		font->sub_yoffset = (int)(xfont->max_bounds.descent * .8);

	/* underline offset */
	if((XGetFontProperty(xfont, XA_UNDERLINE_POSITION, &value)) == True)
		font->ul_offset = (int)value;
	else
		font->ul_offset = (int)(xfont->max_bounds.descent-2);

	/* underline thickness */
	if((XGetFontProperty(xfont, XA_UNDERLINE_THICKNESS, &value)) == True)
		font->ul_thickness = (Cardinal)value;
	else
		font->ul_thickness = (Cardinal)1;

	/* strikeout offset */
	if((XGetFontProperty(xfont, XA_STRIKEOUT_ASCENT, &value)) == True)
		font->st_offset = (int)value;
	else
		font->st_offset = (int)(0.5*(xfont->max_bounds.ascent))+3;

	/* strikeout descent */
	if((XGetFontProperty(xfont, XA_STRIKEOUT_DESCENT, &value)) == True)
		font->st_thickness = font->st_offset + (Cardinal)value;
	else
		font->st_thickness = 1;

	return(font);
} 

/*****
* Name:			loadAndCacheFont
* Return Type: 	XmHTMLfont
* Description: 	retrieves a font from the cache or loads one when it isn't
*				already available.
* In: 
*	w:			XmHTMLWidget id;
*	name:		name of font to be loaded;
*	family:		family to which this font belongs;
*	style:		style of this font, see the FONT_ defines in XmHTMLP.h;
* Returns:
*	a valid font if the font was loaded successfully, NULL if not (which
*	should never happen);
*****/
static XmHTMLfont*
loadAndCacheFont(Widget w, String name, String family, Byte style)
{
	XmHTMLfont *font;
	XFontStruct *xfont;

	_XmHTMLDebug(8,( "fonts.c: loadAndCacheFont: checking fontcache for\n%s: ",
		name));

#ifdef DEBUG
	curr_cache->requests++;
#endif

	/* check if we have loaded this font before */
	if((font = getFont(curr_cache->cache, name, style)) != NULL)
	{
#ifdef DEBUG
		curr_cache->hits++;
#endif
		return(font);
	}
#ifdef DEBUG
	curr_cache->misses++;
#endif

	_XmHTMLDebug(8,( "not cached,\ntrying to load..."));

	/* A new font, try to load it */
	xfont = XLoadQueryFont(XtDisplay(w), name);

	/* store it if successfull */
	if(xfont != NULL)
	{
		_XmHTMLDebug(8,( "found.\n"));

		/* get a new fontentry */
		font = allocFont(xfont, name, family, style);

		/* store in the cache */
#ifdef DEBUG
		curr_cache->nentries++;
#endif
		curr_cache->cache = insertFont(curr_cache->cache, name, font, NULL);

		/* return the new font */
		return(font);
	}
#ifdef DEBUG
	else
		_XmHTMLDebug(8,( "failed.\n"));
#endif
	return((XmHTMLfont*)NULL);
}

/*****
* Name:			_XmHTMLloadQueryFont
* Return Type:	XFontStruct*
* Description:	loads a font from the given family in given size, weight and 
*				slant. Loaded fonts are cached to minimize the overhead spent 
*				in XLoadQueryFont().
* In:
*	w:			Widget for which this font is to be loaded.
*	name:		XmHTML fontFamily spec.
*	family:		font family name of the font to load.
*	ptsz:		size of font to load, in tenths of a point
*	style:		style of this font.
*	*loaded:	indicates whether the requested font was loaded or the current 
*				font was returned. When loaded is initially True, a warning
*				message is displayed if the font can't be loaded.
* Returns:
*	A XFontStruct* for the font in the requested family/size and loaded
*	set to True or the current font and loaded set to False.
* Note: 
*	This routine was based on the LoadQueryScalableFont() routine found in
*	O'Reilly's Xlib Programming Manual by Adrian Nye, but that's no longer
*	recognizable...
*
*	This routine goes through *GREAT* lengths to find the requested font, and
*	it will almost never fail (unless the XmNfontFamily/XmNcharset resources
*	form an invalid pair, but then XmHTML will give up on startup immediatly).
*****/
XmHTMLfont*
_XmHTMLloadQueryFont(Widget w, String name, String family, int ptsz,
	Byte style, Boolean *loaded)
{
	String weight, slant, fontname = NULL, charset = NULL;
	XmHTMLWidget html = (XmHTMLWidget)w;
	char fontfamily[1024], font_mapping[1024];
	XmHTMLfont *font = NULL;

	font_mapping[0] = '\0';

	/***** 
	* Okay, now we are going to try and load a font. 
	* Check weight & slant styles. The order in which they are treated
	* is important: bold overrides any FONT_MEDIUM settings and italic 
	* overrides any FONT_REGULAR settings.
	* First attempts are all made with given charset. If no font is found
	* we wildcard it and try all over again.
	*****/
	if(style & FONT_BOLD)
	{
		int num_total = 0;
		
		while(num_total != 2 && font == NULL)
		{
			int num_outer = 0;

			charset = (num_total == 0 ? html->html.charset : "*-*");

			num_total++;

			while(num_outer != 4 && font == NULL)
			{
				int num_inner = 0;

				/* weight can vary between bold, demibold, medium, regular */
				switch(num_outer)
				{
					case 0 : weight = "bold"    ; break;
					case 1 : weight = "demibold"; break;
					case 2 : weight = "medium"  ; break;
					default: weight = "regular" ; break;
				}

				num_outer++;

				/* slant can vary between italic, oblique and roman */
				if(style & FONT_ITALIC)
				{
					while(num_inner < 3 && font == NULL)
					{
						switch(num_inner)
						{
							case 0 : slant = "i"; break; /* italic */
							case 1 : slant = "o"; break; /* oblique */
							default: slant = "r"; break; /* roman */
						}

						num_inner++;

						fontname = makeFontName(name, family ? "*" : NULL,
							family, weight, slant, ptsz, charset, fontfamily);

						font = loadAndCacheFont(w, fontname, fontfamily, style);
						if(font == NULL && font_mapping[0] == '\0')
						{
							strcpy(font_mapping, fontname);
							font_mapping[strlen(fontname)] = '\0';
						}
					}
				}
				else
				{
					slant = "r"; /* roman */

					fontname = makeFontName(name, family ? "*" : NULL,
						family, weight, slant, ptsz, charset, fontfamily);

					font = loadAndCacheFont(w, fontname, fontfamily, style);
					if(font == NULL && font_mapping[0] == '\0')
					{
						strcpy(font_mapping, fontname);
						font_mapping[strlen(fontname)] = '\0';
					}
				}
			}
		}
	}
	/* regular font style */
	else
	{
		int num_total = 0;

		while(num_total != 2 && font == NULL)
		{
			int num_outer = 0;

			charset = (num_total == 0 ? html->html.charset : "*-*");

			num_total++;

			while(num_outer != 2 && font == NULL)
			{
				int num_inner = 0;

				/* weight can vary between medium and regular */
				if(num_outer == 0)
					weight = "medium";
				else
					weight = "regular";

				num_outer++;

				/* slant can vary between italic, oblique and roman */
				if(style & FONT_ITALIC)
				{
					while(num_inner < 3 && font == NULL)
					{
						switch(num_inner)
						{
							case 0 : slant = "i"; break; /* italic */
							case 1 : slant = "o"; break; /* oblique */
							default: slant = "r"; break; /* roman */
						}

						num_inner++;

						fontname = makeFontName(name, family ? "*" : NULL,
							family, weight, slant, ptsz, charset, fontfamily);

						font = loadAndCacheFont(w, fontname, fontfamily, style);
						if(font == NULL && font_mapping[0] == '\0')
						{
							strcpy(font_mapping, fontname);
							font_mapping[strlen(fontname)] = '\0';
						}
					}
				}
				else
				{
					slant = "r"; /* roman */

					fontname = makeFontName(name, family ? "*" : NULL,
						family, weight, slant, ptsz, charset, fontfamily);

					font = loadAndCacheFont(w, fontname, fontfamily, style);
					if(font == NULL && font_mapping[0] == '\0')
					{
						strcpy(font_mapping, fontname);
						font_mapping[strlen(fontname)] = '\0';
					}
				}
			}
		}
	}

	if(font)
	{
		/*****
		* If the requested font was mapped to another font, store the
		* mapping as well since it will be the same for all subsequent
		* requests for this font.
		* loaded is False only when we are just attempting to load a font
		* and we want this thing to fail, so we sure as hell don't want a
		* default mapping then.
		*****/
		if(font_mapping[0] != '\0' && *loaded == False)
		{
			XmHTMLfont *map = mapFont(font, font_mapping);
#ifdef DEBUG
			curr_cache->nentries++;
			curr_cache->nmaps++;
#endif
			curr_cache->cache = insertFont(curr_cache->cache, font_mapping,
									map, font);
		}
		/* we have the font */
		*loaded = True;

		/* return the new font */
		return(font);
	}

	/* we don't have the font */
	if(*loaded)
	{
		_XmHTMLWarning(__WFUNC__(w, "_XmHTMLloadQueryFont"), "Failed to load "
			"font %s\n    Font probably doesn't exist. Ignored.", fontname);
	}

	*loaded = False;

	/* fix 02/03/07-01, dp */
	return(curr_cache->default_font);
}

/*****
* Name:			_XmHTMLaddFontMapping
* Return Type: 	void
* Description: 	add a fontmapping for the the given font. The name of the
*				font to be mapped is created in such a way that it will
*				be the very first match when the _XmHTMLloadQueryFont routine
*				is called. It's primary use is to reduce loading times when
*				switching between documents (we already know which font we will
*				get).
* In: 
*	font:		actual font.
* Returns:
*	nothing.
*****/
void
_XmHTMLaddFontMapping(XmHTMLWidget html, String name, String family,
	int ptsz, Byte style, XmHTMLfont *font)
{
	String fontname = NULL;
	char fontfamily[1024];
	XmHTMLfont *map;

	/*****
	* Create an XLFD that will match on the first run of _XmHTMLloadQueryFont.
	* !!!INCREDIBLE SPEEDUP!!!
	*****/
	fontname = makeFontName(name, family ? "*" : NULL, family,
			style & FONT_BOLD ? "bold" : "medium", 
			style & FONT_ITALIC ? "i"  : "r", ptsz, html->html.charset,
			fontfamily);

	/* add a mapping */
	map = mapFont(font, fontname);

#ifdef DEBUG
	curr_cache->nentries++;
	curr_cache->nmaps++;
#endif
	curr_cache->cache = insertFont(curr_cache->cache, fontname, map, font);
}

/*****
* Name:			freeFontEntries
* Return Type: 	void
* Description: 	releases all fonts in the given cache;
* In: 
*	dpy:		display on which the fonts were allocated;
*	fonts:		cache to be freed;
* Returns:
*	nothing.
*****/
static void
freeFontEntries(Display *dpy, fontCacheEntry *fonts)
{
	if(fonts)
	{
		freeFontEntries(dpy, fonts->left);
		freeFontEntries(dpy, fonts->right);

		_XmHTMLDebug(8, ("fonts.c: freeFontEntries, releasing font\n\t%s\n",
			fonts->font->font_name));

		/* only free the font if it isn't a mapped one */
		if(!fonts->is_map)
		{
			XFreeFont(dpy, fonts->font->xfont);
			free(fonts->font->font_family);
		}

		/* free all allocated strings */
		free(fonts->font->font_name);

		/* free XmHTMLfont entry */
		free(fonts->font);

		/* free cache entry */
		free(fonts);
	}
}

/*****
* Name: 		initializeFontSizeLists
* Return Type: 	void
* Description: 	fills all arrays of font sizes.
* In: 
*	w:			widget containing font size specs.
* Returns:
*	nothing, but the font lists are updated to reflect the new sizes.
*
* The static size lists can cause unexpected results when multiple instances
* of the Widget with different sizelists are being used.
*****/
static void
initializeFontSizeLists(XmHTMLWidget html)
{
	char *chPtr;
	char size_list[64];
	int i;
	Boolean ok;

	_XmHTMLDebug(8,( "fonts.c: initializeFontSizeLists Start\n"));

	/*** Scalable font size list ***/

	/* copy name, it gets destroyed */
	(void)memset(&size_list, 0, 64);
	strncpy(size_list, html->html.font_sizes, 63);	

	/* This list has 8 elements */
	for(chPtr = strtok(size_list, ","), i = 0; i < 8 && chPtr != NULL; 
		chPtr = strtok(NULL, ","), i++)
	{
		if((xmhtml_fn_sizes[i] = 10*atoi(chPtr)) == 0)
			xmhtml_fn_sizes[i] = def_fn_sizes[i];
	}
	/* fill up list if it is not complete */
	if(i != 8)
	{
		for(; i < 8; i++)
			xmhtml_fn_sizes[i] = def_fn_sizes[i];
	}
#ifdef DEBUG
	_XmHTMLDebug(8, ( "fonts.c: initializeFontSizeLists, scalable font "
		"size list:\n"));
	for(i = 0 ; i < 8 ; i++)
		_XmHTMLDebug(8,("%i ", xmhtml_fn_sizes[i]));
#endif

	/*** Fixed font size list ***/

	/* copy name, it gets destroyed */
	(void)memset(&size_list, 0, 64);
	strncpy(size_list, html->html.font_sizes_fixed, 63);

	/* This list has 2 elements */
	for(chPtr = strtok(size_list, ","), i = 0; i < 2 && chPtr != NULL; 
		chPtr = strtok(NULL, ","), i++)
	{
		if((xmhtml_fn_fixed_sizes[i] = 10*atoi(chPtr)) == 0)
			xmhtml_fn_fixed_sizes[i] = def_fn_fixed_sizes[i];
	}
	/* fill up list if it is not complete */
	if(i != 2)
	{
		for(; i < 2; i++)
			xmhtml_fn_fixed_sizes[i] = def_fn_fixed_sizes[i];
	}

	_XmHTMLDebug(8, ( "\nfonts.c: initializeFontSizeLists, fixed font "
		"size list:\n"));
	_XmHTMLDebug(8, ( "%i %i", xmhtml_fn_fixed_sizes[0],
		xmhtml_fn_fixed_sizes[1]));

	/* list of possible font de/increments using the <FONT SIZE=""> element */
	xmhtml_basefont_sizes[0] = xmhtml_fn_sizes[1];	/* sub/superscript size */
	xmhtml_basefont_sizes[1] = xmhtml_fn_sizes[7];	/* H6 size */
	xmhtml_basefont_sizes[2] = xmhtml_fn_sizes[6];	/* H5 size */
	xmhtml_basefont_sizes[3] = xmhtml_fn_sizes[5];	/* H4 size (def font size)*/
	xmhtml_basefont_sizes[4] = xmhtml_fn_sizes[4];	/* H3 size */
	xmhtml_basefont_sizes[5] = xmhtml_fn_sizes[3];	/* H2 size */
	xmhtml_basefont_sizes[6] = xmhtml_fn_sizes[2];	/* H1 size */

#ifdef DEBUG
	_XmHTMLDebug(8, ( "\nfonts.c: initializeFontSizeLists, fallback "
		"font size list:\n"));
	for(i = 0 ; i < 7 ; i++)
		_XmHTMLDebug(8,( "%i ", xmhtml_basefont_sizes[i]));
	_XmHTMLDebug(8, ("\n"));
#endif

	/* First try to load the default font as specified by the resources */
	ok = False;

	html->html.default_font = _XmHTMLloadQueryFont((Widget)html,
		html->html.font_family, NULL, xmhtml_fn_sizes[0],
		FONT_SCALABLE|FONT_REGULAR|FONT_MEDIUM, &ok);

	/***** 
	* We can't load the default font, try again with a wildcarded family.
	* This time die if it fails
	*****/
	if(html->html.default_font == NULL)
	{
		_XmHTMLWarning(__WFUNC__(html, "initializeFontSizeLists"),
			"Failed to load default font %s\n    Guessing for a default font.",
			html->html.font_family);

		ok = True;
		html->html.default_font = _XmHTMLloadQueryFont((Widget)html,
			html->html.font_family, "*", xmhtml_fn_sizes[0],
			FONT_SCALABLE|FONT_REGULAR|FONT_MEDIUM, &ok);

		/* too bad, we absolutely need it */
		if(ok == False)
		{
			/* die */
			_XmHTMLError(__WFUNC__(html,"initializeFontSizeLists"),
				"Failed to find a default font for %s\n    Check previous "
				"messages and adjust default font", html->html.font_family);
		}
	}

	_XmHTMLDebug(8,( "\nfonts.c: initializeFontSizeLists end.\n"));
}

/*****
* Name:			_XmHTMLSelectFontCache
* Return Type: 	XmHTMLfont*
* Description: 	selects a cache according to the display a widget is
*				being displayed on (or creates one if it isn't present yet)
* In: 
*	html:		XmHTMLWidget id;
*	reset:		hard reset flag (used by SetValues when any of the font
*				resources changes);
* Returns:
*	the id of the default font for this cache;
*****/
XmHTMLfont*
_XmHTMLSelectFontCache(XmHTMLWidget html, Boolean reset)
{
	Display *dpy = XtDisplay((Widget)html);
	fontCache *cache;

	_XmHTMLDebug(8, ("fonts.c: _XmHTMLSelectFontCache start\n"));

	for(cache = master_cache; cache != NULL && cache->dpy != dpy;
		cache = cache->next);

	if(cache == NULL)
	{
		int screen = DefaultScreen(dpy);
		cache = (fontCache*)malloc(sizeof(fontCache));
		cache->dpy = dpy;
		cache->cache = (fontCacheEntry*)NULL;
		cache->default_font = (XmHTMLfont*)NULL;
		cache->nwidgets = 1;
		cache->widgets = (WidgetList)malloc(sizeof(Widget));
		cache->widgets[0] = (Widget)html;
		cache->next = (fontCache*)NULL;

		/* obtain screen resolution in dpi */
		cache->res_x =
			DisplayWidth(dpy, screen)/(DisplayWidthMM(dpy,screen)/25.4);
		cache->res_y =
			DisplayHeight(dpy, screen)/(DisplayHeightMM(dpy,screen)/25.4);

		/* adjust resolutions */
		cache->res_x = (cache->res_x < 87 ? 75 : 100);
		cache->res_y = (cache->res_y < 87 ? 75 : 100);

		/* make sure we have the same resolution in both directions */
		if(cache->res_x != cache->res_y)
		{
			if(cache->res_x > cache->res_y)
				cache->res_y = cache->res_x;
			else
				cache->res_x = cache->res_y;
		}

#ifdef DEBUG
		cache->nentries = 0;
		cache->nmaps    = 0;
		cache->nlookups = 0;
		cache->requests = 0;
		cache->hits     = 0;
		cache->misses   = 0;
#endif

		if(master_cache)
		{
			fontCache *tmp;
			for(tmp = master_cache; tmp->next != NULL; tmp = tmp->next);
			tmp->next = cache;
		}
		else
			master_cache = cache;
		_XmHTMLDebug(8, ("fonts.c: _XmHTMLSelectFontCache, created first "
			"entry.\n"));
	}
	else
	{
		int i;

		/* see if we have got a reference for this widget */
		for(i = 0; i < cache->nwidgets && cache->widgets[i] != (Widget)html;
			i++);

		if(i == cache->nwidgets)
		{ 
			_XmHTMLDebug(8, ("fonts.c: _XmHTMLSelectFontCache, adding "
				"reference entry for widget %s.\n", XtName((Widget)html)));

			cache->widgets = (WidgetList)realloc(cache->widgets,
				(cache->nwidgets+1)*sizeof(Widget));
			cache->widgets[cache->nwidgets++] = (Widget)html;
		}
	}

	/*****
	* Only initialize font lists if the cache has changed, when we
	* are forced to do a reset or we haven't got a default font.
	*****/
	if(curr_cache != cache || reset || html->html.default_font == NULL)
	{
		curr_cache = cache;
		initializeFontSizeLists(html);
	}
	curr_cache->default_font = html->html.default_font;

	return(curr_cache->default_font);
}

/*****
* Name:			_XmHTMLUnloadFonts
* Return Type: 	void
* Description: 	removes a widget from the widget list of a display-bound
*				font cache. When the reference count of this cache reaches
*				zero, the cache is released;
* In: 
*	html:		XmHTMLWidget id;
* Returns:
*	nothing.
*****/
void
_XmHTMLUnloadFonts(XmHTMLWidget html)
{
	Display *dpy = XtDisplay((Widget)html);
	fontCache *cache;
	int i;

	/* get current font cache */
	for(cache = master_cache; cache != NULL && cache->dpy != dpy;
		cache = cache->next);

	if(cache == NULL)
	{
		_XmHTMLWarning(__WFUNC__(html, "_XmHTMLUnloadFonts"),
			"Font cache corrupted: could not find an entry for this display!");
			return;
	}
	for(i = 0; i < cache->nwidgets && cache->widgets[i] != (Widget)html; i++);

	if(i == cache->nwidgets)
	{
		_XmHTMLWarning(__WFUNC__(html, "_XmHTMLUnloadFonts"),
			"Font cache corrupted: could find not an entry for this widget!");
			return;
	}

	_XmHTMLDebug(8,( "\nfonts.c: _XmHTMLUnloadFonts, removing reference for "
		"widget %s.\n", XtName((Widget)html)));

	/* invalidate current cache? */
	if(cache == curr_cache)
		curr_cache = (fontCache*)NULL;

	/* remove this widget */
	cache->widgets[i] = (Widget)NULL;
	for(; i < cache->nwidgets - 1; i++)
		cache->widgets[i] = cache->widgets[i+1];

	cache->nwidgets--;

	/* if this was the last widget, free it */
	if(cache->nwidgets == 0)
	{
		fontCache *tmp;

		_XmHTMLDebug(8, ("fonts.c: _XmHTMLUnloadFonts, releasing font "
			"cache.\n"));

		/* special case, first entry is to be freed */
		if(cache == master_cache)
			master_cache = cache->next;
		else
		{
			for(tmp = master_cache; tmp->next != cache; tmp = tmp->next);

			/* connect next entry */
			tmp->next = cache->next;
		}
		/* free the entire list of cached fonts */
		freeFontEntries(dpy, cache->cache);
		free(cache->widgets);
		free(cache);
	}
#ifdef DEBUG
	else
	{
		_XmHTMLDebug(8, ("fonts.c: _XmHTMLUnloadFonts, cache still "
			"referenced by the following widgets:\n"));
		for(i = 0; i < cache->nwidgets; i++)
			_XmHTMLDebug(8, ("\t%s\n", XtName(cache->widgets[i])));
	}
#endif
}

#ifdef DEBUG
void dumpFontCache(fontCacheEntry *entry)
{
	if(entry)
	{
		dumpFontCache(entry->left);

		_XmHTMLDebug(8, ("    %s, style %i\n", entry->name,
			(int)entry->font->style));
		if(entry->is_map)
			_XmHTMLDebug(8, ("    map: %s, style %i\n",
				entry->map_to->font_name, entry->map_to->style));

		dumpFontCache(entry->right);
	}
}

void
dumpFontCacheStats(void)
{
	int i, k = 0;
	fontCache *tmp;

	_XmHTMLDebug(8, ("fonts.c, dumping font cache statistics\n"));

	for(tmp = master_cache; tmp != NULL; tmp = tmp->next, k++)
	{
		_XmHTMLDebug(8, ("Cache id: %i", k));
		if(tmp == curr_cache)
			_XmHTMLDebug(8, (" (current cache)\n"));
		else
			_XmHTMLDebug(8, ("\n"));
		_XmHTMLDebug(8, ("Widgets refererring this cache:\n"));
		for(i = 0; i < tmp->nwidgets; i++)
			_XmHTMLDebug(8, ("\t%s\n", XtName(tmp->widgets[i])));
		_XmHTMLDebug(8, ("Fonts    : %i (%i mapped)\n", tmp->nentries,
			tmp->nmaps));
		_XmHTMLDebug(8, ("requests : %i\n", tmp->requests));
		_XmHTMLDebug(8, ("lookups  : %i\n", tmp->nlookups));
		_XmHTMLDebug(8, ("misses   : %i\n", tmp->misses));
		_XmHTMLDebug(8, ("hits     : %i\n", tmp->hits));
		_XmHTMLDebug(8, ("hit ratio: %.2f%%\n",
			(float)(tmp->hits*100./(float)tmp->requests)));
		_XmHTMLDebug(8, ("Average lookups per font: %.2f\n",
			(float)(tmp->nlookups/(float)tmp->hits)));
		_XmHTMLDebug(8, ("\n"));

		_XmHTMLDebug(8, ("List of fonts:\n"));
			dumpFontCache(tmp->cache);
	}
}
#endif
