/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module parses Malaga values. */

/* includes =================================================================*/

#include <stdio.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "scanner.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL

#include "value_parser.h"

/* functions ================================================================*/

LOCAL void parse_symbol (void)
/* Parse a symbol and push it on the value stack. */
{
  test_token (TOK_IDENT);
  push_symbol_value (find_symbol (token_name));
  read_next_token ();
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_simple_value (void)
/* Parse a value and leave it on the value_stack. */
{
  int_t n; /* number of values in list or record */
  
  switch (next_token) 
  {
  case '<': /* Parse a list. */
    read_next_token ();
    n = 0;
    if (next_token != '>') 
    {
      parse_simple_value ();
      n++;
      while (next_token == ',') 
      {
	read_next_token ();
	parse_simple_value ();
	n++;
      }
    }
    parse_token ('>');
    build_list (n);
    break;
    
  case '[': /* Parse a record. */
    read_next_token ();
    n = 0;
    if (next_token != ']') 
    {
      parse_symbol ();
      parse_token (':');
      parse_simple_value ();
      n++;
      while (next_token == ',') 
      {
	read_next_token ();
	parse_symbol ();
	parse_token (':');
	parse_simple_value ();
	n++;
      }
    }
    parse_token (']');
    build_record (n);
    break;
    
  case TOK_IDENT: /* Parse a symbol. */
    parse_symbol ();
    break;
    
  case TOK_STRING: /* Parse a string. */
#ifdef HANGUL
    encode_hangul (&token_string);
#endif
    push_string_value (token_string, NULL);
    read_next_token ();
    break;
      
  case TOK_NUMBER: /* Parse a number value. */
    push_number_value (token_number);
    read_next_token ();
    break;
    
  default:
    error ("value expected, not %s", token_as_text (next_token));
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void parse_value_string (string_t value_string)
/* STACK EFFECTS: (nothing) -> <new_value>. *
 * Parse <value_string> which must contain a value in text format.
 * <new_value> is the content of <value_string>. */
{
  set_scanner_input (value_string);
  parse_simple_value ();
  parse_token (EOF);
}

/* end of file ==============================================================*/
