/* 
 *  gstalker stock charter
 * 
 *  Copyright (c) 1998 Stefan S. Stratigakos
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 */

#include "gstalker.h"
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>


/****************************************************************************************
Shutdown gstalker nicely.
****************************************************************************************/
void exit_nicely()
{
  save_state();
  exit_program ();
}
/****************************************************************************************
The chart type button was pressed.
****************************************************************************************/
void set_chart_type()
{
  extern struct record config;
  extern GString *current_symbol;
  extern gint chartflag;
  
  
  /* check if a chart is displayed */
  if (! chartflag)
  	return;
  	
  /* get the current chart type and update it to the following type */
  if (strstr (config.chart_type, "Daily"))
  	strcpy(config.chart_type, "Weekly");
  else
  {
  	if (strstr (config.chart_type, "Weekly"))
  		strcpy(config.chart_type, "Monthly");
  	else
  		strcpy(config.chart_type, "Daily");
  }
  
  /* reload the chart to update the type change */
  load_file (current_symbol->str);
}
/*****************************************************************************************
Add a new item to all the arrays that hold the chart data.
****************************************************************************************/
void
chart_array_append (glong date, gfloat open, gfloat high, gfloat low, gfloat close, glong volume, 					   glong openint)
{
  extern GArray *date_array, *open_array, *high_array, *low_array, *close_array;
  extern GArray *volume_array, *openint_array;
  extern gfloat maxhigh, maxlow;
  extern gulong volume_high;
  
  

  /* append the data to each of the arrays */
  g_array_append_val (date_array, date);
  g_array_append_val (open_array, open);
  g_array_append_val (high_array, high);
  g_array_append_val (low_array, low);
  g_array_append_val (close_array, close);
  g_array_append_val (volume_array, volume);
  g_array_append_val (openint_array, openint);
	  
  /* find the highest close value */
  if (high != 0)
  {
      if (high > maxhigh)
	maxhigh = high;
  }
  else
  {
      if (close > maxhigh)
	maxhigh = close;
  }
	    
  /* find the lowest close value */
  if (low != 0)
  {
      if (low < maxlow)
	maxlow = low;
  }
  else
  {
      if (close < maxlow)
	maxlow = close;
  }
	    
  /* find the highest volume value */
  if (volume > volume_high)
    volume_high = volume;
}
/*****************************************************************************************
Converts a passed string date YYYYMMDD to a global GDate structure called gdate.
*****************************************************************************************/
void
create_gdate_from_string (gchar *date_string)
{
  GDateYear year;
  GDateMonth month;
  GDateDay day;
  extern GDate *gdate;
  GString *tstring;
  
  
  tstring = g_string_new (NULL);
  	
  /* parse the year and create a year structure */
  g_string_assign (tstring, date_string);
  g_string_truncate (tstring, 4);
  year = (GDateYear) atoi(tstring->str);
  g_date_set_year (gdate, year);
      
  /* parse the month and create a month structure */
  g_string_assign (tstring, date_string);
  g_string_erase (tstring, 0, 4);
  g_string_truncate (tstring, 2);
  month = (GDateMonth) atoi(tstring->str);
  g_date_set_month (gdate, month);
      
  /* parse the day and create a day structure */
  g_string_assign (tstring, date_string);
  g_string_erase (tstring, 0, 6);
  day = (GDateDay) atoi(tstring->str);
  g_date_set_day (gdate, day);
  
  g_string_free (tstring, TRUE);
}
/******************************************************************************************
Create an MA array and fill it with values from the current chart.
*******************************************************************************************/
void
create_ma_array (gint type)
{
  guint tuint, records;
  gfloat tfloat=0;
  extern struct record config;
  extern GArray *ma_array;
  extern gint current_records, array_size;


  /* create the array */
  if (ma_array)
  	g_array_free(ma_array, TRUE);
  ma_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  
  /* get the maximum number of records for the array */
  if (strstr(config.chart_type, "Daily"))
  {
  	if (config.bars)
  	{
  		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
			records = config.bars;
	}
	else
		records = current_records;
  }
  else
	records = array_size;
  
  /* loop through all the records in the chart */
  for (tuint = 0; tuint < records; tuint++)
  {
  	/* check if we are < starting point for the MA */
  	if (tuint < config.moving_value)
  		/* we are, so fill the array with a zero */
  		tfloat = 0;
  	else
  	{
  		/* get the MA type and calculate the MA */
  		switch (type)
  		{
  			case 1: tfloat = get_simple_movavg (config.moving_value, tuint);
  				    break;
  			case 2: tfloat = get_weighted_movavg (config.moving_value, tuint);
  				    break;
  			case 3: tfloat = get_exponential_movavg (config.moving_value, tuint);
  				    break;
  			default: break;
  		}
  	}
  	/* put the MA into the array */
  	g_array_append_val (ma_array, tfloat);
  }
}
/******************************************************************************************
Same as the above MA array routine.
******************************************************************************************/
void
create_ma2_array (gint type)
{
  guint tuint, records;
  gfloat tfloat=0;
  extern struct record config;
  extern GArray *ma2_array;
  extern gint current_records, array_size;


  if (ma2_array)
  	g_array_free(ma2_array, TRUE);
  ma2_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  
  if (strstr(config.chart_type, "Daily"))
  {
  	if (config.bars)
  	{
  		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
			records = config.bars;
	}
	else
		records = current_records;
  }
  else
	records = array_size;
  
  for (tuint = 0; tuint < records; tuint++)
  {
  	if (tuint < config.moving2_value)
  		tfloat = 0;
  	else
  	{
  		switch (type)
  		{
  			case 1: tfloat = get_simple_movavg (config.moving2_value, tuint);
  				    break;
  			case 2: tfloat = get_weighted_movavg (config.moving2_value, tuint);
  				    break;
  			case 3: tfloat = get_exponential_movavg (config.moving2_value, tuint);
  				    break;
  			default: break;
  		}
  	}
  	g_array_append_val (ma2_array, tfloat);
  }
}
/******************************************************************************************
Same as the above MA2 array routine.
*******************************************************************************************/
void
create_ma3_array (gint type)
{
  guint tuint, records;
  gfloat tfloat=0;
  extern struct record config;
  extern GArray *ma3_array;
  extern gint current_records, array_size;


  if (ma3_array)
  	g_array_free(ma3_array, TRUE);
  ma3_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  
  if (strstr(config.chart_type, "Daily"))
  {
  	if (config.bars)
  	{
  		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
			records = config.bars;
	}
	else
		records = current_records;
  }
  else
	records = array_size;
  
  for (tuint = 0; tuint < records; tuint++)
  {
  	if (tuint < config.moving3_value)
  		tfloat = 0;
  	else
  	{
  		switch (type)
  		{
  			case 1: tfloat = get_simple_movavg (config.moving3_value, tuint);
  				    break;
  			case 2: tfloat = get_weighted_movavg (config.moving3_value, tuint);
  				    break;
  			case 3: tfloat = get_exponential_movavg (config.moving3_value, tuint);
  				    break;
  			default: break;
  		}
  	}
  	g_array_append_val (ma3_array, tfloat);
  }
}
/*******************************************************************************************
Convert passed file date format YYYYMMDD to display date MM/DD/YYYY and put into global
string gstring.
******************************************************************************************/
void
convert_date(gchar *date)
{
  extern GString *gstring;
  
  g_string_truncate (gstring, 0);  
  g_string_append_c(gstring, *(date + 4));
  g_string_append_c(gstring, *(date + 5));
  g_string_append_c(gstring, '/');
  g_string_append_c(gstring, *(date + 6));
  g_string_append_c(gstring, *(date + 7));
  g_string_append_c(gstring, '/');
  g_string_append_c(gstring, *(date));
  g_string_append_c(gstring, *(date + 1));
  g_string_append_c(gstring, *(date + 2));
  g_string_append_c(gstring, *(date + 3));
}
/****************************************************************************************
Convert passed display date format MM/DD/YYYY to file date YYYYMMDD and put into global
string gstring.
*****************************************************************************************/
void
convert_date2 (gchar *date)
{
  extern GString *gstring;
  
  g_string_truncate (gstring, 0);  
  g_string_append_c(gstring, *(date + 6));
  g_string_append_c(gstring, *(date + 7));
  g_string_append_c(gstring, *(date + 8));
  g_string_append_c(gstring, *(date + 9));
  g_string_append_c(gstring, *(date));
  g_string_append_c(gstring, *(date + 1));
  g_string_append_c(gstring, *(date + 3));
  g_string_append_c(gstring, *(date + 4));
}
/**************************************************************************************
Reads the gstalker config file into the config structure.
***************************************************************************************/
void 
get_config ()
{
  gchar *tstring;
  gint tint;
  gfloat tfloat;
  gboolean tbool;
  extern struct record config;



  /* get the home directory */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/home=", &tbool);
  if (tbool)
    config.home[0] = 0;
  else
    {
      strcpy (config.home, tstring);
      g_free (tstring);
    }

  /* get the group directory */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/grouppath=", &tbool);
  if (tbool)
    config.grouppath[0] = 0;
  else
    {
      strcpy (config.grouppath, tstring);
      g_free (tstring);
    }

  /* get the data directory */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/datapath=", &tbool);
  if (tbool)
    config.datapath[0] = 0;
  else
    {
      strcpy (config.datapath, tstring);
      g_free (tstring);
    }

  /* get the errorlog directory */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/errorlog=", &tbool);
  if (tbool)
    config.errorlog[0] = 0;
  else
    {
      strcpy (config.errorlog, tstring);
      g_free (tstring);
    }
    
  /* get the indexpath directory */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/indexpath=", &tbool);
  if (tbool)
    config.indexpath[0] = 0;
  else
    {
      strcpy (config.indexpath, tstring);
      g_free (tstring);
    }

  /* get the version */
  tstring = gnome_config_get_string_with_default ("/gstalker/Version/version=1.2", &tbool);
  if (tbool)
    strcpy (config.version, "1.2");
  else
    {
      strcpy (config.version, tstring);
      g_free (tstring);
    }

  /* get the last opened group */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/groupfile=", &tbool);
  if (tbool)
    config.groupfile[0] = 0;
  else
    {
      strcpy (config.groupfile, tstring);
      g_free (tstring);
    }
    
  /* get the chart type to display */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/chart_type=", &tbool);
  if (tbool)
    strcpy(config.chart_type, "Daily");
  else
    {
      strcpy (config.chart_type, tstring);
      g_free (tstring);
    }

  /* get the maximum # of data points to load for each chart */
  tint = gnome_config_get_int_with_default ("/gstalker/Preferences/bars=0", &tbool);
  if (tbool)
    config.bars = 0;
  else
    config.bars = tint;

  /* get the display status for the date field */
  tint = gnome_config_get_int_with_default ("/gstalker/Preferences/date_flag=1", &tbool);
  if (tbool)
    config.date_flag = 1;
  else
    config.date_flag = tint;

  /* get the display status for the open field */
  tint = gnome_config_get_int_with_default ("/gstalker/Preferences/open_flag=1", &tbool);
  if (tbool)
    config.open_flag = 1;
  else
    config.open_flag = tint;

  /* get the display status for the high field */
  tint = gnome_config_get_int_with_default ("/gstalker/Preferences/high_flag=1", &tbool);
  if (tbool)
    config.high_flag = 1;
  else
    config.high_flag = tint;

  /* get the display status for the low field */
  tint = gnome_config_get_int_with_default ("/gstalker/Preferences/low_flag=1", &tbool);
  if (tbool)
    config.low_flag = 1;
  else
    config.low_flag = tint;

  /* get the display status for the close field */
  tint = gnome_config_get_int_with_default ("/gstalker/Preferences/close_flag=1", &tbool);
  if (tbool)
    config.close_flag = 1;
  else
    config.close_flag = tint;

  /* get the display status for the volume field */
  tint = gnome_config_get_int_with_default ("/gstalker/Preferences/volume_flag=1", &tbool);
  if (tbool)
    config.volume_flag = 1;
  else
    config.volume_flag = tint;

  /* get the display status for the openi field */
  tint = gnome_config_get_int_with_default ("/gstalker/Preferences/openi_flag=1", &tbool);
  if (tbool)
    config.openi_flag = 1;
  else
    config.openi_flag = tint;

  /* get the display status for the grid */
  tint = gnome_config_get_int_with_default ("/gstalker/State/grid=1", &tbool);
  if (tbool)
    config.grid = 1;
  else
    config.grid = tint;

  /* get the red color for the grid */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/grid_red=0", &tbool);
  if (tbool)
    config.grid_red = 0;
  else
    config.grid_red = tint;

  /* get the green color for the grid */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/grid_green=0", &tbool);
  if (tbool)
    config.grid_green = 0;
  else
    config.grid_green = tint;

  /* get the blue color for the grid */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/grid_blue=100", &tbool);
  if (tbool)
    config.grid_blue = 100;
  else
    config.grid_blue = tint;

  /* get the red color for the price line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/price_red=0", &tbool);
  if (tbool)
    config.price_red = 0;
  else
    config.price_red = tint;

  /* get the green color for the price line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/price_green=255", &tbool);
  if (tbool)
    config.price_green = 255;
  else
    config.price_green = tint;

  /* get the blue color for the price line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/price_blue=0", &tbool);
  if (tbool)
    config.price_blue = 0;
  else
    config.price_blue = tint;

  /* get the red color for the volume line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/volume_red=255", &tbool);
  if (tbool)
    config.volume_red = 255;
  else
    config.volume_red = tint;

  /* get the green color for the volume line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/volume_green=0", &tbool);
  if (tbool)
    config.volume_green = 0;
  else
    config.volume_green = tint;

  /* get the blue color for the volume line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/volume_blue=0", &tbool);
  if (tbool)
    config.volume_blue = 0;
  else
    config.volume_blue = tint;

  /* get the red color for the moving line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving_red=255", &tbool);
  if (tbool)
    config.moving_red = 255;
  else
    config.moving_red = tint;

  /* get the green color for the moving line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving_green=0", &tbool);
  if (tbool)
    config.moving_green = 0;
  else
    config.moving_green = tint;

  /* get the blue color for the moving line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving_blue=0", &tbool);
  if (tbool)
    config.moving_blue = 0;
  else
    config.moving_blue = tint;

  /* get the red color for the moving2 line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving2_red=255", &tbool);
  if (tbool)
    config.moving2_red = 255;
  else
    config.moving2_red = tint;

  /* get the green color for the moving2 line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving2_green=0", &tbool);
  if (tbool)
    config.moving2_green = 0;
  else
    config.moving2_green = tint;

  /* get the blue color for the moving2 line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving2_blue=0", &tbool);
  if (tbool)
    config.moving2_blue = 0;
  else
    config.moving2_blue = tint;

  /* get the red color for the moving3 line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving3_red=255", &tbool);
  if (tbool)
    config.moving3_red = 255;
  else
    config.moving3_red = tint;

  /* get the green color for the moving3 line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving3_green=0", &tbool);
  if (tbool)
    config.moving3_green = 0;
  else
    config.moving3_green = tint;

  /* get the blue color for the moving3 line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/moving3_blue=0", &tbool);
  if (tbool)
    config.moving3_blue = 0;
  else
    config.moving3_blue = tint;

  /* get the red color for the background */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/background_red=0", &tbool);
  if (tbool)
    config.background_red = 0;
  else
    config.background_red = tint;

  /* get the green color for the background line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/background_green=0", &tbool);
  if (tbool)
    config.background_green = 0;
  else
    config.background_green = tint;

  /* get the blue color for the background line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/background_blue=0", &tbool);
  if (tbool)
    config.background_blue = 0;
  else
    config.background_blue = tint;

  /* get the red color for the border */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/border_red=255", &tbool);
  if (tbool)
    config.border_red = 255;
  else
    config.border_red = tint;

  /* get the green color for the border line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/border_green=255", &tbool);
  if (tbool)
    config.border_green = 255;
  else
    config.border_green = tint;

  /* get the blue color for the border line */
  tint = gnome_config_get_int_with_default ("/gstalker/Color/border_blue=255", &tbool);
  if (tbool)
    config.border_blue = 255;
  else
    config.border_blue = tint;

  /* get the style status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/style=0", &tbool);
  if (tbool)
    config.style = 0;
  else
    config.style = tint;

  /* get the import field status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/import_field=0", &tbool);
  if (tbool)
    config.import_field = 0;
  else
    config.import_field = tint;

  /* get the import date status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/import_date=0", &tbool);
  if (tbool)
    config.import_date = 0;
  else
    config.import_date = tint;


  /* get the data source URL */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/source=", &tbool);
  if (tbool)
    config.source[0] = 0;
  else
    {
      strcpy (config.source, tstring);
      g_free (tstring);
    }
    
  /* get the group_pointer */
  tint = gnome_config_get_int_with_default ("/gstalker/State/group_pointer=1", &tbool);
  if (tbool)
    config.group_pointer = 1;
  else
    config.group_pointer = tint;
    
  /* get the group_pointer_size */
  tint = gnome_config_get_int_with_default ("/gstalker/State/group_pointer_size=0", &tbool);
  if (tbool)
    config.group_pointer = 0;
  else
    config.group_pointer = tint;
    
  /* get the moving_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving_status=0", &tbool);
  if (tbool)
    config.moving_status = 0;
  else
    config.moving_status = tint;
    
  /* get the moving2_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving2_status=0", &tbool);
  if (tbool)
    config.moving2_status = 0;
  else
    config.moving2_status = tint;    
    
  /* get the moving3_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving3_status=0", &tbool);
  if (tbool)
    config.moving3_status = 0;
  else
    config.moving3_status = tint;
    
  /* get the closeroc_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/closeroc_status=0", &tbool);
  if (tbool)
    config.closeroc_status = 0;
  else
    config.closeroc_status = tint;
    
  /* get the threshold_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/threshold_status=0", &tbool);
  if (tbool)
    config.threshold_status = 0;
  else
    config.threshold_status = tint;
    
  /* get the ma_alert_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/ma_alert_status=0", &tbool);
  if (tbool)
    config.ma_alert_status = 0;
  else
    config.ma_alert_status = tint;    
    
  /* get the ma2_alert_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/ma2_alert_status=0", &tbool);
  if (tbool)
    config.ma2_alert_status = 0;
  else
    config.ma2_alert_status = tint;   
    
  /* get the ma3_alert_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/ma3_alert_status=0", &tbool);
  if (tbool)
    config.ma3_alert_status = 0;
  else
    config.ma3_alert_status = tint;
    
  /* get the volume_status */
  tint = gnome_config_get_int_with_default ("/gstalker/State/volume_status=0", &tbool);
  if (tbool)
    config.volume_status = 0;
  else
    config.volume_status = tint;
    
  /* get the moving_type */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving_type=1", &tbool);
  if (tbool)
    config.moving_type = 1;
  else
    config.moving_type = tint;
    
   /* get the moving2_type */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving2_type=1", &tbool);
  if (tbool)
    config.moving2_type = 1;
  else
    config.moving2_type = tint;
    
   /* get the moving3_type */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving3_type=1", &tbool);
  if (tbool)
    config.moving3_type = 1;
  else
    config.moving3_type = tint;
    
  /* get the moving_value */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving_value=10", &tbool);
  if (tbool)
    config.moving_value = 10;
  else
    config.moving_value = tint;
    
  /* get the moving2_value */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving2_value=10", &tbool);
  if (tbool)
    config.moving2_value = 10;
  else
    config.moving2_value = tint;
    
  /* get the moving3_value */
  tint = gnome_config_get_int_with_default ("/gstalker/State/moving3_value=10", &tbool);
  if (tbool)
    config.moving3_value = 10;
  else
    config.moving3_value = tint;
    
  config.data_window_open = 0;
  config.edit_indicators_window_open = 0;
  config.edit_alerts_window_open = 0;
  config.goto_chart_window_open = 0;
  config.quotes_window_open = 0;
  config.import_edit_rule_window_open = 0;
  config.edit_chart_window_open = 0;
  config.import_window_open = 0;
    
  /* get the closeroc value */
  tfloat = gnome_config_get_float_with_default ("/gstalker/State/closeroc=0.0", &tbool);
  if (tbool)
    config.closeroc = 0.0;
  else
    config.closeroc = tfloat;
    
  /* get the currency */
  tstring = gnome_config_get_string_with_default ("/gstalker/State/currency=", &tbool);
  if (tbool)
    strcpy (config.currency, "US");
  else
    {
      strcpy (config.currency, tstring);
      g_free (tstring);
    }
    
  /* get the width of the main window */
  tint = gnome_config_get_int_with_default ("/gstalker/Geometry/w=-1", &tbool);
  if (tbool)
    config.w = -1;
  else
    config.w = tint;
    
  /* get the height of the main window */
  tint = gnome_config_get_int_with_default ("/gstalker/Geometry/h=-1", &tbool);
  if (tbool)
    config.h = -1;
  else
    config.h = tint;
    
  /* get the x pos of the main window */
  tint = gnome_config_get_int_with_default ("/gstalker/Geometry/x=-1", &tbool);
  if (tbool)
    config.x = -1;
  else
    config.x = tint;
    
  /* get the y pos of the main window */
  tint = gnome_config_get_int_with_default ("/gstalker/Geometry/y=-1", &tbool);
  if (tbool)
    config.y = -1;
  else
    config.y = tint;
}
/**********************************************************************************
Save the config structure to the gstalker config file.
***********************************************************************************/
void 
save_state ()
{
  extern GtkWidget *main_window;
  extern struct record config;

  gnome_config_push_prefix ("/gstalker/");
  
  gdk_window_get_geometry (main_window->window, &config.x, &config.y, &config.w, &config.h, NULL);

  gnome_config_set_int ("Geometry/x", config.x);
  gnome_config_set_int ("Geometry/y", config.y);
  gnome_config_set_int ("Geometry/w", config.w);
  gnome_config_set_int ("Geometry/h", config.h);
  gnome_config_set_string ("State/grouppath", config.grouppath);
  gnome_config_set_string ("State/datapath", config.datapath);
  gnome_config_set_string ("State/errorlog", config.errorlog);
  gnome_config_set_string ("State/indexpath", config.indexpath);
  gnome_config_set_string ("State/home", config.home);
  gnome_config_set_string ("Version/version", config.version);
  gnome_config_set_string ("State/groupfile", config.groupfile);
  gnome_config_set_string ("State/chart_type", config.chart_type);
  gnome_config_set_int ("Preferences/bars", config.bars);
  gnome_config_set_int ("Preferences/date_flag", config.date_flag);
  gnome_config_set_int ("Preferences/open_flag", config.open_flag);
  gnome_config_set_int ("Preferences/high_flag", config.high_flag);
  gnome_config_set_int ("Preferences/low_flag", config.low_flag);
  gnome_config_set_int ("Preferences/close_flag", config.close_flag);
  gnome_config_set_int ("Preferences/volume_flag", config.volume_flag);
  gnome_config_set_int ("Preferences/openi_flag", config.openi_flag);
  gnome_config_set_int ("State/grid", config.grid);
  gnome_config_set_int ("Color/grid_red", config.grid_red);
  gnome_config_set_int ("Color/grid_green", config.grid_green);
  gnome_config_set_int ("Color/grid_blue", config.grid_blue);
  gnome_config_set_int ("Color/price_red", config.price_red);
  gnome_config_set_int ("Color/price_green", config.price_green);
  gnome_config_set_int ("Color/price_blue", config.price_blue);
  gnome_config_set_int ("Color/volume_red", config.volume_red);
  gnome_config_set_int ("Color/volume_green", config.volume_green);
  gnome_config_set_int ("Color/volume_blue", config.volume_blue);
  gnome_config_set_int ("Color/moving_red", config.moving_red);
  gnome_config_set_int ("Color/moving_green", config.moving_green);
  gnome_config_set_int ("Color/moving_blue", config.moving_blue);
  gnome_config_set_int ("Color/moving2_red", config.moving2_red);
  gnome_config_set_int ("Color/moving2_green", config.moving2_green);
  gnome_config_set_int ("Color/moving2_blue", config.moving2_blue);
  gnome_config_set_int ("Color/moving3_red", config.moving3_red);
  gnome_config_set_int ("Color/moving3_green", config.moving3_green);
  gnome_config_set_int ("Color/moving3_blue", config.moving3_blue);
  gnome_config_set_int ("Color/background_red", config.background_red);
  gnome_config_set_int ("Color/background_green", config.background_green);
  gnome_config_set_int ("Color/background_blue", config.background_blue);
  gnome_config_set_int ("Color/border_red", config.border_red);
  gnome_config_set_int ("Color/border_green", config.border_green);
  gnome_config_set_int ("Color/border_blue", config.border_blue);
  gnome_config_set_int ("State/style", config.style);
  gnome_config_set_int ("State/import_field", config.import_field);
  gnome_config_set_int ("State/import_date", config.import_date);
  gnome_config_set_string ("State/source", config.source);
  gnome_config_set_int ("State/group_pointer", config.group_pointer);
  gnome_config_set_int ("State/moving_status", config.moving_status);
  gnome_config_set_int ("State/moving2_status", config.moving2_status);
  gnome_config_set_int ("State/moving3_status", config.moving3_status);
  gnome_config_set_int ("State/closeroc_status", config.closeroc_status);
  gnome_config_set_int ("State/threshold_status", config.threshold_status);
  gnome_config_set_int ("State/ma_alert_status", config.ma_alert_status);
  gnome_config_set_int ("State/ma2_alert_status", config.ma2_alert_status);
  gnome_config_set_int ("State/ma3_alert_status", config.ma3_alert_status);
  gnome_config_set_int ("State/volume_status", config.volume_status);
  gnome_config_set_int ("State/moving_type", config.moving_type);
  gnome_config_set_int ("State/moving2_type", config.moving2_type);
  gnome_config_set_int ("State/moving3_type", config.moving3_type);
  gnome_config_set_int ("State/moving_value", config.moving_value);
  gnome_config_set_int ("State/moving2_value", config.moving2_value);
  gnome_config_set_int ("State/moving3_value", config.moving3_value);
  gnome_config_set_float ("State/closeroc", config.closeroc);
  gnome_config_set_string ("State/currency", config.currency);
  
  gnome_config_pop_prefix ();
  gnome_config_sync ();
}
/**********************************************************************************
Updates the titlebar of any windows that are open.
***********************************************************************************/
void 
update_name_display ()
{
  GString *tstring;
  extern gint open_file_flag;
  extern GtkWidget *main_window;
  extern struct record config;
  extern GString *current_symbol, *current_name;
  

  tstring = g_string_new(NULL);
  
  if (open_file_flag == 1)
    {
      g_string_sprintf (tstring, "gstalker: %s (%s) ", current_name->str, current_symbol->str);
      if (strstr(config.chart_type, "Daily"))
      	g_string_append (tstring, "Daily");
      else
      {
      	if (strstr(config.chart_type, "Weekly"))
      		g_string_append(tstring, "Weekly");
      	else
      		g_string_append(tstring, "Monthly");
      }
    }
  else
    {
      g_string_sprintf (tstring, "gstalker: [%s] - %s (%s) ", config.groupfile,
      				current_name->str, current_symbol->str);
      if (strstr(config.chart_type, "Daily"))
      	g_string_append(tstring, "Daily");
      else
      {
      	if (strstr(config.chart_type, "Weekly"))
      		g_string_append(tstring, "Weekly");
      	else
      		g_string_append(tstring, "Monthly");
      }
    }
    
  if (config.data_window_open)
	show_data_window ();
  if (config.edit_indicators_window_open)
	edit_indicators ();
  if (config.edit_alerts_window_open)
	edit_alerts ();
  if (config.goto_chart_window_open)
	goto_chart ();
  gtk_window_set_title (GTK_WINDOW (main_window), tstring->str);
  
  g_string_free(tstring, TRUE);
}
/***********************************************************************************
Calculates and resizes the chart area when needed.
***********************************************************************************/
void 
resize_chart ()
{
  gint pixelspace, records;
  extern GtkWidget *chart, *scrolled_win;
  extern struct record config;
  extern gint current_line_pixelspace, current_ohlc_pixelspace, current_records, array_size;


  /* get the current pixelspace density */
  if (config.style == 0)
    pixelspace = current_line_pixelspace;
  else
    pixelspace = current_ohlc_pixelspace;
    
  /* get the number of records used in the chart */
  if (strstr(config.chart_type, "Daily"))
  {
  	if (config.bars)
  	{
  		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
			records = config.bars;
	}
	else
		records = current_records;
  }
  else
	records = array_size;
	
  /* resize the widgets */
  gtk_widget_set_usize (GTK_WIDGET (chart), records * pixelspace, 0);
  gtk_widget_set_usize (GTK_WIDGET (scrolled_win), 0, 0);
}
/***********************************************************************************
returns the space available in the chart widget.
************************************************************************************/
int 
get_current_chart_height ()
{
  gint tint = 0;
  extern GtkWidget *chart;
  extern struct record config;


  tint = chart->allocation.height - DATE_SIZE;
  if (config.volume_status)
    tint = tint - VOLUME_SIZE;
  return tint;
}
/**********************************************************************************
calculates a weighted moving average and returns the moving average
***********************************************************************************/
float 
get_weighted_movavg (gint period, guint start)
{
  gint tint, tint2 = 0, tint3 = 0;
  gfloat tfloat = 0.0, result, close;
  extern GArray *close_array;


  for (tint = (start - period), tint2 = 1; tint < start; tint++, tint2++)
    {
      close = g_array_index(close_array, gfloat, tint);
      tfloat = tfloat + close * tint2;
      tint3 = tint3 + tint2;
    }
  result = tfloat / tint3;
  return result;
}
/**********************************************************************************
calculates an exponential moving average and returns the moving average
***********************************************************************************/
float 
get_exponential_movavg (gint period, guint start)
{
  gfloat result, expo, expo2, yesterday, close;
  extern GArray *close_array;


  expo = 2.0 / (period + 1);
  expo2 = 1.0 - expo;
  yesterday = get_simple_movavg (period, start);
  close = g_array_index(close_array, gfloat, start);
  result = (close * expo) + (yesterday * expo2);
  return result;
}
/**********************************************************************************
calculates a simple moving average and returns the moving average
***********************************************************************************/
float 
get_simple_movavg (gint period, guint start)
{
  gint tint;
  gfloat result, tfloat = 0.0, close;
  extern GArray *close_array;


  for (tint = (start - period); tint < start; tint++)
  {
  	close = g_array_index(close_array, gfloat, tint);
    	tfloat = tfloat + close;
  }
  result = tfloat / period;
  return result;
}
/*********************************************************************************
whenever a new chart is loaded this function updates the stats bar
*******************************************************************************/
void 
update_stats_bar ()
{
  gint records;
  GString *tstring, *tstring2, *ch, *cl, *ld, *lc, *nc;
  gfloat tfloat, tfloat2;
  extern GtkWidget *info_bar;
  extern gfloat maxhigh, maxlow;
  extern GArray *date_array, *close_array;
  extern gint chartflag, current_records, array_size;
  extern GString *gstring;
  extern struct record config;
  
  

  if (!chartflag)
    return;
    
  tstring = g_string_new(NULL);
  tstring2 = g_string_new(NULL);
  ch = g_string_new(NULL);
  cl = g_string_new(NULL);
  ld = g_string_new(NULL);
  lc = g_string_new(NULL);
  nc = g_string_new(NULL);
    
  if (strstr(config.chart_type, "Daily"))
  {
  	if (config.bars)
  	{
  		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
			records = config.bars;
	}
	else
		records = current_records;
  }
  else
	records = array_size;

  g_string_sprintf (ch, "%g", maxhigh);
  g_string_sprintf (cl, "%g", maxlow);
  g_string_sprintf (tstring2, "%ld", g_array_index(date_array, glong, records - 1));
  convert_date(tstring2->str);
  g_string_assign (ld, gstring->str);
  g_string_sprintf (lc, "%g", g_array_index(close_array, gfloat, records - 1));
  tfloat = g_array_index(close_array, gfloat, records - 1);
  tfloat2 = g_array_index(close_array, gfloat, records - 2);
  g_string_sprintf (nc, "%g", tfloat - tfloat2);
  
  g_string_sprintf (tstring, "CH=%s CL=%s LD=%s LC=%s NC=%s", ch->str, cl->str, ld->str,
  			lc->str, nc->str);

  gtk_statusbar_pop (GTK_STATUSBAR (info_bar), 1);
  gtk_statusbar_push (GTK_STATUSBAR (info_bar), 1, tstring->str);
  
  g_string_free(tstring, TRUE);
  g_string_free(tstring2, TRUE);
  g_string_free(ch, TRUE);
  g_string_free(cl, TRUE);
  g_string_free(ld, TRUE);
  g_string_free(lc, TRUE);
  g_string_free(nc, TRUE);
}
/*******************************************************************************
this is the gstalker about box
*******************************************************************************/
void 
about ()
{
  gchar *authors[] = {"Stefan S Stratigakos", NULL};
  GtkWidget *about;

  about = gnome_about_new ("gstalker",
			   	   "1.2",
			   	   "Copyright (c) 1998 Stefan S. Stratigakos",
			   	   (gpointer)authors,
			   	   "A Gnome stock and commodities chart viewer.",
			   	   NULL);

  gtk_widget_show (about);
}
/***********************************************************************************
shuts down gstalker
************************************************************************************/
void 
exit_program ()
{
  extern GString *gstring, *lastdir, *error_message, *edit_groupfilename, *date_alert;
  extern GString *edit_symbol, *current_symbol, *current_name, *current_first_date;
  extern GString *current_last_date, *current_file_type;
  extern GDate *gdate;
  

  /* save the config */
  save_state();

  /* free up all the global strings */
  g_string_free (gstring, TRUE);
  g_string_free (lastdir, TRUE);
  g_string_free (error_message, TRUE);
  g_string_free (edit_groupfilename, TRUE);
  g_string_free (date_alert, TRUE);
  g_string_free (edit_symbol, TRUE);
  g_string_free (current_symbol, TRUE);
  g_string_free (current_name, TRUE);
  g_string_free (current_first_date, TRUE);
  g_string_free (current_last_date, TRUE);
  g_string_free (current_file_type, TRUE);
  g_date_free (gdate);
  
  gtk_main_quit ();
}
/***********************************************************************************
turns on or off volume indiactor
*************************************************************************************/
void 
toggle_volume ()
{
  extern gint chartflag;
  extern struct record config;


  if (chartflag)
    {
      if (config.volume_status)
	config.volume_status--;
      else
	config.volume_status++;
      draw ();
      if (config.edit_indicators_window_open)
	update_indicator_window ();
    }
}
/************************************************************************************
turns on or off grid lines
*************************************************************************************/
void 
toggle_grid ()
{
  extern gint chartflag;
  extern struct record config;


  if (chartflag)
    {
      if (config.grid)
	config.grid--;
      else
	config.grid++;
      draw ();
    }
}
/***************************************************************************************
destroy the data window and update flags.
***************************************************************************************/
void 
destroy_data_window ()
{
  extern GtkWidget *data_window;
  extern struct record config;

  gtk_widget_destroy (data_window);
  config.data_window_open = 0;
}
/*****************************************************************************************
update the data window contents.
******************************************************************************************/
void 
update_data_window ()
{
  GString *tstring, *tstring2, *tstring3, *tstring4, *tstring5, *tstring6;
  GString *tstring7, *tstring8, *tstring9, *tstring10, *tstring11, *tstring12;
  gchar *line[11];
  gfloat tfloat = 0.0;
  guint tuint, records;
  extern GtkWidget *data_window, *data_window_clist;
  extern gint open_file_flag, current_records, array_size;
  extern struct record config;
  extern GString *gstring, *current_symbol, *current_name;
  extern GArray *ma_array, *ma2_array, *ma3_array, *date_array, *open_array, *high_array;
  extern GArray *low_array, *close_array, *volume_array, *openint_array;
  extern gchar *label_data_window;
  


  tstring = g_string_new(NULL);
  tstring2 = g_string_new(NULL);
  tstring3 = g_string_new(NULL);
  tstring4 = g_string_new(NULL);
  tstring5 = g_string_new(NULL);
  tstring6 = g_string_new(NULL);
  tstring7 = g_string_new(NULL);
  tstring8 = g_string_new(NULL);
  tstring9 = g_string_new(NULL);
  tstring10 = g_string_new(NULL);
  tstring11 = g_string_new(NULL);
  tstring12 = g_string_new(NULL);
  
  gtk_clist_freeze (GTK_CLIST (data_window_clist));
  gtk_clist_clear (GTK_CLIST (data_window_clist));
  
  /* get the number of records */
  if (strstr(config.chart_type, "Daily"))
  {
  	if (config.bars)
  	{
  		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
			records = config.bars;
	}
	else
		records = current_records;
  }
  else
	records = array_size;

  /* loop through the arrays and convert to strings */
  for (tuint = 0; tuint < records; tuint++)
    {
      g_string_sprintf (tstring12, "%d", tuint + 1);
      line[0] = tstring12->str;
      g_string_sprintf (tstring, "%ld", g_array_index(date_array, glong, tuint));
      convert_date(tstring->str);
      g_string_assign(tstring2, gstring->str);      
      line[1] = tstring2->str;
      g_string_sprintf (tstring3, "%g", g_array_index(open_array, gfloat, tuint));
      line[2] = tstring3->str;
      g_string_sprintf (tstring4, "%g", g_array_index(high_array, gfloat, tuint));
      line[3] = tstring4->str;
      g_string_sprintf (tstring5, "%g", g_array_index(low_array, gfloat, tuint));
      line[4] = tstring5->str;
      g_string_sprintf (tstring6, "%g", g_array_index(close_array, gfloat, tuint));
      line[5] = tstring6->str;
      g_string_sprintf (tstring7, "%ld", g_array_index(volume_array, glong, tuint));
      line[6] = tstring7->str;
      g_string_sprintf (tstring8, "%ld", g_array_index(openint_array, glong, tuint));
      line[7] = tstring8->str;
      line[8] = 0;
      line[9] = 0;
      line[10] = 0;
      
      if (config.moving_status)
      {
		tfloat = g_array_index(ma_array, gfloat, tuint);
	  	g_string_sprintf (tstring9, "%g", tfloat);
	  	line[8] = tstring9->str;
      }
      
      if (config.moving2_status)
      {
		tfloat = g_array_index(ma2_array, gfloat, tuint);
		g_string_sprintf (tstring10, "%g", tfloat);
		line[9] = tstring10->str;
      }
      
      if (config.moving3_status)
      {
		tfloat = g_array_index(ma3_array, gfloat, tuint);
		g_string_sprintf (tstring11, "%g", tfloat);
		line[10] = tstring11->str;
      }
      gtk_clist_append (GTK_CLIST (data_window_clist), line);
    }
    
  gtk_clist_thaw (GTK_CLIST (data_window_clist));

  /* update the window title bar */
  if (open_file_flag == 1)
      g_string_sprintf (tstring, "%s %s (%s)", label_data_window, current_name->str,
      				current_symbol->str);
  else
      g_string_sprintf (tstring, "%s [%s] - %s (%s)", label_data_window, config.groupfile,
      				current_name->str, current_symbol->str);
      
  gtk_window_set_title (GTK_WINDOW (data_window), tstring->str);
  
  /* show or hide the MA column */
  if (config.moving_status)
  	gtk_clist_set_column_visibility (GTK_CLIST (data_window_clist), 8, TRUE);
  else
  	gtk_clist_set_column_visibility (GTK_CLIST (data_window_clist), 8, FALSE);
  
   /* show or hide the MA2 column */
  if (config.moving2_status)
  	gtk_clist_set_column_visibility (GTK_CLIST (data_window_clist), 9, TRUE);
  else
  	gtk_clist_set_column_visibility (GTK_CLIST (data_window_clist), 9, FALSE);
  	
   /* show or hide the MA3 column */
  if (config.moving3_status)
  	gtk_clist_set_column_visibility (GTK_CLIST (data_window_clist), 10, TRUE);
  else
  	gtk_clist_set_column_visibility (GTK_CLIST (data_window_clist), 10, FALSE);
  
  g_string_free(tstring, TRUE);
  g_string_free(tstring2, TRUE);
  g_string_free(tstring3, TRUE);
  g_string_free(tstring4, TRUE);
  g_string_free(tstring5, TRUE);
  g_string_free(tstring6, TRUE);
  g_string_free(tstring7, TRUE);
  g_string_free(tstring8, TRUE);
  g_string_free(tstring9, TRUE);
  g_string_free(tstring10, TRUE);
  g_string_free(tstring11, TRUE);
  g_string_free(tstring12, TRUE);
}
/*****************************************************************************************
creates or updates the data window
******************************************************************************************/
void 
show_data_window ()
{
  gint tint;
  GtkWidget *vbox = NULL, *swindow;
  extern GtkWidget *data_window, *data_window_clist;
  extern gint chartflag;
  extern struct record config;
  extern gchar *label_date, *label_open, *label_high, *label_low, *label_close, *label_volume;
  extern gchar *label_ointerest, *label_ma, *label_ma2, *label_ma3;



  if (!chartflag)
      return;
      
  if (config.data_window_open)
	update_data_window ();
      else
	{
	  data_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	  gtk_signal_connect (GTK_OBJECT (data_window), "destroy",
			      GTK_SIGNAL_FUNC (destroy_data_window), NULL);
	  gtk_container_border_width (GTK_CONTAINER (data_window), 10);
	  gtk_window_position (GTK_WINDOW (data_window), GTK_WIN_POS_CENTER);
	  gtk_widget_set_usize (GTK_WIDGET (data_window), 585, 300);

	  vbox = gtk_vbox_new (FALSE, 0);
	  gtk_container_add (GTK_CONTAINER (data_window), vbox);
	  gtk_widget_show (vbox);
	  
	  swindow = gtk_scrolled_window_new(NULL, NULL);
      	  gtk_box_pack_start (GTK_BOX (vbox), swindow, TRUE, TRUE, 0);
      	  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swindow),
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
          gtk_widget_show(swindow);

	  data_window_clist = gtk_clist_new_with_titles (11, NULL);
	  gtk_clist_set_column_width (GTK_CLIST (data_window_clist), 0, 40);
	  for (tint = 1; tint < 11; tint++)
	    gtk_clist_set_column_width (GTK_CLIST (data_window_clist), tint, 70);
	  gtk_clist_set_selection_mode (GTK_CLIST (data_window_clist), GTK_SELECTION_BROWSE);
	  gtk_container_add (GTK_CONTAINER (swindow), data_window_clist);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 0, "#");
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 1, label_date);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 2, label_open);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 3, label_high);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 4, label_low);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 5, label_close);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 6, label_volume);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 7, label_ointerest);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 8, label_ma);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 9, label_ma2);
	  gtk_clist_set_column_title (GTK_CLIST(data_window_clist), 10, label_ma3);
	  gtk_clist_column_titles_show (GTK_CLIST(data_window_clist));
	  gtk_widget_show (data_window_clist);
	  
	  update_data_window ();

	  gtk_widget_show (data_window);
	  config.data_window_open = 1;
    }
}
/***************************************************************************************
flips the current chart style
***************************************************************************************/
void 
style_button_pressed ()
{
  extern gint chartflag;
  extern struct record config;


  if (!chartflag)
    return;

  if (config.style == 0)
    config.style = 1;
  else
    config.style = 0;
  resize_chart ();
}
/***********************************************************************************
expands the chart spacing by 1 pixel
************************************************************************************/
void 
expand_chart ()
{
  gint pixelspace;
  extern gint chartflag, current_line_pixelspace, current_ohlc_pixelspace;
  extern struct record config;
  extern GString *current_symbol;



  if (chartflag)
    {
      if (config.style == 0)
	pixelspace = current_line_pixelspace;
      else
	pixelspace = current_ohlc_pixelspace;
      pixelspace++;
      if (config.style == 0)
      {
	save_index_line_pixelspace (current_symbol->str, pixelspace);
	current_line_pixelspace++;
      }
      else
      {
	save_index_ohlc_pixelspace (current_symbol->str, pixelspace);
	current_ohlc_pixelspace++;
      }
      resize_chart ();
    }
}
/***********************************************************************************
shrinks the chart spacing by 1 pixel
*************************************************************************************/
void 
shrink_chart ()
{
  gint pixelspace;
  extern gint chartflag, current_line_pixelspace, current_ohlc_pixelspace;
  extern struct record config;
  extern GString *current_symbol;


  if (!chartflag)
    return;
    
  if (! config.style)
    {
      pixelspace = current_line_pixelspace;
      
      if (strstr (config.chart_type, "Monthly"))
      {
      	if (pixelspace == 3)
      		return;
      }
      
      if (pixelspace > 1)
	{
	  pixelspace--;
	  save_index_line_pixelspace (current_symbol->str, pixelspace);
	  current_line_pixelspace--;
	  resize_chart ();
	}
    }
  else
    {
      pixelspace = current_ohlc_pixelspace;
      if (pixelspace > 6)
	{
	  pixelspace--;
	  save_index_ohlc_pixelspace (current_symbol->str, pixelspace);
	  current_ohlc_pixelspace--;
	  resize_chart ();
	}
    }
}
/***********************************************************************************
advances to the next chart in the current group
************************************************************************************/
void 
next_chart ()
{
  gchar *text;
  extern gint open_file_flag, chartflag;
  extern struct record config;
  extern GPtrArray *group_array;


  if (open_file_flag)
    return;
  if (!chartflag)
    return;
    
  if (config.group_pointer == config.group_pointer_size)
  	config.group_pointer = 0;
  else
  	config.group_pointer++;
  	
  text = g_ptr_array_index (group_array, config.group_pointer);
  load_file (text);
  
  if (config.goto_chart_window_open)
  	goto_chart();
}
/***********************************************************************************
reverses to the previous chart in the current group
*************************************************************************************/
void 
previous_chart ()
{
  gchar *text;
  extern gint open_file_flag, chartflag;
  extern struct record config;
  extern GPtrArray *group_array;


  if (open_file_flag)
    return;
  if (!chartflag)
    return;
    
  if (config.group_pointer == 0)
  	config.group_pointer = config.group_pointer_size;
  else
  	config.group_pointer--;
  	
  text = g_ptr_array_index (group_array, config.group_pointer);
  load_file (text);
  
  if (config.goto_chart_window_open)
  	goto_chart();
}
/*************************************************************************************
called if this is the first time gstalker is run
*************************************************************************************/
void 
new_init ()
{
  gchar *gs_home;
  extern struct record config;


  printf ("\nCreating my data directories in your home directory.\n");

  gs_home = g_get_home_dir();
  if (gs_home != NULL)
    {
      strcpy (config.home, gs_home);
      strcat (config.home, "/gstalker");
    }
  else
    strcpy (config.home, "/tmp/gstalker");

  printf (config.home);
  printf ("\n");

  strcpy (config.grouppath, config.home);
  strcat (config.grouppath, "/groups/");

  strcpy (config.datapath, config.home);
  strcat (config.datapath, "/data/");

  strcpy (config.errorlog, config.home);
  strcat (config.errorlog, "/error.log");
  
  strcpy (config.indexpath, config.home);
  strcat (config.indexpath, "/index/");

  mkdir (config.home, 0777);
  mkdir (config.grouppath, 0777);
  mkdir (config.datapath, 0777);
  mkdir (config.indexpath, 0777);
  
 gnome_config_set_string("/gstalker/Import/Sample=", IMPORT_SAMPLE_RULE);
 gnome_config_set_string("/gstalker/Import/Yahoo=", IMPORT_YAHOO_RULE);
}
/************************************************************************************
sets up environment for gstalker on startup
*************************************************************************************/
void 
init ()
{
  gint tint;
  extern struct record config;


  tint = chdir (config.grouppath);
  if (tint)
    mkdir (config.grouppath, 0777);

  tint = chdir (config.datapath);
  if (tint)
    mkdir (config.datapath, 0777);
    
  tint = chdir (config.indexpath);
  if (tint)
    mkdir (config.indexpath, 0777);

}
/****************************************************************************************
this catches the mouse pointer position over the chart and updates the data bar
*****************************************************************************************/
void 
motion_notify_event (GtkWidget * widget, GdkEventMotion * event)
{
  gint x, y, tint, tint2, current_height, current_width, volume_flag = 0, records;
  gint pixelspace;
  GdkModifierType state;
  GString *tstring, *tstring2;
  extern GtkWidget *data_bar, *chart;
  extern gint chartflag, current_records, current_line_pixelspace, current_ohlc_pixelspace;
  extern gint array_size;
  extern gfloat range, maxlow;
  extern struct record config;
  extern GString *gstring;
  extern GArray *date_array, *open_array, *high_array, *low_array, *close_array;
  extern GArray *volume_array, *openint_array;
  


  if (!chartflag)
    return;
    
  tstring = g_string_new(NULL);
  tstring2 = g_string_new(NULL);
    
  if (event->is_hint)
    gdk_window_get_pointer (event->window, &x, &y, &state);
  else
    {
      x = event->x;
      y = event->y;
      state = event->state;
    }
    
  if (config.volume_status)
    volume_flag++;
    
  current_height = get_current_chart_height ();
  current_width = chart->allocation.width;
  
  if (config.style == 0)
    pixelspace = current_line_pixelspace;
  else
    pixelspace = current_ohlc_pixelspace;
    
  if (strstr(config.chart_type, "Daily"))
  {
  	if (config.bars)
  	{
  		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
			records = config.bars;
	}
	else
		records = current_records;
  }
  else
	records = array_size;
    
  tint = x / pixelspace;
  if (((tint) < records) && (x <= current_width) && (y <= current_height))
    {
      if (config.date_flag)
	{
	  g_string_sprintf (tstring2, "%ld", g_array_index(date_array, glong, tint));
	  convert_date(tstring2->str);
	  g_string_assign (tstring, "D=");
	  g_string_append (tstring, gstring->str);
	}
      if (config.open_flag)
	{
	  g_string_append (tstring, " O=");
	  g_string_sprintfa (tstring, "%g", g_array_index(open_array, gfloat, tint));
	}
      if (config.high_flag)
	{
	  g_string_append (tstring, " H=");
	  g_string_sprintfa (tstring, "%g", g_array_index(high_array, gfloat, tint));
	}
      if (config.low_flag)
	{
	  g_string_append (tstring, " L=");
	  g_string_sprintfa (tstring, "%g", g_array_index(low_array, gfloat, tint));
	}
      if (config.close_flag)
	{
	  g_string_append (tstring, " C=");
	  if (state & GDK_BUTTON1_MASK)
	    {
	      y = current_height - y;
	      g_string_sprintfa (tstring, "%g", ((range / current_height) * y) + maxlow);
	    }
	  else
	    g_string_sprintfa (tstring, "%g", g_array_index(close_array, gfloat, tint));
	}
      if (config.volume_flag)
	{
	  g_string_append (tstring, " V=");
	  g_string_sprintfa (tstring, "%ld", g_array_index(volume_array, glong, tint));
	}
      if (config.openi_flag)
	{
	  g_string_append (tstring, " OI=");
	  g_string_sprintfa (tstring, "%ld", g_array_index(openint_array, glong, tint));
	}
	gnome_appbar_set_status (GNOME_APPBAR(data_bar), tstring->str);
    }
  else
    {
      if (config.volume_flag)
      {
      	tint2 = current_height + VOLUME_SIZE;
      	if (((tint) < records) && (x <= current_width) && (y >= current_height) &&
      	    (y <= tint2) && (volume_flag))
	{
	  g_string_sprintf (tstring2, "%ld", g_array_index(date_array, glong, tint));
	  convert_date(tstring2->str);
	  g_string_assign (tstring, "D=");
	  g_string_append (tstring, gstring->str);
	  g_string_append (tstring, " V=");
	  g_string_sprintfa (tstring, "%ld", g_array_index(volume_array, glong, tint));
	  gnome_appbar_set_status (GNOME_APPBAR(data_bar), tstring->str);
	}
      }
    }
    
    g_string_free(tstring, TRUE);
    g_string_free(tstring2, TRUE);
}
/***************************************************************************************
creates or erases the pixmap on which to draw the chart
****************************************************************************************/
void 
configure_event (GtkWidget * widget, GdkEventConfigure * event)
{
  GdkColor *color;
  GdkGC *gc;
  extern GdkPixmap *chart_pixmap;
  extern gint chartflag;
  extern struct record config;


  gc = gdk_gc_new (widget->window);
  color = (GdkColor *) malloc (sizeof (GdkColor));
  color->red = config.background_red * (65535 / 255);
  color->green = config.background_green * (65535 / 255);
  color->blue = config.background_blue * (65535 / 255);
  gdk_color_alloc (gtk_widget_get_colormap (widget), color);
  gdk_gc_set_foreground (gc, color);

  if (chart_pixmap)
    gdk_pixmap_unref (chart_pixmap);
  chart_pixmap = gdk_pixmap_new (widget->window,
				 widget->allocation.width,
				 widget->allocation.height,
				 -1);

  gdk_draw_rectangle (chart_pixmap, gc, TRUE, 0, 0,
		      widget->allocation.width,
		      widget->allocation.height);

  if (chartflag)
    draw ();
}
/***************************************************************************************
creates or erases the scale pixmap on which to draw the price scale
****************************************************************************************/
void 
scale_configure_event (GtkWidget * widget, GdkEventConfigure * event)
{
  GdkColor *color;
  GdkGC *gc;
  extern GdkPixmap *scale_pixmap;
  extern gint chartflag;
  extern struct record config;


  gc = gdk_gc_new (widget->window);
  color = (GdkColor *) malloc (sizeof (GdkColor));
  color->red = config.background_red * (65535 / 255);
  color->green = config.background_green * (65535 / 255);
  color->blue = config.background_blue * (65535 / 255);
  gdk_color_alloc (gtk_widget_get_colormap (widget), color);
  gdk_gc_set_foreground (gc, color);

  if (scale_pixmap)
    gdk_pixmap_unref (scale_pixmap);
  scale_pixmap = gdk_pixmap_new (widget->window,
				 widget->allocation.width,
				 widget->allocation.height,
				 -1);

  gdk_draw_rectangle (scale_pixmap, gc, TRUE, 0, 0,
		      widget->allocation.width,
		      widget->allocation.height);

  if (chartflag)
    draw_scale ();
}
/*************************************************************************************
this catches the X event to redraw the chart if it's covered by another window
**************************************************************************************/
void 
expose_event (GtkWidget * widget, GdkEventExpose * event)
{
  extern GdkPixmap *chart_pixmap;

  gdk_draw_pixmap (widget->window,
		   widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		   chart_pixmap,
		   event->area.x, event->area.y,
		   event->area.x, event->area.y,
		   event->area.width, event->area.height);
}
/*************************************************************************************
this catches the X event to redraw the price scale if it's covered by another window
*************************************************************************************/
void 
scale_expose_event (GtkWidget * widget, GdkEventExpose * event)
{
  extern GdkPixmap *scale_pixmap;

  gdk_draw_pixmap (widget->window,
		   widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		   scale_pixmap,
		   event->area.x, event->area.y,
		   event->area.x, event->area.y,
		   event->area.width, event->area.height);
}
/****************************************************************************************
this catches the left mouse button press and updates the data window
*****************************************************************************************/
void 
button_press_event (GtkWidget * widget, GdkEventButton * event)
{
  GtkVisibility visible;
  gint x, y, tint, current_width, current_height, pixelspace, records;
  extern GdkPixmap *chart_pixmap;
  extern GtkWidget *chart, *data_window_clist;
  extern gint chartflag, current_records, current_line_pixelspace, current_ohlc_pixelspace;
  extern gint array_size;
  extern struct record config;



  if (!chartflag)
    return;
    
  x = event->x;
  y = event->y;
  
  current_height = get_current_chart_height ();
  current_width = chart->allocation.width;
  
  if (config.style == 0)
    pixelspace = current_line_pixelspace;
  else
    pixelspace = current_ohlc_pixelspace;
    
  if (strstr(config.chart_type, "Daily"))
  {
  	if (config.bars)
  	{
  		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
			records = config.bars;
	}
	else
		records = current_records;
  }
  else
	records = array_size;  
    
  tint = x / pixelspace;
  if (((tint) < records) && (x <= current_width) && (y <= current_height))
    {
      if ((event->button == 1) && (chart_pixmap != NULL) && (config.data_window_open != 0))
	{
	  gtk_clist_freeze (GTK_CLIST (data_window_clist));
	  gtk_clist_select_row (GTK_CLIST (data_window_clist), tint, 0);
	  visible = gtk_clist_row_is_visible (GTK_CLIST (data_window_clist), tint);
	  if ((visible == GTK_VISIBILITY_PARTIAL) || (visible == GTK_VISIBILITY_NONE))
	    gtk_clist_moveto (GTK_CLIST (data_window_clist), tint, 0, 0.5, 0);
	  gtk_clist_thaw (GTK_CLIST (data_window_clist));
	}
    }
}

