/* 
 *  gstalker stock charter
 * 
 *  Copyright (c) 1998 Stefan S. Stratigakos
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 */


#include "gstalker.h"

/***********************************************************************************
Selects all the charts in the alert scan listbox.
************************************************************************************/
void
alert_select_all()
{
  extern GtkWidget *alert_scan_clist;
  
  gtk_clist_select_all (GTK_CLIST (alert_scan_clist));
}
/**********************************************************************************
Clears all the selections in the alert scan listbox.
**********************************************************************************/
void
alert_unselect_all()
{
  extern GtkWidget *alert_scan_clist;
  
  gtk_clist_unselect_all (GTK_CLIST (alert_scan_clist));
}
/*************************************************************************************
Updates the status of the alerts in the alert window.
**************************************************************************************/
void
update_alerts_window()
{
  GString *tstring;
  extern struct record config;
  extern GtkWidget *ea_ma_check, *ea_ma2_check, *ea_ma3_check, *ea_closeroc_check;
  extern GtkWidget *ea_threshold_check, *ea_ma_label, *ea_ma2_label;
  extern GtkWidget *ea_ma3_label, *ea_closeroc_spinner, *ea_threshold_spinner;
  extern GtkWidget *ea_hbox2, *ea_hbox3, *ea_hbox4;
  extern gfloat current_threshold;
  extern gchar *label_simple, *label_weighted, *label_exponential;


  tstring = g_string_new(NULL);
  
  /* only update if the MA indicator is active */
  if (config.moving_status)
  {
  	/* show the MA alert widget */
  	gtk_widget_show (ea_hbox2);
  	
  	/* update the check box */
  	if (config.ma_alert_status)
	  	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(ea_ma_check), TRUE);
	
	/* find out what kind of MA is used and update the label */
	if (config.moving_type == 1)
  	{
  		/* it's a simple MA */
  		g_string_assign(tstring, label_simple);
  		g_string_sprintfa(tstring, "%i", config.moving_value);
		gtk_label_set_text (GTK_LABEL (ea_ma_label), tstring->str);
  	}
  	else
  	{
  		if (config.moving_type == 2)
  		{
  			/* its a weighted MA */
  			g_string_assign(tstring, label_weighted);
  			g_string_sprintfa(tstring, "%i", config.moving_value);
			gtk_label_set_text (GTK_LABEL (ea_ma_label), tstring->str);
		}
		else
		{
			/* its a exponential MA */
			g_string_assign(tstring, label_weighted);
  			g_string_sprintfa(tstring, "%i", config.moving_value);
			gtk_label_set_text (GTK_LABEL (ea_ma_label), tstring->str);
		}
  	}
  }
  else
  	/* the MA indicator is not active so hide the widget */
  	gtk_widget_hide (ea_hbox2);
  
  /* same as the above MA indicator */
  if (config.moving2_status)
  {
  	gtk_widget_show (ea_hbox3);
  	
  	if (config.ma2_alert_status)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(ea_ma2_check), TRUE);
		
	if (config.moving2_type == 1)
  	{
  		g_string_assign(tstring, label_simple);
  		g_string_sprintfa(tstring, "%i", config.moving2_value);
		gtk_label_set_text (GTK_LABEL (ea_ma2_label), tstring->str);
  	}
  	else
  	{
  		if (config.moving2_type == 2)
  		{
  			g_string_assign(tstring, label_weighted);
  			g_string_sprintfa(tstring, "%i", config.moving2_value);
			gtk_label_set_text (GTK_LABEL (ea_ma2_label), tstring->str);
		}
		else
		{
			g_string_assign(tstring, label_exponential);
  			g_string_sprintfa(tstring, "%i", config.moving2_value);
			gtk_label_set_text (GTK_LABEL (ea_ma2_label), tstring->str);
		}
  	}
  }
  else
  	gtk_widget_hide (ea_hbox3);
  
  if (config.moving3_status)
  {
  	gtk_widget_show (ea_hbox4);
  	
  	if (config.ma3_alert_status)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(ea_ma3_check), TRUE);
	
	if (config.moving3_type == 1)
  	{
  		g_string_assign(tstring, label_simple);
  		g_string_sprintfa(tstring, "%i", config.moving3_value);
		gtk_label_set_text (GTK_LABEL (ea_ma3_label), tstring->str);
  	}
  	else
  	{
  		if (config.moving3_type == 2)
  		{
  			g_string_assign(tstring, label_weighted);
  			g_string_sprintfa(tstring, "%i", config.moving3_value);
			gtk_label_set_text (GTK_LABEL (ea_ma3_label), tstring->str);
		}
		else
		{
			g_string_assign(tstring, label_exponential);
  			g_string_sprintfa(tstring, "%i", config.moving3_value);
			gtk_label_set_text (GTK_LABEL (ea_ma3_label), tstring->str);
		}
  	}
  }
  else
  	gtk_widget_hide (ea_hbox4);
		
  /* update the checkboxes for the rest of the alerts */
  if (config.closeroc_status)
	  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(ea_closeroc_check), TRUE);
  if (config.threshold_status)
	  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(ea_threshold_check), TRUE);
	  
  /* set the value of the alerts */
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(ea_closeroc_spinner), config.closeroc);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(ea_threshold_spinner), current_threshold);
  
  g_string_free(tstring, TRUE); 
}
/*************************************************************************************
The MA alert checkbox was checked.
*************************************************************************************/
void
ea_ma_checked()
{
  extern struct record config;
  
  /* update the status flag */
  if (config.ma_alert_status)
  	config.ma_alert_status = 0;
  else
  	config.ma_alert_status = 1;
}
/*************************************************************************************
The MA2 alert checkbox was checked.
*************************************************************************************/
void
ea_ma2_checked()
{
  extern struct record config;
  
  /* update the status flag */
  if (config.ma2_alert_status)
  	config.ma2_alert_status = 0;
  else
  	config.ma2_alert_status = 1;
}
/*************************************************************************************
The MA3 alert checkbox was checked.
**************************************************************************************/
void
ea_ma3_checked()
{
  extern struct record config;
  
  /* update the status flag */
  if (config.ma3_alert_status)
  	config.ma3_alert_status = 0;
  else
  	config.ma3_alert_status = 1;
}
/*************************************************************************************
The closeroc alert checkbox was checked.
*************************************************************************************/
void
ea_closeroc_checked()
{
  extern struct record config;
  
  /* update the status flag */
  if (config.closeroc_status)
  	config.closeroc_status = 0;
  else
  	config.closeroc_status = 1;
}
/*************************************************************************************
The threshold alert checkbox was checked.
**************************************************************************************/
void
ea_threshold_checked()
{
  extern struct record config;
  
  /* update the status flag */
  if (config.threshold_status)
  	config.threshold_status = 0;
  else
  	config.threshold_status = 1;
}
/*************************************************************************************
The closeroc alert spinbutton was changed.
**************************************************************************************/
void
ea_closeroc_spinner_changed()
{
  extern struct record config;
  extern GtkWidget *ea_closeroc_spinner;
  
  /* update the value of the alert from the spinbutton */
  config.closeroc = gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON(ea_closeroc_spinner));
}
/*************************************************************************************
The threshold alert spinbutton was changed.
*************************************************************************************/
void
ea_threshold_spinner_changed()
{
  extern GtkWidget *ea_threshold_spinner;
  extern GString *current_symbol;
  extern gfloat current_threshold;
  
  /* update the value of the alert from the spinbutton and save the value to the index file */
  current_threshold = gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON(ea_threshold_spinner));
  save_index_threshold(current_symbol->str, current_threshold);
  
}
/*****************************************************************************
Find any MA alerts, array is 1,2,3 value, meaning MA,MA2,MA3 arrays.
*****************************************************************************/
void 
scan_ma (gint array)
{
  glong tlong;
  gfloat today, yesterday, ma=0;
  guint tuint, records;
  gchar *line[3];
  GString *tstring;
  extern GtkWidget *alert_window_clist;
  extern GArray *date_array, *close_array, *ma_array, *ma2_array, *ma3_array;
  extern GString *gstring, *current_symbol;
  extern gint current_records;
  extern struct record config;
  extern gchar *label_ma_gt_close, *label_ma2_gt_close, *label_ma3_gt_close, *label_ma_lt_close;
  extern gchar *label_ma2_lt_close, *label_ma3_lt_close;
  
  
  
  tstring = g_string_new(NULL);
  
  if (config.bars)
  {
  	/* get the smallest number of data to load, either total records or maximum */
      	if (current_records < config.bars)
		records = current_records;
      	else
		records = config.bars;
  }
  else
	records = current_records;
  
  /* loop through the MA array */
  for (tuint = 1; tuint < records; tuint++)
  {
  	/* get the close and ma values */ 
  	today  = g_array_index(close_array, gfloat, tuint);
  	yesterday = g_array_index(close_array, gfloat, tuint - 1);
  	switch (array)
  	{
  		case 1: ma = g_array_index(ma_array, gfloat, tuint);
  				break;
  		case 2: ma = g_array_index(ma2_array, gfloat, tuint);
  				break;
  		case 3: ma = g_array_index(ma3_array, gfloat, tuint);
  				break;
  		default: break;
  	}
  	
  	/* check if close plot has crossed the MA plot and was below it previously */
  	if ((today >= ma) && (yesterday < ma))
  	{
  		/* alert was triggered, update the scan alert window with details */
  		tlong = g_array_index (date_array, glong, tuint);
  		g_string_sprintf (tstring, "%ld", tlong);
  		import_check_date(tstring->str);
  		line[0] = gstring->str;
  		line[1] = current_symbol->str;
  		switch (array)
  		{
  			case 1: g_string_assign (tstring, label_ma_gt_close);
  					break;
  			case 2: g_string_assign (tstring, label_ma2_gt_close);
  					break;
  			case 3: g_string_assign (tstring, label_ma3_gt_close);
  					break;
  			default: break;
  		}
  		line[2] = tstring->str;
  		gtk_clist_append (GTK_CLIST (alert_window_clist), line);
  	}
  	
  	/* check if close plot has crossed the MA plot and was above it previously */
  	if ((today <= ma) && (yesterday > ma))
  	{
  		/* alert was triggered, update the scan alert window with details */
  		tlong = g_array_index (date_array, glong, tuint);
  		g_string_sprintf (tstring, "%ld", tlong);
  		import_check_date(tstring->str);
  		line[0] = gstring->str;
  		line[1] = current_symbol->str;
  		switch (array)
  		{
  			case 1: g_string_assign (tstring, label_ma_lt_close);
  					break;
  			case 2: g_string_assign (tstring, label_ma2_lt_close);
  					break;
  			case 3: g_string_assign (tstring, label_ma3_lt_close);
  					break;
  			default: break;
  		}
  		line[2] = tstring->str;
  		gtk_clist_append (GTK_CLIST (alert_window_clist), line);
  	}
  }
  
  g_string_free(tstring, TRUE);
}
/*****************************************************************************
Find any threshold alerts.
******************************************************************************/
void 
scan_threshold ()
{
  glong tlong;
  GString *tstring;
  gchar *line[3];
  gfloat today, yesterday;
  guint tuint, records;
  extern GtkWidget *alert_window_clist;
  extern GArray *close_array, *date_array;
  extern GString *gstring, *current_symbol;
  extern gfloat current_threshold;
  extern gint current_records;
  extern struct record config;
  extern gchar *label_close_lt_threshold, *label_close_gt_threshold;
  
  
  
  tstring = g_string_new(NULL);
  
  if (config.bars)
  {
  	/* get the smallest number of data to load, either total records or maximum */
      	if (current_records < config.bars)
		records = current_records;
      	else
		records = config.bars;
  }
  else
	records = current_records;
  
  /* loop through all the close values */
  for (tuint = 1; tuint < records; tuint++)
  {
  	today = g_array_index(close_array, gfloat, tuint);
  	yesterday = g_array_index(close_array, gfloat, tuint -1 );
  	
  	/* compare the close with previous close */
  	if ((today >= current_threshold) && (yesterday < current_threshold))
  	{
  		/* alert triggered, close => previous close, update alert report */
  		tlong = g_array_index (date_array, glong, tuint);
  		g_string_sprintf (tstring, "%ld", tlong);
  		import_check_date(tstring->str);
  		line[0] = gstring->str;
  		line[1] = current_symbol->str;
  		g_string_assign (tstring, label_close_gt_threshold);
  		line[2] = tstring->str;
  		gtk_clist_append (GTK_CLIST (alert_window_clist), line);
  	}
  	
  	/* compare the close with previous close */
  	if ((today <= current_threshold) && (yesterday > current_threshold))
  	{
  		/* alert triggered, close <= previous close, update alert report */
  		tlong = g_array_index (date_array, glong, tuint);
  		g_string_sprintf (tstring, "%ld", tlong);
  		import_check_date(tstring->str);
  		line[0] = gstring->str;
  		line[1] = current_symbol->str;
  		g_string_assign (tstring, label_close_lt_threshold);
  		line[2] = tstring->str;
  		gtk_clist_append (GTK_CLIST (alert_window_clist), line);
  	}
  }
  
  g_string_free(tstring, TRUE);
}
/******************************************************************************
Find any closeroc alerts.
*****************************************************************************/
void 
scan_closeroc ()
{
  glong tlong;
  gchar *line[3];
  GString *tstring;
  gfloat today, yesterday, percent;
  guint tuint, records;
  extern GArray *close_array, *date_array;
  extern struct record config;
  extern GString *gstring, *current_symbol;
  extern GtkWidget *alert_window, *alert_window_clist;
  extern gint current_records;
  extern gchar *label_close_gt_closeroc, *label_close_lt_closeroc;
  
  
  
  tstring = g_string_new(NULL);
  
  if (config.bars)
  {
  	/* get the smallest number of data to load, either total records or maximum */
      	if (current_records < config.bars)
		records = current_records;
      	else
		records = config.bars;
  }
  else
	records = current_records;
  
  for (tuint = 1; tuint < records; tuint++)
  {
  	today  = g_array_index(close_array, gfloat, tuint);
  	yesterday = g_array_index(close_array, gfloat, tuint - 1);
  	percent = (today - yesterday) / (yesterday / 100);
  	
  	if (config.closeroc < 0)
  	{
  		if (percent < config.closeroc)
  		{
  			if (alert_window)
  			{
  				tlong = g_array_index (date_array, glong, tuint);
  				g_string_sprintf (tstring, "%ld", tlong);
  				import_check_date(tstring->str);
  				line[0] = gstring->str;
  				line[1] = current_symbol->str;
  				g_string_assign (tstring, label_close_gt_closeroc);
  				line[2] = tstring->str;
  				gtk_clist_append (GTK_CLIST (alert_window_clist), line);
  			}
  		}
  	}
  	
  	if (config.closeroc > 0)
  	{
  		if (percent > config.closeroc)
  		{
  			if (alert_window)
  			{
  				tlong = g_array_index (date_array, glong, tuint);
  				g_string_sprintf (tstring, "%ld", tlong);
  				import_check_date(tstring->str);
  				line[0] = gstring->str;
  				line[1] = current_symbol->str;
  				g_string_assign (tstring, label_close_lt_closeroc);
  				line[2] = tstring->str;
  				gtk_clist_append (GTK_CLIST (alert_window_clist), line);
  			}
  		}
  	}
  }
  
  g_string_free(tstring, TRUE);
}
/*********************************************************************************
Destroys the alert window.
*********************************************************************************/
void 
exit_alerts_window ()
{
  extern GtkWidget *edit_alerts_window;
  extern struct record config;


  gtk_widget_destroy (edit_alerts_window);
  
  /* update the window flag to closed */
  config.edit_alerts_window_open = 0;
}
/************************************************************************************
Displays the chart selected from the goto chart window.
*************************************************************************************/
void 
alert_goto_chart ()
{
  GList *list;
  gint tint;
  gchar *text;
  GtkWidget *dialog;
  extern GtkWidget *alert_window_clist, *main_window;
  extern gint open_file_flag;
  extern gchar *mess_nothing_selected;


  /* get the list selection */
  list = GTK_CLIST (alert_window_clist)->selection;
  if (!list)
  {
  	/* nothing was selected */
  	dialog = gnome_app_error (GNOME_APP(main_window), mess_nothing_selected);
	gtk_widget_show(dialog);
	return;
  }

  tint = (int) list->data;
  list = GTK_CLIST (alert_window_clist)->selection;
  
  /* get the symbol */
  gtk_clist_get_text (GTK_CLIST (alert_window_clist), tint, 1, &text);
  open_file_flag = 1;
  
  /* display the chart */
  load_file (text);
}
/*****************************************************************************
Create the alert report window.
*******************************************************************************/
void 
create_alert_report ()
{
  gchar *header[] = {"Date", "Symbol", "Alert"};
  gint tint;
  GtkWidget *vbox, *hbox, *hsep, *button, *button2, *swindow;
  extern GtkWidget *alert_window, *alert_window_clist;
  extern gchar *label_alert_view, *label_goto_chart, *label_close;


  if (!alert_window)
    {
      /* create the alert report window */
      alert_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
      gtk_signal_connect (GTK_OBJECT (alert_window), "destroy",
		     GTK_SIGNAL_FUNC (gtk_widget_destroyed), &alert_window);
      gtk_window_set_title (GTK_WINDOW (alert_window), label_alert_view);
      gtk_container_border_width (GTK_CONTAINER (alert_window), 0);
      gtk_window_position (GTK_WINDOW (alert_window), GTK_WIN_POS_CENTER);
      gtk_widget_set_usize (GTK_WIDGET (alert_window), 560, 200);

      vbox = gtk_vbox_new (FALSE, 0);
      gtk_container_add (GTK_CONTAINER (alert_window), vbox);
      gtk_widget_show (vbox);
      
      swindow = gtk_scrolled_window_new(NULL, NULL);
      gtk_box_pack_start (GTK_BOX (vbox), swindow, TRUE, TRUE, 0);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swindow),
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
      gtk_widget_show(swindow);
      
      alert_window_clist = gtk_clist_new_with_titles (3, header);
      gtk_clist_set_column_width (GTK_CLIST (alert_window_clist), 0, 70);
      gtk_clist_set_column_width (GTK_CLIST (alert_window_clist), 1, 70);
      gtk_clist_set_column_width (GTK_CLIST (alert_window_clist), 2, 200);
      for (tint = 0; tint < 3; tint++)
	gtk_clist_set_column_justification (GTK_CLIST (alert_window_clist), tint, GTK_JUSTIFY_LEFT);
      gtk_clist_set_selection_mode (GTK_CLIST (alert_window_clist), GTK_SELECTION_SINGLE);
      gtk_container_add (GTK_CONTAINER (swindow), alert_window_clist);
      gtk_container_border_width (GTK_CONTAINER (alert_window_clist), 10);
      gtk_widget_show (alert_window_clist);

      hsep = gtk_hseparator_new ();
      gtk_box_pack_start (GTK_BOX (vbox), hsep, FALSE, TRUE, 0);
      gtk_widget_show (hsep);

      hbox = gtk_hbutton_box_new ();
      gtk_container_border_width (GTK_CONTAINER (hbox), 10);
      gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);
      gtk_hbutton_box_set_spacing_default (5);
      gtk_hbutton_box_set_layout_default (GTK_BUTTONBOX_END);
      gtk_widget_show (hbox);

      button = gtk_button_new_with_label (label_goto_chart);
      gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (alert_goto_chart), NULL);
      gtk_widget_show (button);

      button2 = gtk_button_new_with_label (label_close);
      gtk_box_pack_start (GTK_BOX (hbox), button2, FALSE, TRUE, 0);
      gtk_signal_connect_object (GTK_OBJECT (button2), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (alert_window));
      gtk_widget_show (button2);

      gtk_widget_show (alert_window);
    }
  else
    {
      /* the window is already visible, so just clear the contents */
      gtk_clist_freeze (GTK_CLIST (alert_window_clist));
      gtk_clist_clear (GTK_CLIST (alert_window_clist));
      gtk_clist_thaw (GTK_CLIST (alert_window_clist));
    }
}
/****************************************************************************
Perform the alert scan to trigger alerts.
*****************************************************************************/
void 
ok_alert_scan ()
{
  GList *list;
  gchar *text;
  GString *tstring;
  gint tint, importline=0, selcount;
  gfloat tfloat = 0.0;
  GtkWidget *dialog;
  extern GtkWidget *alert_scan_clist, *main_window;
  extern GtkWidget *data_bar;
  extern gchar *mess_nothing_selected, *mess_scan_alert, *mess_alert_scan_complete;
  extern struct record config;


  /* check if anything was selected */
  list = GTK_CLIST (alert_scan_clist)->selection;
  if (!list)
  {
	dialog = gnome_app_error (GNOME_APP(main_window), mess_nothing_selected);
	gtk_widget_show(dialog);
	return;
  }

  /* create the report window */
  create_alert_report ();

  /* update the status bar */
  gnome_appbar_set_status (GNOME_APPBAR(data_bar), mess_scan_alert);
  gnome_appbar_set_progress (GNOME_APPBAR(data_bar), tfloat);
  
  tstring = g_string_new(NULL);
  selcount = g_list_length (list);
  
  /* loop through all the selections in the list */
  while (list)
  {
  	/* get the selected symbol and build the path to the chart data */
  	tint = (int) list->data;
  	gtk_clist_get_text (GTK_CLIST (alert_scan_clist), tint, 0, &text);
  	g_string_sprintf (tstring, "%s%s", config.datapath, text);
  
	/* check for an alert trigger from the active alerts */
	if (config.closeroc_status)
		scan_closeroc ();
	if (config.threshold_status)
	    	scan_threshold ();
	if (config.moving_status)
	    	scan_ma (1);
	if (config.moving2_status)
	    	scan_ma (2);
	if (config.moving3_status)
	    	scan_ma (3);
	
	/* update the number of symbols we have checked so far */    	
        importline++;
        
        /* update the progress bar */
	tfloat = ((float) importline / selcount);
	if (tfloat > 1.0)
		tfloat = 1.0;
	if (tfloat < 0.0)
		tfloat = 0.0;
	gnome_appbar_set_progress (GNOME_APPBAR(data_bar), tfloat);
	while (gtk_events_pending ())
		gtk_main_iteration ();
		
	/* move to the next selected list item */
	list = list->next;
  }
  
  g_string_free(tstring, TRUE);
  
  /* update the status bar that we are done */
  gnome_appbar_set_status (GNOME_APPBAR(data_bar), mess_alert_scan_complete);
  gnome_appbar_set_progress (GNOME_APPBAR(data_bar), 0.0);
}
/****************************************************************************
Create the alert scan window.
****************************************************************************/
void 
alert_scan ()
{
  gchar *text[] = {"Symbol", "Name", "Start", "End"};
  GtkWidget *vbox, *vbox2, *hbox, *scan_button, *cancel_button, *swindow;
  GtkWidget *select_button, *unselect_button;
  extern GtkWidget *alert_scan_clist, *alert_scan_window;
  extern gchar *label_group_scan, *label_scan, *label_select, *label_unselect, *label_cancel;


  /* if window is already displayed then forget it */
  if (alert_scan_window)
    return;

  alert_scan_window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_signal_connect (GTK_OBJECT (alert_scan_window), "destroy",
		GTK_SIGNAL_FUNC (gtk_widget_destroyed), &alert_scan_window);
  gtk_window_position (GTK_WINDOW (alert_scan_window), GTK_WIN_POS_CENTER);
  gtk_window_set_title (GTK_WINDOW (alert_scan_window), label_group_scan);
  gtk_widget_set_usize (GTK_WIDGET (alert_scan_window), 0, 0);
  gtk_container_border_width (GTK_CONTAINER (alert_scan_window), 0);
  
  hbox = gtk_hbox_new (FALSE, 10);
  gtk_container_add (GTK_CONTAINER (alert_scan_window), hbox);
  gtk_container_border_width (GTK_CONTAINER (hbox), 10);
  gtk_widget_show (hbox);

  vbox = gtk_vbox_new (FALSE, 10);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
  gtk_widget_show (vbox);

  vbox2 = gtk_vbutton_box_new ();
  gtk_vbutton_box_set_spacing_default (5);
  gtk_vbutton_box_set_layout_default (GTK_BUTTONBOX_START);
  gtk_box_pack_end (GTK_BOX (hbox), vbox2, FALSE, TRUE, 0);
  gtk_widget_show (vbox2);
  
  swindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start (GTK_BOX (vbox), swindow, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swindow),
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show(swindow);

  alert_scan_clist = gtk_clist_new_with_titles (4, text);
  gtk_clist_set_selection_mode (GTK_CLIST (alert_scan_clist), GTK_SELECTION_MULTIPLE);
  gtk_clist_set_column_width (GTK_CLIST (alert_scan_clist), 0, 70);
  gtk_clist_set_column_width (GTK_CLIST (alert_scan_clist), 1, 200);
  gtk_clist_set_column_width (GTK_CLIST (alert_scan_clist), 2, 70);
  gtk_clist_set_column_width (GTK_CLIST (alert_scan_clist), 3, 70);
  gtk_container_add (GTK_CONTAINER (swindow), alert_scan_clist);
  gtk_container_border_width (GTK_CONTAINER (alert_scan_clist), 0);
  gtk_widget_set_usize (GTK_WIDGET (alert_scan_clist), 350, 200);
  gtk_widget_show (alert_scan_clist);

  scan_button = gtk_button_new_with_label (label_scan);
  gtk_box_pack_start (GTK_BOX (vbox2), scan_button, FALSE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (scan_button), "clicked",
		      GTK_SIGNAL_FUNC (ok_alert_scan), NULL);
  gtk_widget_show (scan_button);
  
  select_button = gtk_button_new_with_label (label_select);
  gtk_box_pack_start (GTK_BOX (vbox2), select_button, FALSE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (select_button), "clicked",
		      GTK_SIGNAL_FUNC (alert_select_all), NULL);
  gtk_widget_show (select_button);
  
  unselect_button = gtk_button_new_with_label (label_unselect);
  gtk_box_pack_start (GTK_BOX (vbox2), unselect_button, FALSE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (unselect_button), "clicked",
		      GTK_SIGNAL_FUNC (alert_unselect_all), NULL);
  gtk_widget_show (unselect_button);

  cancel_button = gtk_button_new_with_label (label_cancel);
  gtk_box_pack_start (GTK_BOX (vbox2), cancel_button, FALSE, TRUE, 0);
  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (alert_scan_window));
  gtk_widget_show (cancel_button);

  show_chart_list (alert_scan_clist);
  gtk_widget_show (alert_scan_window);
}
/****************************************************************************************
Create the edit alerts window.
*****************************************************************************************/
void 
edit_alerts ()
{
  GtkAdjustment *adj, *adj2;
  GString *tstring;
  GtkWidget *close_button, *vbox, *vbox2, *frame, *hbox, *hbox5, *hbox6;
  extern gint chartflag;
  extern GtkWidget *edit_alerts_window, *ea_ma_check, *ea_ma2_check, *ea_ma3_check;
  extern GtkWidget *ea_ma_label, *ea_ma2_label, *ea_ma3_label, *ea_closeroc_spinner;
  extern GtkWidget *ea_threshold_spinner, *ea_closeroc_check;
  extern GtkWidget *ea_threshold_check, *ea_hbox2, *ea_hbox3, *ea_hbox4;
  extern struct record config;
  extern GString *current_symbol, *current_name;
  extern gchar *label_edit_alerts, *label_alerts, *label_close, *label_ma, *label_ma2;
  extern gchar *label_ma3, *label_closeroc, *label_threshold;


  /* dont bother if no chart is displayed */
  if (! chartflag)
      return;
      
  /* check if the window is already open */
  if (config.edit_alerts_window_open)
	{
	  /* its already displayed, so just update the status of the alerts instead */
	  update_alerts_window ();
	  return;
	}
      else
	{
	  /* create the window */
	  edit_alerts_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	  gtk_signal_connect (GTK_OBJECT (edit_alerts_window), "destroy",
			      GTK_SIGNAL_FUNC (exit_alerts_window), NULL);
	  gtk_container_border_width (GTK_CONTAINER (edit_alerts_window), 0);
	  gtk_window_position (GTK_WINDOW (edit_alerts_window), GTK_WIN_POS_CENTER);
	  gtk_widget_set_usize (GTK_WIDGET (edit_alerts_window), 0, 0);

	  tstring = g_string_new(NULL);

	  g_string_sprintf (tstring, "%s - %s (%s)",  label_edit_alerts, current_name->str,
	  				current_symbol->str);
	  gtk_window_set_title (GTK_WINDOW (edit_alerts_window), tstring->str);
	  gtk_widget_realize (edit_alerts_window);

	  hbox = gtk_hbox_new (FALSE, 10);
	  gtk_container_add (GTK_CONTAINER (edit_alerts_window), hbox);
	  gtk_container_border_width (GTK_CONTAINER (hbox), 10);
	  gtk_widget_show (hbox);
	  
	  frame = gtk_frame_new (label_alerts);
  	  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, TRUE, 0);
  	  gtk_container_border_width (GTK_CONTAINER (frame), 0);
  	  gtk_widget_show (frame);
  	  
  	  vbox2 = gtk_vbox_new (FALSE, 5);
  	  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  	  gtk_container_border_width (GTK_CONTAINER (vbox2), 5);
	  gtk_widget_show (vbox2);

	  vbox = gtk_vbutton_box_new ();
	  gtk_vbutton_box_set_spacing_default (5);
	  gtk_vbutton_box_set_layout_default (GTK_BUTTONBOX_START);
	  gtk_box_pack_end (GTK_BOX (hbox), vbox, FALSE, TRUE, 0);
	  gtk_widget_show (vbox);

	  close_button = gtk_button_new_with_label (label_close);
	  gtk_box_pack_start (GTK_BOX (vbox), close_button, FALSE, TRUE, 0);
	  gtk_signal_connect (GTK_OBJECT (close_button), "clicked",
			      GTK_SIGNAL_FUNC (exit_alerts_window), NULL);
	  gtk_widget_show (close_button);
	  
	  adj = (GtkAdjustment *) gtk_adjustment_new ((gfloat) 1, 1.0, (gfloat) G_MAXINT,
	  										  1.0, 5.0, 0.0);
	  adj2 = (GtkAdjustment *) gtk_adjustment_new ((gfloat) 1, 1.0, (gfloat) G_MAXINT,
	  										  1.0, 5.0, 0.0);

  	  ea_hbox2 = gtk_hbox_new (FALSE, 10);
  	  gtk_box_pack_start (GTK_BOX (vbox2), ea_hbox2, FALSE, TRUE, 0);
	  gtk_widget_show (ea_hbox2);
	  
	  ea_ma_check = gtk_check_button_new_with_label (label_ma);
	  gtk_box_pack_start (GTK_BOX (ea_hbox2), ea_ma_check, FALSE, TRUE, 0);
	  gtk_widget_set_usize (GTK_WIDGET (ea_ma_check), 65, 0);
	  gtk_widget_show(ea_ma_check);
	  
	  ea_ma_label = gtk_label_new ("");
      	  gtk_box_pack_start (GTK_BOX (ea_hbox2), ea_ma_label, FALSE, TRUE, 0);
      	  gtk_widget_show (ea_ma_label);
      	  
      	  ea_hbox3 = gtk_hbox_new (FALSE, 10);
  	  gtk_box_pack_start (GTK_BOX (vbox2), ea_hbox3, FALSE, TRUE, 0);
	  gtk_widget_show (ea_hbox3);
	  
	  ea_ma2_check = gtk_check_button_new_with_label (label_ma2);
	  gtk_box_pack_start (GTK_BOX (ea_hbox3), ea_ma2_check, FALSE, TRUE, 0);
	  gtk_widget_set_usize (GTK_WIDGET (ea_ma2_check), 65, 0);
	  gtk_widget_show(ea_ma2_check);
	  
	  ea_ma2_label = gtk_label_new ("");
     	  gtk_box_pack_start (GTK_BOX (ea_hbox3), ea_ma2_label, FALSE, TRUE, 0);
      	  gtk_widget_show (ea_ma2_label);
      	  
      	  ea_hbox4 = gtk_hbox_new (FALSE, 10);
  	  gtk_box_pack_start (GTK_BOX (vbox2), ea_hbox4, FALSE, TRUE, 0);
	  gtk_widget_show (ea_hbox4);
	  
	  ea_ma3_check = gtk_check_button_new_with_label (label_ma3);
	  gtk_box_pack_start (GTK_BOX (ea_hbox4), ea_ma3_check, FALSE, TRUE, 0);
	  gtk_widget_set_usize (GTK_WIDGET (ea_ma3_check), 65, 0);
	  gtk_widget_show(ea_ma3_check);
	  
	  ea_ma3_label = gtk_label_new ("");
      	  gtk_box_pack_start (GTK_BOX (ea_hbox4), ea_ma3_label, FALSE, TRUE, 0);
      	  gtk_widget_show (ea_ma3_label);
      	  
      	  hbox5 = gtk_hbox_new (FALSE, 10);
  	  gtk_box_pack_start (GTK_BOX (vbox2), hbox5, FALSE, TRUE, 0);
	  gtk_widget_show (hbox5);
	  
	  ea_closeroc_check = gtk_check_button_new_with_label (label_closeroc);
	  gtk_box_pack_start (GTK_BOX (hbox5), ea_closeroc_check, FALSE, TRUE, 0);
	  gtk_widget_set_usize (GTK_WIDGET (ea_closeroc_check), 100, 0);
	  gtk_widget_show(ea_closeroc_check);
	  
	  ea_closeroc_spinner = gtk_spin_button_new(adj ,0, 0);
	  gtk_box_pack_start (GTK_BOX (hbox5), ea_closeroc_spinner, FALSE, TRUE, 0);
	  gtk_widget_set_usize (GTK_WIDGET (ea_closeroc_spinner), 50, 0);
	  gtk_widget_show(ea_closeroc_spinner);
	  
	  hbox6 = gtk_hbox_new (FALSE, 10);
  	  gtk_box_pack_start (GTK_BOX (vbox2), hbox6, FALSE, TRUE, 0);
	  gtk_widget_show (hbox6);
	  
	  ea_threshold_check = gtk_check_button_new_with_label (label_threshold);
	  gtk_box_pack_start (GTK_BOX (hbox6), ea_threshold_check, FALSE, TRUE, 0);
	  gtk_widget_set_usize (GTK_WIDGET (ea_threshold_check), 100, 0);
	  gtk_widget_show(ea_threshold_check);
	  
	  ea_threshold_spinner = gtk_spin_button_new(adj2 ,0, 0);
	  gtk_box_pack_start (GTK_BOX (hbox6), ea_threshold_spinner, FALSE, TRUE, 0);
	  gtk_widget_set_usize (GTK_WIDGET (ea_threshold_spinner), 50, 0);
	  gtk_widget_show(ea_threshold_spinner);
	  

	  /* update the alert status */
	  update_alerts_window ();
	  
	  gtk_signal_connect (GTK_OBJECT (ea_ma_check), "toggled",
			   		  GTK_SIGNAL_FUNC (ea_ma_checked), NULL);
	  gtk_signal_connect (GTK_OBJECT (ea_ma2_check), "toggled",
			   		  GTK_SIGNAL_FUNC (ea_ma2_checked), NULL);
	  gtk_signal_connect (GTK_OBJECT (ea_ma3_check), "toggled",
			   		  GTK_SIGNAL_FUNC (ea_ma3_checked), NULL);
	  gtk_signal_connect (GTK_OBJECT (ea_closeroc_check), "toggled",
			   		  GTK_SIGNAL_FUNC (ea_closeroc_checked), NULL);
	  gtk_signal_connect (GTK_OBJECT (ea_threshold_check), "toggled",
			   		  GTK_SIGNAL_FUNC (ea_threshold_checked), NULL);
			   		  
	  gtk_signal_connect (GTK_OBJECT (ea_closeroc_spinner), "changed",
			   		  GTK_SIGNAL_FUNC (ea_closeroc_spinner_changed), NULL);
	  gtk_signal_connect (GTK_OBJECT (ea_threshold_spinner), "changed",
			   		  GTK_SIGNAL_FUNC (ea_threshold_spinner_changed), NULL);

	  /* set the window open flag to true */
	  config.edit_alerts_window_open = 1;
	  g_string_free(tstring, TRUE);
	  gtk_widget_show (edit_alerts_window);
    }
}
