/*
==============================================================================
	iotool.c
		1990/12/14/Fri	Yutaka MYOKI(Nagao Lab., KUEE)
		1990/12/25/Tue	Last Modified
		Oct. 1996       A.Kitauchi <akira-k@is.aist-nara.ac.jp>
==============================================================================
*/

#include "chadic.h"

#ifdef KOCHA
#define RCFILE "/.kocharc"
#else
#ifdef _WIN32
#define RCFILE "\\chasenrc"
#define RC2FILE "\\chasen2rc"
#else
#define RCFILE "/.chasenrc"
#define RC2FILE "/.chasen2rc"
#endif
#endif
#define OLDRCFILE "/.jumanrc"

int LineNo, LineNoForError;
int Cha_errno = 0;

static FILE *cha_stderr = NULL;
static char progpath[CHA_FILENAME_MAX] = "chasen";
static char filepath[CHA_FILENAME_MAX];
static char grammar_dir[CHA_FILENAME_MAX];
static char chasenrc_path[CHA_FILENAME_MAX];

/***********************************************************************
 * convert_escape - convert escape characters
 ***********************************************************************/
char *convert_escape(str, ctrl_only)
    char *str;
    int ctrl_only;
{
    char *s1, *s2;

    for (s1 = s2 = str; *s1; s1++, s2++) {
	if (*s1 != '\\')
	  *s2 = *s1;
	else {
	    switch (*++s1) {
	      case 't':
		*s2 = '\t'; break;
	      case 'n':
		*s2 = '\n'; break;
	      default:
		if (ctrl_only)
		  *s2++ = '\\';
		*s2 = *s1;
		break;
	    }
	}
    }
    *s2 = '\0';

    return str;
}

/***********************************************************************
 * set_progpath - set program pathname
 *
 *	progpath is used in cha_exit() and cha_exit_file()
 ***********************************************************************/
#ifdef _WIN32
static void which(filename, path)
    char *filename;
    char *path;
{
    char *ps, *pe;
    int fl;
    
    strcpy(path, ".\\");
    strcat(path, filename);
    if (fopen(path, "r") != NULL) return;
    
    ps = getenv("PATH");

    for (pe = ps, fl = 0;!fl ; pe++) {
	if (*pe == '\0') {
	    *pe = ';';
	    fl = 1;
	}
	if (*pe == ';') {
	    *pe = '\0';
	    strcpy(path, ps);
	    if (pe[-1] != '\\')
	      strcat(path, "\\");
	    strcat(path, filename);
	    if (fopen(path, "r") != NULL) return;
	    ps = pe + 1;
	}
    }
}
#endif /* _WIN32 */

void set_progpath(path)
    char *path;
{
#ifdef _WIN32
    if (strchr(path, '\\') != NULL)
      strcpy(progpath, path);
    else
      which("chasen.exe", progpath);
#else
    strcpy(progpath, path);
#endif
}

/***********************************************************************
 * set_chasenrc_path - set chasenrc file path
 *
 *	this function is called when -r option is used.
 ***********************************************************************/
void set_chasenrc_path(filename)
    char *filename;
{
    strcpy(chasenrc_path, filename);
}    

/***********************************************************************
 * get_chasenrc_path
 *
 *	called only from chasen.c
 ***********************************************************************/
char *get_chasenrc_path()
{
    return chasenrc_path;
}

/***********************************************************************
 * get_grammar_dir
 *
 *	called only from chasen.c
 ***********************************************************************/
char *get_grammar_dir()
{
    return grammar_dir;
}

void set_filepath(filename)
    char *filename;
{
    strcpy(filepath, filename);
    LineNo = LineNoForError = 0;
}

/***********************************************************************
 * cha_fopen - open file, or error end
 *
 * inputs:
 *	ret - exit code (don't exit if ret < 0)
 ***********************************************************************/
FILE *cha_fopen(filename, mode, ret)
    char *filename, *mode;
    int ret;
{
    FILE *fp;

    if (filename[0] == '-' && filename[1] == '\0')
      return stdin;

    if ((fp = fopen(filename, mode)) != NULL) {
	/* filepath is used in cha_exit_file() */
	if (*mode == 'r') {
	    if (filename != filepath)
	      strcpy(filepath, filename);
	    LineNo = LineNoForError = 0;
	}
    } else if (ret >= 0)
      cha_exit_perror(filename);

    return fp;
}

FILE *cha_fopen2(filename1, filename2, mode, ret)
    char *filename1, *filename2, *mode;
    int ret;
{
    FILE *fp;

    if ((fp = cha_fopen(filename1, mode, -1)) != NULL)
      return fp;

    if ((fp = cha_fopen(filename2, mode, -1)) != NULL)
      return fp;

    cha_exit(ret, "can't open %s or %s", filename1, filename2);

    /* to avoid warning */
    return NULL;
}

/***********************************************************************
 * cha_fopen_grammar - open file from current or grammar directory
 *
 * inputs:
 *	dir - 0: read from current directory
 *	      1: read from grammar directory
 *	      2: read from current directory or grammar directory
 *
 *	ret - return the code when fopen() fails
 *
 * outputs:
 *	filepathp - file path string
 ***********************************************************************/
FILE *cha_fopen_grammar(filename, mode, ret, dir, filepathp)
    char *filename, *mode, **filepathp;
    int ret, dir;
{
    FILE *fp;
    char *s;

    *filepathp = filename;
    switch (dir) {
      case 0:
	/* ȥǥ쥯ȥ꤫ɤ߹ */
	return cha_fopen(filename, mode, ret);
      case 2:
	/* ȥǥ쥯ȥ꤫ɤ߹ */
	if ((fp = cha_fopen(filename, mode, -1)) != NULL)
	  return fp;
	/* FALLTHRU */
      default: /* should be 1 */
	/* ʸˡǥ쥯ȥ꤫ɤ߹ */
	/* ʸˡǥ쥯ȥ꤬ꤵƤʤ .chasenrc ɤ߹ */
	if (grammar_dir[0] == '\0')
	  read_grammar_dir();
	sprintf(filepath, "%s%s", grammar_dir, filename);
	*filepathp = filepath;
	return cha_fopen(filepath, mode, ret);
    }
}

FILE *cha_fopen_grammar2(filename1, filename2, mode, ret, dir, filepathp)
    char *filename1, *filename2, *mode, **filepathp;
    int ret, dir;
{
    FILE *fp;

    if (dir == 2) {
	if ((fp = cha_fopen_grammar(filename1, mode, -1, 0, filepathp)) != NULL)
	  return fp;
	if ((fp = cha_fopen_grammar(filename2, mode, -1, 0, filepathp)) != NULL)
	  return fp;
	if ((fp = cha_fopen_grammar(filename1, mode, -1, 1, filepathp)) != NULL)
	  return fp;
	if ((fp = cha_fopen_grammar(filename2, mode, -1, 1, filepathp)) != NULL)
	  return fp;
    } else {
	if ((fp = cha_fopen_grammar(filename1, mode, -1, dir, filepathp)) != NULL)
	  return fp;
	if ((fp = cha_fopen_grammar(filename2, mode, -1, dir, filepathp)) != NULL)
	  return fp;
    }

    cha_exit(ret, "can't open %s or %s", filename1, filename2);

    /* to avoid warning */
    return NULL;
}

/*
------------------------------------------------------------------------------
        PROCEDURE:
        <cha_malloc>: do "malloc" (library function) and cha_exit processing
------------------------------------------------------------------------------
*/

void *cha_malloc(n)
    int n;
{
    void *p;

    if ((p = (void *)malloc(n)) == NULL)
      cha_exit_perror("malloc");

    return p;
}

#define CHA_MALLOC_SIZE (1024 * 64)
static char *cha_malloc_char(size)
    int size;
{
    static int idx = CHA_MALLOC_SIZE;
    static char *ptr;

    if (idx + size >= CHA_MALLOC_SIZE) {
	ptr = (char *)cha_malloc(CHA_MALLOC_SIZE);
	idx = 0;
    }

    idx += size;
    return ptr + idx - size;
}

char *cha_strdup(str)
    char *str;
{
    char *newstr;

    newstr = cha_malloc_char(strlen(str) + 1);
    strcpy(newstr, str);

    return newstr;
}

/*
 * cha_exit() - print error messages on stderr and exit
 */
void set_cha_stderr(fp)
    FILE *fp;
{
    cha_stderr = fp;
}

void cha_exit(status, format, a, b, c, d, e, f, g, h)
    int status;
    char *format, *a, *b, *c, *d, *e, *f, *g, *h;
{
    if (Cha_errno)
      return;

    if (!cha_stderr)
      cha_stderr = stderr;
    else if (cha_stderr != stderr)
      fputs("500 ", cha_stderr);

    if (progpath)
      fprintf(cha_stderr, "%s: ", progpath);
    fprintf(cha_stderr, format, a, b, c, d, e, f, g, h);

    if (status >= 0) {
	fputc('\n', cha_stderr);
	if (cha_stderr == stderr)
	  exit(status);
	Cha_errno = 1;
    }
}

void cha_exit_file(status, format, a, b, c, d, e, f, g, h)
    int status;
    char *format, *a, *b, *c, *d, *e, *f, *g, *h;
{
    if (Cha_errno)
      return;

    if (!cha_stderr)
      cha_stderr = stderr;
    else if (cha_stderr != stderr)
      fputs("500 ", cha_stderr);

    if (progpath)
      fprintf(cha_stderr, "%s: ", progpath);

    if (LineNo == 0)
      ; /* do nothing */
    else if (LineNo == LineNoForError)
      fprintf(cha_stderr, "%s:%d: ", filepath, LineNo);
    else
      fprintf(cha_stderr, "%s:%d-%d: ", filepath, LineNoForError, LineNo);

    fprintf(cha_stderr, format, a, b, c, d, e, f, g, h);

    if (status >= 0) {
	fputc('\n', cha_stderr); 
	if (cha_stderr == stderr)
	  exit(status);
	Cha_errno = 1;
    }
}

void cha_perror(s)
    char *s;
{
    cha_exit(-1, "");
    perror(s);
}

void cha_exit_perror(s)
    char *s;
{
    cha_perror(s);
    exit(1);
}

FILE *fopen_chasenrc()
{
    FILE *fp;
    char *home_dir, *rc_env, *getenv();

    /* -R option (standard alone) */
    if (!strcmp(chasenrc_path, "*")) {
	/* RCPATH in rcpath.h */
	strcpy(chasenrc_path, RCPATH);
	if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
	  return fp;
	cha_exit(1, "can't open %s", chasenrc_path);
    }

    /* -r option */
    if (chasenrc_path[0])
      return cha_fopen(chasenrc_path, "r", 1);

    /* environment variable CHASENRC */
    if ((rc_env = getenv("CHASENRC")) != NULL) {
	strcpy(chasenrc_path, rc_env);
	return cha_fopen(chasenrc_path, "r", 1);
    }

    /* .chasenrc in the home directory */
    if ((home_dir = getenv("HOME")) != NULL) {
	/* .chasenrc */
	sprintf(chasenrc_path, "%s%s", home_dir, RC2FILE);
	if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
	  return fp;
	sprintf(chasenrc_path, "%s%s", home_dir, RCFILE);
	if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
	  return fp;

#ifndef VGRAM
	/* .jumanrc */
	sprintf(chasenrc_path, "%s%s", home_dir, OLDRCFILE);
	if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
	  return fp;
#endif
    }
#ifdef _WIN32
    else if ((home_dir = getenv("HOMEDRIVE")) != NULL) {
	sprintf(chasenrc_path, "%s%s%s", home_dir, getenv("HOMEPATH"), RC2FILE);
	if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
	  return fp;
	sprintf(chasenrc_path, "%s%s%s", home_dir, getenv("HOMEPATH"), RCFILE);
	if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
	  return fp;
    }
#endif


#ifdef _WIN32
    strcpy(chasenrc_path, progpath);
    sprintf(strrchr(chasenrc_path, '\\'), "\\dic%s", RC2FILE);
    if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
      return fp;
    strcpy(chasenrc_path, progpath);
    sprintf(strrchr(chasenrc_path, '\\'), "\\dic%s", RCFILE);
    if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
      return fp;
#endif

    /* RCPATH in rcpath.h */
    strcpy(chasenrc_path, RCPATH);
    if ((fp = cha_fopen(chasenrc_path, "r", -1)) != NULL)
      return fp;

#ifdef _WIN32
    cha_exit(1, "can't open chasenrc or %s", chasenrc_path);
#else
    cha_exit(1, "can't open .chasenrc, .jumanrc, or %s", chasenrc_path);
#endif

    /* to avoid warning */
    return NULL;
}

/*
 * read .chasenrc and set grammar directory
 */
void read_grammar_dir()
{
    FILE *fp;
    cell_t *cell;

    fp = fopen_chasenrc();

    while (!s_feof(fp)) {
	char *s;
	cell = s_read(fp);
	s = s_atom(car(cell));
	if (strmatch2(s, JSTR_GRAM_FILE, ESTR_GRAM_FILE)) {
	    strcpy(grammar_dir, s_atom(car(cdr(cell))));
	    s = grammar_dir + strlen(grammar_dir);
	    if (s[-1] != PATH_DELIMITER) {
		s[0] = PATH_DELIMITER;
		s[1] = '\0';
	    }
	    break;
	}
    }

    if (grammar_dir[0] == '\0') {
	char *s;
	strcpy(grammar_dir, chasenrc_path);
	if ((s = strrchr(grammar_dir, PATH_DELIMITER)) != NULL)
	  s[1] = '\0';
	else
	  grammar_dir[0] = '\0';
    }

    fclose(fp);
}

