/* Copyright (C) 1999 Aaron Lehmann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */


#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include "gtkplot.h"
#include "gtksheet.h"

#include "app.h"
#include "prefs.h"


extern func_thing *formn;

#include "mp_parser.c"


short initial, flagend=0;
struct mp_stack **entryidx[2][2];

char *c;

static struct mp_stack * mp_new_entry (const int stacknum);
static void mp_free_entry (const int stacknum);


int yyerror (char *s)  /* Dud */
{
    return (1);
}

int yyerror_real (char *s)  /* Called by yyparse on error */
{
    parseerr = 1;
    if (formn != 0) /* If this is a function, set the error text */
    {
	if (formn->error)
	    free (formn->error);
	formn->error = malloc ((strlen(s)+1)*sizeof(char));
	strcpy (formn->error, s);
    }
    return (1);
}

struct init
{
    char *fname;
    double (*fnct)();
};

struct initc
{
    char *cname;
    double cvalue;
};


struct init arith_fncts[]
=
{
    { "sin",    mp_sin    },
    { "asin",   mp_asin   },
    { "cos",    mp_cos    },
    { "acos",   mp_acos   },
    { "tan",    mp_tan    },
    { "atan",   mp_atan   },
    { "ln",     log       },
    { "log",    log10     },
    { "exp",    exp       },
    { "sqrt",   sqrt      },
    { "abs",    fabs      },
    { "tanh",   mp_tanh   },
    { "atanh",  mp_atanh  },
    { "sinh",   mp_sinh   },
    { "asinh",  mp_asinh  },
    { "cosh",   mp_cosh   },
    { "acosh",  mp_acosh  },
    { "cot",    mp_cot    },
    { "csc",    mp_csc    },
    { "sec",    mp_sec    },
    { "sign",   mp_sign   },
    { "lgamma", lgamma    },
    { "erf",    erf       },
    { "erfc",   erfc      },
    { 0,        0         }
};


struct initc constants[]
  = {
      { "e",  M_E   },
      { "pi", M_PI  },
      { "a",  0     },
      { 0,    0     }
  };


/* The symbol table: a chain of `struct symrec'.  */
symrec *sym_table = (symrec *)0;

void init_table ()  /* puts arithmetic functions in table. */
{
    int i;
    symrec *ptr;
    for (i = 0; arith_fncts[i].fname != 0; i++)
    {
	ptr = putsym (arith_fncts[i].fname, FNCT);
	ptr->value.fnctptr = arith_fncts[i].fnct;
    }
    
    for (i = 0; constants[i].cname != 0; i++)
    {
	ptr = putsym (constants[i].cname, CONST);
	ptr->value.var = constants[i].cvalue;
    }
}


symrec *
putsym (sym_name,sym_type)
    char *sym_name;
    int sym_type;
{
    symrec *ptr;
    ptr = (symrec *) malloc (sizeof (symrec));
    ptr->name = (char *) malloc (strlen (sym_name) + 1);
    strcpy (ptr->name,sym_name);
    ptr->type = sym_type;
    ptr->value.var = 0; /* set value to 0 even if fctn.  */
    ptr->next = (struct symrec *)sym_table;
    sym_table = ptr;
    return ptr;
}


symrec *
getsym (sym_name)
    char *sym_name;
{
    symrec *ptr;
    for (ptr = sym_table; ptr != (symrec *) 0;
	 ptr = (symrec *)ptr->next)
	if (strcmp (ptr->name,sym_name) == 0)
	    return ptr;
    return 0;
}


int yylex ()
{
    if (flagend == 1) { flagend = 0; return (0); }

    
    if (!initial){ ++c; }
    
    initial = 0;
    
    
    if (*c == ' ' || *c == '\t')
    {
	/* Ignore whitespace, get first nonwhite character.  */
	while (*++c == ' ' || *c == '\t')
	    ;
    }
    
    if (*c == '\0')
    {
	flagend = 1;
	return ('\n');
    }
    
	/* Char starts a number => parse the number.         */
    if (*c == '.' || isdigit (*c) /*|| *c == ','*/)
    {
	yylval.val = g_strtod(c, (char **)NULL);
	while (*c == '.' || isdigit (*c)) ++c;
	--c;
	return NUM;
    }

    /* Char starts an identifier => read the name.       */
    if (isalpha (*c))
    {
	symrec *s;
	static char *symbuf = 0;
	static int length = 0;
	int i;
	
	/* Initially make the buffer long enough
	   for a 40-character symbol name.  */
	if (length == 0)
	    length = 40, symbuf = (char *)malloc (length + 1);
	
	i = 0;
	do
        {
	    /* If buffer is full, make it bigger.        */
	    if (i == length)
            {
		length *= 2;
		symbuf = (char *)realloc (symbuf, length + 1);
            }
          /* Add this character to the buffer.         */
	    symbuf[i++] = tolower(*c);
	    /* Get another character.                    */
	    ++c;
        }
	while (*c != '\0' && isalnum (*c));
	
	--c;
	
	symbuf[i] = '\0';
	
	s = getsym (symbuf);
	if (s == 0)
	{
	    /* Uh oh... it isn't a function - lets hope its a variable */
	    if (strlen (symbuf) == 1)
	    { /* We're OK */
		s = putsym (symbuf, VAR);
	    }
	    else
	    { /* it's ok */
		/* The user entered a string which is longer than one char but not a function */
		char *namesym;
		namesym = (char *) malloc (strlen(symbuf) + 20);
		sprintf (namesym, "%s \"%s\"", _("Unknown token"), symbuf);
		yyerror_real (namesym);
		free (namesym);
		flagend = 1;
		return ('\n');
	    }
	}	
	yylval.tptr = s;
	return s->type;
    }
    
    
    
    /* Any other character is a token by itself.        */
    return (*c);
}

void mp_push_op (short mptoken)
{
    struct mp_stack *p = mp_new_entry(0);
    p->tokentype = mptoken;
}


void mp_push_var (char mpvar, double value)
{
    struct mp_stack *p = mp_new_entry(0);
    p->tokentype = MP_TOKEN_VARIABLE;
    p->id = mpvar;
    p->value = value;
}


void mp_push_num (double mpnum)
{
    struct mp_stack *p = mp_new_entry(0);
    p->tokentype = MP_TOKEN_CONSTANT;
    p->value = mpnum;
}


void mp_push_fnct (symrec *in)
{
    struct mp_stack *p = mp_new_entry(0);
    p->tokentype = MP_TOKEN_FNCT;
    p->symboltab = in;
}


void mp_push_const (symrec *in)
{
    struct mp_stack *p = mp_new_entry(0);
    p->tokentype = MP_TOKEN_CONST;
    p->symboltab = in;
}


/*void mp_set_eq (char mpname, double value)
  {
  symrec *ptr;
  char mpstring[2];
  mpstring[0] = mpname;
  mpstring[1] = '\0';
  ptr = putsym (mpname, CONST);
  ptr->value.var = value;
  }*/


double mp_eval_exp (double x, int *err)
{
    struct mp_stack *p, *q;
    
    for (p=*entryidx[0][0]; p != 0; p=p->next)
    {
	switch (p->tokentype)
	{
	case MP_TOKEN_VARIABLE:
	    if (p->id == 'x' && !*err)
	    {
		q = mp_new_entry(1);
		q->value = x;
	    }
	    else if (*err)
	    {
		q = mp_new_entry(1);
		q->value=p->value;
	    } 
	    else
	    {
		q = mp_new_entry(1);
		q->value=0;
	    }
	    break;
	case MP_TOKEN_CONSTANT:
	    q = mp_new_entry(1);
	    q->value = p->value;
	    break;
	case MP_TOKEN_CONST: /* Mathematical constant, not to be confused with numerical value above */
	    if (formn != 0 && strcmp(p->symboltab->name, "a") == 0)
	    {
		*err = MP_ERROR_VARIABLE;
		return (0);
	    }
	    q = mp_new_entry(1);
	    q->value = p->symboltab->value.var;
	    break;
	case MP_TOKEN_PLUS:
	    (*entryidx[1][1])->previous->value = (*entryidx[1][1])->previous->value + (*entryidx[1][1])->value;
	    mp_free_entry (1);
	    break;
	case MP_TOKEN_MINUS:
	    (*entryidx[1][1])->previous->value = (*entryidx[1][1])->previous->value - (*entryidx[1][1])->value;
	    mp_free_entry (1);
	    break;
	case MP_TOKEN_MULT:
	    (*entryidx[1][1])->previous->value = (*entryidx[1][1])->previous->value * (*entryidx[1][1])->value;
	    mp_free_entry (1);
	    break;
	case MP_TOKEN_DIV:
	    (*entryidx[1][1])->previous->value = (*entryidx[1][1])->previous->value / (*entryidx[1][1])->value;
	    mp_free_entry (1);
	    break;
	case MP_TOKEN_FACT:
	    (*entryidx[1][1])->value = mp_fact ((*entryidx[1][1])->value);
	    break;
	case MP_TOKEN_POW:
	    (*entryidx[1][1])->previous->value = pow ((*entryidx[1][1])->previous->value, (*entryidx[1][1])->value);
	    mp_free_entry (1);
	    break;
	case MP_TOKEN_FNCT:
	    (*entryidx[1][1])->value = (*(p->symboltab->value.fnctptr))((*entryidx[1][1])->value);
	    break;
	case MP_TOKEN_UNARY_MINUS:
	    /* Take the last number on the stack and reverse the sign */
	    (*entryidx[1][1])->value *= (-1);
	    break;
	}
    }
    return (*entryidx[1][1])->value;
}


void mp_fix_stack (void)
{
    struct mp_stack *p;
    for (p=*entryidx[1][0]; p != 0; p=p->next)
    {
	mp_free_entry (1);
    }
}


void mp_stack_clean (void)
{
    struct mp_stack *p;
    for (p=*entryidx[0][0]; p != 0; p=p->next)
    {
	mp_free_entry (0);
    }
}


double mp_fact (double in)
{
    double lgam = lgamma (in + 1);
    return signgam * exp (lgam);
}


static struct mp_stack * mp_new_entry (const int stacknum)
{
    struct mp_stack *p;
    p = (struct mp_stack *) malloc (sizeof (struct mp_stack));
    
    if (!p)
    {
	/*ERROR*/
	return p;
    }
    if (!(*entryidx[stacknum][1]))
    {
        /* No last entry. This entry will be the first in the list. */
	(*entryidx[stacknum][0]) = p;
	(*entryidx[stacknum][1]) = p;
	p->next = 0;
	p->previous = 0;
    }
    else
    {
	p->next = 0;
	p->previous = (*entryidx[stacknum][1]);
	(*entryidx[stacknum][1])->next = p;
	(*entryidx[stacknum][1]) = p;
    }

    return (p);
}


static void mp_free_entry (const int stacknum)
{
    struct mp_stack *p = *entryidx[stacknum][1];
    if ((*entryidx[stacknum][0]) == p)
    {
	(*entryidx[stacknum][0]) = 0;
	(*entryidx[stacknum][1]) = 0;
	free (p);
	return;
    }
    (*entryidx[stacknum][1]) = p->previous;
    free (p);
    (*entryidx[stacknum][1])->next = 0;
}


/* Trig functions. Right now they defer to the libc functions after doing conversion to radians if necessary. */


/* Sine */
double mp_sin (double x)
{
    if (prf.trigunit == MP_UNIT_RADIAN)
    {
	return (sin(x));
    }
    return (sin((x/180)*M_PI));
}


/* Tangent */
double mp_tan (double x)
{
    if (prf.trigunit == MP_UNIT_RADIAN)
    {
	return (tan(x));
    }
    return (tan((x/180)*M_PI));
}


/* Cosine */
double mp_cos (double x)
{
    if (prf.trigunit == MP_UNIT_RADIAN)
    {
	return (cos(x));
    }
    return (cos((x/180)*M_PI));
}


/* Cotangent */
double mp_cot (double x)
{
    return (1/mp_tan(x));
}


/* Cosecant */
double mp_csc (double x)
{
    return (1/mp_sin(x));
}


/* Secant */
double mp_sec (double x)
{
    return (1/mp_cos(x));
}


/* Arc-sine */
double mp_asin (double x)
{
    if (prf.trigunit == MP_UNIT_RADIAN)
    {
	return (asin(x));
    }
    return (asin(x)/M_PI*180);
}


/* Arc-tangent */
double mp_atan (double x)
{
    if (prf.trigunit == MP_UNIT_RADIAN)
    {
	return (atan(x));
    }
    return (atan(x)/M_PI*180);
}


/* Arc-cosine */
double mp_acos (double x)
{
    if (prf.trigunit == MP_UNIT_RADIAN)
    {
	return (acos(x));
    }
    return (acos(x)/M_PI*180);
}


/* Hyperbolic arc-cosine */
double mp_asinh (double x)
{
    return (asinh(x));
}


/* Hyperbolic arc-tangent */
double mp_atanh (double x)
{
    return (atanh(x));
}


/* Hyperbolic arc-cosine */
double mp_acosh (double x)
{
    return (acosh(x));
}


/* Hyperbolic sine */
double mp_sinh (double x)
{
    return (sinh(x));
}


/* Hyperbolic tangent */
double mp_tanh (double x)
{
    return (tanh(x));
}


/* Hyperbolic cosine */
double mp_cosh (double x)
{
    return (cosh(x));
}

/* Sign */
double mp_sign (double x)
{
    if (x < 0) return (-1);
    if (x == 0) return (0);
    return (1);
}
